/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  assertUnreachable,
  HttpStatusCode,
  TalerError,
} from "@gnu-taler/taler-util";
import {
  Attention,
  ErrorLoading,
  Loading,
  RouteDefinition,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h } from "preact";
import { useServerMeasures } from "../hooks/server-info.js";
import { Profile } from "../pages/Profile.js";
import { computeMeasureInformation } from "../utils/computeAvailableMesaures.js";
import { CurrentMeasureTable } from "./MeasuresTable.js";

const TALER_SCREEN_ID = 124;

export function MeasureList({ routeToNew }: { routeToNew: RouteDefinition }) {
  const { i18n } = useTranslationContext();

  const measures = useServerMeasures();
  // const [custom] = useCustomMeasures();

  if (!measures) {
    return <Loading />;
  }
  if (measures instanceof TalerError) {
    return <ErrorLoading error={measures} />;
  }

  if (measures.type === "fail") {
    switch (measures.case) {
      case HttpStatusCode.Forbidden:
        return (
          <Fragment>
            <Attention type="danger" title={i18n.str`Operation denied`}>
              <i18n.Translate>
                This account signature is invalid, contact administrator or
                create a new one.
              </i18n.Translate>
            </Attention>
            <Profile />
          </Fragment>
        );
      case HttpStatusCode.NotFound:
        return (
          <Fragment>
            <Attention type="danger" title={i18n.str`Operation denied`}>
              <i18n.Translate>
                The designated AML account is not known, contact administrator
                or create a new one.
              </i18n.Translate>
            </Attention>
            <Profile />
          </Fragment>
        );
      case HttpStatusCode.Conflict:
        return (
          <Fragment>
            <Attention type="danger" title={i18n.str`Operation denied`}>
              <i18n.Translate>
                The designated AML account is not enabled, contact administrator
                or create a new one.
              </i18n.Translate>
            </Attention>
            <Profile />
          </Fragment>
        );
      default:
        assertUnreachable(measures);
    }
  }

  const ms = computeMeasureInformation(
    measures.body,
  );

  return (
    <div>
      <div class="px-4 sm:px-6 lg:px-8">
        <div class="sm:flex sm:items-center">
          <div class="sm:flex-auto">
            <h1 class="text-base font-semibold text-gray-900">
              <i18n.Translate>Measures</i18n.Translate>
            </h1>
            <p class="mt-2 text-sm text-gray-700">
              <i18n.Translate>
                A list of all the pre-define measures in your that can used.
              </i18n.Translate>
            </p>
          </div>
          <div class="mt-4 sm:ml-16 sm:mt-0 sm:flex-none">
            <a
              href={routeToNew.url({})}
              class="block rounded-md bg-indigo-600 px-3 py-2 text-center text-sm font-semibold text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
            >
              <i18n.Translate>Add custom measure</i18n.Translate>
            </a>
          </div>
        </div>

        <CurrentMeasureTable measures={ms} />
      </div>
    </div>
  );
}
