/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  AmlProgramRequirement,
  KycCheckInformation,
} from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import {
  MeasureInfo,
  UiMeasureInformation,
} from "../utils/computeAvailableMesaures.js";

const TALER_SCREEN_ID = 123;

export function CurrentMeasureTable({
  measures,
  onSelect,
  hideMeasureNames,
}: {
  measures: UiMeasureInformation;
  hideMeasureNames?: boolean;
  onSelect?: (m: MeasureInfo) => void;
}): VNode {
  const { i18n } = useTranslationContext();
  return (
    <Fragment>
      {!measures.forms.length ? undefined : (
        <div class="mt-4 flow-root">
          <div class="sm:flex sm:items-center">
            <div class="sm:flex-auto">
              <h1 class="text-base font-semibold text-gray-900">
                <i18n.Translate>Forms</i18n.Translate>
              </h1>
              <p class="mt-2 text-sm text-gray-700">
                <i18n.Translate>
                  Measures used to get gather information about the customer.
                </i18n.Translate>
              </p>
            </div>
          </div>

          <div class="inline-block min-w-full py-2 align-middle sm:px-6 lg:px-8">
            <div class="overflow-hidden shadow ring-1 ring-black/5 sm:rounded-lg">
              <table class="min-w-full divide-y divide-gray-300">
                <thead class="bg-gray-50">
                  <tr>
                    {onSelect ? (
                      <th scope="col" class="relative p-2 ">
                        <span class="sr-only">Select</span>
                      </th>
                    ) : (
                      <Fragment />
                    )}
                    {hideMeasureNames ? undefined : (
                      <th
                        scope="col"
                        class="p-2 text-left text-sm font-semibold text-gray-900 sm:pl-6"
                      >
                        <i18n.Translate>Name</i18n.Translate>
                      </th>
                    )}
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900"
                    >
                      <i18n.Translate>Check</i18n.Translate>
                    </th>
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900"
                    >
                      <i18n.Translate>Program</i18n.Translate>
                    </th>
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900"
                    >
                      <i18n.Translate>Context</i18n.Translate>
                    </th>
                  </tr>
                </thead>
                <tbody class="divide-y divide-gray-200 bg-white">
                  {measures.forms.map((m, k) => {
                    return (
                      <tr key={k}>
                        {!onSelect ? undefined : (
                          <td class="relative whitespace-nowrap p-2 text-right text-sm font-medium ">
                            <button
                              onClick={() => onSelect(m)}
                              class="rounded-md w-fit border-0 p-1 mr-1 text-center text-sm bg-indigo-700 text-white shadow-sm hover:bg-indigo-700"
                            >
                              <i18n.Translate>Modify</i18n.Translate>
                            </button>
                          </td>
                        )}
                        {hideMeasureNames ? undefined : (
                          <td class="whitespace-nowrap p-2 text-sm font-medium text-gray-900 ">
                            {m.name}
                          </td>
                        )}
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {m.check?.description ?? ""}
                        </td>
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {m.program?.description}
                        </td>
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {Object.keys(m.context ?? {}).join(", ")}
                        </td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      )}

      {!measures.info.length ? undefined : (
        <div class="mt-4 flow-root">
          <div class="sm:flex sm:items-center">
            <div class="sm:flex-auto">
              <h1 class="text-base font-semibold text-gray-900">
                <i18n.Translate>Info</i18n.Translate>
              </h1>
              <p class="mt-2 text-sm text-gray-700">
                <i18n.Translate>
                  Will show information to the user.
                </i18n.Translate>
              </p>
            </div>
          </div>

          <div class="inline-block min-w-full py-2 align-middle sm:px-6 lg:px-8">
            <div class="overflow-hidden shadow ring-1 ring-black/5 sm:rounded-lg">
              <table class="min-w-full divide-y divide-gray-300">
                <thead class="bg-gray-50">
                  <tr>
                    {onSelect ? (
                      <th scope="col" class="relative p-2 ">
                        <span class="sr-only">Select</span>
                      </th>
                    ) : (
                      <Fragment />
                    )}
                    {hideMeasureNames ? undefined : (
                      <th
                        scope="col"
                        class="p-2 text-left text-sm font-semibold text-gray-900 sm:pl-6"
                      >
                        <i18n.Translate>Name</i18n.Translate>
                      </th>
                    )}
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900 sm:pl-6"
                    >
                      <i18n.Translate>Check</i18n.Translate>
                    </th>
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900"
                    >
                      <i18n.Translate>Context</i18n.Translate>
                    </th>
                  </tr>
                </thead>
                <tbody class="divide-y divide-gray-200 bg-white">
                  {measures.info.map((m, k) => {
                    return (
                      <tr key={k}>
                        {onSelect ? (
                          <td class="relative whitespace-nowrap p-2 text-right text-sm font-medium ">
                            <button
                              onClick={() => onSelect(m)}
                              class="rounded-md w-fit border-0 p-2 text-center text-sm bg-indigo-700 text-white shadow-sm hover:bg-indigo-700"
                            >
                              <i18n.Translate>Modify</i18n.Translate>
                            </button>
                          </td>
                        ) : (
                          <Fragment />
                        )}
                        {hideMeasureNames ? undefined : (
                          <td class="whitespace-nowrap p-2 text-sm font-medium text-gray-900 sm:pl-6">
                            {m.name}
                          </td>
                        )}
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {m.checkName}
                        </td>
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {Object.keys(m.context ?? {}).join(", ")}
                        </td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      )}

      {!measures.procedures.length ? undefined : (
        <div class="mt-4 flow-root">
          <div class="sm:flex sm:items-center">
            <div class="sm:flex-auto">
              <h1 class="text-base font-semibold text-gray-900">
                <i18n.Translate>Instant measures</i18n.Translate>
              </h1>
              <p class="mt-2 text-sm text-gray-700">
                <i18n.Translate>
                  Triggered immediately without customer interaction.
                </i18n.Translate>
              </p>
            </div>
          </div>

          <div class="inline-block min-w-full py-2 align-middle sm:px-6 lg:px-8">
            <div class="overflow-hidden shadow ring-1 ring-black/5 sm:rounded-lg">
              <table class="min-w-full divide-y divide-gray-300">
                <thead class="bg-gray-50">
                  <tr>
                    {onSelect ? (
                      <th scope="col" class="relative p-2 ">
                        <span class="sr-only">Select</span>
                      </th>
                    ) : (
                      <Fragment />
                    )}
                    {hideMeasureNames ? undefined : (
                      <th
                        scope="col"
                        class="p-2 text-left text-sm font-semibold text-gray-900 sm:pl-6"
                      >
                        <i18n.Translate>Name</i18n.Translate>
                      </th>
                    )}
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900"
                    >
                      <i18n.Translate>Program</i18n.Translate>
                    </th>
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900"
                    >
                      <i18n.Translate>Input requirement</i18n.Translate>
                    </th>
                    <th
                      scope="col"
                      class="p-2 text-left text-sm font-semibold text-gray-900"
                    >
                      <i18n.Translate>Context</i18n.Translate>
                    </th>
                  </tr>
                </thead>
                <tbody class="divide-y divide-gray-200 bg-white">
                  {measures.procedures.map((m, k) => {
                    return (
                      <tr key={k}>
                        {onSelect ? (
                          <td class="relative whitespace-nowrap p-2 text-right text-sm font-medium ">
                            <button
                              onClick={() => onSelect(m)}
                              class="rounded-md w-fit border-0 p-2 text-center text-sm bg-indigo-700 text-white shadow-sm hover:bg-indigo-700"
                            >
                              <i18n.Translate>Modify</i18n.Translate>
                            </button>
                          </td>
                        ) : (
                          <Fragment />
                        )}
                        {hideMeasureNames ? undefined : (
                          <td class="whitespace-nowrap p-2 text-sm font-medium text-gray-900 sm:pl-6">
                            {m.name}
                          </td>
                        )}
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {m.program.description}
                        </td>
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {m.program.inputs.join(",")}
                        </td>
                        <td class="whitespace-nowrap p-2 text-sm text-gray-500">
                          {Object.keys(m.context ?? {}).join(", ")}
                        </td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      )}
    </Fragment>
  );
}
