/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  TransactionMajorState,
  TransactionMinorState,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  GlobalTestState,
  getTestHarnessPaytoForLabel,
} from "../harness/harness.js";

/**
 * Run test for basic, bank-integrated withdrawal and payment.
 */
export async function runDepositTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t);

  // Withdraw digital cash into the wallet.

  {
    const withdrawalResult = await withdrawViaBankV3(t, {
      walletClient,
      bankClient,
      exchange,
      amount: "TESTKUDOS:20",
    });
    await withdrawalResult.withdrawalFinishedCond;
  }

  const depositPaytoUri = getTestHarnessPaytoForLabel("foo");

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});

  t.assertAmountEquals(bal.balances[0].available, "TESTKUDOS:19.53");

  const maxDepositResp = await walletClient.call(
    WalletApiOperation.GetMaxDepositAmount,
    {
      currency: "TESTKUDOS",
      depositPaytoUri,
    },
  );

  t.assertAmountEquals(maxDepositResp.rawAmount, "TESTKUDOS:19.09");
  t.assertAmountEquals(maxDepositResp.effectiveAmount, "TESTKUDOS:19.53");

  const dgIdResp = await walletClient.client.call(
    WalletApiOperation.GenerateDepositGroupTxId,
    {},
  );

  const depositTxId = dgIdResp.transactionId;

  const depositGroupResult = await walletClient.client.call(
    WalletApiOperation.CreateDepositGroup,
    {
      amount: "TESTKUDOS:10" as AmountString,
      depositPaytoUri,
      transactionId: depositTxId,
    },
  );

  t.assertDeepEqual(depositGroupResult.transactionId, depositTxId);

  const balDuring = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(`balances during deposit: ${j2s(balDuring)}`);
  t.assertAmountEquals(balDuring.balances[0].pendingOutgoing, "TESTKUDOS:10");

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositTxId,
    txState: {
      major: TransactionMajorState.Finalizing,
      minor: TransactionMinorState.Track,
    },
  });

  t.logStep("before-aggregator");

  await exchange.runAggregatorOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60 * 3,
  });

  await exchange.runTransferOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60 * 3,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositTxId,
    txState: {
      major: TransactionMajorState.Done,
    },
  });

  const transactions = await walletClient.client.call(
    WalletApiOperation.GetTransactions,
    {},
  );

  console.log("transactions", JSON.stringify(transactions, undefined, 2));
  t.assertDeepEqual(transactions.transactions[0].type, "withdrawal");
  t.assertDeepEqual(transactions.transactions[1].type, "deposit");
  // The raw amount is what ends up on the bank account, which includes
  // deposit and wire fees.
  t.assertDeepEqual(transactions.transactions[1].amountRaw, "TESTKUDOS:9.79");

  const balAfter = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(`balances after deposit: ${j2s(balAfter)}`);
  t.assertAmountEquals(balAfter.balances[0].pendingOutgoing, "TESTKUDOS:0");

  // Now, test tiny deposits where fees eat the amount!

  {
    const withdrawalResult = await withdrawViaBankV3(t, {
      walletClient,
      bankClient,
      exchange,
      amount: "TESTKUDOS:5",
    });
    await withdrawalResult.withdrawalFinishedCond;
  }

  {
    const amt = "TESTKUDOS:0.01" as AmountString;
    const res = await walletClient.call(WalletApiOperation.CheckDeposit, {
      amount: amt,
      depositPaytoUri,
    });
    console.log(`check res: ${j2s(res)}`);
    const depositGroupResult = await walletClient.client.call(
      WalletApiOperation.CreateDepositGroup,
      {
        amount: amt,
        depositPaytoUri,
      },
    );

    console.log(`create result:`);
    console.log(j2s(depositGroupResult));

    const transactions = await walletClient.call(
      WalletApiOperation.GetTransactionsV2,
      {},
    );
    console.log(j2s(transactions));
    t.assertTrue(
      transactions.transactions.find(
        (x) => x.transactionId === depositGroupResult.transactionId,
      ) != null,
    );
  }
}

runDepositTest.suites = ["wallet"];
