use itertools::Itertools;
use ndarray::Array4;
use ort::{session::InMemorySession, value::TensorRef};

use super::{
    Point, Stroke,
    point::{ONE_POINT, ZERO_POINT},
    rect::Rect,
};

/// Classifier to classify a LaTeX symbol based on hand-drawn strokes.
pub struct Classifier {
    /// ML-Model used for finding the symbol
    model: InMemorySession<'static>,
}

impl Classifier {
    /// Create a new Classifier.
    ///
    /// This includes setting up and optimizing the model used for classification.
    pub fn new() -> ort::Result<Self> {
        let num_cpus = std::thread::available_parallelism()
            .expect("# CPU cores should be available")
            .get();
        let model = ort::session::Session::builder()?
            .with_optimization_level(ort::session::builder::GraphOptimizationLevel::Level3)?
            .with_intra_threads(num_cpus)?
            .commit_from_memory_directly(&include_bytes!("../../data/model.onnx")[..])?;

        Ok(Self { model })
    }

    /// Tries to classify the given strokes into a symbol.
    pub fn classify(&mut self, sample: Vec<Stroke>) -> Option<Vec<&'static str>> {
        let input_tensor = self.prepate_input(sample);

        let result = self
            .model
            .run(ort::inputs![
                TensorRef::from_array_view(input_tensor.view()).ok()?
            ])
            .ok()?;

        // convert output indices to detexify ids
        let output = result[0].try_extract_array().ok()?;
        let top_indices = top_k_indices(output.as_slice().unwrap(), 25);
        let top_labels: Vec<&'static str> = top_indices.iter().map(|&i| LABELS[i]).collect();
        Some(top_labels)
    }

    /// Prepares the input for the model.
    ///
    /// This is equivalent to drawing the strokes onto an image and
    /// than converting the image data into an array.
    fn prepate_input(&self, sample: Vec<Stroke>) -> Array4<f32> {
        let mut array = Array4::<f32>::zeros((1, 3, 32, 32));

        let strokes = sample
            .into_iter()
            .filter(|s| !s.is_empty())
            .take(10)
            .map(|mut stroke| {
                stroke.dedup();
                stroke.smooth();
                stroke.aspect_refit(Rect::new(ZERO_POINT, ONE_POINT));
                stroke.redistribute(10);
                stroke.dedup();
                stroke.dominant(2.0 * std::f64::consts::PI * 15.0 / 360.0);
                stroke
            })
            .collect_vec();

        for stroke in strokes {
            for (p, q) in stroke
                .points()
                // scale points from [0,1] to [0,31]
                .map(|Point { x, y }| ((x * 31.0).round() as i32, (y * 31.0).round() as i32))
                .tuple_windows()
            {
                self.draw_line(&mut array, p.0, p.1, q.0, q.1);
            }
        }
        array
    }

    /// Draws a line from `(x0, y0)` to `(x1, y1)` using
    /// [Bresenham's line algorithm](https://en.wikipedia.org/wiki/Bresenham%27s_line_algorithm?useskin=vector#Algorithm_for_integer_arithmetic)
    fn draw_line(&self, array: &mut Array4<f32>, x0: i32, y0: i32, x1: i32, y1: i32) {
        let dx = (x1 - x0).abs();
        let dy = -(y1 - y0).abs();
        let sx = if x0 < x1 { 1 } else { -1 };
        let sy = if y0 < y1 { 1 } else { -1 };
        let mut err = dx + dy;

        let mut x = x0;
        let mut y = y0;

        loop {
            array[[0, 0, y as usize, x as usize]] = 1.0;
            array[[0, 1, y as usize, x as usize]] = 1.0;
            array[[0, 2, y as usize, x as usize]] = 1.0;

            if x == x1 && y == y1 {
                break;
            }

            let e2 = 2 * err;
            if e2 >= dy {
                if x == x1 {
                    break;
                }
                err += dy;
                x += sx;
            }
            if e2 <= dx {
                if y == y1 {
                    break;
                }
                err += dx;
                y += sy;
            }
        }
    }
}

fn top_k_indices(arr: &[f32], k: usize) -> Vec<usize> {
    assert_eq!(arr.len(), LABELS.len());
    let mut indices: Vec<usize> = (0..arr.len()).collect();
    indices.sort_by(|&a, &b| arr[b].partial_cmp(&arr[a]).unwrap());
    indices.into_iter().take(k).collect()
}

static LABELS: [&str; 1098] = [
    "Y21sbC1PVDEtX3BhcnI=",
    "Y21sbC1PVDEtX3dpdGg=",
    "YW1zbWF0aC1PVDEtX2RvdHNi",
    "YW1zbWF0aC1PVDEtX2RvdHNj",
    "YW1zbWF0aC1PVDEtX2RvdHNp",
    "YW1zbWF0aC1PVDEtX2RvdHNt",
    "YW1zbWF0aC1PVDEtX2RvdHNv",
    "YW1zbWF0aC1PVDEtX2lkb3RzaW50",
    "YW1zbWF0aC1PVDEtX2lpaW50",
    "YW1zbWF0aC1PVDEtX2lpaWludA==",
    "YW1zbWF0aC1PVDEtX2lpbnQ=",
    "YW1zc3ltYi1PVDEtX0J1bXBlcQ==",
    "YW1zc3ltYi1PVDEtX0JiYms=",
    "YW1zc3ltYi1PVDEtX0N1cA==",
    "YW1zc3ltYi1PVDEtX0NhcA==",
    "YW1zc3ltYi1PVDEtX0ZpbnY=",
    "YW1zc3ltYi1PVDEtX0dhbWU=",
    "YW1zc3ltYi1PVDEtX0xsZWZ0YXJyb3c=",
    "YW1zc3ltYi1PVDEtX0xzaA==",
    "YW1zc3ltYi1PVDEtX1JyaWdodGFycm93",
    "YW1zc3ltYi1PVDEtX1JzaA==",
    "YW1zc3ltYi1PVDEtX1N1YnNldA==",
    "YW1zc3ltYi1PVDEtX1N1cHNldA==",
    "YW1zc3ltYi1PVDEtX1Z2ZGFzaA==",
    "YW1zc3ltYi1PVDEtX1ZkYXNo",
    "YW1zc3ltYi1PVDEtX211bHRpbWFw",
    "YW1zc3ltYi1PVDEtX21hbHRlc2U=",
    "YW1zc3ltYi1PVDEtX21hdGhiYntBfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntCfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntDfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntEfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntFfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntGfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntHfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntIfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntJfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntKfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntLfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntMfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntNfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntOfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntPfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntQfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntRfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntSfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntTfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntUfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntVfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntWfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntXfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntYfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntZfQ==",
    "YW1zc3ltYi1PVDEtX21hdGhiYntafQ==",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Q30=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7QX0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Qn0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7R30=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7RH0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7RX0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Rn0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7S30=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7SH0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7SX0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Sn0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7T30=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7TH0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7TX0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Tn0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7U30=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7UH0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7UX0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Un0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7V30=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7VH0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7VX0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Vn0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7WH0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7WX0=",
    "YW1zc3ltYi1PVDEtX21hdGhjYWx7Wn0=",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre019",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre059",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre099",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0F9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0J9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0N9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0R9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0V9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0Z9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0d9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0h9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0l9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0p9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0t9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre0x9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1B9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1F9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1J9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1N9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1R9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1V9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1Z9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1d9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1h9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1l9",
    "YW1zc3ltYi1PVDEtX21hdGhmcmFre1p9",
    "YW1zc3ltYi1PVDEtX21lYXN1cmVkYW5nbGU=",
    "YW1zc3ltYi1PVDEtX21obw==",
    "YW1zc3ltYi1PVDEtX250cmlhbmdsZWxlZnQ=",
    "YW1zc3ltYi1PVDEtX250cmlhbmdsZWxlZnRlcQ==",
    "YW1zc3ltYi1PVDEtX250cmlhbmdsZXJpZ2h0",
    "YW1zc3ltYi1PVDEtX250cmlhbmdsZXJpZ2h0ZXE=",
    "YW1zc3ltYi1PVDEtX252RGFzaA==",
    "YW1zc3ltYi1PVDEtX252ZGFzaA==",
    "YW1zc3ltYi1PVDEtX25MZWZ0YXJyb3c=",
    "YW1zc3ltYi1PVDEtX25MZWZ0cmlnaHRhcnJvdw==",
    "YW1zc3ltYi1PVDEtX25SaWdodGFycm93",
    "YW1zc3ltYi1PVDEtX25WRGFzaA==",
    "YW1zc3ltYi1PVDEtX25jb25n",
    "YW1zc3ltYi1PVDEtX25lcQ==",
    "YW1zc3ltYi1PVDEtX25leGlzdHM=",
    "YW1zc3ltYi1PVDEtX25nZXE=",
    "YW1zc3ltYi1PVDEtX25nZXFx",
    "YW1zc3ltYi1PVDEtX25nZXFzbGFudA==",
    "YW1zc3ltYi1PVDEtX25ndHI=",
    "YW1zc3ltYi1PVDEtX25sZWZ0YXJyb3c=",
    "YW1zc3ltYi1PVDEtX25sZWZ0cmlnaHRhcnJvdw==",
    "YW1zc3ltYi1PVDEtX25sZXE=",
    "YW1zc3ltYi1PVDEtX25sZXFx",
    "YW1zc3ltYi1PVDEtX25sZXFzbGFudA==",
    "YW1zc3ltYi1PVDEtX25sZXNz",
    "YW1zc3ltYi1PVDEtX25taWQ=",
    "YW1zc3ltYi1PVDEtX25wYXJhbGxlbA==",
    "YW1zc3ltYi1PVDEtX25wcmVj",
    "YW1zc3ltYi1PVDEtX25wcmVjZXE=",
    "YW1zc3ltYi1PVDEtX25yaWdodGFycm93",
    "YW1zc3ltYi1PVDEtX25zaG9ydG1pZA==",
    "YW1zc3ltYi1PVDEtX25zaG9ydHBhcmFsbGVs",
    "YW1zc3ltYi1PVDEtX25zaW0=",
    "YW1zc3ltYi1PVDEtX25zdWJzZXRlcQ==",
    "YW1zc3ltYi1PVDEtX25zdWNj",
    "YW1zc3ltYi1PVDEtX25zdWNjZXE=",
    "YW1zc3ltYi1PVDEtX25zdXBzZXRlcQ==",
    "YW1zc3ltYi1PVDEtX25zdXBzZXRlcXE=",
    "YW1zc3ltYi1PVDEtX2FuZ2xl",
    "YW1zc3ltYi1PVDEtX2FwcHJveGVx",
    "YW1zc3ltYi1PVDEtX2J1bXBlcQ==",
    "YW1zc3ltYi1PVDEtX2JhY2tlcHNpbG9u",
    "YW1zc3ltYi1PVDEtX2JhY2twcmltZQ==",
    "YW1zc3ltYi1PVDEtX2JhY2tzaW0=",
    "YW1zc3ltYi1PVDEtX2JhY2tzaW1lcQ==",
    "YW1zc3ltYi1PVDEtX2JhcndlZGdl",
    "YW1zc3ltYi1PVDEtX2JlY2F1c2U=",
    "YW1zc3ltYi1PVDEtX2JldHdlZW4=",
    "YW1zc3ltYi1PVDEtX2JpZ3N0YXI=",
    "YW1zc3ltYi1PVDEtX2JsYWNrbG96ZW5nZQ==",
    "YW1zc3ltYi1PVDEtX2JsYWNrc3F1YXJl",
    "YW1zc3ltYi1PVDEtX2JsYWNrdHJpYW5nbGU=",
    "YW1zc3ltYi1PVDEtX2JsYWNrdHJpYW5nbGVkb3du",
    "YW1zc3ltYi1PVDEtX2JsYWNrdHJpYW5nbGVsZWZ0",
    "YW1zc3ltYi1PVDEtX2JsYWNrdHJpYW5nbGVyaWdodA==",
    "YW1zc3ltYi1PVDEtX2JveG1pbnVz",
    "YW1zc3ltYi1PVDEtX2JveGRvdA==",
    "YW1zc3ltYi1PVDEtX2JveHBsdXM=",
    "YW1zc3ltYi1PVDEtX2JveHRpbWVz",
    "YW1zc3ltYi1PVDEtX2N1cmx5ZXFwcmVj",
    "YW1zc3ltYi1PVDEtX2N1cmx5ZXFzdWNj",
    "YW1zc3ltYi1PVDEtX2N1cmx5d2VkZ2U=",
    "YW1zc3ltYi1PVDEtX2N1cmx5dmVl",
    "YW1zc3ltYi1PVDEtX2N1cnZlYXJyb3dsZWZ0",
    "YW1zc3ltYi1PVDEtX2N1cnZlYXJyb3dyaWdodA==",
    "YW1zc3ltYi1PVDEtX2NlbnRlcmRvdA==",
    "YW1zc3ltYi1PVDEtX2NoZWNrbWFyaw==",
    "YW1zc3ltYi1PVDEtX2NpcmNlcQ==",
    "YW1zc3ltYi1PVDEtX2NpcmNsZWFycm93bGVmdA==",
    "YW1zc3ltYi1PVDEtX2NpcmNsZWFycm93cmlnaHQ=",
    "YW1zc3ltYi1PVDEtX2NpcmNsZWRS",
    "YW1zc3ltYi1PVDEtX2NpcmNsZWRT",
    "YW1zc3ltYi1PVDEtX2NpcmNsZWRhc3Q=",
    "YW1zc3ltYi1PVDEtX2NpcmNsZWRjaXJj",
    "YW1zc3ltYi1PVDEtX2NpcmNsZWRkYXNo",
    "YW1zc3ltYi1PVDEtX2NvbXBsZW1lbnQ=",
    "YW1zc3ltYi1PVDEtX2Rhc2hsZWZ0YXJyb3c=",
    "YW1zc3ltYi1PVDEtX2Rhc2hyaWdodGFycm93",
    "YW1zc3ltYi1PVDEtX2RpYWd1cA==",
    "YW1zc3ltYi1PVDEtX2RpYWdkb3du",
    "YW1zc3ltYi1PVDEtX2RpZ2FtbWE=",
    "YW1zc3ltYi1PVDEtX2RpdmlkZW9udGltZXM=",
    "YW1zc3ltYi1PVDEtX2Rvd25kb3duYXJyb3dz",
    "YW1zc3ltYi1PVDEtX2Rvd25oYXJwb29ubGVmdA==",
    "YW1zc3ltYi1PVDEtX2Rvd25oYXJwb29ucmlnaHQ=",
    "YW1zc3ltYi1PVDEtX2RvdGVxZG90",
    "YW1zc3ltYi1PVDEtX2RvdHBsdXM=",
    "YW1zc3ltYi1PVDEtX2RvdWJsZWJhcndlZGdl",
    "YW1zc3ltYi1PVDEtX2V0aA==",
    "YW1zc3ltYi1PVDEtX2VxY2lyYw==",
    "YW1zc3ltYi1PVDEtX2Vxc2xhbnRndHI=",
    "YW1zc3ltYi1PVDEtX2Vxc2xhbnRsZXNz",
    "YW1zc3ltYi1PVDEtX2ZhbGxpbmdkb3RzZXE=",
    "YW1zc3ltYi1PVDEtX2d0cmFwcHJveA==",
    "YW1zc3ltYi1PVDEtX2d0cmRvdA==",
    "YW1zc3ltYi1PVDEtX2d0cmVxbGVzcw==",
    "YW1zc3ltYi1PVDEtX2d0cmVxcWxlc3M=",
    "YW1zc3ltYi1PVDEtX2d0cmxlc3M=",
    "YW1zc3ltYi1PVDEtX2d0cnNpbQ==",
    "YW1zc3ltYi1PVDEtX2d2ZXJ0bmVxcQ==",
    "YW1zc3ltYi1PVDEtX2dlcQ==",
    "YW1zc3ltYi1PVDEtX2dlcXE=",
    "YW1zc3ltYi1PVDEtX2dlcXNsYW50",
    "YW1zc3ltYi1PVDEtX2dn",
    "YW1zc3ltYi1PVDEtX2dnZw==",
    "YW1zc3ltYi1PVDEtX2duYXBwcm94",
    "YW1zc3ltYi1PVDEtX2duZXE=",
    "YW1zc3ltYi1PVDEtX2duZXFx",
    "YW1zc3ltYi1PVDEtX2duc2lt",
    "YW1zc3ltYi1PVDEtX2hiYXI=",
    "YW1zc3ltYi1PVDEtX2hzbGFzaA==",
    "YW1zc3ltYi1PVDEtX2ludGVyY2Fs",
    "YW1zc3ltYi1PVDEtX2x0aW1lcw==",
    "YW1zc3ltYi1PVDEtX2x2ZXJ0bmVxcQ==",
    "YW1zc3ltYi1PVDEtX2xlZnR0aHJlZXRpbWVz",
    "YW1zc3ltYi1PVDEtX2xlZnRhcnJvd3RhaWw=",
    "YW1zc3ltYi1PVDEtX2xlZnRsZWZ0YXJyb3dz",
    "YW1zc3ltYi1PVDEtX2xlZnRyaWdodGFycm93cw==",
    "YW1zc3ltYi1PVDEtX2xlZnRyaWdodGhhcnBvb25z",
    "YW1zc3ltYi1PVDEtX2xlZnRyaWdodHNxdWlnYXJyb3c=",
    "YW1zc3ltYi1PVDEtX2xlc3NhcHByb3g=",
    "YW1zc3ltYi1PVDEtX2xlc3Nkb3Q=",
    "YW1zc3ltYi1PVDEtX2xlc3NlcWd0cg==",
    "YW1zc3ltYi1PVDEtX2xlc3NlcXFndHI=",
    "YW1zc3ltYi1PVDEtX2xlc3NndHI=",
    "YW1zc3ltYi1PVDEtX2xlc3NzaW0=",
    "YW1zc3ltYi1PVDEtX2xlcQ==",
    "YW1zc3ltYi1PVDEtX2xlcXE=",
    "YW1zc3ltYi1PVDEtX2xlcXNsYW50",
    "YW1zc3ltYi1PVDEtX2xs",
    "YW1zc3ltYi1PVDEtX2xsbA==",
    "YW1zc3ltYi1PVDEtX2xuYXBwcm94",
    "YW1zc3ltYi1PVDEtX2xuZXE=",
    "YW1zc3ltYi1PVDEtX2xuZXFx",
    "YW1zc3ltYi1PVDEtX2xuc2lt",
    "YW1zc3ltYi1PVDEtX2xvb3BhcnJvd2xlZnQ=",
    "YW1zc3ltYi1PVDEtX2xvb3BhcnJvd3JpZ2h0",
    "YW1zc3ltYi1PVDEtX2xvemVuZ2U=",
    "YW1zc3ltYi1PVDEtX3BpdGNoZm9yaw==",
    "YW1zc3ltYi1PVDEtX3ByZWNhcHByb3g=",
    "YW1zc3ltYi1PVDEtX3ByZWNjdXJseWVx",
    "YW1zc3ltYi1PVDEtX3ByZWNuYXBwcm94",
    "YW1zc3ltYi1PVDEtX3ByZWNuc2lt",
    "YW1zc3ltYi1PVDEtX3ByZWNzaW0=",
    "YW1zc3ltYi1PVDEtX3J0aW1lcw==",
    "YW1zc3ltYi1PVDEtX3JpZ2h0YXJyb3d0YWls",
    "YW1zc3ltYi1PVDEtX3JpZ2h0bGVmdGFycm93cw==",
    "YW1zc3ltYi1PVDEtX3JpZ2h0bGVmdGhhcnBvb25z",
    "YW1zc3ltYi1PVDEtX3JpZ2h0c3F1aWdhcnJvdw==",
    "YW1zc3ltYi1PVDEtX3JpZ2h0cmlnaHRhcnJvd3M=",
    "YW1zc3ltYi1PVDEtX3JpZ2h0dGhyZWV0aW1lcw==",
    "YW1zc3ltYi1PVDEtX3Jpc2luZ2RvdHNlcQ==",
    "YW1zc3ltYi1PVDEtX3N1Y2NhcHByb3g=",
    "YW1zc3ltYi1PVDEtX3N1Y2NjdXJseWVx",
    "YW1zc3ltYi1PVDEtX3N1Y2NuYXBwcm94",
    "YW1zc3ltYi1PVDEtX3N1Y2Nuc2lt",
    "YW1zc3ltYi1PVDEtX3N1Y2NzaW0=",
    "YW1zc3ltYi1PVDEtX3N1YnNldG5lcQ==",
    "YW1zc3ltYi1PVDEtX3N1YnNldG5lcXE=",
    "YW1zc3ltYi1PVDEtX3N1YnNldGVxcQ==",
    "YW1zc3ltYi1PVDEtX3N1cHNldG5lcQ==",
    "YW1zc3ltYi1PVDEtX3N1cHNldG5lcXE=",
    "YW1zc3ltYi1PVDEtX3N1cHNldGVxcQ==",
    "YW1zc3ltYi1PVDEtX3Nob3J0bWlk",
    "YW1zc3ltYi1PVDEtX3Nob3J0cGFyYWxsZWw=",
    "YW1zc3ltYi1PVDEtX3NtYWxsZnJvd24=",
    "YW1zc3ltYi1PVDEtX3NtYWxsc21pbGU=",
    "YW1zc3ltYi1PVDEtX3NtYWxsc2V0bWludXM=",
    "YW1zc3ltYi1PVDEtX3NwaGVyaWNhbGFuZ2xl",
    "YW1zc3ltYi1PVDEtX3Nxc3Vic2V0",
    "YW1zc3ltYi1PVDEtX3Nxc3Vwc2V0",
    "YW1zc3ltYi1PVDEtX3NxdWFyZQ==",
    "YW1zc3ltYi1PVDEtX3R3b2hlYWRsZWZ0YXJyb3c=",
    "YW1zc3ltYi1PVDEtX3R3b2hlYWRyaWdodGFycm93",
    "YW1zc3ltYi1PVDEtX3RoZXJlZm9yZQ==",
    "YW1zc3ltYi1PVDEtX3RoaWNrYXBwcm94",
    "YW1zc3ltYi1PVDEtX3RoaWNrc2lt",
    "YW1zc3ltYi1PVDEtX3RyaWFuZ2xlZG93bg==",
    "YW1zc3ltYi1PVDEtX3RyaWFuZ2xlbGVmdGVx",
    "YW1zc3ltYi1PVDEtX3RyaWFuZ2xlcQ==",
    "YW1zc3ltYi1PVDEtX3RyaWFuZ2xlcmlnaHRlcQ==",
    "YW1zc3ltYi1PVDEtX3VwaGFycG9vbmxlZnQ=",
    "YW1zc3ltYi1PVDEtX3VwaGFycG9vbnJpZ2h0",
    "YW1zc3ltYi1PVDEtX3VwdXBhcnJvd3M=",
    "YW1zc3ltYi1PVDEtX3ZEYXNo",
    "YW1zc3ltYi1PVDEtX3Zhcm5vdGhpbmc=",
    "YW1zc3ltYi1PVDEtX3ZhcmthcHBh",
    "YW1zc3ltYi1PVDEtX3ZhcnByb3B0bw==",
    "YW1zc3ltYi1PVDEtX3ZhcnN1YnNldG5lcQ==",
    "YW1zc3ltYi1PVDEtX3ZhcnN1YnNldG5lcXE=",
    "YW1zc3ltYi1PVDEtX3ZhcnN1cHNldG5lcQ==",
    "YW1zc3ltYi1PVDEtX3ZhcnN1cHNldG5lcXE=",
    "YW1zc3ltYi1PVDEtX3ZhcnRyaWFuZ2xl",
    "YW1zc3ltYi1PVDEtX3ZhcnRyaWFuZ2xlbGVmdA==",
    "YW1zc3ltYi1PVDEtX3ZhcnRyaWFuZ2xlcmlnaHQ=",
    "YW1zc3ltYi1PVDEtX3ZlZWJhcg==",
    "YmJvbGQtT1QxLV9tYXRoYmJ7MX0=",
    "Z2Vuc3ltYi1PVDEtX29obQ==",
    "Z2Vuc3ltYi1PVDEtX2NlbHNpdXM=",
    "Z2Vuc3ltYi1PVDEtX2RlZ3JlZQ==",
    "ZHNmb250LU9UMS1fbWF0aGRze019",
    "ZHNmb250LU9UMS1fbWF0aGRze059",
    "ZHNmb250LU9UMS1fbWF0aGRze099",
    "ZHNmb250LU9UMS1fbWF0aGRze0F9",
    "ZHNmb250LU9UMS1fbWF0aGRze0J9",
    "ZHNmb250LU9UMS1fbWF0aGRze0N9",
    "ZHNmb250LU9UMS1fbWF0aGRze0R9",
    "ZHNmb250LU9UMS1fbWF0aGRze0V9",
    "ZHNmb250LU9UMS1fbWF0aGRze0Z9",
    "ZHNmb250LU9UMS1fbWF0aGRze0d9",
    "ZHNmb250LU9UMS1fbWF0aGRze0h9",
    "ZHNmb250LU9UMS1fbWF0aGRze0l9",
    "ZHNmb250LU9UMS1fbWF0aGRze0p9",
    "ZHNmb250LU9UMS1fbWF0aGRze0t9",
    "ZHNmb250LU9UMS1fbWF0aGRze0x9",
    "ZHNmb250LU9UMS1fbWF0aGRze1B9",
    "ZHNmb250LU9UMS1fbWF0aGRze1F9",
    "ZHNmb250LU9UMS1fbWF0aGRze1J9",
    "ZHNmb250LU9UMS1fbWF0aGRze1N9",
    "ZHNmb250LU9UMS1fbWF0aGRze1R9",
    "ZHNmb250LU9UMS1fbWF0aGRze1V9",
    "ZHNmb250LU9UMS1fbWF0aGRze1Z9",
    "ZHNmb250LU9UMS1fbWF0aGRze1d9",
    "ZHNmb250LU9UMS1fbWF0aGRze1h9",
    "ZHNmb250LU9UMS1fbWF0aGRze1l9",
    "ZHNmb250LU9UMS1fbWF0aGRze1p9",
    "ZHNmb250LU9UMS1fbWF0aGRze2h9",
    "ZHNmb250LU9UMS1fbWF0aGRze2t9",
    "ZHNmb250LU9UMS1fbWF0aGRzezF9",
    "ZXNpbnQtT1QxLV92YXJvaW50Y2xvY2t3aXNl",
    "ZXNpbnQtT1QxLV92YXJvaW50Y3RyY2xvY2t3aXNl",
    "ZXNpbnQtT1QxLV92YXJvaWludA==",
    "ZXNpbnQtT1QxLV9kb3RzaW50",
    "ZXNpbnQtT1QxLV9maW50",
    "ZXNpbnQtT1QxLV9paW50",
    "ZXNpbnQtT1QxLV9paWlpbnQ=",
    "ZXNpbnQtT1QxLV9paWludA==",
    "ZXNpbnQtT1QxLV9sYW5kZG93bmludA==",
    "ZXNpbnQtT1QxLV9sYW5kdXBpbnQ=",
    "ZXNpbnQtT1QxLV9vaW50Y2xvY2t3aXNl",
    "ZXNpbnQtT1QxLV9vaW50Y3RyY2xvY2t3aXNl",
    "ZXNpbnQtT1QxLV9vaWludA==",
    "ZXNpbnQtT1QxLV9zcWlpbnQ=",
    "ZXNpbnQtT1QxLV9zcWludA==",
    "bGF0ZXgyZS1PVDEtIWA=",
    "bGF0ZXgyZS1PVDEtLw==",
    "bGF0ZXgyZS1PVDEtWw==",
    "bGF0ZXgyZS1PVDEtX08=",
    "bGF0ZXgyZS1PVDEtX09F",
    "bGF0ZXgyZS1PVDEtX09tZWdh",
    "bGF0ZXgyZS1PVDEtX0FB",
    "bGF0ZXgyZS1PVDEtX0FF",
    "bGF0ZXgyZS1PVDEtX0RlbHRh",
    "bGF0ZXgyZS1PVDEtX0Rvd25hcnJvdw==",
    "bGF0ZXgyZS1PVDEtX0dhbW1h",
    "bGF0ZXgyZS1PVDEtX0lt",
    "bGF0ZXgyZS1PVDEtX0w=",
    "bGF0ZXgyZS1PVDEtX0xhbWJkYQ==",
    "bGF0ZXgyZS1PVDEtX0xlZnRhcnJvdw==",
    "bGF0ZXgyZS1PVDEtX0xlZnRyaWdodGFycm93",
    "bGF0ZXgyZS1PVDEtX0xvbmdsZWZ0YXJyb3c=",
    "bGF0ZXgyZS1PVDEtX0xvbmdsZWZ0cmlnaHRhcnJvdw==",
    "bGF0ZXgyZS1PVDEtX0xvbmdyaWdodGFycm93",
    "bGF0ZXgyZS1PVDEtX18=",
    "bGF0ZXgyZS1PVDEtX1A=",
    "bGF0ZXgyZS1PVDEtX1BoaQ==",
    "bGF0ZXgyZS1PVDEtX1Bp",
    "bGF0ZXgyZS1PVDEtX1BzaQ==",
    "bGF0ZXgyZS1PVDEtX1Jl",
    "bGF0ZXgyZS1PVDEtX1JpZ2h0YXJyb3c=",
    "bGF0ZXgyZS1PVDEtX1M=",
    "bGF0ZXgyZS1PVDEtX1NT",
    "bGF0ZXgyZS1PVDEtX1NpZ21h",
    "bGF0ZXgyZS1PVDEtX1RoZXRh",
    "bGF0ZXgyZS1PVDEtX1VwYXJyb3c=",
    "bGF0ZXgyZS1PVDEtX1VwZG93bmFycm93",
    "bGF0ZXgyZS1PVDEtX1Vwc2lsb24=",
    "bGF0ZXgyZS1PVDEtX1hp",
    "bGF0ZXgyZS1PVDEtX211",
    "bGF0ZXgyZS1PVDEtX21hcHN0bw==",
    "bGF0ZXgyZS1PVDEtX21hdGh1bmRlcnNjb3Jl",
    "bGF0ZXgyZS1PVDEtX21hdGhkb2xsYXI=",
    "bGF0ZXgyZS1PVDEtX21hdGhlbGxpcHNpcw==",
    "bGF0ZXgyZS1PVDEtX21hdGhwYXJhZ3JhcGg=",
    "bGF0ZXgyZS1PVDEtX21hdGhzZWN0aW9u",
    "bGF0ZXgyZS1PVDEtX21hdGhzdGVybGluZw==",
    "bGF0ZXgyZS1PVDEtX21pZA==",
    "bGF0ZXgyZS1PVDEtX21vZGVscw==",
    "bGF0ZXgyZS1PVDEtX21w",
    "bGF0ZXgyZS1PVDEtX251",
    "bGF0ZXgyZS1PVDEtX253YXJyb3c=",
    "bGF0ZXgyZS1PVDEtX25hYmxh",
    "bGF0ZXgyZS1PVDEtX25hdHVyYWw=",
    "bGF0ZXgyZS1PVDEtX25lYXJyb3c=",
    "bGF0ZXgyZS1PVDEtX25lZw==",
    "bGF0ZXgyZS1PVDEtX25p",
    "bGF0ZXgyZS1PVDEtX25vdF9hcHByb3g=",
    "bGF0ZXgyZS1PVDEtX25vdF9lcXVpdg==",
    "bGF0ZXgyZS1PVDEtX25vdF9zaW0=",
    "bGF0ZXgyZS1PVDEtX25vdF9zaW1lcQ==",
    "bGF0ZXgyZS1PVDEtX25vdGlu",
    "bGF0ZXgyZS1PVDEtX28=",
    "bGF0ZXgyZS1PVDEtX290aW1lcw==",
    "bGF0ZXgyZS1PVDEtX29kb3Q=",
    "bGF0ZXgyZS1PVDEtX29l",
    "bGF0ZXgyZS1PVDEtX29pbnQ=",
    "bGF0ZXgyZS1PVDEtX29tZWdh",
    "bGF0ZXgyZS1PVDEtX29taW51cw==",
    "bGF0ZXgyZS1PVDEtX29wbHVz",
    "bGF0ZXgyZS1PVDEtX29zbGFzaA==",
    "bGF0ZXgyZS1PVDEtX2Fh",
    "bGF0ZXgyZS1PVDEtX2Fl",
    "bGF0ZXgyZS1PVDEtX2FsZXBo",
    "bGF0ZXgyZS1PVDEtX2FscGhh",
    "bGF0ZXgyZS1PVDEtX2FtYWxn",
    "bGF0ZXgyZS1PVDEtX2FuZ2xl",
    "bGF0ZXgyZS1PVDEtX2FwcHJveA==",
    "bGF0ZXgyZS1PVDEtX2FzdA==",
    "bGF0ZXgyZS1PVDEtX2FzeW1w",
    "bGF0ZXgyZS1PVDEtX2J1bGxldA==",
    "bGF0ZXgyZS1PVDEtX2JhY2tzbGFzaA==",
    "bGF0ZXgyZS1PVDEtX2JldGE=",
    "bGF0ZXgyZS1PVDEtX2JpZ290aW1lcw==",
    "bGF0ZXgyZS1PVDEtX2JpZ29kb3Q=",
    "bGF0ZXgyZS1PVDEtX2JpZ29wbHVz",
    "bGF0ZXgyZS1PVDEtX2JpZ2N1cA==",
    "bGF0ZXgyZS1PVDEtX2JpZ2NhcA==",
    "bGF0ZXgyZS1PVDEtX2JpZ2NpcmM=",
    "bGF0ZXgyZS1PVDEtX2JpZ3NxY3Vw",
    "bGF0ZXgyZS1PVDEtX2JpZ3RyaWFuZ2xlZG93bg==",
    "bGF0ZXgyZS1PVDEtX2JpZ3RyaWFuZ2xldXA=",
    "bGF0ZXgyZS1PVDEtX2JpZ3VwbHVz",
    "bGF0ZXgyZS1PVDEtX2JpZ3ZlZQ==",
    "bGF0ZXgyZS1PVDEtX2JpZ3dlZGdl",
    "bGF0ZXgyZS1PVDEtX2Jvd3RpZQ==",
    "bGF0ZXgyZS1PVDEtX2JvdA==",
    "bGF0ZXgyZS1PVDEtX2N1cA==",
    "bGF0ZXgyZS1PVDEtX2NhcA==",
    "bGF0ZXgyZS1PVDEtX2Nkb3Q=",
    "bGF0ZXgyZS1PVDEtX2Nkb3Rw",
    "bGF0ZXgyZS1PVDEtX2Nkb3Rz",
    "bGF0ZXgyZS1PVDEtX2NoaQ==",
    "bGF0ZXgyZS1PVDEtX2NpcmM=",
    "bGF0ZXgyZS1PVDEtX2NsdWJzdWl0",
    "bGF0ZXgyZS1PVDEtX2NvbG9u",
    "bGF0ZXgyZS1PVDEtX2Nvbmc=",
    "bGF0ZXgyZS1PVDEtX2NvcHJvZA==",
    "bGF0ZXgyZS1PVDEtX2NvcHlyaWdodA==",
    "bGF0ZXgyZS1PVDEtX2RhZ2dlcg==",
    "bGF0ZXgyZS1PVDEtX2RhZw==",
    "bGF0ZXgyZS1PVDEtX2Rhc2h2",
    "bGF0ZXgyZS1PVDEtX2RkYWc=",
    "bGF0ZXgyZS1PVDEtX2RkYWdnZXI=",
    "bGF0ZXgyZS1PVDEtX2Rkb3Rz",
    "bGF0ZXgyZS1PVDEtX2RlbHRh",
    "bGF0ZXgyZS1PVDEtX2RpYW1vbmQ=",
    "bGF0ZXgyZS1PVDEtX2RpYW1vbmRzdWl0",
    "bGF0ZXgyZS1PVDEtX2Rpdg==",
    "bGF0ZXgyZS1PVDEtX2Rvd25hcnJvdw==",
    "bGF0ZXgyZS1PVDEtX2RvdGVx",
    "bGF0ZXgyZS1PVDEtX2RvdHM=",
    "bGF0ZXgyZS1PVDEtX2V0YQ==",
    "bGF0ZXgyZS1PVDEtX2V4aXN0cw==",
    "bGF0ZXgyZS1PVDEtX2VsbA==",
    "bGF0ZXgyZS1PVDEtX2VtcHR5c2V0",
    "bGF0ZXgyZS1PVDEtX2Vwc2lsb24=",
    "bGF0ZXgyZS1PVDEtX2VxdWl2",
    "bGF0ZXgyZS1PVDEtX2ZsYXQ=",
    "bGF0ZXgyZS1PVDEtX2ZvcmFsbA==",
    "bGF0ZXgyZS1PVDEtX2Zyb3du",
    "bGF0ZXgyZS1PVDEtX2dhbW1h",
    "bGF0ZXgyZS1PVDEtX2hiYXI=",
    "bGF0ZXgyZS1PVDEtX2hlYXJ0c3VpdA==",
    "bGF0ZXgyZS1PVDEtX2hvb2tsZWZ0YXJyb3c=",
    "bGF0ZXgyZS1PVDEtX2hvb2tyaWdodGFycm93",
    "bGF0ZXgyZS1PVDEtX2ltYXRo",
    "bGF0ZXgyZS1PVDEtX2lu",
    "bGF0ZXgyZS1PVDEtX2luZnR5",
    "bGF0ZXgyZS1PVDEtX2ludA==",
    "bGF0ZXgyZS1PVDEtX2lvdGE=",
    "bGF0ZXgyZS1PVDEtX2ptYXRo",
    "bGF0ZXgyZS1PVDEtX2thcHBh",
    "bGF0ZXgyZS1PVDEtX2w=",
    "bGF0ZXgyZS1PVDEtX2xhbWJkYQ==",
    "bGF0ZXgyZS1PVDEtX2xhbmdsZQ==",
    "bGF0ZXgyZS1PVDEtX2xjZWls",
    "bGF0ZXgyZS1PVDEtX2xkb3Rw",
    "bGF0ZXgyZS1PVDEtX2xkb3Rz",
    "bGF0ZXgyZS1PVDEtX2xlZnRhcnJvdw==",
    "bGF0ZXgyZS1PVDEtX2xlZnRoYXJwb29uZG93bg==",
    "bGF0ZXgyZS1PVDEtX2xlZnRoYXJwb29udXA=",
    "bGF0ZXgyZS1PVDEtX2xlZnRyaWdodGFycm93",
    "bGF0ZXgyZS1PVDEtX2xmbG9vcg==",
    "bGF0ZXgyZS1PVDEtX2xvbmdsZWZ0YXJyb3c=",
    "bGF0ZXgyZS1PVDEtX2xvbmdsZWZ0cmlnaHRhcnJvdw==",
    "bGF0ZXgyZS1PVDEtX2xvbmdtYXBzdG8=",
    "bGF0ZXgyZS1PVDEtX2xvbmdyaWdodGFycm93",
    "bGF0ZXgyZS1PVDEtX30=",
    "bGF0ZXgyZS1PVDEtX3BhcmFsbGVs",
    "bGF0ZXgyZS1PVDEtX3BhcnRpYWw=",
    "bGF0ZXgyZS1PVDEtX3BlcnA=",
    "bGF0ZXgyZS1PVDEtX3BoaQ==",
    "bGF0ZXgyZS1PVDEtX3Bp",
    "bGF0ZXgyZS1PVDEtX3Bt",
    "bGF0ZXgyZS1PVDEtX3BvdW5kcw==",
    "bGF0ZXgyZS1PVDEtX3ByZWM=",
    "bGF0ZXgyZS1PVDEtX3ByZWNlcQ==",
    "bGF0ZXgyZS1PVDEtX3ByaW1l",
    "bGF0ZXgyZS1PVDEtX3Byb2Q=",
    "bGF0ZXgyZS1PVDEtX3Byb3B0bw==",
    "bGF0ZXgyZS1PVDEtX3BzaQ==",
    "bGF0ZXgyZS1PVDEtX3JhbmdsZQ==",
    "bGF0ZXgyZS1PVDEtX3JjZWls",
    "bGF0ZXgyZS1PVDEtX3JmbG9vcg==",
    "bGF0ZXgyZS1PVDEtX3Jobw==",
    "bGF0ZXgyZS1PVDEtX3JpZ2h0YXJyb3c=",
    "bGF0ZXgyZS1PVDEtX3JpZ2h0aGFycG9vbmRvd24=",
    "bGF0ZXgyZS1PVDEtX3JpZ2h0aGFycG9vbnVw",
    "bGF0ZXgyZS1PVDEtX3JpZ2h0bGVmdGhhcnBvb25z",
    "bGF0ZXgyZS1PVDEtX3N0YXI=",
    "bGF0ZXgyZS1PVDEtX3N1Y2M=",
    "bGF0ZXgyZS1PVDEtX3N1Y2NlcQ==",
    "bGF0ZXgyZS1PVDEtX3N1YnNldA==",
    "bGF0ZXgyZS1PVDEtX3N1YnNldGVx",
    "bGF0ZXgyZS1PVDEtX3N1bQ==",
    "bGF0ZXgyZS1PVDEtX3N1cHNldA==",
    "bGF0ZXgyZS1PVDEtX3N1cHNldGVx",
    "bGF0ZXgyZS1PVDEtX3N1cmQ=",
    "bGF0ZXgyZS1PVDEtX3N3YXJyb3c=",
    "bGF0ZXgyZS1PVDEtX3NlYXJyb3c=",
    "bGF0ZXgyZS1PVDEtX3NldG1pbnVz",
    "bGF0ZXgyZS1PVDEtX3NoYXJw",
    "bGF0ZXgyZS1PVDEtX3NpZ21h",
    "bGF0ZXgyZS1PVDEtX3NpbQ==",
    "bGF0ZXgyZS1PVDEtX3NpbWVx",
    "bGF0ZXgyZS1PVDEtX3NtaWxl",
    "bGF0ZXgyZS1PVDEtX3NwYWRlc3VpdA==",
    "bGF0ZXgyZS1PVDEtX3NxY2Fw",
    "bGF0ZXgyZS1PVDEtX3NxY3Vw",
    "bGF0ZXgyZS1PVDEtX3Nxc3Vic2V0ZXE=",
    "bGF0ZXgyZS1PVDEtX3Nxc3Vwc2V0ZXE=",
    "bGF0ZXgyZS1PVDEtX3NxcnR7fQ==",
    "bGF0ZXgyZS1PVDEtX3Nz",
    "bGF0ZXgyZS1PVDEtX3RhdQ==",
    "bGF0ZXgyZS1PVDEtX3RleHR0cmFkZW1hcms=",
    "bGF0ZXgyZS1PVDEtX3RleHR1bmRlcnNjb3Jl",
    "bGF0ZXgyZS1PVDEtX3RleHR2aXNpYmxlc3BhY2U=",
    "bGF0ZXgyZS1PVDEtX3RleHRhc2NpaWNpcmN1bQ==",
    "bGF0ZXgyZS1PVDEtX3RleHRhc2NpaXRpbGRl",
    "bGF0ZXgyZS1PVDEtX3RleHRhc3Rlcmlza2NlbnRlcmVk",
    "bGF0ZXgyZS1PVDEtX3RleHRiYWNrc2xhc2g=",
    "bGF0ZXgyZS1PVDEtX3RleHRiYXI=",
    "bGF0ZXgyZS1PVDEtX3RleHRicmFjZWxlZnQ=",
    "bGF0ZXgyZS1PVDEtX3RleHRicmFjZXJpZ2h0",
    "bGF0ZXgyZS1PVDEtX3RleHRidWxsZXQ=",
    "bGF0ZXgyZS1PVDEtX3RleHRjb3B5cmlnaHQ=",
    "bGF0ZXgyZS1PVDEtX3RleHRkYWdnZXI=",
    "bGF0ZXgyZS1PVDEtX3RleHRkYWdnZXJkYmw=",
    "bGF0ZXgyZS1PVDEtX3RleHRkb2xsYXI=",
    "bGF0ZXgyZS1PVDEtX3RleHRlbGxpcHNpcw==",
    "bGF0ZXgyZS1PVDEtX3RleHRlbWRhc2g=",
    "bGF0ZXgyZS1PVDEtX3RleHRlbmRhc2g=",
    "bGF0ZXgyZS1PVDEtX3RleHRleGNsYW1kb3du",
    "bGF0ZXgyZS1PVDEtX3RleHRncmVhdGVy",
    "bGF0ZXgyZS1PVDEtX3RleHRsZXNz",
    "bGF0ZXgyZS1PVDEtX3RleHRvcmRmZW1pbmluZQ==",
    "bGF0ZXgyZS1PVDEtX3RleHRvcmRtYXNjdWxpbmU=",
    "bGF0ZXgyZS1PVDEtX3RleHRwYXJhZ3JhcGg=",
    "bGF0ZXgyZS1PVDEtX3RleHRwZXJpb2RjZW50ZXJlZA==",
    "bGF0ZXgyZS1PVDEtX3RleHRxdW90ZWRibGxlZnQ=",
    "bGF0ZXgyZS1PVDEtX3RleHRxdW90ZWRibHJpZ2h0",
    "bGF0ZXgyZS1PVDEtX3RleHRxdW90ZWxlZnQ=",
    "bGF0ZXgyZS1PVDEtX3RleHRxdW90ZXJpZ2h0",
    "bGF0ZXgyZS1PVDEtX3RleHRxdWVzdGlvbmRvd24=",
    "bGF0ZXgyZS1PVDEtX3RleHRyZWdpc3RlcmVk",
    "bGF0ZXgyZS1PVDEtX3RleHRzZWN0aW9u",
    "bGF0ZXgyZS1PVDEtX3RleHRzdGVybGluZw==",
    "bGF0ZXgyZS1PVDEtX3RoZXRh",
    "bGF0ZXgyZS1PVDEtX3RpbWVz",
    "bGF0ZXgyZS1PVDEtX3RvcA==",
    "bGF0ZXgyZS1PVDEtX3RyaWFuZ2xl",
    "bGF0ZXgyZS1PVDEtX3RyaWFuZ2xlbGVmdA==",
    "bGF0ZXgyZS1PVDEtX3RyaWFuZ2xlcmlnaHQ=",
    "bGF0ZXgyZS1PVDEtX3VwYXJyb3c=",
    "bGF0ZXgyZS1PVDEtX3VwZG93bmFycm93",
    "bGF0ZXgyZS1PVDEtX3VwbHVz",
    "bGF0ZXgyZS1PVDEtX3Vwc2lsb24=",
    "bGF0ZXgyZS1PVDEtX3ZhcmVwc2lsb24=",
    "bGF0ZXgyZS1PVDEtX3ZhcnBoaQ==",
    "bGF0ZXgyZS1PVDEtX3ZhcnBp",
    "bGF0ZXgyZS1PVDEtX3ZhcnJobw==",
    "bGF0ZXgyZS1PVDEtX3ZhcnNpZ21h",
    "bGF0ZXgyZS1PVDEtX3ZhcnRoZXRh",
    "bGF0ZXgyZS1PVDEtX3ZkYXNo",
    "bGF0ZXgyZS1PVDEtX3Zkb3Rz",
    "bGF0ZXgyZS1PVDEtX3ZlZQ==",
    "bGF0ZXgyZS1PVDEtX3dlZGdl",
    "bGF0ZXgyZS1PVDEtX3dw",
    "bGF0ZXgyZS1PVDEtX3dy",
    "bGF0ZXgyZS1PVDEtX3hp",
    "bGF0ZXgyZS1PVDEtX3pldGE=",
    "bGF0ZXgyZS1PVDEtX3s=",
    "bGF0ZXgyZS1PVDEtX3w=",
    "bGF0ZXgyZS1PVDEtXQ==",
    "bGF0ZXgyZS1PVDEtXy0t",
    "bGF0ZXgyZS1PVDEtXy0tLQ==",
    "bGF0ZXgyZS1PVDEtXy0tLS0=",
    "bGF0ZXgyZS1PVDEtXyM=",
    "bGF0ZXgyZS1PVDEtXyQ=",
    "bGF0ZXgyZS1PVDEtXyU=",
    "bGF0ZXgyZS1PVDEtXyY=",
    "bGF0ZXgyZS1PVDEtfA==",
    "bGF0ZXgyZS1UMS1fREg=",
    "bGF0ZXgyZS1UMS1fREo=",
    "bGF0ZXgyZS1UMS1fTkc=",
    "bGF0ZXgyZS1UMS1fVEg=",
    "bGF0ZXgyZS1UMS1fZ3VpbGxlbW90bGVmdA==",
    "bGF0ZXgyZS1UMS1fZ3VpbGxlbW90cmlnaHQ=",
    "bGF0ZXgyZS1UMS1fZ3VpbHNpbmdsbGVmdA==",
    "bGF0ZXgyZS1UMS1fZ3VpbHNpbmdscmlnaHQ=",
    "bGF0ZXgyZS1UMS1fZGg=",
    "bGF0ZXgyZS1UMS1fZGo=",
    "bGF0ZXgyZS1UMS1fbmc=",
    "bGF0ZXgyZS1UMS1fcXVvdGVkYmxiYXNl",
    "bGF0ZXgyZS1UMS1fcXVvdGVzaW5nbGJhc2U=",
    "bGF0ZXgyZS1UMS1fdGV4dHF1b3RlZGJs",
    "bGF0ZXgyZS1UMS1fdGg=",
    "bGF0ZXhzeW0tT1QxLV9sZWFkc3Rv",
    "bWF0aGRvdHMtT1QxLV9pZGRvdHM=",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye019",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye059",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye099",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0F9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0J9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0N9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0R9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0V9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0Z9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0d9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0h9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0l9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0p9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0t9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye0x9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1B9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1F9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1J9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1N9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1R9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1V9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1Z9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1d9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1h9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1l9",
    "bWF0aHJzZnMtT1QxLV9tYXRoc2Nye1p9",
    "bWFydm9zeW0tT1QxLV9Bbmto",
    "bWFydm9zeW0tT1QxLV9BcXVhcml1cw==",
    "bWFydm9zeW0tT1QxLV9Bcmllcw==",
    "bWFydm9zeW0tT1QxLV9CYXQ=",
    "bWFydm9zeW0tT1QxLV9DYW5jZXI=",
    "bWFydm9zeW0tT1QxLV9DYXByaWNvcm4=",
    "bWFydm9zeW0tT1QxLV9DZWx0Y3Jvc3M=",
    "bWFydm9zeW0tT1QxLV9DaXJjbGVkQQ==",
    "bWFydm9zeW0tT1QxLV9Dcm9zcw==",
    "bWFydm9zeW0tT1QxLV9EZW5hcml1cw==",
    "bWFydm9zeW0tT1QxLV9FVVI=",
    "bWFydm9zeW0tT1QxLV9FVVJ0bQ==",
    "bWFydm9zeW0tT1QxLV9FVVJjcg==",
    "bWFydm9zeW0tT1QxLV9FVVJkaWc=",
    "bWFydm9zeW0tT1QxLV9FVVJodg==",
    "bWFydm9zeW0tT1QxLV9FY29tbWVyY2U=",
    "bWFydm9zeW0tT1QxLV9FYXJ0aA==",
    "bWFydm9zeW0tT1QxLV9FbWFpbA==",
    "bWFydm9zeW0tT1QxLV9FbWFpbGN0",
    "bWFydm9zeW0tT1QxLV9FeWVzRG9sbGFy",
    "bWFydm9zeW0tT1QxLV9GQVg=",
    "bWFydm9zeW0tT1QxLV9GYXhtYWNoaW5l",
    "bWFydm9zeW0tT1QxLV9Gcm93bnk=",
    "bWFydm9zeW0tT1QxLV9HZW1pbmk=",
    "bWFydm9zeW0tT1QxLV9IZWFydA==",
    "bWFydm9zeW0tT1QxLV9KdXBpdGVy",
    "bWFydm9zeW0tT1QxLV9MZW8=",
    "bWFydm9zeW0tT1QxLV9MZXR0ZXI=",
    "bWFydm9zeW0tT1QxLV9MaWJyYQ==",
    "bWFydm9zeW0tT1QxLV9MaWdodG5pbmc=",
    "bWFydm9zeW0tT1QxLV9NVkF0",
    "bWFydm9zeW0tT1QxLV9NVlJpZ2h0YXJyb3c=",
    "bWFydm9zeW0tT1QxLV9NYXJz",
    "bWFydm9zeW0tT1QxLV9NZXJjdXJ5",
    "bWFydm9zeW0tT1QxLV9Nb29u",
    "bWFydm9zeW0tT1QxLV9Nb2JpbGVmb25l",
    "bWFydm9zeW0tT1QxLV9NdW5kdXM=",
    "bWFydm9zeW0tT1QxLV9OZXB0dW5l",
    "bWFydm9zeW0tT1QxLV9QZnVuZA==",
    "bWFydm9zeW0tT1QxLV9QaWNrdXA=",
    "bWFydm9zeW0tT1QxLV9QaXNjZXM=",
    "bWFydm9zeW0tT1QxLV9QbHV0bw==",
    "bWFydm9zeW0tT1QxLV9TY29ycGlv",
    "bWFydm9zeW0tT1QxLV9TYWdpdHRhcml1cw==",
    "bWFydm9zeW0tT1QxLV9TYXR1cm4=",
    "bWFydm9zeW0tT1QxLV9TaGlsbGluZw==",
    "bWFydm9zeW0tT1QxLV9TbWlsZXk=",
    "bWFydm9zeW0tT1QxLV9TdW4=",
    "bWFydm9zeW0tT1QxLV9UYXVydXM=",
    "bWFydm9zeW0tT1QxLV9UZWxlZm9u",
    "bWFydm9zeW0tT1QxLV9VcmFudXM=",
    "bWFydm9zeW0tT1QxLV9WZW51cw==",
    "bWFydm9zeW0tT1QxLV9WaXJnbw==",
    "bWFydm9zeW0tT1QxLV9ZaW55YW5n",
    "bWFydm9zeW0tT1QxLV9mYXg=",
    "c2t1bGwtT1QxLV9za3VsbA==",
    "c3RtYXJ5cmQtT1QxLV9MYmFn",
    "c3RtYXJ5cmQtT1QxLV9Mb25nbWFwc2Zyb20=",
    "c3RtYXJ5cmQtT1QxLV9Mb25nbWFwc3Rv",
    "c3RtYXJ5cmQtT1QxLV9NYXBzZnJvbQ==",
    "c3RtYXJ5cmQtT1QxLV9NYXBzdG8=",
    "c3RtYXJ5cmQtT1QxLV9SYmFn",
    "c3RtYXJ5cmQtT1QxLV9pbnBsdXM=",
    "c3RtYXJ5cmQtT1QxLV9sYmFn",
    "c3RtYXJ5cmQtT1QxLV9sZWZ0YXJyb3d0cmlhbmdsZQ==",
    "c3RtYXJ5cmQtT1QxLV9sZWZ0cmlnaHRhcnJvd2Vx",
    "c3RtYXJ5cmQtT1QxLV9sZWZ0cmlnaHRhcnJvd3RyaWFuZ2xl",
    "c3RtYXJ5cmQtT1QxLV9saWdodG5pbmc=",
    "c3RtYXJ5cmQtT1QxLV9sb25nbWFwc2Zyb20=",
    "c3RtYXJ5cmQtT1QxLV9sbGJyYWNrZXQ=",
    "c3RtYXJ5cmQtT1QxLV9sbGNlaWw=",
    "c3RtYXJ5cmQtT1QxLV9sbGZsb29y",
    "c3RtYXJ5cmQtT1QxLV9sbHBhcmVudGhlc2lz",
    "c3RtYXJ5cmQtT1QxLV9tYXBzZnJvbQ==",
    "c3RtYXJ5cmQtT1QxLV9uaXBsdXM=",
    "c3RtYXJ5cmQtT1QxLV9ubmVhcnJvdw==",
    "c3RtYXJ5cmQtT1QxLV9ubndhcnJvdw==",
    "c3RtYXJ5cmQtT1QxLV9yYmFn",
    "c3RtYXJ5cmQtT1QxLV9yaWdodGFycm93dHJpYW5nbGU=",
    "c3RtYXJ5cmQtT1QxLV9ycmJyYWNrZXQ=",
    "c3RtYXJ5cmQtT1QxLV9ycmNlaWw=",
    "c3RtYXJ5cmQtT1QxLV9ycmZsb29y",
    "c3RtYXJ5cmQtT1QxLV9ycnBhcmVudGhlc2lz",
    "c3RtYXJ5cmQtT1QxLV9zaG9ydGRvd25hcnJvdw==",
    "c3RtYXJ5cmQtT1QxLV9zaG9ydGxlZnRhcnJvdw==",
    "c3RtYXJ5cmQtT1QxLV9zaG9ydHJpZ2h0YXJyb3c=",
    "c3RtYXJ5cmQtT1QxLV9zaG9ydHVwYXJyb3c=",
    "c3RtYXJ5cmQtT1QxLV9zc2VhcnJvdw==",
    "c3RtYXJ5cmQtT1QxLV9zc3dhcnJvdw==",
    "d2FzeXN5bS1PVDEtX0Jvd3RpZQ==",
    "d2FzeXN5bS1PVDEtX0RPV05hcnJvdw==",
    "d2FzeXN5bS1PVDEtX0xFRlRhcnJvdw==",
    "d2FzeXN5bS1PVDEtX0xIRA==",
    "d2FzeXN5bS1PVDEtX1JIRA==",
    "d2FzeXN5bS1PVDEtX1JJR0hUYXJyb3c=",
    "d2FzeXN5bS1PVDEtX1VQYXJyb3c=",
    "d2FzeXN5bS1PVDEtX21hbGU=",
    "d2FzeXN5bS1PVDEtX21hcnM=",
    "d2FzeXN5bS1PVDEtX21lcmN1cnk=",
    "d2FzeXN5bS1PVDEtX25lcHR1bmU=",
    "d2FzeXN5bS1PVDEtX25ld21vb24=",
    "d2FzeXN5bS1PVDEtX29jaXJjbGU=",
    "d2FzeXN5bS1PVDEtX29wcG9zaXRpb24=",
    "d2FzeXN5bS1PVDEtX2F0YXJpYm94",
    "d2FzeXN5bS1PVDEtX2FxdWFyaXVz",
    "d2FzeXN5bS1PVDEtX2FyaWVz",
    "d2FzeXN5bS1PVDEtX2FzY25vZGU=",
    "d2FzeXN5bS1PVDEtX2FzdHJvc3Vu",
    "d2FzeXN5bS1PVDEtX2JlbGw=",
    "d2FzeXN5bS1PVDEtX2JsYWNrc21pbGV5",
    "d2FzeXN5bS1PVDEtX2Jyb2tlbnZlcnQ=",
    "d2FzeXN5bS1PVDEtX2N1cnJlbmN5",
    "d2FzeXN5bS1PVDEtX2NhbmNlcg==",
    "d2FzeXN5bS1PVDEtX2NhcHJpY29ybnVz",
    "d2FzeXN5bS1PVDEtX2NlbnQ=",
    "d2FzeXN5bS1PVDEtX2NoZWNrZWQ=",
    "d2FzeXN5bS1PVDEtX2Nsb2Nr",
    "d2FzeXN5bS1PVDEtX2Nvbmp1bmN0aW9u",
    "d2FzeXN5bS1PVDEtX2Rlc2Nub2Rl",
    "d2FzeXN5bS1PVDEtX2RpYW1ldGVy",
    "d2FzeXN5bS1PVDEtX2VhcnRo",
    "d2FzeXN5bS1PVDEtX2Z1bGxtb29u",
    "d2FzeXN5bS1PVDEtX2ZlbWFsZQ==",
    "d2FzeXN5bS1PVDEtX2Zyb3duaWU=",
    "d2FzeXN5bS1PVDEtX2dlbWluaQ==",
    "d2FzeXN5bS1PVDEtX2ludmRpYW1ldGVy",
    "d2FzeXN5bS1PVDEtX2p1cGl0ZXI=",
    "d2FzeXN5bS1PVDEtX2tyZXV6",
    "d2FzeXN5bS1PVDEtX2xlZnRtb29u",
    "d2FzeXN5bS1PVDEtX2xlbw==",
    "d2FzeXN5bS1PVDEtX2xoZA==",
    "d2FzeXN5bS1PVDEtX2xpYnJh",
    "d2FzeXN5bS1PVDEtX2xpZ2h0bmluZw==",
    "d2FzeXN5bS1PVDEtX3Blcm1pbA==",
    "d2FzeXN5bS1PVDEtX3Bob25l",
    "d2FzeXN5bS1PVDEtX3Bpc2Nlcw==",
    "d2FzeXN5bS1PVDEtX3BsdXRv",
    "d2FzeXN5bS1PVDEtX3BvaW50ZXI=",
    "d2FzeXN5bS1PVDEtX3JlY29yZGVy",
    "d2FzeXN5bS1PVDEtX3JoZA==",
    "d2FzeXN5bS1PVDEtX3JpZ2h0bW9vbg==",
    "d2FzeXN5bS1PVDEtX3N1bg==",
    "d2FzeXN5bS1PVDEtX3NhZ2l0dGFyaXVz",
    "d2FzeXN5bS1PVDEtX3NhdHVybg==",
    "d2FzeXN5bS1PVDEtX3Njb3JwaW8=",
    "d2FzeXN5bS1PVDEtX3NtaWxleQ==",
    "d2FzeXN5bS1PVDEtX3RhdXJ1cw==",
    "d2FzeXN5bS1PVDEtX3VubGhk",
    "d2FzeXN5bS1PVDEtX3Vucmhk",
    "d2FzeXN5bS1PVDEtX3VyYW51cw==",
    "d2FzeXN5bS1PVDEtX3ZlbnVz",
    "d2FzeXN5bS1PVDEtX3Zlcm5hbA==",
    "d2FzeXN5bS1PVDEtX3Zpcmdv",
    "d2FzeXN5bS1PVDEtX3dhc3lsb3plbmdl",
    "dGV4dGNvbXAtT1QxLV90ZXh0Y29sb25tb25ldGFyeQ==",
    "dGV4dGNvbXAtT1QxLV90ZXh0Y29weWxlZnQ=",
    "dGV4dGNvbXAtT1QxLV90ZXh0Y29weXJpZ2h0",
    "dGV4dGNvbXAtT1QxLV90ZXh0Y2VudA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0Y2VudG9sZHN0eWxl",
    "dGV4dGNvbXAtT1QxLV90ZXh0Y2lyY2xlZFA=",
    "dGV4dGNvbXAtT1QxLV90ZXh0Y3VycmVuY3k=",
    "dGV4dGNvbXAtT1QxLV90ZXh0YXN0ZXJpc2tjZW50ZXJlZA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0YmFodA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0YmFyZGJs",
    "dGV4dGNvbXAtT1QxLV90ZXh0YmlnY2lyY2xl",
    "dGV4dGNvbXAtT1QxLV90ZXh0Ymxhbms=",
    "dGV4dGNvbXAtT1QxLV90ZXh0YnJva2VuYmFy",
    "dGV4dGNvbXAtT1QxLV90ZXh0YnVsbGV0",
    "dGV4dGNvbXAtT1QxLV90ZXh0Z3VhcmFuaQ==",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZG93bmFycm93",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZG9sbGFy",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZG9sbGFyb2xkc3R5bGU=",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZG9uZw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZGFnZ2Vy",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZGFnZ2VyZGJs",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZGJsaHlwaGVu",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZGJsaHlwaGVuY2hhcg==",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZGVncmVl",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZGl2",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZGlzY291bnQ=",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZXN0aW1hdGVk",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZXVybw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0Zmxvcmlu",
    "dGV4dGNvbXAtT1QxLV90ZXh0ZnJhY3Rpb25zb2xpZHVz",
    "dGV4dGNvbXAtT1QxLV90ZXh0aW50ZXJyb2Jhbmc=",
    "dGV4dGNvbXAtT1QxLV90ZXh0aW50ZXJyb2Jhbmdkb3du",
    "dGV4dGNvbXAtT1QxLV90ZXh0b25laGFsZg==",
    "dGV4dGNvbXAtT1QxLV90ZXh0b25lc3VwZXJpb3I=",
    "dGV4dGNvbXAtT1QxLV90ZXh0b25lcXVhcnRlcg==",
    "dGV4dGNvbXAtT1QxLV90ZXh0b3BlbmJ1bGxldA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0b3JkZmVtaW5pbmU=",
    "dGV4dGNvbXAtT1QxLV90ZXh0b3JkbWFzY3VsaW5l",
    "dGV4dGNvbXAtT1QxLV90ZXh0bG5vdA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0bGFuZ2xl",
    "dGV4dGNvbXAtT1QxLV90ZXh0bGJyYWNrZGJs",
    "dGV4dGNvbXAtT1QxLV90ZXh0bGVmdGFycm93",
    "dGV4dGNvbXAtT1QxLV90ZXh0bGlyYQ==",
    "dGV4dGNvbXAtT1QxLV90ZXh0bHF1aWxs",
    "dGV4dGNvbXAtT1QxLV90ZXh0bWludXM=",
    "dGV4dGNvbXAtT1QxLV90ZXh0bXVzaWNhbG5vdGU=",
    "dGV4dGNvbXAtT1QxLV90ZXh0bmFpcmE=",
    "dGV4dGNvbXAtT1QxLV90ZXh0bnVtZXJv",
    "dGV4dGNvbXAtT1QxLV90ZXh0c2VjdGlvbg==",
    "dGV4dGNvbXAtT1QxLV90ZXh0c2VydmljZW1hcms=",
    "dGV4dGNvbXAtT1QxLV90ZXh0c3Rlcmxpbmc=",
    "dGV4dGNvbXAtT1QxLV90ZXh0c3VyZA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0cG0=",
    "dGV4dGNvbXAtT1QxLV90ZXh0cGFyYWdyYXBo",
    "dGV4dGNvbXAtT1QxLV90ZXh0cGVyaW9kY2VudGVyZWQ=",
    "dGV4dGNvbXAtT1QxLV90ZXh0cGVydGVudGhvdXNhbmQ=",
    "dGV4dGNvbXAtT1QxLV90ZXh0cGVydGhvdXNhbmQ=",
    "dGV4dGNvbXAtT1QxLV90ZXh0cGVzbw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0cGlsY3Jvdw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0cXVvdGVzaW5nbGU=",
    "dGV4dGNvbXAtT1QxLV90ZXh0cXVvdGVzdHJhaWdodGJhc2U=",
    "dGV4dGNvbXAtT1QxLV90ZXh0cXVvdGVzdHJhaWdodGRibGJhc2U=",
    "dGV4dGNvbXAtT1QxLV90ZXh0cmFuZ2xl",
    "dGV4dGNvbXAtT1QxLV90ZXh0cmJyYWNrZGJs",
    "dGV4dGNvbXAtT1QxLV90ZXh0cmVjaXBl",
    "dGV4dGNvbXAtT1QxLV90ZXh0cmVmZXJlbmNlbWFyaw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0cmVnaXN0ZXJlZA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0cmlnaHRhcnJvdw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0cnF1aWxs",
    "dGV4dGNvbXAtT1QxLV90ZXh0d29u",
    "dGV4dGNvbXAtT1QxLV90ZXh0dGhyZWVxdWFydGVyc2VtZGFzaA==",
    "dGV4dGNvbXAtT1QxLV90ZXh0dGhyZWVxdWFydGVycw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0dGhyZWVzdXBlcmlvcg==",
    "dGV4dGNvbXAtT1QxLV90ZXh0dGlsZGVsb3c=",
    "dGV4dGNvbXAtT1QxLV90ZXh0dGltZXM=",
    "dGV4dGNvbXAtT1QxLV90ZXh0dHJhZGVtYXJr",
    "dGV4dGNvbXAtT1QxLV90ZXh0dHdlbHZldWRhc2g=",
    "dGV4dGNvbXAtT1QxLV90ZXh0dHdvc3VwZXJpb3I=",
    "dGV4dGNvbXAtT1QxLV90ZXh0dXBhcnJvdw==",
    "dGV4dGNvbXAtT1QxLV90ZXh0eWVu",
    "dGlwYS1PVDEtX3RleHR0Y3RjbGln",
    "dGlwYS1PVDEtX3RleHR0ZXNobGln",
    "dGlwYS1PVDEtX3RleHR0aG9ybg==",
    "dGlwYS1PVDEtX3RleHR0aGV0YQ==",
    "dGlwYS1PVDEtX3RleHR0b25lbGV0dGVyc3RlbQ==",
    "dGlwYS1PVDEtX3RleHR0c2xpZw==",
    "dGlwYS1PVDEtX3RleHR0dXJuY2VsaWc=",
    "dGlwYS1PVDEtX3RleHR0dXJuYQ==",
    "dGlwYS1PVDEtX3RleHR0dXJuaA==",
    "dGlwYS1PVDEtX3RleHR0dXJuaw==",
    "dGlwYS1PVDEtX3RleHR0dXJubG9uZ2xlZ3I=",
    "dGlwYS1PVDEtX3RleHR0dXJubQ==",
    "dGlwYS1PVDEtX3RleHR0dXJubXJsZWc=",
    "dGlwYS1PVDEtX3RleHR0dXJuc2NyaXB0YQ==",
    "dGlwYS1PVDEtX3RleHR0dXJucg==",
    "dGlwYS1PVDEtX3RleHR0dXJucnJ0YWls",
    "dGlwYS1PVDEtX3RleHR0dXJudA==",
    "dGlwYS1PVDEtX3RleHR0dXJudg==",
    "dGlwYS1PVDEtX3RleHR0dXJudw==",
    "dGlwYS1PVDEtX3RleHR0dXJueQ==",
    "dGlwYS1PVDEtX3RleHR1cHN0ZXA=",
    "dGlwYS1PVDEtX3RleHR1cHNpbG9u",
    "dGlwYS1PVDEtX3RleHR2ZXJ0bGluZQ==",
    "dGlwYS1PVDEtX3RleHR2aWJ5aQ==",
    "dGlwYS1PVDEtX3RleHR2aWJ5eQ==",
    "dGlwYS1PVDEtX3RleHR3eW5u",
    "dGlwYS1PVDEtX3RleHR5b2do",
    "dGlwYS1PVDEtX3RleHRPYmFyZG90bGVzc2o=",
    "dGlwYS1PVDEtX3RleHRPbHlvZ2hsaWc=",
    "dGlwYS1PVDEtX3RleHRiYWJ5Z2FtbWE=",
    "dGlwYS1PVDEtX3RleHRiYXJ1",
    "dGlwYS1PVDEtX3RleHRiYXJi",
    "dGlwYS1PVDEtX3RleHRiYXJj",
    "dGlwYS1PVDEtX3RleHRiYXJk",
    "dGlwYS1PVDEtX3RleHRiYXJkb3RsZXNzag==",
    "dGlwYS1PVDEtX3RleHRiYXJn",
    "dGlwYS1PVDEtX3RleHRiYXJnbG90c3RvcA==",
    "dGlwYS1PVDEtX3RleHRiYXJp",
    "dGlwYS1PVDEtX3RleHRiYXJs",
    "dGlwYS1PVDEtX3RleHRiYXJv",
    "dGlwYS1PVDEtX3RleHRiYXJyZXZnbG90c3RvcA==",
    "dGlwYS1PVDEtX3RleHRiZWx0bA==",
    "dGlwYS1PVDEtX3RleHRiZXRh",
    "dGlwYS1PVDEtX3RleHRidWxsc2V5ZQ==",
    "dGlwYS1PVDEtX3RleHRjZWx0cGFs",
    "dGlwYS1PVDEtX3RleHRjaGk=",
    "dGlwYS1PVDEtX3RleHRjb21tYXRhaWx6",
    "dGlwYS1PVDEtX3RleHRjb3JuZXI=",
    "dGlwYS1PVDEtX3RleHRjbG9zZW9tZWdh",
    "dGlwYS1PVDEtX3RleHRjbG9zZWVwc2lsb24=",
    "dGlwYS1PVDEtX3RleHRjbG9zZXJldmVwc2lsb24=",
    "dGlwYS1PVDEtX3RleHRjcmI=",
    "dGlwYS1PVDEtX3RleHRjcmQ=",
    "dGlwYS1PVDEtX3RleHRjcmc=",
    "dGlwYS1PVDEtX3RleHRjcmg=",
    "dGlwYS1PVDEtX3RleHRjcmludmdsb3RzdG9w",
    "dGlwYS1PVDEtX3RleHRjcmxhbWJkYQ==",
    "dGlwYS1PVDEtX3RleHRjcnR3bw==",
    "dGlwYS1PVDEtX3RleHRjdG4=",
    "dGlwYS1PVDEtX3RleHRjdGM=",
    "dGlwYS1PVDEtX3RleHRjdGQ=",
    "dGlwYS1PVDEtX3RleHRjdGRjdHpsaWc=",
    "dGlwYS1PVDEtX3RleHRjdGVzaA==",
    "dGlwYS1PVDEtX3RleHRjdGo=",
    "dGlwYS1PVDEtX3RleHRjdHQ=",
    "dGlwYS1PVDEtX3RleHRjdHRjdGNsaWc=",
    "dGlwYS1PVDEtX3RleHRjdHlvZ2g=",
    "dGlwYS1PVDEtX3RleHRjdHo=",
    "dGlwYS1PVDEtX3RleHRkY3R6bGln",
    "dGlwYS1PVDEtX3RleHRkb3VibGV2ZXJ0bGluZQ==",
    "dGlwYS1PVDEtX3RleHRkb3VibGViYXJlc2g=",
    "dGlwYS1PVDEtX3RleHRkb3VibGViYXJwaXBl",
    "dGlwYS1PVDEtX3RleHRkb3VibGViYXJzbGFzaA==",
    "dGlwYS1PVDEtX3RleHRkb3VibGVwaXBl",
    "dGlwYS1PVDEtX3RleHRkb3duc3RlcA==",
    "dGlwYS1PVDEtX3RleHRkeW9naGxpZw==",
    "dGlwYS1PVDEtX3RleHRkemxpZw==",
    "dGlwYS1PVDEtX3RleHRlc2g=",
    "dGlwYS1PVDEtX3RleHRlcHNpbG9u",
    "dGlwYS1PVDEtX3RleHRmaXNoaG9va3I=",
    "dGlwYS1PVDEtX3RleHRn",
    "dGlwYS1PVDEtX3RleHRnYW1tYQ==",
    "dGlwYS1PVDEtX3RleHRnbG90c3RvcA==",
    "dGlwYS1PVDEtX3RleHRnbG9iZmFsbA==",
    "dGlwYS1PVDEtX3RleHRnbG9icmlzZQ==",
    "dGlwYS1PVDEtX3RleHRoYWxmbGVuZ3Ro",
    "dGlwYS1PVDEtX3RleHRoYXJkc2lnbg==",
    "dGlwYS1PVDEtX3RleHRob29rdG9w",
    "dGlwYS1PVDEtX3RleHRodGI=",
    "dGlwYS1PVDEtX3RleHRodGJhcmRvdGxlc3Nq",
    "dGlwYS1PVDEtX3RleHRodGM=",
    "dGlwYS1PVDEtX3RleHRodGQ=",
    "dGlwYS1PVDEtX3RleHRodGc=",
    "dGlwYS1PVDEtX3RleHRodGg=",
    "dGlwYS1PVDEtX3RleHRodGhlbmc=",
    "dGlwYS1PVDEtX3RleHRodGs=",
    "dGlwYS1PVDEtX3RleHRodHA=",
    "dGlwYS1PVDEtX3RleHRodHE=",
    "dGlwYS1PVDEtX3RleHRodHJ0YWlsZA==",
    "dGlwYS1PVDEtX3RleHRodHNjZw==",
    "dGlwYS1PVDEtX3RleHRodHQ=",
    "dGlwYS1PVDEtX3RleHRodmxpZw==",
    "dGlwYS1PVDEtX3RleHRpb3Rh",
    "dGlwYS1PVDEtX3RleHRpbnZnbG90c3RvcA==",
    "dGlwYS1PVDEtX3RleHRpbnZzY3I=",
    "dGlwYS1PVDEtX3RleHRsYW1iZGE=",
    "dGlwYS1PVDEtX3RleHRsZW5ndGhtYXJr",
    "dGlwYS1PVDEtX3RleHRsaG9va3Q=",
    "dGlwYS1PVDEtX3RleHRsaHRsb25naQ==",
    "dGlwYS1PVDEtX3RleHRsaHRsb25neQ==",
    "dGlwYS1PVDEtX3RleHRsb25nbGVncg==",
    "dGlwYS1PVDEtX3RleHRscHRy",
    "dGlwYS1PVDEtX3RleHRsdGFpbG0=",
    "dGlwYS1PVDEtX3RleHRsdGFpbG4=",
    "dGlwYS1PVDEtX3RleHRsdGlsZGU=",
    "dGlwYS1PVDEtX3RleHRseW9naGxpZw==",
    "dGlwYS1PVDEtX3RleHRvbWVnYQ==",
    "dGlwYS1PVDEtX3RleHRvcGVuY29ybmVy",
    "dGlwYS1PVDEtX3RleHRvcGVubw==",
    "dGlwYS1PVDEtX3RleHRwYWxob29r",
    "dGlwYS1PVDEtX3RleHRwaGk=",
    "dGlwYS1PVDEtX3RleHRwaXBl",
    "dGlwYS1PVDEtX3RleHRwcmltc3RyZXNz",
    "dGlwYS1PVDEtX3RleHRyYW1zaG9ybnM=",
    "dGlwYS1PVDEtX3RleHRyYWlzZWdsb3RzdG9w",
    "dGlwYS1PVDEtX3RleHRyYWlzZXZpYnlp",
    "dGlwYS1PVDEtX3RleHRyZXZ5b2do",
    "dGlwYS1PVDEtX3RleHRyZXZhcG9zdHJvcGhl",
    "dGlwYS1PVDEtX3RleHRyZXZl",
    "dGlwYS1PVDEtX3RleHRyZXZlcHNpbG9u",
    "dGlwYS1PVDEtX3RleHRyZXZnbG90c3RvcA==",
    "dGlwYS1PVDEtX3RleHRyaG90aWNpdHk=",
    "dGlwYS1PVDEtX3RleHRyaG9va3JldmVwc2lsb24=",
    "dGlwYS1PVDEtX3RleHRyaG9va3NjaHdh",
    "dGlwYS1PVDEtX3RleHRycHRy",
    "dGlwYS1PVDEtX3RleHRydGFpbG4=",
    "dGlwYS1PVDEtX3RleHRydGFpbGQ=",
    "dGlwYS1PVDEtX3RleHRydGFpbGw=",
    "dGlwYS1PVDEtX3RleHRydGFpbHI=",
    "dGlwYS1PVDEtX3RleHRydGFpbHM=",
    "dGlwYS1PVDEtX3RleHRydGFpbHQ=",
    "dGlwYS1PVDEtX3RleHRydGFpbHo=",
    "dGlwYS1PVDEtX3RleHRydGhvb2s=",
    "dGlwYS1PVDEtX3RleHRzY24=",
    "dGlwYS1PVDEtX3RleHRzY29lbGln",
    "dGlwYS1PVDEtX3RleHRzY29tZWdh",
    "dGlwYS1PVDEtX3RleHRzY2E=",
    "dGlwYS1PVDEtX3RleHRzY2I=",
    "dGlwYS1PVDEtX3RleHRzY2U=",
    "dGlwYS1PVDEtX3RleHRzY2c=",
    "dGlwYS1PVDEtX3RleHRzY2g=",
    "dGlwYS1PVDEtX3RleHRzY2h3YQ==",
    "dGlwYS1PVDEtX3RleHRzY2k=",
    "dGlwYS1PVDEtX3RleHRzY2o=",
    "dGlwYS1PVDEtX3RleHRzY2w=",
    "dGlwYS1PVDEtX3RleHRzY3I=",
    "dGlwYS1PVDEtX3RleHRzY3JpcHR2",
    "dGlwYS1PVDEtX3RleHRzY3JpcHRh",
    "dGlwYS1PVDEtX3RleHRzY3JpcHRn",
    "dGlwYS1PVDEtX3RleHRzY3U=",
    "dGlwYS1PVDEtX3RleHRzY3k=",
    "dGlwYS1PVDEtX3RleHRzZWNzdHJlc3M=",
    "dGlwYS1PVDEtX3RleHRzb2Z0c2lnbg==",
    "dGlwYS1PVDEtX3RleHRzdHJldGNoYw==",
    "dXBncmVlay1PVDEtX1VwZ2FtbWE=",
    "dXBncmVlay1PVDEtX1VwZGVsdGE=",
    "dXBncmVlay1PVDEtX1Vwb21lZ2E=",
    "dXBncmVlay1PVDEtX1VwbGFtYmRh",
    "dXBncmVlay1PVDEtX1Vwc2lnbWE=",
    "dXBncmVlay1PVDEtX1VwcGhp",
    "dXBncmVlay1PVDEtX1VwcGk=",
    "dXBncmVlay1PVDEtX1VwcHNp",
    "dXBncmVlay1PVDEtX1VwdGhldGE=",
    "dXBncmVlay1PVDEtX1VwdXBzaWxvbg==",
    "dXBncmVlay1PVDEtX1VweGk=",
    "dXBncmVlay1PVDEtX3VwY2hp",
    "dXBncmVlay1PVDEtX3VwYWxwaGE=",
    "dXBncmVlay1PVDEtX3VwYmV0YQ==",
    "dXBncmVlay1PVDEtX3VwZ2FtbWE=",
    "dXBncmVlay1PVDEtX3VwZGVsdGE=",
    "dXBncmVlay1PVDEtX3VwZXBzaWxvbg==",
    "dXBncmVlay1PVDEtX3VwZXRh",
    "dXBncmVlay1PVDEtX3Vwa2FwcGE=",
    "dXBncmVlay1PVDEtX3VwaW90YQ==",
    "dXBncmVlay1PVDEtX3Vwb21lZ2E=",
    "dXBncmVlay1PVDEtX3VwbGFtYmRh",
    "dXBncmVlay1PVDEtX3VwbXU=",
    "dXBncmVlay1PVDEtX3VwbnU=",
    "dXBncmVlay1PVDEtX3Vwc2lnbWE=",
    "dXBncmVlay1PVDEtX3VwcGhp",
    "dXBncmVlay1PVDEtX3VwcGk=",
    "dXBncmVlay1PVDEtX3VwcHNp",
    "dXBncmVlay1PVDEtX3Vwcmhv",
    "dXBncmVlay1PVDEtX3VwdGF1",
    "dXBncmVlay1PVDEtX3VwdGhldGE=",
    "dXBncmVlay1PVDEtX3VwdXBzaWxvbg==",
    "dXBncmVlay1PVDEtX3VwdmFyZXBzaWxvbg==",
    "dXBncmVlay1PVDEtX3VwdmFyc2lnbWE=",
    "dXBncmVlay1PVDEtX3VwdmFycGhp",
    "dXBncmVlay1PVDEtX3VwdmFycGk=",
    "dXBncmVlay1PVDEtX3VwdmFycmhv",
    "dXBncmVlay1PVDEtX3VwdmFydGhldGE=",
    "dXBncmVlay1PVDEtX3VweGk=",
    "dXBncmVlay1PVDEtX3VwemV0YQ==",
];
