// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A2__F167

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_256_0_T_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1101)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xd538a7f4u, 0x99213229u, 0, 109 }, // d538a7f499213229 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1101
    { 0x0ba55e56u, 0x688c577au, 0, 6 }, // 0ba55e56688c577a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1101
    { 0x68c5c56cu, 0xa76635e4u, 0, 143 }, // 68c5c56ca76635e4 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1101
    { 0x2b7a119eu, 0x37355b28u, 0, 86 }, // 2b7a119e37355b28 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1101
    { 0xcb804964u, 0xcb816ba7u, 23, 109 }, // cb804964cb816ba7 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1101
    { 0x9ae3adb6u, 0xbe6d9004u, 23, 6 }, // 9ae3adb6be6d9004 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1101
    { 0xcf5de35eu, 0x62fc7359u, 23, 143 }, // cf5de35e62fc7359 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1101
    { 0xeab71699u, 0xa58881cdu, 63, 126 }, // eab71699a58881cd = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1101
    { 0x4f3d1f81u, 0xefbe9d2cu, 63, 69 }, // 4f3d1f81efbe9d2c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1101
    { 0xaff95a63u, 0xadf76d19u, 63, 29 }, // aff95a63adf76d19 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1101
    { 0xa98b76b2u, 0x3122d07au, 63, 46 }, // a98b76b23122d07a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1101
    { 0x75b1b584u, 0x129bdd46u, 160, 69 }, // 75b1b584129bdd46 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1101
    { 0x6c9cae74u, 0x61a62edau, 160, 46 }, // 6c9cae7461a62eda = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_fuse/FONLY__＊fp16@16_256_0_T_T_1___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1101
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx1101_mod0
{{ 5, 1, 2, 4, 5, 4, 6},
 { 7, 2, 0, 0, 0, 0, 0},
 { 8, 0, 0, 0, 0, 0, 0},
 { 8, 0, 0, 0, 0, 0, 0},
 { 7, 7, 0, 0, 0, 0, 0},
 { 9,11, 7, 0, 0, 2, 1},
 {10,12, 7, 0, 0, 3, 0}}
// End of GPU gfx1101_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A2__F167 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 256
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 256 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

