﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/drs/DrsRequest.h>
#include <aws/drs/Drs_EXPORTS.h>
#include <aws/drs/model/PITPolicyRule.h>
#include <aws/drs/model/ReplicationConfigurationDataPlaneRouting.h>
#include <aws/drs/model/ReplicationConfigurationDefaultLargeStagingDiskType.h>
#include <aws/drs/model/ReplicationConfigurationEbsEncryption.h>

#include <utility>

namespace Aws {
namespace drs {
namespace Model {

/**
 */
class CreateReplicationConfigurationTemplateRequest : public DrsRequest {
 public:
  AWS_DRS_API CreateReplicationConfigurationTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateReplicationConfigurationTemplate"; }

  AWS_DRS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Whether to associate the default Elastic Disaster Recovery Security group
   * with the Replication Configuration Template.</p>
   */
  inline bool GetAssociateDefaultSecurityGroup() const { return m_associateDefaultSecurityGroup; }
  inline bool AssociateDefaultSecurityGroupHasBeenSet() const { return m_associateDefaultSecurityGroupHasBeenSet; }
  inline void SetAssociateDefaultSecurityGroup(bool value) {
    m_associateDefaultSecurityGroupHasBeenSet = true;
    m_associateDefaultSecurityGroup = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithAssociateDefaultSecurityGroup(bool value) {
    SetAssociateDefaultSecurityGroup(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether to allow the AWS replication agent to automatically replicate newly
   * added disks.</p>
   */
  inline bool GetAutoReplicateNewDisks() const { return m_autoReplicateNewDisks; }
  inline bool AutoReplicateNewDisksHasBeenSet() const { return m_autoReplicateNewDisksHasBeenSet; }
  inline void SetAutoReplicateNewDisks(bool value) {
    m_autoReplicateNewDisksHasBeenSet = true;
    m_autoReplicateNewDisks = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithAutoReplicateNewDisks(bool value) {
    SetAutoReplicateNewDisks(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configure bandwidth throttling for the outbound data transfer rate of the
   * Source Server in Mbps.</p>
   */
  inline long long GetBandwidthThrottling() const { return m_bandwidthThrottling; }
  inline bool BandwidthThrottlingHasBeenSet() const { return m_bandwidthThrottlingHasBeenSet; }
  inline void SetBandwidthThrottling(long long value) {
    m_bandwidthThrottlingHasBeenSet = true;
    m_bandwidthThrottling = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithBandwidthThrottling(long long value) {
    SetBandwidthThrottling(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether to create a Public IP for the Recovery Instance by default.</p>
   */
  inline bool GetCreatePublicIP() const { return m_createPublicIP; }
  inline bool CreatePublicIPHasBeenSet() const { return m_createPublicIPHasBeenSet; }
  inline void SetCreatePublicIP(bool value) {
    m_createPublicIPHasBeenSet = true;
    m_createPublicIP = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithCreatePublicIP(bool value) {
    SetCreatePublicIP(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data plane routing mechanism that will be used for replication.</p>
   */
  inline ReplicationConfigurationDataPlaneRouting GetDataPlaneRouting() const { return m_dataPlaneRouting; }
  inline bool DataPlaneRoutingHasBeenSet() const { return m_dataPlaneRoutingHasBeenSet; }
  inline void SetDataPlaneRouting(ReplicationConfigurationDataPlaneRouting value) {
    m_dataPlaneRoutingHasBeenSet = true;
    m_dataPlaneRouting = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithDataPlaneRouting(ReplicationConfigurationDataPlaneRouting value) {
    SetDataPlaneRouting(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Staging Disk EBS volume type to be used during replication.</p>
   */
  inline ReplicationConfigurationDefaultLargeStagingDiskType GetDefaultLargeStagingDiskType() const {
    return m_defaultLargeStagingDiskType;
  }
  inline bool DefaultLargeStagingDiskTypeHasBeenSet() const { return m_defaultLargeStagingDiskTypeHasBeenSet; }
  inline void SetDefaultLargeStagingDiskType(ReplicationConfigurationDefaultLargeStagingDiskType value) {
    m_defaultLargeStagingDiskTypeHasBeenSet = true;
    m_defaultLargeStagingDiskType = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithDefaultLargeStagingDiskType(
      ReplicationConfigurationDefaultLargeStagingDiskType value) {
    SetDefaultLargeStagingDiskType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of EBS encryption to be used during replication.</p>
   */
  inline ReplicationConfigurationEbsEncryption GetEbsEncryption() const { return m_ebsEncryption; }
  inline bool EbsEncryptionHasBeenSet() const { return m_ebsEncryptionHasBeenSet; }
  inline void SetEbsEncryption(ReplicationConfigurationEbsEncryption value) {
    m_ebsEncryptionHasBeenSet = true;
    m_ebsEncryption = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithEbsEncryption(ReplicationConfigurationEbsEncryption value) {
    SetEbsEncryption(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the EBS encryption key to be used during replication.</p>
   */
  inline const Aws::String& GetEbsEncryptionKeyArn() const { return m_ebsEncryptionKeyArn; }
  inline bool EbsEncryptionKeyArnHasBeenSet() const { return m_ebsEncryptionKeyArnHasBeenSet; }
  template <typename EbsEncryptionKeyArnT = Aws::String>
  void SetEbsEncryptionKeyArn(EbsEncryptionKeyArnT&& value) {
    m_ebsEncryptionKeyArnHasBeenSet = true;
    m_ebsEncryptionKeyArn = std::forward<EbsEncryptionKeyArnT>(value);
  }
  template <typename EbsEncryptionKeyArnT = Aws::String>
  CreateReplicationConfigurationTemplateRequest& WithEbsEncryptionKeyArn(EbsEncryptionKeyArnT&& value) {
    SetEbsEncryptionKeyArn(std::forward<EbsEncryptionKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Point in time (PIT) policy to manage snapshots taken during
   * replication.</p>
   */
  inline const Aws::Vector<PITPolicyRule>& GetPitPolicy() const { return m_pitPolicy; }
  inline bool PitPolicyHasBeenSet() const { return m_pitPolicyHasBeenSet; }
  template <typename PitPolicyT = Aws::Vector<PITPolicyRule>>
  void SetPitPolicy(PitPolicyT&& value) {
    m_pitPolicyHasBeenSet = true;
    m_pitPolicy = std::forward<PitPolicyT>(value);
  }
  template <typename PitPolicyT = Aws::Vector<PITPolicyRule>>
  CreateReplicationConfigurationTemplateRequest& WithPitPolicy(PitPolicyT&& value) {
    SetPitPolicy(std::forward<PitPolicyT>(value));
    return *this;
  }
  template <typename PitPolicyT = PITPolicyRule>
  CreateReplicationConfigurationTemplateRequest& AddPitPolicy(PitPolicyT&& value) {
    m_pitPolicyHasBeenSet = true;
    m_pitPolicy.emplace_back(std::forward<PitPolicyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instance type to be used for the replication server.</p>
   */
  inline const Aws::String& GetReplicationServerInstanceType() const { return m_replicationServerInstanceType; }
  inline bool ReplicationServerInstanceTypeHasBeenSet() const { return m_replicationServerInstanceTypeHasBeenSet; }
  template <typename ReplicationServerInstanceTypeT = Aws::String>
  void SetReplicationServerInstanceType(ReplicationServerInstanceTypeT&& value) {
    m_replicationServerInstanceTypeHasBeenSet = true;
    m_replicationServerInstanceType = std::forward<ReplicationServerInstanceTypeT>(value);
  }
  template <typename ReplicationServerInstanceTypeT = Aws::String>
  CreateReplicationConfigurationTemplateRequest& WithReplicationServerInstanceType(ReplicationServerInstanceTypeT&& value) {
    SetReplicationServerInstanceType(std::forward<ReplicationServerInstanceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The security group IDs that will be used by the replication server.</p>
   */
  inline const Aws::Vector<Aws::String>& GetReplicationServersSecurityGroupsIDs() const { return m_replicationServersSecurityGroupsIDs; }
  inline bool ReplicationServersSecurityGroupsIDsHasBeenSet() const { return m_replicationServersSecurityGroupsIDsHasBeenSet; }
  template <typename ReplicationServersSecurityGroupsIDsT = Aws::Vector<Aws::String>>
  void SetReplicationServersSecurityGroupsIDs(ReplicationServersSecurityGroupsIDsT&& value) {
    m_replicationServersSecurityGroupsIDsHasBeenSet = true;
    m_replicationServersSecurityGroupsIDs = std::forward<ReplicationServersSecurityGroupsIDsT>(value);
  }
  template <typename ReplicationServersSecurityGroupsIDsT = Aws::Vector<Aws::String>>
  CreateReplicationConfigurationTemplateRequest& WithReplicationServersSecurityGroupsIDs(ReplicationServersSecurityGroupsIDsT&& value) {
    SetReplicationServersSecurityGroupsIDs(std::forward<ReplicationServersSecurityGroupsIDsT>(value));
    return *this;
  }
  template <typename ReplicationServersSecurityGroupsIDsT = Aws::String>
  CreateReplicationConfigurationTemplateRequest& AddReplicationServersSecurityGroupsIDs(ReplicationServersSecurityGroupsIDsT&& value) {
    m_replicationServersSecurityGroupsIDsHasBeenSet = true;
    m_replicationServersSecurityGroupsIDs.emplace_back(std::forward<ReplicationServersSecurityGroupsIDsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The subnet to be used by the replication staging area.</p>
   */
  inline const Aws::String& GetStagingAreaSubnetId() const { return m_stagingAreaSubnetId; }
  inline bool StagingAreaSubnetIdHasBeenSet() const { return m_stagingAreaSubnetIdHasBeenSet; }
  template <typename StagingAreaSubnetIdT = Aws::String>
  void SetStagingAreaSubnetId(StagingAreaSubnetIdT&& value) {
    m_stagingAreaSubnetIdHasBeenSet = true;
    m_stagingAreaSubnetId = std::forward<StagingAreaSubnetIdT>(value);
  }
  template <typename StagingAreaSubnetIdT = Aws::String>
  CreateReplicationConfigurationTemplateRequest& WithStagingAreaSubnetId(StagingAreaSubnetIdT&& value) {
    SetStagingAreaSubnetId(std::forward<StagingAreaSubnetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of tags to be associated with all resources created in the replication
   * staging area: EC2 replication server, EBS volumes, EBS snapshots, etc.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetStagingAreaTags() const { return m_stagingAreaTags; }
  inline bool StagingAreaTagsHasBeenSet() const { return m_stagingAreaTagsHasBeenSet; }
  template <typename StagingAreaTagsT = Aws::Map<Aws::String, Aws::String>>
  void SetStagingAreaTags(StagingAreaTagsT&& value) {
    m_stagingAreaTagsHasBeenSet = true;
    m_stagingAreaTags = std::forward<StagingAreaTagsT>(value);
  }
  template <typename StagingAreaTagsT = Aws::Map<Aws::String, Aws::String>>
  CreateReplicationConfigurationTemplateRequest& WithStagingAreaTags(StagingAreaTagsT&& value) {
    SetStagingAreaTags(std::forward<StagingAreaTagsT>(value));
    return *this;
  }
  template <typename StagingAreaTagsKeyT = Aws::String, typename StagingAreaTagsValueT = Aws::String>
  CreateReplicationConfigurationTemplateRequest& AddStagingAreaTags(StagingAreaTagsKeyT&& key, StagingAreaTagsValueT&& value) {
    m_stagingAreaTagsHasBeenSet = true;
    m_stagingAreaTags.emplace(std::forward<StagingAreaTagsKeyT>(key), std::forward<StagingAreaTagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of tags to be associated with the Replication Configuration Template
   * resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateReplicationConfigurationTemplateRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateReplicationConfigurationTemplateRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether to use a dedicated Replication Server in the replication staging
   * area.</p>
   */
  inline bool GetUseDedicatedReplicationServer() const { return m_useDedicatedReplicationServer; }
  inline bool UseDedicatedReplicationServerHasBeenSet() const { return m_useDedicatedReplicationServerHasBeenSet; }
  inline void SetUseDedicatedReplicationServer(bool value) {
    m_useDedicatedReplicationServerHasBeenSet = true;
    m_useDedicatedReplicationServer = value;
  }
  inline CreateReplicationConfigurationTemplateRequest& WithUseDedicatedReplicationServer(bool value) {
    SetUseDedicatedReplicationServer(value);
    return *this;
  }
  ///@}
 private:
  bool m_associateDefaultSecurityGroup{false};

  bool m_autoReplicateNewDisks{false};

  long long m_bandwidthThrottling{0};

  bool m_createPublicIP{false};

  ReplicationConfigurationDataPlaneRouting m_dataPlaneRouting{ReplicationConfigurationDataPlaneRouting::NOT_SET};

  ReplicationConfigurationDefaultLargeStagingDiskType m_defaultLargeStagingDiskType{
      ReplicationConfigurationDefaultLargeStagingDiskType::NOT_SET};

  ReplicationConfigurationEbsEncryption m_ebsEncryption{ReplicationConfigurationEbsEncryption::NOT_SET};

  Aws::String m_ebsEncryptionKeyArn;

  Aws::Vector<PITPolicyRule> m_pitPolicy;

  Aws::String m_replicationServerInstanceType;

  Aws::Vector<Aws::String> m_replicationServersSecurityGroupsIDs;

  Aws::String m_stagingAreaSubnetId;

  Aws::Map<Aws::String, Aws::String> m_stagingAreaTags;

  Aws::Map<Aws::String, Aws::String> m_tags;

  bool m_useDedicatedReplicationServer{false};
  bool m_associateDefaultSecurityGroupHasBeenSet = false;
  bool m_autoReplicateNewDisksHasBeenSet = false;
  bool m_bandwidthThrottlingHasBeenSet = false;
  bool m_createPublicIPHasBeenSet = false;
  bool m_dataPlaneRoutingHasBeenSet = false;
  bool m_defaultLargeStagingDiskTypeHasBeenSet = false;
  bool m_ebsEncryptionHasBeenSet = false;
  bool m_ebsEncryptionKeyArnHasBeenSet = false;
  bool m_pitPolicyHasBeenSet = false;
  bool m_replicationServerInstanceTypeHasBeenSet = false;
  bool m_replicationServersSecurityGroupsIDsHasBeenSet = false;
  bool m_stagingAreaSubnetIdHasBeenSet = false;
  bool m_stagingAreaTagsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_useDedicatedReplicationServerHasBeenSet = false;
};

}  // namespace Model
}  // namespace drs
}  // namespace Aws
