/**
 *    Copyright (C) 2023-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include <boost/optional.hpp>
#include <compare>
#include <string>
#include <type_traits>
#include <variant>

#include "mongo/base/string_data.h"
#include "mongo/bson/bsonelement.h"
#include "mongo/bson/bsonobj.h"
#include "mongo/bson/bsonobjbuilder.h"
#include "mongo/bson/util/builder_fwd.h"

namespace mongo {
using IndexKeyPattern = BSONObj;
using IndexName = std::string;

/**
 * Struct representing a $natural hint. The natural sort hint $natural is used to:
 * - include collection scan into a list of valid plans generated by the optimizer, which will
 * return the documents of the collection in natural order.
 * - specify a natural sort when running a find operation against a view.
 */
struct NaturalOrderHint {
    enum class Direction : int8_t {
        kForward = 1,
        kBackward = -1,
    };

    explicit NaturalOrderHint(Direction direction) : direction(direction) {}

    auto operator<=>(const NaturalOrderHint& other) const = default;

    Direction direction;
};

namespace detail {
// Concept to avoid ambiguity with BSON builder types which implement differing streaming operators.
template <class T>
concept string_builder = std::is_same_v<T, StringBuilder> || std::is_same_v<T, StackStringBuilder>;
}  // namespace detail

auto& operator<<(detail::string_builder auto& os, NaturalOrderHint::Direction dir) {
    switch (dir) {
        case NaturalOrderHint::Direction::kForward:
            os << "forward";
            return os;
        case NaturalOrderHint::Direction::kBackward:
            os << "backward";
            return os;
    }
    MONGO_UNREACHABLE;
}

std::string toString(NaturalOrderHint::Direction dir);
bool isForward(NaturalOrderHint::Direction dir);

/**
 * Class represents all possible index hint definitions. Index hint may be specified as:
 * - Index key pattern, defined as a BSONObj
 * - Index name
 * - $natural hint
 */
class IndexHint {
public:
    explicit IndexHint(IndexKeyPattern keyPattern) : _hint(keyPattern) {}
    explicit IndexHint(IndexName indexName) : _hint(indexName) {}
    explicit IndexHint(NaturalOrderHint hint) : _hint(hint) {}

    static IndexHint parse(const BSONElement& element);
    static void append(const IndexHint& hint, StringData fieldName, BSONObjBuilder* builder);

    void append(BSONArrayBuilder* builder) const;

    const std::variant<IndexKeyPattern, IndexName, NaturalOrderHint>& getHint() const {
        return _hint;
    }
    boost::optional<const IndexKeyPattern&> getIndexKeyPattern() const;
    boost::optional<const IndexName&> getIndexName() const;
    boost::optional<const NaturalOrderHint&> getNaturalHint() const;

    std::strong_ordering operator<=>(const IndexHint& other) const;
    bool operator==(const IndexHint& other) const;
    bool operator!=(const IndexHint& other) const = default;

    size_t hash() const;

private:
    std::variant<IndexKeyPattern, IndexName, NaturalOrderHint> _hint;
};

// Helper for boost::hash.
size_t hash_value(const IndexHint& hint);

}  // namespace mongo
