/***************************************************************************
 *   Copyright (C) 2005 by Ken Werner                                      *
 *   ken.werner@web.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef KIMA_H
#define KIMA_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <kpanelapplet.h>
#include "sources/source.h"
#include "cpufreqd/cpufreqd.h"

// Qt
#include <qtooltip.h>
#include <qfont.h>
#include <qcolor.h>
#include <qptrlist.h>
#include <qlabel.h>

// Forward Declarations
class FlowLayout;
class KAboutApplication;
class KPopupMenu;
class KSelectAction;
class QMouseEvent;
class KDialogBase;
class Prefs;
class QListViewItem;
class KConfig;
class QLayoutItem;

/**
@author Ken Werner
*/
class Kima : public KPanelApplet, QToolTip {
	Q_OBJECT //macro which activates signals and slots (moc)

public:
	/**
	 * Construct a @ref KPanelApplet just like any other widget.
	 *
	 * @param configFile The configFile handed over in the factory function.
	 * @param Type The applet @ref type().
	 * @param actions Standard RMB menu actions supported by the applet (see @ref action() ).
	 * @param parent The pointer to the parent widget handed over in the factory function.
	 * @param name A Qt object name for your applet.
	 **/
	Kima(const QString& inConfigFile, Type inType = Normal, int inActions = 0,
	           QWidget* inParent = 0, const char* inName = 0);
	/** destructor */
	virtual ~Kima();

	/**
	 * Retrieve a suggested width for a given height.
	 *
	 * Every applet should reimplement this function.
	 *
	 * Depending on the panel orientation the height (horizontal panel) or the
	 * width (vertical panel) of the applets is fixed.
	 * The exact values of the fixed size component depend on the panel size.
	 *
	 * On a horizontal panel the applet height is fixed, the panel will
	 * call @ref widthForHeight(int height) with @p height
	 * equal to 'the fixed applet height'
	 * when laying out the applets.
	 *
	 * The applet can now choose the other size component (width)
	 * based on the given height.
	 *
	 * The width you return is granted.
	 **/
	virtual int widthForHeight(int inHeight) const;

	/**
	 * @return A suggested height for a given width.
	 *
	 * Every applet should reimplement this function.
	 *
	 * Depending on the panel orientation the height (horizontal panel) or the
	 * width (vertical panel) of the applets is fixed.
	 * The exact values of the fixed size component depend on the panel size.
	 *
	 * On a vertical panel the applet width is fixed, the panel will
	 * call @ref heightForWidth(int width) with @p width
	 * equal to 'the fixed applet width'
	 * when laying out the applets.
	 *
	 * The applet can now choose the other size component (height)
	 * based on the given width.
	 *
	 * The height you return is granted.
	 **/
	virtual int heightForWidth(int inWidth) const;

public slots:
	/**
	 * called if the preferences dialog is cancelled
	 **/
	void cancelPreferences();

	/**
	 * Saves the preferences
	 **/
	void savePreferences();

	/**
	 * Is called when the user selects "About" from the applets RMB menu.
	 * Reimplement this function to launch a about dialog.
	 *
	 * Note that this is called only when your applet supports the About action.
	 * See @ref Action and @ref KPanelApplet().
	 **/
	virtual void about();

	/**
	 * Is called when the user selects "Help" from the applets RMB menu.
	 * Reimplement this function to launch a manual or help page.
	 *
	 * Note that this is called only when your applet supports the Help action.
	 * See @ref Action and @ref KPanelApplet().
	 **/
	virtual void help();

	/**
	 * Is called when the user selects "Preferences" from the applets RMB menu.
	 * Reimplement this function to launch a preferences dialog or kcontrol module.
	 *
	 * Note that this is called only when your applet supports the preferences action.
	 * See @ref Action and @ref KPanelApplet().
	 **/
	virtual void preferences();

	/**
	 * raises the preferences widgets of the given list item
	 */	
	void raiseSourcePrefsWidget(QListViewItem* inListViewItem);

	/**
	 * adds or removes the given source from the display (layout) of the kicker applet
	 */	
	void displaySource(bool inDisplay, Source* inSource);

protected:
	void paintEvent(QPaintEvent* inEvent);
	void mousePressEvent(QMouseEvent* inEvent);
	void mouseMoveEvent(QMouseEvent* inEvent);
	void mouseReleaseEvent(QMouseEvent* inEvent);
	void maybeTip(const QPoint& inPos);
	void registerSource(Source* source);

protected slots:
	void updateSourceWidgets();

private:
	/**
	 * Adds the entries of given the STL to mSources
	 */
	void addSources(const std::list<Source*>& inList);

	/**
	 * Searches for the Source that has inWidget
	 * Returns the found Soucre, otherwise NULL
	 */
	Source* findSource(QWidget* inWidget);
	/**
	 * Searches for the Source that has inPosition
	 * Returns the found Soucre, otherwise NULL
	 */
	Source* findSource(int inPosition);

	KConfig* mKConfig;
	FlowLayout* mLayout;
	mutable int mCachedWFH;
	mutable int mCachedHeight;
	KAboutApplication* mAboutDialog;
	KPopupMenu* mMenu;
	KDialogBase* mPrefsDlg;
	Prefs* mPrefs;

	/**
	 * stores sources
	 */
	QPtrList<Source> mSources;

	/**
	 * points to the source which is currently
	 * dragged, otherwise NULL
	 */
	QLayoutItem* mDraggedSourceItem;

	/**
	 * 'drag' threshold multiplier
	 */
	double mDragFactor;

	/**
	 * contains the cpufreq support
	 */
	CPUFreqd mCpufreqd;
};

#endif // KIMA_H
