/***************************************************************************
 *   Copyright (C) 2007 by Johannes Schaub                                 *
 *   schaub-johannes@web.de                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "cpuusagesrc.h"
#include <qtextstream.h> 
#include <qfile.h>
#include <qstringlist.h>
#include <klocale.h>
#include <algorithm>
#include <cstring>
#include <cstdio>

CpuUsageSrc::CpuUsageSrc(QWidget* inParent, const QFile& inSourceFile, int index):
        LabelSource(inParent),
        mSourceFile(inSourceFile.name()), mIndex(index),
        mTrigger(this) {
	mID = QString("CPUUsage%1").arg(index);
    if(index == 0) {
        mName = "CPU Usage";
        mDescription = i18n("This source displays the CPU usage.");
        mPattern = "cpu %u %u %u %u %u %u %u";
    } else {
        mName = QString("CPU%1 Usage").arg(index);
        mDescription = i18n("This source displays the CPU usage of CPU%1.").arg(index);
        mPattern = QString("cpu%1 %u %u %u %u %u %u %u").arg(index-1);
        mEnabled = false; // disabled by default
    }
    std::memset(mLastUsage, 0, sizeof(mLastUsage));
}

CpuUsageSrc::~CpuUsageSrc(){
}

std::list<Source*>CpuUsageSrc::createInstances(QWidget* inParent){
	std::list<Source*> list;
	QFile cpuUsageFile( "/proc/stat" );
	if(cpuUsageFile.open(IO_ReadOnly)) {
        int cpuCount;
        QString line;
        for(cpuCount = 0; cpuUsageFile.readLine( line, 255 ) != -1; cpuCount++)
            if(!line.startsWith("cpu"))
                break;
        // cpu and cpu0 are the same if there are no more cpuX lines
        cpuCount = cpuCount == 2 ? 1 : cpuCount;
        while(--cpuCount >= 0)
            list.push_back(new CpuUsageSrc(inParent, cpuUsageFile, cpuCount));
    }
	return list;
}

QString CpuUsageSrc::fetchValue(){
	QString s = "n/a";
	if(mSourceFile.open(IO_ReadOnly)){
        for(int i=0; i<=mIndex; i++)
            mSourceFile.readLine( s, 255 );
 
        // read the cu usage since the cpu has started
        unsigned int currentUsage[7];
        if( std::sscanf(s.ascii(), mPattern.ascii(),
                   &currentUsage[0], &currentUsage[1], &currentUsage[2],
                   &currentUsage[3], &currentUsage[4], &currentUsage[5],
                   &currentUsage[6]) != 7 ) {
            mSourceFile.close();
            return "n/a";
        }
        // calculate the difference between this and the last fetch
        unsigned int relativeUsage = 0;
        for(int i=0; i<7; i++)
            relativeUsage += currentUsage[i]-mLastUsage[i];
        
        // currentUsage[3] is the idle usage.
        s = QString("%1 %").arg(100 - (currentUsage[3]-mLastUsage[3])*100/relativeUsage);
        std::memcpy(mLastUsage, currentUsage, sizeof(currentUsage));
		mSourceFile.close();
	}
	return s;
}
