/***************************************************************************
 *   Copyright (C) 2007 by the KSmoothDock team   *
 *   dangvd@yahoo.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/// KSmoothDock main class. VERSION: 4.5

#include "ksmoothdock.h"

#include <iostream>
#include <iterator>

#include <qdir.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdialogbase.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kwin.h>
#include <dcopclient.h>

#include "aboutdialog.h"
#include "buttonisedstate.h"
#include "clock.h"
#include "configdialog.h"
#include "desktopselector.h"
#include "dockstate.h"
#include "launcher.h"
#include "launcherconfigdialog.h"
#include "normalzoomstate.h"
#include "paraboliczoomstate.h"
#include "separator.h"
#include "task.h"
#include "welcomedialog.h"

/// PUBLIC ///

/** 
 * Constructor
 */

KSmoothDock::KSmoothDock(bool resetConfig, bool resetLaunchers) : QWidget(0, "KSmoothDock", WStyle_Customize | WStyle_NoBorder | WNoAutoErase) {
    // init the dock
    init();

    // load configuration
    loadConfig(resetConfig);

    // initiate variables
    initVariables();
    
    // load quick launchers
    loadLaunchers(resetLaunchers);

    // initiate the pager
    initPager();

    // load the running tasks
    loadTasks();

    // init clock
    initClock();

    // update layout
    updateLayout();

    // check kicker position
    checkKickerPosition();
}

/** 
 * Destructor
 */

KSmoothDock::~KSmoothDock() {
    // free resources
    for (unsigned int i = 0; i < m_items.size(); i++) {
        delete m_items[i];
    }
    delete m_wallpaperManager;

    if (m_normalZoomState != NULL)
        delete m_normalZoomState;
    if (m_parabolicZoomState != NULL)
        delete m_parabolicZoomState;
    if (m_buttonisedState != NULL)
        delete m_buttonisedState;
}

/**
 * Get the current active window
 */

WId KSmoothDock::activeWindow() {
    return m_windowManager.activeWindow();
}

/**
 * Get the position of the dock
 */

PanelPosition KSmoothDock::getPosition() {
    return m_position;
}

/**
 * Get the zoom mode of the dock
 */

ZoomMode KSmoothDock::getZoomMode() {
    return m_zoomMode;
}

/**
 * Get active desktop color
 */

QColor KSmoothDock::getActiveDesktopColor() {
    return m_activeDesktopColor;
}

/**
 * Get inactive desktop color
 */

QColor KSmoothDock::getInactiveDesktopColor() {
    return m_inactiveDesktopColor;
}

/**
 * Get separator color
 */

QColor KSmoothDock::getSeparatorColor() {
    return m_separatorColor;
}

/**
 * Get clock font face
 */

QString KSmoothDock::getClockFontFace() {
    return m_clockFontFace;
}

/**
 * Get small icon size
 */

int KSmoothDock::getSmallIconSize() {
    return m_smallIconSize;
}

/**
 * Update tooltip
 */

void KSmoothDock::updateTooltip() {
    if (m_showTooltip) {
        m_tooltip.setFontFace(m_tooltipFontFace);
        m_tooltip.setFontSize(m_tooltipFontSize);
        m_tooltip.setFontBold(m_tooltipFontIsBold);
        m_tooltip.setFontItalic(m_tooltipFontIsItalic);
        m_tooltip.setFontColor(m_tooltipFontColor);
        m_tooltip.setBackgroundColor(m_tooltipBackgroundColor);
    }
}

/**
 * Update clock
 */

void KSmoothDock::updateClock() {
    if (m_showClock) {
        DockItem* item = m_items[m_items.size() - 1];
        Clock c_test;
        if (typeid(*item) == typeid(c_test)) {
            Clock* c = dynamic_cast<Clock*>(item);
            c->set24HourClock(m_use24HourClock);
            c->setFontFace(m_clockFontFace);
            c->setFontSize(m_clockFontSize);
            c->setFontBold(m_clockFontIsBold);
            c->setFontItalic(m_clockFontIsItalic);
            c->setFontColor(m_clockFontColor);
            repaint();
        }
    }
}

/**
 * Show the popup menu
 */

void KSmoothDock::showPopupMenu(const QPoint& pos) {
    KPopupMenu popup(this);
    popup.insertItem(i18n("Edit Quick Launch &Menu"), this, SLOT(configLaunchers()));
    popup.insertItem(i18n("Edit &Preferences"), this, SLOT(config()));
    popup.insertSeparator();
    if (m_zoomMode == NORMAL_ZOOM)
        popup.insertItem(i18n("Switch to Parabolic &Zooming Mode"), this, SLOT(switchZoomMode()));
    else
        popup.insertItem(i18n("Switch to Normal &Zooming Mode"), this, SLOT(switchZoomMode()));
    popup.insertSeparator();
    
    KPopupMenu position(&popup);
    position.insertItem(i18n("Top"), this, SLOT(setPositionTop()), 0, 0);
    position.insertItem(i18n("Bottom"), this, SLOT(setPositionBottom()), 0, 1);
    position.insertItem(i18n("Left"), this, SLOT(setPositionLeft()), 0, 2);
    position.insertItem(i18n("Right"), this, SLOT(setPositionRight()), 0, 3);

    switch(m_position) {
    case LEFT:
        position.setItemChecked(2, true);
        break;
    case RIGHT:
        position.setItemChecked(3, true);
        break;
    case TOP:
        position.setItemChecked(0, true);
        break;
    case BOTTOM:
        position.setItemChecked(1, true);
        break;
    default:
        break;
    }

    popup.insertItem("Position", &position);

    popup.insertItem(i18n("&Hide"), this, SLOT(buttonise()), 0, 4);
    if (m_isAutohide) {
        popup.setItemEnabled(4, false);
    }
        
    popup.insertItem(i18n("H&ide Automatically"), this, SLOT(switchAutohideMode()), 0, 5);
    if (m_isAutohide) {
        popup.setItemChecked(5, true);
    }

    popup.insertSeparator();
    popup.insertItem(i18n("&About"), this, SLOT(about()));
    popup.insertSeparator();
    popup.insertItem(i18n("E&xit"), this, SLOT(close()));

    popup.exec(pos);
}

/**
 * Config the dock
 */

void KSmoothDock::config(int tabIndex) {
    if (tabIndex > -1)
        m_configDialog->setCurrentPage(tabIndex);
    config();
}

/**
 * Set the desktop icons area
 */

void KSmoothDock::setDesktopIconsArea(int left, int top, int width, int height) {
    DCOPClient* client = KApplication::dcopClient();
    if (!client->isAttached()) {
        client->attach();
        client->registerAs("KSmoothDock");
    }

    for (int i = 0; i < KWin::numberOfDesktops(); i++) {
        QByteArray data;
        QDataStream args(data, IO_WriteOnly);
        args << left << top << width << height << i;
        client->send("kdesktop", "KDesktopIface", "desktopIconsAreaChanged(QRect, int)", data);
    }
}

/**
 * Exit the dock
 */

void KSmoothDock::exit() {
    close();
}

/**
 * Update the dock when the background has been changed, reload the wallpaper if necessary
 */

void KSmoothDock::updateBackground(const QPixmap& pix) {
    if (m_wallpaperManager->wallpapersChanged())
        m_wallpaperManager->loadWallpaper();
    else
        updateBackground();
}

/**
 * Update background, no wallpaper reload
 */

void KSmoothDock::updateBackground() {
    m_state->updateBackground();
}

/** 
 * Update the dock when a window has been added
 */

void KSmoothDock::windowAdded(WId id) {
    if (id == winId()) {
        m_state->dockCreatedEvent();
        return;
    }

    if (!m_showTaskbar)
        return;

    if (!m_showWindowsAllDesktops) {
        KWin::WindowInfo info(id, 0, 0);
        if (!info.onAllDesktops() && info.desktop() != KWin::currentDesktop())
            return;
    }

    addTask(id, true);
    updateLayout(false);
}

/**
 * Update the dock when a window has been removed
 */

void KSmoothDock::windowRemoved(WId id) {
    if (!m_showTaskbar)
        return;

    Task task;
    
    std::vector<DockItem*>::iterator it = m_items.begin();
    for (unsigned int i = 0; i < m_items.size(); i++, it++) {
        // if the item is a task, check its id
        if (typeid(*m_items[i]) == typeid(task)) {
            Task* t = dynamic_cast<Task*>(m_items[i]);
            if (t->getWinId() == id) {
                // remove the task
                m_items.erase(it);
                // free memory
                delete t;
                // update layout
                updateLayout(false);
                break;
            }
        }
    }    
}

/**
 * Update the dock when a window has been changed
 */

void KSmoothDock::windowChanged(WId id, unsigned int properties) {
    if (id == winId())
        return;

    if (!m_showTaskbar)
        return;

    bool iconChanged = (properties & NET::WMIcon);
    bool nameChanged = (properties & NET::WMName);
    bool deskChanged = (properties & NET::WMDesktop);
    bool stateChanged = (properties & NET::WMState);

    KWin::WindowInfo info(id, 0, 0);

    if (!m_showWindowsAllDesktops && !info.onAllDesktops() && info.desktop() != KWin::currentDesktop() && !deskChanged)
        return;

    if (deskChanged && !m_showWindowsAllDesktops) {
        if (info.desktop() == KWin::currentDesktop()) { // a window has been moved to the current desktop, add it
            windowAdded(id);
        } else { // a window may have been moved to a different desktop, remove it
            windowRemoved(id);
        }
    }

    Task task;

    for (unsigned int i = 0; i < m_items.size(); i++) {
        // if the item is a task, check its id
        Task task;
        if (typeid(*m_items[i]) == typeid(task)) {
            Task* t = dynamic_cast<Task*>(m_items[i]);

            if (t->getWinId() == id) {
                if (nameChanged) {
                    // Window title changed.
                    t->setDescription(info.name());
                }

                if (iconChanged) {
                    // Window icon changed.
                    QPixmap pix = KWin::icon(id, m_baseIconSize, m_baseIconSize);
                    t->setIcon(pix);
                    m_state->updateItem(i);
                }

                if (stateChanged) {
                    if (info.state() & NET::DemandsAttention) { // demand attention
                        t->setDemandsAttention(true);
                    } else if (t->getDemandsAttention()) {
                        t->setDemandsAttention(false);
                    }
                }
                break;
            }
        }
    }
}

/**
 * Update the dock when the active window has been changed
 */

void KSmoothDock::activeWindowChanged(WId id) {
    m_state->activeWindowChanged(id);
}

/**
 * Update the dock when the current desktop has been changed
 */

void KSmoothDock::currentDesktopChanged(int desktop) {
    if (!m_showWindowsAllDesktops) {
        loadLaunchers();
        initPager();
        loadTasks();
        initClock();
        updateLayout(false);
    }
}

/**
 * Update a specific item
 */

void KSmoothDock::updateItem(int itemId) {
    for (unsigned int i = 0; i < m_items.size(); i++) {
        if (m_items[i]->getItemId() == itemId) {
            m_state->updateItem(i);
            break;
        }
    }
}

/**
 * Config the launchers
 */

void KSmoothDock::configLaunchers() {
    m_launcherConfigDialog->update();
    m_launcherConfigDialog->show();
}

/**
 * Config the dock
 */

void KSmoothDock::config() {
    m_configDialog->updateValues();
    m_configDialog->show();
}

/**
 * Switch the zoom mode
 */

void KSmoothDock::switchZoomMode() {
    if (m_zoomMode == NORMAL_ZOOM) {
        m_zoomMode = PARABOLIC_ZOOM;

        if (m_parabolicZoomState == NULL)
            m_parabolicZoomState = new ParabolicZoomState(this);
        m_state = m_parabolicZoomState;
    } else { // Parabolic zoom
        m_zoomMode = NORMAL_ZOOM;
        if (m_normalZoomState == NULL)
            m_normalZoomState = new NormalZoomState(this);
        m_state = m_normalZoomState;
    }

    // init icon sizes    
    if (m_zoomMode == NORMAL_ZOOM) {
        m_baseIconSize = m_NzBaseIconSize;
        m_smallIconSize = m_NzSmallIconSize;
        m_bigIconSize = m_NzBigIconSize;
    } else {
        m_baseIconSize = m_PzBaseIconSize;
        m_smallIconSize = m_PzSmallIconSize;
        m_bigIconSize = m_PzBigIconSize;
    }
    loadLaunchers();
    initPager();
    loadTasks();
    initClock();
    updateLayout();
    updateLayout();

    saveConfig();
}

/**
 * Hide the dock (to a buttonised state)
 */

void KSmoothDock::buttonise() {
    if (m_buttonisedState == NULL)
        m_buttonisedState = new ButtonisedState(this);
    m_state = m_buttonisedState;
    updateLayout();
}

/**
 * Show the about dialog
 */

void KSmoothDock::about() {
    m_aboutDialog->show();
}

/**
 * Update the launchers
 */

void KSmoothDock::updateLaunchers() {
    // load quick launchers
    loadLaunchers();

    // initiate the pager
    initPager();

    // load the running tasks
    loadTasks();

    // init clock
    initClock();

    // update layout
    updateLayout();
}

/**
 * Set position to the top of the screen
 */

void KSmoothDock::setPositionTop() {
    setPosition(TOP);
}

/**
 * Set position to the bottom of the screen
 */

void KSmoothDock::setPositionBottom() {
    setPosition(BOTTOM);
}

/**
 * Set position to the left of the screen
 */

void KSmoothDock::setPositionLeft() {
    setPosition(LEFT);
}

/**
 * Set position to the right of the screen
 */

void KSmoothDock::setPositionRight() {
    setPosition(RIGHT);
}

/**
 * Set position of the dock
 * @param position the new position
 */

void KSmoothDock::setPosition(PanelPosition position) {
    if (m_position == position)
        return;

    m_position = position;
    m_orientation = ((position == TOP) || (position == BOTTOM)) ? Qt::Horizontal : Qt::Vertical;

    loadLaunchers();
    initPager();
    loadTasks();
    initClock();
    updateLayout();
    updateLayout();    

    saveConfig();
}

/**
 * Switch autohide mode on/off
 */

void KSmoothDock::switchAutohideMode() {
    m_isAutohide = 1 - m_isAutohide;
    m_state->updateAutohideMode();
    saveConfig();
}

/// PROTECTED ///

/** 
 * Paint event handler
 *
 * @param e the paint event
 */

void KSmoothDock::paintEvent(QPaintEvent* e) {
    //std::cout << "Paint event" << std::endl;
    m_state->paintEvent(e);
}

// Mouse pressed event handler
void KSmoothDock::mousePressEvent(QMouseEvent* e) {
    m_state->mousePressEvent(e);
}

// Mouse moved event handler
void KSmoothDock::mouseMoveEvent(QMouseEvent* e) {
    m_state->mouseMoveEvent(e);
}

// Enter event handler
void KSmoothDock::enterEvent(QEvent* e) {
    m_state->enterEvent(e);
}

// Leave event handler
void KSmoothDock::leaveEvent(QEvent* e) {
    m_state->leaveEvent(e);
}

/// PRIVATE ///

/**
 * Init the dock
 */

void KSmoothDock::init() {
    KWin::setOnAllDesktops(winId(), true);
    KWin::setState(winId(), NET::SkipTaskbar | NET::SkipPager);
    KWin::setType(winId(), NET::Dock);
    setBackgroundMode(NoBackground);
    setMouseTracking(TRUE);
}

/**
 * Load configuration data
 */

void KSmoothDock::loadConfig(bool reset) {
    KConfig* config = KGlobal::config();
    config->setGroup("ksmoothdock");

    QDir home_dir = QDir::home();
    QString conf_file = locateLocal("config", "ksmoothdockrc");
    bool first_run;
    if (reset) {
        first_run = true;
    } else {
        first_run = !home_dir.exists(conf_file);
    }
    if (first_run) { // first run of KSmoothDock, or in case of reset
        WelcomeDialog welcome(this, "Welcome to KSmoothDock", TRUE);
        welcome.exec();

        m_zoomMode = welcome.getZoomMode();
        m_showPager = welcome.isShowingPager();
        m_showTaskbar = welcome.isShowingTaskbar();
        m_showWindowsAllDesktops = welcome.isShowingWindowsAllDesktops();
        m_showClock = welcome.isShowingClock();

        m_isAutohide = welcome.isAutohide();
        m_position = welcome.getPosition();
    } else {
        m_zoomMode = static_cast<ZoomMode> (config->readNumEntry("ZoomMode", NORMAL_ZOOM));
        m_showPager = config->readNumEntry("ShowPager", 1);
        m_showTaskbar = config->readNumEntry("ShowTaskbar", 1);
        m_showWindowsAllDesktops = config->readNumEntry("ShowWindowsFromAllDesktops", 1);
        m_showClock = config->readNumEntry("ShowClock", 1);

        m_isAutohide = config->readNumEntry("Autohide", 0);
        m_position = static_cast<PanelPosition> (config->readNumEntry("PanelPosition", RIGHT));
    }    

    m_NzZoomSpeed = config->readNumEntry("NormalZoomSpeed", 16);
    m_PzZoomSpeed = config->readNumEntry("ParabolicZoomSpeed", 16);
    m_showNotification = config->readNumEntry("ShowNotification", 0);
    m_showBorders = config->readNumEntry("ShowBorders", 1);
    m_borderColor = QColor(config->readEntry("BorderColor", "#b1c4de"));

    m_NzSmallIconSize = config->readNumEntry("SmallIconSizeNormalZoom", NZ_DEFAULT_SMALL_SIZE);
    m_NzBigIconSize = config->readNumEntry("BigIconSizeNormalZoom", NZ_DEFAULT_BIG_SIZE);
    m_NzBaseIconSize = config->readNumEntry("BaseIconSizeNormalZoom", NZ_DEFAULT_BASE_SIZE);

    m_PzSmallIconSize = config->readNumEntry("SmallIconSizeParabolicZoom", PZ_DEFAULT_SMALL_SIZE);
    m_PzBigIconSize = config->readNumEntry("BigIconSizeParabolicZoom", PZ_DEFAULT_BIG_SIZE);
    m_PzBaseIconSize = config->readNumEntry("BaseIconSizeParabolicZoom", PZ_DEFAULT_BASE_SIZE);

    m_dockOpacity = config->readNumEntry("DockOpacity", DEFAULT_DOCK_OPACITY);
    m_backgroundColor = QColor(config->readEntry("BackgroundColor", "#638abd"));
    m_separatorColor = QColor(config->readEntry("SeparatorColor", "#b1c4de"));
    m_screenWidth = config->readNumEntry("ScreenWidth", 0);
    m_activeDesktopColor = QColor(config->readEntry("ActiveDesktopColor", "yellow"));
    m_inactiveDesktopColor = QColor(config->readEntry("InactiveDesktopColor", "white"));

    m_showTooltip = config->readNumEntry("ShowTooltip", 1);
    m_tooltipFontFace = config->readEntry("TooltipFontFace", "Sans Serif");
    m_tooltipFontSize = config->readNumEntry("TooltipFontSize", 18);
    m_tooltipFontIsItalic = config->readNumEntry("TooltipFontItalic", 1);
    m_tooltipFontIsBold = config->readNumEntry("TooltipFontBold", 1);
    m_tooltipFontColor = QColor(config->readEntry("TooltipFontColor", "white"));
    m_tooltipBackgroundColor = QColor(config->readEntry("TooltipFontBgColor", "black"));

    m_use24HourClock = config->readNumEntry("Use24HourClock", 1);
    m_clockFontColor = QColor(config->readEntry("ClockFontColor", "white"));
    m_clockFontFace = config->readEntry("ClockFontFace", "Sans Serif");

    int def_size = (m_zoomMode == NORMAL_ZOOM) ? m_NzSmallIconSize/2 : m_PzSmallIconSize/2;
    m_clockFontSize = config->readNumEntry("ClockFontSize", def_size);

    m_clockFontIsItalic = config->readNumEntry("ClockFontItalic", 0);
    m_clockFontIsBold = config->readNumEntry("ClockFontBold", 0);

    if (m_position == TOP || m_position == BOTTOM) {
        m_orientation = Horizontal;
    } else {
        m_orientation = Vertical;
    }

    // correct the icon sizes if needed

    if (m_NzSmallIconSize > m_NzBigIconSize || m_PzSmallIconSize > m_PzBigIconSize) {
        m_NzSmallIconSize = NZ_DEFAULT_SMALL_SIZE;
        m_NzBigIconSize = NZ_DEFAULT_BIG_SIZE;
        
        m_PzSmallIconSize = PZ_DEFAULT_SMALL_SIZE;
        m_PzBigIconSize = PZ_DEFAULT_BIG_SIZE;
    }
    
    if (!m_showTaskbar) {
        m_showNotification = 0;
    }

    if (first_run) {    
        saveConfig();
    }
}

/**
 * Save configuration data
 */

void KSmoothDock::saveConfig() {
    KConfig* config = KGlobal::config();
    config->setGroup("ksmoothdock");

    config->writeEntry("ZoomMode", m_zoomMode);
    config->writeEntry("NormalZoomSpeed", m_NzZoomSpeed);
    config->writeEntry("ParabolicZoomSpeed", m_PzZoomSpeed);

    config->writeEntry("ShowPager", m_showPager);
    config->writeEntry("ShowTaskbar", m_showTaskbar);
    config->writeEntry("ShowWindowsFromAllDesktops", m_showWindowsAllDesktops);
    config->writeEntry("ShowClock", m_showClock);

    config->writeEntry("ShowNotification", m_showNotification);
    config->writeEntry("ShowBorders", m_showBorders);
    config->writeEntry("BorderColor", m_borderColor.name());

    config->writeEntry("PanelPosition", m_position);

    config->writeEntry("SmallIconSizeNormalZoom", m_NzSmallIconSize);
    config->writeEntry("BigIconSizeNormalZoom", m_NzBigIconSize);
    config->writeEntry("BaseIconSizeNormalZoom", m_NzBaseIconSize);  

    config->writeEntry("SmallIconSizeParabolicZoom", m_PzSmallIconSize);
    config->writeEntry("BigIconSizeParabolicZoom", m_PzBigIconSize);
    config->writeEntry("BaseIconSizeParabolicZoom", m_PzBaseIconSize);

    config->writeEntry("Autohide", m_isAutohide);

    config->writeEntry("DockOpacity", m_dockOpacity);
    config->writeEntry("BackgroundColor", m_backgroundColor.name());
    config->writeEntry("SeparatorColor", m_separatorColor.name());
    config->writeEntry("ScreenWidth", m_screenWidth);
    config->writeEntry("ActiveDesktopColor", m_activeDesktopColor.name());
    config->writeEntry("InactiveDesktopColor", m_inactiveDesktopColor.name());

    config->writeEntry("ShowTooltip", m_showTooltip);
    config->writeEntry("TooltipFontFace", m_tooltipFontFace);
    config->writeEntry("TooltipFontSize", m_tooltipFontSize);
    config->writeEntry("TooltipFontItalic", m_tooltipFontIsItalic);
    config->writeEntry("TooltipFontBold", m_tooltipFontIsBold);
    config->writeEntry("TooltipFontColor", m_tooltipFontColor.name());
    config->writeEntry("TooltipFontBgColor", m_tooltipBackgroundColor.name());

    config->writeEntry("Use24HourClock", m_use24HourClock);
    config->writeEntry("ClockFontColor", m_clockFontColor.name());
    config->writeEntry("ClockFontFace", m_clockFontFace);
    config->writeEntry("ClockFontSize", m_clockFontSize);
    config->writeEntry("ClockFontItalic", m_clockFontIsItalic);
    config->writeEntry("ClockFontBold", m_clockFontIsBold);

    config->sync();
}

/** 
 * Initiate variables
 */

void KSmoothDock::initVariables() {
    // init path    
    m_dataPath = locateLocal("data", "ksmoothdock");
    m_launchersPath = m_dataPath + QString("/menu/");
    
    // init icon sizes    
    if (m_zoomMode == NORMAL_ZOOM) {
        m_baseIconSize = m_NzBaseIconSize;
        m_smallIconSize = m_NzSmallIconSize;
        m_bigIconSize = m_NzBigIconSize;
    } else {
        m_baseIconSize = m_PzBaseIconSize;
        m_smallIconSize = m_PzSmallIconSize;
        m_bigIconSize = m_PzBigIconSize;
    }

    // state
    m_normalZoomState = NULL;
    m_parabolicZoomState = NULL;
    m_buttonisedState = NULL;

    if (m_zoomMode == NORMAL_ZOOM) {
        m_normalZoomState = new NormalZoomState(this);
        m_state = m_normalZoomState;
    } else { // Parabolic
        m_parabolicZoomState = new ParabolicZoomState(this);
        m_state = m_parabolicZoomState;
    }

    // tooltip
    if (m_showTooltip) {
        m_tooltip.setFontFace(m_tooltipFontFace);
        m_tooltip.setFontSize(m_tooltipFontSize);
        m_tooltip.setFontBold(m_tooltipFontIsBold);
        m_tooltip.setFontItalic(m_tooltipFontIsItalic);
        m_tooltip.setFontColor(m_tooltipFontColor);
        m_tooltip.setBackgroundColor(m_tooltipBackgroundColor);
    }

    m_itemIndexInitialiser = 0;

    m_aboutDialog.reset(new AboutDialog());
    m_configDialog.reset(new ConfigDialog(this));
    m_launcherConfigDialog.reset(new LauncherConfigDialog());
    connect(m_launcherConfigDialog.get(), SIGNAL(updated()), this, SLOT(updateLaunchers()));

    m_wallpaperManager = new WallpaperManager(KWin::numberOfDesktops());
    connect(m_wallpaperManager, SIGNAL(wallpaperLoaded()), this, SLOT(updateBackground()));

    m_rootPix.reset(new KRootPixmap(this));
    m_rootPix->setCustomPainting(true);
    connect(m_rootPix.get(), SIGNAL(backgroundUpdated(const QPixmap&)), this, SLOT(updateBackground(const QPixmap&)));
    m_rootPix->start();
//    else
//        m_state = new HiddenState();

    connect(&m_windowManager, SIGNAL(currentDesktopChanged(int)), this, SLOT(currentDesktopChanged(int)));
    connect(&m_windowManager, SIGNAL(windowAdded (WId)), this, SLOT(windowAdded (WId)));
    connect(&m_windowManager, SIGNAL(windowRemoved (WId)), this, SLOT(windowRemoved (WId)));
    connect(&m_windowManager, SIGNAL(windowChanged(WId, unsigned int)), this, SLOT(windowChanged(WId, unsigned int)));    
    connect(&m_windowManager, SIGNAL(activeWindowChanged(WId)), this, SLOT(activeWindowChanged(WId)));    
}

/**
 * Load quick launchers
 */

void KSmoothDock::loadLaunchers(bool reset) {
    m_items.clear();
    QDir home_dir = QDir::home();
    
    if (reset) {
        if (home_dir.exists(m_launchersPath)) {  // delete the launchers directory
            QDir dir(m_launchersPath);
            QStringList file_list = dir.entryList("*");
            for (QStringList::Iterator it = file_list.begin(); it != file_list.end(); it++) {
                if ((*it) && (*it != ".") && (*it != "..")) {
                    dir.remove(*it);
                }
            }
            home_dir.rmdir(m_launchersPath);
            home_dir.rmdir(m_dataPath);
        }
        createDefaultLaunchers();
    } else {
        if (home_dir.exists(m_launchersPath)) { // if the launchers directory exists
            QDir dir(m_launchersPath);
            QStringList file_list = dir.entryList("*");
            for (QStringList::Iterator it = file_list.begin(); it != file_list.end(); it++) {
                if ((*it) && (*it != ".") && (*it != "..")) {
                    m_items.push_back(new Launcher(this, m_itemIndexInitialiser++, m_launchersPath + (*it), m_baseIconSize, m_smallIconSize, m_bigIconSize, m_orientation));
                }
            }
        } else { // otherwise create default quick launchers
            createDefaultLaunchers();
        }
    }

    //std::cout << "Loaded " << m_items.size() << " launchers" << std::endl;
}

/**
 * Create default quick launchers
 */
 
void KSmoothDock::createDefaultLaunchers() {
    QDir home_dir = QDir::home();
    if (!home_dir.mkdir(m_dataPath) || !home_dir.mkdir(m_launchersPath)) {
        KMessageBox::information(this, QString("Could not create data folder: ") + m_launchersPath);
        return;
    }
    
    Launcher* item1 = new Launcher(this, m_itemIndexInitialiser++, "Start Applications", ON_ALL_DESKTOPS, "kmenu", m_baseIconSize, m_smallIconSize, m_bigIconSize, m_orientation, "dcop kicker kicker popupKMenu 0");
    item1->saveToFile(m_launchersPath + "0 - KMenu");
    m_items.push_back(item1);
    
    Launcher* item2 = new Launcher(this, m_itemIndexInitialiser++, "Show Desktop", ON_ALL_DESKTOPS, "desktop", m_baseIconSize, m_smallIconSize, m_bigIconSize, m_orientation, "SHOW_DESKTOP");
    item2->saveToFile(m_launchersPath + "1 - show desktop");
    m_items.push_back(item2);
    
    Launcher* item3 = new Launcher(this, m_itemIndexInitialiser++, "Konsole", ON_ALL_DESKTOPS, "konsole", m_baseIconSize, m_smallIconSize, m_bigIconSize, m_orientation, "konsole");
    item3->saveToFile(m_launchersPath + "2 - konsole");
    m_items.push_back(item3);

    Launcher* item4 = new Launcher(this, m_itemIndexInitialiser++, "Home Directory", ON_ALL_DESKTOPS, "kfm_home", m_baseIconSize, m_smallIconSize, m_bigIconSize, m_orientation, "kfmclient openProfile filemanagement");
    item4->saveToFile(m_launchersPath + "3 - home");
    m_items.push_back(item4);

    Launcher* item5 = new Launcher(this, m_itemIndexInitialiser++, "Web Browser", ON_ALL_DESKTOPS, "package_network", m_baseIconSize, m_smallIconSize, m_bigIconSize, m_orientation, "konqueror --profile webbrowsing");
    item5->saveToFile(m_launchersPath + "4 - konqueror");
    m_items.push_back(item5);

    Launcher* item6 = new Launcher(this, m_itemIndexInitialiser++, "Control Center", ON_ALL_DESKTOPS, "kcontrol", m_baseIconSize, m_smallIconSize, m_bigIconSize, m_orientation, "kcontrol");
    item6->saveToFile(m_launchersPath + "5 - control center");
    m_items.push_back(item6);
}

/**
 * Initiate the pager
 */

void KSmoothDock::initPager() {
    if (m_showPager) {
        int num_desks = KWin::numberOfDesktops();
    
        for (int i = 1; i <= num_desks; i++) {
            DesktopSelector* item = new DesktopSelector(this, m_itemIndexInitialiser++, "Desktop " + QString::number(i), ON_ALL_DESKTOPS, m_wallpaperManager->getWallpaper(i), m_smallIconSize, m_bigIconSize, m_orientation, i, m_wallpaperManager);
            m_items.push_back(item);
            connect(m_wallpaperManager, SIGNAL(wallpaperLoaded()), item, SLOT(updateIcon()));
        }
    } else if (m_showTaskbar) { // Add a seperator between the launchers and the taskbar
        Separator* item = new Separator(this, m_itemIndexInitialiser++, ON_ALL_DESKTOPS, m_smallIconSize, m_bigIconSize, 0.0, m_orientation);
        m_items.push_back(item);
    }
}

/**
 * Load running tasks
 */

void KSmoothDock::loadTasks() {
    if (m_showTaskbar) {
        for (QValueList<WId>::ConstIterator it = m_windowManager.windows().begin(); it != m_windowManager.windows().end(); it++) {
            if (m_windowManager.hasWId(*it)) {
                addTask(*it);
            }
        }
    }
    // std::cout << "Loaded " << m_items.size() << " launchers and tasks" << std::endl;
}

/**
 * Add a task
 */

void KSmoothDock::addTask(WId id, bool checkPosition) {
    QPixmap pix = KWin::icon(id, m_baseIconSize, m_baseIconSize);
    KWin::WindowInfo info(id, 0, 0);
    QString name = info.name();

    if (!m_showWindowsAllDesktops) {
        if (!info.onAllDesktops() && info.desktop() != KWin::currentDesktop()) {
            return;
        }
    }  

    NET::WindowType type = info.windowType(0xffff);
    if (type == NET::Normal || type == NET::Dialog || type == NET::Unknown || type == NET::Override)
        if ((info.state() & NET::SkipTaskbar) == 0)
            if (name.compare("XMMS Playlist") != 0 && name.compare("XMMS Equalizer") != 0) {
                Task* t = new Task(this, m_itemIndexInitialiser++, name, info.desktop(), pix, m_smallIconSize, m_bigIconSize, m_orientation, id);

                if (checkPosition && m_showClock) {
                    m_items.insert(m_items.end() - 1, t);
                } else
                    m_items.push_back(t);
            }
}

/**
 * Init the clock
 */

void KSmoothDock::initClock() {
    if (m_showClock) {
        Clock* c = new Clock(this, m_itemIndexInitialiser++, "", ON_ALL_DESKTOPS, m_smallIconSize, m_bigIconSize, 2.0, m_orientation);
        c->set24HourClock(m_use24HourClock);
        c->setFontFace(m_clockFontFace);
        c->setFontSize(m_clockFontSize);
        c->setFontBold(m_clockFontIsBold);
        c->setFontItalic(m_clockFontIsItalic);
        c->setFontColor(m_clockFontColor);
        m_items.push_back(c);
    }
}

/**
 * Update layout
 */

void KSmoothDock::updateLayout(bool reset) {
    m_state->updateLayout(reset);
    if (reset && m_showTooltip) {
        m_tooltip.hide();
    }
}

/**
 * Set dock above all applications
 */

void KSmoothDock::setDockAbove() {    
    KWin::setState(winId(), NET::KeepAbove);
}

/**
 * Set dock below all applications
 */

void KSmoothDock::setDockBelow() {
    KWin::setState(winId(), NET::KeepBelow);
    KWin::lowerWindow(winId());
}

/**
 * Check kicker position to avoid collision
 */

void KSmoothDock::checkKickerPosition() {
    DCOPClient *client = KApplication::dcopClient();
    if (!client->isAttached()) {
        client->attach();
        client->registerAs("KSmoothDock");
    }

    QByteArray data, replyData;
    QCString replyType;
    client->call("kicker", "Panel", "panelPosition()", data, replyType, replyData );
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "int") {
        int result;
        PanelPosition kicker_pos;

        reply >> result;
        //cout << "the result is: " << result << endl;
        switch(result) {
        case 0:
            kicker_pos = LEFT;
            break;
        case 1:
            kicker_pos = RIGHT;
            break;
        case 2:
            kicker_pos = TOP;
            break;
        case 3:
            kicker_pos = BOTTOM;
            break;
        }

        if (kicker_pos == m_position) {
            KMessageBox::information(this, i18n("KSmoothDock and kicker have the same position on the screen and hence can overlap each other visually. Please move one of these to another position."), i18n("KSmoothDock and kicker"), "ShowAgainCheckKickerPosition");
        }
    }
}

#include "ksmoothdock.moc"
