/***************************************************************************
 *   Copyright (C) 2006 by the KSmoothDock team   *
 *   dangvd@yahoo.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "wallpapermanager.h"

#include <iostream>

#include <qcstring.h>
#include <qdatastream.h>
#include <qfileinfo.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <kstandarddirs.h>
#include <kwin.h>
#include <dcopclient.h>

#include "constants.h"

/// PUBLIC ///

/** 
 * Constructor
 * @param numDesks number of virtual desktops
 */

WallpaperManager::WallpaperManager(int numDesks) {
    m_configFile = locateLocal("config", "kdesktoprc");
    QFileInfo fi(m_configFile);
    m_lastUpdateTime = fi.lastModified();

    m_numDesktops = numDesks;
    for (int i = 0; i < m_numDesktops; i++) {
        m_wallpapers.push_back(new KSharedPixmap());
        m_wallpapersLoaded.push_back(false);
    }

    initWallpapers();
}

/** 
 * Destructor
 */

WallpaperManager::~WallpaperManager() {
    for (int i = 0; i < m_numDesktops; i++) {
        delete m_wallpapers[i];
    }
}

/** 
 * Check if desktop wallpapers have been changed
 */

bool WallpaperManager::wallpapersChanged() {
    // if the wallpaper has never been loaded, return true
    if (!m_wallpapersLoaded[KWin::currentDesktop()])
        return true;

    // check the date of the desktop config file
    QFileInfo fi(m_configFile);
    QDateTime t = fi.lastModified();
    if (m_lastUpdateTime != t) {
        m_lastUpdateTime = t;
        return true;
    } else {
        return false;
    }
}

/** 
 * Load the current wallpaper
 *
 * @NOTE: as of KDE 3.x, if we import from kdesktop, we can only import wallpapers that have been displayed, so there is no point importing other wallpapers as well (as other wallpapers may or may not be loadable)
 */

void WallpaperManager::loadWallpaper() {
    DCOPClient* client = KApplication::dcopClient();
    if (!client->isAttached()) {
        client->attach();
        client->registerAs("KSmoothDock");
    }

    QByteArray data;
    QDataStream args(data, IO_WriteOnly);
    args << 1;
    client->send("kdesktop", "KBackgroundIface", "setExport(int)", data);

    int cur_desk = KWin::currentDesktop();
    connect(m_wallpapers[cur_desk - 1], SIGNAL(done(bool)), this, SLOT(receiveWallpaper(bool)));

    m_wallpapers[cur_desk - 1]->loadFromShared(QString("DESKTOP%1").arg(cur_desk));
}

/** 
 * Get a wallpaper
 * @param desk desktop number (value = 1...numberOfDesktops)
 */
    
QPixmap& WallpaperManager::getWallpaper(int desk) {
    return *m_wallpapers[desk-1];
}

/** 
 * Handle the event when the wallpaper has been loaded
 */

void WallpaperManager::receiveWallpaper(bool success) {
    if (success) {
        // std::cout << "loaded wallpaper" << std::endl;
        m_wallpapersLoaded[KWin::currentDesktop()] = true;
        emit wallpaperLoaded();
    }
}

/// PRIVATE ///

/**
 * Initiate the wallpapers
 */

void WallpaperManager::initWallpapers() {
    // std::cout << "init wallpapers" << std::endl;
    DCOPClient* client = KApplication::dcopClient();
    if (!client->isAttached()) {
        client->attach();
        client->registerAs("KSmoothDock");
    }

    for (int i = 1; i <= m_numDesktops; i++) {
        QByteArray data;
        QByteArray reply;
        QDataStream args(data, IO_WriteOnly);
        args << i;
        QCString type;
        if (client->call("kdesktop", "KBackgroundIface", "currentWallpaper(int)", data, type, reply, false, 1000) && type == "QString") {
            QDataStream ds(reply, IO_ReadOnly);
            QString path; // path to the wallpaper image file
            ds >> path;
            // std::cout << "current wallpaper " << i << ": " << path << std::endl;
            if (!m_wallpapers[i-1]->load(path)) { // load the image
                // std::cout << "could not load. load backup" << std::endl;
                // if loading failed, load a replacement default image
                QString s = KGlobal::iconLoader()->iconPath("desktop", KIcon::Desktop);
                m_wallpapers[i-1]->load(s);
            }
        }
    }
}
