/******************************************************************************
 * Copyright (C) 2006 Tetsuya Kimata <kimata@acapulco.dyndns.org>
 *
 * All rights reserved.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty.  In no event will the authors be held liable for any
 * damages arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any
 * purpose, including commercial applications, and to alter it and
 * redistribute it freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must
 *    not claim that you wrote the original software. If you use this
 *    software in a product, an acknowledgment in the product
 *    documentation would be appreciated but is not required.
 *
 * 2. Altered source versions must be plainly marked as such, and must
 *    not be misrepresented as being the original software.
 *
 * 3. This notice may not be removed or altered from any source
 *    distribution.
 *
 * $Id: DownloadFlowController.cpp 2221 2006-12-10 06:11:38Z svn $
 *****************************************************************************/

#include "Environment.h"

#include <iomanip>

#include "DownloadFlowController.h"
#include "WriteLocker.h"
#include "SourceInfo.h"

SOURCE_INFO_ADD("$Id: DownloadFlowController.cpp 2221 2006-12-10 06:11:38Z svn $");

/******************************************************************************
 * public メソッド
 *****************************************************************************/
DownloadFlowController::DownloadFlowController()
  : downloader_count_(0)
{
    memset(downloader_list_, 0, sizeof(downloader_list_));
    apr_atomic_set(&lock_, ReadWriteLocker::NOT_LOCKER);
}

bool DownloadFlowController::add_session(apr_sockaddr_t *ip_address)
{
    downloader_t *downloader;

    WriteLocker lock(&lock_);

    downloader = downloader_list_get(ip_address);

    if (downloader == NULL) {
        return downloader_list_add(ip_address);
    } else {
        if (downloader->session_count == DLD_MAX_SESSION_COUNT_PER_IP) {
            return false;
        } else {
            downloader->session_count++;
            return true;
        }
    }
}

void DownloadFlowController::remove_session(apr_sockaddr_t *ip_address)
{
    downloader_t *downloader;

    WriteLocker lock(&lock_);

    downloader = downloader_list_get(ip_address);

    if ((downloader == NULL) || (downloader->session_count == 0)) {
        return;
    }

    downloader->session_count--;

    if (downloader->session_count != 0) {
        downloader_count_--;
    }
}

DownloadFlowController *DownloadFlowController::get_instance(apr_shm_t *shm)
{
    DownloadFlowController *flow_controller;

#ifdef DEBUG
    if (apr_shm_size_get(shm) != get_memory_size()) {
        THROW(MESSAGE_SHM_SIZE_INVALID);
    }
#endif

    flow_controller =
        reinterpret_cast<DownloadFlowController *>(apr_shm_baseaddr_get(shm));
    new(flow_controller) DownloadFlowController;

    return flow_controller;
}

apr_size_t DownloadFlowController::get_memory_size()
{
    return sizeof(DownloadFlowController);
}

void DownloadFlowController::dump_list(DownloadFlowController *flow_controller)
{
    for (apr_size_t i = 0; i < DLD_MAX_SESSION_COUNT; i++) {
        if (flow_controller->downloader_list_[i].session_count == 0) {
            continue;
        }
        dump_downloader(flow_controller->downloader_list_ + i);
    }
}


/******************************************************************************
 * private メソッド
 *****************************************************************************/
bool DownloadFlowController::downloader_list_add(apr_sockaddr_t *ip_address)
{
    apr_size_t i;

    if (downloader_count_ == DLD_MAX_SESSION_COUNT) {
        return false;
    }

    for (i = 0; i < ARRAY_SIZE_OF(downloader_list_); i++) {
        if (downloader_list_[i].session_count == 0) {
            break;
        }
    }

#ifdef DEBUG
    if (i == DLD_MAX_SESSION_COUNT) {
        THROW(MESSAGE_BUG_FOUND);
    }
#endif

    memcpy(&(downloader_list_[i].ip_address), &(ip_address->sa),
           sizeof(ip_address->sa));
    downloader_list_[i].session_count = 1;
    downloader_count_++;

    return true;
}

DownloadFlowController::downloader_t *
DownloadFlowController::downloader_list_get(apr_sockaddr_t *ip_address)
{
    for (apr_size_t i = 0; i < ARRAY_SIZE_OF(downloader_list_); i++) {
        if (memcmp(&(downloader_list_[i].ip_address), &(ip_address->sa),
#if APR_HAVE_IPV6
                   sizeof(struct sockaddr_in6)
#else
                   sizeof(struct sockaddr_in)
#endif
                   ) != 0) {
            continue;
        }
        return (downloader_list_ + i);
    }

    return NULL;
}

void DownloadFlowController::dump_downloader(downloader_t *downloader)
{
    apr_size_t *address;

    // 強引にキャスト
    address = reinterpret_cast<apr_size_t *>(&(downloader->ip_address));

    cerr << setw(24) << setiosflags(ios::left) << *address;
    cerr << ": " << downloader->session_count << endl;
}


/******************************************************************************
 * テスト
 *****************************************************************************/
#ifdef DEBUG_DownloadFlowController
#include "TestRunner.h"

const char DUMMY_IP_ADDRESS[]           = "127.0.0.1";
const apr_size_t ATTACK_COUNT           = 100;

void show_usage(const char *prog_name)
{
    cerr << "Usage: " << prog_name << endl;
}

void run_can_download(DownloadFlowController *flow_controller)
{
    show_test_name("can_download");

    for (apr_size_t i = 0; i < DLD_MAX_SESSION_COUNT_PER_IP; i++) {
        flow_controller->add_session(DUMMY_IP_ADDRESS);
    }
    if (flow_controller->add_session(DUMMY_IP_ADDRESS)) {
        THROW(MESSAGE_BUG_FOUND);
    }
    flow_controller->remove_session(DUMMY_IP_ADDRESS);
    if (!flow_controller->add_session(DUMMY_IP_ADDRESS)) {
        THROW(MESSAGE_BUG_FOUND);
    }
}

void run_all(apr_pool_t *pool, int argc, const char * const *argv)
{
    if (argc != 1) {
        THROW(MESSAGE_ARGUMENT_INVALID);
    }

    DownloadFlowController flow_controller;
    show_line();
    run_can_download(&flow_controller);
}

#endif

// Local Variables:
// mode: c++
// coding: utf-8-dos
// End:
