/**
 * $Id: mxCellMarker.java,v 1.6 2009/01/14 14:39:55 gaudenz Exp $
 * Copyright (c) 2008, Gaudenz Alder
 */
package com.mxgraph.swing.handler;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.event.MouseEvent;

import javax.swing.JComponent;

import com.mxgraph.swing.mxGraphComponent;
import com.mxgraph.util.mxConstants;
import com.mxgraph.util.mxEventSource;
import com.mxgraph.util.mxUtils;
import com.mxgraph.util.mxEventSource.mxIEventListener;
import com.mxgraph.view.mxCellState;
import com.mxgraph.view.mxGraphView;

/**
 * 
 */
public class mxCellMarker extends JComponent
{

	/**
	 * Holds the name for the mark event, which fires after a cell has been
	 * marked. First and only argument in the array is the cell state that has
	 * been marked or null, if no state has been marked.
	 * 
	 * To add a mark listener to the cell marker:
	 * 
	 * <code>
	 * addListener(
	 *   mxCellMarker.EVENT_MARK, new mxEventListener()
	 *   {
	 *     public void invoke(Object source, Object[] args)
	 *     {
	 *       cellMarked((mxCellMarker) source, (mxCellState) args[0]);
	 *     }
	 *   });
	 * </code>
	 */
	public static String EVENT_MARK = "mark";

	/**
	 * Specifies the default stroke for the marker.
	 */
	public static Stroke DEFAULT_STROKE = new BasicStroke(3);

	/**
	 * Holds the event source.
	 */
	protected mxEventSource eventSource = new mxEventSource(this);

	/**
	 * Holds the enclosing graph component.
	 */
	protected mxGraphComponent graphComponent;

	/**
	 * Specifies if the marker is enabled. Default is true.
	 */
	protected boolean enabled = true;

	/**
	 * Specifies the portion of the width and height that should trigger
	 * a highlight. The area around the center of the cell to be marked is used
	 * as the hotspot. Possible values are between 0 and 1. Default is
	 * mxConstants.DEFAULT_HOTSPOT.
	 */
	protected double hotspot;

	/**
	 * Specifies if the hotspot is enabled. Default is false.
	 */
	protected boolean hotspotEnabled = false;

	/**
	 * Specifies if the the content area of swimlane should be non-transparent
	 * to mouse events. Default is false.
	 */
	protected boolean swimlaneContentEnabled = false;

	/**
	 * Specifies the valid- and invalidColor for the marker.
	 */
	protected Color validColor, invalidColor;

	/**
	 * Holds the current marker color.
	 */
	protected transient Color currentColor;

	/**
	 * Holds the marked state if it is valid.
	 */
	protected transient mxCellState validState;

	/**
	 * Holds the marked state.
	 */
	protected transient mxCellState markedState;

	/**
	 * Constructs a new marker for the given graph component.
	 * 
	 * @param graphComponent
	 */
	public mxCellMarker(mxGraphComponent graphComponent)
	{
		this(graphComponent, mxConstants.DEFAULT_VALID_COLOR);
	}

	/**
	 * Constructs a new marker for the given graph component.
	 */
	public mxCellMarker(mxGraphComponent graphComponent, Color validColor)
	{
		this(graphComponent, validColor, mxConstants.DEFAULT_INVALID_COLOR);
	}

	/**
	 * Constructs a new marker for the given graph component.
	 */
	public mxCellMarker(mxGraphComponent graphComponent, Color validColor,
			Color invalidColor)
	{
		this(graphComponent, validColor, invalidColor,
				mxConstants.DEFAULT_HOTSPOT);
	}

	/**
	 * Constructs a new marker for the given graph component.
	 */
	public mxCellMarker(mxGraphComponent graphComponent, Color validColor,
			Color invalidColor, double hotspot)
	{
		this.graphComponent = graphComponent;
		this.validColor = validColor;
		this.invalidColor = invalidColor;
		this.hotspot = hotspot;
	}

	/**
	 * Sets the enabled state of the marker.
	 */
	public void setEnabled(boolean enabled)
	{
		this.enabled = enabled;
	}

	/**
	 * Returns true if the marker is enabled, that is, if it processes events
	 * in process.
	 */
	public boolean isEnabled()
	{
		return enabled;
	}

	/**
	 * Sets the hotspot.
	 */
	public void setHotspot(double hotspot)
	{
		this.hotspot = hotspot;
	}

	/**
	 * Returns the hotspot.
	 */
	public double getHotspot()
	{
		return hotspot;
	}

	/**
	 * Specifies whether the hotspot should be used in intersects.
	 */
	public void setHotspotEnabled(boolean enabled)
	{
		this.hotspotEnabled = enabled;
	}

	/**
	 * Returns true if hotspot is used in intersects.
	 */
	public boolean isHotspotEnabled()
	{
		return hotspotEnabled;
	}

	/**
	 * Sets if the content area of swimlanes should not be transparent to
	 * events.
	 */
	public void setSwimlaneContentEnabled(boolean swimlaneContentEnabled)
	{
		this.swimlaneContentEnabled = swimlaneContentEnabled;
	}

	/**
	 * Returns true if the content area of swimlanes is non-transparent to
	 * events.
	 */
	public boolean isSwimlaneContentEnabled()
	{
		return swimlaneContentEnabled;
	}

	/**
	 * Returns true if validState is not null.
	 */
	public boolean hasValidState()
	{
		return (validState != null);
	}

	/**
	 * Returns the valid state.
	 */
	public mxCellState getValidState()
	{
		return validState;
	}

	/**
	 * Returns the marked state.
	 */
	public mxCellState getMarkedState()
	{
		return markedState;
	}

	/**
	 * Resets the state of the cell marker.
	 */
	public void reset()
	{
		validState = null;

		if (markedState != null)
		{
			markedState = null;
			unmark();
		}
	}

	/**
	 * Processes the given event and marks the state returned by getStateAt
	 * with the color returned by getMarkerColor. If the markerColor is not
	 * null, then the state is stored in markedState. If isValidState returns
	 * true, then the state is stored in validState regardless of the marker
	 * color. The state is returned regardless of the marker color and
	 * valid state. 
	 */
	public mxCellState process(MouseEvent e)
	{
		mxCellState state = null;

		if (isEnabled())
		{
			state = getState(e);
			boolean isValid = (state != null) ? isValidState(state) : false;
			Color color = getMarkerColor(e, state, isValid);

			if (isValid)
			{
				validState = state;
			}
			else
			{
				validState = null;
			}

			if (state != markedState || color != currentColor)
			{
				currentColor = color;

				if (state != null && currentColor != null)
				{
					markedState = state;
					mark();
				}
				else if (markedState != null)
				{
					markedState = null;
					unmark();
				}
			}
		}

		return state;
	}

	/**
	 * Marks the markedState and fires a EVENT_MARK event.
	 */
	protected void mark()
	{
		if (markedState != null)
		{
			Rectangle bounds = markedState.getRectangle();
			bounds.grow(3, 3);
			bounds.width += 1;
			bounds.height += 1;
			setBounds(bounds);

			if (this.getParent() == null)
			{
				setVisible(true);
				graphComponent.getGraphControl().add(this);
			}

			repaint();
			eventSource.fireEvent(EVENT_MARK, new Object[] { markedState });
		}
	}

	/**
	 * Hides the marker and fires a EVENT_MARK event.
	 */
	protected void unmark()
	{
		if (this.getParent() != null)
		{
			setVisible(false);
			getParent().remove(this);
			eventSource.fireEvent(EVENT_MARK);
		}
	}

	/**
	 * Returns true if the given state is a valid state. If this returns true,
	 * then the state is stored in validState. The return value of this method
	 * is used as the argument for getMarkerColor.
	 */
	protected boolean isValidState(mxCellState state)
	{
		return true;
	}

	/**
	 * Returns the valid- or invalidColor depending on the value of isValid.
	 * The given state is ignored by this implementation.
	 */
	protected Color getMarkerColor(MouseEvent e, mxCellState state,
			boolean isValid)
	{
		return (isValid) ? validColor : invalidColor;
	}

	/**
	 * Uses getCell, getMarkedState and intersects to return the state for
	 * the given event.
	 */
	protected mxCellState getState(MouseEvent e)
	{
		mxGraphView view = graphComponent.getGraph().getView();
		Object cell = getCell(e);
		mxCellState state = getStateToMark(view.getState(cell));

		return (state != null && intersects(state, e)) ? state : null;
	}

	/**
	 * Returns the state at the given location. This uses mxGraph.getCellAt.
	 */
	protected Object getCell(MouseEvent e)
	{
		return graphComponent.getCellAt(e.getX(), e.getY(),
				swimlaneContentEnabled);
	}

	/**
	 * Returns the state to be marked for the given state under the mouse. This
	 * returns the given state.
	 */
	protected mxCellState getStateToMark(mxCellState state)
	{
		return state;
	}

	/**
	 * Returns true if the given mouse event intersects the given state. This
	 * returns true if the hotspot is 0 or the event is inside the hotspot for
	 * the given cell state.
	 */
	protected boolean intersects(mxCellState state, MouseEvent e)
	{
		if (hotspotEnabled && hotspot > 0)
		{
			int x = (int) state.getCenterX();
			int y = (int) state.getCenterY();
			int width = (int) state.getWidth();
			int height = (int) state.getHeight();

			int start = mxUtils.getInt(state.getStyle(),
					mxConstants.STYLE_STARTSIZE);

			if (start > 0)
			{
				if (mxUtils.isTrue(state.getStyle(),
						mxConstants.STYLE_HORIZONTAL, true))
				{
					y = (int) (state.getY() + start / 2);
					height = start;
				}
				else
				{
					x = (int) (state.getX() + start / 2);
					width = start;
				}
			}

			int w = (int) Math.max(mxConstants.MIN_HOTSPOT_SIZE, width
					* hotspot);
			int h = (int) Math.max(mxConstants.MIN_HOTSPOT_SIZE, height
					* hotspot);

			if (mxConstants.MAX_HOTSPOT_SIZE > 0)
			{
				w = Math.min(w, mxConstants.MAX_HOTSPOT_SIZE);
				h = Math.min(h, mxConstants.MAX_HOTSPOT_SIZE);
			}

			Rectangle rect = new Rectangle((int) (x - w / 2),
					(int) (y - h / 2), w, h);

			return rect.contains(e.getPoint());
		}

		return true;
	}

	/**
	 * @param eventName
	 * @param listener
	 * @see com.mxgraph.util.mxEventSource#addListener(java.lang.String, com.mxgraph.util.mxEventSource.mxIEventListener)
	 */
	public void addListener(String eventName, mxIEventListener listener)
	{
		eventSource.addListener(eventName, listener);
	}

	/**
	 * @param listener
	 * @see com.mxgraph.util.mxEventSource#removeListener(com.mxgraph.util.mxEventSource.mxIEventListener)
	 */
	public void removeListener(mxIEventListener listener)
	{
		eventSource.removeListener(listener);
	}

	/**
	 * @param eventName
	 * @param listener
	 * @see com.mxgraph.util.mxEventSource#removeListener(java.lang.String, com.mxgraph.util.mxEventSource.mxIEventListener)
	 */
	public void removeListener(mxIEventListener listener, String eventName)
	{
		eventSource.removeListener(listener, eventName);
	}

	/**
	 * Paints the outline of the markedState with the currentColor.
	 */
	public void paint(Graphics g)
	{
		if (markedState != null && currentColor != null)
		{
			((Graphics2D) g).setStroke(DEFAULT_STROKE);
			g.setColor(currentColor);

			if (markedState.getAbsolutePointCount() > 0)
			{
				Point last = markedState.getAbsolutePoint(0).getPoint();

				for (int i = 1; i < markedState.getAbsolutePointCount(); i++)
				{
					Point current = markedState.getAbsolutePoint(i).getPoint();
					g.drawLine(last.x - getX(), last.y - getY(), current.x
							- getX(), current.y - getY());
					last = current;
				}
			}
			else
			{
				g.drawRect(1, 1, getWidth() - 3, getHeight() - 3);
			}
		}
	}

}
