%-*- Prolog -*-
:- module(middle_chinese, [initial/3, initial/1, transcript/3,
			   palatal/1, labial/1]).

% Transcriptions             Initials
% p   ph   b   m             labials
% t   th   d   n             dentals
%                     l      lateral
% tr  trh  dr  nr            retroflex stops
% ts  tsh  dz      s  z      dental sibilants
% tsr tsrh dzr     sr zr     retroflex sibilants
% tsy tsyh dzy ny  sy zy y   palatals
% k   kh   g   ng            velars
% ?                x  h      laryngeals

% 聲母            聲類
% 幫 滂 並 明     脣音
% 端 透 定 泥     舌頭音
% 知 徹 澄 孃     舌上音
% 精 清 從 心 邪  齒頭音
% 莊 初 崇 生 俟  正齒音莊組 
% 章 昌 常 書 船  正齒音章組 
% 見 溪 羣 疑     牙音
% 曉 匣 影 于 以  喉音 
% 來              半舌音 
% 日              半齒音 

% initial(Transcription, TraditionalName, Type)
initial('p',    '幫', 'P').  % Labials 唇音
initial('ph',   '滂', 'P').
initial('b',    '並', 'P').
initial('m',    '明', 'P').
initial('t',    '端', 'T').  % Dentals 舌頭音
initial('th',   '透', 'T').
initial('d',    '定', 'T').
initial('n',    '泥', 'T').
initial('l',    '來', 'l').  % Laterals 半舌音
initial('tr',   '知', 'Tr'). % Retroflex stops 舌上音
initial('trh',  '徹', 'Tr').
initial('dr',   '澄', 'Tr').
initial('nr',   '娘', 'Tr').     /* 孃 */
initial('ts',   '精', 'TS'). % Dental sibilants 齒頭音
initial('tsh',  '清', 'TS').
initial('dz',   '從', 'TS').
initial('s',    '心', 'TS').
initial('z',    '邪', 'TS').
initial('tsr',  '莊', 'TSr'). % Retroflex sibilants 正齒音莊組(二等)
initial('tsrh', '初', 'TSr').
initial('dzr',  '崇', 'TSr').
initial('sr',   '生', 'TSr').
initial('zr',   '俟', 'TSr').       /* no standard name */
initial('tsy',  '章', 'TSy'). % Palatal sibilants 正齒音章組(三等)
initial('tsyh', '昌', 'TSy').
initial('dzy',  '禪', 'TSy').       /* 常? */
initial('sy',   '書', 'TSy').
initial('zy',   '船', 'TSy').
initial('k',    '見', 'K'). % Velars 牙音
initial('kh',   '溪', 'K').
initial('g',    '群', 'K').      /* 羣 */
initial('ng',   '疑', 'K').
initial('?',    '影', 'K'). % Laryngeals 喉音
initial('x',    '曉', 'K').
initial('h',    '匣', 'K').
initial('y',    '喩', 'y').
%initial('hj',  '云', 'y').     /* 于? */
%initial('y',   '以', 'y').
initial('ny',  '日', 'TSy').      /* Type? */

initial(X) :- initial(X, _, _).
% initial(X) :-
%     sub_atom(X, _, _, 0, 'w'), !,
%     atom_length(X, XLength),
%     XL is XLength - 1,
%     sub_atom(X, 0, XL, XL, XPrefix),
%     initial(XPrefix, _, _).

% transcript(Name, Pinyin, EMC)
transcript('幫', 'bang1', 'pang'). % Labials 唇音
transcript('滂', 'pang1', 'phang').
transcript('並', 'bing4', 'bengX').
transcript('明', 'ming2', 'mjaeng').
transcript('端', 'duan1', 'twan'). % Dental 舌頭音
transcript('透', 'tou4',  'thuwH').
transcript('定', 'ding4', 'dengH').
transcript('泥', 'ni2',   'nej').
transcript('來', 'lai2', 'loj').   % Lateral 半舌音
transcript('', '', '').

labial(X) :-             initial(X, _, 'P').
dental(X) :-             initial(X, _, 'T').
lateral(X) :-            initial(X, _, 'l').
retroflex_stop(X) :-     initial(X, _, 'Tr').
dental_sibilant(X) :-    initial(X, _, 'TS').
retroflex_sibilant(X) :- initial(X, _, 'TSr').
palatal(X) :-            initial(X, _, 'TSy').
palatal(X) :-            initial(X, _, 'y').  /* ? */
velar(X) :-              initial(X, _, 'K').
laryngeal(X) :-          initial(X, _, '').
