﻿using UnityEngine;
using UnityEditor;
using System.Collections;
using System.Collections.Generic;
using System.Text;


namespace MMD
{
	namespace PMD
	{
		public class PMDConverter
		{
			Vector3[] EntryVertices(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				Vector3[] vpos = new Vector3[vcount];
				for (int i = 0; i < vcount; i++)
					vpos[i] = format.vertex_list.vertex[i].pos;
				return vpos;
			}
			
			Vector3[] EntryNormals(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				Vector3[] normals = new Vector3[vcount];
				for (int i = 0; i < vcount; i++)
					normals[i] = format.vertex_list.vertex[i].normal_vec;
				return normals;
			}
			
			Vector2[] EntryUVs(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				Vector2[] uvs = new Vector2[vcount];
				for (int i = 0; i < vcount; i++)
					uvs[i] = format.vertex_list.vertex[i].uv;
				return uvs;
			}
			
			BoneWeight[] EntryBoneWeights(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				BoneWeight[] weights = new BoneWeight[vcount];
				for (int i = 0; i < vcount; i++)
				{
					weights[i].boneIndex0 = (int)format.vertex_list.vertex[i].bone_num[0];
					weights[i].boneIndex1 = (int)format.vertex_list.vertex[i].bone_num[1];
					weights[i].weight0 = format.vertex_list.vertex[i].bone_weight;
					weights[i].weight1 = 100 - format.vertex_list.vertex[i].bone_weight;
				}
				return weights;
			}
			
			// 頂点座標やUVなどの登録だけ
			void EntryAttributesForMesh(PMD.PMDFormat format, Mesh mesh)
			{
				//mesh.vertexCount = (int)format.vertex_list.vert_count;
				mesh.vertices = EntryVertices(format);
				mesh.normals = EntryNormals(format);
				mesh.uv = EntryUVs(format);
				mesh.boneWeights = EntryBoneWeights(format);
			}
			
			void SetSubMesh(PMD.PMDFormat format, Mesh mesh)
			{
				// マテリアル対サブメッシュ
				// サブメッシュとはマテリアルに適用したい面頂点データのこと
				// 面ごとに設定するマテリアルはここ
				mesh.subMeshCount = (int)format.material_list.material_count;
				
				int sum = 0;
				for (int i = 0; i < mesh.subMeshCount; i++)
				{
					int count = (int)format.material_list.material[i].face_vert_count;
					int[] indices = new int[count*3];
					
					// 面頂点は材質0から順番に加算されている
					for (int j = 0; j < count; j++)
						indices[j] = format.face_vertex_list.face_vert_index[j+sum];
					mesh.SetTriangles(indices, i);
					sum += (int)format.material_list.material[i].face_vert_count;
				}
			}
			
			// メッシュをProjectに登録
			void CreateAssetForMesh(PMD.PMDFormat format, Mesh mesh)
			{
				AssetDatabase.CreateAsset(mesh, format.folder + "/" + format.name + ".asset");
			}
			
			public Mesh CreateMesh(PMD.PMDFormat format)
			{
				Mesh mesh = new Mesh();
				EntryAttributesForMesh(format, mesh);
				SetSubMesh(format, mesh);
				CreateAssetForMesh(format, mesh);
				return mesh;
			}
			
			void EntryColors(PMD.PMDFormat format, Material[] mats)
			{
				for (int i = 0; i < mats.Length; i++)
				{
					string[] buf = format.material_list.material[i].texture_file_name.Split('.');
					if (buf[buf.Length-1] == "tga")
						mats[i] = new Material(Shader.Find("Transparent/Diffuse"));
					else
						mats[i] = new Material(Shader.Find("Diffuse"));
					mats[i].color = format.material_list.material[i].diffuse_color;
					
					// テクスチャが空でなければ登録
					if (format.material_list.material[i].texture_file_name != "") {
						mats[i].mainTexture = AssetDatabase.LoadAssetAtPath(format.folder + "/" + format.material_list.material[i].texture_file_name, typeof(Texture)) as Texture;
						mats[i].mainTextureScale = new Vector2(1, -1);
					}
				}
			}
			
			// マテリアルに必要な色などを登録
			Material[] EntryAttributesForMaterials(PMD.PMDFormat format)
			{
				int count = (int)format.material_list.material_count;
				Material[] mats = new Material[count];
				EntryColors(format, mats);
				return mats;
			}
			
			// マテリアルの登録
			void CreateAssetForMaterials(PMD.PMDFormat format, Material[] mats)
			{
				// 適当なフォルダに投げる
				string path = format.folder + "/Materials/";
				AssetDatabase.CreateFolder(format.folder, "Materials");
				
				for (int i = 0; i < mats.Length; i++)
				{
					string fname = path + format.name + "_material" + i + ".asset";
					AssetDatabase.CreateAsset(mats[i], fname);
				}
			}
			
			public Material[] CreateMaterials(PMD.PMDFormat format)
			{
				Material[] materials;
				materials = EntryAttributesForMaterials(format);
				CreateAssetForMaterials(format, materials);
				return materials;
			}
			
			// 親子関係の構築
			void AttachParentsForBone(PMD.PMDFormat format, GameObject[] bones)
			{
				for (int i = 0; i < bones.Length; i++)
				{
					int index = format.bone_list.bone[i].parent_bone_index;
					if (index != 0xFFFF)
						bones[i].transform.parent = bones[index].transform;
					else
						bones[i].transform.parent = format.caller.transform;
				}
			}

			// ボーンの位置決めや親子関係の整備など
			GameObject[] EntryAttributeForBones(PMD.PMDFormat format)
			{
				int count = format.bone_list.bone_count;
				GameObject[] bones = new GameObject[count];
				
				for (int i = 0; i < count; i++) {
					bones[i] = new GameObject(format.bone_list.bone[i].bone_name);
					bones[i].transform.position = format.bone_list.bone[i].bone_head_pos;
				}
				return bones;
			}
			
			public GameObject[] CreateBones(PMD.PMDFormat format)
			{
				GameObject[] bones;
				bones = EntryAttributeForBones(format);
				AttachParentsForBone(format, bones);
				return bones;
			}
			
			// バインドポーズの作成
			public void BuildingBindpose(PMD.PMDFormat format, Mesh mesh, Material[] materials, GameObject[] bones)
			{
				// 行列とかトランスフォームとか
				Matrix4x4[] bindpose = new Matrix4x4[bones.Length];
				Transform[] trans = new Transform[bones.Length];
				for (int i = 0; i < bones.Length; i++) {
					trans[i] = bones[i].transform;
					bindpose[i] = bones[i].transform.worldToLocalMatrix;
				}
				
				// ここで本格的な適用
				SkinnedMeshRenderer smr = format.caller.AddComponent<SkinnedMeshRenderer>() as SkinnedMeshRenderer;
				mesh.bindposes = bindpose;
				smr.sharedMesh = mesh;
				smr.bones = trans;
				smr.materials = materials;
			}
		}
	}
}

/*
namespace MMD
{
	namespace PMD
	{
		public class PMDBurner
		{
			static Vector3 EntryVector3(float[] arr)
			{
				return new Vector3(arr[0], arr[1], arr[2]);
			}
			
			static Vector2 EntryVector2(float[] arr)
			{
				return new Vector2(arr[0], arr[1]);
			}
			
			static void CreateVertex(MMD.PMD.PMDFormat format, Mesh mesh)
			{
				// 頂点の登録
				int vcount = (int)format.vertex_list.vert_count;
				Vector3[] vpos = new Vector3[vcount];
				for (int i = 0; i < vcount; i++)
					vpos[i] = EntryVector3(format.vertex_list.vertex[i].pos);
				
				mesh.vertices = vpos;
			}
			
			static void CreateUV(MMD.PMD.PMDFormat format, Mesh mesh)
			{
				// UVの登録
				int uvcount = (int)format.vertex_list.vert_count;
				Vector2[] uv = new Vector2[uvcount];
				for (int i = 0; i < uvcount; i++)
					uv[i] = EntryVector2(format.vertex_list.vertex[i].uv);
				
				mesh.uv = uv;
			}
			
			static void CreateNormalVector(MMD.PMD.PMDFormat format, Mesh mesh)
			{
				// 法線の登録
				int vcount = (int)format.vertex_list.vert_count;
				Vector3[] nml = new Vector3[vcount];
				for (int i = 0; i < vcount; i++)
					nml[i] = EntryVector3(format.vertex_list.vertex[i].normal_vec);
				
				mesh.normals = nml;
			}
			
			static void CreateBoneWeight(MMD.PMD.PMDFormat format, Mesh mesh)
			{
				// ウェイト値の登録
				int vcount = (int)format.vertex_list.vert_count;
				BoneWeight[] weights = new BoneWeight[vcount];
				for (int i = 0; i < vcount; i++)
				{
					weights[i].boneIndex0 = format.vertex_list.vertex[i].bone_num[0];
					weights[i].boneIndex1 = format.vertex_list.vertex[i].bone_num[1];
					weights[i].weight0 = format.vertex_list.vertex[i].bone_weight;
					weights[i].weight1 = 100 - format.vertex_list.vertex[i].bone_weight;
				}
				
				mesh.boneWeights = weights;
			}
			
			static void CreateSubMesh(MMD.PMD.PMDFormat format, Mesh mesh)
			{
				// 面を張る
				int vcount = (int)format.face_vertex_list.face_vert_count/3;
				mesh.subMeshCount = vcount;	// 面の数を登録
				
				for (int i = 0; i < vcount; i++)
				{
					int[] triangle = new int[3];
					for (int j = 0; j < 3; j++)
						triangle[j] = (int)format.face_vertex_list.face_vert_index[i*3+j];
					mesh.SetTriangles(triangle, i);	// ここで面を張る
				}
				Debug.Log("SubMeshCount:" + mesh.subMeshCount);
			}
			
			// メッシュの生成
			static public Mesh BurnMesh(MMD.PMD.PMDFormat format)
			{
				Mesh mesh = new Mesh();
				CreateVertex(format, mesh);
				CreateUV(format, mesh);
				CreateNormalVector(format, mesh);
				CreateBoneWeight(format, mesh);
				CreateSubMesh(format, mesh);
				string mesh_name = format.path + "_mesh.asset";
				AssetDatabase.CreateAsset(mesh, mesh_name);		// Projectに登録
				return mesh;
			}
			
			// フォルダ名+ファイル名のパスを作成
			static string CreateObjectPath(string master_path, string file_name)
			{
				string buffer = "";
				string[] arr = master_path.Split('/');
				for (int i = 0; i < arr.Length-1; i++) buffer += (arr[i] + "/");
				return buffer + file_name;
			}
			
			// テクスチャの指定
			static void SetMaterialTexture(Material material, string path)
			{
				material.mainTexture = (Texture)AssetDatabase.LoadAssetAtPath(path, typeof(Texture));
			}
			
			// マテリアル色の指定
			static void SetMaterialColor(Material material, float[] diffuse_color, float alpha)
			{
				material.color = new Color(diffuse_color[0], diffuse_color[1], diffuse_color[2], alpha);
			}
			
			// Projectにアセットの登録
			static void CreateAssetForMaterial(Material material, string path)
			{
				path += ".mat";
				AssetDatabase.CreateAsset(material, path);
			}
			
			static void CreateMaterialAttribute(MMD.PMD.PMDFormat format, Material[] materials)
			{
				// ここでマテリアルごとに色なりSpecularityなどを設定する
				for (int i = 0; i < materials.Length; i++)
				{
					// マテリアルの名前を作る。
					// ex. 初音ミク_material1
					string subject = "_material" + i;
					string material_name = format.name + subject;
					materials[i] = new Material(Shader.Find("Diffuse"));
					
					// ファイルの名前がおかしいときはテクスチャを使ってない
					if (format.material_list.material[i].texture_file_name.Length > 0)
					{
						string path = CreateObjectPath(format.path, format.material_list.material[i].texture_file_name);
						SetMaterialTexture(materials[i], path);
					}
					SetMaterialColor(materials[i], format.material_list.material[i].diffuse_color, format.material_list.material[i].alpha);
					CreateAssetForMaterial(materials[i], format.path + subject);// パス+サブジェクトで登録する
				}
			}
			
			// マテリアルの移植＋アセットの登録
			// 完成したアセットが返ってくる
			static public Material[] BurnMaterial(MMD.PMD.PMDFormat format)
			{
				Material[] materials = new Material[format.material_list.material_count];
				CreateMaterialAttribute(format, materials);
				return materials;
			}
			
			static GameObject CreateGameObjectForBone(string bone_name, Material[] materials)
			{
				// ゲームオブジェクトの作成
				GameObject go = new GameObject(bone_name);
				go.AddComponent<SkinnedMeshRenderer>();
				
				go.renderer.materials = materials;
				return go;
			}
			
			static void SetGameObjectPosition(GameObject go, float[] pos)
			{
				// ボーンの位置決め
				go.transform.position = new Vector3(pos[0], pos[1], pos[2]);
			}
			
			// ボーンの親子関係を構築
			static void SetBoneParent(GameObject[] source, PMD.PMDFormat format)
			{
				for (int dest = 0; dest < source.Length; dest++)
				{
					for (int parent = 0; parent < source.Length; parent++)
					{
						if (parent == dest) continue;
						if (format.bone_list.bone[dest].parent_bone_index == parent)	// 走査対象と親番号が一致すれば
						{
							// 親は必ず一つ
							source[dest].transform.parent = source[parent].transform;
							break;
						}
					}
				}
			}
									
			// ボーンの生成
			static public GameObject[] BurnBone(MMD.PMD.PMDFormat format, Material[] materials)
			{
				// GameObjectを一つ一つ生成
				GameObject[] gos = new GameObject[format.bone_list.bone_count];
				
				for (int i = 0; i < format.bone_list.bone_count; i++)
				{
					string bone_name = format.bone_list.bone[i].bone_name;
					float[] bone_pos = format.bone_list.bone[i].bone_head_pos;
					
					gos[i] = CreateGameObjectForBone(bone_name, materials);
					SetGameObjectPosition(gos[i], bone_pos);
				}
				SetBoneParent(gos, format);
				return gos;
			}
			
			// バインドポーズするための行列を生成
			static Matrix4x4[] CreateBindPoseMatrix(GameObject[] bones)
			{
				Matrix4x4[] mat = new Matrix4x4[bones.Length];
				for (int i = 0; i < bones.Length; i++)
					mat[i] = bones[i].transform.localToWorldMatrix;
				return mat;
			}
			
			// shared_meshの設定
			static void SetSharedMesh(Mesh mesh, GameObject[] bones)
			{
				for (int i = 0; i < bones.Length; i++)
				{
					SkinnedMeshRenderer render = bones[i].renderer as SkinnedMeshRenderer;
					render.sharedMesh = mesh;
					
					Transform[] transform = new Transform[bones.Length];
					for (int j = 0; j < bones.Length; j++)
						transform[i] = bones[i].transform;
					render.bones = transform;
				}
			}
			
			// バインドポーズの構築
			static public void BurnBindpose(MMD.PMD.PMDFormat format, Mesh mesh, GameObject[] bones)
			{
				mesh.bindposes = CreateBindPoseMatrix(bones);
				SetSharedMesh(mesh, bones);
			}
			
			// プレハブのパスを作成
			static string CreatePrefabName(string path)
			{
				string[] prefab_path = path.Split('.');
				string comp = "";
				for (int i = 0; i < prefab_path.Length-1; i++)
					comp += prefab_path[i];
				return comp + ".prefab";
			}
			
			// プレハブの作成
			static public void CreatePrefab(string path, GameObject[] bones)
			{
				path = CreatePrefabName(path);
				Object obj = EditorUtility.CreateEmptyPrefab(path);
				
				for (int i = 0; i < bones.Length; i++)
					EditorUtility.ReplacePrefab(bones[i], obj);
				//EditorUtility.ReplacePrefab(ddd, obj);
			}
		}
	}
}
*/