﻿using UnityEngine;
using UnityEditor;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System;


namespace MMD
{
	namespace PMD
	{
		public class PMDConverter
		{
			Vector3[] EntryVertices(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				Vector3[] vpos = new Vector3[vcount];
				for (int i = 0; i < vcount; i++)
					vpos[i] = format.vertex_list.vertex[i].pos;
				return vpos;
			}
			
			Vector3[] EntryNormals(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				Vector3[] normals = new Vector3[vcount];
				for (int i = 0; i < vcount; i++)
					normals[i] = format.vertex_list.vertex[i].normal_vec;
				return normals;
			}
			
			Vector2[] EntryUVs(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				Vector2[] uvs = new Vector2[vcount];
				for (int i = 0; i < vcount; i++)
					uvs[i] = format.vertex_list.vertex[i].uv;
				return uvs;
			}
			
			BoneWeight[] EntryBoneWeights(PMD.PMDFormat format)
			{
				int vcount = (int)format.vertex_list.vert_count;
				BoneWeight[] weights = new BoneWeight[vcount];
				for (int i = 0; i < vcount; i++)
				{
					weights[i].boneIndex0 = (int)format.vertex_list.vertex[i].bone_num[0];
					weights[i].boneIndex1 = (int)format.vertex_list.vertex[i].bone_num[1];
					weights[i].weight0 = format.vertex_list.vertex[i].bone_weight;
					weights[i].weight1 = 100 - format.vertex_list.vertex[i].bone_weight;
				}
				return weights;
			}
			
			// 頂点座標やUVなどの登録だけ
			void EntryAttributesForMesh(PMD.PMDFormat format, Mesh mesh)
			{
				//mesh.vertexCount = (int)format.vertex_list.vert_count;
				mesh.vertices = EntryVertices(format);
				mesh.normals = EntryNormals(format);
				mesh.uv = EntryUVs(format);
				mesh.boneWeights = EntryBoneWeights(format);
			}
			
			void SetSubMesh(PMD.PMDFormat format, Mesh mesh)
			{
				// マテリアル対サブメッシュ
				// サブメッシュとはマテリアルに適用したい面頂点データのこと
				// 面ごとに設定するマテリアルはここ
				mesh.subMeshCount = (int)format.material_list.material_count;
				
				int sum = 0;
				for (int i = 0; i < mesh.subMeshCount; i++)
				{
					int count = (int)format.material_list.material[i].face_vert_count;
					int[] indices = new int[count*3];
					
					// 面頂点は材質0から順番に加算されている
					for (int j = 0; j < count; j++)
						indices[j] = format.face_vertex_list.face_vert_index[j+sum];
					mesh.SetTriangles(indices, i);
					sum += (int)format.material_list.material[i].face_vert_count;
				}
			}
			
			// メッシュをProjectに登録
			void CreateAssetForMesh(PMD.PMDFormat format, Mesh mesh)
			{
				AssetDatabase.CreateAsset(mesh, format.folder + "/" + format.name + ".asset");
			}
			
			public Mesh CreateMesh(PMD.PMDFormat format)
			{
				Mesh mesh = new Mesh();
				EntryAttributesForMesh(format, mesh);
				SetSubMesh(format, mesh);
				CreateAssetForMesh(format, mesh);
				return mesh;
			}
			
			void EntryColors(PMD.PMDFormat format, Material[] mats, bool cutoutFlag)
			{
				// マテリアルの生成 
				for (int i = 0; i < mats.Length; i++)
				{
					// PMDフォーマットのマテリアルを取得 
					PMD.PMDFormat.Material pmdMat = format.material_list.material[i];
					
					string[] buf = pmdMat.texture_file_name.Split('.');
					if (buf[buf.Length-1] == "tga") {
						if (cutoutFlag)
							mats[i] = new Material(Shader.Find("Custom/CharModel"));
						else
							mats[i] = new Material(Shader.Find("Transparent/Diffuse"));
					}
					else
						if (cutoutFlag)
							mats[i] = new Material(Shader.Find("Custom/CharModel"));
						else
							mats[i] = new Material(Shader.Find("Diffuse"));
					
					mats[i].color = pmdMat.diffuse_color;
					
					if (cutoutFlag) {
						mats[i].SetFloat("_Cutoff", 1-pmdMat.alpha);
					}
					
					// テクスチャが空でなければ登録
					if (pmdMat.texture_file_name != "") {
						string path = format.folder + "/" + pmdMat.texture_file_name;
						mats[i].mainTexture = AssetDatabase.LoadAssetAtPath(path, typeof(Texture)) as Texture;
						mats[i].mainTextureScale = new Vector2(1, -1);
					}
				}
			}
			
			// マテリアルに必要な色などを登録
			Material[] EntryAttributesForMaterials(PMD.PMDFormat format, bool cutoutFlag)
			{
				int count = (int)format.material_list.material_count;
				Material[] mats = new Material[count];
				EntryColors(format, mats, cutoutFlag);
				return mats;
			}
			
			// マテリアルの登録
			void CreateAssetForMaterials(PMD.PMDFormat format, Material[] mats)
			{
				// 適当なフォルダに投げる
				string path = format.folder + "/Materials/";
				AssetDatabase.CreateFolder(format.folder, "Materials");
				
				for (int i = 0; i < mats.Length; i++)
				{
					string fname = path + format.name + "_material" + i + ".asset";
					AssetDatabase.CreateAsset(mats[i], fname);
				}
			}
			
			public Material[] CreateMaterials(PMD.PMDFormat format, bool cutoutFlag)
			{
				Material[] materials;
				materials = EntryAttributesForMaterials(format, cutoutFlag);
				CreateAssetForMaterials(format, materials);
				return materials;
			}
			
			// 親子関係の構築
			void AttachParentsForBone(PMD.PMDFormat format, GameObject[] bones)
			{
				for (int i = 0; i < bones.Length; i++)
				{
					int index = format.bone_list.bone[i].parent_bone_index;
					if (index != 0xFFFF)
						bones[i].transform.parent = bones[index].transform;
					else
						bones[i].transform.parent = format.caller.transform;
				}
			}

			// ボーンの位置決めや親子関係の整備など
			GameObject[] EntryAttributeForBones(PMD.PMDFormat format)
			{
				int count = format.bone_list.bone_count;
				GameObject[] bones = new GameObject[count];
				
				for (int i = 0; i < count; i++) {
					bones[i] = new GameObject(format.bone_list.bone[i].bone_name);
					bones[i].transform.name = bones[i].name;
					bones[i].transform.position = format.bone_list.bone[i].bone_head_pos;
				}
				return bones;
			}
			
			public GameObject[] CreateBones(PMD.PMDFormat format)
			{
				GameObject[] bones;
				bones = EntryAttributeForBones(format);
				AttachParentsForBone(format, bones);
				return bones;
			}
			
			// バインドポーズの作成
			public void BuildingBindpose(PMD.PMDFormat format, Mesh mesh, Material[] materials, GameObject[] bones)
			{
				// 行列とかトランスフォームとか
				Matrix4x4[] bindpose = new Matrix4x4[bones.Length];
				Transform[] trans = new Transform[bones.Length];
				for (int i = 0; i < bones.Length; i++) {
					trans[i] = bones[i].transform;
					bindpose[i] = bones[i].transform.worldToLocalMatrix;
				}
				
				// ここで本格的な適用
				SkinnedMeshRenderer smr = format.caller.AddComponent<SkinnedMeshRenderer>() as SkinnedMeshRenderer;
				mesh.bindposes = bindpose;
				smr.sharedMesh = mesh;
				smr.bones = trans;
				smr.materials = materials;
			}
			
			IKSolverScript[] CreateIKBoneList(PMD.PMDFormat.IKList ik_list, GameObject[] bones)
			{
				IKSolverScript[] solvers = new IKSolverScript[ik_list.ik_data_count];
				
				// IKの数だけ回す
				for (int i = 0; i < ik_list.ik_data_count; i++)
				{
					// IKについた子の数だけ回す
					PMD.PMDFormat.IK ik = ik_list.ik_data[i];
					
					// 子ボーンの取得
					GameObject[] childs = new GameObject[ik.ik_chain_length];
					for (int j = 0; j < ik.ik_chain_length; j++)
						childs[j] = bones[ik.ik_child_bone_index[j]];
					
					// コンポーネントを追加しつつ、根本ボーンの登録
					solvers[i] = bones[ik_list.ik_data[i].ik_bone_index].AddComponent<IKSolverScript>();
					IKSolverScript.SetAttribute(solvers[i], bones[ik.ik_target_bone_index], childs, ik.iterations, ik.control_weight);
				}
				return solvers;
			}
			
			// IKの登録
			// IKは基本的にスクリプトを利用
			public void EntryIKSolver(PMD.PMDFormat format, GameObject[] bones)
			{
				PMD.PMDFormat.IKList ik_list = format.ik_list;
				
				// 悪さする可能性があるのでIKはOFF 
				//IKSolverScript[] solvers = CreateIKBoneList(ik_list, bones);
			}
		}
	}
	
	namespace VMD
	{
		public class VMDConverter
		{
			// tangentを求める
			float GetTangent(byte[] interpolation, int type, int ab)
			{
				// 0=X, 1=Y, 2=Z, 3=R
				// abはa?かb?のどちらを使いたいか
				Vector2 itpl = new Vector2((float)interpolation[ab*8+type], (float)interpolation[ab*8+4+type]);
				if (ab == 1) {
					// わからんけどマイナスになるので直す
					itpl.x = -(itpl.x-128f);
					itpl.y = -(itpl.y-128f);
				}
				itpl.Normalize();
				
				//Vector2 r = Vector2.right;
				//return (Vector2.Dot(itpl, r) * Mathf.PI) * 180.0f / Mathf.PI;
				return 0;
			}
			
			// あるボーンに含まれるキーフレを抽出
			// これは回転のみ
			void CreateKeysForRotation(MMD.VMD.VMDFormat format, AnimationClip clip, string current_bone, string bone_path)
			{
				try 
				{
					List<MMD.VMD.VMDFormat.Motion> mlist = format.motion_list.motion[current_bone];
				
					Keyframe[] rx_keys = new Keyframe[mlist.Count];
					Keyframe[] ry_keys = new Keyframe[mlist.Count];
					Keyframe[] rz_keys = new Keyframe[mlist.Count];
					Keyframe[] rw_keys = new Keyframe[mlist.Count];
					for (int i = 0; i < mlist.Count; i++)
					{
						const float tick_time = 1.0f / 30.0f;
						float tick = mlist[i].flame_no * tick_time;
						float a = GetTangent(mlist[i].interpolation, 3, 0);	// inTangent
						float b;	// outTangent
						
						// -1フレにはoutTangentは存在しないのでcatch
						try { b = GetTangent(mlist[i-1].interpolation, 3, 1); }
						catch { b = 0; }
						rx_keys[i] = new Keyframe(tick, mlist[i].rotation.x, a, b);
						ry_keys[i] = new Keyframe(tick, mlist[i].rotation.y, a, b);
						rz_keys[i] = new Keyframe(tick, mlist[i].rotation.z, a, b);
						rw_keys[i] = new Keyframe(tick, mlist[i].rotation.w, a, b);
					}
				
					AnimationCurve curve_x = new AnimationCurve(rx_keys);
					AnimationCurve curve_y = new AnimationCurve(ry_keys);
					AnimationCurve curve_z = new AnimationCurve(rz_keys);
					AnimationCurve curve_w = new AnimationCurve(rw_keys);
				
					// ここで回転クォータニオンをセット
					clip.SetCurve(bone_path, typeof(Transform), "localRotation.x", curve_x);
					clip.SetCurve(bone_path, typeof(Transform), "localRotation.y", curve_y);
					clip.SetCurve(bone_path, typeof(Transform), "localRotation.z", curve_z);
					clip.SetCurve(bone_path, typeof(Transform), "localRotation.w", curve_w);
				}
				catch { /* 互換性のないボーンが読み込まれた */ }
			}
			
			// 移動のみの抽出
			void CreateKeysForLocation(MMD.VMD.VMDFormat format, AnimationClip clip, string current_bone, string bone_path)
			{
				try
				{
					List<MMD.VMD.VMDFormat.Motion> mlist = format.motion_list.motion[current_bone];
				
					List<Keyframe> lx_keys = new List<Keyframe>();
					List<Keyframe> ly_keys = new List<Keyframe>();
					List<Keyframe> lz_keys = new List<Keyframe>();
					for (int i = 0; i < mlist.Count; i++)
					{
						const float tick_time = 1.0f / 30.0f;
					
						// 0だけのいらんデータを捨てる
						if (Vector3.zero != mlist[i].location)
						{
							float tick = mlist[i].flame_no * tick_time;
							float a_x, a_y, a_z, b_x, b_y, b_z;
					
							// 各軸別々に補間が付いてる
							a_x = GetTangent(mlist[i].interpolation, 0, 0);
							a_y = GetTangent(mlist[i].interpolation, 1, 0);
							a_z = GetTangent(mlist[i].interpolation, 2, 0);
							b_x = GetTangent(mlist[i].interpolation, 0, 1);
							b_y = GetTangent(mlist[i].interpolation, 1, 1);
							b_z = GetTangent(mlist[i].interpolation, 2, 1);
					
							lx_keys.Add(new Keyframe(tick, mlist[i].location.x, a_x, b_x));
							ly_keys.Add(new Keyframe(tick, mlist[i].location.y, a_y, b_y));
							lz_keys.Add(new Keyframe(tick, mlist[i].location.z, a_z, b_z));
						}
					}
				
					// 回転ボーンの場合はデータが入ってないはず
					int count = lx_keys.Count;
					if (count != 0)
					{
						Keyframe[] x = new Keyframe[count];
						Keyframe[] y = new Keyframe[count];
						Keyframe[] z = new Keyframe[count];
						for (int i = 0; i < count; i++) x[i] = lx_keys[i];
						for (int i = 0; i < count; i++) y[i] = ly_keys[i];
						for (int i = 0; i < count; i++) z[i] = lz_keys[i];
						AnimationCurve curve_x = new AnimationCurve(x);
						AnimationCurve curve_y = new AnimationCurve(y);
						AnimationCurve curve_z = new AnimationCurve(z);
						clip.SetCurve(bone_path, typeof(Transform), "localPosition.x", curve_x);
						clip.SetCurve(bone_path, typeof(Transform), "localPosition.y", curve_y);
						clip.SetCurve(bone_path, typeof(Transform), "localPosition.z", curve_z);
					}
				}
				catch { /* 互換性のないボーンが読み込まれた */}
			}
			
			// ボーンのパスを取得する
			string GetBonePath(Transform transform)
			{
				string buf;
				if (transform.parent == null)
					return transform.name;
				else 
					buf = GetBonePath(transform.parent);
				return buf + "/" + transform.name;
			}
			
			// ボーンの子供を走査
			void FullSearchBonePath(Transform transform, Dictionary<string, string> dic)
			{
				int count = transform.GetChildCount();
				for (int i = 0; i < count; i++)
				{
					Transform t = transform.GetChild(i);
					FullSearchBonePath(t, dic);
				}
				
				// オブジェクト名が足されてしまうので抜く
				string buf = "";
				string[] spl = GetBonePath(transform).Split('/');
				for (int i = 1; i < spl.Length-1; i++)
					buf += spl[i] + "/";
				buf += spl[spl.Length-1];
				dic.Add(transform.name, buf);
			}
			
			// 無駄なカーブを登録してるけどどうするか
			void FullEntryBoneAnimation(MMD.VMD.VMDFormat format, AnimationClip clip, Dictionary<string, string> dic)
			{
				foreach (KeyValuePair<string, string> p in dic)
				{
					CreateKeysForLocation(format, clip, p.Key, p.Value);
					CreateKeysForRotation(format, clip, p.Key, p.Value);
				}
			}
			
			// クリップをアニメーションに登録する
			public void CreateAnimationClip(MMD.VMD.VMDFormat format, GameObject assign_pmd, Animation anim)
			{
				//Animation anim = assign_pmd.GetComponent<Animation>();
				
				// クリップの作成
				AnimationClip clip = new AnimationClip();
				clip.name = format.clip_name;
				
				Dictionary<string, string> bone_path = new Dictionary<string, string>();
				FullSearchBonePath(assign_pmd.transform, bone_path);
				FullEntryBoneAnimation(format, clip, bone_path);
				
				// ここで登録
				//anim.AddClip(clip, format.clip_name);
				Debug.Log(AssetDatabase.GetAssetPath(assign_pmd));
				AssetDatabase.AddObjectToAsset(clip, AssetDatabase.GetAssetPath(assign_pmd));
				
				AssetDatabase.ImportAsset(AssetDatabase.GetAssetPath(clip));
			}
		}
	}
}
