# -*- coding: utf-8 -*-
#jptools/harness.py
#A part of NonVisual Desktop Access (NVDA)
#This file is covered by the GNU General Public License.
#See the file COPYING for more details.
#Copyright (C) 2013 Masataka.Shinke, Takuya Nishimoto, NVDA Japanese Team
# 
# For output field, blank should be 0x20 (not 0x2800).
# output の空白は 0x2800 ではなく 0x20 を使います

from __future__ import unicode_literals

tests = [
	{ 'note': '+ 基本テスト +' },
	{
		'input': 'アイウエオカキクケコサシスセソタチツテトナニヌネノハヒフヘホマミムメモヤユヨン',
		'output': '⠁⠃⠉⠋⠊⠡⠣⠩⠫⠪⠱⠳⠹⠻⠺⠕⠗⠝⠟⠞⠅⠇⠍⠏⠎⠥⠧⠭⠯⠮⠵⠷⠽⠿⠾⠌⠬⠜⠴',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38],
		'cursorPos': 0,
		'brlCursorPos': 0,
		'mode': '',
		'comment': '',
	},
	{
		'input': 'ガギグゲゴザジズゼゾダジズデドパピプペポキャキュキョチャチュチョヒャヒュヒョジャジュジョピャピュピョ',
		'output': '⠐⠡⠐⠣⠐⠩⠐⠫⠐⠪⠐⠱⠐⠳⠐⠹⠐⠻⠐⠺⠐⠕⠐⠳⠐⠹⠐⠟⠐⠞⠠⠥⠠⠧⠠⠭⠠⠯⠠⠮⠈⠡⠈⠩⠈⠪⠈⠕⠈⠝⠈⠞⠈⠥⠈⠭⠈⠮⠘⠱⠘⠹⠘⠺⠨⠥⠨⠭⠨⠮',
	},
	{
		'text': '1234567890',
		'input': '1234567890',
		'output': '⠼⠁⠃⠉⠙⠑⠋⠛⠓⠊⠚',
		'inpos2': [0,1,2,3,4,5,6,7,8,9],
		'inpos1': [0,0,1,2,3,4,5,6,7,8,9],
		'inpos':  [0,0,1,2,3,4,5,6,7,8,9],
	},
	{
		'text': '１２３４５６７８９０',
		'input': '1234567890',
		'output': '⠼⠁⠃⠉⠙⠑⠋⠛⠓⠊⠚',
		'inpos2': [0,1,2,3,4,5,6,7,8,9],
		'inpos1': [0,0,1,2,3,4,5,6,7,8,9],
		'inpos':  [0,0,1,2,3,4,5,6,7,8,9],
	},
	{
		'text': 'abcdefghijklmnopqrstuvwxyz',
		'input': 'abcdefghijklmnopqrstuvwxyz',
		'output': '⠰⠁⠃⠉⠙⠑⠋⠛⠓⠊⠚⠅⠇⠍⠝⠕⠏⠟⠗⠎⠞⠥⠧⠺⠭⠽⠵',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
		'inpos1': [0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
		'inpos':  [0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
	},
	{
		'text': 'ａｂｃｄｅｆｇｈｉｊｋｌｍｎｏｐｑｒｓｔｕｖｗｘｙｚ',
		'input': 'abcdefghijklmnopqrstuvwxyz',
		'output': '⠰⠁⠃⠉⠙⠑⠋⠛⠓⠊⠚⠅⠇⠍⠝⠕⠏⠟⠗⠎⠞⠥⠧⠺⠭⠽⠵',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
		'inpos1': [0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
		'inpos':  [0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
	},
	{
		'text': 'abc defghijklmnopqrstuvwxyz',
		'input': 'abc defghijklmnopqrstuvwxyz',
		'output': '⠰⠁⠃⠉ ⠰⠙⠑⠋⠛⠓⠊⠚⠅⠇⠍⠝⠕⠏⠟⠗⠎⠞⠥⠧⠺⠭⠽⠵',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26],
		'inpos1': [0,0,1,2,3,4,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26],
		'inpos':  [0,0,1,2,3,4,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26],
	},
	{
		'text': 'ABCDEFGHIJKLMNOPQRSTUVWXYZ',
		'input': 'ABCDEFGHIJKLMNOPQRSTUVWXYZ',
		'output': '⠰⠠⠠⠁⠃⠉⠙⠑⠋⠛⠓⠊⠚⠅⠇⠍⠝⠕⠏⠟⠗⠎⠞⠥⠧⠺⠭⠽⠵',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
		'inpos1': [0,0,0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
		'inpos':  [0,0,0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25],
	},
	{
		'text': 'ABC abc 123 ＡＢＣ　ａｂｃ　１２３',
		'input': 'ABC abc 123 ABC abc 123',
		'output': '⠰⠠⠠⠁⠃⠉ ⠰⠁⠃⠉ ⠼⠁⠃⠉ ⠰⠠⠠⠁⠃⠉ ⠰⠁⠃⠉ ⠼⠁⠃⠉',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22],
		'inpos1': [0,0,0,0,1,2,3,4,4,5,6,7,8,8,9,10,11,12,12,12,12,13,14,15,16,16,17,18,19,20,20,21,22],
		'inpos':  [0,0,0,0,1,2,3,4,4,5,6,7,8,8,9,10,11,12,12,12,12,13,14,15,16,16,17,18,19,20,20,21,22],
	},
	{
		'text': 'a\ta',
		'input': 'a a',
		'output': '⠰⠁ ⠰⠁',
	},
	{
		'text': 'あ\tあ',
		'input': 'ア ア',
		'output': '⠁ ⠁',
	},
	{
		'text': 'あ\u200bあ',
		'input': 'ア ア',
		'output': '⠁ ⠁',
	},
	{
		'text': 'ab 123　あab 123　あ',
		'input': 'ab 123 ア ab 123 ア',
		'output': '⠰⠁⠃ ⠼⠁⠃⠉ ⠁ ⠰⠁⠃ ⠼⠁⠃⠉ ⠁',
		'inpos2': [0,1,2,3,4,5,6,7,7,8,9,10,11,12,13,14,15],
		'inpos1': [0,0,1,2,3,3,4,5,6,7,8,9,9,10,11,12,12,13,14,15,16],
		'inpos':  [0,0,1,2,3,3,4,5,6,7,7,8,8,9,10,11,11,12,13,14,15],
	},
	{
		'text': 'ヒロイノ カン  カンスージノ ニ',
		'input': 'ヒロイノ カン  カンスージノ ニ',
		'output': '⠧⠚⠃⠎ ⠡⠴  ⠡⠴⠹⠒⠐⠳⠎ ⠇',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16],
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,13,14,15,16],
		'inpos':  [0,1,2,3,4,5,6,7,8,9,10,11,12,13,13,14,15,16],
	},
	{
		'text': 'トーヨーカンジノ カン  モジノ ジ 候補2',
		'input': 'トーヨーカンジノ カン  モジノ ジ コーホ2',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,19,20,21],
		'inpos1': [0,1,2,3,4,5,6,6,7,8,9,10,11,12,13,14,14,15,16,17,17,18,19,20,21,22,22],
		'inpos':  [0,1,2,3,4,5,6,6,7,8,9,10,11,12,13,14,14,15,16,17,17,18,19,19,20,21,21],
		'output': '⠞⠒⠜⠒⠡⠴⠐⠳⠎ ⠡⠴  ⠾⠐⠳⠎ ⠐⠳ ⠪⠒⠮⠼⠃',
	},
	{
		'text': 'カンジルノ カン  ヒラガナ じ 候補3',
		'input': 'カンジルノ カン  ヒラガナ ジ コーホ3',
		'inpos1': [0,1,2,2,3,4,5,6,7,8,9,10,11,12,12,13,14,15,15,16,17,18,19,20,20],
		'output': '⠡⠴⠐⠳⠙⠎ ⠡⠴  ⠧⠑⠐⠡⠅ ⠐⠳ ⠪⠒⠮⠼⠉',
	},
	{
		'text': 'カンムリノ カン  シカイシャノ シ 候補4',
		'input': 'カンムリノ カン  シカイシャノ シ コーホ4',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,22],
		'output': '⠡⠴⠽⠓⠎ ⠡⠴  ⠳⠡⠃⠈⠱⠎ ⠳ ⠪⠒⠮⠼⠙',
	},
	{
		'text': 'ミキノ カン  コト ジケンノ ジ 候補5',
		'input': 'ミキノ カン  コト ジケンノ ジ コーホ5',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,11,12,13,14,15,16,16,17,18,19,20,21,21],
		'output': '⠷⠣⠎ ⠡⠴  ⠪⠞ ⠐⠳⠫⠴⠎ ⠐⠳ ⠪⠒⠮⠼⠑',
	},
	{
		'text': 'ヒラガナ かんじ 候補6',
		'input': 'ヒラガナ カンジ コーホ6',
		'inpos1': [0,1,2,2,3,4,5,6,7,7,8,9,10,11,12,12],
		'output': '⠧⠑⠐⠡⠅ ⠡⠴⠐⠳ ⠪⠒⠮⠼⠋',
	},
	{
		'text': 'ヒロイノ カン  オサメルノ ジ 候補7',
		'input': 'ヒロイノ カン  オサメルノ ジ コーホ7',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,15,16,17,18,19,20,20],
		'output': '⠧⠚⠃⠎ ⠡⠴  ⠊⠱⠿⠙⠎ ⠐⠳ ⠪⠒⠮⠼⠛',
	},
	{
		'text': 'クサカンムリニ カンノ カン  カンジノ ジ 候補8',
		'input': 'クサカンムリニ カンノ カン  カンジノ ジ コーホ8',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,18,19,20,21,21,22,23,24,25,26,26],
		'output': '⠩⠱⠡⠴⠽⠓⠇ ⠡⠴⠎ ⠡⠴  ⠡⠴⠐⠳⠎ ⠐⠳ ⠪⠒⠮⠼⠓',
	},
	{
		'text': '(U)',
		'input': '(U)',
		'output': '⠶⠰⠠⠥⠶',
		'inpos2': [0,1,2],
		'inpos1': [0,1,1,1,2],
		'inpos':  [0,1,1,1,2],
		'comment': 'nvdajp ticket 28226 (comment 2012-05-16)',
	},
	{
		'text': '（U）',
		'input': '(U)',
		'output': '⠶⠰⠠⠥⠶',
		'inpos2': [0,1,2],
		'inpos1': [0,1,1,1,2],
		'inpos':  [0,1,1,1,2],
		'comment': '記号はUnicode正規化',
	},
	{
		'text': '(日)',
		'input': '(ニチ)',
		'output': '⠶⠇⠗⠶',
		'inpos2': [0,1,1,2],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,1,2],
		'comment': 'nvdajp ticket34973',
	},
	{
		'text': '(月)',
		'input': '(ゲツ)',
		'comment': 'nvdajp ticket34973',
	},
	{
		'text': '(火)',
		'input': '(カ)',
		'output': '⠶⠡⠶',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2],
		'inpos':  [0,1,2],
		'comment': 'nvdajp ticket34973',
	},
	{
		'text': '(水)',
		'input': '(スイ)',
		'comment': 'nvdajp ticket34973',
	},
	{
		'text': '(木)',
		'input': '(モク)',
		'comment': 'nvdajp ticket34973',
	},
	{
		'text': '(金)',
		'input': '(キン)',
		'comment': 'nvdajp ticket34973',
	},
	{
		'text': '(土)',
		'input': '(ド)',
		'comment': 'nvdajp ticket34973',
	},
	{
		'text': '[ア]',
		'input': '[ア]',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '［ア］',
		'input': '[ア]',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '記号はUnicode正規化',
	},
	{
		'text': '“ア”',
		'input': '“ア”',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '{ア}',
		'input': '{ア}',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '‘ア’',
		'input': '‘ア’',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '〔ア〕',
		'input': '〔ア〕',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '｛ア｝',
		'input': '{ア}',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '記号はUnicode正規化',
	},
	{
		'text': '〈ア〉',
		'input': '〈ア〉',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '《ア》',
		'input': '《ア》',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '【ア】',
		'input': '【ア】',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '〝ア〟',
		'input': '〝ア〟',
		'output': '⠐⠶⠁⠶⠂',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '「ア」',
		'input': '「ア」',
		'output': '⠤⠁⠤',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2],
		'inpos':  [0,1,2],
		'comment': '',
	},
	{
		'text': '『ア』',
		'input': '『ア』',
		'output': '⠰⠤⠁⠤⠆',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2,2],
		'inpos':  [0,0,1,2,2],
		'comment': '',
	},
	{
		'text': '｢ア｣',
		'input': '「ア」',
		'output': '⠤⠁⠤',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2],
		'inpos':  [0,1,2],
		'comment': '記号はUnicode正規化',
	},
	{
		'text': 'Ctrl-Z',
		'input': 'Ctrl-Z',
		'output': '⠰⠠⠉⠞⠗⠇⠤⠰⠠⠵',
		'inpos2': [0,1,2,3,4,5],
		'inpos1': [0,0,0,1,2,3,4,5,5,5],
		'inpos':  [0,0,0,1,2,3,4,5,5,5],
		'comment': 'Latinの 0x002D HYPHEN-MINUS は 0x2212 MUNUS SIGN に正規化しない',
	},
	{'text':'ト゚', 'input':'ト'}, # U+309a
	{'text':'ト　゚', 'input':'ト'}, # U+309a
	{
		'input':'ｶﾞｰﾝ',
		'output': '⠐⠡⠒⠴',
		'inpos1': [0,0,1,2],
		'comment':'半角カタカナのポジションマッピング(level1)',
	},
	{
		'text':'ｶﾞｰﾝ',
		'input':'ｶﾞｰﾝ',
		'output': '⠐⠡⠒⠴',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,2],
		'inpos':  [0,0,1,2],
		'comment':'半角カタカナのポジションマッピング(level2+level1)',
	},
	{
		'input': 'Aア',
		'output': '⠰⠠⠁⠤⠁',
		'inpos1': [0,0,0,0,1],
	},
	{
		'input': 'aアットb',
		'output': '⠰⠁⠤⠁⠂⠞⠰⠃',
		'inpos1': [0,0,0,1,2,3,4,4],
	},
	{
		'text': 'a@b',
		'input': '⠠⠦a@b⠠⠴',
		'output': '⠠⠦⠁⠪⠃⠠⠴',
		'inpos2': [0,0,0,1,2,2,2],
		'inpos1': [0,1,2,3,4,5,6],
		'comment': '情報処理点字を使う。@は246で。',
	},
	{
		'text': '<a@b>',
		'input': '⠠⠦<a@b>⠠⠴',
		'output': '⠠⠦⠔⠔⠁⠪⠃⠢⠢⠠⠴',
		'inpos2': [0,0,0,1,2,3,4,4,4],
		'inpos1': [0,1,2,2,3,4,5,6,6,7,8],
		'inpos':  [0,0,0,0,1,2,3,4,4,4,4],
		'comment': 'メールアドレス前後の山かっこ',
	},
	{
		'input': 'Cプラスプラス',
		'output': '⠰⠠⠉⠤⠠⠭⠑⠹⠠⠭⠑⠹',
		'inpos1': [0,0,0,0,1,1,2,3,4,4,5,6],
	},
	{
		'text': 'notepad++',
		'input': 'notepad++',
		'output': '⠰⠝⠕⠞⠑⠏⠁⠙⠢⠢',
		'inpos2': [0,1,2,3,4,5,6,7,8],
		'inpos1': [0,0,1,2,3,4,5,6,7,8],
		'comment': 'プラスは情報処理点字 356 だが 26 のまま。',
	},
	{
		'text': '+@/#$%&*;',
		'input': '⠠⠦+@/#$%&*;⠠⠴',
		'output': '⠠⠦⠬⠪⠌⠩⠹⠻⠯⠡⠆⠠⠴',
		'inpos2': [0,0,0,1,2,3,4,5,6,7,8,8,8],
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12],
		'inpos':  [0,0,0,1,2,3,4,5,6,7,8,8,8],
		'comment': '情報処理点字',
	},
	{
		'text': ',.:;!?@#\\$%&*|+-/=<>^_~',
		'input': '⠠⠦,.:;!?@#\\$%&*|+-/=<>^_~⠠⠴',
		'output': '⠠⠦⠂⠲⠐⠂⠆⠖⠐⠦⠪⠩⠫⠹⠻⠯⠡⠳⠬⠤⠌⠒⠒⠔⠔⠢⠢⠘⠐⠤⠐⠉⠠⠴',
		'inpos2': [0,0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,22,22],
		'inpos1': [0,1,2,3,4,4,5,6,7,7,8,9,10,11,12,13,14,15,16,17,18,19,19,20,20,21,21,22,23,23,24,24,25,26],
		'inpos':  [0,0,0,1,2,2,3,4,5,5,6,7,8,9,10,11,12,13,14,15,16,17,17,18,18,19,19,20,21,21,22,22,22,22],
		'comment': '情報処理点字 バックスラッシュではなく円記号',
	},
	{
		'text': '"\'`',
		'input': '”’‘',
		#'output': '⠶⠄⠐⠑',
		'inpos2': [0,1,2],
		'comment': '「”’‘」は正規化',
	},
	{
		'text': '@{}[]', # ()
		'input': '⠠⠦@{}[]⠠⠴', # ()
		'output': '⠠⠦⠪⠣⠜⠷⠾⠠⠴', # ⠦⠴
		'inpos2': [0,0,0,1,2,3,4,4,4],
		'inpos1': [0,1,2,3,4,5,6,7,8],
		'inpos':  [0,0,0,1,2,3,4,4,4],
	},
	{
		'text': 'アドレスはnvda@nvda.jpです。',
		'input': 'アドレスワ ⠠⠦nvda@nvda.jp⠠⠴ デス。',
		'output': '⠁⠐⠞⠛⠹⠄ ⠠⠦⠝⠧⠙⠁⠪⠝⠧⠙⠁⠲⠚⠏⠠⠴ ⠐⠟⠹⠲',
		'inpos2': [0,1,2,3,4,4,5,5,5,6,7,8,9,10,11,12,13,14,15,16,16,16,16,17,18,19],
		'inpos1': [0,1,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,23,24,25],
		'inpos':  [0,1,1,2,3,4,4,5,5,5,6,7,8,9,10,11,12,13,14,15,16,16,16,16,17,17,18,19],
		'comment': '情報処理点字',
	},
	{
		'text': 'アドレスはabc@123.jpです。',
		'input': 'アドレスワ ⠠⠦abc@123.jp⠠⠴ デス。',
		'output': '⠁⠐⠞⠛⠹⠄ ⠠⠦⠁⠃⠉⠪⠼⠁⠃⠉⠲⠰⠚⠏⠠⠴ ⠐⠟⠹⠲',
		'inpos2': [0,1,2,3,4,4,5,5,5,6,7,8,9,10,11,12,13,14,14,14,14,15,16,17],
		'inpos1': [0,1,1,2,3,4,5,6,7,8,9,10,11,12,12,13,14,15,16,16,17,18,19,20,21,21,22,23],
		'inpos':  [0,1,1,2,3,4,4,5,5,5,6,7,8,9,9,10,11,12,13,13,14,14,14,14,15,15,16,17],
		'comment': '情報処理点字。数字、ピリオドのあとにａ～ｊが続くときは小文字フラグ(56)が必要。',
	},
	{
		'text': 'アドレスはa-b_c@123.jpです。',
		'input': 'アドレスワ ⠠⠦a-b_c@123.jp⠠⠴ デス。',
		'output': '⠁⠐⠞⠛⠹⠄ ⠠⠦⠁⠤⠃⠐⠤⠉⠪⠼⠁⠃⠉⠲⠰⠚⠏⠠⠴ ⠐⠟⠹⠲',
		'inpos2': [0,1,2,3,4,4,5,5,5,6,7,8,9,10,11,12,13,14,15,16,16,16,16,17,18,19],
		'inpos1': [0,1,1,2,3,4,5,6,7,8,9,10,11,11,12,13,14,14,15,16,17,18,18,19,20,21,22,23,23,24,25],
		'inpos':  [0,1,1,2,3,4,4,5,5,5,6,7,8,8,9,10,11,11,12,13,14,15,15,16,16,16,16,17,17,18,19],
		'comment': '情報処理点字。ｊの前に小文字フラグ。',
	},
	{
		'text': 'アドレスはhttp://abc.123.jp/~a/?a&b#a+b-c%56です。',
		'input': 'アドレスワ ⠠⠦http://abc.123.jp/~a/?a&b#a+b-c%56⠠⠴ デス。',
		'output': '⠁⠐⠞⠛⠹⠄ ⠠⠦⠓⠞⠞⠏⠐⠂⠌⠌⠁⠃⠉⠲⠼⠁⠃⠉⠲⠰⠚⠏⠌⠐⠉⠁⠌⠐⠦⠁⠯⠃⠩⠁⠬⠃⠤⠉⠻⠼⠑⠋⠠⠴ ⠐⠟⠹⠲',
		'inpos2': [0,1,2,3,4,4,5,5,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,38,38,38,39,40,41],
		'inpos1': [0,1,1,2,3,4,5,6,7,8,9,10,11,12,12,13,14,15,16,17,18,19,19,20,21,22,23,23,24,25,26,26,27,28,29,29,30,31,32,33,34,35,36,37,38,39,40,40,41,42,43,44,45,45,46,47],
		'inpos':  [0,1,1,2,3,4,4,5,5,5,6,7,8,9,9,10,11,12,13,14,15,16,16,17,18,19,20,20,21,22,23,23,24,25,26,26,27,28,29,30,31,32,33,34,35,36,37,37,38,38,38,38,39,39,40,41],
		'comment': '情報処理点字。ｊの前に小文字フラグ。',
	},
	{
		'text': 'アドレスはabc.123.jpです。',
		'input': 'アドレスワ ⠦abc.123.jp⠴ デス。',
		'output': '⠁⠐⠞⠛⠹⠄ ⠦⠁⠃⠉⠲⠼⠁⠃⠉⠲⠚⠏⠴ ⠐⠟⠹⠲',
		'inpos2': [0,1,2,3,4,4,5,5,6,7,8,9,10,11,12,13,14,14,14,15,16,17],
		'inpos1': [0,1,1,2,3,4,5,6,7,8,9,10,11,11,12,13,14,15,16,17,18,19,19,20,21],
		'inpos':  [0,1,1,2,3,4,4,5,5,6,7,8,9,9,10,11,12,13,14,14,14,15,15,16,17],
		'comment': '外国語引用符。数字の後ろのドットは小数点ではない。',
	},
	{
		'text': 'C:\\Program Files',
		'input': '⠠⠦C:\\Program Files⠠⠴',
		'output': '⠠⠦⠠⠉⠐⠂⠫⠠⠏⠗⠕⠛⠗⠁⠍ ⠠⠋⠊⠇⠑⠎⠠⠴',
		'inpos1': [0,1,2,2,3,3,4,5,5,6,7,8,9,10,11,12,13,13,14,15,16,17,18,19],
		'comment': '情報処理点字',
	},
	{
		'text': 'C:\\Users\\ユーザー',
		'input': '⠠⠦C:\\Users\\⠠⠴ ユーザー',
		'output': '⠠⠦⠠⠉⠐⠂⠫⠠⠥⠎⠑⠗⠎⠫⠠⠴ ⠬⠒⠐⠱⠒',
		'inpos1': [0,1,2,2,3,3,4,5,5,6,7,8,9,10,11,12,13,14,15,16,16,17],
		'comment': '情報処理点字を前半のみに使う',
	},
	{
		'text': 'v1.4',
		'input': '⠦v1.4⠴',
		'output': '⠦⠧⠼⠁⠂⠙⠴',
		'inpos2': [0,0,1,2,3,3],
		'inpos1': [0,1,2,2,3,4,5],
		'inpos':  [0,0,1,1,2,3,3],
		'comment': 'nvdajp ticket 28226 (comment 2012-05-16)',
	},
	{
		'text': 'c:\\',
		'input': '⠠⠦c:\\⠠⠴',
		'output': '⠠⠦⠉⠐⠂⠫⠠⠴',
		'inpos1': [0,1,2,3,3,4,5,6],
		'comment': '円記号',
	},
	{
		'text': 'c:\\work\\nvda\\jpmain>',
		'input': '⠠⠦c:\\work\\nvda\\jpmain>⠠⠴',
		'output': '⠠⠦⠉⠐⠂⠫⠺⠕⠗⠅⠫⠝⠧⠙⠁⠫⠚⠏⠍⠁⠊⠝⠢⠢⠠⠴',
		'inpos1': [0,1,2,3,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,21,22,23],
		'comment': '円記号 jpとmainが既知語であるがマスアケしない',
	},
	{
		'text': '<=>',
		'input': '<=>',
		'output': '⠔⠔⠒⠒⠢⠢',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,1,2,2],
		'inpos':  [0,0,1,1,2,2],
	},
	{
		'input': 'アー',
		'output': '⠁⠒',
		'inpos1': [0,1],
	},
	{
		'text': 'きゃりーぱみゅぱみゅ',
		'input': 'キャリーパミュパミュ',
		'output': '⠈⠡⠓⠒⠠⠥⠈⠽⠠⠥⠈⠽',
		'inpos2': [0,1,2,3,4,5,6,7,8,9],
		'inpos1': [0,1,2,3,4,4,5,6,7,7,8,9],
		'inpos':  [0,1,2,3,4,4,5,6,7,7,8,9],
	},
	{
		'comment': '点訳のてびき 第4章 その1 句読符',
		'input': 'ア、イ。',
		'output': '⠁⠰ ⠃⠲',
		'inpos1': [0,1,1,2,3],
	},
	{
		'comment': '点訳のてびき 第4章 その1 句読符',
		'input': 'ア？ア?',
		'output': '⠁⠢ ⠁⠢',
		'inpos1': [0,1,1,2,3],
	},
	{
		'comment': '点訳のてびき 第4章 その1 句読符',
		'input': 'ア！ア!',
		'output': '⠁⠖ ⠁⠖',
		'inpos1': [0,1,1,2,3],
	},
	{
		'comment': '点訳のてびき 第4章 その1 句読符',
		'text': 'いいえ。はい、そうです。',
		'input': 'イイエ。ハイ、ソーデス。',
		'output': '⠃⠃⠋⠲  ⠥⠃⠰ ⠺⠒⠐⠟⠹⠲',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11],
		'inpos1': [0,1,2,3,3,3,4,5,6,6,7,8,9,9,10,11],
		'inpos':  [0,1,2,3,3,3,4,5,6,6,7,8,9,9,10,11],
	},
	{
		'comment': '点訳のてびき 第4章 その1 句読符 4.中点',
		'input': 'ア・イ',
		'output': '⠁⠐ ⠃',
		'inpos1': [0,1,1,2],
	},
	{
		'input': '123アイウ',
		'output': '⠼⠁⠃⠉⠤⠁⠃⠉',
		'inpos1': [0,0,1,2,2,3,4,5],
		'comment': 'numbers and kana characters should be separated by 36 symbol.',
	},
	{
		'input': '0ア1イ2ウ3エ4オ5ラ6リ7ル8レ9ロ',
		'output': '⠼⠚⠤⠁⠼⠁⠤⠃⠼⠃⠤⠉⠼⠉⠤⠋⠼⠙⠤⠊⠼⠑⠤⠑⠼⠋⠤⠓⠼⠛⠤⠙⠼⠓⠤⠛⠼⠊⠤⠚',
		'comment': 'numbers and some kana characters should be separated by 36 symbol.',
	},
	{
		'input': '123キャ',
		'output': '⠼⠁⠃⠉⠈⠡',
		'comment': 'should not be separated by 36 symbol.',
	},
	{
		'input': '0カ1キ2ク3ケ4コ5サ6シ7ス8セ9ソ0タ1チ2ツ3テ4ト5ナ6ニ7ヌ8ネ9ノ',
		'output': '⠼⠚⠡⠼⠁⠣⠼⠃⠩⠼⠉⠫⠼⠙⠪⠼⠑⠱⠼⠋⠳⠼⠛⠹⠼⠓⠻⠼⠊⠺⠼⠚⠕⠼⠁⠗⠼⠃⠝⠼⠉⠟⠼⠙⠞⠼⠑⠅⠼⠋⠇⠼⠛⠍⠼⠓⠏⠼⠊⠎',
		'comment': 'should not be separated by 36 symbol.',
	},
	{
		'input': '1ア1キャ1ア',
		'output': '⠼⠁⠤⠁⠼⠁⠈⠡⠼⠁⠤⠁',
		'comment': 'numbers and some kana characters should be separated by 36 symbol.',
	},
	{
		'input': '123a',
		'output': '⠼⠁⠃⠉⠰⠁',
	},
	{
		'text': 'h26a.pdf',
		'input': '⠦h26a.pdf⠴',
		'output': '⠦⠓⠼⠃⠋⠰⠁⠲⠏⠙⠋⠴',
	},
	{
		'text': 'http://h26a.pdf',
		'input': '⠠⠦http://h26a.pdf⠠⠴',
		'output': '⠠⠦⠓⠞⠞⠏⠐⠂⠌⠌⠓⠼⠃⠋⠰⠁⠲⠏⠙⠋⠠⠴',
	},
	{
		'text': 'http://0x1b',
		'input': '⠠⠦http://0x1b⠠⠴',
		'output': '⠠⠦⠓⠞⠞⠏⠐⠂⠌⠌⠼⠚⠭⠼⠁⠰⠃⠠⠴',
	},
	{
		'input': '123A',
		'output': '⠼⠁⠃⠉⠰⠠⠁',
	},
	{
		'input': '1アa1aキャA1Aア',
		'output': '⠼⠁⠤⠁⠰⠁⠼⠁⠰⠁⠤⠈⠡⠰⠠⠁⠼⠁⠰⠠⠁⠤⠁',
	},
	{
		'input': '⠀ア',
		'output': ' ⠁',
		'inpos1': [0,1],
		'comment': 'braille space should be changed to 0x20.',
	},
	{
		'text': '⠁⠂⠃⠄⠅⠆⠇⠈⠉⠊⠋⠌⠍⠎⠏⠐⠑⠒⠓⠔⠕⠖⠗⠘⠙⠚⠛⠜⠝⠞⠟',
		'input': '⠁⠂⠃⠄⠅⠆⠇⠈⠉⠊⠋⠌⠍⠎⠏⠐⠑⠒⠓⠔⠕⠖⠗⠘⠙⠚⠛⠜⠝⠞⠟',
		'output': '⠁⠂⠃⠄⠅⠆⠇⠈⠉⠊⠋⠌⠍⠎⠏⠐⠑⠒⠓⠔⠕⠖⠗⠘⠙⠚⠛⠜⠝⠞⠟',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30],
		'comment': 'braille code should not be changed.',
	},
	{
		'text': '⠁⡀あ',
		'input': '⠁⡀ア',
		'output': '⠁⡀⠁',
		'comment': 'braille code should not be changed.',
	},
	{
		'text': '⠁⡀',
		'input': '⠁⡀',
		'output': '⠁⡀',
		'comment': 'braille code should not be changed.',
	},
	{
		'text': '⠁⡀ ',
		'input': '⠁⡀ ',
		'output': '⠁⡀ ',
		'comment': 'braille code (with space) should not be changed.',
	},
	{
		'text': '⠒⠓⠔⠕⠖⠗⠘⠙あ⠚⠛⠜⠝⠞⠟ ',
		'input': '⠒⠓⠔⠕⠖⠗⠘⠙ア⠚⠛⠜⠝⠞⠟ ',
		'comment': 'braille code (with space) should not be changed.',
	},


	{
		'text': '⠠⠡⠢⠣⠤⠥⠦⠧⠨⠩⠪⠫⠬⠭⠮⠯⠰⠱⠲⠳⠴⠵⠶⠷⠸⠹⠺⠻⠼⠽⠾⠿',
		'input': '⠠⠡⠢⠣⠤⠥⠦⠧⠨⠩⠪⠫⠬⠭⠮⠯⠰⠱⠲⠳⠴⠵⠶⠷⠸⠹⠺⠻⠼⠽⠾⠿',
		'output': '⠠⠡⠢⠣⠤⠥⠦⠧⠨⠩⠪⠫⠬⠭⠮⠯⠰⠱⠲⠳⠴⠵⠶⠷⠸⠹⠺⠻⠼⠽⠾⠿',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31],
		'comment': 'braille code should not be changed.',
	},
	{
		'text': '⡀⡁⡂⡃⡄⡅⡆⡇⡈⡉⡊⡋⡌⡍⡎⡏⡐⡑⡒⡓⡔⡕⡖⡗⡘⡙⡚⡛⡜⡝⡞⡟',
		'input': '⡀⡁⡂⡃⡄⡅⡆⡇⡈⡉⡊⡋⡌⡍⡎⡏⡐⡑⡒⡓⡔⡕⡖⡗⡘⡙⡚⡛⡜⡝⡞⡟',
		'output': '⡀⡁⡂⡃⡄⡅⡆⡇⡈⡉⡊⡋⡌⡍⡎⡏⡐⡑⡒⡓⡔⡕⡖⡗⡘⡙⡚⡛⡜⡝⡞⡟',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31],
		'comment': 'braille code should not be changed.',
	},
	{
		'text': '《本章のねらい》',
		'input': '《ホンショーノ ネライ》',
		'output': '⠐⠶⠮⠴⠈⠺⠒⠎ ⠏⠑⠃⠶⠂',
	},
	{
		'text': '二十五',
		'input': '25',
		'output': '⠼⠃⠑',
		'inpos1': [0,0,1],
	},
	{
		'text': 'ピクチャ3の12',
		'input': 'ピクチャ3ノ 12',
		'output': '⠠⠧⠩⠈⠕⠼⠉⠎ ⠼⠁⠃',
	},
	{
		'text': '    雨。  晴れ。',
		'input': '    アメ。  ハレ。',
		'output': '    ⠁⠿⠲    ⠥⠛⠲',
		'comment': '先頭のスペースを除去しない'
	},
	{
		'text': '大文字',
		'input': 'オーモジ',
	},
	{
		'text': 'システムキャレット', 
		'input': 'システム キャレット',
	},
	{
		'text': 'フィードバック',
		'input': 'フィード バック',
	},
	{
		'text': 'インターフェース',
		'input': 'インター フェース',
	},
	{
		'text': 'オペレーティングシステム',
		'input': 'オペレーティング システム',
	},
	{
		'text': 'インストーラー',
		'input': 'インストーラー',
	},
	{
		'text': 'トーキングインストーラー',
		'input': 'トーキング インストーラー',
	},
	{
		'text': 'アイスクリーム', 
		'input': 'アイス クリーム',
	},
	{
		'text': '日本点字図書館',
		'input': 'ニッポン テンジ トショカン'
	},
	{
		'text': '通り',
		'input': 'トオリ',
	},
	{
		'text': 'ねえ',
		'input': 'ネエ',
	},
	{
		'text': '通り', 'input': 'トオリ'
	},
	{
		'text': '狼', 'input': 'オオカミ'
	},
	{
		'text': '通る', 'input': 'トオル'
	},
	{
		'text': '多い', 'input': 'オオイ'
	},
	{
		'text': '多く', 'input': 'オオク'
	},
	{
		'text': '大晦日', 'input': 'オオミソカ'
	},
	{
		'text': '手作り', 'input': 'テヅクリ'
	},
	{
		'text': '南半球', 'input': 'ミナミ ハンキュー'
	},
	{	
		'text': 'アメリカ合衆国', 'input': 'アメリカ ガッシューコク'
	},
	{
		'text': '第一人者', 'input': 'ダイ1ニンシャ'
	},
	{
		'text': '一流', 'input': '1リュー'
	},
	{
		'text': '一月', 'input': '1ガツ'
	},
	{
		'text': '二月', 'input': '2ガツ'
	},
	{
		'text': '四方', 'input': '4ホー'
	},
	{
		'text': '六法全書', 'input': '6ポー ゼンショ'
	},
	{
		'text': '百人一首', 'input': '100ニン 1シュ'
	},
	{
		'text': 'ヱビスビール', 'input': 'エビス ビール'
	},
	{
		'text': '日本コロムビア', 'input': 'ニッポン コロムビア'
	},
	{
		'text': 'ビタミンＥ', 'input': 'ビタミン E'
	},
	{
		'text': '劇団四季', 'input': 'ゲキダン 4キ'
	},
	{
		'text': '四季', 'input': '4キ'
	},
	{
		'text': '四半期', 'input': '4ハンキ'
	},
	{
		'text': '四角形', 'input': '4カクケイ'
	},
	{
		'text': '四条', 'input': '4ジョー'
	},
	{
		'text': '二男', 'input': '2ナン'
	},
	{
		'text': '十数', 'input': '10スー'
	},
	{
		'text': '一輪車', 'input': '1リンシャ'
	},
	{
		'text': '三塁打', 'input': '3ルイダ'
	},
	{
		'text': '一汁一菜', 'input': '1ジュー 1サイ'
	},
	{
		'text': '五臓六腑', 'input': '5ゾー 6プ'
	},
	{
		'text': '一段', 'input': '1ダン'
	},
	{
		'text': '七転び八起き', 'input': 'ナナコロビ ヤオキ'
	},
	{
		'text': '十重二十重', 'input': 'トエ ハタエ'
	},
	{
		'text': '３ラン', 'input': '3ラン'
	},
	{
		'text': 'さんりんしゃ', 'input': '3リンシャ'
	},
	{
		'text': 'いちばん', 'input': '1バン'
	},
	{
		'text': 'Ｘ線', 'input': 'Xセン'
	},

	{'text':'触読', 'input':'ショクドク'},
	{'text':'触読式時計', 'input':'ショクドクシキ トケイ'},
	{'text':'触手話', 'input':'ショクシュワ'},
	{'text':'触読手話', 'input':'ショクドク シュワ'},

	{'text':'泉質', 'input':'センシツ'},
	{'text':'硫酸塩', 'input':'リューサンエン'},
	{'text':'硫酸塩泉', 'input':'リューサンエンセン'},
	{'text':'塩化物泉', 'input':'エンカブッセン'},
	{'text':'泉温', 'input':'センオン'},
	{'text':'冷鉱泉', 'input':'レイコーセン'},
	{'text':'微温泉', 'input':'ビオンセン'},
	{'text':'療養泉', 'input':'リョーヨーセン'},
	{'text':'低張性', 'input':'テイチョーセイ'},
	{'text':'等張性', 'input':'トーチョーセイ'},
	{'text':'高張性', 'input':'コーチョーセイ'},
	{'text':'酸性泉', 'input':'サンセイセン'},
	{'text':'放射能泉', 'input':'ホーシャノーセン'},

	{'text':'晴れ所により一時雨', 'input':'ハレ トコロニ ヨリ イチジ アメ'},

	{ 'note': '+ NVDA点字表示の誤りへの対応 +' },
	{
		'comment': 'NVDA点字表示の誤り No.1',
		'text': '誤判定',
		'input': 'ゴハンテイ',
		'output': '⠐⠪⠥⠴⠟⠃',
	},
	{
		'text': '判定',
		'input': 'ハンテイ',
		'output': '⠥⠴⠟⠃',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': 'NVDA点字表示の誤り No.2',
		'text': '50音順',
		'input': '50オンジュン',
		'output': '⠼⠑⠚⠤⠊⠴⠘⠹⠴',
	},
	{
		'comment': 'NVDA点字表示の誤り No.3',
		'text': '下矢印',
		'input': 'シタヤジルシ',
		'output': '⠳⠕⠌⠐⠳⠙⠳',
	},
	{
		'comment': 'NVDA点字表示の誤り No.4',
		'text': '上矢印',
		'input': 'ウエヤジルシ',
		'output': '⠉⠋⠌⠐⠳⠙⠳',
	},
	{
		'comment': 'NVDA点字表示の誤り No.5',
		'text': '大見出し',
		'input': 'オオミダシ',
		'output': '⠊⠊⠷⠐⠕⠳',
	},
	{
		'comment': 'NVDA点字表示の誤り No.6',
		'text': '前景色',
		'input': 'ゼンケイショク',
		'output': '⠐⠻⠴⠫⠃⠈⠺⠩',
	},
	{
		'comment': 'NVDA点字表示の誤り No.7',
		'text': '表計算',
		'input': 'ヒョーケイサン',
		'output': '⠈⠮⠒⠫⠃⠱⠴',
	},
	{
		'comment': 'NVDA点字表示の誤り No.8',
		'text': '材販',
		'input': 'ザイハン',
		'output': '⠐⠱⠃⠥⠴',
	},
	{
		'comment': 'NVDA点字表示の誤り No.9',
		'text': '孫正義',
		'input': 'ソン マサヨシ',
		'output': '⠺⠴ ⠵⠱⠜⠳',
		'inpos1': [0,1,2,3,4,5,6],
	},
	{
		'comment': 'NVDA点字表示の誤り No.10',
		'text': '昔々',
		'input': 'ムカシ ムカシ',
		'output': '⠽⠡⠳ ⠽⠡⠳',
		'inpos1': [0,1,2,3,4,5,6],
	},
	{
		'comment': 'NVDA点字表示の誤り No.11',
		'text': '梅雨前線',
		'input': 'バイウ ゼンセン',
		'output': '⠐⠥⠃⠉ ⠐⠻⠴⠻⠴',
	},
	{
		'comment': 'NVDA点字表示の誤り No.12',
		'text': '八ッ場ダム',
		'input': 'ヤンバダム',
		'output': '⠌⠴⠐⠥⠐⠕⠽',
	},
	{
		'comment': 'NVDA点字表示の誤り No.13',
		'text': '１都５県',
		'input': 'イットゴケン',
		'output': '⠃⠂⠞⠐⠪⠫⠴',
	},
	{
		'comment': 'NVDA点字表示の誤り No.14',
		'text': 'お姉さん',
		'input': 'オネエサン',
		'output': '⠊⠏⠋⠱⠴',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': 'NVDA点字表示の誤り No.15',
		'text': 'お兄さん',
		'input': 'オニイサン',
		'output': '⠊⠇⠃⠱⠴',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': 'NVDA点字表示の誤り No.16',
		'text': 'お婆さん',
		'input': 'オバアサン',
		'output': '⠊⠐⠥⠁⠱⠴',
	},
	{
		'comment': 'NVDA点字表示の誤り No.17',
		'text': 'お爺さん',
		'input': 'オジイサン',
		'output': '⠊⠐⠳⠃⠱⠴',
	},
	{
		'comment': 'NVDA点字表示の誤り No.18',
		'text': '1月',
		'input': '1ガツ',
		'output': '⠼⠁⠐⠡⠝',
	},
	{
		'comment': 'NVDA点字表示の誤り No.19',
		'text': '盲ろう者',
		'input': 'モーローシャ',
		'output': '⠾⠒⠚⠒⠈⠱',
	},
	{
		'comment': 'NVDA点字表示の誤り No.20',
		'text': 'のように',
		'input': 'ノヨーニ',
		'output': '⠎⠜⠒⠇',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': 'NVDA点字表示の誤り No.21',
		'text': '1回',
		'input': '1カイ',
		'output': '⠼⠁⠡⠃',
	},
	{
		'comment': 'NVDA点字表示の誤り No.22',
		'text': 'いいんですけど',
		'input': 'イインデスケド',
		'output': '⠃⠃⠴⠐⠟⠹⠫⠐⠞',
	},
	{ 'note': '+ 点訳のてびき第3版 +' },
	{ 'note': '== 第２章　語の書き表し方 ==' },
	{ 'note': '=== その１　仮名遣い ===' },
	{ 'note': '==== １　基本的な仮名遣い ====' },
	{ 'note': '１、直音（清音・濁音・半濁音）は、現代仮名遣いに準じて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 1 No.1',
		'text': '未曾有',
		'input': 'ミゾウ',
		'output': '⠷⠐⠺⠉',
	},
	{ 'note': '【備考１】助詞の「を」は、発音に関わりなく「ヲ」と書くが、助詞の「は」「へ」は、発音するとおりに「ワ」「エ」と書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 1 No.17',
		'text': '駅へは',
		'input': 'エキエワ',
		'output': '⠋⠣⠋⠄',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 1 No.22',
		'text': 'えきへは',
		'input': 'エキエワ',
		'output': '⠋⠣⠋⠄',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 1 No.23',
		'text': 'ははへの',
		'input': 'ハハエノ',
		'output': '⠥⠥⠋⠎',
		'inpos1': [0,1,2,3],
	},
	{ 'note': '【備考２】「言う」の語幹は、「ユ」と発音しても常に「イ」と書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 1 No.25',
		'text': '言うて',
		'input': 'イウテ',
		'output': '⠃⠉⠟',
		'inpos1': [0,1,2],
	},
	{ 'note': '２、拗音（拗濁音・拗半濁音を含む）は、拗音点字を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 2 No.1',
		'text': '距離',
		'input': 'キョリ',
		'output': '⠈⠪⠓',
	},

	{ 'note': '３、撥音（ん）は、撥音符を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 3 No.4',
		'text': '何百',
		'input': 'ナンビャク',
		'output': '⠅⠴⠘⠥⠩',
	},
	{ 'note': '４、促音（っ）は、促音符を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 4 No.11',
		'text': '夫',
		'input': 'オット',
		'output': '⠊⠂⠞',
	},
	{ 'note': '〈処理〉「き」「く」「つ」が促音化しているかどうかは、辞書で確認するが、辞書によって表記が異なるなど判断が難しい場合は、促音符を用いず、その部分を「キ」「ク」「ツ」と書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 4 No.10',
		'text': '万国旗',
		'input': 'バンコクキ',
		'output': '⠐⠥⠴⠪⠩⠣',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 4 No.11',
		'text': '多角形',
		'input': 'タカクケイ',
		'output': '⠕⠡⠩⠫⠃',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 4 No.14',
		'text': '高脂血症',
		'input': 'コーシケツショー',
		'output': '⠪⠒⠳⠫⠝⠈⠺⠒',
	},
	{ 'note': '５、長音の書き表し方' },
	{ 'note': '（１）ア列の長音はア列の仮名に「ア」を添え、イ列とエ列の長音はイ列・エ列の仮名に「イ」を添えて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.1',
		'text': 'ああ',
		'input': 'アア',
		'output': '⠁⠁',
		'inpos1': [0,1],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.2',
		'text': 'お母さん',
		'input': 'オカアサン',
		'output': '⠊⠡⠁⠱⠴',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.3',
		'text': 'いいえ',
		'input': 'イイエ',
		'output': '⠃⠃⠋',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2],
		'inpos':  [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.4',
		'text': 'お兄さん',
		'input': 'オニイサン',
		'output': '⠊⠇⠃⠱⠴',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.5',
		'text': '先生',
		'input': 'センセイ',
		'output': '⠻⠴⠻⠃',
		'inpos1': [0,1,2,3],
	},
	{ 'note': '【備考】エ列の長音のうち和語は「エ」を添えて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.13',
		'text': 'お姉さん',
		'input': 'オネエサン',
		'output': '⠊⠏⠋⠱⠴',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.14',
		'text': 'へえ',
		'input': 'ヘエ',
		'output': '⠯⠋',
		'inpos2': [0,1],
		'inpos1': [0,1],
		'inpos':  [0,1],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.15',
		'text': 'べらんめえ',
		'input': 'ベランメエ',
		'output': '⠐⠯⠑⠴⠿⠋',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,0,1,2,3,4],
		'inpos':  [0,0,1,2,3,4],
	},
	{ 'note': '〈処理１〉原文で小文字の仮名が長音を表しているときは、原文に使われている仮名を添えて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.16',
		'text': 'やったぁ',
		'input': 'ヤッタア',
		'output': '⠌⠂⠕⠁',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.17',
		'text': '君ィ',
		'input': 'キミイ',
		'output': '⠣⠷⠃',
		'inpos1': [0,1,2],
	},
	{ 'note': '〈処理２〉原文に長音符号（－）が使われているときは、それにしたがって長音符を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.18',
		'text': 'はんたーい',
		'input': 'ハンターイ',
		'output': '⠥⠴⠕⠒⠃',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,1,2,3,4],
		'inpos':  [0,1,2,3,4],
	},
	{ 'note': '（２）ウ列とオ列の長音は、ウ列・オ列の仮名に長音符を添えて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.22',
		'text': '宇宙',
		'input': 'ウチュー',
		'output': '⠉⠈⠝⠒',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.30',
		'text': 'お父さん',
		'input': 'オトーサン',
		'output': '⠊⠞⠒⠱⠴',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.34',
		'text': '八日',
		'input': 'ヨーカ',
		'output': '⠜⠒⠡',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.35',
		'text': '絵を見よう',
		'input': 'エヲ ミヨー',
		'output': '⠋⠔ ⠷⠜⠒',
		'inpos1': [0,1,2,3,4,5],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.36',
		'text': '運動しよう',
		'input': 'ウンドー シヨー',
		'output': '⠉⠴⠐⠞⠒ ⠳⠜⠒',
		'inpos1': [0,1,2,2,3,4,5,6,7],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.37',
		'text': 'そのような',
		'input': 'ソノヨーナ',
		'output': '⠺⠎⠜⠒⠅',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.38',
		'text': 'そうだろう',
		'input': 'ソーダロー',
		'output': '⠺⠒⠐⠕⠚⠒',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,1,2,2,3,4],
		'inpos':  [0,1,2,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.39',
		'text': '何と言おうか',
		'input': 'ナント イオーカ',
		'output': '⠅⠴⠞ ⠃⠊⠒⠡',
		'inpos1': [0,1,2,3,4,5,6,7],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.40',
		'text': '見まごうばかり',
		'input': 'ミマゴーバカリ',
		'output': '⠷⠵⠐⠪⠒⠐⠥⠡⠓',
		'inpos2': [0,1,2,3,4,5,6],
		'inpos1': [0,1,2,2,3,4,4,5,6],
		'inpos':  [0,1,2,2,3,4,4,5,6],
	},
	{ 'note': '【備考１】動詞の語尾の「う」は、「ウ」と書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.42',
		'text': '縫う',
		'input': 'ヌウ',
		'output': '⠍⠉',
		'inpos1': [0,1],
	},
	{ 'note': '【備考２】ウ音便はウ列・オ列の長音の表記との関連で、長音符を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.50',
		'text': 'ゆうて',
		'input': 'ユーテ',
		'output': '⠬⠒⠟',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2],
		'inpos':  [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.51',
		'text': '思うて',
		'input': 'オモーテ',
		'output': '⠊⠾⠒⠟',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.52',
		'text': '吸うたり',
		'input': 'スータリ',
		'output': '⠹⠒⠕⠓',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.53',
		'text': '繕うたり',
		'input': 'ツクロータリ',
		'output': '⠝⠩⠚⠒⠕⠓',
		'inpos1': [0,1,2,3,4,5],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.54',
		'text': '縫うた',
		'input': 'ヌータ',
		'output': '⠍⠒⠕',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2],
		'inpos':  [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.55',
		'text': '買うた',
		'input': 'コータ',
		'output': '⠪⠒⠕',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2],
		'inpos':  [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.56',
		'text': 'うれしゅう',
		'input': 'ウレシュー',
		'output': '⠉⠛⠈⠹⠒',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,1,2,3,4],
		'inpos':  [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.57',
		'text': '面白うて',
		'input': 'オモシローテ',
		'output': '⠊⠾⠳⠚⠒⠟',
		'inpos1': [0,1,2,3,4,5],
	},
	{ 'note': '【備考３】オ列の長音のうち、次の一覧表に掲げた和語とその派生語や複合語は「オ」を添えて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.58',
		'text': '多い',
		'input': 'オオイ',
		'output': '⠊⠊⠃',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.59',
		'text': '覆う',
		'input': 'オオウ',
		'output': '⠊⠊⠉',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.60',
		'text': '狼',
		'input': 'オオカミ',
		'output': '⠊⠊⠡⠷',
		'inpos2': [0,0,0,0],
		'inpos1': [0,1,2,3],
		'inpos':  [0,0,0,0],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.61',
		'text': '大きい',
		'input': 'オオキイ',
		'output': '⠊⠊⠣⠃',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5',
		'text': 'おおきい',
		'input': 'オオキイ',
		'output': '⠊⠊⠣⠃',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 「大きい」の派生語',
		'text': '大まか',
		'input': 'オオマカ',
		'output': '⠊⠊⠵⠡',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 「大きい」の派生語',
		'text': 'おおまか',
		'input': 'オオマカ',
		'output': '⠊⠊⠵⠡',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.62',
		'text': '仰せ',
		'input': 'オオセ',
		'output': '⠊⠊⠻',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.63',
		'text': '為おおせる',
		'input': 'シオオセル',
		'output': '⠳⠊⠊⠻⠙',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,1,2,3,4],
		'inpos':  [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.64',
		'text': '車前草',
		'input': 'オオバコ',
		'output': '⠊⠊⠐⠥⠪',
		'inpos1': [0,1,2,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.65',
		'text': '概ね',
		'input': 'オオムネ',
		'output': '⠊⠊⠽⠏',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.66',
		'text': '公',
		'input': 'オオヤケ',
		'output': '⠊⠊⠌⠫',
		'inpos2': [0,0,0,0],
		'inpos1': [0,1,2,3],
		'inpos':  [0,0,0,0],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.67',
		'text': '凡そ',
		'input': 'オオヨソ',
		'output': '⠊⠊⠜⠺',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.68',
		'text': '氷',
		'input': 'コオリ',
		'output': '⠪⠊⠓',
		'inpos2': [0,0,0],
		'inpos1': [0,1,2],
		'inpos':  [0,0,0],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.69',
		# 'text': '郡',
		'input': 'コオリ',
		'output': '⠪⠊⠓',
		'inpos1': [0,1,2],
		'comment': '郡: 文脈がないと読みを確定できない',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.70',
		'text': '凍る',
		'input': 'コオル',
		'output': '⠪⠊⠙',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.71',
		'text': '滞る',
		'input': 'トドコオル',
		'output': '⠞⠐⠞⠪⠊⠙',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.72',
		'text': '蟋蟀',
		'input': 'コオロギ',
		'output': '⠪⠊⠚⠐⠣',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.73',
		'text': '装う',
		'input': 'ヨソオウ',
		'output': '⠜⠺⠊⠉',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.74',
		# 'text': '十',
		'input': 'トオ',
		'output': '⠞⠊',
		'inpos1': [0,1],
		'comment': '十: 文脈がないと読みを確定できない',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.75',
		'text': '遠い',
		'input': 'トオイ',
		'output': '⠞⠊⠃',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.76',
		'text': '通る',
		'input': 'トオル',
		'output': '⠞⠊⠙',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.77',
		'text': '憤る',
		'input': 'イキドオル',
		'output': '⠃⠣⠐⠞⠊⠙',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.78',
		'text': 'いとおしい',
		'input': 'イトオシイ',
		'output': '⠃⠞⠊⠳⠃',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.79',
		'text': '炎',
		'input': 'ホノオ',
		'output': '⠮⠎⠊',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.80',
		'text': '頬',
		'input': 'ホオ',
		'output': '⠮⠊',
		'inpos1': [0,1],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.81',
		'text': '朴',
		'input': 'ホオ',
		'output': '⠮⠊',
		'inpos1': [0,1],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.82',
		'text': '酸漿',
		'input': 'ホオズキ',
		'output': '⠮⠊⠐⠹⠣',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.83',
		'text': '催す',
		'input': 'モヨオス',
		'output': '⠾⠜⠊⠹',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.84',
		'text': '大目',
		'input': 'オオメ',
		'output': '⠊⠊⠿',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.85',
		'text': '大通り',
		'input': 'オオドオリ',
		'output': '⠊⠊⠐⠞⠊⠓',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.86',
		'text': '凍り付く',
		'input': 'コオリツク',
		'output': '⠪⠊⠓⠝⠩',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.87',
		'text': '透徹る',
		'input': 'スキトオル',
		'output': '⠹⠣⠞⠊⠙',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.88',
		'text': '十日',
		'input': 'トオカ',
		'output': '⠞⠊⠡',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.89',
		'text': '遠ざかる',
		'input': 'トオザカル',
		'output': '⠞⠊⠐⠱⠡⠙',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.90',
		'text': '通す',
		'input': 'トオス',
		'output': '⠞⠊⠹',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 5 No.91',
		'text': '頬張る',
		'input': 'ホオバル',
		'output': '⠮⠊⠐⠥⠙',
	},
	{ 'note': '６、「ジ　ズ　ジャ　ジュ　ジョ」と「ヂ　ヅ　ヂャ　ヂョ」の使い分け' },
	{ 'note': '（１）サ行を用いることを原則とする。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.1',
		'text': '根治',
		'input': 'コンジ',
		'output': '⠪⠴⠐⠳',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.2',
		'text': '砂地',
		'input': 'スナジ',
		'output': '⠹⠅⠐⠳',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.3',
		'text': '無花果',
		'input': 'イチジク',
		'output': '⠃⠗⠐⠳⠩',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.10',
		'text': '砂利道',
		'input': 'ジャリミチ',
		'output': '⠘⠱⠓⠷⠗',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.20',
		'text': '少しずつ',
		'input': 'スコシズツ',
		'output': '⠹⠪⠳⠐⠹⠝',
	},
	{ 'note': '（２）タ行同音の連呼によって後ろの音節が濁る場合は、「ヂ」「ヅ」を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.25',
		'text': '縮む',
		'input': 'チヂム',
		'output': '⠗⠐⠗⠽',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.26',
		'text': '縮れ',
		'input': 'チヂレ',
		'output': '⠗⠐⠗⠛',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.27',
		'text': '続く',
		'input': 'ツヅク',
		'output': '⠝⠐⠝⠩',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.28',
		'text': '手続き',
		'input': 'テツヅキ',
		'output': '⠟⠝⠐⠝⠣',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.29',
		'text': '綴る',
		'input': 'ツヅル',
		'output': '⠝⠐⠝⠙',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.30',
		'text': '綴り方',
		'input': 'ツヅリカタ',
		'output': '⠝⠐⠝⠓⠡⠕',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.31',
		'text': '約やか',
		'input': 'ツヅマヤカ',
		'output': '⠝⠐⠝⠵⠌⠡',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.32',
		'text': '鼓',
		'input': 'ツヅミ',
		'output': '⠝⠐⠝⠷',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.33',
		'text': '葛籠',
		'input': 'ツヅラ',
		'output': '⠝⠐⠝⠑',
	},
	{ 'note': '（３）２語の複合によってタ行の後半の語頭が濁音に変わる連濁の場合は、「ヂ」「ヅ」「ヂャ」「ヂュ」「ヂョ」を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.34',
		'text': '鼻血',
		'input': 'ハナヂ',
		'output': '⠥⠅⠐⠗',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.35',
		'text': '身近',
		'input': 'ミヂカ',
		'output': '⠷⠐⠗⠡',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.36',
		'text': '底力',
		'input': 'ソコヂカラ',
		'output': '⠺⠪⠐⠗⠡⠑',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.37',
		'text': '湯呑茶碗',
		'input': 'ユノミヂャワン',
		'output': '⠬⠎⠷⠘⠕⠄⠴',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.38',
		'text': '一本調子',
		'input': 'イッポンヂョーシ',
		'output': '⠃⠂⠠⠮⠴⠘⠞⠒⠳',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.39',
		'text': '紅提灯',
		'input': 'ベニヂョーチン',
		'output': '⠐⠯⠇⠘⠞⠒⠗⠴',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.40',
		'text': '三日月',
		'input': 'ミカヅキ',
		'output': '⠷⠡⠐⠝⠣',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.41',
		'text': '気付く',
		'input': 'キヅク',
		'output': '⠣⠐⠝⠩',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.42',
		'text': '小突く',
		'input': 'コヅク',
		'output': '⠪⠐⠝⠩',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.43',
		'text': '近づく',
		'input': 'チカヅク',
		'output': '⠗⠡⠐⠝⠩',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.44',
		'text': 'つくづく',
		'input': 'ツクヅク',
		'output': '⠝⠩⠐⠝⠩',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.45',
		'text': '基づく',
		'input': 'モトヅク',
		'output': '⠾⠞⠐⠝⠩',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.46',
		'text': '心尽くし',
		'input': 'ココロヅクシ',
		'output': '⠪⠪⠚⠐⠝⠩⠳',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.47',
		'text': '小包',
		'input': 'コヅツミ',
		'output': '⠪⠐⠝⠝⠷',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.48',
		'text': '手綱',
		'input': 'タヅナ',
		'output': '⠕⠐⠝⠅',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.49',
		'text': '常々',
		'input': 'ツネヅネ',
		'output': '⠝⠏⠐⠝⠏',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.50',
		'text': '働き詰め',
		'input': 'ハタラキヅメ',
		'output': '⠥⠕⠑⠣⠐⠝⠿',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.51',
		'text': '理詰め',
		'input': 'リヅメ',
		'output': '⠓⠐⠝⠿',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 6 No.52',
		'text': '我慢強い',
		'input': 'ガマンヅヨイ',
		'output': '⠐⠡⠵⠴⠐⠝⠜⠃',
	},

	{ 'note': '７、現代仮名遣いで書かれた文章の中に歴史的仮名遣いが混じっているときは、現代仮名遣いに直して書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.1',
		'text': 'まづ',
		'input': 'マズ',
		'output': '⠵⠐⠹',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.2',
		'text': '一つづつ',
		'input': 'ヒトツズツ',
		'output': '⠧⠞⠝⠐⠹⠝',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.3',
		# 'text': 'なほ',
		'input': 'ナオ',
		'output': '⠅⠊',
		'inpos1': [0,1],
		'comment': '人名「なほ」と区別がつかない',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.4',
		'text': '思ふ',
		'input': 'オモウ',
		'output': '⠊⠾⠉',
		'inpos1': [0,1,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.5',
		'text': '大きう',
		'input': 'オオキュー',
		'output': '⠊⠊⠈⠩⠒',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.6',
		'text': 'うれしう',
		'input': 'ウレシュー',
		'output': '⠉⠛⠈⠹⠒',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.7',
		'text': 'もみぢ',
		'input': 'モミジ',
		'output': '⠾⠷⠐⠳',
		'inpos2': [0,1,2],
		'inpos1': [0,1,2,2],
		'inpos':  [0,1,2,2],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 1 7 No.8',
		'text': 'みづうみ',
		'input': 'ミズウミ',
		'output': '⠷⠐⠹⠉⠷',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,1,2,3],
		'inpos':  [0,1,1,2,3],
	},
	# 第２章　語の書き表し方
	# その１　仮名遣い
	{ 'note': '==== ２　その他の仮名遣い ====' },
	{ 'note': '１．外来語や外国語は原文の表記に従って書くことを原則とする。'},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.1',
		'text': 'フィルム',
		'input': 'フィルム',
		'output': '⠢⠧⠙⠽',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.2',
		'text': 'フイルム',
		'input': 'フイルム',
		'output': '⠭⠃⠙⠽',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.3',
		'text': 'バイオリン',
		'input': 'バイオリン',
		'output': '⠐⠥⠃⠊⠓⠴',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,0,1,2,3,4],
		'inpos':  [0,0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.4',
		'text': 'ヴァイオリン',
		'input': 'ヴァイオリン',
		'output': '⠲⠥⠃⠊⠓⠴',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.5',
		'text': 'ビタミン',
		'input': 'ビタミン',
		'output': '⠐⠧⠕⠷⠴',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.6',
		'text': 'ヴィタミン',
		'input': 'ヴィタミン',
		'output': '⠲⠧⠕⠷⠴',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.7',
		'text': 'カルシウム',
		'input': 'カルシウム',
		'output': '⠡⠙⠳⠉⠽',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.8',
		'text': 'カルシューム',
		'input': 'カルシューム',
		'output': '⠡⠙⠈⠹⠒⠽',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.9',
		'text': 'パーティー',
		'input': 'パーティー',
		'output': '⠠⠥⠒⠈⠗⠒',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.10',
		'text': 'パーティ',
		'input': 'パーティ',
		'output': '⠠⠥⠒⠈⠗',
	},
	{ 'note': '＜処理１＞外来語外国語に使われている「ヂ」「ヅ」「ヂャ」「ヂュ」「ヂョ」は、「ジ」「ズ」「ジャ」「ジュ」「ジョ」を用いて書いてよい。なお、特に必要があれば原文のかなづかいで書くことができる。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.22',
		'text': 'ラヂオ',
		'input': 'ラジオ',
		'output': '⠑⠐⠳⠊',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.23',
		'text': 'ヂャケット',
		'input': 'ジャケット',
		'output': '⠘⠱⠫⠂⠞',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.24',
		'text': '映画「ラヂオの時間」',
		'input': 'エイガ 「ラジオノ ジカン」',
		'output': '⠋⠃⠐⠡ ⠤⠑⠐⠳⠊⠎ ⠐⠳⠡⠴⠤', # 「」に 36 を使う
	},
	{ 'note': '＜処理2＞外来語や外国語に使われている「ヰ」「ヱ」「ヲ」は、「イ」「エ」「オ」あるいは「ウィ」「ウェ」「ウォ」を用いて書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.25',
		'text': 'ウヰスキー',
		'input': 'ウイスキー',
		'output': '⠉⠃⠹⠣⠒',
		'inpos1': [0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.26',
		'text': 'スヰフト',
		'input': 'スイフト', #スウィフト
		'output': '⠹⠃⠭⠞',
		'inpos1': [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.27',
		'text': 'ヱルテル',
		'input': 'ウェルテル',
		'output': '⠢⠋⠙⠟⠙',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.28',
		'text': 'ヲルポール',
		'input': 'ウォルポール',
		'output': '⠢⠊⠙⠠⠮⠒⠙',
	},
	{ 'note': '＜処理3＞児童書などで、全文ひらがな書きのため長音符を使わずに書かれている外来語や外国語は、発音を考慮して適宜長音符に書き換えてよい。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.29',
		'text': 'ヘリコプタア',
		'input': 'ヘリコプター',
		'output': '⠯⠓⠪⠠⠭⠕⠒',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.30',
		'text': 'ちゅうりっぷ',
		'input': 'チューリップ',
		'output': '⠈⠝⠒⠓⠂⠠⠭',
	},
	{ 'note': '＜処理4＞漢字で書かれた外来語や外国語を原文で書く場合は、ルビや辞書などで示される仮名の表記に従って書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 1 No.31',
		# 'text': '老酒',
		'input': 'ラオチュウ',
		'output': '⠑⠊⠈⠝⠉',
		'comment': '老酒: ルビがないと読みを確定できない',
	},
	{ 'note': '２．擬声語・擬態語・嘆声などは「基本的な仮名遣い」に準じて書くが、原文に長音記号や特殊音が使われているときは、それに従って長音符や特殊音点字を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.3',
		'text': 'シーンと',
		'input': 'シーント',
		'output': '⠳⠒⠴⠞',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.4',
		'text': 'しいんと',
		'input': 'シイント',
		'output': '⠳⠃⠴⠞',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.5',
		#'text': 'ちゅうちゅう',
		'input': 'チューチュー',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.6',
		#'text': 'ぐうぐう',
		'input': 'グーグー',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.10',
		#'text': 'めえめえ',
		'input': 'メエメエ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.12',
		#'text': 'ごおごお',
		'input': 'ゴオゴオ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.13',
		#'text': 'ぎゃおーっ',
		'input': 'ギャオーッ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.14',
		#'text': 'があーん',
		'input': 'ガアーン',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.17',
		#'text': 'がおううううう',
		'input': 'ガオウウウウー',
	},
	{ 'note': '＜処理＞原文で長音記号に代えて波線や棒線などが使われているときも、長音符を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.18',
		'text': 'ふにゃ～',
		'input': 'フニャー',
		'output': '⠭⠈⠅⠒',
		'inpos2': [0,1,2,3],
		'inpos1': [0,1,2,3],
		'inpos':  [0,1,2,3],
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 2 No.19',
		#'text': 'まぁ――っ',
		'input': 'マアーッ',
	},
	{ 'note': '３．方言は原文の表記に従って書くことを原則とするが、長音は「基本的な仮名遣い」に準じて書いてよい。なお、原文に長音符号が使われているときは、それに従って長音符を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 3 No.2',
		'text': 'しぇんしぇい',
		'input': 'シェンシェイ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 3 No.4',
		'text': 'おおきに',
		'input': 'オオキニ',
	},
	{ 'note': '＜処理＞原文の表記に従うと不自然な音になる場合は、その発音に使いと思われる点字を用いて書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 3 No.6',
		'text': 'おみやぁさん',
		'input': 'オミャアサン',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 3 No.7',
		'text': '先生ぇさまぁ',
		'input': 'センセエサマア',
	},
	{ 'note': '４．原文に特殊音点字を用いても書き表せない文字が使われているときは、なるべくその発音に近い点字を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.1',
		'text': 'おとゥ、おかァ',
		'input': 'オトー、オカア',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.2',
		'text': 'ヱ゛ルレエヌ',
		'input': 'ヴェルレエヌ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.3',
		'text': 'ヲ゛ルガ',
		'input': 'ヴォルガ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.4',
		'text': 'ガリワ゛ー',
		'input': 'ガリヴァー',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.5',
		'text': 'クヮルテット',
		'input': 'クァルテット',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.6',
		'text': 'ヂェスチャー',
		'input': 'ジェスチャー',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.7',
		'text': 'スェーター',
		'input': 'スエーター',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.8',
		'text': 'ヒァーッたまげた',
		'input': 'ヒャーッ タマゲタ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 4 No.9',
		'text': 'あ゛ー',
		'input': 'アー',
	},
	{ 'note': '５．固有名詞の仮名遣い' },
	{ 'note': '(1) 漢字で書かれた固有名詞は「基本的な仮名遣い」に準じて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.1',
		'text': '大阪',
		'input': 'オオサカ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.2',
		'text': '遠野',
		'input': 'トオノ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.3',
		'text': '東井',
		'input': 'トーイ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.10',
		'text': '青梅',
		'input': 'オーメ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.11',
		'text': '透',
		'input': 'トオル',
	},

	{ 'note': '(2)仮名で書かれた固有名詞は原本の仮名遣いに従って書くが、長音は「基本的な仮名遣い」に準じて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.13',
		'text': 'みさを',
		'input': 'ミサヲ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.14',
		'text': 'かほる',
		'input': 'カホル',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.19',
		#'text': 'さをり織り',
		'input': 'サヲリオリ',
	},
	{ 'note': '＜処理1＞ワ行の「ヱ」は、感嘆符と形が同じなので、ア行の「エ」に変えることを原則とする。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.22',
		'text': 'ヱビスビール',
		'input': 'エビス ビール',
	},
	{ 'note': '＜処理2＞原文で長音符号が使われているときは、長音符を用いて書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.23',
		#'text': 'オーヤマ書店',
		'input': 'オーヤマ ショテン',
	},
	{ 'note': '＜処理3＞主となる固有名詞部分以外の語に使われている歴史的仮名遣いは、現代仮名遣いに直して書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.24',
		'text': 'みやこをどり',
		'input': 'ミヤコ オドリ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.25',
		'text': '青木をりがみ教室',
		'input': 'アオキ オリガミ キョーシツ',
	},
	{ 'note': '＜処理4＞外来語や疑似外来語などを使って仮名で表記されている法人名や商品名などは、その仮名遣いに従って書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 5 No.26',
		'text': '日本コロムビア',
		'input': 'ニッポン コロムビア',
	},
	{ 'note': '６．歴史的仮名遣いで書かれた文や語句が現代文の中に挿入されているときは、次の４種類の方法の中から原文に応じて最もふさわしい方法を選んで点訳する。なお、特に差し支えがない場合は(ｱ)の方法を用いる。' },
	{ 'note': '(ｱ)現代文の仮名遣いで書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 6 No.3',
		#'text': '問ふ',
		'input': 'トウ',
	},
	{ 'note': '(ｲ)現代文の仮名遣いで書いたあとに、古文の仮名遣いをカッコ類で囲んで書く。' },
	{ 'note': '(ｳ)古文の仮名遣いで書いたあとに、現代文の仮名遣いをカッコ類で囲んで書く。' },
	{ 'note': '(ｴ)古文の仮名遣いで書く。' },
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 6 No.10',
		#'text': 'あはれ',
		'input': 'アハレ',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 6 No.11',
		# 'text': '弥生',
		'input': 'ヤヨヒ',
		'output': '⠌⠜⠧',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 6 No.12',
		# 'text': '問ふ',
		'input': 'トフ',
		'output': '⠞⠭',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 6 No.13',
		# 'text': '末',
		'input': 'スヱ',
		'output': '⠹⠖',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 6 No.14',
		# 'text': '居り',
		'input': 'ヲリ',
		'output': '⠔⠓',
	},
	{
		'comment': '点訳のてびき第3版 第2章 その1 2 6 No.15',
		# 'text': '静けさ',
		'input': 'シヅケサ',
		'output': '⠳⠐⠝⠫⠱',
	},

	{ 'note': '=== その２　数字 ===' },
	{ 'note': '==== １　数の書き方 ====' },
	{ 'note': '１．数字は数符（⠼）を前置し、４桁までは一続きに書く。' },
	{
		'text': '1',
		'input': '1',
		'output': '⠼⠁',
		'inpos2': [0],
		'inpos1': [0,0],
		'inpos':  [0,0],
	},
	{
		'text': '二十三',
		'input': '23',
	},
	{
		'text': '五百',
		'input': '500',
	},
	{
		'text': '二千三百',
		'input': '2300',
	},
	{
		'text': '2001',
		'input': '2001',
		'output': '⠼⠃⠚⠚⠁',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,2,3],
		'inpos':  [0,0,1,2,3],
	},
	{
		'text': '5000',
		'input': '5000',
		'output': '⠼⠑⠚⠚⠚',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,2,3],
		'inpos':  [0,0,1,2,3],
	},
	{
		#'text': '一千万',
		'input': '1センマン',
	},
	{
		#'text': '三千億',
		'input': '3ゼンオク',
	},
	{ 'note': '２．万・億・兆などはその発音を仮名で書き、４桁ごとに区切る。' },
	{
		#'text': '54309',
		'input': '5マン 4309',
	},
	{
		#'text': '10500',
		'input': '1マン 500',
	},
	{
		'text': '三兆二千四百万',
		'input': '3チョー 2400マン',
	},
	{ 'note': '３．小数は小数点（⠂）を用いて書く。' },
	{
		'text': '3.14',
		'input': '3.14',
		'output': '⠼⠉⠂⠁⠙',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,2,3],
		'inpos':  [0,0,1,2,3],
	},
	{
		'text': '0.01',
		'input': '0.01',
		'output': '⠼⠚⠂⠚⠁',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,2,3],
		'inpos':  [0,0,1,2,3],
	},
	{ 'note': '４．分数' },
	{ 'note': '５．「千」「百」「十」が位を表している場合には、それぞれの仮名を用いて書く。' },
	{
		'text': '数千',
		'input': 'スーセン',
	},
	{
		'text': '数百',
		'input': 'スーヒャク',
	},
	{
		'text': '何百',
		'input': 'ナンビャク',
	},
	{
		'text': '何十',
		'input': 'ナンジュー', # ナン10
	},
	{ 'note': '６．およその数で数が重なる場合は、読点などを用いず、それぞれに数符を前置して続けて書く。' },
	{
		'text': '二、三',
		'input': '2⠼3',
		'output': '⠼⠃⠼⠉',
	},
	{
		#'text': '二百五、六十',
		'input': '2ヒャク 5⠼60',
		'output': '⠼⠃⠈⠥⠩ ⠼⠑⠼⠋⠚',
	},
	{
		'text': '二一',
		'input': '21',
		'output': '⠼⠃⠁',
	},
	{
		'text': '二十一',
		'input': '21',
		'output': '⠼⠃⠁',
	},
	{
		'text': '二十一二',
		'input': '21⠼2',
		'output': '⠼⠃⠁⠼⠃',
	},
	{
		'text': '二十二三',
		'input': '22⠼3',
		'output': '⠼⠃⠃⠼⠉',
	},

	{ 'note': '７．特に必要があればそのまま数字を並べて書くことができる。その場合は３桁ごとに位取り点（⠄）を用いて書いてよい。' },
	{
		'text': '53,000',
		'input': '53,000',
		'output': '⠼⠑⠉⠄⠚⠚⠚',
		'inpos2': [0,1,2,3,4,5],
		'inpos1': [0,0,1,2,3,4,5],
		'inpos':  [0,0,1,2,3,4,5],
	},
	{
		'text': '１，２３０',
		'input': '1,230',
		'output': '⠼⠁⠄⠃⠉⠚',
	},
	{
		'text': '１，',
		'input': '1,',
		'output': '⠼⠁⠄',
	},
	{
		'text': '，',
		'input': ',',
		'output': '⠄',
	},
	{
		'text': '１２３ー４５',
		'input': '123ー45',
		'output': '⠼⠁⠃⠉⠒⠼⠙⠑',
	},
	{
		'text': '１２３－４５',
		'input': '123-45',
		'output': '⠼⠁⠃⠉⠤⠼⠙⠑',
	},
	{ 'note': '８．年号の略記に使われているアポストロフィは、数符の直後にアポストロフィ(3)を用いて書く。' },
	{
		'text': "'02",
		'input': "'02", # text2mecab で全角に変換されるが元に戻す
		'output': '⠼⠄⠚⠃',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,2],
		'inpos':  [0,0,1,2],
	},
	{ 'note': '==== ２　数を含む言葉の書き方 ====' },
	{ 'note': '１．数字を漢字音で発音する場合' },
	{ 'note': '(1) 数を含む言葉は数字で表すことを原則とし、数字に続く言葉（助数詞など）は続けて書く。また、数に続く言葉はア行・ラ行のいずれかで始まる場合は、数字と形が同じなので、第一つなぎ符(36)をはさんで続けて書く。' },
	{
		'text': '1枚',
		'input': '1マイ',
	},
	{
		'text': '三十センチメートル',
		'input': '30センチメートル',
	},
	{
		'text': '五割',
		'input': '5ワリ',
	},
	{
		'text': '六百人',
		'input': '600ニン',
	},
	{
		'text': '一流品',
		'input': '1リューヒン',
	},
	{
		'text': '第2章',
		'input': 'ダイ2ショー',
	},
	{
		'text': '中2階',
		'input': 'チュー2カイ',
	},
	{
		#'text': 'お三方',
		'input': 'オ3カタ',
	},
	{
		#'text': '3三振',
		'input': '3⠼3シン',
		'output': '⠼⠉⠼⠉⠳⠴',
	},
	{
		'text': '四季',
		'input': '4キ',
	},
	{
		'text': '四半期',
		'input': '4ハンキ',
	},
	{
		'text': '四角形',
		'input': '4カクケイ',
	},
	{
		#'text': '二男一女',
		'input': '2ナン 1ジョ',
	},
	{
		'text': '十数人',
		'input': '10スーニン',
	},
	{
		#'text': '五千円札',
		'input': '5センエンサツ',
	},
	{
		#'text': '1000万光年',
		'input': '1センマンコーネン',
	},
	{
		'text': '西暦2000年',
		'input': 'セイレキ 2000ネン',
	},
	{
		'text': '零時十五分',
		'input': '0ジ 15フン',
	},
	{
		'text': '一列',
		'input': '1レツ',
		'output': '⠼⠁⠤⠛⠝',
	},
	{
		'text': '一輪車',
		'input': '1リンシャ',
		'output': '⠼⠁⠤⠓⠴⠈⠱',
	},
	{
		'text': '二億年',
		'input': '2オクネン',
		'output': '⠼⠃⠤⠊⠩⠏⠴',
	},
	{
		'text': '三塁打',
		'input': '3ルイダ',
		'output': '⠼⠉⠤⠙⠃⠐⠕',
	},
	{
		'text': '五十円',
		'input': '50エン',
		'output': '⠼⠑⠚⠤⠋⠴',
	},
	{ 'note': '【備考】数字の後ろに助詞・助動詞が続く場合も続けて書く。その場合、ア行・ラ行で始まる助詞・助動詞との間は、第一つなぎ符をはさんで続けて書く。' },
	{
		'text': '101の',
		'input': '101ノ',
		'output': '⠼⠁⠚⠁⠎',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,2,3],
		'inpos':  [0,0,1,2,3],
	},
	{
		'text': '3です',
		'input': '3デス',
		'output': '⠼⠉⠐⠟⠹',
		'inpos2': [0,1,2],
		'inpos1': [0,0,1,1,2],
		'inpos':  [0,0,1,1,2],
	},
	{
		'text': '4120へ',
		'input': '4120エ',
		'output': '⠼⠙⠁⠃⠚⠤⠋',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,0,1,2,3,3,4],
		'inpos':  [0,0,1,2,3,3,4],
	},
	{
		'text': '18らしい',
		'input': '18ラシイ',
		'output': '⠼⠁⠓⠤⠑⠳⠃',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,0,1,1,2,3,4],
		'inpos':  [0,0,1,1,2,3,4],
	},
	{ 'note': '＜処理＞数字に続く言葉との関係で数字の読みが促音化しても、促音符は用いず、数字の後ろに仮名を続けて書く。' },
	{
		'text': '1足',
		'input': '1ソク',
	},
	{
		#'text': '10本',
		'input': '10ポン',
	},
	{
		'text': '一汁一菜',
		'input': '1ジュー 1サイ',
	},
	{
		'text': '一丁目一番地',
		'input': '1チョーメ 1バンチ',
	},
	{
		'text': '五臓六腑',
		'input': '5ゾー 6プ',
	},
	{ 'note': '(2) 一語中で、月と日を省略して書いてあるときや、数字が並んでいるときは、中点などを省略して続けて書く。' },
	{
		'text': '二・二六事件',
		'input': '2⠼26 ジケン',
		'output': '⠼⠃⠼⠃⠋ ⠐⠳⠫⠴',
		'inpos1': [0,0,1,2,3,4,5,5,6,7],
	},
	{
		'text': '七・五・三',
		'input': '7⠼5⠼3',
		'output': '⠼⠛⠼⠑⠼⠉',
		'inpos1': [0,0,1,2,3,4],
	},
	{
		'text': '六・三・三制',
		'input': '6⠼3⠼3セイ',
		'output': '⠼⠋⠼⠉⠼⠉⠻⠃',
		'inpos1': [0,0,1,2,3,4,5,6],
	},
	{ 'note': '(3) 数量や順序の意味が薄れた言葉や、漢数字の形を表している場合などは仮名で書く。' },
	{
		'text': '青二才',
		'input': 'アオニサイ',
	},
	{
		'text': '一足飛び',
		'input': 'イッソクトビ',
	},
	{
		'text': '一般的',
		'input': 'イッパンテキ',
	},
	{
		'text': '一生',
		'input': 'イッショー',
	},
	{
		'text': '無二',
		'input': 'ムニ',
	},
	{
		'text': '一辺倒',
		'input': 'イッペントー',
	},
	{
		'text': '四角い',
		'input': 'シカクイ',
	},
	{
		'text': '七面鳥',
		'input': 'シチメンチョー',
	},
	{
		'text': '零下30度',
		'input': 'レイカ 30ド',
	},
	{
		'text': '真一文字',
		'input': 'マイチモンジ',
	},
	{
		'text': '十字路',
		'input': 'ジュージロ',
	},
	{
		#'text': '八の字をよせる',
		'input': 'ハチノジヲ ヨセル',
	},
	{ 'note': '＜処理＞明らかにアラビア数字の形を表している場合は、数字を用いて書く。' },
	{
		#'text': '8の字形',
		'input': '8ノジガタ',
	},
	{ 'note': '(4) 同じ発音の語でも、数量や順序の意味のあるなしで、数字と仮名に書き分ける。' },
	{
		#'text': '一番になった',
		'input': '1バンニ ナッタ',
	},
	{
		'text': '一番大きい',
		'input': 'イチバン オオキイ',
	},
	{
		'text': '一段昇る',
		'input': '1ダン ノボル',
	},
	{
		'text': '一段ときれいだ',
		'input': 'イチダント キレイダ',
	},
	{
		#'text': 'ちょっと一杯',
		'input': 'チョット 1パイ',
	},
	{
		#'text': '一杯になった',
		'input': 'イッパイニ ナッタ',
	},
	{
		'text': '第一段階',
		'input': 'ダイ1 ダンカイ',
	},
	{
		#'text': '第一、金がない',
		'input': 'ダイイチ、カネガ ナイ',
	},
	{ 'note': '(5) 数字の読みが仮名で書かれていても、数量や順序の意味がある場合は数字を用いて書く。' },
	{
		#'text': 'いちにち',
		'input': '1ニチ',
	},
	{
		#'text': 'もういちど',
		'input': 'モー 1ド',
	},
	{
		#'text': 'いちばん星',
		'input': '1バンボシ',
	},
	{
		'text': 'さんりんしゃ',
		'input': '3リンシャ',
	},
	{ 'note': '２．数字を和語読みする場合' },
	{ 'note': ' (1) 数を含む言葉で、和語読みするものは仮名で書く。' },
	{
		'text': '一人',
		'input': 'ヒトリ',
	},
	{
		'text': '二人',
		'input': 'フタリ',
	},
	{
		'text': '一言',
		'input': 'ヒトコト',
	},
	{
		'text': '三日月',
		'input': 'ミカヅキ',
	},
	{
		'text': '四隅',
		'input': 'ヨスミ',
	},
	{
		'text': '五十路',
		'input': 'イソジ',
	},
	{
		'text': '明け六つ',
		'input': 'アケムツ',
	},
	{
		'text': '七草',
		'input': 'ナナクサ',
	},
	{
		'text': '七転び八起き',
		'input': 'ナナコロビ ヤオキ',
	},
	{
		'text': '九重',
		'input': 'ココノエ',
	},
	{
		'text': '十重二十重',
		'input': 'トエ ハタエ',
	},
	{
		'text': '二十日',
		'input': 'ハツカ',
	},
	{
		'text': '二十歳',
		'input': 'ハタチ',
	},
	#{
	#	'text': '三十日',
	#	'input': '30ニチ', # 'ミソカ',
	#	},
	{
		'text': '一つから十',
		'input': 'ヒトツカラ 10', # 'ヒトツカラ トオ',
	},
	{
		'text': '一日から十日',
		'input': 'ツイタチカラ トオカ',
	},
	{
		'text': '1日から10日',
		'input': 'ツイタチカラ トオカ',
	},
	{
		'text': '一日から一〇日',
		'input': 'ツイタチカラ トオカ',
	},
	{
		'text': '1から10',
		'input': '1カラ 10',
	},

	{ 'note': '【備考】漢字音と和語読みが混じる場合も、和語読みの部分は仮名で書く。' },
	{
		'text': '3泊4日',
		'input': '3パク ヨッカ',
	},
	{
		'text': '二百十日',
		'input': '2ヒャク トオカ',
	},
	{ 'note': '(2) 漢字音の系列に入っているものは、読み替えによって和語と同じ発音になっても数字で書く。' },
	{
		'text': '四時',
		'input': '4ジ',
	},
	{
		'text': '四人',
		'input': '4ニン',
	},
	{
		'text': '四冊',
		'input': '4サツ',
	},
	{
		'text': '十四日',
		'input': '14カ',
	},
	{
		'text': '二十四日',
		'input': '24カ',
	},
	{
		#'text': '三、四日',
		'input': '3⠼4カ',
	},
	{
		'text': '三三七拍子',
		'input': '3⠼3⠼7ビョーシ',
	},
	{
		'text': '七個',
		'input': '7コ',
	},
	{
		'text': '十七歳',
		'input': '17サイ',
	},
	{
		'text': '百十番',
		'input': '110バン',
	},

	{ 'text': '１日', 'input': 'ツイタチ',},
	{ 'text': '２日', 'input': 'フツカ',},
	{ 'text': '３日', 'input': 'ミッカ',},
	{ 'text': '４日', 'input': 'ヨッカ',},
	{ 'text': '５日', 'input': 'イツカ',},
	{ 'text': '６日', 'input': 'ムイカ',},
	{ 'text': '７日', 'input': 'ナノカ',},
	{ 'text': '８日', 'input': 'ヨーカ',},
	{ 'text': '９日', 'input': 'ココノカ',},
	{ 'text': '１０日', 'input': 'トオカ',},
	{ 'text': '１１日', 'input': '11ニチ',},
	{ 'text': '１２日', 'input': '12ニチ',},
	{ 'text': '１３日', 'input': '13ニチ',},
	{ 'text': '１４日', 'input': '14カ',},
	{ 'text': '１５日', 'input': '15ニチ',},
	{ 'text': '１６日', 'input': '16ニチ',},
	{ 'text': '１７日', 'input': '17ニチ',},
	{ 'text': '１８日', 'input': '18ニチ',},
	{ 'text': '１９日', 'input': '19ニチ',},
	{ 'text': '２０日', 'input': 'ハツカ',},
	{ 'text': '２１日', 'input': '21ニチ',},
	{ 'text': '２２日', 'input': '22ニチ',},
	{ 'text': '２３日', 'input': '23ニチ',},
	{ 'text': '２４日', 'input': '24カ',},
	{ 'text': '２５日', 'input': '25ニチ',},
	{ 'text': '２６日', 'input': '26ニチ',},
	{ 'text': '２７日', 'input': '27ニチ',},
	{ 'text': '２８日', 'input': '28ニチ',},
	{ 'text': '２９日', 'input': '29ニチ',},
	{ 'text': '３０日', 'input': '30ニチ',},
	{ 'text': '３１日', 'input': '31ニチ',},

	{ 'text': '一日', 'input': 'ツイタチ',},
	{ 'text': '二日', 'input': 'フツカ',},
	{ 'text': '三日', 'input': 'ミッカ',},
	{ 'text': '四日', 'input': 'ヨッカ',},
	{ 'text': '五日', 'input': 'イツカ',},
	{ 'text': '六日', 'input': 'ムイカ',},
	{ 'text': '七日', 'input': 'ナノカ',},
	{ 'text': '八日', 'input': 'ヨーカ',},
	{ 'text': '九日', 'input': 'ココノカ',},
	{ 'text': '一〇日', 'input': 'トオカ',},
	{ 'text': '一一日', 'input': '11ニチ',},
	{ 'text': '一二日', 'input': '12ニチ',},
	{ 'text': '一三日', 'input': '13ニチ',},
	{ 'text': '一四日', 'input': '14カ',},
	{ 'text': '一五日', 'input': '15ニチ',},
	{ 'text': '一六日', 'input': '16ニチ',},
	{ 'text': '一七日', 'input': '17ニチ',},
	{ 'text': '一八日', 'input': '18ニチ',},
	{ 'text': '一九日', 'input': '19ニチ',},
	{ 'text': '二〇日', 'input': 'ハツカ',},
	{ 'text': '二一日', 'input': '21ニチ',},
	{ 'text': '二二日', 'input': '22ニチ',},
	{ 'text': '二三日', 'input': '23ニチ',},
	{ 'text': '二四日', 'input': '24カ',},
	{ 'text': '二五日', 'input': '25ニチ',},
	{ 'text': '二六日', 'input': '26ニチ',},
	{ 'text': '二七日', 'input': '27ニチ',},
	{ 'text': '二八日', 'input': '28ニチ',},
	{ 'text': '二九日', 'input': '29ニチ',},
	{ 'text': '三〇日', 'input': '30ニチ',},
	{ 'text': '三一日', 'input': '31ニチ',},

	{ 'text': '十日', 'input': 'トオカ',},
	{ 'text': '十一日', 'input': '11ニチ',},
	{ 'text': '十二日', 'input': '12ニチ',},
	{ 'text': '十三日', 'input': '13ニチ',},
	{ 'text': '十四日', 'input': '14カ',},
	{ 'text': '十五日', 'input': '15ニチ',},
	{ 'text': '十六日', 'input': '16ニチ',},
	{ 'text': '十七日', 'input': '17ニチ',},
	{ 'text': '十八日', 'input': '18ニチ',},
	{ 'text': '十九日', 'input': '19ニチ',},
	{ 'text': '二十日', 'input': 'ハツカ',},
	{ 'text': '二十一日', 'input': '21ニチ',},
	{ 'text': '二十二日', 'input': '22ニチ',},
	{ 'text': '二十三日', 'input': '23ニチ',},
	{ 'text': '二十四日', 'input': '24カ',},
	{ 'text': '二十五日', 'input': '25ニチ',},
	{ 'text': '二十六日', 'input': '26ニチ',},
	{ 'text': '二十七日', 'input': '27ニチ',},
	{ 'text': '二十八日', 'input': '28ニチ',},
	{ 'text': '二十九日', 'input': '29ニチ',},
	{ 'text': '三十日', 'input': '30ニチ',},
	{ 'text': '三十一日', 'input': '31ニチ',},

	{ 'note': '３．外国語の読みをする場合は漢字音に準じて書く。' },
	{
		'text': 'ボーイング747',
		'input': 'ボーイング 747',
	},
	{
		'text': 'ベスト10',
		'input': 'ベスト 10',
	},
	{
		'text': '3ランホームラン',
		'input': '3ラン ホームラン',
	},
	{ 'note': '４．地名や人名などの固有名詞の中に使われている数字は、仮名で書くことを原則とする。' },
	{
		'text': '八戸市',
		'input': 'ハチノヘシ',
	},
	{
		'text': '九州',
		'input': 'キューシュー',
	},
	{
		'text': '四国',
		'input': 'シコク',
	},
	{
		#'text': '八幡平',
		'input': 'ハチマンタイ',
	},
	{
		'text': '一郎',
		'input': 'イチロー',
	},
	{
		'text': '直木三十五',
		'input': 'ナオキ サンジューゴ',
	},
	{
		'text': '赤十字社',
		'input': 'セキジュージシャ',
	},
	{ 'note': '【備考】固有名詞であっても、数量や順序の意味を明確にする必要がある場合などには数字を用いて書く。' },
	{
		'text': 'ルイ十四世',
		'input': 'ルイ 14セイ',
	},
	{
		'text': '二月堂',
		'input': '2ガツドー',
	},
	{
		#'text': '市立一中',
		'input': 'シリツ 1チュー',
	},
	{
		'text': '三十三間堂',
		'input': '33ゲンドー',
	},
	{
		'text': '四条河原町',
		'input': '4ジョー カワラマチ',
	},
	{
		'text': '劇団四季',
		'input': 'ゲキダン 4キ',
	},
	{ 'note': '５．アルファベットやハイフンなど他の記号と一緒に使われている数字や、4桁で区切ったり仮名で書くと不自然になるような数字は、原文の表記に従って書く。' },
	{
		'text': '連絡先090-8000-7603',
		'input': 'レンラクサキ 090-8000-7603',
		'output': '⠛⠴⠑⠩⠱⠣ ⠼⠚⠊⠚⠤⠼⠓⠚⠚⠚⠤⠼⠛⠋⠚⠉',
		'inpos1': [0,1,2,3,4,5,6,7,7,8,9,10,11,11,12,13,14,15,16,16,17,18,19],
	},
	{
		'text': '振替00150-8-44522',
		'input': 'フリカエ 00150-8-44522',
		'output': '⠭⠓⠡⠋ ⠼⠚⠚⠁⠑⠚⠤⠼⠓⠤⠼⠙⠙⠑⠃⠃',
		'inpos1': [0,1,2,3,4,5,5,6,7,8,9,10,11,11,12,13,13,14,15,16,17],
	},
	{
		'text': 'ISBN 4-925053-14-0',
		'input': 'ISBN 4-925053-14-0',
		'output': '⠰⠠⠠⠊⠎⠃⠝ ⠼⠙⠤⠼⠊⠃⠑⠚⠑⠉⠤⠼⠁⠙⠤⠼⠚',
		'inpos2': [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17],
		'inpos1': [0,0,0,0,1,2,3,4,5,5,6,7,7,8,9,10,11,12,13,14,14,15,16,17,17],
		'inpos':  [0,0,0,0,1,2,3,4,5,5,6,7,7,8,9,10,11,12,13,14,14,15,16,17,17],
	},
	{
		'text': 'No.2514269',
		'input': 'No. 2514269',
		'output': '⠰⠠⠝⠕⠲ ⠼⠃⠑⠁⠙⠃⠋⠊',
		'inpos1': [0,0,0,1,2,3,4,4,5,6,7,8,9,10],
	},

	{ 'note': '==== ３ ローマ数字の書き方 ===='},
	{
		'text': 'ⅰ',
		'input': 'i',
		'output': '⠰⠊',
		'inpos2': [0],
		'inpos1': [0,0],
		'inpos':  [0,0],
	},
	{
		'text': 'Ⅴ',
		'input': 'V',
		'output': '⠰⠠⠧',
		'inpos2': [0],
		'inpos1': [0,0,0],
		'inpos':  [0,0,0],
	},
	{
		'text': 'Ⅶ',
		'input': 'VII',
		'output': '⠰⠠⠠⠧⠊⠊',
		'inpos1': [0,0,0,0,1,2],
	},
	{
		'text': 'Ⅹ',
		'input': 'X',
		'output': '⠰⠠⠭',
		'inpos2': [0],
		'inpos1': [0,0,0],
		'inpos':  [0,0,0],
	},

	{ 'note': '=== その３　アルファベット ===' },
	{ 'note': '==== １　文字として書き表す場合 ====' },
	{ 'note': '１．アルファベットは外字符（56）を前置して書く。略字など２文字以上の場合でも、一つの外字符に続けて書く。大文字は該当する文字に大文字符(6)を前置して書くが、一続きの文字がすべて大文字の場合は、外字符の直後に二重大文字符（6）を続けて書く。なお、一続きの文字の一部でも、後ろに続く文字すべてが大文字であれば、その位置に二重大文字符を用いて書くことができる。' },
	{
		'text': 'ビタミンE',
		'input': 'ビタミン E',
		'output': '⠐⠧⠕⠷⠴ ⠰⠠⠑',
		'inpos1': [0,0,1,2,3,4,5,5,5],
	},
	{
		'text': 'TV',
		'input': 'TV',
		'output': '⠰⠠⠠⠞⠧',
		'inpos2': [0,1],
		'inpos1': [0,0,0,0,1],
		'inpos':  [0,0,0,0,1],
	},
	{
		'text': 'PTA',
		'input': 'PTA',
		'output': '⠰⠠⠠⠏⠞⠁',
		'inpos2': [0,1,2],
		'inpos1': [0,0,0,0,1,2],
		'inpos':  [0,0,0,0,1,2],
	},
	{
		'text': 'OPEC',
		'input': 'OPEC',
		'output': '⠰⠠⠠⠕⠏⠑⠉',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,0,0,1,2,3],
		'inpos':  [0,0,0,0,1,2,3],
	},
	{
		'text': 'AM',
		'input': 'AM',
		'output': '⠰⠠⠠⠁⠍',
		'inpos2': [0,1],
		'inpos1': [0,0,0,0,1],
		'inpos':  [0,0,0,0,1],
	},
	{
		'text': 'AMe',
		'input': 'AMe',
		'output': '⠰⠠⠁⠠⠍⠑',
		'inpos2': [0,1,2],
		'inpos1': [0,0,0,1,1,2],
		'inpos':  [0,0,0,1,1,2],
	},
	{
		'text': 'AMeDAS',
		'input': 'AMeDAS',
		'output': '⠰⠠⠁⠠⠍⠑⠠⠙⠠⠁⠠⠎',
		'inpos2': [0,1,2,3,4,5],
		'inpos1': [0,0,0,1,1,2,3,3,4,4,5,5],
		'inpos':  [0,0,0,1,1,2,3,3,4,4,5,5],
		'comment': '大文字と小文字が混在する単語では二重大文字符を使わないほうが読みやすい',
	},
	{ 'note': '【備考１】' },
	{
		#'text': 'Ｐ・Ｔ・Ａ',
		'input': 'PTA',
		'output': '⠰⠠⠠⠏⠞⠁',
		'inpos1': [0,0,0,0,1,2],
	},
	{ 'note': '【備考２】' },
	{
		#'text': 'Ｕ．Ｓ．Ａ．',
		'input': 'USA',
		'output': '⠰⠠⠠⠥⠎⠁',
		'inpos1': [0,0,0,0,1,2],
	},
	{ 'note': '【備考３】二つ以上の略称がハイフンで結ばれているときは、それぞれに外字符や大文字符・二重大文字符を前置し、第１つなぎ符を用いて書く。' },
	{
		'text': 'MS-DOS',
		'input': 'MS-DOS',
		'output': '⠰⠠⠠⠍⠎⠤⠰⠠⠠⠙⠕⠎',
		'inpos2': [0,1,2,3,4,5],
		'inpos1': [0,0,0,0,1,2,3,3,3,3,4,5],
		'inpos':  [0,0,0,0,1,2,3,3,3,3,4,5],
		'comment': '36でつなぐとカナに戻る（ハイフンで外字符の効力がなくなる）',
	},
	{
		'text': 'ＭＳ－ＤＯＳ',
		'input': 'MS-DOS',
		'output': '⠰⠠⠠⠍⠎⠤⠰⠠⠠⠙⠕⠎',
		'inpos2': [0,1,2,3,4,5],
		'inpos1': [0,0,0,0,1,2,3,3,3,3,4,5],
		'inpos':  [0,0,0,0,1,2,3,3,3,3,4,5],
	},
	{ 'note': '【備考４】アルファベットで書かれた略称の間の「／」に限り、(34) の記号を用いて書く。一つの外字符にアルファベットと記号を続けて書くが、大文字符や二重大文字符は改めて前置する。' },
	{
		'text': 'DOS/V',
		'input': 'DOS/V',
		'output': '⠰⠠⠠⠙⠕⠎⠌⠠⠧',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,0,0,0,1,2,3,4,4],
		'inpos':  [0,0,0,0,1,2,3,4,4],
		'comment': '／は456-34という新記号がある',
	},
	{
		'text': 'ＤＯＳ／Ｖ',
		'input': 'DOS/V',
		'output': '⠰⠠⠠⠙⠕⠎⠌⠠⠧',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,0,0,0,1,2,3,4,4],
		'inpos':  [0,0,0,0,1,2,3,4,4],
		'comment': '／は456-34という新記号がある',
	},
	{ 'note': '２．一続きに書き表すべき１語中のアルファベットと仮名の間は、第１つなぎ符をはさんで続けて書く。' },
	{
		'text': 'A型',
		'input': 'Aガタ',
		'output': '⠰⠠⠁⠤⠐⠡⠕',
		'inpos1': [0,0,0,0,1,1,2],
	},
	{
		'text': 'X線',
		'input': 'Xセン',
		'output': '⠰⠠⠭⠤⠻⠴',
		'inpos1': [0,0,0,0,1,2],
	},
	{ 'note': '【備考】アルファベットや省略符としてのピリオドの後ろに助詞・助動詞が続く場合は、間を区切って書く。' },
	{
		'text': 'Aは',
		'input': 'A ワ',
	},
	{
		'text': 'NHKまで',
		'input': 'NHK マデ',
	},
	{
		'text': 'PRだった',
		'input': 'PR ダッタ',
	},
	{
		'text': 'No.を',
		'input': 'No. ヲ',
	},
	{ 'note': '３．一続きに書き表すべき1語中の仮名とアルファベットの間は続けて書く。' },
	{
		'text': '何㎝',
		'input': 'ナンcm',
	},
	{
		'text': '何cm',
		'input': 'ナンcm',
	},
	{
		'text': '数㎏',
		'input': 'スーkg',
	},
	{
		'text': '数kg',
		'input': 'スーkg',
	},
	{ 'note': '４．一続きに書き表すべき1語中のアルファベットと数字、数字とアルファベットの間は続けて書く。' },
	{
		'text': 'p105',
		'input': 'p105',
		'output': '⠰⠏⠼⠁⠚⠑',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,1,2,3],
		'inpos':  [0,0,1,1,2,3],
	},
	{
		'text': 'B5判',
		'input': 'B5ハン',
	},
	{ 'note': '＜処理＞アルファベットに省略符としてのピリオドが付いているときは、後ろに続く数字との間を区切って書く。' },
	{
		'text': 'No.1から',
		'input': 'No. 1カラ',
		'output': '⠰⠠⠝⠕⠲ ⠼⠁⠡⠑',
		'inpos1': [0,0,0,1,2,3,4,4,5,6],
	},
	{ 'note': '==== ２　語や文を書き表す場合 ====' },
	{ 'note': '１．アルファベットで書かれた語句や文は、その前後ろを外国語引用符(236～356)で囲んで書く。外国語引用符の内側は英文記号を用いて、英語点字の表記法に従って書く。' },
	{
		'text': 'チェコ製のglass',
		'input': 'チェコセイノ glass',
		'output': '⠈⠟⠪⠻⠃⠎ ⠰⠛⠇⠁⠎⠎',
		'inpos1': [0,1,2,3,4,5,6,7,7,8,9,10,11],
		'comment': '規則変更→ 空白をはさまない1単語は外国語引用符ではなく外字符でもよい',
	},
	{
		'text': '世界保健機関(World Health Organization)の活動',
		'input': 'セカイ ホケン キカン(⠦World Health Organization⠴)ノ カツドー',
		'output': '⠻⠡⠃ ⠮⠫⠴ ⠣⠡⠴⠶⠦⠠⠺⠕⠗⠇⠙ ⠠⠓⠑⠁⠇⠞⠓ ⠠⠕⠗⠛⠁⠝⠊⠵⠁⠞⠊⠕⠝⠴⠶⠎ ⠡⠝⠐⠞⠒',
	},
	{
		'text': 'MY FAIR LADY, a Broadway play, is a musical.',
		'input': '⠦MY FAIR LADY, a Broadway play, is a musical.⠴',
	},
	{ 'note': '２．一続きに書き表すべき１語中の外国語引用符と仮名の間は、第１つなぎ符をはさんで続けて書き、仮名と外国語引用符の間は続けて書く。' },
	{ 'note': '規則変更→ 外国語引用符の後は必ずマスアケする' },
	{
		'text': 'Gift券',
		'input': 'Giftケン',
		'output': '⠰⠠⠛⠊⠋⠞⠤⠫⠴',
		'inpos1': [0,0,0,1,2,3,3,4,5],
		'comment': '規則変更→1単語なので外字符を使う。マスアケではなくつなぎ符を使う。'
	},
	{
		'text': '赤wine',
		'input': 'アカwine',
		'output': '⠁⠡⠰⠺⠊⠝⠑',
		'inpos1': [0,1,2,2,3,4,5],
	},
	{
		'text': '名speech集',
		'input': 'メイspeechシュー',
		'output': '⠿⠃⠰⠎⠏⠑⠑⠉⠓⠤⠈⠹⠒',
		'inpos1': [0,1,2,2,3,4,5,6,7,7,8,9,10],
	},
	{ 'note': '【備考】外国語引用符の後ろに助詞・助動詞が続く場合は、間を区切って書く。' },
	{
		'text': 'I love you.に相当する日本語は何か',
		'input': '⠦I love you.⠴ ニ ソートー スル ニホンゴワ ナニカ',
		'output': '⠦⠠⠊ ⠇⠕⠧⠑ ⠽⠕⠥⠲⠴ ⠇ ⠺⠒⠞⠒ ⠹⠙ ⠇⠮⠴⠐⠪⠄ ⠅⠇⠡',
		'inpos1': [0,1,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,27,28,29,30,31,32],
	},
 	{
		'text': 'tea roomらしい',
		'input': '⠦tea room⠴ ラシイ',
		'output': '⠦⠞⠑⠁ ⠗⠕⠕⠍⠴ ⠑⠳⠃',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,12,13],
	},
 	{
		'text': 'Windows Live メール',
		'input': '⠦Windows Live⠴ メール',
	},
	{ 'note': '３．一般書に挿入されているドイツ語やフランス語など' },

	{ 'note': '４．ローマ字で書かれた日本語' },

	{ 'note': '５．人名は、原文の書き方によって外国語引用符と外字符を使い分ける。なお、外国語引用符の中ではピリオドは省略することができない。' },

	{ 'note': '== 第３章　語の書き表し方 ==' },
	{ 'note': '=== その１　分かち書き ===' },
	{ 'note': '１．自立語は前を区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 1',
		'text': '美しい山桜',
		'input': 'ウツクシイ ヤマザクラ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 1',
		'text': '早くする',
		'input': 'ハヤク スル',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 1',
		'text': 'ゆっくり歩く',
		'input': 'ユックリ アルク',
	},
	{
		'text': '運動をした。',
		'input': 'ウンドーヲ シタ。',
	},
	{
		'text': 'マッサージをしている。',
		'input': 'マッサージヲ シテ イル。',
	},
	{
		'text': '取り止めにした。',
		'input': 'トリヤメニ シタ。',
	},
	{ 'note': '２．付属語（助詞・助動詞は）は、自立語または他の付属語に続けて書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.14',
		'text': '山ん中',
		'input': 'ヤマン ナカ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.16',
		'text': '好かんやつ',
		'input': 'スカン ヤツ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.17',
		'text': '嘘みたいな話',
		'input': 'ウソミタイナ ハナシ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.18',
		'text': 'どうなのですか',
		'input': 'ドーナノデスカ',
	},
	{ 'note': '【備考１】ア行・ラ行で始まる助詞・助動詞が数字の後ろに続く場合は、第1つなぎ符をはさんで続けて書く。' },
	{ 'note': '【備考２】助詞・助動詞がアルファベットや省略符としてのピリオド、および外国語引用符の後ろに続く場合は、前を区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.31',
		'text': 'ＣＤを聞く',
		'input': 'CD ヲ キク',
		'output': '⠰⠠⠠⠉⠙ ⠔ ⠣⠩',
		'inpos1': [0,0,0,0,1,2,3,4,5,6],
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.32',
		'text': 'ＳＦみたいな話',
		'input': 'SF ミタイナ ハナシ',
		'output': '⠰⠠⠠⠎⠋ ⠷⠕⠃⠅ ⠥⠅⠳',
		'inpos1': [0,0,0,0,1,2,3,4,5,6,7,8,9,10],
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.33',
		'text': 'ｃｆ．は参照を示す',
		'input': 'cf. ワ サンショーヲ シメス',
		'output': '⠰⠉⠋⠲ ⠄ ⠱⠴⠈⠺⠒⠔ ⠳⠿⠹',
		'inpos1': [0,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15],
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.34',
		'text': 'あなたのモットーはchallengeですね',
		'input': 'アナタノ モットーワ challenge デスネ',
		'output': '⠁⠅⠕⠎ ⠾⠂⠞⠒⠄ ⠰⠉⠓⠁⠇⠇⠑⠝⠛⠑ ⠐⠟⠹⠏',
		'inpos1': [0,1,2,3,4,5,6,7,8,9,10,11,11,12,13,14,15,16,17,18,19,20,21,21,22,23],
		'comment': '規則変更→1単語は外国語引用符を使わない'
	},
	{ 'note': '＜処理＞アルファベットで書かれた外国語に「な」や「だ」などが続いて、形容動詞と考えられる場合でも、助詞・助動詞と同様に区切って書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 2 No.35',
		'text': 'Basicな装い',
		'input': 'Basic ナ ヨソオイ',
		'output': '⠰⠠⠃⠁⠎⠊⠉ ⠅ ⠜⠺⠊⠃',
		'inpos1': [0,0,0,1,2,3,4,5,6,7,8,9,10,11],
	},
	{ 'note': '３．実質的な意味が薄れた名詞（形容詞なども）、自立語であるから前を区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.36',	
		'text': '東京辺り',	
		'input': 'トーキョー アタリ',	
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.37',	
		'text': '力の限り',	
		'input': 'チカラノ カギリ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.38',	
		'text': '悲しみのあまり',
		'input': 'カナシミノ アマリ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.39',	
		'text': '忘れずに読むこと',
		'input': 'ワスレズニ ヨム コト',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.40',	
		'text': 'わたしの生まれた頃',
		'input': 'ワタシノ ウマレタ コロ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.41',	
		'text': '遅れたため',
		'input': 'オクレタ タメ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.42',	
		#'text': '１時間近く',
		'input': '1ジカン チカク',
		'output': '⠼⠁⠐⠳⠡⠴ ⠗⠡⠩',
		'inpos1': [0,0,1,1,2,3,4,5,6,7],
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.43',	
		'text': 'その点について',
		'input': 'ソノ テンニ ツイテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.51',
		'text': 'そういうわけ',
		'input': 'ソー イウ ワケ',
	},

	{ 'note': '【備考１】次のような言葉は、接尾語や助詞なので前に続けて書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.52',
		'text': '一人当り10個ずつ',
		'input': 'ヒトリアタリ 10コズツ',
		'output': '⠧⠞⠓⠁⠕⠓ ⠼⠁⠚⠪⠐⠹⠝',
		'inpos1': [0,1,2,3,4,5,6,7,7,8,9,10,10,11],
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.54',
		'text': 'いく度か',
		'input': 'イクドカ', # 'イクタビカ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.55',
		'text': 'まあ楽しみですこと',
		'input': 'マア タノシミデスコト',
	},
	{ 'note': '＜処理＞同じ漢字でも読み方の違いによって書き分ける。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.59',
		'text': 'ここら辺り',
		'input': 'ココラ アタリ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.63',
		'text': 'その後',
		'input': 'ソノゴ',# ソノ アト, ソノ ノチ, ソノ ウシロ
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 3 No.64',
		'text': '今月末',
		'input': 'コンゲツマツ', # コンゲツ スエ
	},
	{ 'note': '【備考２】会話などで名詞の語形や音韻が変化した場合でも、前を区切って書いてよい。' },
	{ 'note': '４．本来の意味が薄れて付属語的に用いられている動詞（補助動詞など）も、自立語なので前を区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.75',
		'text': 'お読み頂く',
		'input': 'オヨミ イタダク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.76',
		'text': '発車いたします',
		'input': 'ハッシャ イタシマス',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.77',
		'text': '行っていらっしゃい',
		'input': 'イッテ イラッシャイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.78',
		'text': '見ている',
		'input': 'ミテ イル',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.79',
		'text': '現代における',
		'input': 'ゲンダイニ オケル',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.80',
		'text': 'しまっておく',
		'input': 'シマッテ オク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.81',
		'text': '日本に於て',
		'input': 'ニッポンニ オイテ', # 'ニホンニ オイテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.82',
		'text': '出てきた',
		'input': 'デテ キタ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.83',
		'text': '行ってくる',
		'input': 'イッテ クル',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.84',
		'text': 'ごめんください',
		'input': 'ゴメン クダサイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.85',
		'text': 'お遊びください',
		'input': 'オアソビ クダサイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.86',
		'text': 'おはようございます',
		'input': 'オハヨー ゴザイマス',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.87',
		'text': '休館日につき',
		'input': 'キューカンビニ ツキ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.88',
		'text': 'これについて',
		'input': 'コレニ ツイテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.89',
		'text': '彼にとって',
		'input': 'カレニ トッテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 4 No.96',
		'text': '計画により',
		'input': 'ケイカクニ ヨリ',
	},
	{ 'note': '５．副詞「こう・そう・ああ・どう」、連体詞「この・その・あの・どの・ある・わが」なども自立語なので、いずれも後ろの自立語との間を区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.2',
		'text': 'そういう',
		'input': 'ソー イウ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.5',
		'text': 'どうして帰りましょう',
		'input': 'ドー シテ カエリマショー',
	},
	{ 'note': '【備考】後ろの語と結びついて1語になっている場合は続ける。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.20',
		'text': 'この間',
		'input': 'コノアイダ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.21',
		'text': 'この期',
		'input': 'コノゴ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.22',
		'text': 'この程',
		'input': 'コノホド',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.23',
		'text': 'この度',
		'input': 'コノタビ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.26',
		'text': 'そのくせ',
		'input': 'ソノクセ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.30',
		'text': 'そのうちに',
		'input': 'ソノウチニ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.31',
		'text': 'そのまま',
		'input': 'ソノママ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.32',
		'text': 'その節',
		'input': 'ソノセツ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.36',
		'text': 'わがまま',
		'input': 'ワガママ',
	},
	{ 'note': '６．「にして」「をして」「ずして」などの、「して」が文語的表現の助詞である場合は前に続けて書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.37',
		'text': '不幸にして',
		'input': 'フコーニシテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.38',
		'text': '今にして',
		'input': 'イマニシテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.39',
		'text': '居ながらにして',
		'input': 'イナガラニシテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.40',
		'text': '労せずして',
		'input': 'ローセズシテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.41',
		'text': '私をして言わしむれば',
		'input': 'ワタシヲシテ イワシムレバ', # 'ワタクシヲシテ イワシムレバ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.42',
		'text': '年わかくして',
		'input': 'トシ ワカクシテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.43',
		'text': '水清くして',
		'input': 'ミズ キヨクシテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.45',
		#'text': '二人して',
		'input': 'フタリシテ', # 2ニンシテ
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.46',
		'text': 'それだからして',
		'input': 'ソレダカラシテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.48',
		'text': '見るべくして',
		'input': 'ミルベクシテ',
	},
	{ 'note': '【備考１】' },
	{ 'note': '【備考２】' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.50',
		'text': '人として',
		'input': 'ヒトト シテ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.51',
		'text': '私としては',
		'input': 'ワタシト シテワ', # 'ワタクシト シテワ',
	},
	{
		'comment': '前の語と結びついている場合は続ける',
		'text':'主として', 'input':'シュトシテ'
	},
	{ 'note': '７．「なさい・なさる」などが、独立の動詞であれば前を区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.65',
		'text': '立ってなさい',
		'input': 'タッテ ナサイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.66',
		'text': '見てなさい',
		'input': 'ミテ ナサイ',
	},
	{ 'note': '【備考】動詞の連用形に続く場合は続けて書く。「い」が省略されているときは続けて書く。「お」がついて名詞化した語に「なさい・なさる」が続く場合は区切って書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.67',
		'text': 'お黙りなさい',
		'input': 'オダマリ ナサイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.70',
		'text': 'お休みなされる',
		'input': 'オヤスミ ナサレル',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.71',
		'text': 'お読みなされ',
		'input': 'オヨミ ナサレ',
	},
	{ 'note': '８．形容詞の「ない」は前を区切って書き、助動詞の「ない」は前に続けて書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '関係ない',
		'input': 'カンケイ ナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '異存なければ',
		'input': 'イゾン ナケレバ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.74',
		'text': '問題なかった',
		'input': 'モンダイ ナカッタ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.75',
		'text': 'わかりっこない',
		'input': 'ワカリッコ ナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.76',
		'text': '間違いなし',
		'input': 'マチガイ ナシ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '声なき声',
		'input': 'コエ ナキ コエ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': 'こだわりなく',
		'input': 'コダワリ ナク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.79',
		'text': 'そうじゃない',
		'input': 'ソージャ ナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.80',
		'text': '目じゃない',
		'input': 'メジャ ナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '意気地ない',
		'input': 'イクジ ナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '自信なげ',
		'input': 'ジシン ナゲ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '起きない',
		'input': 'オキナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '歩かない',
		'input': 'アルカナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '捨てない',
		'input': 'ステナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8',
		'text': '間違えない',
		'input': 'マチガエナイ',
	},
	{ 'note': '【備考】「ない」などが前の語と複合して、1語になっている場合は続けて書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.87',
		'text': '味気ない',
		'input': 'アジケナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '呆気ない',
		'input': 'アッケナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.95',
		'text': '隈なく',
		'input': 'クマナク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '難なく',
		'input': 'ナンナク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '何となく',
		'input': 'ナントナク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.98',
		'text': '心置きなく',
		'input': 'ココロオキナク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.99',
		'text': '満遍無く',
		'input': 'マンベンナク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '程なく',
		'input': 'ホドナク',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '形なし',
		'input': 'カタナシ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '底無し',
		'input': 'ソコナシ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '意気地無し',
		'input': 'イクジナシ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': 'ろくでなし',
		'input': 'ロクデナシ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 8 備考',
		'text': '事も無げ',
		'input': 'コトモナゲ',
	},
	{ 'note': '＜処理＞「て」「で」に続く「ない」は、「ない」の前を区切って書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.106',
		'text': '彼にはまだ話してない',
		'input': 'カレニワ マダ ハナシテ ナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.107',
		'text': 'しばらく休んでない',
		'input': 'シバラク ヤスンデ ナイ',
	},
	{ 'note': '９．形容詞などの「～く」の後ろは区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.110',
		'text': '良くない',
		'input': 'ヨク ナイ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.112',
		'text': '食べたくなくなる',
		'input': 'タベタク ナク ナル',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その1 5 No.114',
		'text': 'それどころではなくなる',
		'input': 'ソレドコロデワ ナク ナル',
	},

	{ 'note': '=== その２　複合語 ===' },
	{ 'note': '１．区切ると意味の理解を損なう短い複合語や短い略語は、一続きに書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 1',
		'text': '朝夕', 
		'input': 'アサユー'
	},
	{ 'note': '【備考】内部に助詞などを含んでいても、１語として熟している短い複合語は一続きに書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 1 備考',
		'text': '息の根', 
		'input': 'イキノネ'
	},
	{ 'note': '２．接頭語・接尾語・造語要素と自立語の間は続けて書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 2',
		'text': '相たずさえる', 
		'input': 'アイタズサエル'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2',
		'text': '各種', 
		'input': 'カクシュ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2',
		'text': '各国', 
		'input': 'カッコク'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2',
		'text': '旧街道', 
		'input': 'キューカイドー'
	},
	{ 'note': '複合語（接頭語・接尾語・造語要素）【備考１】接頭語・接尾語・造語要素であっても、意味の理解を助ける場合には、発音上の切れ目を考慮して区切って書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '各党', 
		'input': 'カク トー'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '各方面', 
		'input': 'カク ホーメン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '貴商店',
		'input': 'キ ショーテン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '旧陸軍', 
		'input': 'キュー リクグン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '故池上先生',
		'input': 'コ イケガミ センセイ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '前校長',
		'input': 'ゼン コーチョー'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '全5巻',
		'input': 'ゼン 5カン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '超現実的',
		'input': 'チョー ゲンジツテキ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '当施設',
		'input': 'トー シセツ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '反社会的',
		'input': 'ハン シャカイテキ'
	},
	#{
	#	'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
	#	'text': '富栄養化',
	#	'input': 'フ エイヨーカ'
	#   'comment': 'not トミ エイヨーカ',
	#	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '非人道的',
		'input': 'ヒ ジンドーテキ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '満3年',
		'input': 'マン 3ネン'
	},
	#{
	#	'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
	#	'text': '丸1日',
	#	'input': 'マル 1ニチ'
	#   'comment': '「マルツイタチ」にしないために辞書登録が必要',
	#	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '元副総理',
		'input': 'モト フクソーリ',
		'comment': 'not モト フク ソーリ',
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': '新時刻表',
		'input': 'シン ジコクヒョー'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
		'text': 'ご挨拶かたがた',
		'input': 'ゴアイサツ カタガタ'
	},
	#{
	#	'comment': '点訳のてびき第3版 第3章 その2 2 備考1',
	#	'text': '1回こっきり',
	#	'input': '1カイ コッキリ'
	#   'comment': '「こっきり」が解析失敗',
	#	},
	{ 'note': '【備考２】語頭にある接頭語や造語要素が、マスあけを含む複合語全体にかかる場合には、その後ろを区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考2',
		'text': '新東京名所', 
		'input': 'シン トーキョー メイショ'
	},
	{ 'note': '【備考３】外来語の接辞も続けて書くが、意味の理解を助ける場合には、区切って書いてよい。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考3',
		'text': 'フレンドシップ', 
		'input': 'フレンドシップ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 備考3',
		'text': 'インターネット', 
		'input': 'インター ネット'
	},
	{ 'note': '＜処理＞「等」は「など」「ら」と読む場合には前に続けて書くが、「とう」と読む場合は前を区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 処理',
		'text': '山や川等', 
		'input': 'ヤマヤ カワナド'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 2 処理',
		'text': '我等は', 
		'input': 'ワレラワ'
	},
	{ 'note': '３．複合名詞の内部に、３拍以上の意味のまとまりが二つ以上あれば、その境目で区切って書くことを原則とする。' },
	{   'comment': '点訳のてびき第3版 第3章 その2 3.',
		'text': '桜並木', 'input': 'サクラ ナミキ' },
	{   'comment': '点訳のてびき第3版 第3章 その2 3.',
		'text': '花嫁姿', 'input': 'ハナヨメ スガタ' },
	#{   'comment': '点訳のてびき第3版 第3章 その2 3.',
	#	'text': '一人娘', 'input': 'ヒトリ ムスメ' },
	{   'comment': '点訳のてびき第3版 第3章 その2 3.',
		'text': '左半身', 'input': 'ヒダリ ハンシン' },

	{ 'note': '【備考１】接尾語的に用いられている語であっても、３拍以上の漢語は区切って書いてよい。' },
	#{  'comment': '点訳のてびき第3版 第3章 その2 3. 備考1',
	#   'text': '手当たり次第', 'input': 'テアタリ シダイ' },
	{   'comment': '点訳のてびき第3版 第3章 その2 3. 備考1',
		'text': '若者同士', 'input': 'ワカモノ ドーシ' },

	{ 'note': '【備考２】複合動詞や複合形容詞から転成したものや、形容詞の語幹を含んでいる場合など、区切ると意味の理解を損なうおそれのある語は続けて書く。' },
	#{   'comment': '点訳のてびき第3版 第3章 その2 3. 備考2',
	#	'text': '扱い始め', 'input': 'アツカイハジメ' },

	{ 'note': '【備考３】動詞から転成した３拍以上の語も区切って書くが、自立性が弱いと思われるものは続けて書いてよい。' },
	{ 'note': '【備考４】外来語の複合名詞で、区切ると意味の理解を損なうおそれのある語は続けて書く。' },
	{   'comment': '点訳のてびき第3版 第3章 その2 3. 備考4', 'text': 'プラットホーム', 'input': 'プラットホーム' },

	{ 'note': '４．複合名詞内部の２拍以下の意味のまとまりは、続けて書くことを原則とする。' },
	#{   'comment': '点訳のてびき第3版 第3章 その2 4.',
	#	'text': '松並木', 'input': 'マツナミキ' },
	{   'comment': '点訳のてびき第3版 第3章 その2 4.',
		'text': '蝉時雨', 'input': 'セミシグレ' },
	{   'comment': '点訳のてびき第3版 第3章 その2 4.',
		'text': '右半身', 'input': 'ミギハンシン' },
	{   'comment': '点訳のてびき第3版 第3章 その2 4.',
		'text': '腕次第', 'input': 'ウデシダイ' },

	{ 'note': '【備考１】２字漢語は、２拍であっても３拍以上の意味のまとまりと同様に扱う。' },
	#{   'comment': '点訳のてびき第3版 第3章 その2 4. 備考1',
	#	'text': '母子年金', 'input': 'ボシ ネンキン' },

	{ 'note': '【備考２】複合名詞などの意味のまとまりが２拍以下であっても、自立性が強く、意味の理解を助ける場合には、区切って書いてよい。' },
	#{   'comment': '点訳のてびき第3版 第3章 その2 4. 備考2',
	#	'text': '要問い合わせ', 'input': 'ヨー トイアワセ' },

	{ 'note': '＜処理＞２拍の外来語を含む混種語は、意味の理解を助ける場合には区切って書いてよい。' },
	#{   'comment': '点訳のてびき第3版 第3章 その2 4. 備考2',
	#	'text': 'バス停留所', 'input': 'バス テイリュージョ' },

	{ 'note': '５．内部に二つ以上の意味のまとまりがない場合や、区切ると意味の理解を損なうおそれのある複合名詞は続けて書く。' },
	{   'comment': '点訳のてびき第3版 第3章 その2 5.',
		'text': '小中学校', 'input': 'ショーチューガッコー' },

	{ 'note': '６．アルファベットを含む複合名詞は、語句のまとまりを考慮して続けて書くか、適宜区切って書く。' },
	{ 'note': '７．漢字１字ずつが対等な関係で並んでいる複合名詞などは、意味の理解を容易にするために、語句のまとまりや発音などを考慮して適宜区切るか、すべてを続けて書く。' },
	{ 'note': '８．複合名詞は続けて書くことを原則とする。' },
	{   'comment': '点訳のてびき第3版 第3章 その2 8.',
		'text': '飛び込む', 'input': 'トビコム' },

	{ 'note': '【備考】内部に助詞「て」を含む場合は「て」の後ろで区切って書いてよい。' },
	{   'comment': '点訳のてびき第3版 第3章 その2 8. 備考',
		'text': 'やってくる', 'input': 'ヤッテ クル' },

	{ 'note': '９．名詞や副詞に動詞の「する」が続く場合は、「する」の前を区切って書くことを原則とする。' },
	{ 'note': '【備考１】１字漢語などに「する」が続いて音韻が変化したり、連濁する語は続けて書く。' },
	{ 'note': '【備考２】１字漢語に「する」が続く場合は一続きに書くが、「する」の前に助詞「を」をはさむことができる場合は区切って書いてよい。' },
	{ 'note': '【備考３】和語に「する」が続く語は、区切って書くが、和語の自立性が弱いと思われる場合は続けて書いてよい。' },
	{ 'note': '１０．複合形容詞は続けて書く。' },
	#{   'comment': '点訳のてびき第3版 第3章 その2 10.',
	#	'text': '面白おかしい', 'input': 'オモシロオカシイ' },

	{ 'note': '１１．接続詞句や副詞句は、意味の理解を容易にするために、発音上の切れ目などを考慮して適宜区切って書く。' },
	{ 'note': '１２．連濁を生じた複合語は続けて書く。' },
	{   'comment': '点訳のてびき第3版 第3章 その2 12.',
		'text': '柱時計', 'input': 'ハシラドケイ' },

	{ 'note': '１３．短い語からなる慣用句などは、すべて続けて書くか、意味のまとまりを目安に適宜区切って書く。' },
	{ 'note': '１４．年月日や名数などは、その段階ごとに区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その2 14',
		'text': '４月１５日', 
		'input': '4ガツ 15ニチ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その2 14',
		'text': '０４月', 
		'input': '04ガツ'
	},
	{ 'note': '【備考】年月日や名数などの後ろに続く２拍以下の語は、意味の理解を助ける場合には発音上の切れ目を考慮して区切って書いてよい。' },
	{ 'note': '１５．二つ以上の意味のまとまりからなる繰り返し言葉は、区切って書くことを原則とする。' },
	{ 'note': '【備考１】連濁する語は続けて書く。' },
	{ 'note': '【備考２】区切ると意味の理解を損なう場合は続けて書く。' },
	{ 'note': '【備考３】擬声語・擬態語は、３拍以上の繰り返しは区切って書き、２拍以下の繰り返しは続けて書く。' },
	{ 'note': '１６．動植物名や理化学用語なども、複合動詞の切れ続きに準じて書くことを原則とする。' },
	{ 'note': '【備考】区切ると意味の理解を損なうと思われる場合は、第１つなぎ符をはさんで続けて書くか、一続きに書くことができる。' },
	{ 'note': '１７．医学用語などは、発音や意味のまとまりに留意して、複合動詞の切れ続きに準じて書く。' },

	{ 'note': '=== その３　固有名詞 ===' },
	{ 'note': '１．人名の名字と名前の間は、区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その3 1.',
		'text': '湯川秀樹', 
		'input': 'ユカワ ヒデキ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 1.',
		'text': '石川倉次', 
		'input': 'イシカワ クラジ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 1.',
		'text': '武者小路実篤', 
		'input': 'ムシャノコージ サネアツ'
	},
	{ 'note': '３．人名に続く「さん」「様」「君」「殿」「氏（し）」「氏（うじ）」は、区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '鈴木さん', 'input': 'スズキ サン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '鈴木様', 'input': 'スズキ サマ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '鈴木さま', 'input': 'スズキ サマ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '鈴木君', 'input': 'スズキ クン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '鈴木くん', 'input': 'スズキ クン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '鈴木殿', 'input': 'スズキ ドノ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '鈴木氏', 'input': 'スズキ シ'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '一郎君', 'input': 'イチロー クン'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3.',
		'text': '一郎くん', 'input': 'イチロー クン'
	},
	{   'text':'久代さん', 'input':'ヒサヨ サン'},
	{   'text':'新家さん', 'input':'シンケ サン'},
	{
		'comment': '点訳のてびき第3版 第3章 その3 3. 備考1',
		'text': '魚屋さん', 'input': 'サカナヤサン'
	},

	{ 'note': '４．地名（国名を含む）および地名と接尾語や造語要素または普通名詞などとの複合語は、段階（都道府県・市区町村）ごとに区切って書く。' },
	{
		'comment': '点訳のてびき第3版 第3章 その3 4.',
		'text': '東京都千代田区永田町１の７',
		'input': 'トーキョート チヨダク ナガタチョー 1ノ 7'
	},
	{
		'text': '東京都千代田区永田町1-7',
		'input': 'トーキョート チヨダク ナガタチョー 1-7'
	},
	{
		'text': '東京都千代田区永田町一の七',
		'input': 'トーキョート チヨダク ナガタチョー 1ノ 7'
	},
	{
		'comment': '点訳のてびき第3版 第3章 その3 4.',
		'text': '福島県伊達市月舘町月舘',
		'input': 'フクシマケン ダテシ ツキダテマチ ツキダテ'
	},
	{'text':'綾部市保健福祉センター', 'input':'アヤベシ ホケン フクシ センター'},
	{'text':'東京都交通局', 'input':'トーキョート コーツーキョク'},
	{'text':'府中町役場', 'input':'フチューマチ ヤクバ'},
	{'text':'町役場', 'input':'マチヤクバ'},

	{ 'note': '=== その４　方言・古文など ===' },

	{ 'note': '== 第４章　記号類の使い方 ==' },
	{ 'note': '=== その１　句読符 ===' },
	{ 'note': '==== １．句点 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その1 1.',
		'text': '雨が降っていた。けれどぼくは出かけた。',
		'input': 'アメガ フッテ イタ。ケレド ボクワ デカケタ。',
		'output': '⠁⠿⠐⠡ ⠭⠂⠟ ⠃⠕⠲  ⠫⠛⠐⠞ ⠐⠮⠩⠄ ⠐⠟⠡⠫⠕⠲',
		'inpos1': [0,1,2,2,3,4,5,6,7,8,9,10,10,10,11,12,13,13,14,15,15,16,17,18,19,19,20,21,22,23],
	},
	{ 'note': '==== ２．疑問符、感嘆符 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その1 2.',
		'text': 'お元気？ 久しぶりですね。',
		'input': 'オゲンキ? ヒサシブリデスネ。',
		'output': '⠊⠐⠫⠴⠣⠢  ⠧⠱⠳⠐⠭⠓⠐⠟⠹⠏⠲',
		'inpos1': [0,1,1,2,3,4,4,5,6,7,8,9,9,10,11,11,12,13,14],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その1 2.',
		'text': 'えっ！ 彼が哲学者!?',
		'input': 'エッ! カレガ テツガクシャ!?',
		#'output': '⠋⠂⠖  ⠡⠛⠐⠡ ⠟⠝⠐⠡⠩⠈⠱⠖⠢',
	},
	{ 'note': '==== ３．読点 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その1 3.',
		'text': '森には、いろいろな動物が住んでいる。',
		'input': 'モリニワ、イロイロナ ドーブツガ スンデ イル。',
		'output': '⠾⠓⠇⠄⠰ ⠃⠚⠃⠚⠅ ⠐⠞⠒⠐⠭⠝⠐⠡ ⠹⠴⠐⠟ ⠃⠙⠲',
		'inpos1': [0,1,2,3,4,4,5,6,7,8,9,10,11,11,12,13,13,14,15,15,16,17,18,19,19,20,21,22,23],
	},
	{ 'note': '==== ４．中点 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その1 4.',
		'text': '松・杉・檜は、針葉樹です。',
		'input': 'マツ・スギ・ヒノキワ、シンヨージュデス。',
		'output': '⠵⠝⠐ ⠹⠐⠣⠐ ⠧⠎⠣⠄⠰ ⠳⠴⠜⠒⠘⠹⠐⠟⠹⠲',
		'inpos1': [0,1,2,2,3,4,4,5,5,6,7,8,9,10,10,11,12,13,14,15,16,17,17,18,19],
	},
	{ 'note': '=== その２　囲みの記号 ===' },
	{ 'note': '==== １．カギ類 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その2 1.',
		'text': '「住所」「氏名」「電話番号」を書いてください。',
		'input': '「ジューショ」 「シメイ」 「デンワ バンゴー」ヲ カイテ クダサイ。',
		#'output': '⠤⠘⠹⠒⠈⠺⠤ ⠤⠳⠿⠃⠤ ⠤⠐⠟⠴⠄ ⠐⠥⠴⠐⠪⠒⠤⠔ ⠡⠃⠟ ⠩⠐⠕⠱⠃⠲',
	},
	{ 'note': '==== ２．カッコ類 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その2 2.',
		'text': 'Ａｕ（金）の比重',
		'input': 'Au(キン)ノ ヒジュー',
		'output': '⠰⠠⠁⠥⠶⠣⠴⠶⠎ ⠧⠘⠹⠒',
		'inpos1': [0,0,0,1,2,3,4,5,6,7,8,9,10,11],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その2 2.',
		'text': 'すぐ（に）（副詞）陽が暮れる。', # 日が暮れる→読み付与誤り回避の変更
		'input': 'スグ(ニ) (フクシ) ヒガ クレル。',
		'output': '⠹⠐⠩⠶⠇⠶ ⠶⠭⠩⠳⠶ ⠧⠐⠡ ⠩⠛⠙⠲',
		'inpos1': [0,1,1,2,3,4,5,6,7,8,9,10,11,12,13,13,14,15,16,17,18],
	},
	{ 'note': '==== ３．指示符類 ====' },
	{ 'note': '==== ４．外国語引用符 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その2 4.',
		#'text': '“MODERN TIMES”',
		'input': '⠦MODERN TIMES⠴',
		'output': '⠦⠠⠠⠍⠕⠙⠑⠗⠝ ⠠⠠⠞⠊⠍⠑⠎⠴',
		'inpos1': [0,1,1,1,2,3,4,5,6,7,8,8,8,9,10,11,12,13],
	},
	{ 'note': '==== ５．発音記号符 ====' },
	{ 'note': '==== ６．点訳者挿入符 ====' },
	{ 'note': '==== ７．段落挿入符類 ====' },

	{ 'note': '=== その３　線類 ===' },
	{ 'note': '==== １．棒線・点線 ====' },
	{ 'note': '==== ２．矢印類 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その3 2.',
		'text': '札幌→東京',
		'input': 'サッポロ→トーキョー',
		'output': '⠱⠂⠠⠮⠚ ⠒⠒⠕ ⠞⠒⠈⠪⠒',
	},
	{
		'comment': '点訳のてびき第3版 第4章 その3 2.',
		'text': '札幌←東京',
		'input': 'サッポロ←トーキョー',
		'output': '⠱⠂⠠⠮⠚ ⠪⠒⠒ ⠞⠒⠈⠪⠒',
	},
	{ 'note': '==== ３．波線 ====' },

	{ 'note': '=== その４　その他の記号類 ===' },
	{ 'note': '==== １．つなぎ符類 ====' },
	{ 'note': '【備考1】数字に続く「ワ」の後ろにア行・ラ行のカナが3つ続くときは、位取り点と誤読されるので、その場合に限り第1つなぎ符を用いる。' },
	{
		'comment': '点訳のてびき第3版 第4章 その4 1. 備考1',
		'text': '4割入り',
		'input': '4ワリイリ',
		'output': '⠼⠙⠤⠄⠓⠃⠓',
		'inpos1': [0,0,0,1,2,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その4 1. 備考1に該当しない場合',
		'text': '4割抜き',
		'input': '4ワリ ヌキ',
		'output': '⠼⠙⠄⠓ ⠍⠣',
		'inpos1': [0,0,1,2,3,4,5],
	},
	{ 'note': '==== ２．小見出し符類 ====' },
	{ 'note': '==== ３．文中注記符 ====' },
	{ 'note': '==== ４．星印類 ====' },
	{
		'comment': '白星印を 56-2346-2 と定義',
		'text': '☆',
		'input': '☆',
		'output': '⠰⠮⠂',
	},
	{
		'comment': '黒星印を 56-2346-23 と定義',
		'text': '★',
		'input': '★',
		'output': '⠰⠮⠆',
	},
	{
		'comment': '第1星印 35-35 (後ろを1マスあける)',
		'text': '※あ',
		'input': '※ア',
		'output': '⠔⠔ ⠁',
	},
	{ 'note': '==== ５．詩行符類 ====' },
	{ 'note': '==== ６．空欄符号 ====' },
	{ 'note': '==== ７．伏せ字 ====' },
	{
		'comment': '白丸を 6-1356-2 と定義',
		'text': '○',
		'input': '○',
		'output': '⠠⠵⠂',
	},
	{
		'comment': '黒丸を 6-1356-23 と定義',
		'text': '●',
		'input': '●',
		'output': '⠠⠵⠆',
	},
	{
		'comment': '二重丸を 6-1356-256 と定義',
		'text': '◎',
		'input': '◎',
		'output': '⠠⠵⠲',
	},
	{
		'comment': '白四角を 6-1256-2 と定義',
		'text': '□',
		'input': '□',
		'output': '⠠⠳⠂',
	},
	{
		'comment': '黒四角を 6-1256-23 と定義',
		'text': '■', 
		'input': '■',
		'output': '⠠⠳⠆',
	},
	{
		'comment': '白上向き三角を 6-156-2 と定義', 
		'text': '△', 
		'input': '△',
		'output': '⠠⠱⠂',
	},
	{
		'comment': '黒上向き三角を 6-156-23 と定義', 
		'text': '▲', 
		'input': '▲',
		'output': '⠠⠱⠆',
	},
	{
		'comment': '白下向き三角を 56-156-2 と定義', 
		'text': '▽', 
		'input': '▽',
		'output': '⠰⠱⠂',
	},
	{
		'comment': '黒下向き三角を 56-156-23 と定義', 
		'text': '▼', 
		'input': '▼',
		'output': '⠰⠱⠆',
	},
	{
		'comment': '白菱形を 46-1236-2 と定義(1236=ひ)',
		'text': '◇',
		'input': '◇',
		'output': '⠨⠧⠂',
	},
	{
		'comment': '黒菱形を 46-1256-23 と定義(1236=ひ)',
		'text': '◆', 
		'input': '◆',
		'output': '⠨⠧⠆',
	},
	{
		'comment': 'バツ印を 56-16-2 と定義', 
		'text': '×', 
		'input': '×',
		'output': '⠰⠡⠂',
	},
	{
		'comment': '郵便番号記号 56 2356 ユービン 0 バンゴー 2356 23',
		'text': '〒',
		'input': '〒',
		'output': '⠰⠶⠬⠒⠐⠧⠴ ⠐⠥⠴⠐⠪⠒⠶⠆',
	},
	{ 'note': '==== ８．パーセント ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その4 8.',
		'text': '100%',
		'input': '100%',
		'output': '⠼⠁⠚⠚⠰⠏',
		'inpos2': [0,1,2,3],
		'inpos1': [0,0,1,2,3,3],
		'inpos':  [0,0,1,2,3,3],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その4 8.',
		'text': '数%',
		'input': 'スー%',
		'output': '⠹⠒⠰⠏',
		'inpos1': [0,1,2,2],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その4 8.',
		'text': 'A%',
		'input': 'A%',
		'output': '⠰⠠⠁⠰⠏',
		'inpos2': [0,1],
		'inpos1': [0,0,0,1,1],
		'inpos':  [0,0,0,1,1],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その4 8.',
		'text': '25%引き',
		'input': '25%ビキ',
		'output': '⠼⠃⠑⠰⠏⠐⠧⠣',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,0,1,2,2,3,3,4],
		'inpos':  [0,0,1,2,2,3,3,4],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その4 8.',
		'text': '××％',
		'input': '××%',
		#'output': '⠐⠿⠐⠿⠤⠰⠏',
		'inpos2': [0,1,2],
	},
	{ 'note': '==== ９．アンドマーク ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その4 9.',
		'text': 'Ｑ＆Ａ',
		'input': 'Q & A',
		'output': '⠰⠠⠟ ⠰⠯ ⠰⠠⠁',
		'inpos1': [0,0,0,1,2,2,3,4,4,4],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その4 9.',
		'text': 'シャープス＆フラッツ',
		'input': 'シャープス & フラッツ',
		'output': '⠈⠱⠒⠠⠭⠹ ⠰⠯ ⠭⠑⠂⠝',
		'inpos1': [0,1,2,3,3,4,5,6,6,7,8,9,10,11],
	},
	{ 'note': '==== １０．ナンバーマーク ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その4 10.',
		'text': '４＃で応答メッセージが録音できます。',
		'input': '4# デ オートー メッセージガ ロクオン デキマス。',
		'output': '⠼⠙⠰⠩ ⠐⠟ ⠊⠒⠞⠒ ⠿⠂⠻⠒⠐⠳⠐⠡ ⠚⠩⠊⠴ ⠐⠟⠣⠵⠹⠲',
		'inpos1': [0,0,1,1,2,3,3,4,5,6,7,8,9,10,11,12,13,14,14,15,15,16,17,18,19,20,21,22,22,23,24,25,26],
	},
	{
		'comment': '点訳のてびき第3版 第4章 その4 10.',
		'text': '#50のミシン糸',
		'input': '#50ノ ミシンイト',
		'output': '⠰⠩⠼⠑⠚⠎ ⠷⠳⠴⠃⠞',
		'inpos1': [0,0,1,1,2,3,4,5,6,7,8,9],
	},
	{ 'note': '==== １１．アステリスク ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その4 11.',
		'text': '＊または＃を押すたびに',
		'input': '* マタワ # ヲ オス タビニ',
		'output': '⠰⠡ ⠵⠕⠄ ⠰⠩ ⠔ ⠊⠹ ⠕⠐⠧⠇',
		'inpos1': [0,0,1,2,3,4,5,6,6,7,8,9,10,11,12,13,14,14,15],
	},
	{ 'note': '==== １２．小文字符 ====' },
	{
		'comment': '点訳のてびき第3版 第4章 その4 12.',
		#'text': '「ァ」は、',
		'input': '「ァ」ワ、',
		#'output': '⠤⠘⠁⠤⠄⠰',
	},
	{ 'note': '+ ポジションマッピング +' },
	{
		'text': '12345 ',
		'input': '12345 ',
		'output': '⠼⠁⠃⠉⠙⠑ ',
		'inpos2': [0,1,2,3,4,5],
		'inpos1': [0,0,1,2,3,4,5],
		'inpos':  [0,0,1,2,3,4,5],
		'outpos': [0,2,3,4,5,6],
		'comment': '行末の空白を保存する',
	},
	{
		'text': 'やっぱり ',
		'input': 'ヤッパリ ',
		'output': '⠌⠂⠠⠥⠓ ',
		'inpos2': [0,1,2,3,4],
		'inpos1': [0,1,2,2,3,4],
		'inpos':  [0,1,2,2,3,4],
		'outpos': [0,1,2,4,5],
	},
	{
		'text': '機能 ',
		'input': 'キノー ',
		'output': '⠣⠎⠒ ',
		'inpos2': [0,0,1,2],
		'inpos1': [0,1,2,3],
		'inpos':  [0,0,1,2],
		'outpos': [0,2,3],
	},
	{
		'text': '実技 ',
		'input': 'ジツギ ',
		'output': '⠐⠳⠝⠐⠣ ',
		'inpos2': [0,0,1,2],
		'inpos1': [0,0,1,2,2,3],
		'inpos':  [0,0,0,1,1,2],
		'outpos': [0,3,5],
	},
	{
		'text': '東京 ',
		'input': 'トーキョー ',
		'output': '⠞⠒⠈⠪⠒ ',
		'inpos2': [0,0,0,1,1,2],
		'inpos1': [0,1,2,3,4,5],
		'inpos':  [0,0,0,1,1,2],
		'outpos': [0,3,5],
	},

	{ 'note': '+ 点字表記辞典 +' },
	{	'text': '相対する', 'input': 'アイタイスル'},
	{	'text': '相たずさえて', 'input': 'アイタズサエテ'},
	{	'text': '相整う', 'input': 'アイトトノウ'},
	{	'text': '相憐れむ', 'input': 'アイアワレム'},
	{	'text': '木立の間', 'input': 'コダチノ アイダ'},
	{	'text': '開いた口', 'input': 'アイタ クチ'},
	{	'text': '相無く', 'input': 'アイナク'},
	{	'text': 'お生憎様', 'input': 'オアイニクサマ'},
	{	'text': '開かずの間', 'input': 'アカズノ マ'},
	{	'text': '不開の間', 'input': 'アカズノ マ'},
	{	'text': '山田県主', 'input': 'ヤマダノ アガタヌシ'},
	{	'text': '暁闇', 'input': 'アカツキヤミ'},
	{	'text': '上がり降り', 'input': 'アガリオリ'},
	{	'text': '四条上ル', 'input': '4ジョー アガル'},
	{	'text': '秋津国', 'input': 'アキツクニ'},
	{	'text': '現つ神', 'input': 'アキツカミ'},
	{	'text': '阿Ｑ正伝', 'input': 'アQ セイデン'},
	{	'text': '悪源太', 'input': 'アクゲンタ'},
	{	'text': '明くる朝', 'input': 'アクル アサ'},
	{	'text': '明くる年', 'input': 'アクル トシ'},
	{	'text': '明の星', 'input': 'アケノ ホシ'},
	{	'text': '麻布十番', 'input': 'アザブ ジューバン'},
	{	'text': '男漁り', 'input': 'オトコ アサリ'},
	{	'text': '古本漁り', 'input': 'フルホン アサリ'},
	{	'text': '足の甲', 'input': 'アシノ コー'},
	{	'text': '日の足', 'input': 'ヒノアシ'},
	{	'text': '醤油味', 'input': 'ショーユ アジ'},
	{	'text': '源朝臣頼政', 'input': 'ミナモトノ アソン ヨリマサ'},
	{	'text': '兄貴風を吹かす', 'input': 'アニキカゼヲ フカス'},
	{	'text': '東漢直駒', 'input': 'ヤマトノ アヤノ アタイノ コマ'},
	{   'text': '天照大神', 'input': 'アマテラス オオミカミ'},
	{   'text': '天の岩戸', 'input': 'アマノ イワト'},
	{   'text': '天香具山', 'input': 'アマノ カグヤマ'},
	{   'text': '天の羽衣', 'input': 'アマノ ハゴロモ'},
	{   'text': '天の原', 'input': 'アマノハラ'},
	{   'text': '天鈿女命', 'input': 'アメノ ウズメノ ミコト'},
	{   'text': '水争', 'input': 'ミズアラソイ'},
	#{   'text': 'あらぬ方を向く', 'input': 'アラヌ カタヲ ムク'},
	{   'text': '蟻の塔草', 'input': 'アリノトーグサ'},
	{   'text': '有りの儘に打ち明ける', 'input': 'アリノママニ ウチアケル'},
	{   'text': '有りの実', 'input': 'アリノミ'},
	{   'text': '栄えある', 'input': 'ハエ アル'},
	{   'text': '或る程度', 'input': 'アル テイド'},
	{   'text': '有るが儘', 'input': 'アルガ ママ'},
	{   'text': 'アルカリ泉', 'input': 'アルカリセン'},
	{   'text': '合わせ鏡', 'input': 'アワセ カガミ'},
	{   'text': '鬘合わせ', 'input': 'カツラ アワセ'},
	{   'text': '暗順応', 'input': 'アンジュンノー'},

	{   'text': '大慌て', 'input': 'オオアワテ'},
	{   'text': '大いなる', 'input': 'オオイナル'},
	{   'text': '大男', 'input': 'オオオトコ'},
	{   'text': '大川', 'input': 'オオカワ'},
	#{   'text': '大君', 'input': 'オオキミ'},
	{   'text': '大阪', 'input': 'オオサカ'},
	{   'text': '大旦那', 'input': 'オオダンナ'},
	{   'text': '大づかみ', 'input': 'オオヅカミ'},
	{   'text': '大人数', 'input': 'オオニンズー'},
	{   'text': '大田', 'input': 'オオタ'},

	{ 'note': '+ その他の収集事例 +' },

	{   'text':'↓最新号はこちらからお楽しみください↓', 'input':'↓サイシンゴーワ コチラカラ オタノシミクダサイ↓'},
	{   'text':'http://www.mag2.com/o/tabi/2013/1121.html', 'input':'⠠⠦http://www.mag2.com/o/tabi/2013/1121.html⠠⠴'},
	{   'text':'このメルマガはHTML形式です。HTMLメール表示をオフにしている', 'input':'コノ メルマガワ HTML ケイシキデス。HTML メール ヒョージヲ オフニシテ イル'},
	{   'text':'　　　　　　　　　　a', 'input':'          a'},
	{   'text':'http://www.mag2.com/o/tabi/2013/1121', 'input':'⠠⠦http://www.mag2.com/o/tabi/2013/1121⠠⠴'},
	{   'text':'http:', 'input':'http:'},
	{   'text':'http://www.mag2.com/o/tabi/2013/', 'input':'⠠⠦http://www.mag2.com/o/tabi/2013/⠠⠴'},
	{   'text':'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa', 'input':'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa'},
	{   'text':'http:///////aaaaaaaaaaaaaaaaaaaaaaaaaaa', 'input':'⠠⠦http:///////aaaaaaaaaaaaaaaaaaaaaaaaaaa⠠⠴'},
	{   'text':'http///////aaaaaaaaaaaaaaaaaaaaaaaaaa', 'input':'http///////aaaaaaaaaaaaaaaaaaaaaaaaaa'},
	{   'text':'http://////aaaaaaaaaaaaaaaaaaaaaaaaaaa', 'input':'⠠⠦http://////aaaaaaaaaaaaaaaaaaaaaaaaaaa⠠⠴'},
	{   'text':'http://///aaaaaaaaaaaaaaaaaaaaaaaaaaaa', 'input':'⠠⠦http://///aaaaaaaaaaaaaaaaaaaaaaaaaaaa⠠⠴'},
	{   'text':'http:/aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa', 'input':'http:/aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa'},
	{   'text':'http://aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa', 'input':'⠠⠦http://aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa⠠⠴'},

	{   'text':'障害者協会', 'input':'ショーガイシャ キョーカイ'},
	{   'text':'久代', 'input':'ヒサヨ'},
	{   'text':"　御園　", 'input':' ミソノ'},
	#{   'text':"　西本　", 'input':' ニシモト'},
	{   'text':'参加者　御園、田中、中原、御園', 'input':'サンカシャ ミソノ、タナカ、ナカハラ、ミソノ'},
	{   'text':'有り難うございました', 'input':'アリガトー ゴザイマシタ'},
	#{   'text':'いう', 'input':'イウ'},
	{   'text':'森田知事', 'input':'モリタ チジ'},
	#{   'text':'自ら行う', 'input':'ミズカラ オコナウ'},
	{   'text':'おはようございます', 'input':'オハヨー ゴザイマス'},
	{   'text':'森田県知事', 'input':'モリタ ケンチジ'},
	{   'text':'森田副知事', 'input':'モリタ フクチジ'},
	{   'text':'京丹後市', 'input':'キョータンゴシ'},
	#{   'text':'二人して', 'input':'フタリシテ'},
	{   'text':'京丹後', 'input':'キョータンゴ'},
	{   'text':'次世代型点字ピンディスプレイ', 'input':'ジセダイガタ テンジピンディスプレイ'},
	{   'text':'世界初', 'input':'セカイ ハツ'},
	{   'text':'見出しレベル３', 'input':'ミダシ レベル 3'},
	{   'text':'もうすぐ', 'input':'モー スグ'},
	#{   'text':'最上', 'input':'モガミ'},
	#{   'text':'間人', 'input':'タイザ'},

	{   'text':'秋子嬢', 'input':'アキコジョー'},
	{   'text':'宗谷岬', 'input':'ソーヤ ミサキ'},
	{   'text':'宗谷岬周辺の丘陵地帯', 'input':'ソーヤ ミサキ シューヘンノ キューリョー チタイ'},
	{   'text':'比治山峠', 'input':'ヒジヤマ トーゲ'},
	{   'text':'丹後半島', 'input':'タンゴ ハントー'},

	# 人名に造語要素が続く場合で、2拍以下の場合は
	# 続けるのが原則ですが、自立性が強く、意味の理解を助ける
	# 場合は、前を区切って書く
	{   'text':'上田訳', 'input':'ウエダ ヤク'},
	{   'text':'村岡花子訳', 'input':'ムラオカ ハナコ ヤク'},
	{   'text':'西本訳', 'input':'ニシモト ヤク'},

	{   'text':'鈴木作', 'input':'スズキ サク'},
	{   'text':'運慶作', 'input':'ウンケイ サク'},
	{   'text':'快慶作', 'input':'カイケイ サク'},
	{   'text':'ゴッホ作', 'input':'ゴッホ サク'},

	{   'text':'リンカーン談', 'input':'リンカーン ダン'},
	{   'text':'細川談', 'input':'ホソカワ ダン'},
	{   'text':'西本談', 'input':'ニシモト ダン'},

	{   'text':'田中著', 'input':'タナカ チョ'},
	{   'text':'山田耕筰曲', 'input':'ヤマダ コーサク キョク'},
	{   'text':'正岡子規記', 'input':'マサオカ シキ キ'},
	{   'text':'ゴッホ絵', 'input':'ゴッホ エ'},
	{   'text':'鈴木アナ', 'input':'スズキ アナ'},
	{   'text':'岡本プロ', 'input':'オカモト プロ'},

	#{   'text':'びっくら放いた', 'input':'ビックラ コイタ'},
	#{   'text':'五倍子', 'input':'ゴバイシ'},

	{ 'note': '+ 踊り字 +' },
	# http://ja.wikipedia.org/wiki/%E8%B8%8A%E3%82%8A%E5%AD%97

	# 同の字点
	# 々	3005	[クリカエシキゴー]	オドリジ
	{   'text':'呉々', 'input':'クレグレ'},
	{   'text':'呉々も', 'input':'クレグレモ'},
	{   'text':'時々', 'input':'トキドキ'},
	{   'text':'代々木', 'input':'ヨヨギ'},
	{   'text':'複々線', 'input':'フクフクセン'},
	{   'text':'複々々線', 'input':'フクフクフクセン'},
	{   'text':'複々々々線', 'input':'フクフクフクフクセン'},
	{   'text':'複々々々々線', 'input':'フクフクフクフクフクセン'},
	{   'text':'小々々支川', 'input':'ショーショーショーシセン'},
	{   'text':'結婚式々場', 'input':'ケッコンシキ シキジョー'},

	# 平仮名繰り返し記号
	# ゝ	309d	[ヒラガナクリカエシ]	クリカエシ
	# ゞ	309e	[ヒラガナダクテンクリカエシ]	クリカエシ ダクテン
	{   'text':'すゝめ', 'input':'ススメ'},
	{   'text':'すゞめ', 'input':'スズメ'},
	{   'text':'いすゞ', 'input':'イスズ'},
	{   'text':'づゝ', 'input':'ヅツ'},
	{   'text':'ぶゞ', 'input':'ブブ'},
	{   'text':'ふゞ', 'input':'フブ'},

	# 片仮名繰り返し記号
	# ヽ	30fd	[カタカナクリカエシ]	クリカエシ
	# ヾ	30fe	[カタカナダクテンクリカエシ]	クリカエシ ダクテン
	{   'text':'スヽメ', 'input':'ススメ'},
	{   'text':'イスヽ', 'input':'イスス'},
	{   'text':'イスヾ', 'input':'イスズ'},
	{   'text':'ヅヽ', 'input':'ヅツ'},
	{   'text':'ブヾ', 'input':'ブブ'},

	# 同上記号
	# 仝	4edd	[ドージョー]	ドージョー
	{   'text':'仝', 'input':'ドージョー'},

	# ノノ字点（ののじてん）・同じく記号
	# 〃	3003	[オナジク]	オナジク
	{   'text':'〃', 'input':'オナジク'},

	# 二の字点（にのじてん）
	# 文字説明に未登録
	# 〻    303b
	{   'text':'〻', 'input':'ニノジテン'},
	{   'text':'各〻', 'input':'オノオノ'},
	{   'text':'屡〻', 'input':'シバシバ'},

	# くの字点（くのじてん）
	# 文字説明に未登録
	# 〱    3031 くの字点
	# 〲    3032 くの字点(濁点)
	# 〳    3033 くの字点上
	# 〴    3034 くの字点上(濁点)
	# 〵    3035 くの字点下
	{   'text':'〱', 'input':'クノジテン'},
	{   'text':'〲', 'input':'クノジテン ダクテン'},
	{   'text':'〳', 'input':'クノジテン ウエ'},
	{   'text':'〴', 'input':'クノジテン ウエ ダクテン'},
	{   'text':'〵', 'input':'クノジテン シタ'},

]
