/*
 * OpenI2CRADIO
 * Shell Handler
 * Copyright (C) 2013-11-07 K.Ohta <whatisthis.sowhat ai gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#if defined(__SDCC)
#include <sdcc-lib.h>
#include <pic18fregs.h> /* ONLY FOR PIC18x */
#include <delay.h>
#else
#include <xc.h>
#endif
#include <signal.h>

#include "iodef.h"
#include "idle.h"
#include "ioports.h"
#include "euart.h"
#include "menu.h"
#include "power.h"
#include "shell_strutl.h"
#include "uart_termio.h"
#include "term_shell.h"
#include "eeprom_util.h"


unsigned char cmd_shellstr[130];
static char shell_strbuf[130];
static char xarg1[66];
static char xarg2[66];
static char xarg3[66];
static unsigned char tmparea[256];


static void shell_printnum(unsigned int v)
{
    char s[32];
    s[0] = '\0';
    term_printnum(s, v);
    uart_term_putstr(s);
}

/*
 * Set Frequency: Arg1 = Freq.
 */
static void term_freq(char *p)
{
    unsigned int freq;
    long l;

    l = term_getuint(p);
    uart_term_putstr("\nSet Freq:");
    if((l < 100) || (l > 30000)) {
        uart_term_putstr("\nE:Illegal freq range.\n");
        return;
    }
    freq = (unsigned int)l;
    if(setup.fm == 0){
        shell_printnum(setup.amfreq);
        // AM
        setup.amfreq = freq;
    } else {
        shell_printnum(setup.fmfreq);
        // FM
        setup.fmfreq = freq;
    }
    uart_term_putstr("->");
    shell_printnum(freq);

    akc6955_set_freq(freq);
    uart_term_putstr("\ndone.\n");
}

/*
 * Print status
 * Arg = none.
 */
static void term_printstatus(char *p)
{
    update_status();
    uart_term_putstr("\nStatus:\n");
    if(setup.fm == 0){
        uart_term_putstr("AM \nBAND = ");
        shell_printnum(setup.amband);
        uart_term_putstr("\nFREQ = ");
        shell_printnum(setup.amfreq);
        uart_term_putstr("KHz");
        uart_term_putstr("\nDIFF = ");
        shell_printnum(diffstat);
    } else {
        uart_term_putstr("FM \nBAND = ");
        shell_printnum(setup.fmband);
        uart_term_putstr("\nFREQ = ");
        shell_printnum(setup.fmfreq / 100);
        uart_term_putstr(".");
        shell_printnum(setup.fmfreq % 100);
        uart_term_putstr("MHz");
        uart_term_putstr("\nDIFF = ");
        shell_printnum(diffstat);
        uart_term_putstr("\nSTEREO = ");
        if(stereoflag == 0) {
	    uart_term_putstr("NO");
	} else {
	    uart_term_putstr("YES");
	}
    }
    uart_term_putstr("\nSIGNAL-LEVEL = ");
    shell_printnum(recv_signal);
    uart_term_putstr("dB");
   
    uart_term_putstr("\nCNR-LEVEL = ");
    shell_printnum(cnrlevel);
    uart_term_putstr("dB");
   
    uart_term_putstr("\nBATT = ");
    shell_printnum(battlevel / 100);
    uart_term_putstr(".");
    shell_printnum(battlevel % 100);
    uart_term_putstr("V\n");
   
    uart_term_putstr("\n");
}

static void term_fm(char *p)
{
    unsigned int freq;
    setup.fm = 0xff;
    setup.amfreq_bank[setup.amband] = setup.amfreq;
    freq = setup.fmfreq_bank[setup.fmband];
    akc6955_chg_fm(setup.fm, freq);
}

static void term_am(char *p)
{
    unsigned int freq;
    setup.fm = 0x00;
    setup.fmfreq_bank[setup.fmband] = setup.fmfreq;
    freq = setup.amfreq_bank[setup.amband];
    akc6955_chg_fm(setup.fm, freq);
}


/*
 * Poweroff.
 * Arg1: 'N'/'n' : not save.
 */
static void term_poweroff(char *p)
{
    unsigned char s[16];
    unsigned char slen;
    unsigned char savef = 0xff;
    xarg1[0] = '\0';
    slen = shell_gettok(xarg1, p);
    if(shell_strlen(xarg1) != 0) {
        if((xarg1[0] == 'N') || (xarg1[0] == 'n')) savef = 0;
    }
    uart_term_putstr("\n*** Power OFF ***\nOK? (Yes = Y)");
    uart_term_getstr(s, 15, 0xff);
    if(shell_strlen(s) >= 3) return;
    if(s[0] == 'Y') shutdown(savef);
}


static void term_poweron(char *p)
{
}


static void term_band(char *p)
{
   unsigned char slen;
   unsigned char n;
   unsigned char band;
   slen = shell_gettok(xarg1, p);
}


static unsigned char put_hexline(unsigned char *dat, unsigned long addr, unsigned char len)
{
    unsigned char l;
    l = str_bin2hex((unsigned char *)shell_strbuf, dat, addr, len);
    uart_term_putstr(shell_strbuf);
    return l;
}

static void put_hexheader(char *fname)
{
    uart_term_putstr("Start of save value.\n--- BEGIN ---\n");
    str_put_shexheader(shell_strbuf, fname);
    uart_term_putstr(shell_strbuf);
}


static void put_hexfooter(void)
{
    str_put_shexfooter(shell_strbuf);
    uart_term_putstr(shell_strbuf);
    uart_term_putstr("\n--- END---\n");
}



static void save_hex_page(unsigned char *p, unsigned int len, unsigned long baseaddr)
{
    unsigned int pp;
    unsigned int l = len;
        for(pp = 0; pp < len; pp += 16){
            if(l > 16) {
                put_hexline(&p[pp], pp + baseaddr, 16);
                l -= 16;
            } else {
                put_hexline(&p[pp], pp + baseaddr, l);
                break;
            }
	   idle_time_ms(100); // Line wait.
        }
}


void shell_memcpy(unsigned char *to, unsigned char *from, unsigned char len)
{
    unsigned char l;
    for(l = 0; l < len; l++) to[l] = from[l];
}


static char wait_for_sheader(unsigned char *file, unsigned char retry)
{
    unsigned char _try = 0;
    unsigned char _err = 0;
    int stat;
    file[0] = '\0';
   
    do {
        shell_strbuf[0] = '\0';
        uart_term_getstr(shell_strbuf, 100, 0); // With Echo, timeout=10Sec.
        stat = search_head_s(shell_strbuf);
        if(stat < 0) {
	   _err++;
	   if(_err > 100) return -1; // Error 
	   continue;
	}
        _err = 0;
        stat = str_shexheader(&shell_strbuf[stat], file);
        if(stat == TERM_OK) break;
        _try++;
    } while(_try <= retry);
    return stat;
}

static unsigned long load_hex_page(unsigned char *p, unsigned long *addr, unsigned int len)
{
    int stat;
    unsigned char l;
    unsigned long a;
    unsigned char __err = 0;
    unsigned long bytes = 0;
    int pos;
    unsigned char sbuf[20];
   
    while(len > 0){
_l0:

        shell_strbuf[0] = '\0';
        uart_term_getstr(shell_strbuf, 1000, 0); // Without Echo, timeout=100Sec.

        stat = search_head_s(shell_strbuf);
        if(stat < 0) {
	   __err++;
	   if(__err > 100) {
	      return 0xffffffff; // Error
	   }
	   ClrWdt();
	   idle_time_ms(10);
	   goto _l0;
	}
        __err = 0;
        pos = stat;
        stat = str_shex2bin(&shell_strbuf[stat], p, &a, &l);
        if(stat == TERM_SRECEND) {
	   return bytes; // Return head
	}
       
        if(stat != TERM_OK) {
	   return 0xffffffff; // Some error
	}
        
        bytes += l;
        *addr = a;
        if(l >= len) break; // End addr
        p += l;
        len -= l;
    }
    // Check return header
    shell_strbuf[0] = '\0';
    uart_term_getstr(shell_strbuf, 1000, 0); // Without Echo, timeout=100Sec.
    stat = search_head_s(shell_strbuf);
    if(stat < 0) return 0xffffffff;
    stat = str_shex2bin(&shell_strbuf[stat], p, &a, &l);
    if(stat != TERM_SRECEND) return 0xffffffff;
    return bytes;
}
/*
 * Load from S record.
 */
static void load_from_term(unsigned char *p)
{
    unsigned char slen = shell_gettok(xarg1, p);
    unsigned long addr = 0;
    unsigned char *pv;
    unsigned int len;
    unsigned char fbuf[128];
    unsigned long l;
    int stat;

    if(shell_strcmp(xarg1, "INT") > 0) { // Internal EEPROM
        uart_term_putstr("\nPls. start internal eeprom data...\n");

        stat = wait_for_sheader(fbuf, 10);

        if(stat != TERM_OK) goto _loaderr;
        if(shell_strcmp(fbuf, "INT_EEPROM") < 0) goto _fileerr;
        len = sizeof(__radioset_t);
        pv = tmparea;
//        pv = &setup;
        l = load_hex_page(pv, &addr, len);
        if(l == 0xffffffff) goto _loaderr;
        if(addr >= 255) goto _loaderr; // Too large
       
        shell_memcpy((unsigned char *)(&setup) , &tmparea, len);
        setup_akc6955(); // DO!
    } else if(shell_strcmp(xarg1, "FTBL") > 0) { // External EEPROM, Freq TBL
    } else if(shell_strcmp(xarg1, "BAND") > 0) { // band
    }
_OK:

    uart_term_putstr("\nOK.\n");
    return;

_loaderr:
    uart_term_putstr("\nERR: LOAD Error\n");
    return;
_addrerr:
    uart_term_putstr("\nERR: Address Error\n");
    return;
_fileerr:
    uart_term_putstr("\nERR: Not correct filename\n");
    return;
}

/*
 * Save ROM as S record.
 */
static void save_to_term(unsigned char *p)
{
    unsigned char slen;
    unsigned int len;
    unsigned int pp;
    int addr;
    unsigned char *pv;
    unsigned long base;

    slen = shell_gettok(xarg1, p);
    addr = 0;

    if(shell_strcmp(xarg1, "INT") > 0) { // Internal EEPROM
        put_hexheader("INT_EEPROM");
        len = sizeof(__radioset_t);
        pv = &setup;
        save_hex_page(pv, len,  0);
        put_hexfooter();
    } else if(shell_strcmp(xarg1, "FTBL") > 0) { // External EEPROM, Freq TBL
        put_hexheader("EXT_FREQTBL");
        base = 0x100000;
        for(pp = 0; pp < USER_MEMORY_BANKS; pp++) {
            if(load_frequencies(pp, 0xff) != 0) break; // Error
            save_hex_page((unsigned char *)(&freqset_temp), sizeof(__freqset_t), base + (pp << 9)) ;
        }
        put_hexfooter();
    } else if(shell_strcmp(xarg1, "BAND") > 0) { // band
        put_hexheader("EXT_BANDTBL");
        put_hexfooter();
    }
}

/*
 * Open the Shell: Return cmdcode.
 */
const char *cmdstr[SHELL_TOPCMDS] = {
    "HELP",
    "FREQ",
    "STAT",
    "LOAD",
    "SAVE",
    "FM",
    "AM",
    "POFF",
    "PON",
    "DUMP",
    "LOG",
    "SCAN",
    "BAND",
    "SET",
    "PRINT",
    "EXIT"
};


static const char shell_helpstr_help[] = "HELP [about-command]\n -- Online help.\n";
static const char shell_helpstr_freq[] = "FREQ frequency\n -- Set frequency.\n";
static const char shell_helpstr_stat[] = "STAT\n -- Print status.\n";
static const char shell_helpstr_load[] = "LOAD section [S]\n -- Load settings from terminal.\n Arg2: if 'S' then Save to internal EEPROM.\n";
static const char shell_helpstr_save[] = "SAVE section\n -- Save settings to terminal.\n";
static const char shell_helpstr_fm[] = "FM\n -- Switch to FM.\n";
static const char shell_helpstr_am[] = "AM\n -- Switch to AM.\n";
static const char shell_helpstr_poff[] = "\nPOFF [savef]\n -- Power OFF.\n  savef: if 'N' then power-off without saving.\n";
static const char shell_helpstr_pon[] = "\nPON\n -- Power ON.\n";
static const char shell_helpstr_dump[] = "\nDUMP section [number]\n -- Dump section(s).\n";
static const char shell_helpstr_log[] = "\nLOG\n -- View Log.\n";
static const char shell_helpstr_scan[] = "\nSCAN [direction]\n -- DO Scan.\n    direction : UP / DOWN\n";
static const char shell_helpstr_band[] = "\nBAND [band-name]\n -- Set band\n";
static const char shell_helpstr_set[] = "\nSET entry [value]\n -- Set setting / viewing value.\n   value : Decimal.\n";
static const char shell_helpstr_print[] = "\nPRINT [entry]\n -- Print system value(s).\n";
static const char shell_helpstr_exit[] = "\nEXIT\n -- EXIT FROM SHELL.\n";


static const char *shell_helpstr_sections[2] = {
   "\nsection :\n    INT : Internal EEPROM\n    FTBL : Freq table\n",
   "\n   FREQ num : Save user freq #num.\n   BAND : Save user band table."
};


static void cmd_printhelp(char cmdnum)
{
  const char *p;
  unsigned char i;
  unsigned char sectionf = 0;
   
   switch(cmdnum) {
    case 0:
      p = shell_helpstr_help;
      break;
    case 1:
      p = shell_helpstr_freq;
      break;
    case 2:
      p = shell_helpstr_stat;
      break;
    case 3:
      p = shell_helpstr_load;
      sectionf = 0xff;
      break;
    case 4:
      p = shell_helpstr_save;
      sectionf = 0xff;
      break;
    case 5:
      p = shell_helpstr_fm;
      break;
    case 6:
      p = shell_helpstr_am;
      break;
    case 7:
      p = shell_helpstr_poff;
      break;
    case 8:
      p = shell_helpstr_pon;
      break;
    case 9:
      p = shell_helpstr_dump;
      sectionf = 0xff;
      break;
    case 10:
      p = shell_helpstr_log;
      break;
    case 11:
      p = shell_helpstr_scan;
      break;
    case 12:
      p = shell_helpstr_band;
      break;
    case 13:
      p = shell_helpstr_set;
      break;
    case 14:
      p = shell_helpstr_print;
      break;
    case 15:
      p = shell_helpstr_exit;
      break;
    default:
      return;
   };
   uart_term_putstr(p);
   if(sectionf != 0) {
       for(i = 0; i < 2; i++) {
	  uart_term_putstr(shell_helpstr_sections[i]);
       }
   }
}

static void cmd_help(char *p)
{
    unsigned char i;
    unsigned char slen;
    int f;

    slen = shell_gettok(xarg1, p);
    if(slen > 1) {
        for(i = 0; i < SHELL_TOPCMDS; i++){
            f = shell_strcmp(xarg1, cmdstr[i]);
            if(f > 0) {
                cmd_printhelp(i);
                return;
            }
        }
    }
   uart_term_putstr("\nAvailable commands:\n");
   for(i = 0; i < SHELL_TOPCMDS; i++){
       uart_term_putstr(cmdstr[i]);
       uart_term_putstr("\n");
    }
    uart_term_putstr("See Details : HELP [cmd].\n");
}

char term_shell(unsigned int timeout)
{
    unsigned int t;
    unsigned char pp;
    unsigned char c;
    int i;
    unsigned int ii;
    char pool[127];

    //cmd_shellstr[0] = '\0';
    if(timeout != 0) {
        t = timeout;
        while((uart_getstat() & UART_WAKEUP) == 0) {
            if(t == 0) return SHELL_CMD_NONE;
            t--;
            idle_time_ms(1);
        }
    } else {
        if((uart_getstat() & UART_WAKEUP) == 0) return SHELL_CMD_NONE;
    }

    uart_term_putstr("\nOpen I2C Radio v2.0\n(C)2013- Kyuma Ohta\n");
    uart_term_putstr("\n\nWelcome to shell mode (^^).\nPress help to show usage.\n");
    do {
        cmd_shellstr[0] = '\0';
        uart_term_putstr("\n$>");
        uart_term_getstr(cmd_shellstr, 100, 1); // With Echo, timeout=10Sec.
        if(cmd_shellstr[0] == TERM_CHAR_TIMEOUT){
	   uart_term_putstr("\nTimeout.\n");
	   uart_init();
	   return SHELL_CMD_NONE; // TimeOut 
	}

   
       
        ClrWdt();
        ii = shell_gettok(pool, cmd_shellstr);
//        if(ii >= 126) return SHELL_CMD_TOOLONG;
        if(ii >= 126) continue; // Discard
        for(t = 0; t < SHELL_TOPCMDS; t++){
            i = shell_strcmp((char *)cmdstr[t], pool);
            if(i > 0) break;
        }

        if(cmd_shellstr[ii] == ' ') {
	   //if(i <= 0) return SHELL_CMD_NOTFOUND;
	   ii = shell_gettok(pool, &cmd_shellstr[ii + 1]);
	}
        if(ii >= 126) t = 127; // Error
        switch(t){
            case 0:
                cmd_help(pool);
                break;
            case 1: // Freq
                term_freq(pool);
                break;
            case 2:
                term_printstatus(pool);
                break;
            case 3:
                load_from_term(pool);
                break;
            case 4:
                save_to_term(pool);
                break;
            case 5:
                term_fm(pool);
                break;
            case 6:
                term_am(pool);
                break;
            case 7:
                term_poweroff(pool);
                break;
            case 8:
                term_poweron(pool);
                break;
            case 12: // BAND
	        term_band(pool);
	        break;
            case 15: // Exit
                uart_term_putstr("\nBye... (^^)/~~\n");
	        goto _l0;
	        break;
            default:
                uart_term_putstr("\n?? CMD Error\n");
                break;
        }

    } while(1);
_l0:
    uart_init();
    return SHELL_CMD_OK;
}
