'use strict';

const neo4j = require('neo4j-driver').v1;
const logger = require('../common/log-utils.js');
const fs = require('fs');
const exec = require('child_process').execSync;
const path = require('path');

const graphDb = process.env.GRAPH_DB_IP || '52.11.150.230';
const uri = 'bolt://' + graphDb + ':7687';
const user = process.env.GRAPH_USER || '';
const password = process.env.GRAPH_PWD || '';
const C_MODEL = '[xmail-edit]';

/*
 * XMAILデータ登録処理
 *
 * ・引数に指定されたファイルを一度添付ファイルとして保存する
 * ・xml2cypher.pyを実行する
 * ・GrpahDBに接続する
 * ・cypher実行：(1) XMAILファイル内のXML構造作成
 * ・(1)で登録できたcypherのreturn = IDを取得する
 * ・(2) XMAIL内のペトリネット構造作成用のCypher生成
 * ・cypher実行：(2) XMAIL内のペトリネット構造作成
 * ・(1), (2)共に正常終了した場合のみcommit、以外の場合はrollbackしてエラーを表示
 * ・一時ファイルを削除する
 *
 * @param {filename} filename - XMAILファイル名
 * @param {xmail} xmail - XMAILデータ内容
 * @see https://neo4j.com/docs/api/javascript-driver/current/
 * @returns [0]: JSON for XHR
 * @throws {error}
 */
exports.upload = async function(filename, xmail) {
	var idx = 0;
	var cypher1 = '';
	var idNx = '';
	var graphJson = '[0]';
	var uploadResult = false;

	// 引数に指定されたファイルを一時ファイルとして保存する
	const tmpfile = path.join(__dirname, 'tmp', filename);
	logger.app.debug(C_MODEL + 'filename:' + tmpfile);

	try {
		fs.writeFileSync(tmpfile, xmail);
		logger.app.debug(C_MODEL + 'Temporary file created.');
	} catch (err) {
		logger.app.error(
			C_MODEL + 'Temporary file creation failed：' + err.message
		);
		throw err;
	}

	// xml2cypher.pyを実行する
	try {
		var cmd = '';
		cmd =
			'PYTHONIOENCODING=utf-8 ' +
			path.join(__dirname, 'python', 'xml2cypher.py') +
			' ' +
			'"' +
			tmpfile +
			'"';
		cypher1 = cmdExecute(cmd);
		//logger.app.debug(C_MODEL + 'xml2cypher.py executed : ' + cypher1.toString());
	} catch (err) {
		logger.app.error(C_MODEL + err.name + ':' + err.message);
		throw err;
	}

	const driver = neo4j.driver(uri, neo4j.auth.basic(user, password));
	logger.app.debug(C_MODEL + 'Graph DB connected');

	var session = driver.session();
	var tx = session.beginTransaction();
	logger.app.debug(C_MODEL + 'tx created.');

	// cypher実行：(1) XMAILファイル内のXML構造を作成する
	try {
		await tx
			.run(cypher1.toString())
			.then(function(result) {
				result.records.forEach(function(record) {
					idNx = record.get('id(nx)');
					logger.app.debug(
						C_MODEL + 'Cypher1 Record:' + idx + ' id(nx): ' + idNx
					);
					logger.app.debug(
						C_MODEL + 'Cypher1 Counter:' + result.summary.counters
					);
					logger.app.debug(
						C_MODEL +
							'Cypher1 Notification:' +
							result.summary.notifications
					);
					idx = idx + 1;
				});
			})
			.catch(function(err) {
				tx.rollback();
				session.close();
				logger.app.error(
					C_MODEL +
						'Cypher1 error rollbacked. Session closed:' +
						err.message
				);
				throw err;
			});
	} catch (err) {
		await tx.rollback();
		await session.close();
		logger.app.error(
			C_MODEL + 'Error rollbacked. Session closed : ' + err.message
		);
		throw err;
	}

	// (1) で登録できたcypherのreturn = IDを取得する
	// (2) XMAIL内のペトリネット構造作成用のCypherを生成する
	var cypher2 =
		`match (a:XMAIL)
					-[:XML_Root]->(d)
					-[:XML_Child]->(pr {__tag: 'protocol'})
					-[:XML_Child]->(me {__tag: 'method'})
					-[:XML_Child]->(pg {__tag: 'program'})
					-[:XML_Child]->(pn {__tag: 'pnml'}),
					(pn)-[:XML_Child]->(ar {__tag: 'arc'}),
					(pn)-[:XML_Child]->(s),
					(pn)-[:XML_Child]->(t)
				where
					id(a)=` +
		idNx +
		`
					and ar.source=s.id
					and ar.target=t.id
				merge (s)-[:PNarc]->(t);
				`;
	logger.app.debug(C_MODEL + 'Cypher2 : ' + cypher2);

	// cypher実行：(2) XMAIL内のペトリネット構造を作成する
	try {
		await tx
			.run(cypher2)
			.then(function(result) {
				logger.app.debug(
					C_MODEL + 'Cypher2 Counter:' + result.summary.counters
				);
				logger.app.debug(
					C_MODEL +
						'Cypher2 Notification:' +
						result.summary.notifications
				);
				uploadResult = true;
			})
			.catch(function(err) {
				if (err.message === undefined) {
					logger.app.debug(C_MODEL + 'Cypher2 Not affected.');
				} else {
					logger.app.error(
						C_MODEL + 'Cypher2 error rollbacked : ' + err.message
					);
					tx.rollback();
					throw err;
				}
			});
	} catch (err) {
		await tx.rollback();
		logger.app.error(C_MODEL + 'Error rollbacked : ' + err.message);
		throw err;
	} finally {
		if (uploadResult == true) {
			// (1), (2)共に正常終了した場合のみcommit、以外の場合はrollbackしてエラーを表示
			await tx.commit();
			logger.app.debug(C_MODEL + 'Commit completed.');
		}
		await session.close();
		logger.app.debug(C_MODEL + 'Finally session closed.');
	}

	//一時ファイルを削除する
	try {
		fs.unlinkSync(tmpfile);
		logger.app.debug('[xmail-edit]Temporary file deleted.');
	} catch (err) {
		logger.app.error(
			'[xmail-edit]Temporary file deletion failed ： ' + err.message
		);
		throw err;
	}

	return graphJson;
};

/*
 * コマンド実行
 *
 * ・引数に指定されたコマンドを実行する。
 * ・実行環境においてコマンドライン上で"python"コマンドが実行できることが前提
 *
 * @param {cmd} cmd - コマンド（pythonスクリプト実行を想定)
 * @returns {stdout} コマンド実行後の標準出力結果
 * @throws {error}
 */
function cmdExecute(cmd) {
	try {
		logger.app.debug('[cmdExecute]' + cmd);
		//var result = exec('python -c "import sys; print("hoge"); sys.exit(-1)"');
		var result =  exec(cmd, { timeout: 3000 });
		return result.toString();

	} catch (err) {
		logger.app.error('[cmdExecute]' + err.status);
		logger.app.error('[cmdExecute]' + err.stdout);
		logger.app.error('[cmdExecute]' + err.stderr);
		throw err;
	}
}

/*
 * XMAILデータ登録処理
 *
 * ・引数に指定されたNIDをキーに、GraphDBのXMAILデータ一式を削除する
 * ・GrpahDBに接続する
 * ・cypher実行：XMAILデータ削除
 *
 * @param {nid} nid - XMAIL一意キー
 * @see https://neo4j.com/docs/api/javascript-driver/current/
 * @returns [0]: JSON for XHR
 * @throws {error}
 */
exports.delete = async function(nid) {
	var graphJson = '[0]';

	const driver = neo4j.driver(uri, neo4j.auth.basic(user, password));
	logger.app.debug(C_MODEL + 'Graph DB connected');

	var session = driver.session();
	var tx = session.beginTransaction();
	logger.app.debug(C_MODEL + 'tx created.');

	var cypher =
		`match p=(a:XMAIL)
					-[:XML_Root]->(r)
					-[:XML_Child|XML_Data*1..]->(n)
					where id(a)=` +
		nid +
		`
					detach delete p;`;
	logger.app.debug(C_MODEL + 'Delete Cypher : ' + cypher);

	try {
		await tx
			.run(cypher)
			.then(function(result) {
				logger.app.debug(
					C_MODEL + 'Cypher Counter:' + result.summary.counters
				);
				logger.app.debug(
					C_MODEL +
						'Cypher Notification:' +
						result.summary.notifications
				);
			})
			.catch(function(err) {
				if (err.message === undefined) {
					logger.app.debug(C_MODEL + 'Cypher Not affected.');
				} else {
					logger.app.error(
						C_MODEL + 'Cypher error rollbacked : ' + err.message
					);
					tx.rollback();
					throw err;
				}
			});
	} catch (err) {
		await tx.rollback();
		logger.app.error(C_MODEL + 'Error rollbacked : ' + err.message);
		throw err;
	} finally {
		await tx.commit();
		logger.app.debug(C_MODEL + 'Commit completed.');
		await session.close();
		logger.app.debug(C_MODEL + 'Finally session closed.');
	}

	return graphJson;
};
