/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.file.dao.standard;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;

import jp.terasoluna.fw.file.annotation.FileFormat;
import jp.terasoluna.fw.file.annotation.InputFileColumn;
import jp.terasoluna.fw.file.annotation.StringConverter;
import jp.terasoluna.fw.file.dao.FileException;
import jp.terasoluna.fw.file.dao.FileLineException;
import jp.terasoluna.fw.file.dao.FileLineIterator;

import org.apache.commons.lang.StringUtils;

/**
 * t@CANZX(f[^擾)p̋ʃNXB
 * 
 * <p>
 * t@CANZX(f[^擾)s3̃NX(CSVAŒ蒷Aϒ) ɋʂ鏈܂Ƃ߂ۃNXB
 * t@C̎ނɑΉTuNXsB<br>
 * gp{@link jp.terasoluna.fw.file.dao.FileLineIterator}QƂ̂ƁB
 * </p>
 * 
 * t@C擾̓f[^擪珇Ԃɓǂݍނ߁AL̎菇ŌĂяo悤ɎKv܂B<br>
 * <ul>
 * <li>wb_擾(getHeader())</li>
 * <li>XLbv(skip())</li>
 * <li>f[^擾(hasNext()AreadLine())</li>
 * <li>gC擾(getTrailer())</li>
 * </ul>
 * 
 * AgC̎擾sƓŎcĂf[^SXLbv邽߁A
 * rɃgC擾ƃf[^̎擾oȂȂ܂B<br>
 * gC̎擾f[^擾s<code>IllegalStateException<code>܂B<br>
 * 
 * @see jp.terasoluna.fw.file.dao.FileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.CSVFileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.FixedFileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.VariableFileLineIterator
 * @see jp.terasoluna.fw.file.dao.standard.PlainFileLineIterator
 * 
 * @param <T> t@CsIuWFNgB
 */
public abstract class AbstractFileLineIterator<T> implements
        FileLineIterator<T> {

    /**
     * 킷sԍB 
     */
    private static final int INITIAL_LINE_NO = -1;
    
    /**
     * t@CB
     */
    private String fileName = null;

    /**
     * ʃNXB
     */
    private Class<T> clazz = null;

    /**
     * s؂蕶B
     */
    private String lineFeedChar = System.getProperty("line.separator");

    /**
     * t@CGR[fBOB
     */
    private String fileEncoding = System.getProperty("file.encoding");

    /**
     * wb_sB
     */
    private int headerLineCount = 0;

    /**
     * gCsB
     */
    private int trailerLineCount = 0;

    /**
     * t@C͏ς݂̃f[^̍sB
     */
    private int currentLineCount = 0;

    /**
     * t@CANZXp̕Xg[B
     */
    private Reader reader = null;

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐB
     */
    private Field[] fields = null;

    /**
     * t@CsIuWFNg̃XgORo[^i[ϐB
     */    
    private StringConverter[] stringConverters = null;
    
    /**
     * t@CsIuWFNg̃XgORo[^i[}bvB
     */
    private static Map<Class, StringConverter> stringConverterCacheMap = 
        new HashMap<Class, StringConverter>();
 
    /**
     * t@CsIuWFNgFieldɑΉsetter\bhi[B
     */
    private Method[] methods = null;

    /**
     * Jp[T[i[}bvB
     */
    private Map<String, ColumnParser> columnParserMap = null;

    /**
     * wb_̕񃊃XgB
     */
    private List<String> header = new ArrayList<String>();

    /**
     * gC̕񃊃XgB
     */
    private List<String> trailer = new ArrayList<String>();

    /**
     * gCmFptOB
     */
    private boolean readTrailer = false;

    /**
     * gC̈ꎞi[p̃L[B
     */
    private Queue<String> trailerQueue = null;

    /**
     * 1s̕ǂݍރIuWFNg
     */
    private LineReader lineReader = null;
    
    /**
     * mFptOB
     */
    private boolean calledInit = false;

    /**
     * RXgN^B<br>
     * ̃`FbNyсAt@CsIuWFNgFileFormatAme[V
     * ݒ̃`FbNsB<br>
     * `FbNʖ肪ꍇ͗O𔭐B<br>
     * 
     * @param fileName t@CŌɈړ
     * @param clazz t@CsIuWFNgNX
     * @param columnParserMap tH[}bgXg
     * @throws FileException ŎsꍇB
     */
    public AbstractFileLineIterator(String fileName, Class<T> clazz,
            Map<String, ColumnParser> columnParserMap) {
        // t@C̕K{`FbNsB
        if (fileName == null || "".equals(fileName)) {
            throw new FileException("fileName is required.", 
                    new IllegalArgumentException(), fileName);
        }

        // t@CsIuWFNgNX̕K{`FbNsB
        if (clazz == null) {
            throw new FileException("clazz is required.", 
                    new IllegalArgumentException(), fileName);
        }

        // tH[}bgXg̕K{`FbNsB
        if (columnParserMap == null || columnParserMap.isEmpty()) {
            throw new FileException("columnFormaterMap is required.", 
                    new IllegalArgumentException(), fileName);
        }
        
        // t@CsIuWFNgNXCX^Xł邩`FbNB
        try {
            clazz.newInstance();
        } catch (InstantiationException e) {
            throw new FileException("Failed in instantiation of clazz.",
                    e, fileName);
        } catch (IllegalAccessException e) {
            throw new FileException(
                    "clazz's nullary  constructor is not accessible",
                    e, fileName);
        }

        this.fileName = fileName;
        this.clazz = clazz;
        this.columnParserMap = columnParserMap;
        
        // FileFormatAme[V̐ݒ`FbNB
        FileFormat fileFormat = clazz.getAnnotation(FileFormat.class);
        
        // t@CsIuWFNgClassFileFormatAme[V邩`FbNB
        if (fileFormat == null) {
            throw new FileException("FileFormat annotation is not found.", 
                    new IllegalStateException(), fileName);
        }
    
        // ؂蕶ƈ͂ݕꍇAOX[B
        if (fileFormat.delimiter() == fileFormat.encloseChar()) {
            throw new FileException(
                    "Delimiter is the same as EncloseChar and is no use.",
                    new IllegalStateException(), fileName);
        }
        
        // s؂蕶`FbNBݒ肪Ȃꍇ̓VXeftHgl𗘗pB
        if (fileFormat.lineFeedChar() != null
                && !"".equals(fileFormat.lineFeedChar())) {
            this.lineFeedChar = fileFormat.lineFeedChar();
        }
        
        // t@CGR[fBO`FbNBݒ肪Ȃꍇ̓VXeftHgl𗘗pB
        if (fileFormat.fileEncoding() != null
                && !"".equals(fileFormat.fileEncoding())) {
            this.fileEncoding = fileFormat.fileEncoding();
        }
        
        // wb_sݒ肷B
        this.headerLineCount = fileFormat.headerLineCount();
        
        // gCsݒ肷B
        this.trailerLineCount = fileFormat.trailerLineCount();
    }

    /**
     * ̍s̃R[h邩ǂmFB<br>
     * JԂłɗvfꍇ true Ԃ܂B
     * 
     * @return JԂłɗvfꍇ <code>true</code>
     * @throws FileException [_IOExceptionꍇB
     */
    public boolean hasNext() {
        try {
            if (reader.ready()) {
                return true;
            }
        } catch (IOException e) {
            throw new FileException("Processing of reader was failed.",
                    e, fileName);
        }
        return false;
    }

    /**
     * JԂŃt@CsIuWFNgԋpB<br>
     * 
     * <p>
     * ̍s̃R[h̏t@CsIuWFNgɊi[ĕԋp܂B<br>
     * JԂŎ̗vfԂ܂B<br>
     * </p>
     * ̍s̃R[h̏̓t@CsIuWFNgInputFileColumn̒`
     * ÂĊi[B<br>
     * At@CsIuWFNg̃}bsOtB[h̐ƍȂ
     * R[h񂪗ꍇ͗O𔭐B<br>
     * ܂AInputFileColumnɐݒ肳ꂽoCgƈႤ񂪗ꍇO𔭐B<br>
     * ł͂Ȃꍇ͈ȉ̏ԂŃf[^i[B<br>
     * <ul>
     * @@<li>g</li>
     * @@<li>pfBO</li>
     * @@<li>ϊ</li>
     * @@<li>^ϊ(}bsO)</li>
     * </ul>
     * 
     * @return t@CsIuWFNg
     * @throws FileException t@CsIuWFNg̐ɎsꍇB
     * @throws FileLineException t@CsIuWFNg̎擾ɎsꍇB
     */
    public T next() {
        if (readTrailer) {
            throw new FileLineException(
                    "Data part should be called before trailer part.", 
                    new IllegalStateException(),
                    fileName, currentLineCount);
        }
        
        if (!hasNext()) {
            throw new FileLineException(
                    "The data which can be acquired doesn't exist.",
                    new NoSuchElementException(), 
                    fileName, currentLineCount);
        }
        
        T fileLineObject = null;
        
        // ̍sf[^ǂށBhasNext()`FbNs߁Anull̏ꍇȂB
        String currentString = readLine();

        // t@CsIuWFNgVɐ鏈B
        try {
            fileLineObject = clazz.newInstance();
        } catch (InstantiationException e) {
            throw new FileException(
                    "Failed in an instantiate of a FileLineObject.",
                    e, fileName);
        } catch (IllegalAccessException e) {
            throw new FileException(
                    "Failed in an instantiate of a FileLineObject.",
                    e, fileName);
        }

        // CSV̋؂蕶ɂē̓f[^𕪉B
        // ؂蕶̓Ame[V擾B
        String[] columns = separateColumns(currentString);

        // t@CǂݎJƃt@CsIuWFNg̃JrB
        if (fields.length != columns.length) {
            throw new FileLineException("Column Count is different from "
                    + "FileLineObject's column counts",
                    new IllegalStateException(),
                    fileName, currentLineCount + 1);
        }
        
        InputFileColumn inputFileColumn = null;
        int columnIndex = -1;
        String columnString = null;
        
        for (int i = 0; i < fields.length; i++) {
            
            // JavaBean̓͗p̃Ame[Vݒ肷B
            inputFileColumn = fields[i].getAnnotation(InputFileColumn.class);
            columnIndex = inputFileColumn.columnIndex();

            // 1J̕ZbgB
            columnString = columns[columnIndex];

            // J̃oCg`FbNB
            if (isCheckByte(inputFileColumn)) {
                try {
                    if (columnString.getBytes(fileEncoding).length
                            != inputFileColumn.bytes()) {
                        throw new FileLineException(
                                "Data size is different from a set point "
                                + "of a column.", new IllegalStateException(),
                                fileName, currentLineCount + 1,
                                fields[i].getName(),
                                columnIndex);
                    }
                } catch (UnsupportedEncodingException e) {
                    throw new FileException(
                            "fileEncoding which isn't supported was set.",
                            e, fileName);
                }
            }

            // g
            columnString = FileDAOUtility.trim(columnString, fileEncoding,
                    inputFileColumn.trimChar(), inputFileColumn.trimType());

            // pfBO
            columnString = FileDAOUtility.padding(columnString,
                    fileEncoding, inputFileColumn.bytes(), inputFileColumn
                            .paddingChar(), inputFileColumn.paddingType());

            // ϊ̏B
            columnString = stringConverters[i].convert(columnString);

            // li[鏈B
            // JavaBean̑̌^̖OɂďU蕪B
            ColumnParser columnParser = columnParserMap.get(fields[i]
                    .getType().getName());
            try {
                columnParser.parse(columnString, fileLineObject, methods[i],
                        inputFileColumn.columnFormat());
            } catch (IllegalArgumentException e) {
                throw new FileLineException("Failed in coluomn data parsing.",
                        e, fileName, currentLineCount + 1, fields[i].getName(),
                        columnIndex);
            } catch (IllegalAccessException e) {
                throw new FileLineException("Failed in coluomn data parsing.",
                        e, fileName, currentLineCount + 1, fields[i].getName(),
                        columnIndex);
            } catch (InvocationTargetException e) {
                throw new FileLineException("Failed in coluomn data parsing.",
                        e, fileName, currentLineCount + 1, fields[i].getName(),
                        columnIndex);
            } catch (ParseException e) {
                throw new FileLineException("Failed in coluomn data parsing.",
                        e, fileName, currentLineCount + 1, fields[i].getName(),
                        columnIndex);
            }

        }

        currentLineCount++;
        return fileLineObject;
    }

    /**
     * T|[gȂB<br>
     * IteratorŒ`Ă郁\bhB<br>
     * FileQueryDAOł͎Ȃ̂ŁÃNXĂяoꍇA
     * UnsupportedOperationExceptionX[B
     * @throws UnsupportedOperationException ̃\bh̓T|[gȂB
     */
    public void remove() {
        throw new UnsupportedOperationException("remove() isn't supported.");
    }

    /**
     * sB<br>
     * ōs͈ȉłBB
     * <ul>
     *   <li>t@CsIuWFNg̑(Field)̎擾</li>
     *   <li>ϊʃIuWFNg(stringConverters)̐</li>
     *   <li>t@CsIuWFNg̑ɑ΂Zb^\bh(methods)̎擾</li>
     *   <li>t@Cf[^Ǎނ߂LineReader̐</li>
     *   <li>wb_̎擾</li>
     *   <li>gCL[̏</li>
     * </ul>
     * 
     * init()AbstracFileLineIteratorpNX̃RXgN^
     * Ăԃ\bhłB<br>
     * ʌ݊̂߁A2ȏsłȂ悤ɂĂB
     * 
     * @throws FileException ŎsꍇB
     * @throws FileLineException JɊ֘A鏉ŎsꍇB
     */
    protected void init() {
        if (!calledInit) {
            // tNV̊{𐶐B
            buildFields();
            buildStringConverters();
            buildMethods();
            
            // t@Cf[^Ǎނ߂LineReader𐶐B
            buildLineReader();
            
            // wb_ƃgC̎擾邽߂̊{𐶐B
            buildHeader();
            buildTrailerQueue();
            
            calledInit = true;
        }
    }

    /**
     * t@Cf[^Ǎނ߂LineReader𐶐B<br>
     * t@CReader̐сApׂLineReader̐sB<br>
     * 
     * s؂蕶1,2ł͂Ȃꍇ͗O𔭐B
     * 
     * @throws FileException LineReader̐ɎsꍇB
     */
    private void buildLineReader() {
        // Ώۃt@Cɑ΂Reader擾B
        try {
            this.reader = new BufferedReader(new InputStreamReader(
                    (new FileInputStream(fileName)), fileEncoding));
        } catch (UnsupportedEncodingException e) {
            throw new FileException("Failed in generation of reader.",
                    e, fileName);
        } catch (FileNotFoundException e) {
            throw new FileException("Failed in generation of reader.",
                    e, fileName);
        }

        // s؂蕶ƈ͂ݕ̏ɊÂLineReader𐶐B
        if (lineFeedChar.length() == 2) {
            // s؂蕶2
            if (getEncloseChar() == Character.MIN_VALUE) {
                // ͂ݕ
                lineReader = new LineFeed2LineReader(reader, lineFeedChar);
            } else {
                // ͂ݕ
                lineReader = new EncloseCharLineFeed2LineReader(
                        getDelimiter(), getEncloseChar(), reader,
                        lineFeedChar);
            }
        } else if (lineFeedChar.length() == 1) {
            // s؂蕶1
            if (getEncloseChar() == Character.MIN_VALUE) {
                // ͂ݕ
                lineReader = new LineFeed1LineReader(reader, lineFeedChar);
            } else {
                // ͂ݕ
                lineReader = new EncloseCharLineFeed1LineReader(
                        getDelimiter(), getEncloseChar(), reader,
                        lineFeedChar);
            }
        } else {
            throw new FileException(
                    "lineFeedChar length must be 1 or 2. but: "
                    + lineFeedChar.length(), new IllegalStateException(),
                    fileName);
        }
    }

    /**
     * InputFileColumnAme[Vݒ肳Ăt@CsIuWFNg
     * ̔z𐶐B<br>
     * 
     * 擾Ώۑ̓t@CsIuWFNgƌp̑SNX̑łB<br>
     * 擾{@link InputFileColumn#columnIndex()}Əd
     * ꍇ͗OB<br>
     * ܂A{@link InputFileColumn#columnIndex()}̍őlJ̐ƍȂ
     * ꍇOB<br>
     * 
     * t@CsIuWFNg̑̐ݒɖ肪ꍇ
     * InputFileColumnAme[Vݒ肪鑮̂ݐzɂB<br>
     * 
     * @throws FileException JCfbNXdꍇB
     */
    private void buildFields() {
        // tB[hIuWFNg𐶐
        List<Field[]> allFields = new ArrayList<Field[]>();

        // tB[hIuWFNg𐶐
        Class tempClass = clazz;
        Field[] declaredFieldArray = null;
        int allFieldCount = 0;
        while (tempClass != null) {
            declaredFieldArray = tempClass.getDeclaredFields();
            allFields.add(declaredFieldArray);
            allFieldCount += declaredFieldArray.length;
            tempClass = tempClass.getSuperclass();
        }
        
        // JCfbNX̒`̏Ԃɕёւ
        Field[] dataColumnFields = new Field[allFieldCount];

        InputFileColumn inputFileColumn = null;
        int maxColumnIndex = -1;
        int columnIndex = -1;
        int columnCount = 0;
        
        for (Field[] fields : allFields) {
            for (Field field : fields) {
                inputFileColumn = field.getAnnotation(InputFileColumn.class);
                if (inputFileColumn != null) {
                    // }bsO\Ȍ^̃tB[hȂ̂mFB
                    if (columnParserMap.get(field.getType().getName())
                            == null) {
                        throw new FileException(
                                "There is a type which isn't supported in a "
                                + "mapping target field in FileLineObject.",
                                new IllegalStateException(), fileName);
                    }
                    
                    columnIndex = inputFileColumn.columnIndex();
                    // JIndex}CiXlȂ̂mFB
                    if (columnIndex < 0) {
                        throw new FileException(
                                "Column Index in FileLineObject is the minus " +
                                "number.",
                                new IllegalStateException(), fileName);
                    }
                    // JIndextB[h𒴂Ă邩邩mFB
                    if (dataColumnFields.length <= columnIndex) {
                        throw new FileException(
                                "Column Index in FileLineObject is bigger than "
                                + "the total number of the field.",
                                new IllegalStateException(), fileName);
                    }
                    // JIndexdĂȂ̂mFB
                    if (dataColumnFields[columnIndex] == null) {
                        dataColumnFields[columnIndex] = field;
                        if (maxColumnIndex < columnIndex) {
                            maxColumnIndex = columnIndex;
                        }
                        columnCount++;
                    } else {
                        throw new FileException("Column Index is duplicate : "
                                + columnIndex, fileName);
                    }
                }
            }
        }
        
        // columnIndexAԂŒ`Ă邩`FbN
        if (columnCount != (maxColumnIndex + 1)) {
            throw new FileException(
                    "columnIndex in FileLineObject is not sequential order.",
                    new IllegalStateException(), fileName);
        }
        
        // tB[hRs[(null̕폜)
        if (dataColumnFields.length == columnCount) {
            this.fields = dataColumnFields;
        } else {
            this.fields = new Field[columnCount];
            System.arraycopy(dataColumnFields, 0, this.fields, 0, columnCount);
        }
    }

    /**
     * t@CsIuWFNg̑̕ϊʃIuWFNg̔z𐶐B<br>
     * ꂽϊʃIuWFNgCX^X̓LbVA
     * l̐ݒ肪鑮ŗpB<br>
     * ݒ肳ꂽϊʃIuWFNgCX^XoȂ̂̏ꍇ
     * OB<br>
     * 
     * @throws FileLineException ϊʃIuWFNg̐ɎsꍇB
     */
    private void buildStringConverters() {
 
        // ϊʂ̔z𐶐
        StringConverter[] dataColumnStringConverters =
            new StringConverter[fields.length];
        
        InputFileColumn inputFileColumn = null;
        Class<? extends StringConverter> converterKind = null;
            
        for (int i = 0; i < fields.length; i++) {
            
            // JavaBean̓͗p̃Ame[V擾B
            inputFileColumn = fields[i].getAnnotation(InputFileColumn.class);
            
            // inputFileColumn.stringConverter()̓eɂ菈U蕪B
            try {
                // ϊʂ̃Ame[V擾B
                converterKind = inputFileColumn.stringConverter();
                
                // }bvɎ擾ϊʂƈvL[݂邩肷B
                if (stringConverterCacheMap.containsKey(converterKind)) {
                    // }bvIuWFNg擾Aϊʂ̔zɃZbgB
                    dataColumnStringConverters[i] =
                        stringConverterCacheMap.get(converterKind);  
                
                } else {
                    // CX^X𐶐Aϊʂ̔zɃZbgB
                    dataColumnStringConverters[i] = converterKind.newInstance();
                    stringConverterCacheMap.put(converterKind,
                            dataColumnStringConverters[i]);
                }           
                
            } catch (InstantiationException e) {
                throw new FileLineException(
                        "Failed in an instantiate of a stringConverter.",
                        e, fileName, INITIAL_LINE_NO,
                        fields[i].getName(), inputFileColumn.columnIndex());
                
            } catch (IllegalAccessException e) {
                throw new FileLineException(
                        "Failed in an instantiate of a stringConverter.",
                        e, fileName, INITIAL_LINE_NO,
                        fields[i].getName(), inputFileColumn.columnIndex());
            }
        }
        this.stringConverters = dataColumnStringConverters;
    }

    /**
     * t@CsIuWFNg̑setter\bh̃\bhIuWFNg̔z𐶐B<br>
     * ɑ΂setter\bh͈ȉ̃[ŌB<br>
     * <ul>
     * <li>
     * ̍ŏ̕啶ɂ̐擪Ɂusetv́B
     * </li>
     * </ul>
     * 
     * setter\bhłȂꍇ͗OB
     * 
     * @throws FileException setter\bhȂꍇB
     */
    private void buildMethods() {
        Method[] dataColumnSetMethods = new Method[fields.length];
        StringBuilder setterName = new StringBuilder();
        String fieldName = null;

        for (int i = 0; i < fields.length; i++) {
            // JavaBean珈̑ΏۂƂȂ鑮̑擾B
            fieldName = fields[i].getName();

            // ɁAsetter\bh̖O𐶐B
            setterName.setLength(0);
            setterName.append("set");
            setterName.append(StringUtils.upperCase(fieldName.substring(0,
                    1)));
            setterName.append(fieldName.substring(1, fieldName.length()));

            // setter̃tNVIuWFNg擾B
            // fields[i].getType()ň̌^w肵ĂB
            try {
                dataColumnSetMethods[i] = clazz.getMethod(
                        setterName.toString(),
                        new Class[] { fields[i].getType() });
            } catch (NoSuchMethodException e) {
                throw new FileException(
                        "The setter method of column doesn't exist.",
                        e, fileName);
            }
        }
        this.methods = dataColumnSetMethods;
    }

    /**
     * wb_̎擾sB<br>
     * w肳ꂽs̃f[^݂ȂꍇɗOԂB<br>
     * 
     * @throws FileException wb_̎擾ɎsꍇB
     */
    private void buildHeader() {
        if (0 < headerLineCount) {
            for (int i = 0; i < headerLineCount; i++) {
                if (!hasNext()) {
                    throw new FileException(
                            "The data which can be acquired doesn't exist.",
                            new NoSuchElementException(),
                            fileName);
                }
                try {
                    header.add(lineReader.readLine());
                } catch (FileException e) {
                    throw new FileException(
                            "Error occurred by reading processing of a File.",
                            e, fileName);
                }
            }
        }
    }

    /**
     * gCL[̏sB<br>
     * gC̓f[^Sǂ񂾌̕ō\܂A
     * t@C͑Oɏɓǂ܂邽߁A<br>
     * 擾f[^f[^̏Ȃ̂gC̏񂩂fłȂB<br>
     * ̂߁AL[Ƀf[^Ď擾B<br>
     * w肳ꂽs̃f[^݂ȂꍇɗOԂB<br>
     * 
     * @throws FileException gCL[̏sꍇB
     */
    private void buildTrailerQueue() {
        if (0 < trailerLineCount) {
            // gCL[CX^X𐶐B
            trailerQueue = new ArrayBlockingQueue<String>(trailerLineCount);
            
            // gCL[̃gCs̃f[^ǉB
            for (int i = 0; i < trailerLineCount; i++) {
                if (!hasNext()) {
                    throw new FileException(
                            "The data which can be acquired doesn't exist.",
                            new NoSuchElementException(),
                            fileName);
                }
                try {
                    trailerQueue.add(lineReader.readLine());
                } catch (FileException e) {
                    throw new FileException(
                            "Error occurred by reading processing of a File.",
                            e, fileName);
                }
            }
        }
    }

    /**
     * t@CǏsB<br>
     * 
     * @throws FileException t@CǏŎsꍇB
     */
    public void closeFile() {
        try {
            reader.close();
        } catch (IOException e) {
            throw new FileException("Processing of reader was failed.",
                    e, fileName);
        }
    }

    /**
     * wb_̃f[^擾B<br>
     * f[^ƃgC̎擾̎sۂƊ֌WȂwb_擾邱Ƃ
     * oB
     * 
     * @return header wb_̕񃊃Xg
     */
    public List<String> getHeader() {
        return header;
    }

    /**
     * gC̃f[^擾B<br>
     * gC̃f[^擾ƃf[^̃f[^擾邱Ƃ͏oȂB<br>
     * R̓gC̃f[^擾ɁAf[^̏SXLbv邽߂łB<br>
     * 
     * <b>ӎ</b><br>
     * f[^̃f[^S擾ȑOɃgC擾ȂƁB<br> 
     * 
     * @return gC̕񃊃Xg
     * @throws FileException f[^s擾ŎsꍇB
     */
    public List<String> getTrailer() {
        // gC̃LbVȂꍇɎsB
        if (!readTrailer) {
            String currentData = null;
            // cĂf[^΂sB
            while (hasNext()) {
                try {
                    currentData = lineReader.readLine();
                } catch (FileException e) {
                    throw new FileException(
                            "Processing of lineReader was failed.",
                            e, fileName);
                }
                if (0 < trailerLineCount) {
                    trailerQueue.poll();
                    trailerQueue.add(currentData);
                }
            }
            
            // t@C̃gCLbVɊi[B
            if (0 < trailerLineCount) {
                int trailerQueueLength = trailerQueue.size();
                
                for (int i = 0; i < trailerQueueLength; i++) {
                    trailer.add(trailerQueue.poll());
                }
            }
            readTrailer = true;
        }
        return trailer;
    }

    /**
     * t@Cf[^̃f[^1sǂݎAƂČďoɕԋpB<br>
     * gC݂ꍇ̓gCL[f[^擾Čʕ
     * B<br>
     * ̌LineReader1s̕擾gCL[Ɋi[B<br>
     * gC݂ȂꍇLineReader擾1s̕ʕ
     * B<br>
     * 
     * A1s̃f[^ȂꍇnullԂB
     * 
     * @return f[^̂Ps̕
     * @throws FileException f[^s擾ŎsꍇB
     */
    protected String readLine() {
        
        // ̍sf[^ȂꍇnullԂB
        if (!hasNext()) {
            return null;
        }
        
        // 1s̕擾B
        String currentReadLineString = null;
        try {
            currentReadLineString = lineReader.readLine();
        } catch (FileException e) {
            throw new FileException("Processing of lineReader was failed.",
                    e, fileName);
        }

        // gCL[݂ꍇ́AʂƂăL[̐擪f[^ԂB
        // 擾1s̓̕gCL[ɓB
        if (0 < trailerLineCount) {
            String pollingLineString = trailerQueue.poll();
            trailerQueue.add(currentReadLineString);
            return pollingLineString;
        }

        return currentReadLineString;
    }

    /**
     * f[^̃f[^ǂݔ΂sB<br>
     * 
     * @param skipLines ǂݔ΂sB
     */
    public void skip(int skipLines) {
        for (int i = 0; i < skipLines; i++) {
            if (!hasNext()) {
                throw new FileLineException(
                        "The data which can be acquired doesn't exist.",
                        new NoSuchElementException(), 
                        fileName, currentLineCount + 1);
            }
            readLine();
            currentLineCount++;
        }
    }

    /**
     * ؂蕶擾B
     * 
     * @return s؂蕶B
     */
    protected abstract char getDelimiter();

    /**
     * ͂ݕ擾B
     * 
     * @return ͂ݕB
     */
    protected abstract char getEncloseChar();

    /**
     * f[^̃f[^Pst@CsIuWFNg̃Ame[V̋Lq
     * ]JɕB<br>
     * <code>fileLineString</code><code>null</code>
     * 󕶎̏ꍇ́AvfȂ<code>String</code>zԂ܂B<br>
     * TuNX͂̃\bhI[o[Ch܂B 
     * 
     * @param fileLineString f[^̃f[^Ps
     * @return f[^Ps̕𕪉z
     */
    protected abstract String[] separateColumns(String fileLineString);

    /**
     * ΏۃJɑ΂oCg`FbNsԂB
     * 
     * @param inputFileColumn ΏۃJInputFileColumn
     * @return oCgݒ肳Ă(1oCgȏ)ꍇtrueB
     */
    protected boolean isCheckByte(InputFileColumn inputFileColumn) {
        if (0 < inputFileColumn.bytes()) {
            return true;
        }
        return false;
    }
    
    /**
     * s؂蕶擾B
     * 
     * @return s؂蕶
     */
    protected String getLineFeedChar() {
        return lineFeedChar;
    }

    /**
     * t@CGR[fBO擾B
     * 
     * @return t@CGR[fBO
     */
    protected String getFileEncoding() {
        return fileEncoding;
    }

    /**
     * wb_s擾B
     * 
     * @return wb_s
     */
    protected int getHeaderLineCount() {
        return headerLineCount;
    }

    /**
     * gCs擾B
     * 
     * @return gCs
     */
    protected int getTrailerLineCount() {
        return trailerLineCount;
    }

    /**
     * ݃t@C͏ς݂̃f[^̍s擾B
     * 
     * @return t@C͏ς݂̃f[^̍sB
     */
    public int getCurrentLineCount() {
        return currentLineCount;
    }

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐ擾B
     * 
     * @return t@CsIuWFNgFieldiAnnotationji[ϐ
     */
    protected Field[] getFields() {
        return fields;
    }

    /**
     * t@C擾B
     * @return fileName t@C
     */
    protected String getFileName() {
        return fileName;
    }
}
