/*
 * cexcel - A library to read data from Excel spread sheets
 *
 * Copyright (C) 2007  RIKEN Japan, All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#include "cole.h"
#include "libsl4.h"
#include "cexcel.h"
#include "cexcel_internal.h"

static void _cexcel_context_sst_del_func( void *data, void *arg );

cexcel_context_t *cexcel_context_new( cexcel_file_t * file )
{
  cexcel_context_t *ctx;
  ctx = ( cexcel_context_t * ) malloc( sizeof( cexcel_context_t ) );
  if ( ctx == NULL ) {
    return NULL;
  }
  ctx->file = file;
  ctx->has_global = 0;
  ctx->codepage = 0x16f;        /* ASCII */
  ctx->global_btype = CEXCEL_BTYPE_UNDEF;
  ctx->sheet_btype = CEXCEL_BTYPE_UNDEF;
  ctx->book = cexcel_book_new( sl4_file_get_path( file->file ) );
  ctx->worksheet = NULL;
  if ( ctx->book == NULL ) {
    free( ctx );
    return NULL;
  }
  ctx->sst = cexcel_hash_new(  );
  if ( ctx->sst == NULL ) {
    cexcel_book_delete( ctx->book );
    free( ctx );
    return NULL;
  }
  ctx->sheet = cexcel_hash_new(  );
  if ( ctx->sheet == NULL ) {
    cexcel_book_delete( ctx->book );
    sl4_hash_table_delete( ctx->sst, _cexcel_context_sst_del_func, NULL );
    free( ctx );
    return NULL;
  }
  return ctx;
}

int cexcel_context_delete( cexcel_context_t * ctx )
{
  /* cexcel_book_delete( ctx->book ); */
  sl4_hash_table_delete( ctx->sheet, NULL, NULL );
  sl4_hash_table_delete( ctx->sst, _cexcel_context_sst_del_func, NULL );
  free( ctx );
  return 0;
}

int cexcel_context_add_sst( cexcel_context_t * ctx, sl4_string_t *str )
{
  CEXCEL_DWORD nextkey = ( CEXCEL_DWORD ) ctx->sst->nkeys;
  return sl4_hash_table_insert( ctx->sst, &nextkey, str );
}

sl4_string_t *cexcel_context_get_sst( cexcel_context_t * ctx,
                                      CEXCEL_DWORD key )
{
  return ( sl4_string_t * ) sl4_hash_table_lookup( ctx->sst, &key );
}

int cexcel_context_add_sheet( cexcel_context_t * ctx, CEXCEL_DWORD offset,
                              const char *str )
{
  cexcel_sheet_t *sheet;
  sheet = cexcel_book_add_sheet( ctx->book, str );
  if ( sheet == NULL ) {
    return -1;
  }
  if ( sl4_hash_table_insert( ctx->sheet, &offset, sheet ) ) {
    return -1;
  }
  return 0;
}

int cexcel_context_get_sheet( cexcel_context_t * ctx, CEXCEL_DWORD offset )
{
  cexcel_sheet_t *sheet;
  sheet =
    ( cexcel_sheet_t * ) sl4_hash_table_lookup( ctx->sheet, &offset );
  if ( sheet == NULL ) {
    return -1;
  }
  ctx->worksheet = sheet;
  return 0;
}

static void _cexcel_context_sst_del_func( void *data, void *arg )
{
  sl4_string_t *str = ( sl4_string_t * ) data;
  sl4_string_delete( str );
}
