/*
  This file is part of f-palette-CANopen

  2011/11/08
  Copyright (C): Future Robotics Technology Center, Chiba institute of technology
  TI F2803x Port: Masaharu Shimizu@Future Robotics Technology Center, Chiba institute of technology
  
  This work is based on
   -    Canfestival
        http://sourceforge.net/projects/canfestival/
  Copyright (C): Edouard TISSERANT and Francis DUPIN

See COPYING file for copyrights details.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "DSP2803x_Device.h"     // DSP2803x Headerfile Include File
#include "DSP2803x_Examples.h"   // DSP2803x Examples Include File
#include "can_TI_F2803x.h"
#include "canfestival.h"
#include "ObjDict.h"

extern void InitECana(void);

#define TXMBOXREGS	ECanaMboxes.MBOX0
#define WAIT_US		(10)


unsigned char canInit(unsigned int bitrate)
/******************************************************************************
Initialize the hardware to receive CAN messages and start the timer for the
CANopen stack.
INPUT	bitrate		bitrate in kilobit
OUTPUT	1 if successful	
******************************************************************************/
{
	struct ECAN_REGS ECanaShadow; 
	volatile struct MBOX	*p_mbox; 
	Uint16		i,uiNodeID;
	Uint32		u32canBitrateReg;
	

	uiNodeID =  getNodeId(&ObjDict_Data);/* call after "setNodeId" */
	InitECana(); /* found in DSP2803x_ECan.c */
	/* set bitrate register */
	switch(bitrate){/* CanClocks = 30MHz :F20803x */
		case 10 :	u32canBitrateReg =   0x00C70052;break; /*  10K bit/s */
		case 20 :	u32canBitrateReg =   0x00630052;break; /*  20K bit/s */
		case 50 :	u32canBitrateReg =   0x00270052;break; /*  50K bit/s */
		case 100 :	u32canBitrateReg =   0x00130052;break; /* 100K bit/s */
		case 125 :	u32canBitrateReg =   0x000F0052;break; /* 125K bit/s */
		case 250 :	u32canBitrateReg =   0x00070052;break; /* 250K bit/s */
		case 500 :	u32canBitrateReg =   0x00030052;break; /* 500K bit/s */
		case 1000 :	u32canBitrateReg =   0x00020031;break; /*   1M bit/s */
		default:	u32canBitrateReg =   0x00030052;break; /* 500K bit/s */
	}
   EALLOW;  // This is needed to write to EALLOW protected registers

/* Configure bit timing parameters for eCANA*/
    ECanaShadow.CANMC.all = ECanaRegs.CANMC.all;
    ECanaShadow.CANMC.bit.CCR = 1 ;            // Set CCR = 1
    ECanaRegs.CANMC.all = ECanaShadow.CANMC.all;
    // Wait until the CPU has been granted permission to change the configuration registers
    do{
      ECanaShadow.CANES.all = ECanaRegs.CANES.all;
    } while(ECanaShadow.CANES.bit.CCE != 1 );       // Wait for CCE bit to be set..

    ECanaRegs.CANBTC.all = u32canBitrateReg;

    ECanaShadow.CANMC.all = ECanaRegs.CANMC.all;
    ECanaShadow.CANMC.bit.CCR = 0 ;            // Set CCR = 0
    ECanaRegs.CANMC.all = ECanaShadow.CANMC.all;
    // Wait until the CPU no longer has permission to change the configuration registers
    do{
      ECanaShadow.CANES.all = ECanaRegs.CANES.all;
    } while(ECanaShadow.CANES.bit.CCE != 0 );       // Wait for CCE bit to be  cleared..
   EDIS;    // This is needed to disable write to EALLOW protected registers


/* setp CAN interrupt */
   EALLOW;  // This is needed to write to EALLOW protected registers
   PieVectTable.ECAN0INTA = &ecanaRx_isr;
   EDIS;    // This is needed to disable write to EALLOW protected registers

// Enable CPU INT9 which is connected to ECAN0A INT:
   IER |= M_INT9;
// Enable eCAN INTn in the PIE: Group 9 interrupt 5
   PieCtrlRegs.PIEIER9.bit.INTx5 = 1;


	/* initialize MSGIDL register --------------------------------------------------------- */
	for(i=0;i<32;i++){
		p_mbox = &(ECanaMboxes.MBOX0) + i;
		(*p_mbox).MSGID.all = 0;	/*	initialize all 0 */
	}
	
	ECanaRegs.CANMC.bit.MBNR = 6;   /* Chande Data Request, set to don't use mbox No. */
	TXMBOXREGS.MSGID.bit.STDMSGID = 0x789; /* Set cob id Dummy*/
	TXMBOXREGS.MSGCTRL.bit.DLC = 8;

	
	ECanaMboxes.MBOX16.MSGID.bit.AME = 1;			/* local acceptance mask used (for all rx mboxes) */
		/* Define Local Acceptance Masks for rx mailboxes */
	ECanaLAMRegs.LAM16.bit.LAMI = 0;
	ECanaLAMRegs.LAM16.bit.LAM_L = 0;
	ECanaLAMRegs.LAM16.bit.LAM_H = 0x600;	/* bits 7 and 8 from 11-bit identifier are ignored STD_ID:0x180 */	
							/* valid for ID = 0x000 (NMT), 0x080 (sync) and 0x100 (timestamp) */
	ECanaMboxes.MBOX17.MSGID.bit.AME = 1;			/* local acceptance mask used (for all rx mboxes) */
	ECanaMboxes.MBOX17.MSGID.bit.STDMSGID = uiNodeID; /* set NodeID */
		/* Define Local Acceptance Masks for rx mailboxes */
	ECanaLAMRegs.LAM17.bit.LAMI = 0;
	ECanaLAMRegs.LAM17.bit.LAM_L = 0;
	ECanaLAMRegs.LAM17.bit.LAM_H = 0x1E00;	/* only messages for this node STD_ID:0x780 */
	
	/* ----- Tx & Rx Configuration -------------------------------------------------------- */
	ECanaRegs.CANMD.all = 0xFFFFFC00;
	/* MB 0-9 as 0: Tx, MB 10-31 as 1: Rx */

	/* ----- Enable all mailboxes, enable: 1, disable: 0 ---------------------------------- */
	/* this register is only write accessible by 32-bits() accesses */
	ECanaRegs.CANME.all = 0x00030001; /* Enable MBOX0,MBOX16,MBOX17 */

	/* Since this write is to the entire register (instead of a bit field)
	   a shadow register is not required. */

	EALLOW;
	ECanaRegs.CANGIM.all = 0;
	ECanaRegs.CANGIM.bit.I0EN = 1;//Interrupt 0 enable
	
	/* ----- MB Interrupt Mask Register --------------------------------------------------- */
	/* this register is only write accessible by 32-bits() accesses */
	ECanaRegs.CANMIM.all = 0x00030000;	/* Rx MBOX16 &MBOX17 arrow Interrupt */
	
	/* Configure the eCAN for self test mode Enable the enhanced features of the eCAN. */
	ECanaShadow.CANMC.all = ECanaRegs.CANMC.all;
	ECanaShadow.CANMC.bit.STM = 0;    /* 1:Configure CAN for self-test mode  */
	ECanaShadow.CANMC.bit.SCB = 1;    /* 0: SCC, 1: eCAN mode (reqd to access 32 mailboxes)*/
	ECanaShadow.CANMC.bit.CDR = 1;	   /* Chande Data Request set */
	ECanaShadow.CANMC.bit.ABO = 1;/* Activate Auto Bus On (to recover after a "buss-off" state) */
	ECanaRegs.CANMC.all = ECanaShadow.CANMC.all;
	EDIS;

  return 1;
}

unsigned char canSend(CAN_PORT notused, Message *m)
/******************************************************************************
The driver send a CAN message passed from the CANopen stack
INPUT	CAN_PORT is not used (only 1 avaiable)
	Message *m pointer to message to send
OUTPUT	1 if  hardware -> CAN frame
******************************************************************************/
{
	union CANME_REG ECanaShadow_CANME;

	while(ECanaRegs.CANTRS.bit.TRS0){/* ECanaRegs.CANTRS.bit.TRS0 == 1 */
	DELAY_US(WAIT_US);	/* wait 10us */
	}

	/* disable mailbox for changing message identifier */
	ECanaShadow_CANME.all = ECanaRegs.CANME.all;	
	ECanaShadow_CANME.bit.ME0 = 0;		/* disable the mailbox that is going to be used to transmit */
	ECanaRegs.CANME.all = ECanaShadow_CANME.all; 

	TXMBOXREGS.MSGID.all= 0; /* reset MSGID */
	TXMBOXREGS.MSGID.bit.STDMSGID = (m->cob_id & 0x07FF); /* Set cob id */
	TXMBOXREGS.MSGCTRL.bit.RTR = m->rtr;
	TXMBOXREGS.MSGCTRL.bit.DLC = m->len;/* Set DLC */

	switch(m->len){/* copy to mail box */
		case 8:TXMBOXREGS.MDH.byte.BYTE7 = m->data[7];
		case 7:TXMBOXREGS.MDH.byte.BYTE6 = m->data[6];
		case 6:TXMBOXREGS.MDH.byte.BYTE5 = m->data[5];
		case 5:TXMBOXREGS.MDH.byte.BYTE4 = m->data[4];
		case 4:TXMBOXREGS.MDL.byte.BYTE3 = m->data[3];
		case 3:TXMBOXREGS.MDL.byte.BYTE2 = m->data[2];
		case 2:TXMBOXREGS.MDL.byte.BYTE1 = m->data[1];
		case 1:TXMBOXREGS.MDL.byte.BYTE0 = m->data[0];
		case 0:break;
		default:break;
	}
	/* Enable mailbox  */
	ECanaShadow_CANME.all = ECanaRegs.CANME.all;	
	ECanaShadow_CANME.bit.ME0 = 1;		/* Enable the mailbox that is going to be used to transmit */
	ECanaRegs.CANME.all = ECanaShadow_CANME.all; 

	if(ECanaRegs.CANTRS.bit.TRS0 == 0){
		if(ECanaRegs.CANTA.bit.TA0 == 1){ECanaRegs.CANTA.bit.TA0 = 1;}
		ECanaRegs.CANTRS.bit.TRS0 = 1;  
		return 1;	// succesful
	}
	else{
		return 0;		// no message received
	}
}

unsigned char canReceive(Message *m)
/******************************************************************************
The driver pass a received CAN message to the stack
INPUT	Message *m pointer to received CAN message
OUTPUT	1 if a message received
******************************************************************************/
{/* Read MBOX16 & MBOX17 */
	volatile struct MBOX	*p_mbox; 
	Uint16		uiMboxNo;
	
	uiMboxNo = 0x00;
	/* initialize MSGIDL register --------------------------------------------------------- */
	
	if(ECanaRegs.CANRMP.bit.RMP16 == 1){
		ECanaRegs.CANRMP.bit.RMP16 = 1;  // clear RMP by writing '1' 
		if(ECanaRegs.CANRFP.bit.RFP16 == 1){ECanaRegs.CANRFP.bit.RFP16 = 1; }
		else{
			uiMboxNo = 16;
	    }
	}
	if(ECanaRegs.CANRMP.bit.RMP17 == 1){
		ECanaRegs.CANRMP.bit.RMP17 = 1;  // clear RMP by writing '1' 
		if(ECanaRegs.CANRFP.bit.RFP17 == 1){ECanaRegs.CANRFP.bit.RFP17 = 1; }
		else{
	    	uiMboxNo = 17;
		}
	}
	
	if(uiMboxNo){/* uiMboxNo != 0 */
		p_mbox = &(ECanaMboxes.MBOX0) + uiMboxNo;
		m->cob_id 	= (*p_mbox).MSGID.bit.STDMSGID; /* Set cob id */
		m->rtr		= (*p_mbox).MSGCTRL.bit.RTR;
		m->len		= (*p_mbox).MSGCTRL.bit.DLC;/* Set DLC */
		switch(m->len){/* copy from mail box */
			case 8:m->data[7] = (*p_mbox).MDH.byte.BYTE7;
			case 7:m->data[6] = (*p_mbox).MDH.byte.BYTE6;
			case 6:m->data[5] = (*p_mbox).MDH.byte.BYTE5;
			case 5:m->data[4] = (*p_mbox).MDH.byte.BYTE4;
			case 4:m->data[3] = (*p_mbox).MDL.byte.BYTE3;
			case 3:m->data[2] = (*p_mbox).MDL.byte.BYTE2;
			case 2:m->data[1] = (*p_mbox).MDL.byte.BYTE1;
			case 1:m->data[0] = (*p_mbox).MDL.byte.BYTE0;
			case 0:break;
			default:break;
		}
		return 1;	// succesful
	}
	else{
		return 0;		// no message received
	}
}/* end of canReceive */

/***************************************************************************/
unsigned char canChangeBaudRate_driver( CAN_HANDLE fd, char* baud)
{

	return 0;
}

interrupt void ecanaRx_isr(void){
Message m_main = Message_Initializer;

	if(canReceive(&m_main)){			/* a message reveived */
		canDispatch(&ObjDict_Data, &m_main);         /* process it			*/
	}

	/* Acknowledge this interrupt to receive more interrupts from group 9 */
	PieCtrlRegs.PIEACK.all = PIEACK_GROUP9;
}/* end of ecanaRx_isr */

/* end of file can_TI_F2803x.c */


