package org.maachang.dbm.engine ;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

/**
 * Value管理オブジェクト.
 * 
 * @version 2008/01/17
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MValue {
    
    /**
     * セクター管理オブジェクト.
     */
    private MSctArray sectors = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private MValue() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を指定してオブジェクトを生成します.
     * <BR>
     * @param sectors 対象のセクターオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public MValue( MSctArray sectors ) throws Exception {
        if( sectors == null || sectors.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.sectors = sectors ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        synchronized( sync ) {
            if( sectors != null ) {
                sectors.destroy() ;
            }
            sectors = null ;
        }
    }
    
    /**
     * オブジェクト更新.
     * <BR><BR>
     * オブジェクトを更新します.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            sectors.flush() ;
        }
    }
    
    /**
     * データ追加チェック処理.
     * <BR><BR>
     * データを追加する場合にデータ入力可能かチェックします.
     * <BR>
     * @param length 追加対象のデータ長を設定します.
     * @return boolean [true]の場合、データセット可能です.
     * @exception Exception 例外.
     */
    protected boolean putCheck( int length ) throws Exception {
        if( length <= 0 ) {
            return false ;
        }
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            int rem = length - sectors.useSector() ;
            if( rem <= -1 ) {
                return true ;
            }
            int sec = ( ( rem & 0xffff0000 ) >> 16 ) + ( ( ( rem & 0x0000ffff ) != 0 ) ? 1 : 0 ) ;
            if( sec + sectors.size() >= MKey.MAX_VALUE_FILE_SIZE ) {
                return false ;
            }
            long all = MSector.oneSectorFileSize() * sec ;
            long sz = new File( sectors.getDirectory() ).getFreeSpace() ;
            if( all >= sz ) {
                return false ;
            }
        }
        return true ;
    }
    
    /**
     * 新しいデータを追加.
     * <BR><BR>
     * 新しいデータを追加します.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @return int[] 追加された先頭の[0:ファイルNo,1:ファイルポジション,
     *                                2:最終ファイルNo,3:最終ファイルポジション]が返されます.
     * @exception Exception 例外.
     */
    public int[] put( byte[] binary ) throws Exception {
        if( binary == null || binary.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        // 書き込みセクターがない場合は、作成.
        int binLen = binary.length ;
        int secLen = MSctArray.sectorLength( binLen ) ;
        if( secLen * 2 >= ary.useSector() ) {
            ary.add() ;
        }
        MSector sector = null ;
        ArrayList<ValueSector> cache = new ArrayList<ValueSector>() ;
        int secCnt = secLen ;
        // ポジション予約.
        for( int i = 0 ; i < secLen ; i ++,secCnt -- ) {
            //if( sector == null ) {
            //    sector = getUseObject( ary,secCnt ) ;
            //}
            ValueSector value = new ValueSector() ;
            sector = addSector( ary,sector,value,secCnt ) ;
            cache.add( value ) ;
            ary.removeUseSector() ;
        }
        // 予約ポジションに対して、データ出力.
        int[] ret = new int[ 4 ] ;
        int len = cache.size() ;
        int offset = 0 ;
        ValueSector bef = null ;
        for( int i = 0 ; i < len ; i ++ ) {
            ValueSector now = cache.get( i ) ;
            // 開始書き込み位置を取得.
            if( bef == null ) {
                ret[ 0 ] = now.getFileNo() ;
                ret[ 1 ] = now.getPosition() ;
            }
            // 最終書き込み位置を取得.
            ret[ 2 ] = now.getFileNo() ;
            ret[ 3 ] = now.getPosition() ;
            int oneLen = ( binLen >= MSector.ONE_DATA ) ? MSector.ONE_DATA : binLen ;
            now.write( binary,0,offset,oneLen ) ;
            offset += oneLen ;
            binLen -= oneLen ;
            if( bef != null ) {
                now.setBeforeFileNo( bef.getFileNo() ) ;
                now.setBeforeFilePos( bef.getPosition() ) ;
                bef.setNextFileNo( now.getFileNo() ) ;
                bef.setNextFilePos( now.getPosition() ) ;
                sector = ary.get( bef.getFileNo() ) ;
                sector.put( bef.getPosition(),bef ) ;
            }
            bef = now ;
            now = null ;
        }
        // 一番最後の領域を出力.
        sector = ary.get( bef.getFileNo() ) ;
        sector.put( bef.getPosition(),bef ) ;
        return ret ;
    }
    
    /**
     * 指定データを削除.
     * <BR><BR>
     * 指定されたデータを削除します.
     * <BR>
     * @param fileNo 対象のファイルNoを設定します.
     * @param filePos 対象のファイル項番を設定します.
     * @exception Exception 例外.
     */
    public void remove( int fileNo,int filePos ) throws Exception {
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        MSector sector = null ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            ValueSector val = sector.remove( filePos ) ;
            if( val == null || val.getLength() <= 0 ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")は不正な条件です" ) ;
            }
            ary.addUseSector() ;
            fileNo = val.getNextFileNo() ;
            filePos = val.getNextFilePos() ;
            if( fileNo <= -1 || filePos <= -1 ) {
                break ;
            }
            if( val.getFileNo() != fileNo ) {
                sector = null ;
            }
        }
    }
    
    /**
     * 指定データを取得.
     * <BR><BR>
     * 指定されたデータを取得します.
     * <BR>
     * @param fileNo 対象のファイルNoを設定します.
     * @param filePos 対象のファイル項番を設定します.
     * @param length 対象のデータ長を設定します.
     * @return byte[] 対象のデータが返されます.
     * @exception Exception 例外.
     */
    public byte[] get( int fileNo,int filePos,int length ) throws Exception {
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        MSector sector = null ;
        byte[] ret = new byte[ length ] ;
        int pnt = 0 ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            ValueSector val = sector.get( filePos ) ;
            if( val == null || val.getLength() <= 0 ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")は不正な条件です" ) ;
            }
            byte[] b = val.read() ;
            if( b == null ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")には、データが存在しません" ) ;
            }
            System.arraycopy( b,0,ret,pnt,b.length ) ;
            pnt += b.length ;
            fileNo = val.getNextFileNo() ;
            filePos = val.getNextFilePos() ;
            if( fileNo <= -1 || filePos <= -1 ) {
                break ;
            }
            if( val.getFileNo() != fileNo ) {
                sector = null ;
            }
        }
        return ret ;
    }
    
    /**
     * 指定データ長を取得.
     * <BR><BR>
     * 指定されたデータ長を取得します.
     * <BR>
     * @param fileNo 対象のファイルNoを設定します.
     * @param filePos 対象のファイル項番を設定します.
     * @return int 対象のデータ長が返されます.
     * @exception Exception 例外.
     */
    public int getLength( int fileNo,int filePos ) throws Exception {
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        MSector sector = null ;
        int ret = 0 ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            ValueSector val = sector.getHeader( filePos ) ;
            if( val == null || val.getLength() <= 0 ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")は不正な条件です" ) ;
            }
            ret += val.getLength() ;
            fileNo = val.getNextFileNo() ;
            filePos = val.getNextFilePos() ;
            if( fileNo <= -1 || filePos <= -1 ) {
                break ;
            }
            if( val.getFileNo() != fileNo ) {
                sector = null ;
            }
        }
        return ret ;
    }
    
    /**
     * 最後のセクタにデータを追加.
     * <BR><BR>
     * 最後のセクタにデータを追加します.
     * <BR>
     * @param ch 対象のキー子要素を設定します.
     * @param binary 対象のバイナリを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @exception Exception 例外.
     */
    public void addLast( MKeyChild ch,byte[] binary,int off,int length )
        throws Exception {
        if( ch == null || binary == null || binary.length <= 0 || off < 0 ||
            length <= 0 || off + length > binary.length ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        int fileNo = ch.getLastValueFileNo() ;
        int filePos = ch.getLastValueFilePos() ;
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        MSector sector = null ;
        ValueSector nextVal = null ;
        int dataLen = 0 ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            ValueSector val = null ;
            if( nextVal == null ) {
                val = sector.get( filePos ) ;
            }
            else {
                val = nextVal ;
                nextVal = null ;
            }
            int sectorLen = val.getLength() ;
            int cnt = 0 ;
            byte[] b = val.getData() ;
            for( int i = sectorLen ; i < MSector.ONE_DATA ; i ++ ) {
                if( dataLen >= length ) {
                    break ;
                }
                b[ i ] = binary[ off+dataLen ] ;
                dataLen ++ ;
                cnt ++ ;
            }
            if( cnt > 0 ) {
                val.setLength( sectorLen + cnt ) ;
                if( dataLen >= length ) {
                    sector.put( filePos,val ) ;
                    break ;
                }
            }
            else if( dataLen >= length ) {
                break ;
            }
            nextVal = new ValueSector() ;
            MSector befSector = sector ;
            sector = addSector( ary,sector,nextVal,1 ) ;
            nextVal.setBeforeFileNo( val.getFileNo() ) ;
            nextVal.setBeforeFilePos( val.getPosition() ) ;
            val.setNextFileNo( nextVal.getFileNo() ) ;
            val.setNextFilePos( nextVal.getPosition() ) ;
            befSector.put( filePos,val ) ;
            befSector = null ;
            fileNo = nextVal.getFileNo() ;
            filePos = nextVal.getPosition() ;
            ch.setLastValueFileNo( fileNo ) ;
            ch.setLastValueFilePos( filePos ) ;
        }
        ch.setValueLength( ch.getValueLength() + length ) ;
    }
    
    /**
     * 指定データを書き込む.
     * <BR><BR>
     * 指定位置にデータを書き込みます.
     * <BR>
     * @param ch 対象のキー子要素を設定します.
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 書き込まれたデータ長が返されます.
     * @exception Exception 例外.
     */
    public int write( MKeyChild ch,byte[] binary,int pos,int off,int length )
        throws Exception {
        if( ch == null || binary == null || binary.length <= 0 || off < 0 || length <= 0 ||
            off + length > binary.length || pos + length > ch.getValueLength() ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        int fileNo = ch.getValueFileNo() ;
        int filePos = ch.getValueFilePos() ;
        MSector sector = null ;
        int pnt = 0 ;
        int dataLen = 0 ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            boolean bodyFlag = false ;
            ValueSector val = null ;
            if( pnt <= pos ) {
                val = sector.get( filePos ) ;
                bodyFlag = true ;
            }
            else {
                val = sector.getHeader( filePos ) ;
            }
            if( val == null || val.getLength() <= 0 ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")は不正な条件です" ) ;
            }
            int len = val.getLength() ;
            if( pnt+len >= pos ) {
                if( bodyFlag == false ) {
                    val = sector.get( filePos ) ;
                }
                int startPos = pos - pnt ;
                if( startPos <= 0 ) {
                    startPos = 0 ;
                }
                byte[] b = val.getData() ;
                for( int i = startPos ; i < len ; i ++ ) {
                    if( dataLen >= length ) {
                        sector.put( filePos,val ) ;
                        return length ;
                    }
                    b[ i ] = binary[ dataLen+off ] ;
                    dataLen ++ ;
                }
                sector.put( filePos,val ) ;
                if( len != MSector.ONE_DATA ) {
                    return dataLen ;
                }
            }
            pnt += len ;
            fileNo = val.getNextFileNo() ;
            filePos = val.getNextFilePos() ;
            if( fileNo <= -1 || filePos <= -1 ) {
                return dataLen ;
            }
            if( val.getFileNo() != fileNo ) {
                sector = null ;
            }
        }
    }
    
    /**
     * 指定データを取得.
     * <BR><BR>
     * @param ch 対象のキー子要素を設定します.
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 取得されたデータ長が返されます.
     * @exception Exception 例外.
     */
    public int read( MKeyChild ch,byte[] binary,int pos,int off,int length )
        throws Exception {
        if( ch == null || binary == null || binary.length <= 0 || off < 0 || length <= 0 ||
            off + length > binary.length || pos + length > ch.getValueLength() ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        int fileNo = ch.getValueFileNo() ;
        int filePos = ch.getValueFilePos() ;
        MSector sector = null ;
        int pnt = 0 ;
        int dataLen = 0 ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            boolean bodyFlag = false ;
            ValueSector val = null ;
            if( pnt <= pos ) {
                val = sector.get( filePos ) ;
                bodyFlag = true ;
            }
            else {
                val = sector.getHeader( filePos ) ;
            }
            if( val == null || val.getLength() <= 0 ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")は不正な条件です" ) ;
            }
            int len = val.getLength() ;
            if( pnt+len >= pos ) {
                if( bodyFlag == false ) {
                    val = sector.get( filePos ) ;
                }
                int startPos = pos - pnt ;
                if( startPos <= 0 ) {
                    startPos = 0 ;
                }
                byte[] b = val.getData() ;
                for( int i = startPos ; i < len ; i ++ ) {
                    if( dataLen >= length ) {
                        return length ;
                    }
                    binary[ dataLen+off ] = b[ i ] ;
                    dataLen ++ ;
                }
                if( len != MSector.ONE_DATA ) {
                    return dataLen ;
                }
            }
            pnt += len ;
            fileNo = val.getNextFileNo() ;
            filePos = val.getNextFilePos() ;
            if( fileNo <= -1 || filePos <= -1 ) {
                return dataLen ;
            }
            if( val.getFileNo() != fileNo ) {
                sector = null ;
            }
        }
    }
    
    /**
     * セクター管理オブジェクトを取得.
     * <BR><BR>
     * セクター管理オブジェクトを取得します.
     * <BR>
     * @return MSctArray セクター管理オブジェクトが返されます.
     */
    public MSctArray getArray() {
        MSctArray ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = null ;
            }
            else {
                ret = sectors ;
            }
        }
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret = false ;
        synchronized( sync ) {
            ret = check() ;
        }
        return ret ;
    }
    
    private boolean check() {
        if( sectors == null ) {
            return false ;
        }
        return true ;
    }
    
    private static final MSector getUseObject( MSctArray ary,int secLen ) throws Exception {
        MSector ret = null ;
        for( ;; ) {
            ret = ary.getMaxUseObject( secLen ) ;
            if( ret == null ) {
                ary.add() ;
            }
            else {
                break ;
            }
        }
        return ret ;
    }
    
    private static final MSector addSector( MSctArray ary,MSector sector,ValueSector value,int secLen )
        throws Exception {
        for( ;; ) {
            if( sector != null && sector.add( value ) == true ) {
                break ;
            }
            sector = getUseObject( ary,secLen ) ;
        }
        return sector ;
    }
    
}
