package org.maachang.dbm.service.client ;

import java.net.InetAddress;

import org.maachang.dbm.MDbm;
import org.maachang.dbm.MDbmOp;
import org.maachang.dbm.service.ProtocolDef;

/**
 * MDbmドライバーマネージャ.
 * <BR><BR>
 * MDbmサーバと接続するためのドライバーマネージャです.<BR>
 * このオブジェクトは、動作しているMDBMサーバと接続し操作する、
 * MDbmオブジェクト及びMDbmOpオブジェクトを生成するためのマネージャです.<BR>
 * 利用方法は、下記の通りです.<BR>
 * <BR>
 * <div style="border:1px solid gray; padding:5px;">
 * <pre>
 * <font color="green">// MDBサーバの接続先ホスト名.</font>
 * InetAddress addr = InetAddress.getByName( "server" ) ;
 * 
 * <font color="green">// MDBMオブジェクトを取得.</font>
 * MDbm dbm = MDbmDriverManager.getConnection( addr ) ;
 *
 * <font color="green">// MDBM操作オブジェクトを取得.</font>
 * MDbmOp op = MDbmDriverManager.getConnectionOp( addr ) ;
 * 
 * </pre>
 * </div>
 * <BR>
 * 
 * @version 2008/01/20
 * @author masahito suzuki
 * @since MaachangDBM 1.03
 */
public class MDbmDriverManager {
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( String addr )
        throws Exception {
        return getConnection( addr,-1 ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( InetAddress addr )
        throws Exception {
        return getConnection( addr,-1 ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( String addr,int port )
        throws Exception {
        if( addr == null ) {
            addr = "127.0.0.1" ;
        }
        return MDbmDriverManager.getInstance().connection( InetAddress.getByName( addr ),port ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return Mdbm MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbm getConnection( InetAddress addr,int port )
        throws Exception {
        return MDbmDriverManager.getInstance().connection( addr,port ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( String addr )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,-1 ) ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( InetAddress addr )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,-1 ) ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( String addr,int port )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,port ) ) ;
    }
    
    /**
     * MDBMコネクションを取得.
     * <BR><BR>
     * MDBMコネクションを取得します.
     * <BR>
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @return MdbmOp MDBMコネクションが返されます.
     * @exception Exception 例外.
     */
    public static final MDbmOp getConnectionOp( InetAddress addr,int port )
        throws Exception {
        return new MDbmOpConnection( getConnection( addr,port ) ) ;
    }
    
    /**
     * 監視スレッド数を設定.
     * <BR><BR>
     * 監視スレッド数を設定します.<BR>
     * また、このメソッドは、コネクションを１度も生成しないときに呼び出さないと
     * 意味を持ちません.
     * <BR>
     * @param length 監視スレッド数を設定します.
     */
    public synchronized void setThread( int length ) {
        if( length <= THREAD_LENGTH ) {
            length = THREAD_LENGTH ;
        }
        if( length >= 25 ) {
            length = 25 ;
        }
        this.threadLength = length ;
    }
    
    private static final int THREAD_LENGTH = 3 ;
    private static final MDbmDriverManager SNGL = new MDbmDriverManager() ;
    
    private MDbmDriverManager() {
        
    }
    
    private static final MDbmDriverManager getInstance() {
        return SNGL ;
    }
    
    private int threadLength = THREAD_LENGTH ;
    private ClientConnectManager man = null ;
    private ClientHeartBeatThread[] heartBeat = null ;
    private final Object sync = new Object() ;
    
    private synchronized MDbm connection( InetAddress addr,int port ) throws Exception {
        if( port < 0 || port > 65535 ) {
            port = ProtocolDef.BIND_PORT ;
        }
        MDbmConnection ret = null ;
        synchronized( sync ) {
            ret = new MDbmConnection( sync,addr,port ) ;
            if( man == null ) {
                man = new ClientConnectManager() ;
                heartBeat = new ClientHeartBeatThread[ this.threadLength ] ;
                for( int i = 0 ; i < this.threadLength ; i ++ ) {
                    heartBeat[ i ] = new ClientHeartBeatThread( sync,man ) ;
                }
            }
            man.put( ret.clientConnection().getSession() ) ;
        }
        return ret ;
    }
}

