package org.maachang.proxy.engine.conf;

import org.maachang.util.conf.Config;

/**
 * サーバオプション.
 * 
 * @version 2008/03/25
 * @author masahito suzuki
 * @since MaachangProxy 1.00
 */
public class ServerOption {
    /** 最大コネクション数. */
    private static final int MAX_CONNECT_LENGTH = 999 ;
    /** デフォルト最大コネクション数. */
    private static final int DEF_CONNECT_LENGTH = 128 ;
    /** 最大プーリング数. */
    private static final int MAX_POOL_LENGTH = 250 ;
    /** デフォルトプーリング数. */
    private static final int DEF_POOL_LENGTH = 15 ;
    /** デフォルトサーバポート番号. */
    private static final int DEF_PORT = 80 ;
    /** サーバセクション名 */
    private static final String SERVER_SECTION = "server" ;
    
    /** 最大コネクション数 */
    private int connectLength = -1 ;
    
    /** 最大プーリング数 */
    private int poolLength = -1 ;
    
    /** サーバポート番号. */
    private int port = -1 ;
    
    /** バインドアドレス. */
    private String bindAddress = null ;
    
    private ServerOption() {
        
    }
    
    /**
     * コンフィグ情報から、ServerOptionを生成.
     * <BR>
     * @param conf 対象のコンフィグを設定します.
     * @return ServerOption ServerOptionが返されます.
     */
    public static final ServerOption create( Config conf ) {
        int connectLength = conf.getInt( SERVER_SECTION,"max-connect",0 ) ;
        int poolLength = conf.getInt( SERVER_SECTION,"max-pool",0 ) ;
        int port = conf.getInt( SERVER_SECTION,"server-port",0 ) ;
        String bindAddress = conf.get( SERVER_SECTION,"server-address",0 ) ;
        if( bindAddress == null || ( bindAddress = bindAddress.trim() ).length() <= 0 ) {
            bindAddress = null ;
        }
        if( connectLength <= -1 ) {
            connectLength = DEF_CONNECT_LENGTH ;
        }
        if( connectLength >= MAX_CONNECT_LENGTH ) {
            connectLength = MAX_CONNECT_LENGTH ;
        }
        if( poolLength <= -1 ) {
            poolLength = DEF_POOL_LENGTH ;
        }
        if( poolLength >= MAX_POOL_LENGTH ) {
            poolLength = MAX_POOL_LENGTH ;
        }
        if( port <= -1 || port > 65535 ) {
            port = DEF_PORT ;
        }
        if( poolLength * 8 >= connectLength ) {
            poolLength = connectLength / 8 ;
        }
        if( poolLength <= 0 ) {
            poolLength = 1 ;
        }
        ServerOption ret = new ServerOption() ;
        ret.connectLength = connectLength ;
        ret.poolLength = poolLength ;
        ret.port = port ;
        ret.bindAddress = bindAddress ;
        return ret ;
    }
    
    /**
     * bindAddress を取得.
     * <BR><BR>
     * @return bindAddress が返されます.
     */
    public String getBindAddress() {
        return bindAddress;
    }
    
    /**
     * connectLength を取得.
     * <BR><BR>
     * @return connectLength が返されます.
     */
    public int getConnectLength() {
        return connectLength;
    }
    
    /**
     * poolLength を取得.
     * <BR><BR>
     * @return poolLength が返されます.
     */
    public int getPoolLength() {
        return poolLength;
    }
    
    /**
     * port を取得.
     * <BR><BR>
     * @return port が返されます.
     */
    public int getPort() {
        return port;
    }
}
