#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - "do"/"redo" command operation -
# Copyright (C) 2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Operation of redo command before the database reconstruction =============
command_do_do__command_all_exec_command_specific_preconfiguration ()
{
	program_deregister_stage_complete FORGET::PREPARATION_INSPECT_MASTER
	program_deregister_stage_complete ALL_COMPLETE
}

# ============= Operation of redo command before the database reconstruction =============
command_do_redo__command_all_exec_command_specific_preconfiguration ()
{
	program_deregister_stage_complete REDO_INIT
	if [ $opt_reset_targets = yes ]
	then
		program_deregister_stage_complete DETERMINE_SPECIFIED_TARGETS
		program_deregister_stage_complete PREPARE_FOR_INSPECT_ALL_DEPENDENCIES
		rm -rf "${DBDIR}/targets"
	fi
	command_do_do__command_all_exec_command_specific_preconfiguration
}

# ============= Meta process for redo =============
command_do_meta_process_for_redo ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=''
	_program_exec_and_record_completion__operation ()
	{
		rm -f "${DBDIR}/new_success_in_current_run"
		[ "x$COMMAND_MODE" = xredo ] || return 0
		message_echo "[REDO mode]"
		message_echo
	}
	program_exec_and_record_completion REDO_INIT
}

# ============= Determine specified targets =============
command_do_determine_specified_targets ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=''
	_program_exec_and_record_completion__operation ()
	{
		local tag level dbsuffix
		message_section_title "Determining specified targets"
		cat "${DBDIR}/stage.loop_list/target_itself.specified" \
			"${DBDIR}/stage.loop_list/target_dependents.specified" \
			"${DBDIR}/stage.loop_list/target_requirements.specified" \
			"${DBDIR}/need.list" \
			"${DBDIR}/targets_specified_so_far" 2> /dev/null \
			| sort -u > ${DBDIR}/targets_specified_so_far.tmp
		mv "${DBDIR}/targets_specified_so_far.tmp" "${DBDIR}/targets_specified_so_far"
		rm -f "${DBDIR}/stage.loop_list/target_itself.specified" \
			"${DBDIR}/stage.loop_list/target_dependents.specified" \
			"${DBDIR}/stage.loop_list/target_requirements.specified"
		if [ -n "$opt_target_itself$opt_target_dependents$opt_target_requirements" ]
		then
			options_select_new_ports_if_duplicated O \
				"${DBDIR}/stage.loop_list/target_itself.specified" "$opt_target_itself"
			options_select_new_ports_if_duplicated t \
				"${DBDIR}/stage.loop_list/target_dependents.specified" "$opt_target_dependents"
			options_select_new_ports_if_duplicated T \
				"${DBDIR}/stage.loop_list/target_requirements.specified" "$opt_target_requirements"
			if [ `cat "${DBDIR}/stage.loop_list/target_itself.specified" \
				"${DBDIR}/stage.loop_list/target_dependents.specified" \
				"${DBDIR}/stage.loop_list/target_requirements.specified" | wc -l` -eq 0 ]
			then
				message_echo "ERROR: No matching port for target globs." >&2
				message_echo >&2
				exit 1
			fi
			cat "${DBDIR}/stage.loop_list/target_itself.specified" \
				"${DBDIR}/stage.loop_list/target_dependents.specified" \
				"${DBDIR}/stage.loop_list/target_requirements.specified" \
				| sort -u > ${TMPDIR}/DETERMINE_SPECIFIED_TARGETS.reset
			cat "${TMPDIR}/DETERMINE_SPECIFIED_TARGETS.reset" "${DBDIR}/need.list" 2> /dev/null \
				| sort -u > ${DBDIR}/need.list.tmp
			mv "${DBDIR}/need.list.tmp" "${DBDIR}/need.list"
			sort -u "${DBDIR}/need.list" "${DBDIR}/targets_specified_so_far" \
				> ${DBDIR}/targets_specified_so_far.tmp
			mv "${DBDIR}/targets_specified_so_far.tmp" "${DBDIR}/targets_specified_so_far"
			for tag in all run build none
			do
				for level in direct full
				do
					dbsuffix=$tag.$level
					{
						cat "${TMPDIR}/DETERMINE_SPECIFIED_TARGETS.reset"
						cat "${DBDIR}/stage.loop_list/target_dependents.specified" | while read origin
						do
							nodedir=${DBDIR}/requires/$origin
							cat "$nodedir/dependents.$dbsuffix" 2> /dev/null
						done
						cat "${DBDIR}/stage.loop_list/target_requirements.specified" | while read origin
						do
							nodedir=${DBDIR}/requires/$origin
							cat "$nodedir/requirements.$dbsuffix" 2> /dev/null
						done
					} | sort -u | while read origin
					do
						fileedit_rm_a_line "$origin" "${DBDIR}/success.$dbsuffix.list"
						fileedit_rm_a_line "$origin" "${DBDIR}/todo_after_requirements_succeed.$dbsuffix.list"
						rm -f "${DBDIR}/requires/$origin/succeeded_once"
					done
				done
			done
		fi
		message_echo
	}
	program_exec_and_record_completion DETERMINE_SPECIFIED_TARGETS
}

# ============= Show specified targets =============
command_do_show_specified_targets ()
{
	if [ -n "$opt_target_itself$opt_target_dependents$opt_target_requirements" -a $opt_batch_mode = no ]
	then
		message_echo "INFO: (Re/de-)installation will be carried out only for the targets:"
		echo
		if [ `wc -l < ${DBDIR}/stage.loop_list/target_itself.specified` -gt 0 ]
		then
			message_echo "[Targets only]"
			echo "----------------------------------------"
			cat "${DBDIR}/stage.loop_list/target_itself.specified"
			echo "----------------------------------------"
			echo
		fi
		if [ `wc -l < ${DBDIR}/stage.loop_list/target_dependents.specified` -gt 0 ]
		then
			message_echo "[Targets with their `options_get_dependency_msgterm` dependents]"
			echo "----------------------------------------"
			cat "${DBDIR}/stage.loop_list/target_dependents.specified"
			echo "----------------------------------------"
			echo
		fi
		if [ `wc -l < ${DBDIR}/stage.loop_list/target_requirements.specified` -gt 0 ]
		then
			message_echo "[Targets with their `options_get_dependency_msgterm` requirements]"
			echo "----------------------------------------"
			cat "${DBDIR}/stage.loop_list/target_requirements.specified"
			echo "----------------------------------------"
			echo
		fi
	fi
}

# ============= Determine all target ports =============
command_do_determine_all_target_ports ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='DETERMINE_SPECIFIED_TARGETS COLLECT_ALL_INSTALLED_PACKAGES'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Determining all target ports"
		{
			if [ $opt_only_target_scope = no ]
			then
				cat "${DBDIR}/installed_ports" 2> /dev/null || :
			fi
			cat "${DBDIR}/stage.loop_list/target_itself.specified" || :
			cat "${DBDIR}/stage.loop_list/target_dependents.specified" || :
			cat "${DBDIR}/stage.loop_list/target_requirements.specified" || :
			cat "${DBDIR}/targets_specified_so_far" || :
		}  2> /dev/null | sort -u > "${DBDIR}/stage.loop_list/ports_to_inspect"
		cp "${DBDIR}/stage.loop_list/ports_to_inspect" "${DBDIR}/stage.loop_list/ports_to_inspect_initial"
		while read origin
		do
			[ -d "${PORTSDIR}/$origin" ] || echo "$origin"
		done < ${DBDIR}/stage.loop_list/ports_to_inspect > ${DBDIR}/obsoletes_to_escape_for_inspection
		message_echo
	}
	program_exec_and_record_completion DETERMINE_ALL_TARGET_PORTS
}

# ============= Inspection of all initial dependencies =============
command_do_inspection_of_all_initial_dependencies ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='DETERMINE_ALL_TARGET_PORTS'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin
		origin=$1
		database_build_setup_initial_node "$origin"
	}
	_program_exec_and_record_completion__operation ()
	{
		local DEPTH_INDEX 
		message_section_title "Inspecting initial dependencies of the all installed packages"
		[ $opt_only_target_scope = no -a -n "$opt_target_itself$opt_target_dependents$opt_target_requirements" ] && \
			message_echo "INFO: Ports which seem irrelevant to the targets are also inspected in order to get complete information."
		program_exec_restartable_loop_operation ports_to_inspect_initial
		database_build_post_inspect_initial_dependencies
		message_echo
	}
	program_exec_and_record_completion INSPECT_ALL_INITIAL_DEPENDENCIES
}

# ============= Prepare for inspecting all dependencies =============
command_do_prepare_for_inspect_all_dependencies ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='PARSE_CONF  INSPECT_ALL_INITIAL_DEPENDENCIES'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Preparing for inspection of all dependencies"
		database_build_escape "${DBDIR}/escaped_obsoletes" < ${DBDIR}/obsoletes_to_escape_for_inspection
		cp /dev/null "${DBDIR}/done_required_ports_to_inspect"
		message_echo
	}
	program_exec_and_record_completion PREPARE_FOR_INSPECT_ALL_DEPENDENCIES
}

# ============= Check ports of which the port options are changed and reset their databases =============
# Return status 0 for no need to update of inspected ports and 1 for needed to do it
command_do_reset_changed_portdb ()
{
	local tmp_ls
	program_chk_stage_complete INSPECT_DEPENDENTS || return
	pkgsys_exists_saved_port_oprions_timestamps || return
	message_section_title "Inspecting changes of port options"
	tmp_ls=${TMPDIR}/command_do_reset_changed_portdb
	rm -f "$tmp_ls.updated"
	if [ ! -d "${PORT_DBDIR}" ]
	then
		message_echo "WARNING: ${PORT_DBDIR} is missing. All port options are recognized as default."
	fi
	pkgsys_get_saved_port_oprions_timestamps_all > $tmp_ls.db
	pkgsys_get_current_port_oprions_timestamp > $tmp_ls.log
	pkgsys_get_changed_port_oprions "$tmp_ls.db" "$tmp_ls.log" > $tmp_ls.diff
	if ! [ `wc -l < $tmp_ls.diff` -gt 0 ]
	then
		message_echo
		return
	fi
	message_echo "INFO: There are some changes. Inspecting the detail..."
	cp /dev/null "$tmp_ls.uninspected.dbfile"
	cp /dev/null "$tmp_ls.uninspected.origin"
	cp /dev/null "$tmp_ls.nonexistent.db"
	pkgsys_conv_portoptiondbs_to_globs < $tmp_ls.diff > $tmp_ls.glob
	str_escape_replaceval_filter < $tmp_ls.diff > $tmp_ls.diff.rpl
	paste "$tmp_ls.diff" "$tmp_ls.diff.rpl" "$tmp_ls.glob" | while read -r dbfile dbfile_rpl glob
	do
		pkgsys_eval_ports_glob "$glob" > $tmp_ls.origins
		if [ `wc -l < $tmp_ls.origins` -gt 0 ]
		then
			sed -E "s/^/$dbfile_rpl /" < $tmp_ls.origins
		else
			echo $dbfile >> $tmp_ls.nonexistent.db
		fi
	done | while read dbfile origin
	do
		if [ -d "${DBDIR}/requires/$origin" ]
		then
			message_echo "Change detected on $origin: Resetting"
			database_build_patch_reconf "$origin"
			program_deregister_stage_complete PREPARE_FOR_INSPECT_ALL_DEPENDENCIES
			program_deregister_stage_complete ALL_COMPLETE
			touch "$tmp_ls.updated"
		else
			echo "$dbfile" >> $tmp_ls.uninspected.dbfile
			echo "$origin" >> $tmp_ls.uninspected.origin
		fi
	done
	message_echo "INFO: Recording for not inspected ports..."
	pkgsys_conv_portorigin_to_port_oprion_timestamp_logname < $tmp_ls.uninspected.origin > $tmp_ls.uninspected.logname
	str_escape_regexp_filter < $tmp_ls.uninspected.dbfile | sed 's/^/^/;s/$/[[:space:]]/' > $tmp_ls.uninspected.dbfile_ptn
	paste "$tmp_ls.uninspected.dbfile_ptn" "$tmp_ls.uninspected.logname" | while read -r dbfile_ptn logname
	do
		grep -E "$dbfile_ptn" < $tmp_ls.log > ${DBDIR}/ls_dbdir/$logname.log 2> /dev/null || :
	done
	str_escape_regexp_filter < $tmp_ls.nonexistent.db | sed 's/^/^/;s/$/[[:space:]]/' > $tmp_ls.nonexistent.db_ptn
	grep -E -f "$tmp_ls.nonexistent.db_ptn" < $tmp_ls.log | pkgsys_register_list_nonexistent_portopriondb
	message_echo
	[ ! -e "$tmp_ls.updated" ]
}

# ============= Inspection of all dependencies =============
command_do_inspection_of_all_dependencies ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='PREPARE_FOR_INSPECT_ALL_DEPENDENCIES'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin
		origin=$1
		database_build_inspect_dependencies "$origin"
	}
	_program_exec_and_record_completion__operation ()
	{
		local DEPTH_INDEX 
		message_section_title "Inspecting dependencies of the all installed packages"
		[ $opt_only_target_scope = no -a -n "$opt_target_itself$opt_target_dependents$opt_target_requirements" ] && \
			message_echo "INFO: Ports which seem irrelevant to the targets are also inspected in order to get complete information."
		DEPTH_INDEX='--'
		program_exec_restartable_loop_operation ports_to_inspect
		database_build_post_inspect_dependencies
		database_maintain_clear_prevset
		message_echo
	}
	program_exec_and_record_completion INSPECT_ALL_DEPENDENCIES
}

# ============= Convert dependency-lists to actual ones =============
command_do_convert_dependency_lists_to_actual_ones ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin table dbtag level tag target
		origin=$1
		for table in dependents requirements
		do
			for dbtag in requires obsolete
			do
				for level in direct full
				do
					for tag in all run build
					do
						target=${DBDIR}/$dbtag/$origin/${table}.${tag}.${level}
						if [ -e "$target.src" ]
						then
							sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" "$target.src" \
								| grep -v '^$' | sort -u > $target.tmp
							[ -e "$target" ] && ! diff -q "$target.tmp" "$target" > /dev/null \
								&& echo "$origin" >> ${DBDIR}/update_dependencies
							mv "$target.tmp" "$target"
						else
							[ -e "$target" ] && echo "$origin" >> ${DBDIR}/update_dependencies
							rm -f "$target"
						fi
					done
				done
			done
		done
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Conversion of dependency-lists to actual ones"
		program_exec_restartable_loop_operation convert_dependency_lists
		sort -u "${DBDIR}/update_dependencies" > ${DBDIR}/update_dependencies.tmp
		mv "${DBDIR}/update_dependencies.tmp" "${DBDIR}/update_dependencies"
		for tag in all run build
		do
			( cd "${DBDIR}/requires" && \
				find . -depth 3 -type f -name requirements.${tag}.full -or -name requirements.${tag}.full.orig ) \
				| sort -u \
				| sed 's|^./||;s|/[^/]*$||' \
				| grep -v -Fx -f "${DBDIR}/update_dependencies" \
				> ${TMPDIR}/convert_requirements_list:full_complete.grep_pattern || :
			( cd "${DBDIR}/requires" && \
				find . -depth 3 -type f -name requirements.${tag}.direct -or -name requirements.${tag}.direct.orig ) \
				| sort -u \
				| sed 's|^./||;s|/[^/]*$||' \
				| grep -v -Fx -f "${TMPDIR}/convert_requirements_list:full_complete.grep_pattern" \
				> ${DBDIR}/stage.loop_list/complete_recursive_${tag}time_reqlists || :
		done
		for inspected_level_tmp in direct node
		do
			cat "${DBDIR}/ports.inspected.${inspected_level_tmp}.list" || :
		done 2> /dev/null | sort -u > ${DBDIR}/stage.loop_list/trim_dependency_lists_rm_uninspected_ports
		find "${DBDIR}/requires" -depth 2 -type d > ${DBDIR}/stage.loop_list/inspect_dependent
		[ -e "${DBDIR}/dependents_files" ] && \
			mv "${DBDIR}/dependents_files" "${DBDIR}/dependents_files.prev"
		rm -f "${DBDIR}/stage.loop_list/make_dependents_lists_unique.tmp" "${DBDIR}/dependents_files.tmp"
		message_echo
	}
	program_exec_and_record_completion CONVERT_REQUIREMENTS_LIST
}

# ============= Completion of recursive requirement lists =============
command_do_completion_of_recursive_requirement_lists ()
{
	local PROGRAM_DEPENDS _REQUIREMENT_LISTS_tag
	for _REQUIREMENT_LISTS_tag in all run build
	do
		PROGRAM_DEPENDS='CONVERT_REQUIREMENTS_LIST'
		_program_exec_restartable_loop_operation__routine ()
		{
			local tag dbpath origin suffix
			tag=${_REQUIREMENT_LISTS_tag}
			dbpath=$1
			origin=`str_dirpath_to_origin "$dbpath"`
			for suffix in '' .orig
			do
				database_build_get_complete_recursive_dependency "$tag" "$origin" "$suffix" > /dev/null
			done
		}
		_program_exec_and_record_completion__operation ()
		{
			local tag
			tag=${_REQUIREMENT_LISTS_tag}
			message_section_title "Completion of ${tag}-time requirement lists"
			program_exec_restartable_loop_operation complete_recursive_${tag}time_reqlists
			message_echo
		}
		program_exec_and_record_completion RECURSIVE_REQUIREMENT_LISTS:${_REQUIREMENT_LISTS_tag}
	done
}

# ============= Trim dependency lists by removing uninspected ports =============
command_do_trim_dependency_lists_by_removing_uninspected_ports ()
{
	local PROGRAM_DEPENDS
	if [ $opt_only_target_scope = yes ]
	then
		PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES CONVERT_REQUIREMENTS_LIST'
		_program_exec_restartable_loop_operation__routine ()
		{
			local dbpath tag level srcdb
			dbpath=$1
			for tag in all run build
			do
				for level in direct full
				do
					srcdb=requirements.${tag}.${level}
					[ -e "$dbpath/$srcdb" ] || continue
					grep -Fx -f "${DBDIR}/inspected_ports" "$dbpath/$srcdb" > $dbpath/$srcdb.tmp || :
					mv "$dbpath/$srcdb.tmp" "$dbpath/$srcdb"
				done
			done
		}
		_program_exec_and_record_completion__operation ()
		{
			message_section_title "Trimming dependency lists by removing uninspected ports"
			program_exec_restartable_loop_operation trim_dependency_lists_rm_uninspected_ports
			message_echo
		}
		program_exec_and_record_completion TRIM_DEPENDENCY_LISTS_RM_UNINSPECTED_PORTS
	fi
}

# ============= Inspection of dependents =============
command_do_inspection_of_dependents ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES CONVERT_REQUIREMENTS_LIST TRIM_DEPENDENCY_LISTS_RM_UNINSPECTED_PORTS'
	_program_exec_restartable_loop_operation__routine ()
	{
		local dbpath origin tag level suffix srcdb dstdb
		dbpath=$1
		origin=`str_dirpath_to_origin "$dbpath"`
		for tag in all run build
		do
			for suffix in '' .orig
			do
				for level in direct full
				do
					srcdb=requirements.${tag}.${level}${suffix}
					[ -e "$dbpath/$srcdb" ] || continue
					dstdb=dependents.${tag}.${level}${suffix}
					while read origin_requirement
					do
						dstpath=${DBDIR}/requires/$origin_requirement
						echo "$dstpath/$dstdb" >> ${DBDIR}/dependents_files.tmp
						[ "$dstpath/$dstdb" -nt "$dbpath/$srcdb" ] && continue
						mkdir -p "$dstpath"
						echo "$origin" >> $dstpath/$dstdb.raw
						echo "$dstpath/$dstdb" >> ${DBDIR}/stage.loop_list/make_dependents_lists_unique.tmp
					done < $dbpath/$srcdb
				done
				srcdb=ignored_requirements.${tag}${suffix}
				[ -e "$dbpath/$srcdb" ] || continue
				dstdb=ignored_dependents.${tag}${suffix}
				while read origin_requirement
				do
					dstpath=${DBDIR}/requires/$origin_requirement
					echo "$dstpath/$dstdb" >> ${DBDIR}/ignored_dependents_files.tmp
					[ "$dstpath/$dstdb" -nt "$dbpath/$srcdb" ] && continue
					mkdir -p "$dstpath"
					echo "$origin" >> $dstpath/$dstdb.raw
					echo "$dstpath/$dstdb" >> ${DBDIR}/stage.loop_list/make_ignored_dependents_lists_unique.tmp
				done < $dbpath/$srcdb
			done
		done
	}
	_program_exec_and_record_completion__operation ()
	{
		local dbrequires_valesc
		message_section_title "Inspection of dependents"
		dbrequires_valesc=`str_escape_replaceval "${DBDIR}/requires/"`
		program_exec_restartable_loop_operation inspect_dependent
		for dbtype in dependents ignored_dependents
		do
			sort -u "${DBDIR}/stage.loop_list/make_${dbtype}_lists_unique.tmp" 2> /dev/null \
				> ${DBDIR}/stage.loop_list/make_${dbtype}_lists_unique || :
			sort -u "${DBDIR}/${dbtype}_files.tmp" 2> /dev/null > ${DBDIR}/${dbtype}_files || :
			[ -e "${DBDIR}/make_${dbtype}_lists_unique.prev" ] && \
				fileedit_manipulate_old_lines "${DBDIR}/${dbtype}_files.prev" "${DBDIR}/${dbtype}_files" \
				| xargs rm -f
		done
		message_echo
	}
	program_exec_and_record_completion INSPECT_DEPENDENTS
}

# ============= Restore escaped obsolete packages for inspection =============
command_do_restore_escaped_obsoletes ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='INSPECT_DEPENDENTS'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Restoring escaped obsolete packages for inspection"
		cat "${DBDIR}/escaped_obsoletes" 2> /dev/null | database_build_restore
		rm -f "${DBDIR}/escaped_obsoletes"
		message_echo
	}
	program_exec_and_record_completion RESTORE_ESCAPED_OBSOLETE_PACKAGES_FOR_INSPECTION
}

# ============= Remove duplicated lines in dependents lists =============
command_do_remove_duplicated_lines_in_dependents_lists ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='CONVERT_REQUIREMENTS_LIST INSPECT_DEPENDENTS'
	_program_exec_restartable_loop_operation__routine ()
	{
		local dbpath tag level dstdb
		dstdb=$1
		cat "$dstdb" "$dstdb.raw" 2> /dev/null | sort -u > $dstdb.tmp
		mv "$dstdb.tmp" "$dstdb"
		rm -f "$dstdb.raw"
	}
	_program_exec_and_record_completion__operation ()
	{
		local dbrequires_valesc
		message_section_title "Removing duplicated items in dependents lists"
		program_exec_restartable_loop_operation make_dependents_lists_unique
		message_echo
	}
	program_exec_and_record_completion MAKE_DEPENDENTS_LISTS_UNIQUE
}

# ============= Remove duplicated lines in ignored dependents lists =============
command_do_remove_duplicated_lines_in_ignored_dependents_lists ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='CONVERT_REQUIREMENTS_LIST INSPECT_DEPENDENTS'
	_program_exec_restartable_loop_operation__routine ()
	{
		local dbpath tag level dstdb
		dstdb=$1
		cat "$dstdb" "$dstdb.raw" 2> /dev/null | sort -u > $dstdb.tmp
		mv "$dstdb.tmp" "$dstdb"
		rm -f "$dstdb.raw"
	}
	_program_exec_and_record_completion__operation ()
	{
		local dbrequires_valesc
		message_section_title "Removing duplicated items in ignored dependents lists"
		program_exec_restartable_loop_operation make_ignored_dependents_lists_unique
		message_echo
	}
	program_exec_and_record_completion MAKE_IGNORED_DEPENDENTS_LISTS_UNIQUE
}

# ============= Preparation of target attribute information =============
command_do_preparation_of_target_attribute_information ()
{
	local PROGRAM_DEPENDS _TARGET_ATTR_INFO_table
	for _TARGET_ATTR_INFO_table in requirements dependents itself
	do
		[ `cat "${DBDIR}/stage.loop_list/target_${_TARGET_ATTR_INFO_table}.replaced.specified" 2> /dev/null \
			| wc -l` -gt 0 ] || continue
		PROGRAM_DEPENDS='DETERMINE_SPECIFIED_TARGETS CONVERT_REQUIREMENTS_LIST'
		_program_exec_restartable_loop_operation__routine ()
		{
			local origin dbtargets_valesc table
			origin=$1
			dbtargets_valesc=`str_escape_replaceval "${DBDIR}/targets/"`
			table=${_TARGET_ATTR_INFO_table}
			for database in requires initial
			do
				dbpath=${DBDIR}/$database/$origin
				dstpath=${DBDIR}/targets/$origin
				mkdir -p "$dstpath"
				touch "$dstpath/target_itself"
				echo "$origin" >> ${DBDIR}/all_targets.lst
				[ $table = itself ] && continue
				for tag in all run build
				do
					for level in direct full
					do
						srcdb=${table}.${tag}.${level}
						dstdb=target_${database}_${table}.${tag}.${level}
						[ -e "$dbpath/$srcdb" ] || continue
						cat "$dbpath/$srcdb" >> ${DBDIR}/all_targets.lst
						sed -E "s/^/$dbtargets_valesc/; s|$|/$dstdb|" "$dbpath/$srcdb" \
							| fileedit_add_a_line_to_files_if_new "$origin"
					done
				done
			done
		}
		_program_exec_and_record_completion__operation ()
		{
			local table
			table=${_TARGET_ATTR_INFO_table}
			message_section_title "Preparation of target attribute information for dependency [$table]"
			program_exec_restartable_loop_operation target_$table.replaced.specified
			message_echo
		}
		program_exec_and_record_completion TARGET_ATTR_INFO:${_TARGET_ATTR_INFO_table}
	done
}

# ============= Post-process after the preparation of target attribute information =============
command_do_post_process_after_the_preparation_of_target_attribute_information ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='MAKE_DEPENDENTS_LISTS_UNIQUE MAKE_IGNORED_DEPENDENTS_LISTS_UNIQUE TARGET_ATTR_INFO:requirements TARGET_ATTR_INFO:dependents'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Post-process after the preparation of target attribute information"
		sort -u "${DBDIR}/all_targets.lst" 2> /dev/null \
			| grep -Fx -f "${DBDIR}/inspected_ports" \
			| sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" 2> /dev/null \
			> ${DBDIR}/all_targets.lst.tmp || :
		mv "${DBDIR}/all_targets.lst.tmp" "${DBDIR}/all_targets.lst"
		find "${DBDIR}/targets" -depth 2 -type d > ${DBDIR}/stage.loop_list/build_complement_to_new_dependents_for_targets 2> /dev/null || :
		{
			cat "${DBDIR}/all_targets.lst" "${DBDIR}/need.with_replaced.list" 2> /dev/null || :
			find "${DBDIR}/moved_from" -depth 3 -type f -name installed_version \
				| sed -E 's|.*/([^/]+/[^/]+)/[^/]*$|\1|'
		} | sort -u > ${DBDIR}/stage.loop_list/inspect_necessity
		cp /dev/null "${DBDIR}/stage.loop_list/parse_target_attr_info"
		find -E "${DBDIR}/requires" -depth 3 -type f -regex '.*/necessary_port\.(direct|full)$' -delete
		message_echo
	}
	program_exec_and_record_completion TARGET_ATTR_INFO_POSTPROCESS
}

# ============= Build of data on complement to new dependents for target attribute information =============
command_do_build_of_data_on_complement_to_new_dependents_for_target_attribute_information ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='TARGET_ATTR_INFO_POSTPROCESS TARGET_ATTR_INFO:requirements TARGET_ATTR_INFO:dependents CONVERT_REQUIREMENTS_LIST'
	_program_exec_restartable_loop_operation__routine ()
	{
		local dbpath origin
		dbpath=$1
		origin=`str_dirpath_to_origin "$dbpath"`
		database_build_complement_to_new_dependents_for_targets "$origin"
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Build of data on complement to new dependents for target attribute information"
		program_exec_restartable_loop_operation build_complement_to_new_dependents_for_targets
		sort -u "${DBDIR}/stage.loop_list/parse_target_attr_info" > ${DBDIR}/stage.loop_list/parse_target_attr_info.tmp
		mv "${DBDIR}/stage.loop_list/parse_target_attr_info.tmp" "${DBDIR}/stage.loop_list/parse_target_attr_info"
		message_echo
	}
	program_exec_and_record_completion COMPLEMENT_TO_NEW_DEPENDENTS_FOR_TARGET_ATTR_INFO
}

# ============= Parse target attribute information =============
command_do_parse_target_attribute_information ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='COMPLEMENT_TO_NEW_DEPENDENTS_FOR_TARGET_ATTR_INFO'
	_program_exec_restartable_loop_operation__routine ()
	{
		local dbpath origin
		dbpath=$1
		origin=`str_dirpath_to_origin "$dbpath"`
		database_build_target_attributes "$origin"
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Parsing target attribute information"
		program_exec_restartable_loop_operation parse_target_attr_info
		message_echo
	}
	program_exec_and_record_completion PARSE_TARGET_ATTR_INFO
}

# ============= Inspection of necessity =============
command_do_inspection_of_necessity ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='TARGET_ATTR_INFO_POSTPROCESS RECURSIVE_REQUIREMENT_LISTS:run RECURSIVE_REQUIREMENT_LISTS:build INSPECT_ALL_DEPENDENCIES'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin
		origin=$1
		for level in direct full
		do
			database_build_inspect_necessity_for_only_new_upgrade "$origin" "$level"
		done
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Inspection of necessity"
		program_exec_restartable_loop_operation inspect_necessity
		for level in direct full
		do
			find "${DBDIR}/requires" -depth 3 -type f -name "necessary_port.${level}" | \
				sed -E 's|.*/([^/]+/[^/]+)/[^/]*$|\1|' > ${DBDIR}/stage.loop_list/necessary_ports.${level}
		done
		message_echo
	}
	program_exec_and_record_completion INSPECT_NECESSITY
}

# ============= Inspection of necessary upgrades =============
command_do_inspection_of_necessary_upgrades ()
{
	local PROGRAM_DEPENDS _NECESSARY_UPDATES_level
	for _NECESSARY_UPDATES_level in direct full
	do
		PROGRAM_DEPENDS='INSPECT_NECESSITY INSPECT_ALL_DEPENDENCIES MAKE_DEPENDENTS_LISTS_UNIQUE MAKE_IGNORED_DEPENDENTS_LISTS_UNIQUE'
		_program_exec_restartable_loop_operation__routine ()
		{
			local markerpath level dbpath origin tag
			origin=$1
			level=${_NECESSARY_UPDATES_level}
			dbpath=${DBDIR}/requires/$origin
			database_query_does_a_port_need_update "$origin" || return 0
			for tag in all run build none
			do
				touch "$dbpath/necessary_upgrade.${tag}.${level}"
				[ -e "$dbpath/dependents.${tag}.${level}" -o "$dbpath/ignored_dependents.${tag}" ] || continue
				cat "$dbpath/dependents.${tag}.${level}" "$dbpath/ignored_dependents.${tag}" 2> /dev/null | \
					while read origin_dependent
					do
						touch "${DBDIR}/requires/$origin_dependent/necessary_upgrade.${tag}.${level}"
					done
			done
		}
		_program_exec_and_record_completion__operation ()
		{
			local level
			level=${_NECESSARY_UPDATES_level}
			message_section_title "Inspection of necessary upgrades at the $level level"
			program_exec_restartable_loop_operation necessary_ports.${level}
			find "${DBDIR}/requires" -depth 3 -type f -name "necessary_upgrade.run.${level}" | \
				sed -E 's|.*/([^/]+/[^/]+)/[^/]*$|\1|' > ${DBDIR}/stage.loop_list/complete_necessary_ports.${level}
			message_echo
		}
		program_exec_and_record_completion NECESSARY_UPDATES:${_NECESSARY_UPDATES_level}
	done
}

# ============= Completion of necessary upgrades for build =============
command_do_complete_necessary_upgrades_for_build ()
{
	local PROGRAM_DEPENDS _NECESSARY_UPDATES_level
	for _NECESSARY_UPDATES_level in direct full
	do
		PROGRAM_DEPENDS="INSPECT_NECESSITY NECESSARY_UPDATES:${_NECESSARY_UPDATES_level}"
		_program_exec_restartable_loop_operation__routine ()
		{
			local origin level
			origin=$1
			level=${_NECESSARY_UPDATES_level}
			database_build_complete_necessary_upgrades_for_build "$origin" "$level"
		}
		_program_exec_and_record_completion__operation ()
		{
			local level
			level=${_NECESSARY_UPDATES_level}
			message_section_title "Completion of necessary upgrades for build at the $level level"
			program_exec_restartable_loop_operation complete_necessary_ports.${level}
			message_echo
		}
		program_exec_and_record_completion COMPLETE_NECESSARY_UPDATES:${_NECESSARY_UPDATES_level}
	done
}

# ============= Preparation for inspection of new leaf ports =============
command_do_preparation_for_inspection_of_new_leaf_ports ()
{
	local PROGRAM_DEPENDS
	if [ ! -e "${DBDIR}/inspected_ports_only_partially" ]
	then
		PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES MAKE_DEPENDENTS_LISTS_UNIQUE MAKE_IGNORED_DEPENDENTS_LISTS_UNIQUE PARSE_CONF'
		_program_exec_and_record_completion__operation ()
		{
			message_section_title "Preparation for inspection of new leaf ports"
			find "${DBDIR}/requires" -depth 3 -type f -name dependents.all.full -or -name ignored_dependents.all \
				| sed -E 's|.*/([^/]+/[^/]+)/[^/]+$|\1|' \
				| sort -u > ${TMPDIR}/PREPARE_INSPECT_LEAF_PORTS:nonleaf_ports
			sort -u "${DBDIR}/inspected_ports" > ${TMPDIR}/PREPARE_INSPECT_LEAF_PORTS:inspected_ports
			fileedit_manipulate_new_lines \
				"${TMPDIR}/PREPARE_INSPECT_LEAF_PORTS:nonleaf_ports" \
				"${TMPDIR}/PREPARE_INSPECT_LEAF_PORTS:inspected_ports" \
				| grep -v -Fx -f "${DBDIR}/conf/HOLD:PORTS.parsed" 2> /dev/null \
				> ${DBDIR}/stage.loop_list/leaf_ports_primary_candidates || :
			cp /dev/null "${DBDIR}/leaf_ports.filter"
			cp /dev/null "${DBDIR}/stage.loop_list/leaf_ports_secondary_candidates"
			message_echo
		}
		program_exec_and_record_completion PREPARE_INSPECT_LEAF_PORTS
	fi
}

# ============= Inspection of new primary leaf ports =============
command_do_inspection_of_new_primary_leaf_ports ()
{
	local PROGRAM_DEPENDS
	if [ ! -e "${DBDIR}/inspected_ports_only_partially" ]
	then
		PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES MAKE_DEPENDENTS_LISTS_UNIQUE MAKE_IGNORED_DEPENDENTS_LISTS_UNIQUE PREPARE_INSPECT_LEAF_PORTS PARSE_CONF'
		_program_exec_restartable_loop_operation__routine ()
		{
			local origin origin_ini dbpath origin_req
			origin=$1
			pkgsys_is_pkgtool "$origin" && return
			dbpath=${DBDIR}/requires/$origin
			grep -q -Fx "$origin" "${DBDIR}/need.with_replaced.list" 2> /dev/null && return
			if ! grep -q -Fx "$origin" "${DBDIR}/noneed.list" 2> /dev/null
			then
				if [ -e "$dbpath/initial_orig" ]
				then
					origin_ini=`cat "$dbpath/initial_orig"`
					[ -e "${DBDIR}/initial/$origin_ini/installed_version" \
						-a `cat "${DBDIR}/initial/$origin_ini/dependents.all.full" 2> /dev/null | wc -l` -eq 0 ] \
						&& return
				fi
				[ -e "${DBDIR}/initial/$origin/installed_version" \
					-a `cat "${DBDIR}/initial/$origin/dependents.all.full" 2> /dev/null | wc -l` -eq 0 ] \
					&& return
			fi
			if [ -e "$dbpath/requirements.all.full" -o -e "$dbpath/ignored_requirements.all" ]
			then
				cat "$dbpath/requirements.all.full" "$dbpath/ignored_requirements.all" 2> /dev/null | \
					grep -v -Fx -f "${DBDIR}/conf/HOLD:PORTS.parsed" 2> /dev/null| \
					fileedit_add_lines_if_new "${DBDIR}/stage.loop_list/leaf_ports_secondary_candidates" || :
			fi
			fileedit_add_a_line_if_new "$origin" "${DBDIR}/leaf_ports.filter"
		}
		_program_exec_and_record_completion__operation ()
		{
			local num_leaves num_leaves_prev
			message_section_title "Inspection of new primary leaf ports"
			program_exec_restartable_loop_operation leaf_ports_primary_candidates
			wc -l < ${DBDIR}/leaf_ports.filter | tr -d ' ' > ${DBDIR}/num_leaves
			cp /dev/null "${DBDIR}/leaf_ports_secondary_candidates.new_requirements"
			message_echo "  `cat "${DBDIR}/num_leaves"` primary leaf port(s) is/are found."
			message_echo
		}
		program_exec_and_record_completion INSPECT_PRIMARY_LEAF_PORTS
	fi
}

# ============= Inspection of requirements of new leaf ports =============
command_do_inspection_of_requirements_of_new_leaf_ports ()
{
	local PROGRAM_DEPENDS
	if [ ! -e "${DBDIR}/inspected_ports_only_partially" ]
	then
		PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES MAKE_DEPENDENTS_LISTS_UNIQUE MAKE_IGNORED_DEPENDENTS_LISTS_UNIQUE INSPECT_PRIMARY_LEAF_PORTS PARSE_CONF'
		_program_exec_and_record_completion__operation ()
		{
			local num_leaves num_leaves_prev num_inspect num_leaves_new
			message_section_title "Inspection of requirements of new leaf ports"
			message_echo "INFO: The inspection proceeds by iterative method."
			while :
			do
				_program_exec_restartable_loop_operation__routine ()
				{
					local origin dbpath
					origin=$1
					pkgsys_is_pkgtool "$origin" && return
					dbpath=${DBDIR}/requires/$origin
					grep -q -Fx "$origin" "${DBDIR}/need.with_replaced.list" 2> /dev/null && return
					cat "$dbpath/dependents.all.full" "$dbpath/ignored_dependents.all"  2> /dev/null \
						| grep -Fxq -v -f "${DBDIR}/leaf_ports.filter" 2> /dev/null && return
					cat "$dbpath/requirements.all.full" "$dbpath/ignored_requirements.all" 2> /dev/null \
						>> ${DBDIR}/leaf_ports_secondary_candidates.new_requirements || :
					fileedit_add_a_line_if_new "$origin" "${DBDIR}/leaf_ports.filter"
				}
				program_exec_restartable_loop_operation leaf_ports_secondary_candidates
				num_leaves_prev=`cat "${DBDIR}/num_leaves"`
				num_leaves=`wc -l < ${DBDIR}/leaf_ports.filter | tr -d ' '`
				num_leaves_new=`echo $(($num_leaves-$num_leaves_prev)) | tr -d ' '`
				if [ $num_leaves_new -eq 0 ]
				then
					message_echo "  No more leaf port is found."
					message_echo "  $num_leaves leaf port(s) is/are found in total."
					break
				fi
				{
					grep -Fx -v -f "${DBDIR}/leaf_ports.filter" \
						"${DBDIR}/stage.loop_list/leaf_ports_secondary_candidates" || :
					cat "${DBDIR}/leaf_ports_secondary_candidates.new_requirements" || :
				} 2> /dev/null | grep -v -Fx -f "${DBDIR}/conf/HOLD:PORTS.parsed" 2> /dev/null | sort -u \
					> ${DBDIR}/stage.loop_list/leaf_ports_secondary_candidates.tmp || :
				program_reset_loop_for_stage INSPECT_REQUIREMENTS_OF_LEAF_PORTS
				mv "${DBDIR}/stage.loop_list/leaf_ports_secondary_candidates.tmp" \
					"${DBDIR}/stage.loop_list/leaf_ports_secondary_candidates"
				cp /dev/null "${DBDIR}/leaf_ports_secondary_candidates.new_requirements"
				echo "$num_leaves" > ${DBDIR}/num_leaves
				num_inspect=`wc -l < ${DBDIR}/stage.loop_list/leaf_ports_secondary_candidates | tr -d ' '`
				message_echo "  $num_leaves_new leaf port(s) is/are newly found; continue for $num_inspect candidate(s)."
			done
			grep -Fx -f "${DBDIR}/leaf_ports.filter" "${DBDIR}/inspected_ports" 2> /dev/null | sort -u > ${DBDIR}/leaf_ports || :
			message_echo
		}
		program_exec_and_record_completion INSPECT_REQUIREMENTS_OF_LEAF_PORTS
	fi
}

# ============= Order the ports considering dependencies =============
command_do_order_the_ports_considering_dependencies ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='CONVERT_REQUIREMENTS_LIST'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Ordering dependencies"
		if ! database_build_order_ports_considering_dependencies
		then
			message_echo "ERROR: Unsatisfied dependencies are remained:" >&2
			message_cat "${DBDIR}/unsatisfied.list"
			message_echo "*** Aborted by ${APPNAME}"
			message_echo "The ports tree seems broken. You might have caught an incomplete version."
			message_echo "You are encouraged to update the ports tree by portsnap(8)."
			message_echo "Then execute"
			message_echo " ${APPNAME} clean"
			message_echo "before restart."
			temp_terminate_process () { :; }
			exit 1
		fi
		message_echo
	}
	program_exec_and_record_completion ORDER_ALL_DEPENDENCIES
}

# ============= Selection of removing leaf ports =============
command_do_selection_of_removing_leaf_ports ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='INSPECT_REQUIREMENTS_OF_LEAF_PORTS'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Selection of removing leaf ports"
		deinstall_select_leaf_ports_to_delete
		message_echo
	}
	program_exec_and_record_completion SELECT_LEAF_PORTS
}

# ============= Selection of removing obsolete ports =============
command_do_selection_of_removing_obsolete_ports ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES PARSE_CONF'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Selection of removing obsolete ports"
		deinstall_select_obsolete_ports_to_delete
		message_echo
	}
	program_exec_and_record_completion SELECT_OBSOLETE_PORTS
}

# ============= Collection of leaf ports to delete =============
command_do_collection_of_leaf_ports_to_delete ()
{
	local PROGRAM_DEPENDS
	if [ ! -e "${DBDIR}/inspected_ports_only_partially" ]
	then
		PROGRAM_DEPENDS='SELECT_LEAF_PORTS'
		_program_exec_and_record_completion__operation ()
		{
			local src src_unselected reqptn_file src_with_initial_origins
			message_section_title "Collecting leaf ports to delete"
			src=${DBDIR}/leaf_ports
			src_unselected=${DBDIR}/leaf_ports_to_delete.unselected
			src_with_initial_origins=${DBDIR}/leaf_ports.with_ini
			reqptn_file=${DBDIR}/leaf_ports.requirements_of_unselected
			cat "$src_unselected" 2> /dev/null | while read origin
			do
				cat "${DBDIR}/requires/$origin/requirements.all.full" "${DBDIR}/requires/$origin/ignored_requirements.all" || :
			done 2> /dev/null | sort -u > $reqptn_file
			database_query_add_initial_origins < $src > $src_with_initial_origins
			message_echo
		}
		program_exec_and_record_completion COLLECT_LEAF_PORTS_TO_DELETE
	fi
}

# ============= Collection of obsolete ports to delete =============
command_do_collection_of_obsolete_ports_to_delete ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='SELECT_OBSOLETE_PORTS'
	_program_exec_and_record_completion__operation ()
	{
		local src src_selected src_unselected dst_selected reqptn_file
		message_section_title "Collecting obsolete ports to delete"
		src=${DBDIR}/obsolete_ports.can_be_deleted
		src_selected=${DBDIR}/obsolete_ports_to_delete.selected
		src_unselected=${DBDIR}/obsolete_ports_to_delete.unselected
		dst_selected=${DBDIR}/obsolete_ports_to_delete
		reqptn_file=${DBDIR}/obsolete_ports.requirements_of_unselected
		cat "$src_unselected" 2> /dev/null | while read origin
		do
			cat "${DBDIR}/initial/$origin/requirements.run.full" || :
			cat "${DBDIR}/obsolete/$origin/requirements.run.full" || :
		done | sort -u > $reqptn_file
		grep -v -Fx -f "$reqptn_file" "$src_selected" > $dst_selected 2> /dev/null || :
		message_echo
	}
	program_exec_and_record_completion COLLECT_OBSOLETE_PORTS_TO_DELETE
}

# ============= Set up the list of ports to reinstall =============
command_do_set_up_the_list_of_ports_to_reinstall ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='ORDER_ALL_DEPENDENCIES'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Setting up the list of ports to reinstall"
		cp -p "${DBDIR}/reinst_order.list" "${DBDIR}/stage.loop_list/reinst_todo"
		message_echo
	}
	program_exec_and_record_completion SETUP_REINST_TODO
}

# ============= Composition of a list for deinstallation of obsolete and leaf packages =============
command_do_composition_of_a_list_for_deinstallation_of_obsolete_and_leaf_packages ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='COLLECT_LEAF_PORTS_TO_DELETE COLLECT_OBSOLETE_PORTS_TO_DELETE'
	_program_exec_and_record_completion__operation ()
	{
		local reqptn_leaf reqptn_obs leaf_selected leaf_selected_src obs_selected obs_selected_src grepptn preserved
		message_section_title "Composing a list for deinstallation of obsolete and leaf packages"
		reqptn_leaf=${DBDIR}/leaf_ports.requirements_of_unselected
		reqptn_obs=${DBDIR}/obsolete_ports.requirements_of_unselected
		leaf_selected_src=${DBDIR}/leaf_ports_to_delete.selected
		leaf_selected=${DBDIR}/leaf_ports_to_delete
		obs_selected_src=${DBDIR}/obsolete_ports_to_delete.selected
		obs_selected=${DBDIR}/obsolete_ports_to_delete
		grepptn=${DBDIR}/ports_to_delete.grep_pattern
		grepptn_col1=${DBDIR}/ports_to_delete.grep_pattern_col1
		preserved=${TMPDIR}/LIST_DEINST_PKGS::preserved
		if [ ! -e "${DBDIR}/inspected_ports_only_partially" ]
		then
			cat "$reqptn_leaf" "$reqptn_obs" 2> /dev/null | sort -u > $grepptn
			grep -v -Fx -f "$grepptn" "$leaf_selected_src" 2> /dev/null \
				| database_query_add_initial_origins > $leaf_selected || :
			cat "$obs_selected" "$leaf_selected" 2> /dev/null || :
		else
			rm -f "$leaf_selected"
			cat "$obs_selected" 2> /dev/null
		fi | sort -u > ${DBDIR}/stage.loop_list/ports_to_delete
		str_escape_regexp_filter < ${DBDIR}/stage.loop_list/ports_to_delete \
			| sed 's/^/^/;s/$/[[:space:]]/' > $grepptn_col1
		cat "${DBDIR}/leaf_ports.with_ini" "${DBDIR}/obsolete_ports" 2> /dev/null \
			| grep -Fx -v -f "${DBDIR}/stage.loop_list/ports_to_delete" > ${DBDIR}/stage.loop_list/ports_to_restore 2> /dev/null || :
		if [ $opt_batch_mode = no ]
		then
			if [ ! -e "${DBDIR}/inspected_ports_only_partially" ] && \
				grep -v -Fx -f "${DBDIR}/stage.loop_list/ports_to_delete" "$leaf_selected_src" > $preserved 2> /dev/null
			then
				message_echo "INFO: Following leaf ports are preserved because required by other preserved leaf/obsolete ports."
				message_echo "----------------"
				while read origin
				do
					pkgtag=`cat "${DBDIR}/moved_from/$origin/pkgtag" 2> /dev/null` || :
					if [ -n "$pkgtag" ]
					then
						echo "$origin" "($pkgtag)"
					else
						echo "$origin"
					fi
				done < $preserved
				message_echo "----------------"
			fi
			if grep -v -Fx -f "${DBDIR}/stage.loop_list/ports_to_delete" "$obs_selected_src" > $preserved 2> /dev/null
			then
				message_echo "INFO: Following obsolete ports are preserved because required by other obsolete ports."
				message_echo "----------------"
				while read origin
				do
					pkgtag=`cat "${DBDIR}/initial/$origin/installed_version" 2> /dev/null` || :
					if [ -n "$pkgtag" ]
					then
						echo "$origin" "($pkgtag)"
					else
						echo "$origin"
					fi
				done < $preserved
				message_echo "----------------"
			fi
		fi
		message_echo
	}
	program_exec_and_record_completion LIST_DEINST_PKGS
}

# ============= Collect entire distfiles list =============
command_do_collect_entire_distfiles_list ()
{
	local PROGRAM_DEPENDS
	if [ $opt_inspect_entire_distinfo = yes ]
	then
		PROGRAM_DEPENDS=''
		_program_exec_and_record_completion__operation ()
		{
			message_section_title "Collecting entire distfiles list"
			find "${PORTSDIR}" -depth 3 -name distinfo -exec cat {} \; \
				| grep '^SHA256 ' | sed -E 's/^SHA256 \(([^)]*)\).*/\1/' \
				| sort -u > ${DBDIR}/distfiles.entire.tmp
			mv "${DBDIR}/distfiles.entire.tmp" "${DBDIR}/distfiles.entire"
			message_echo
		}
		program_exec_and_record_completion COLLECT_ALL_DISTFILES_LIST
	fi
}

# ============= Inspection of all required distfiles =============
command_do_inspection_of_all_required_distfiles ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='INSPECT_ALL_DEPENDENCIES COLLECT_ALL_DISTFILES_LIST'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Summarizing distfiles list"
		{
			cat "${DBDIR}/distfiles.entire" || :
			cat "${DBDIR}/distfiles.inspected" || :
		} 2> /dev/null | sort -u | str_escape_regexp_filter \
			| sed 's|^|^\\.\\/|; s|$|$|' > ${DBDIR}/distfiles.grep.pattern || :
		message_echo
	}
	program_exec_and_record_completion DISTFILES_LIST
}

# ============= Clean up of reinstallation status for preparation =============
command_do_clean_up_of_reinstallation_status_for_preparation ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='REDO_INIT INSPECT_ALL_DEPENDENCIES'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Cleaning up of reinstallation status for preparation"
		rm -rf "${DBDIR}/status.ports"
		message_echo
	}
	program_exec_and_record_completion CLEANUP_REINST_STATUS
}

# ============= Completion of building the temporary database =============
command_do_completion_of_building_the_temporary_database ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='REDO_INIT SETUP_REINST_TODO CLEANUP_REINST_STATUS PARSE_CONF INSPECT_ALL_DEPENDENCIES RESTORE_ESCAPED_OBSOLETE_PACKAGES_FOR_INSPECTION COMPLETE_NECESSARY_UPDATES:direct COMPLETE_NECESSARY_UPDATES:full PARSE_TARGET_ATTR_INFO MAKE_DEPENDENTS_LISTS_UNIQUE MAKE_IGNORED_DEPENDENTS_LISTS_UNIQUE COLLECT_LEAF_PORTS_TO_DELETE'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "The temporary database is completely built up"
		message_echo
	}
	program_exec_and_record_completion PREPARATION
}

# ============= End of do/redo prepare  =============
command_do_end_at_prepare_complete ()
{
	local tmp_msg
	tmp_msg=${TMPDIR}/command_do_end_at_prepare_complete:msg
	cat > "$tmp_msg" << eof
Done (skipped reinstallation) at `message_timestamp`

 You can restart this process from the aborted/terminated point by executing without options or arguments as:
  ${APPNAME}
eof
	message_cat "$tmp_msg"
	temp_terminate_process () { :; }
}

# ============= Reinstallation of remained ports =============
command_do_reinstallation_of_remained_ports ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='PREPARATION'
	_program_exec_restartable_loop_operation__routine ()
	{
		reinstall_exec "$@"
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
	}
	_program_exec_and_record_completion__operation ()
	{
		local _MSG_CURRENT_STAGE_general
		_MSG_CURRENT_STAGE_general="reinstallation"
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
		message_section_title "Reinstallation"
		program_exec_restartable_loop_operation reinst_todo
		[ "x$PROGRAM_STEP_COUNTER_LAST_SKIPPED" = xyes ] && message_stage_title "$PROGRAM_STEP_COUNTER"
		reinstall_restore_conflicts
		temp_set_msg_current_stage
		message_echo
	}
	program_exec_and_record_completion REINSTALLATION
}

# ============= Restoration of obsolete and leaf packages which have been deinstalled but unselected from the deletion list again  =============
command_do_restore_needed_obsolete_and_leaf_packages ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='REDO_INIT LIST_DEINST_PKGS'
	_program_exec_restartable_loop_operation__routine ()
	{
		deinstall_restore "$@"
	}
	_program_exec_and_record_completion__operation ()
	{
		local _MSG_CURRENT_STAGE_general
		_MSG_CURRENT_STAGE_general="restoration of unselected obsolete/leaf packages"
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
		message_section_title "Restoration of unselected obsolete/leaf packages"
		program_exec_restartable_loop_operation ports_to_restore
		[ "x$PROGRAM_STEP_COUNTER_LAST_SKIPPED" = xyes ] && message_stage_title "$PROGRAM_STEP_COUNTER"
		temp_set_msg_current_stage
		message_echo
	}
	program_exec_and_record_completion RESTORE_ONCE_DEINST_PKGS
}

# ============= Deinstallation of unused obsolete and leaf packages =============
command_do_deinstallation_of_unused_obsolete_and_leaf_packages ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='REDO_INIT LIST_DEINST_PKGS'
	_program_exec_restartable_loop_operation__routine ()
	{
		deinstall_exec "$@"
	}
	_program_exec_and_record_completion__operation ()
	{
		local _MSG_CURRENT_STAGE_general
		_MSG_CURRENT_STAGE_general="deinstallation of obsolete/leaf packages"
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
		message_section_title "Deinstallation of unused obsolete/leaf packages"
		program_exec_restartable_loop_operation ports_to_delete
		[ "x$PROGRAM_STEP_COUNTER_LAST_SKIPPED" = xyes ] && message_stage_title "$PROGRAM_STEP_COUNTER"
		temp_set_msg_current_stage
		message_echo
	}
	program_exec_and_record_completion DEINST_UNUSED_PKGS
}

# ============= Clean up obsolete or unused distfiles =============
command_do_clean_up_obsolete_or_unused_distfiles ()
{
	local PROGRAM_DEPENDS
	if [ $opt_only_target_scope = no -a $opt_keep_distfiles = no ]
	then
		PROGRAM_DEPENDS='REINSTALLATION DISTFILES_LIST'
		_program_exec_and_record_completion__operation ()
		{
			local tmp_distfiles_exists
			message_section_title "Cleaning up obsolete or unused distfiles"
			tmp_distfiles_exists=${TMPDIR}/CLEANUP_OBSLETE_DISTFILES::distfiles_exists
			[ $opt_dry_run = yes ] && message_echo "INFO: The operations are not actually carried out."
			( set -e; cd "${DISTDIR}" && find . -type f ) \
				| sed 's|^\./||' | sort -u > $tmp_distfiles_exists
			fileedit_manipulate_old_lines "$tmp_distfiles_exists" "${DBDIR}/distfiles.entire" \
				| while read distfile
			do
				if [ $opt_batch_mode = no ]
				then
					echo "  $distfile"
				fi
				[ $opt_dry_run = yes ] && continue
				rm -f "${DISTDIR}/$distfile"
			done
			message_echo
		}
		program_exec_and_record_completion CLEANUP_OBSLETE_DISTFILES
	fi
}

# ============= Rebuild of package database =============
command_do_rebuild_of_package_database ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='REINSTALLATION RESTORE_ONCE_DEINST_PKGS DEINST_UNUSED_PKGS'
	_program_exec_and_record_completion__operation ()
	{
		which -s pkgdb || return 0
		message_section_title "Rebuilding package database for portupgrade"
		if grep -q @ "${DBDIR}/stage.loop_list/reinst_todo"
		then
			message_echo "INFO: Skipped because of lacking flavor support."
		else
			pkgdb -fu
		fi
		message_echo
	}
	program_exec_and_record_completion REBUILD_PKGDB
}

# =============Operation of redo command irrespective of option settings =============
command_do_redo__command_all_exec_irrespective_of_saved_options ()
{
	[ $opt_reload_conf = yes ] || return 0
	program_deregister_stage_complete SAVE_PREV_CONF
	program_deregister_stage_complete ALL_COMPLETE
	rm -f "${DBDIR}"/complete
}

# ============= Preparation of the temporary database =============
command_do_prepare ()
{
	# Meta process for redo
	command_do_meta_process_for_redo
	
	# Determine specified targets
	command_do_determine_specified_targets
	
	# Show specified targets
	command_do_show_specified_targets
	
	# Determine all target ports
	command_do_determine_all_target_ports
	
	# Inspection of all initial dependencies
	command_do_inspection_of_all_initial_dependencies
	
	# Loop considering cases that port options are changed after inspection
	while true
	do
		# Prepare for inspecting all dependencies
		command_do_prepare_for_inspect_all_dependencies
		
		# Inspection of all dependencies
		command_do_inspection_of_all_dependencies
		
		# Convert dependency-lists to actual ones
		command_do_convert_dependency_lists_to_actual_ones
		
		# Completion of recursive requirement lists
		command_do_completion_of_recursive_requirement_lists
		
		# Trim dependency lists by removing uninspected ports
		command_do_trim_dependency_lists_by_removing_uninspected_ports
		
		# Inspection of dependents
		command_do_inspection_of_dependents
		
		# End the loop when no change is made on port options after the inspection
		command_do_reset_changed_portdb && break
	done
	
	# Restore escaped obsolete packages for inspection
	command_do_restore_escaped_obsoletes
	
	# Remove duplicated lines in dependents lists
	command_do_remove_duplicated_lines_in_dependents_lists
	
	# Remove duplicated lines in ignored dependents lists
	command_do_remove_duplicated_lines_in_ignored_dependents_lists
	
	# Preparation of target attribute information
	command_do_preparation_of_target_attribute_information
	
	# Post-process after the preparation of target attribute information
	command_do_post_process_after_the_preparation_of_target_attribute_information
	
	# Build of data on complement to new dependents for target attribute information
	command_do_build_of_data_on_complement_to_new_dependents_for_target_attribute_information
	
	# Parse target attribute information
	command_do_parse_target_attribute_information
	
	# Inspection of necessity
	command_do_inspection_of_necessity
	
	# Inspection of necessary upgrades
	command_do_inspection_of_necessary_upgrades
	
	# Completion of necessary upgrades for build
	command_do_complete_necessary_upgrades_for_build
	
	# Preparation for inspection of new leaf ports
	command_do_preparation_for_inspection_of_new_leaf_ports
	
	# Inspection of new primary leaf ports
	command_do_inspection_of_new_primary_leaf_ports
	
	# Inspection of requirements of new leaf ports
	command_do_inspection_of_requirements_of_new_leaf_ports
	
	# Order the ports considering dependencies
	command_do_order_the_ports_considering_dependencies
	
	# Selection of removing leaf ports
	command_do_selection_of_removing_leaf_ports
	
	# Selection of removing obsolete ports
	command_do_selection_of_removing_obsolete_ports
	
	# Collection of leaf ports to delete
	command_do_collection_of_leaf_ports_to_delete
	
	# Collection of obsolete ports to delete
	command_do_collection_of_obsolete_ports_to_delete
	
	# Set up the list of ports to reinstall
	command_do_set_up_the_list_of_ports_to_reinstall
	
	# Composition of a list for deinstallation of obsolete and leaf packages
	command_do_composition_of_a_list_for_deinstallation_of_obsolete_and_leaf_packages
	
	# Collect entire distfiles list
	command_do_collect_entire_distfiles_list
	
	# Inspection of all required distfiles
	command_do_inspection_of_all_required_distfiles
	
	# Clean up of reinstallation status for preparation
	command_do_clean_up_of_reinstallation_status_for_preparation
	
	# Completion of building the temporary database
	command_do_completion_of_building_the_temporary_database
}

# ============= Reset the progress of reinstallation for retrial =============
command_do_reset_for_reinst_retrial ()
{
	program_deregister_stage_complete REINSTALLATION
	program_deregister_stage_complete RESTORE_ONCE_DEINST_PKGS
	program_deregister_stage_complete DEINST_UNUSED_PKGS
	rm -rf "${DBDIR}/status.ports" "${DBDIR}/new_success_in_current_run"
}

# ============= Main operation of do/redo =============
command_do_main ()
{
	local ntrial
	# Reset termination messages
	temp_reset_termination_messages_common
	
	ntrial=0
	while true
	do
		if [ $ntrial -gt 0 ]
		then
			message_echo "########## RETRIAL ($ntrial) OF (RE)INSTALLATION ##########"
			message_echo
		fi
		
		if [ $opt_fetch_only = yes ]
		then
			# Reinstallation of remained ports
			command_do_reinstallation_of_remained_ports
			
			message_echo "Completed the fetch only mode."
			exit
		elif [ $opt_delete_then_reinstall = no ]
		then
			# Reinstallation of remained ports
			command_do_reinstallation_of_remained_ports
			
			# Restoration of obsolete and leaf packages which have been deinstalled but unselected from the deletion list again 
			command_do_restore_needed_obsolete_and_leaf_packages
			
			# Deinstallation of unused obsolete and leaf packages
			command_do_deinstallation_of_unused_obsolete_and_leaf_packages
		else
			# Deinstallation of unused obsolete and leaf packages
			command_do_deinstallation_of_unused_obsolete_and_leaf_packages
			
			# Reinstallation of remained ports
			command_do_reinstallation_of_remained_ports
			# Restoration of obsolete and leaf packages which have been deinstalled but unselected from the deletion list again 
			
			command_do_restore_needed_obsolete_and_leaf_packages
		fi
		
		# Clean up obsolete or unused distfiles
		command_do_clean_up_obsolete_or_unused_distfiles
		
		# Rebuild of package database
		command_do_rebuild_of_package_database
		
		# Retry if incomplete
		command_do_is_everything_resolved && break
		database_query_is_any_progress || break
		command_do_reset_for_reinst_retrial
		ntrial=$(($ntrial+1))
	done
	:
}

# ============= Check whether everything is resolved =============
command_do_is_everything_resolved ()
{
	local subject
	for subject in failure redo conflict
	do
		database_query_show_single_list_exec "$subject" \
			`options_get_dependency_type` `options_get_dependency_level` > /dev/null 2> /dev/null && return 1
	done
	:
}

# ============= Notice of failures =============
command_do_failure_notice ()
{
	local exists_unresolved_ports
	exists_unresolved_ports=
	message_summary_dependents_of_failed_reinstallation failure || exists_unresolved_ports=y
	message_summary_dependents_of_failed_reinstallation redo || exists_unresolved_ports=y
	message_summary_dependents_of_failed_reinstallation conflict || exists_unresolved_ports=y
	[ "x$exists_unresolved_ports" = xy ] || return 0
	message_summary_advice_on_manual_solution
	return 1
}

# ============= Ending process =============
command_do_ending_process ()
{
	local PROGRAM_DEPENDS
	temp_terminate_process () { :; }
	if command_do_failure_notice
	then
		if [ $opt_no_opening_message = yes ]
		then
			message_echo "Done as ${APPNAME}"
		else
			message_section_title "COMPLETELY DONE"
		fi
		if [ $opt_fetch_only = no -a $opt_dry_run = no ]
		then
			PROGRAM_DEPENDS='REBUILD_PKGDB CLEANUP_OBSLETE_DISTFILES '
			_program_exec_and_record_completion__operation ()
			{
			}
			program_exec_and_record_completion ALL_COMPLETE
			[ $opt_no_opening_message = yes ] || message_echo "- E N D -"
		else
			message_echo "INFO: Redo for the real (re)installation."
		fi
	else
		message_warn_no_achieved_progress
		message_section_title "Done with some unresolved problems"
		message_echo "- To be continued -"
	fi
}
