#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Operations for building the temporary database -
# Copyright (C) 2013-2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Register an obsolete flavored origin =============
database_build_register_obsolete_port ()
{
	local origin dbpath pkgtag
	origin=$1
	dbpath=${DBDIR}/obsolete/$origin
	[ -e "$dbpath/complete_as_node" ] && return
	mkdir -p "${DBDIR}/obsolete/$origin"
	pkgtag=`pkgsys_get_init_pkg_from_orig "$origin"`
	[ -n "$pkgtag" ] || pkgtag='[not installed]'
	echo "$pkgtag" > ${DBDIR}/obsolete/$origin/pkgtag
	for table in dependents requirements
	do
		for level in direct full
		do
			for tag in all run build
			do
				srcfile=${DBDIR}/initial/$origin/${table}.${tag}.${level}
				[ -e "$srcfile" ] && ln -f "$srcfile" "$dbpath/${table}.${tag}.${level}.src"
			done
		done
	done
	touch "$dbpath/complete_as_node"
}

# ============= Convert and register if a flavored origin is obsolete =============
database_build_convert_and_register_origin_if_obsolete ()
{
	local origin recursedb_in recursedb output_origin iline_db origin_new date_moved why_moved
	origin=$1
	recursedb_in=$2
	recursedb=${recursedb_in:-${PORTS_MOVED_DB}}
	echo "$origin" > ${TMPDIR}/database_build_convert_and_register_origin_if_obsolete:origin
	pkgsys_exists_port "$origin" && return
	database_build_register_obsolete_port "$origin"
	grep -n -m 1 -E "^`str_escape_regexp \"$origin\"`\|" "$recursedb" 2> /dev/null > ${TMPDIR}/moved.info || :
	if [ `wc -l < ${TMPDIR}/moved.info` -eq 0 ]
	then
		fileedit_add_a_line_if_new "$origin" "${DBDIR}/obsolete_ports"
		fileedit_rm_a_line "$origin" "${DBDIR}/moved_ports"
		if [ -n "$recursedb_in" ]
		then
			message_echo "${DEPTH_INDEX}  ===> Disappeared port (MOVED broken?)"
		else
			message_echo "${DEPTH_INDEX}  ===> Nonexistent port (your original?)"
		fi
		return 1
	else
		iline_db=`cut -d : -f 1 "${TMPDIR}/moved.info"`
		sed 1,${iline_db}d "${PORTS_MOVED_DB}" > ${TMPDIR}/MOVED.DB
		origin_new=`sed -E 's/^[0-9]+://' "${TMPDIR}/moved.info" | cut -d '|' -f 2 || :`
		date_moved=`cut -d '|' -f 3 "${TMPDIR}/moved.info" || :`
		why_moved=`cut -d '|' -f 4 "${TMPDIR}/moved.info" || :`
		if [ -n "$origin_new" ]
		then
			message_echo "${DEPTH_INDEX}  ===> Moved to $origin_new at $date_moved because \"$why_moved\""
			fileedit_add_a_line_if_new "$origin" "${TMPDIR}/database_build_convert_and_register_origin_if_obsolete:origins_old"
			fileedit_add_a_line_if_new "$origin" "${DBDIR}/moved_ports"
			fileedit_rm_a_line "$origin" "${DBDIR}/obsolete_ports"
			database_build_convert_and_register_origin_if_obsolete "$origin_new" "${TMPDIR}/MOVED.DB" || return 1
		else
			message_echo "${DEPTH_INDEX}  ===> Deleted at $date_moved because \"$why_moved\""
			fileedit_add_a_line_if_new "$origin" "${DBDIR}/obsolete_ports"
			fileedit_rm_a_line "$origin" "${DBDIR}/moved_ports"
			return 1
		fi
	fi
}

# ============= [Sub-function] Get the true latest flavored origin =============
database_build_convert_and_register_origin_if_obsolete__get_origin ()
{
	cat "${TMPDIR}/database_build_convert_and_register_origin_if_obsolete:origin"
}

# ============= [Sub-function] Reset the all old origins of the tested flavored origin =============
database_build_convert_and_register_origin_if_obsolete__reset_origins_old ()
{
	rm -rf "${TMPDIR}/database_build_convert_and_register_origin_if_obsolete:origins_old"
}

# ============= [Sub-function] Save the all old origins of the tested flavored origin =============
database_build_convert_and_register_origin_if_obsolete__save_origins_old ()
{
	cat "${TMPDIR}/database_build_convert_and_register_origin_if_obsolete:origins_old" 2> /dev/null
}

# ============= Get the make arguments for building the temporary database =============
database_build_setup_make_args ()
{
	local origin
	origin=$1
	{
		for key in LOCALBASE LINUXBASE PORT_DBDIR PORTSDIR DISTDIR PACKAGES PKGREPOSITORY
		do
			eval echo $key=\$$key
		done
		echo 'DISABLE_VULNERABILITIES=yes'
		if [ $opt_apply_default_config = yes ]
		then
			if pkgsys_is_dialog4ports_used
			then
				echo 'NO_DIALOG=yes'
			else
				echo 'BATCH=yes'
			fi
		fi
		dbdir=${DBDIR}/requires/$origin
		[ -d "$dbdir" ] || dbdir=${DBDIR}/conf/each_port/$origin
		cat "$dbdir/MARG.conf" 2> /dev/null || :
		flavor=`pkgsys_get_flavor_from_origin "$origin"`
		[ -z "$flavor" ] || echo "FLAVOR=$flavor"
	} | tr '\n' ' '
}

# ============= Get the make environment variables for building the temporary database =============
database_build_setup_make_envs ()
{
	local origin dbdir
	origin=$1
	dbdir=${DBDIR}/requires/$origin
	[ -d "$dbdir" ] || dbdir=${DBDIR}/conf/each_port/$origin
	cat "$dbdir/MENV.conf" 2> /dev/null | tr '\n' ' '
}

# ============= Execute make command for building the temporary database =============
database_build_make ()
{
	local origin MAKE_ARGS MAKE_ENVS port_path flavor
	origin=$1
	shift
	MAKE_ARGS=`database_build_setup_make_args "$origin"`
	MAKE_ENVS=`database_build_setup_make_envs "$origin"`
	port_path=`pkgsys_get_portpath_from_origin "$origin"`
	fs_fix_unionfs_image_if_hidden "$port_path"
	env $MAKE_ENVS make -C "$port_path" "$@" $MAKE_ARGS
}

# ============= Set up a temporary database node for the initial state of a port =============
database_build_setup_initial_node ()
{
	local origin pkg dbpath
	origin=$1
	dbpath=${DBDIR}/initial/$origin
	[ -e "$dbpath/complete_as_node" ] && return
	rm -rf "$dbpath"
	mkdir -p "$dbpath"
	pkg=`pkgsys_get_installed_pkg_from_origin "$origin"`
	if [ -n "$pkg" ]
	then
		echo "$pkg" > $dbpath/installed_version
		ln -f "$dbpath/installed_version" "$dbpath/pkgtag"
		pkg_info_qr "$pkg" | while read requirement
		do
			origin_requirement=`pkgsys_init_pkg_orig_by_ambiguous_matching "$requirement" || :`
			[ -n "$origin_requirement" ] && echo "$origin_requirement"
			:
		done > $dbpath/requirements.all.full
		pkg_info_qR "$pkg" | while read dependent
		do
			origin_dependent=`pkgsys_init_pkg_orig_by_ambiguous_matching "$dependent" || :`
			[ -n "$origin_dependent" ] && echo "$origin_dependent"
			:
		done > $dbpath/dependents.all.full
		for table in dependents requirements
		do
			for level in direct full
			do
				for tag in all run build
				do
					[ "${tag}.${level}" = all.full ] && continue
					ln -f "$dbpath/${table}.all.full" "$dbpath/${table}.${tag}.${level}"
				done
			done
		done
	fi
	if [ `expr "$pkg" : "^${APPNAME}-[0-9].*"` -gt 0 ]
	then
		[ -d "$dbpath" ] && touch "$dbpath/SUPPRESSED_SELF"
	elif [ `expr "$pkg" : "^pkg-[0-9].*"` -gt 0 ]
	then
		[ -d "$dbpath" ] && touch "$dbpath/SUPPRESSED_PKGNG"
	fi
	touch "$dbpath/complete_as_node"
}

# ============= Set up a temporary database node for the replaced/moved information of a port =============
database_build_setup_replace_node ()
{
	local origin_orig portoption_before portoption_after origin_trial origin dbpath tmp_msg
	origin_orig=$1
	portoption_before=$2
	portoption_after=$3
	dbpath=${DBDIR}/replace/$origin_orig
	tmp_msg=${TMPDIR}/database_build_setup_replace_node:meg
	if [ ! -e "$dbpath/complete_as_node" ]
	then
		rm -rf "$dbpath"
		mkdir -p "$dbpath"
		origin_trial=$origin_orig
		if echo "$origin_orig" | grep -q -Fx -f "${DBDIR}/conf/REPLACE.filter" 2> /dev/null
		then
			origin_trial=`echo "$origin_orig" | sed -E -f "${DBDIR}/conf/REPLACE.sed_pattern"`
			if [ "x$origin_orig" != "x$origin_trial" ]
			then
				if [ -n "$origin_trial" ]
				then
					message_echo "${DEPTH_INDEX}  ===> Replaced with $origin_trial by user configuration"
				else
					database_build_register_obsolete_port "$origin_orig"
					message_echo "${DEPTH_INDEX}  ===> Deleted by user configuration"
				fi > $dbpath/message
			fi
		fi
		if [ -n "$origin_trial" ]
		then
			if database_build_convert_and_register_origin_if_obsolete "$origin_trial" >> $dbpath/message
			then
				origin_trial=`database_build_convert_and_register_origin_if_obsolete__get_origin`
			else
				if [ "x$origin_trial" != "x$origin_orig" ]
				then
					message_echo "${DEPTH_INDEX}  ===> Going back to the original port $origin_orig"
					if database_build_convert_and_register_origin_if_obsolete "$origin_orig"
					then
						origin_trial=`database_build_convert_and_register_origin_if_obsolete__get_origin`
					else
						origin_trial=
					fi
				else
					origin_trial=
				fi >> $dbpath/message
			fi
		fi
		cat "$dbpath/message" 2> /dev/null || :
		if [ -n "$origin_trial" ]
		then
			database_build_make "$origin_trial" showconfig > $portoption_before 2> /dev/null || :
			database_build_determine_port_option "$origin_trial"
			database_build_make "$origin_trial" showconfig > $portoption_after 2> /dev/null || :
			origin=`database_build_determine_flavored_origin "$origin_trial"`
			if [ "x$origin_trial" != "x$origin" ]
			then
				if [ -n "$origin" ]
				then
					message_echo "${DEPTH_INDEX}  ===> Transferred to $origin"
				else
					fileedit_add_a_line_if_new "$origin" "${DBDIR}/obsolete_ports"
					message_echo "${DEPTH_INDEX}  ===> Lost (Something is wrong!)"
				fi
			fi > $tmp_msg
			cat "$tmp_msg" >> $dbpath/message
		else
			origin=
		fi
		[ "x$origin_orig" = "x$origin" ] || echo "$origin" > $dbpath/origin
		cat "$tmp_msg" 2> /dev/null || :
		touch "$dbpath/complete_as_node"
	else
		cat "$dbpath/message" 2> /dev/null || :
	fi
}

# ============= Get the inspected level for a port with the current option settings =============
database_build_get_inspected_level ()
{
	local origin origin_dependent
	origin=$1
	origin_dependent=$2
	if [ $opt_only_target_scope = no ]
	then
		echo full
	elif ! pkgsys_exists_or_existed_from_orig "$origin" \
		|| grep -qFx "$origin" "${DBDIR}/stage.loop_list/ports_to_inspect" 2> /dev/null \
		|| [ ! -e "${DBDIR}/moved_from/$origin_dependent/installed_version" ] \
		|| grep -qFx "$origin_dependent" "${DBDIR}/stage.loop_list/ports_to_inspect" 2> /dev/null
	then
		echo direct
	else
		echo node
	fi
}

# ============= Check whether a port has been inspected in a required level =============
database_build_is_port_already_inspected_in_required_level ()
{
	local origin origin_dependent origin_actual inspected_level
	origin=$1
	origin_dependent=$2
	origin_actual=`echo "$origin" | sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" 2> /dev/null || :`
	[ -n "$origin_actual" ] || origin_actual=$origin
	inspected_level=`database_build_get_inspected_level "$origin_actual" "$origin_dependent"`
	{
		cat "${DBDIR}/ports.inspected.${inspected_level}.list" || :
		cat "${DBDIR}/moved_ports" || :
		cat "${DBDIR}/obsolete_ports" || :
	}  2> /dev/null | grep -q -Fx "$origin_actual" || return
	fileedit_rm_a_line "$origin" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
	[ "x$origin" = "$origin_actual" ] || \
		fileedit_rm_a_line "$origin_actual" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
	echo "$origin_actual"
}

# ============= Update the current package name of a port =============
database_build_update_pkgname ()
{
	local origin dbdir pkg savefile
	origin=$1
	dbdir=${DBDIR}/moved_from/$origin
	mkdir -p "$dbdir"
	savefile=$dbdir/current_version
	pkg=`pkgsys_get_installed_pkg_from_origin "$origin"`
	if [ -z "$pkg" -a -e "$dbdir/initial_orig" ]
	then
		while read orig_init
		do
			pkgsys_get_installed_pkg_from_origin "$orig_init"
		done < $dbdir/initial_orig
	elif [ -n "$pkg" ]
	then
		echo "$pkg"
	fi | sort -u > $savefile
	cat "$savefile"
}

# ============= Get the current package name of a port =============
database_build_get_pkgname ()
{
	local origin savefile
	origin=$1
	savefile=${DBDIR}/moved_from/$origin/current_version
	if [ -e "$savefile" ]
	then
		cat "$savefile"
	else
		database_build_update_pkgname "$origin"
	fi
}

# ============= Get the new package name of a port =============
database_build_get_new_pkgname ()
{
	local origin savefile
	origin=$1
	savefile=${DBDIR}/requires/$origin/new_version
	if [ ! -e "$savefile" ]
	then
		database_build_make "$origin" package-name \
			| tr -d '\n' > $savefile
	fi
	cat "$savefile"
}

# ============= Get the package list of full-run-time requirements =============
database_build_get_full_run_requirement_pkgs ()
{
	local origin srcfile savefile
	origin=$1
	srcfile=${DBDIR}/requires/$origin/requirements.run.full
	savefile=${DBDIR}/requires/$origin/pkg_requirements.run.full
	if [ ! -e "$srcfile" ]
	then
		rm -f "$savefile"
	elif [ ! -e "$savefile" -o "$savefile" -ot "$srcfile" ]
	then
		while read origin
		do
			database_build_get_new_pkgname "$origin"
		done < $srcfile | grep -v '^[[:space:]]*$' | sort -u > $savefile
	fi
	cat "$savefile" 2> /dev/null || :
}

# ============= Build the original message tag showing the version upgrade =============
database_build_create_pkgtag ()
{
	local origin dbdir pkg_init pkg_new pkgtag
	origin=$1
	dbdir=${DBDIR}/moved_from/$origin
	mkdir -p "$dbdir"
	pkg_init=`database_build_get_pkgname "$origin"`
	[ -n "$pkg_init" -a ! -e "$dbdir/installed_version" ] && \
		echo -n "$pkg_init" > $dbdir/installed_version
	pkg_new=`database_build_get_new_pkgname "$origin"`
	pkgtag=$pkg_init
	[ -n "$pkgtag" ] || pkgtag=$pkg_new
	if [ -z "$pkgtag" ]
	then
		pkgtag='?'
	
	elif [ "x$pkg_init" != "x$pkg_new" ]
	then
		if [ -n "$pkg_init" ]
		then
			pkgtag="$pkg_init => $pkg_new"
		else
			pkgtag="[new] $pkg_new"
		fi
	fi
	echo "$pkgtag" > $dbdir/pkgtag.orig
	echo "$pkgtag" > $dbdir/pkgtag
}

# ============= Update the message tag showing the version upgrade =============
database_build_update_pkgtag ()
{
	local origin dbdir pkg_init pkg_bak pkg_cur detail pkgtag
	origin=$1
	dbdir=${DBDIR}/moved_from/$origin
	mkdir -p "$dbdir"
	pkg_init=`cat "$dbdir/installed_version" 2> /dev/null | tr '\n' ' ' | sed 's/ *$//'`
	pkg_bak=`cat "$dbdir/backedup_version" 2> /dev/null | tr '\n' ' ' | sed 's/ *$//'`
	pkg_cur=`database_build_get_pkgname "$origin" | tr '\n' ' ' | sed 's/ *$//'`
	detail=
	if [ "x$pkg_init" != "x$pkg_cur" ]
	then
		if [ -n "$pkg_cur" ]
		then
			detail=" [currently installed version: $pkg_cur]"
		elif [ -n "$pkg_bak" ]
		then
			detail=" [currently deinstalled, previously installed version: $pkg_bak]"
		else
			detail=" [currently deinstalled]"
		fi
	fi
	pkgtag=`cat "$dbdir/pkgtag.orig"`
	echo "$pkgtag$detail" > $dbdir/pkgtag
}

# ============= Check whether the currently installed package version is the latest =============
database_build_is_currentpkg_latest ()
{
	local origin pkg_cur pkg_new
	origin=$1
	pkg_cur=`database_build_get_pkgname "$origin" | tr '\n' ' ' | sed 's/ *$//'`
	pkg_new=`database_build_get_new_pkgname "$origin"`
	[ "x$pkg_cur" = "x$pkg_new" ]
}
# ============= Configure the port option to determine the true flavor =============
database_build_determine_port_option ()
{
	local origin
	origin=$1
	if [ $opt_apply_default_config = yes ]
	then
		if ! pkgsys_is_dialog4ports_used
		then
			printf '\t\n' | database_build_make "$origin" config-conditional > /dev/null
		fi
	else
		database_build_make "$origin" config-conditional
	fi
}

# ============= Get the true flavor to determine the flavored port origin =============
database_build_determine_flavored_origin ()
{
	local origin origin_unflavored flavor_trial origin_trial flavors_ptn flavor
	origin=$1
	origin_unflavored=`pkgsys_get_unflavored_origin "$origin"`
	flavor_trial=`pkgsys_get_flavor_from_origin "$origin"`
	if [ -n "$flavor_trial" ]
	then
		flavors_ptn="^("`database_build_make "$origin" -V FLAVORS 2> /dev/null | sed -E 's/[[:space:]]+/|/g'`")$"
		echo "$flavor_trial" | grep -qE "$flavors_ptn" || flavor_trial=
	fi
	origin_trial=`pkgsys_compose_flavored_origin "$origin_unflavored" "$flavor_trial"`
	flavor=`database_build_make "$origin_trial" -V FLAVOR 2> /dev/null || :`
	pkgsys_compose_flavored_origin "$origin_unflavored" "$flavor"
}

# ============= Escape of inspection conflict =============
database_build_escape_inspect_conflict ()
{
	local origin dbpath
	origin=$1
	dbpath=${DBDIR}/requires/$origin
	if [ -e "$dbpath/CONFLICT.conf" ]
	then
		message_echo "${DEPTH_INDEX}  ===> Escaping inspection conflict..."
		pkgsys_eval_ports_glob `cat "$dbpath/CONFLICT.conf"` | database_build_escape "$dbpath/CONFLICT_pkgarc.lst"
	fi
}

# ============= Escape of conflicts =============
# A list of conflict package origins are given by stdin.
database_build_escape ()
{
	local backup_list origin_conflict pkg_conflict pkgarc
	backup_list=$1
	while read origin_conflict
	do
		pkg_conflict=`pkgsys_get_installed_pkg_from_origin "$origin_conflict"`
		[ -n "$pkg_conflict" ] || continue
		message_echo "${DEPTH_INDEX}  ===> Escaping $pkg_conflict..."
		pkgarc=`pkgsys_create_backup_pkg "$pkg_conflict" "${DBDIR}/backup_packages" 2> /dev/null` || \
			message_echo "${DEPTH_INDEX}  ===> (WARNING: Failed to back up)"
		pkg_delete_f "$pkg_conflict" > /dev/null 2>&1 || \
			message_echo "${DEPTH_INDEX}  ===> (WARNING: Failed to delete)"
		[ -n "$pkgarc" ] && printf '%s\t%s\n' "$pkg_conflict" "$pkgarc" >> $backup_list
	done
}

# ============= Restoration of escaped inspection conflict =============
database_build_restore_inspect_conflict ()
{
	local origin dbpath
	origin=$1
	dbpath=${DBDIR}/requires/$origin
	if [ -e "$dbpath/CONFLICT.conf" ]
	then
		message_echo "${DEPTH_INDEX}  ===> Restoring inspection conflict..."
		cat "$dbpath/CONFLICT_pkgarc.lst" 2> /dev/null | database_build_restore
		rm -f "$dbpath/CONFLICT_pkgarc.lst"
	fi
}

# ============= Restoration of escaped conflicts =============
# The backup list is given by stdin.
database_build_restore ()
{
	local pkg_conflict pkgarc
	while read pkg_conflict pkgarc
	do
		pkg_info_e "$pkg_conflict" && continue
		message_echo "${DEPTH_INDEX}  ===> Restoring $pkg_conflict..."
		pkg_add_f "$pkgarc" > /dev/null 2>&1 || \
			message_echo "${DEPTH_INDEX}  ===> (WARNING: Failed to restore $pkg_conflict)"
	done
}

# ============= Recursively inspect dependencies of a port and build a node database of the information =============
database_build_inspect_dependencies ()
{
	local origin_orig origin_dependent origin_orig_regexp origin_replace origin tag stage level dbpath dbpath_prev origin_id tmp_config origin_dependency DEPTH_INDEX_orig nlines iline dist_subdir_rpl inspected_level inspected_levels_compatible origin_tmp inspected_level_tmp conf_updated tmp_portsdb_work tmp_portopt same_as_prevset dbfile installed_version frompath variable pkg_new
	origin_orig=$1
	origin_dependent=$2
	[ -z "$origin_orig" ] && return
	if ! origin=`database_build_is_port_already_inspected_in_required_level "$origin_orig" "$origin_dependent"`
	then
		DEPTH_INDEX_orig=${DEPTH_INDEX}
		DEPTH_INDEX="${DEPTH_INDEX}--"
		message_echo "${DEPTH_INDEX} $origin_orig"
		database_build_convert_and_register_origin_if_obsolete__reset_origins_old
		origin_id=`echo "$origin_orig" | tr / :`
		# Replacement specified by the configuration file, knobs and port options
		tmp_config=${TMPDIR}/database_build_inspect_dependencies:config
		database_build_setup_replace_node "$origin_orig" "$tmp_config.before" "$tmp_config.after"
		if [ -e "${DBDIR}/replace/$origin_orig/origin" ]
		then
			origin=`cat "${DBDIR}/replace/$origin_orig/origin"`
		else
			origin=$origin_orig
		fi
		# Build the database for this port
		inspected_level=
		inspected_levels_compatible=
		if [ -n "$origin" ]
		then
			frompath=${DBDIR}/moved_from/$origin
			mkdir -p "$frompath"
			database_build_convert_and_register_origin_if_obsolete__save_origins_old | \
				fileedit_add_lines_if_new "$frompath/old_origs"
			# Check the package name initially installed
			installed_version=`cat "${DBDIR}/initial/$origin_orig/installed_version" 2> /dev/null || :`
			[ -n "$installed_version" ] && fileedit_add_a_line_if_new "$installed_version" "$frompath/installed_version"
			[ "x$origin_orig" = "x$origin" ] || fileedit_add_a_line_if_new "$origin_orig" "$frompath/initial_orig"
			[ -e "${DBDIR}/initial/$origin_orig/SUPPRESSED_SELF" ] && touch "$frompath/SUPPRESSED_SELF"
			[ -e "${DBDIR}/initial/$origin_orig/SUPPRESSED_PKGNG" ] && touch "$frompath/SUPPRESSED_PKGNG"
			if ! grep -qFx "$origin" "${DBDIR}/done_required_ports_to_inspect" 2> /dev/null
			then
				fileedit_rm_a_line "$origin" "${DBDIR}/obsolete_ports"
				fileedit_rm_a_line "$origin" "${DBDIR}/moved_ports"
				dbpath=${DBDIR}/requires/$origin
				dbpath_prev=${DBDIR}/prevset/requires/$origin
				if [ ! -e "$dbpath/complete_as_node" ] || \
					! diff "$tmp_config.before" "$tmp_config.after" > /dev/null 2> /dev/null
				then
					# Notify reconfiguration of the port option
					conf_updated=
					if [ -e "$dbpath/complete_as_node" ] || \
						grep -q -Fx -e "$origin_orig" -e "$origin" "${DBDIR}/to_be_reconf" 2> /dev/null
					then
						message_echo "${DEPTH_INDEX}  ===> Reconfigured"
						conf_updated=y
					fi
					# Reset the database
					rm -rf "$dbpath"
					mkdir -p "$dbpath"
					[ -n "$conf_updated" ] && touch "$dbpath/conf_updated"
					if [ -d "${DBDIR}/conf/each_port/$origin" ]
					then
						cp -R "${DBDIR}/conf/each_port/$origin/"* "$dbpath/" > /dev/null 2>&1 || :
					fi
					# Escape of inspection conflict
					database_build_escape_inspect_conflict "$origin"
					# Check the change of the port option from the default
					tmp_portsdb_work=${TMPDIR}/database_build_inspect_dependencies:portsdb_work
					tmp_portopt=${TMPDIR}/database_build_inspect_dependencies:portopt
					[ -d "$tmp_portsdb_work" ] || mkdir "$tmp_portsdb_work"
					( set -e
						export PORT_DBDIR=$tmp_portsdb_work
						export __MAKE_CONF=
						if database_build_make "$origin" showconfig > $tmp_portopt
						then
							cp "$tmp_portopt" "$dbpath"/ports_options.default
						else
							cat "$tmp_portopt" >&2
							message_echo 'Error in detecting the default options.' 2>&1
							exit 1
						fi
					)
					if [ `wc -c < $dbpath/ports_options.default` -gt 0 ]
					then
						if database_build_make "$origin" showconfig > $tmp_portopt
						then
							cp "$tmp_portopt" "$dbpath"/ports_options.current
						else
							cat "$tmp_portopt" >&2
							message_echo 'Error in detecting the current options.' 2>&1
							exit 1
						fi
					else
						cp /dev/null "$dbpath/ports_options.current"
					fi
					pkgsys_save_port_oprion_timestamp "$origin"
					# Get the lists of requirements in the flavored form
					database_build_create_pkgtag "$origin"
					for variable in PKG_DEPENDS EXTRACT_DEPENDS PATCH_DEPENDS FETCH_DEPENDS BUILD_DEPENDS LIB_DEPENDS
					do
						database_build_make "$origin" -V $variable
					done > ${TMPDIR}/database_build_inspect_dependencies:build_depends_list
					for variable in LIB_DEPENDS RUN_DEPENDS
					do
						database_build_make "$origin" -V $variable
					done > ${TMPDIR}/database_build_inspect_dependencies:run_depends_list
					for tag in run build
					do
						tr ' ' '\n' < ${TMPDIR}/database_build_inspect_dependencies:${tag}_depends_list | \
							sed -E 's#.*:([^:/]+/[^:/]+|[^:/]+/[^:/]+:[^:/]*)$#\1#' | cut -d : -f 1 | sort -u | \
							grep -Ev "`pkgsys_pkgtools_ports_filter_regexp`" \
								| while read origin_flavor_incomplete
								do
									( set -e
										opt_apply_default_config=yes
										database_build_determine_port_option "$origin_flavor_incomplete"
									)
									database_build_determine_flavored_origin "$origin_flavor_incomplete"
								done | grep -v '^$' | sort -u > $dbpath/requirements.$tag.orig || :
						sed -E -f "${DBDIR}/conf/REPLACE.sed_pattern" "$dbpath/requirements.$tag.orig" \
							| grep -v '^$' | sort -u > $dbpath/requirements.$tag.src || :
					done
					for stage in orig src
					do
						sort -u "$dbpath/requirements.run.${stage}" "$dbpath/requirements.build.${stage}" \
							> $dbpath/requirements.all.direct.${stage}
						mv "$dbpath/requirements.build.${stage}" "$dbpath/requirements.build.direct.${stage}"
						mv "$dbpath/requirements.run.${stage}" "$dbpath/requirements.run.direct.${stage}"
					done
					# Record the completion
					touch "$dbpath/complete_as_node"
					fileedit_rm_a_line "$origin_orig" "${DBDIR}/to_be_reconf"
					fileedit_rm_a_line "$origin" "${DBDIR}/to_be_reconf"
					# Restoration of escaped inspection conflict
					database_build_restore_inspect_conflict "$origin"
				else
					# Restoration of escaped inspection conflict
					database_build_restore_inspect_conflict "$origin"
					# Reset the complied lists of requirements
					for level in direct full
					do
						for tag in run build
						do
							rm -f "$dbpath/requirements.${tag}.${level}"
						done
					done
					for origin_tmp in "$origin" "$origin_orig"
					do
						for inspected_level_tmp in full direct node
						do
							fileedit_rm_a_line "$origin_tmp" "${DBDIR}/ports.inspected.${inspected_level_tmp}.list"
						done
					done
				fi
				# Inspect the requirements
				inspected_level=`database_build_get_inspected_level "$origin" "$origin_dependent"`
				case $inspected_level in
				full )
					grep -v -Fx -f "${DBDIR}/installed_ports" \
						"$dbpath/requirements.all.direct.src" > ${TMPDIR}/missing.$origin_id || :
					inspected_levels_compatible='full direct node'
					;;
				direct )
					grep -v -Fx -f "${DBDIR}/installed_ports" \
						"$dbpath/requirements.all.direct.src" > ${TMPDIR}/missing.$origin_id || :
					inspected_levels_compatible='direct node'
					;;
				node )
					cp /dev/null "${TMPDIR}/missing.$origin_id"
					inspected_levels_compatible='node'
					;;
				esac
				rm -rf "$dbpath/rename_requirements.sed.pattern"
				nlines=`wc -l < ${TMPDIR}/missing.$origin_id`
				iline=1
				while [ $iline -le $nlines ]
				do
					origin_dependency=`sed -n ${iline}p "${TMPDIR}/missing.$origin_id"`
					iline=$(($iline+1))
					grep -q -Fx "$origin_dependency" "${DBDIR}/ports.inspected.list" 2> /dev/null && \
							continue
					database_build_inspect_dependencies "$origin_dependency" "$origin"
				done
				if [ -e "$dbpath/rename_requirements.sed.pattern" ]
				then
					for tag in run build all
					do
						sed -E -i .unrenamed -f "$dbpath/rename_requirements.sed.pattern" "$dbpath/requirements.${tag}.direct.src"
					done
				fi 
				rm -f "${TMPDIR}/missing.$origin_id"
				# Inspect the distfiles
				dist_subdir_rpl=`database_query_get_makevar_val "$origin" DIST_SUBDIR | str_escape_replaceval_filter` || :
				[ -n "$dist_subdir_rpl" ] && dist_subdir_rpl="$dist_subdir_rpl\/"
				database_build_make "$origin" fetch-urlall-list \
					| sed -E "s/.*\/([^\/]+)$/$dist_subdir_rpl\1/" \
					| sort -u | fileedit_add_lines_if_new "${DBDIR}/distfiles.inspected"
				# Transfer data from the previous database if existent, successful and no change
				pkg_new=`cat "$dbpath/new_version"`
				if [ ! -e "$dbpath/conf_updated" -a -d "$dbpath_prev" ] && pkg_info_e "$pkg_new" 2> /dev/null && \
					! grep -qFx "$origin" "$dbpath_prev/to_be_reconf" 2> /dev/null
				then
					same_as_prevset=yes
					for dbfile in new_version CONFLICT.conf BUILDCONFLICT.conf MARG.conf MENV.conf \
						ports_options.default ports_options.current \
						rename_requirements.sed.pattern \
						requirements.run.direct.orig requirements.run.direct.src \
						requirements.build.direct.orig requirements.build.direct.src
					do
						[ ! -e "$dbpath/$dbfile" -a ! -e "$dbpath_prev/$dbfile" ] && continue
						if ! diff "$dbpath/$dbfile"  "$dbpath_prev/$dbfile" 2> /dev/null > /dev/null
						then
							same_as_prevset=no
							break
						fi
					done
					if [ $same_as_prevset = yes ]
					then
						nlines=`wc -l < $dbpath/requirements.all.direct.src`
						iline=1
						while [ $iline -le $nlines ]
						do
							origin_dependency=`sed -n ${iline}p "$dbpath/requirements.all.direct.src"`
							iline=$(($iline+1))
							if [ ! -e "${DBDIR}/requires/$origin_dependency/same_as_prevset" ]
							then
								same_as_prevset=no
								break
							fi
						done
					fi
					if [ $same_as_prevset = yes ]
					then
						for dbfile in failed.list damaged_package manually_done.list
						do
							if grep -qFx "$origin" "${DBDIR}/prevset/$dbfile" 2> /dev/null
							then
								same_as_prevset=no
								break
							fi
						done
					fi
					if [ $same_as_prevset = yes ]
					then
						rm -rf "$dbpath"
						cp -Rp "$dbpath_prev" "$dbpath"
						if [ -d "${DBDIR}/prevset/notes/$origin" ]
						then
							rm -rf "${DBDIR}/notes/$origin"
							mkdir -p "${DBDIR}/notes"
							cp -Rp "${DBDIR}/prevset/notes/$origin" "${DBDIR}/notes/$origin"
						fi
						{
							for tag in all run build none
							do
								for level in full direct
								do
									echo "success.$tag.$level.list"
								done
							done
						} | while read dbfile
						do
							if grep -Fxq "$origin" "${DBDIR}/prevset/$dbfile" 2> /dev/null
							then
								fileedit_add_a_line_if_new "$origin" "${DBDIR}/$dbfile"
							fi
						done
						touch "$dbpath/same_as_prevset"
						message_echo "${DEPTH_INDEX}  ===> No update from the previous reinstallation. Progress status inherited."
					fi
				fi
				# Record the completion
				if [ "x$origin_orig" != "x$origin" ]
				then
					for inspected_level_tmp in $inspected_levels_compatible
					do
						fileedit_add_a_line_if_new "$origin" "${DBDIR}/ports.inspected.${inspected_level_tmp}.list"
					done
					fileedit_add_a_line_if_new "$origin" "${DBDIR}/ports.inspected.list"
					fileedit_add_a_line_if_new "$origin" "${DBDIR}/inspected_ports.update"
					fileedit_rm_a_line "$origin" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
				fi
				fileedit_add_a_line_if_new "$origin" "${DBDIR}/done_required_ports_to_inspect"
			else
				message_echo "${DEPTH_INDEX}  ===> Already inspected merged port"
				fileedit_rm_a_line "$origin_orig" "${DBDIR}/to_be_reconf"
				for inspected_level_tmp in full direct node
				do
					fileedit_rm_a_line "$origin_orig" "${DBDIR}/ports.inspected.${inspected_level_tmp}.list"
				done
			fi
		fi
		# Record the completion
		for inspected_level_tmp in $inspected_levels_compatible
		do
			fileedit_add_a_line_if_new "$origin_orig" "${DBDIR}/ports.inspected.${inspected_level_tmp}.list"
		done
		fileedit_add_a_line_if_new "$origin_orig" "${DBDIR}/ports.inspected.list"
		fileedit_add_a_line_if_new "$origin_orig" "${DBDIR}/inspected_ports.update"
		fileedit_rm_a_line "$origin_orig" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
		message_echo "${DEPTH_INDEX}  ===> ok"
		DEPTH_INDEX=${DEPTH_INDEX_orig}
	fi
	if [ -n "$origin_dependent" -a "x$origin_orig" != "x$origin" ]
	then
		origin_orig_regexp=`str_escape_regexp "$origin_orig"`
		origin_replace=`str_escape_replaceval "$origin"`
		fileedit_add_a_line_if_new "s/^$origin_orig_regexp$/$origin_replace/" "${DBDIR}/requires/$origin_dependent/rename_requirements.sed.pattern"
	fi
}

# ============= Filter ignored dependencies from a list given by the standard input =============
database_build_filter_ignored_requirements ()
{
	local origin pattern
	origin=$1
	pattern=${DBDIR}/requires/$origin/ignored_requirements.filter
	if [ -e "$pattern" ]
	then
		grep -Fxvq -f "$pattern"
	else
		cat
	fi 2> /dev/null || :
}

# ============= Build and get a list of the complete recursive dependencies of a port =============
database_build_get_complete_recursive_dependency ()
{
	local tag origin suffix tmppath dbpath srcfile dstfile num_parents loophead dstfile_tmp tmpdstpath index_loop loop_len origin_target origin_ref dbpath_target
	tag=$1
	origin=$2
	suffix=$3
	tmppath=${TMPDIR}/database_build_get_complete_recursive_dependency
	dbpath=${DBDIR}/requires/$origin
	srcfile=$dbpath/requirements.${tag}.direct${suffix}
	dstfile=$dbpath/requirements.${tag}.full${suffix}
	if [ ! -e "$srcfile" ]
	then
		[ -d "$dbpath" ] && rm -f "$dstfile"
		return 0
	fi
	if [ ! -e "$dstfile" ]
	then
		touch "$tmppath.parents"
		num_parents=`wc -l < $tmppath.parents`
		if grep -Fxq "$origin" "$tmppath.parents"
		then
			loophead=`grep -Fxn "$origin" "$tmppath.parents" | tail -n 1 | cut -d : -f 1`
			if [ "x$opt_force_continuation_for_looped_dependency" = xno ]
			then
				message_echo "ERROR: The following loop was found for requirements.${tag}${suffix}:" >&2
			else
				message_echo "WARNING: The following loop was found for requirements.${tag}${suffix}:" >&2
			fi
			message_echo "  $origin" >&2
			ports_in_loop=`sed -n $(($loophead+1)),\$p "$tmppath.parents"`
			echo "$ports_in_loop" | sed 's/^/  -->/' | message_cat >&2
			message_echo "  -->$origin" >&2
			if [ "x$opt_disallow_force_continuation_for_looped_dependency" = xyes ]
			then
				message_echo 'Resolve the problem manually and then retry by executing' >&2
				message_echo "        ${APPNAME} reset keepopts" >&2
				message_echo "        ${APPNAME}" >&2
				exit 1
			else
				message_echo 'Exploring a node port to terminate the loop by evaluating only build-and-run-time dependencies as essential ones.' >&2
				echo "$origin" > $tmppath.ports_in_loop
				echo "$ports_in_loop" >> $tmppath.ports_in_loop
				echo "$origin" >> $tmppath.ports_in_loop
				index_loop=1
				loop_len=`wc -l < $tmppath.ports_in_loop`
				while [ $index_loop -lt $loop_len ]
				do
					origin_target=`sed -n ${index_loop}p "$tmppath.ports_in_loop"`
					origin_ref=`sed -n $((${index_loop}+1))p "$tmppath.ports_in_loop"`
					dbpath_target=${DBDIR}/requires/$origin_target
					if ! grep -Fxq "$origin_ref" "$dbpath_target/requirements.run.direct${suffix}" 2> /dev/null || \
						! grep -Fxq "$origin_ref" "$dbpath_target/requirements.build.direct${suffix}" 2> /dev/null
					then
						message_echo 'INFO: The dependency of '$origin_target' on '$origin_ref' is ignored to terminate the loop.' >&2
						break
					fi
					index_loop=$((${index_loop}+1))
				done
				if [ $index_loop -eq $loop_len ]
				then
					message_echo 'WARNING: The loop cannot be resolved. Continuing by forcible ignorance of the dependency of '$origin_target' on '$origin_ref'. This may cause confusion in the later processes.' >&2
				fi
				for suffix_tmp in '' .orig
				do
					for tag_tmp in run build
					do
						grep -Fxq "$origin_ref" "$dbpath_target/requirements.${tag_tmp}.direct${suffix_tmp}" 2> /dev/null && \
							fileedit_add_a_line_if_new "$origin_ref" "$dbpath_target/ignored_requirements.${tag_tmp}${suffix_tmp}"
					done
					cat "$dbpath_target/ignored_requirements.run${suffix_tmp}" "$dbpath_target/ignored_requirements.build${suffix_tmp}" 2> /dev/null | sort -u > $dbpath_target/ignored_requirements.all${suffix_tmp}
				done
				cat "$dbpath_target/ignored_requirements.all"* 2> /dev/null | sort -u > $dbpath_target/ignored_requirements.filter
			fi
		fi
		echo "$origin" >> $tmppath.parents
		tmpdstpath=${TMPDIR}/requires/$origin
		dstfile_tmp=$tmpdstpath/requirements.${tag}.full${suffix}
		mkdir -p "$tmpdstpath"
		database_build_filter_ignored_requirements "$origin" < $srcfile | while read origin_requirement
		do
			# The least requirements of build are the direct build-time requirements and the full run-time requirements of the direct build-time requirements.
			tag_dep=$tag
			[ "x$tag" = xbuild ] && tag_dep=run
			recursive_dependency=`database_build_get_complete_recursive_dependency "$tag_dep" "$origin_requirement" "$suffix"`
			[ -e "$dbpath/ignored_requirements.filter" ] && \
				echo "$origin_requirement" | grep -Fxq -f "$dbpath/ignored_requirements.filter" 2> /dev/null && \
				continue
			echo "$origin_requirement"
			echo "$recursive_dependency"
		done > $dstfile_tmp
		sed -n ${num_parents}p "$tmppath.parents" > $tmppath.parents.tmp
		mv "$tmppath.parents.tmp" "$tmppath.parents"
		if [ ! -e "$dstfile" ]
		then
			sort -u "$dstfile_tmp" | grep -v '^$' > $dstfile || :
			rm "$dstfile_tmp"
		fi
	fi
	cat "$dstfile"
}

# ============= Inspect the necessity of requirements of a necessary port even in case of skipping unchanged ports =============
# Here the "necessity" means that a port must be kept installed, newly installed or reinstalled for run- or build-time operations.
# In other words, "unnecessary" ports are what are not yet installed and not required by any "necessary" ports neither in run- or build-time.
database_build_inspect_necessity_for_only_new_upgrade ()
{
	local origin level dbpath tmpfile
	origin=$1
	level=$2
	dbpath=${DBDIR}/requires/$origin
	[ ! -d "$dbpath" -o -e "$dbpath/necessary_port.${level}" ] && return
	tmpfile=${TMPDIR}/database_build_inspect_necessity_for_only_new_upgrade:`echo "$origin" | tr / :`
	if database_query_does_a_port_need_update "$origin"
	then
		sort -u "$dbpath/requirements.build.direct" "$dbpath/requirements.run.${level}" || :
	else
		cat "$dbpath/requirements.run.${level}" || :
	fi 2> /dev/null | database_build_filter_ignored_requirements "$origin" > $tmpfile
	while read origin_requirement
	do
		database_build_inspect_necessity_for_only_new_upgrade "$origin_requirement" "$level"
	done < $tmpfile
	rm "$tmpfile"
	touch "$dbpath/necessary_port.${level}"
}

# ============= Build complement relations to new dependents for targets recursively =============
database_build_complement_to_new_dependents_for_targets ()
{
	local origin dbdir reqdir level
	origin=$1
	dbdir=${DBDIR}/targets/$origin
	reqdir=${DBDIR}/requires/$origin
	[ -d "$dbdir" ] || return 0
	echo "$origin" >> ${DBDIR}/stage.loop_list/parse_target_attr_info
	if [ -e "$dbdir/target_itself" ] && database_query_does_a_port_need_update "$origin"
	then
		for level in direct full
		do
			cat "$reqdir/requirements.all.${level}" 2> /dev/null | \
				database_build_filter_ignored_requirements "$origin" | \
				while read origin_requirement
			do
				fileedit_add_a_line_if_new "$origin" \
					"${DBDIR}/targets/$origin_requirement/complement_for_new_dependents.${level}"
			done
		done
		cat "$reqdir/requirements.all.direct" "$reqdir/requirements.all.full" 2> /dev/null | \
			sort -u | database_build_filter_ignored_requirements "$origin" | \
			while read origin_requirement
		do
			database_build_complement_to_new_dependents_for_targets "$origin_requirement"
		done
	fi
}

# ============= Recursively complete necessary upgrades for build =============
database_build_complete_necessary_upgrades_for_build ()
{
	local origin level dbpath origin_requirement dbpath_requirement
	origin=$1
	level=$2
	dbpath=${DBDIR}/requires/$origin
	[ -e "$dbpath/requirements.build.${level}" ] || return 0
	while read origin_requirement
	do
		dbpath_requirement=${DBDIR}/requires/$origin_requirement
		[ -e "$dbpath_requirement/necessary_upgrade.build.${level}" ] || continue
		[ `cat "${DBDIR}/moved_from/$origin_requirement/current_version" 2> /dev/null | wc -l` -eq 0 ] || continue
		touch "$dbpath_requirement/necessary_upgrade.run.${level}"
		database_build_complete_necessary_upgrades_for_build "$origin_requirement" "$level"
	done < $dbpath/requirements.build.${level}
	:
}

# ============= Build target attribute files recursively =============
database_build_target_attributes ()
{
	local origin dbdir reqdir _is_target _is_requires_requirements _is_initial_requirements _is_requires_dependents _is_initial_dependents _is_requires_requirements_complement _is_relevant tag level database table
	origin=$1
	dbdir=${DBDIR}/targets/$origin
	reqdir=${DBDIR}/requires/$origin
	[ -d "$dbdir" ] || return 0
	[ -e "$dbdir/COMPLETE_ATTRS" ] && return
	_is_target=
	[ -e "$dbdir/target_itself" ] && _is_target=y
	for tag in all run build none
	do
		for level in direct full
		do
			_is_relevant=${_is_target}
			_is_requires_requirements=
			_is_initial_requirements=
			_is_requires_dependents=
			_is_initial_dependents=
			_is_requires_requirements_complement=
			for database in requires initial
			do
				for table in requirements dependents
				do
					if [ -e "$dbdir/target_${database}_${table}.${tag}.${level}" ]
					then
						eval _is_${database}_${table}=y
						_is_relevant=y
					fi
				done
			done
			if [ -z "${_is_requires_requirements}" -a -e "$dbdir/complement_for_new_dependents.${level}" ]
			then
				_is_requires_requirements_complement=y
				_is_relevant=y
			fi
			[ -n "${_is_relevant}" ] && cat > $dbdir/attrs.${tag}.${level} << eof
_is_target=${_is_target}
_is_requires_requirements=${_is_requires_requirements}
_is_initial_requirements=${_is_initial_requirements}
_is_requires_dependents=${_is_requires_dependents}
_is_initial_dependents=${_is_initial_dependents}
_is_requires_requirements_complement=${_is_requires_requirements_complement}
eof
		done
	done
	touch "$dbdir/COMPLETE_ATTRS"
}

# ============= Order the ports considering dependencies =============
database_build_order_ports_considering_dependencies ()
{
	touch "${DBDIR}/reinst_order.list.tmp"
	(cd "${DBDIR}/requires" && \
		find . -depth 3 -name requirements.all.full -exec echo {} \; -exec cat {} \; -exec echo \;) | \
		env "DBDIR=${DBDIR}" awk -f "${LIBEXECDIR}"/order_dependencies.awk || return
	grep -v '^$' "${DBDIR}/reinst_order.list.tmp" > "${DBDIR}/reinst_order.list" || :
}

# ============= [Sub-function] Common operations for resetting configurations for a port =============
_database_build_reset_a_port_confdb ()
{
	local origin origin_replace db
	origin=$1
	[ -d "${DBDIR}/initial/$origin" -o -d "${DBDIR}/replace/$origin" \
		-o -d "${DBDIR}/requires/$origin" -o -d "${DBDIR}/targets/$origin" ] || return 0
	touch "${DBDIR}/inspected_ports_only_partially"
	{
		echo "$origin"
		cat "${DBDIR}/replace/$origin/origin" 2> /dev/null || :
		for dbtag in initial requires
		do
			for tabel in requirements dependents
			do
				cat "${DBDIR}/$dbtag/$origin/$tabel.all.full" 2> /dev/null || :
				cat "${DBDIR}/$dbtag/$origin/$tabel.all.full.orig" 2> /dev/null || :
				rm -rf "${DBDIR}/$dbtag/$origin/$tabel.run.full" \
					"${DBDIR}/$dbtag/$origin/$tabel.build.full"
			done
			rm -rf "${DBDIR}/$dbtag/$origin/is_customized"
		done
	} | sort -u >> ${DBDIR}/inspected_ports.update
	for level in full direct node
	do
		fileedit_rm_a_line "$origin" "${DBDIR}/ports.inspected.${level}.list"
	done
	fileedit_rm_a_line "$origin" "${DBDIR}/obsolete_ports"
	fileedit_rm_a_line "$origin" "${DBDIR}/moved_ports"
	fileedit_rm_a_line "$origin" "${DBDIR}/ports.inspected.list"
	cat "${DBDIR}/replace/$origin/origin" 2> /dev/null || :
}

# ============= Clear database directories for a flavored origin =============
database_build_clear_db_dirs ()
{
	local origin db
	origin=$1
	for db in requires replace targets obsolete
	do
		rm -rf "${DBDIR}/$db/$origin"
	done
}

# ============= Remove configurations for a port permanently =============
database_build_forget ()
{
	local origin origin_replace
	origin=$1
	origin_replace=`_database_build_reset_a_port_confdb "$origin"`
	fileedit_rm_a_line "$origin" "${DBDIR}/targets_specified_so_far"
	[ -z "$origin_replace" ] || database_build_forget "$origin_replace"
	database_build_clear_db_dirs "$origin"
}

# ============= Overlay onto the temporary database so as to re-inspect and reinstall ports whose configurations were changed =============
database_build_patch_reconf ()
{
	local origin origin_replace
	origin=$1
	origin_replace=`_database_build_reset_a_port_confdb "$origin"`
	[ -d "${DBDIR}/initial/$origin" -o -d "${DBDIR}/requires/$origin" ] && \
		fileedit_add_a_line_if_new "$origin" "${DBDIR}/stage.loop_list/ports_to_inspect"
	fileedit_add_a_line_if_new "$origin" "${DBDIR}/to_be_reconf"
	fileedit_rm_a_line "$origin" "${DBDIR}/done_required_ports_to_inspect"
	[ -z "$origin_replace" ] || database_build_patch_reconf "$origin_replace"
	database_build_clear_db_dirs "$origin"
}

# ============= Post-processes after finishing to inspect initial dependencies =============
database_build_post_inspect_initial_dependencies ()
{
	find "${DBDIR}/initial" -depth 2 -type d \
		| sed -E 's|.*/([^/]+/[^/]+)$|\1|' > ${DBDIR}/inspected_ports.initial
}

# ============= Post-processes after finishing to inspect dependencies =============
database_build_post_inspect_dependencies ()
{
	local table tmpfile
	tmpfile=${TMPDIR}/database_build_post_inspect_dependencies
	touch "${DBDIR}/obsolete_ports" "${DBDIR}/inspected_ports.update"
	find "${DBDIR}/requires" -depth 2 -type d \
		| sed -E 's|.*/([^/]+/[^/]+)$|\1|' > ${DBDIR}/inspected_ports
	sort -u "${DBDIR}/inspected_ports" "${DBDIR}/inspected_ports.initial" > ${DBDIR}/inspected_ports.all
	{
		cat "${DBDIR}/conf/HOLD:PORTS.parsed" || :
		cat "${DBDIR}/need.list" || :
	} 2> /dev/null | sort -u > $tmpfile.obsolete_ports.exclude
	grep -v -Fx -f "$tmpfile.obsolete_ports.exclude" "${DBDIR}/obsolete_ports" > ${DBDIR}/obsolete_ports.can_be_deleted || :
	cp /dev/null "${DBDIR}/REPLACE.complete_sed_pattern.tmp"
	find "${DBDIR}/replace" -depth 3 -type f -name origin | while read nodepath
	do
		origin_orig=`expr "$nodepath" : '.*/\([^/][^/]*/[^/][^/]*\)/origin$'`
		origin=`cat "$nodepath"`
		origin_orig_regexp=`str_escape_regexp "$origin_orig"`
		origin_esc=`str_escape_replaceval "$origin"`
		echo "s/^$origin_orig_regexp$/$origin_esc/" >> ${DBDIR}/REPLACE.complete_sed_pattern.tmp
	done
	mv "${DBDIR}/REPLACE.complete_sed_pattern.tmp" "${DBDIR}/REPLACE.complete_sed_pattern"
	if [ `cat "${DBDIR}/inspected_ports.update" 2> /dev/null | wc -l` -gt 0 ]
	then
		grep -Fx -f "${DBDIR}/inspected_ports.all" "${DBDIR}/inspected_ports.update" \
			> ${DBDIR}/inspected_ports.update.tmp 2> /dev/null || :
		mv "${DBDIR}/inspected_ports.update.tmp" "${DBDIR}/inspected_ports.update"
		if [ $opt_only_target_scope = yes ]
		then
			touch "${DBDIR}/inspected_ports_only_partially"
		else
			rm -f "${DBDIR}/inspected_ports_only_partially"
		fi
		if program_chk_stage_loop_complete CONVERT_REQUIREMENTS_LIST
		then
			mv "${DBDIR}/inspected_ports.update" "${DBDIR}/stage.loop_list/convert_dependency_lists"
		else
			cat "${DBDIR}/inspected_ports.update" "${DBDIR}/stage.loop_list/convert_dependency_lists" \
				2> /dev/null | sort -u > ${DBDIR}/stage.loop_list/convert_dependency_lists.tmp
			mv "${DBDIR}/stage.loop_list/convert_dependency_lists.tmp" \
				"${DBDIR}/stage.loop_list/convert_dependency_lists"
			rm -f "${DBDIR}/inspected_ports.update"
		fi
	else
		program_chk_stage_loop_complete CONVERT_REQUIREMENTS_LIST \
			&& rm -f "${DBDIR}/stage.loop_list/convert_dependency_lists"
	fi
	{
		sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" "${DBDIR}/need.list" || :
		cat "${DBDIR}/need.list" || :
	} 2> /dev/null | sort -u > ${DBDIR}/need.with_replaced.list
	for table in requirements dependents itself
	do
		[ -e "${DBDIR}/stage.loop_list/target_$table.specified" ] || continue
		sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" \
			"${DBDIR}/stage.loop_list/target_$table.specified" \
			> ${DBDIR}/stage.loop_list/target_$table.replaced.specified
	done
	cp /dev/null "${DBDIR}/update_dependencies"
}
