#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Operations for maintaining the temporary database -
# Copyright (C) 2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================


# ============= Variables =============
DATABASE_VERSION=
DATABASE_IS_OBSOLETE=no


# ============= Cleaning of the temporary database =============
database_maintain_clean_all ()
{
	rm -rf "${DBDIR}"
}

# ============= Reset the execution result flags =============
database_maintain_reset_execflag ()
{
	rm -rf "${DBDIR}/execflag"
}

# ============= Load a temporary database from an archive =============
database_maintain_load ()
{
	local filepath dbdir_parent
	filepath=$1
	dbdir_parent=`dirname "${DBDIR}"`
	mkdir -p "$dbdir_parent"
	tar xzf "$filepath" -C "$dbdir_parent" --exclude "*/.lock"
}

# ============= Save the temporary database to an archive =============
database_maintain_save ()
{
	local filepath dbdir_parent dbdir_node
	filepath=$1
	dbdir_parent=`dirname "${DBDIR}"`
	dbdir_node=`basename "${DBDIR}"`
	tar czf "$filepath" -C "$dbdir_parent" "$dbdir_node"
}

# ============= Creation of the temporary database =============
database_maintain_create ()
{
	local subdir
	[ `id -u` -eq 0 ] && mkdir -p "${DBDIR}"
	misc_lock_duplicated_executions "${DBDIR}/.lock"
	if [ -e "${DBDIR}/MYVERSION" ]
	then
		if ! grep -q -E "$COMPATIBLE_VERSIONS" "${DBDIR}/MYVERSION" 2> /dev/null
		then
			message_echo "ERROR: The current temporary database is incompatible. You must delete it by" >&2
			message_echo "        ${APPNAME} clean force" >&2
			message_echo "       in order to enable the current version." >&2
			exit 1
		fi
	elif misc_is_superuser_privilege
	then
		echo "$MYVERSION" > ${DBDIR}/MYVERSION
	fi
	DATABASE_VERSION=`cat "${DBDIR}"/MYVERSION 2> /dev/null || :`
	misc_is_superuser_privilege || return 0
	for subdir in initial requires replace targets obsolete backup_packages \
		stage.loop_list stage.complete stage.reinit_loop stage.depends
	do
		mkdir -p "${DBDIR}/$subdir"
	done
}

# ============= Mark the use of the temporary database =============
database_maintain_mark_use ()
{
	 touch "${DBDIR}/in_use"
}

# ============= Check the use of the temporary database =============
database_maintain_chk_use ()
{
	 [ -e "${DBDIR}/in_use" ]
}

# ============= Refresh the temporary database =============
database_maintain_refresh ()
{
	misc_is_superuser_privilege || return
	[ $opt_suppress_obsolete_db_clean = no ] || return
	message_echo "INFO: The temporary database is cleaned up."
	message_echo
	[ -d "${DBDIR}" -a ! -d "${DBDIR}.tmp" ] && mv "${DBDIR}" "${DBDIR}.tmp"
	database_maintain_create
	mv "${DBDIR}.tmp/saved_options.sh" "${DBDIR}" 2> /dev/null || :
	mv "${DBDIR}.tmp/backup_packages" "${DBDIR}" 2> /dev/null || :
	mv "${DBDIR}.tmp/backup_pkgarcs.lst" "${DBDIR}" 2> /dev/null || :
	rm -rf "${DBDIR}.tmp"
}

# ============= Clean up the temporary database for upgrade of this utility =============
database_maintain_clean_for_self_upgrade ()
{
	misc_is_superuser_privilege || return
	[ $opt_suppress_obsolete_db_clean = no ] || return
	rm -rf "${DBDIR}"
	database_maintain_create
	[ -e "${DBDIR}/MYVERSION" ] && mv "${DBDIR}/MYVERSION" "${DBDIR}/MYVERSION.prev"
	:
}

# ============= Check whether the temporary database is newer than the ports tree and refresh if so =============
database_maintain_refresh_if_obsolete ()
{
	if [ "${PORTS_INDEX_DB}" -nt "${DBDIR}"/MYVERSION ] && misc_is_superuser_privilege
	then
		if [ $opt_suppress_obsolete_db_clean = no -a "z${command}" = zclean ]
		then
			DATABASE_IS_OBSOLETE=no
			message_echo "WARNING: The temporary database is older than the ports tree." >&2
			database_maintain_refresh || DATABASE_IS_OBSOLETE=yes
		else
			DATABASE_IS_OBSOLETE=yes
		fi
	else
		DATABASE_IS_OBSOLETE=no
	fi
}

# ============= Resetting of the temporary database =============
database_maintain_reset ()
{
	local mode
	mode=$1
	if [ $mode = keepstatus ]
	then
		if [ ! -d "${DBDIR}/prevset" ]
		then
			mkdir -p "${DBDIR}/prevset"
			{
				echo requires
				echo notes
				echo to_be_reconf
				echo failed.list
				echo damaged_package
				echo manually_done.list
				for tag in all run build none
				do
					for level in full direct
					do
						echo "success.$tag.$level.list"
					done
				done
			} | while read content
			do
				[ -e "${DBDIR}/$content" ] && mv "${DBDIR}/$content" "${DBDIR}/prevset"
			done
		fi
	else
		rm -rf "${DBDIR}"/requires.prev "${DBDIR}"/notes.prev
	fi
	find "${DBDIR}" -depth 1 -maxdepth 1 \
		-not \( -name saved_options.sh -or -name initial \
			-or -name fossil_pkgs -or -name moved_from \
			-or -name MYVERSION -or -name .lock \
			-or -name journal -or -name prevset \
			-or -name backup_failure -or -name installed_ports \
			-or -name installed_ports:pkg_vs_origin.tbl \) \
		-exec rm -rf {} \; 2> /dev/null || :
	touch "${DBDIR}/MYVERSION"
	case $mode in
	all )
		command_exec_without_pkgtools__notify_reset_options
		rm -f "${DBDIR}/saved_options.sh" "${DBDIR}/prevset" "${DBDIR}/journal"*
		;;
	keepopts | keepstatus )
		(
			opt_batch_mode=yes
			while read srcline
			do
				command_flexconf_update_taboo $srcline
			done < ${DBDIR}/journal/taboo || :
			while read srcline
			do
				command_flexconf_update_freeze $srcline
			done < ${DBDIR}/journal/freeze || :
			while read srcline
			do
				command_flexconf_update_need $srcline
			done < ${DBDIR}/journal/need || :
			while read srcline
			do
				command_flexconf_update_noneed $srcline
			done < ${DBDIR}/journal/noneed || :
		) 2> /dev/null
		;;
	esac
}

# ============= Clear the data of previous progress status =============
database_maintain_clear_prevset ()
{
	rm -rf "${DBDIR}/prevset"
}
