#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Deinstallation processes -
# Copyright (C) 2013-2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Dialog for selection of removing new leaf ports =============
deinstall_select_leaf_ports_to_delete_dialog ()
{
	local src dst_selected dst_unselected itemlist title desc
	src=$1
	dst_selected=$2
	dst_unselected=$3
	itemlist=${TMPDIR}/deinstall_select_leaf_ports_to_delete_dialog::itemlist
	title='New leaf ports and their exclusive requirements to be deleted'
	desc='Unchecked ones are preserved with their requirements.\nCancellation means to preserve all.'
	while read origin
	do
		pkgtag=`cat "${DBDIR}/moved_from/$origin/pkgtag" 2> /dev/null` || :
		val=on
		grep -Fxq "$origin" "$dst_unselected" 2> /dev/null && val=off
		printf '%s\t"(%s)"\t%s\n' "$origin" "$pkgtag" "$val"
	done < $src > $itemlist
	misc_dialog_checklist "$title" "$desc" "$dst_selected" "$itemlist"
	grep -Fx -v -f "$dst_selected" "$src" > $dst_unselected || :
}

# ============= Automatic selection of removing ports =============
deinstall_select_auto ()
{
	local src dst_add dst_rem tmp_srcprev tmp_srcnew diff_removed diff_added
	src=$1
	dst_add=$2
	dst_rem=$3
	tmp_srcprev=${TMPDIR}/deinstall_select_auto:src.prev
	tmp_srcnew=${TMPDIR}/deinstall_select_auto:src.new
	diff_removed=${TMPDIR}/deinstall_select_auto:diff_removed
	diff_added=${TMPDIR}/deinstall_select_auto:diff_added
	cat "$dst_add" "$dst_rem" 2> /dev/null | sort -u > $tmp_srcprev
	cat "$src" 2> /dev/null | sort -u > $tmp_srcnew
	fileedit_manipulate_old_new_lines "$tmp_srcprev" "$tmp_srcnew" "$diff_removed" "$diff_added" || return 0
	grep -v -Fx -f "$diff_removed" "$dst_rem" 2> /dev/null | \
		grep -Fx -f "$src" > $dst_rem.tmp || :
	mv "$dst_rem.tmp" "$dst_rem"
	cat "$dst_add" "$diff_added" 2> /dev/null | sort -u | \
		grep -Fx -f "$src" > $dst_add.tmp || :
	mv "$dst_add.tmp" "$dst_add"
}

# ============= Selection of removing new leaf ports =============
deinstall_select_leaf_ports_to_delete ()
{
	local mode src dst_selected dst_unselected tmp_srcprev
	mode=$1
	src=${DBDIR}/leaf_ports
	dst_selected=${DBDIR}/leaf_ports_to_delete.selected
	dst_unselected=${DBDIR}/leaf_ports_to_delete.unselected
	tmp_srcprev=${TMPDIR}/deinstall_select_leaf_ports_to_delete:src.prev
	if [ "x$mode" = xforce ]
	then
		if [ -e "${DBDIR}/inspected_ports_only_partially" ]
		then
			return 3
		elif [ `cat "$src" 2> /dev/null | wc -l` -eq 0 ]
		then
			cp /dev/null "$dst_selected"
			cp /dev/null "$dst_unselected"
			return 2
		fi
		deinstall_select_leaf_ports_to_delete_dialog "$src" "$dst_selected" "$dst_unselected"
	elif [ ! -e "${DBDIR}/inspected_ports_only_partially" ]
	then
		case $opt_dialog_auto in
		no )
			touch "$dst_selected" "$dst_unselected" "$src"
			if [ `cat "$src" 2> /dev/null | wc -l` -gt 0 ]
			then
				cat "$dst_selected" "$dst_unselected" 2> /dev/null | sort -u > $tmp_srcprev
				diff -q "$tmp_srcprev" "$src" > /dev/null 2>&1 || \
					deinstall_select_leaf_ports_to_delete_dialog "$src" "$dst_selected" "$dst_unselected"
			else
				cp /dev/null "$dst_selected"
				cp /dev/null "$dst_unselected"
			fi
			;;
		all )
			deinstall_select_auto "$src" "$dst_selected" "$dst_unselected"
			;;
		none )
			deinstall_select_auto "$src" "$dst_unselected" "$dst_selected"
			;;
		esac
	fi
}

# ============= Dialog for selection of removing obsolete ports =============
deinstall_select_obsolete_ports_to_delete_dialog ()
{
	local src dst_selected dst_unselected itemlist title desc
	src=$1
	dst_selected=$2
	dst_unselected=$3
	itemlist=${TMPDIR}/deinstall_select_obsolete_ports_to_delete_dialog::itemlist
	title='Obsolete packages to be deleted'
	desc='Unchecked ones are preserved.\nCancellation means to preserve all.'
	while read origin
	do
		pkgtag=`cat "${DBDIR}/initial/$origin/installed_version"`
		val=on
		grep -Fxq "$origin" "$dst_unselected" 2> /dev/null && val=off
		printf '%s\t"(%s)"\t%s\n' "$origin" "$pkgtag" "$val"
	done < $src > $itemlist
	misc_dialog_checklist "$title" "$desc" "$dst_selected" "$itemlist"
	grep -Fx -v -f "$dst_selected" "$src" > $dst_unselected || :
}

# ============= Selection of removing obsolete ports =============
deinstall_select_obsolete_ports_to_delete ()
{
	local mode src dst_selected dst_unselected tmp_srcprev
	mode=$1
	src=${DBDIR}/obsolete_ports.can_be_deleted
	dst_selected=${DBDIR}/obsolete_ports_to_delete.selected
	dst_unselected=${DBDIR}/obsolete_ports_to_delete.unselected
	tmp_srcprev=${TMPDIR}/deinstall_select_obsolete_ports_to_delete:src.prev
	if [ "x$mode" = xforce ]
	then
		if [ `cat "$src" 2> /dev/null | wc -l` -eq 0 ]
		then
			cp /dev/null "$dst_selected"
			cp /dev/null "$dst_unselected"
			return 2
		fi
		deinstall_select_obsolete_ports_to_delete_dialog "$src" "$dst_selected" "$dst_unselected"
	else
		case $opt_dialog_auto in
		no )
			touch "$dst_selected" "$dst_unselected" "$src"
			if [ `cat "$src" 2> /dev/null | wc -l` -gt 0 ]
			then
				cat "$dst_selected" "$dst_unselected" 2> /dev/null | sort -u > $tmp_srcprev
				diff -q "$tmp_srcprev" "$src" > /dev/null 2>&1 || \
					deinstall_select_obsolete_ports_to_delete_dialog "$src" "$dst_selected" "$dst_unselected"
			else
				cp /dev/null "$dst_selected"
				cp /dev/null "$dst_unselected"
			fi
			;;
		all )
			deinstall_select_auto "$src" "$dst_selected" "$dst_unselected"
			;;
		none )
			deinstall_select_auto "$src" "$dst_unselected" "$dst_selected"
			;;
		esac
	fi
}

# ============= Restoration of a package for a flavored origin =============
deinstall_restore ()
{
	local origin tmp_forbidden currentpkg backup_pkg pkg
	origin=$1
	tmp_forbidden=${TMPDIR}/deinstall_restore:forbidden
	pkgsys_exists_from_orig "$origin" && return
	backup_pkg=`pkgsys_get_backup_pkg "$origin"` || return 0
	currentpkg=`pkgsys_pkgarc_to_pkgname "$backup_pkg"`
	message_stage_title "$PROGRAM_STEP_COUNTER $origin ($currentpkg)"
	message_target_relations "$origin"
	temp_set_msg_current_stage "a ${_MSG_CURRENT_STAGE_general} process for $origin ($currentpkg) $PROGRAM_STEP_COUNTER"
	database_query_get_target_attributes currentorigin "$origin"
	if [ -z "${currentorigin_is_relevant}" ]
	then
		message_echo "-- (Skipping an irrelevant package for obsolete port $origin as $currentpkg)"
		message_echo
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
		return
	fi
	if database_query_is_a_port_suppressed "$origin"
	then
		message_echo "-- (Skipped because being suppressed)"
		message_echo
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
		return
	fi
	pkg=`pkgsys_pkgarc_to_pkgname "$backup_pkg"`
	if reinstall_chk_forbidden_conflicts "$pkg"
	then
		message_echo "-- (Skipped because conflicting with installed packages)"
		return
	fi
	if [ $opt_dry_run = yes ]
	then
		message_dry_run
		message_echo
		temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
		return
	elif ! pkg_add_fF "$backup_pkg"
	then
		message_echo "*** Going on to the next process anyway..."
	fi
	temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
	message_echo
}

# ============= Deinstallation of a package for a flavored origin =============
deinstall_exec ()
{
	local origin_master origin origin_regexp currentpkg backup_pkgdir backup_pkg pkg
	origin_master=$1
	currentpkg=`pkgsys_get_installed_pkg_from_origin "$origin_master" | tr '\n' ' ' | sed 's/ *$//'`
	[ -n "$currentpkg" ] || currentpkg='deinstalled'
	message_stage_title "$PROGRAM_STEP_COUNTER $origin_master ($currentpkg)"
	message_target_relations "$origin_master"
	for origin in `{ echo "$origin_master"; database_query_initial_orgins "$origin_master"; } | sort -u`
	do
		origin_regexp=`str_escape_regexp "$origin"`
		grep -v -E "^${origin_regexp}[[:space:]]" "${DBDIR}/deleted_conflicts" > ${DBDIR}/deleted_conflicts.tmp 2> /dev/null || :
		mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
		currentpkg=`pkgsys_get_installed_pkg_from_origin "$origin" | tr '\n' ' ' | sed 's/ *$//'`
		[ -n "$currentpkg" ] || continue
		temp_set_msg_current_stage "a ${_MSG_CURRENT_STAGE_general} process for $origin ($currentpkg) $PROGRAM_STEP_COUNTER"
		database_query_get_target_attributes currentorigin "$origin"
		if [ -z "${currentorigin_is_relevant}" ]
		then
			message_echo "-- (Skipping an non-target package for obsolete port $origin as $currentpkg)"
			continue
		fi
		if database_query_is_a_port_suppressed "$origin"
		then
			message_echo "-- (Skipped for $origin as $currentpkg because being suppressed)"
			continue
		fi
		if [ $opt_dry_run = yes ]
		then
			message_echo "-- (Create backup package for $origin as $currentpkg)"
			continue
		fi
		message_echo "-- (Creating backup package for $origin as $currentpkg)"
		if ! pkgsys_get_backup_pkg "$origin" > /dev/null
		then
			if [ -d "${DBDIR}/requires/$origin" ]
			then
				backup_pkgdir=${DBDIR}/backup_packages
			else
				backup_pkgdir=${PKGREPOSITORY}
			fi
			mkdir -p "$backup_pkgdir"
			pkg=`echo "$currentpkg" | tr ' ' '\n' | grep -v '^$' | tail -n 1`
			backup_pkg=`pkgsys_create_backup_pkg "$pkg" "$backup_pkgdir"` || :
			if [ -n "$backup_pkg" ]
			then
				message_echo "INFO: The backup is saved at $backup_pkg."
				message_echo
			else
				message_echo "*** Continuating forcibly by keeping the installed package"
				continue
			fi
		fi
		message_echo "-- (Deleting package for $origin as $currentpkg)"
		echo "$currentpkg" | tr ' ' '\n' | grep -v '^$' | while read pkg
		do
			if ! pkg_delete_f "$currentpkg"
			then
				message_echo "*** Continuating forcibly by hoping success..."
			fi
		done
	done
	[ $opt_dry_run = yes ] && message_dry_run
	temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
	message_echo
}
