#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# Overlay onto lib/libcommand.sh
# - Operations of commands as well as check of command line arguments -
# Copyright (C) 2013-2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Check the necessity of opening notice =============
command_all_chk_need_opening_notice ()
{
	[ $COMMAND_MODE = do -o $COMMAND_MODE = redo -o $COMMAND_MODE = forget ]
}

# ============= Execute command operations before getting the temporary database ready =============
command_all_exec_before_db_creation ()
{
	local COMMAND_RESTART COMMAND_MODE COMMAND_OPERATION
	COMMAND_RESTART="$@"
	COMMAND_MODE=${1:-do}
	shift || :
	case $COMMAND_MODE in
	clean )
		COMMAND_OPERATION=${1:-normal}
		shift || :
		_command_parse_args__chk_no_arg $#
		case $COMMAND_OPERATION in
		force )
			message_echo "INFO: The temporary database is tried to be cleaned up without checking the privilege."
			database_maintain_clean_all
			[ $opt_no_opening_message = yes ] || message_echo "Done"
			exit
			;;
		esac
		;;
	esac
}

# ============= Check and parse command line arguments =============
command_all_parse_args ()
{
	local num_args_init
	num_args_init=$#
	COMMAND_RESTART="$@"
	COMMAND_MODE=${1:-do}
	shift || :
	case $COMMAND_MODE in
	clean )
		COMMAND_OPERATION=${1:-normal}
		shift || :
		case $COMMAND_OPERATION in
		normal )
			misc_chk_privilege
			;;
		esac
		_command_parse_args__chk_no_arg $#
		;;
	reset )
		misc_chk_privilege
		COMMAND_OPERATION=${1:-all}
		shift || :
		case $COMMAND_OPERATION in
		all | keepopts | keepstatus )
			;;
		*)
			message_echo "ERROR: Invalid operation [$COMMAND_OPERATION]; it must be empty, \"all\" or \"keepopts\"." >&2
			exit 1
			;;
		esac
		_command_parse_args__chk_no_arg $#
		;;
	ok | taboo | freeze | need | noneed )
		misc_chk_privilege
		[ $COMMAND_MODE = ok ] && database_query_chk_preparation_completion
		temp_warn_obsolete_temp_db >&2
		COMMAND_OPERATION=$1
		shift || :
		case $COMMAND_OPERATION in
		add | del )
			;;
		'' )
			message_echo "ERROR: Missing operation which must be \"add\" or \"del\"." >&2
			exit 1
			;;
		*)
			message_echo "ERROR: Invalid operation [$COMMAND_OPERATION]; it must be \"add\" or \"del\"." >&2
			exit 1
			;;
		esac
		_command_parse_args__chk_glob_args $#
		;;
	reselect )
		misc_chk_privilege
		database_query_chk_preparation_completion
		temp_warn_obsolete_temp_db >&2
		COMMAND_OPERATION=$1
		shift || :
		case $COMMAND_OPERATION in
		leaves )
			if [ -e "${DBDIR}/inspected_ports_only_partially" ]
			then
				message_echo "ERROR: Leaf ports cannot be analyzed because the dependency inspection is partial." >&2
				message_echo "Executing redo command with -N option by disabling -o option fixes this situation." >&2
				exit 1
			fi
			;;
		obsolete )
			;;
		'' )
			message_echo "ERROR: Missing operation which must be \"leaves\" or \"obsolete\"." >&2
			exit 1
			;;
		*)
			message_echo "ERROR: Invalid operation [$COMMAND_OPERATION]; it must be \"leaves\" or \"obsolete\"." >&2
			exit 1
			;;
		esac
		_command_parse_args__chk_glob_args $#
		;;
	save )
		misc_chk_privilege
		if [ ! -d "${DBDIR}" ]
		then
			message_echo "ERROR: The temporary database has not been created yet." >&2
			exit 1
		fi
		COMMAND_SAVE_DIR=$1
		shift || :
		_command_parse_args__chk_no_arg $#
		if [ -z "$COMMAND_SAVE_DIR" ]
		then
			message_echo "ERROR: Directory to save the temporary database archive is not specified." >&2
			exit 1
		fi
		if [ ! -d "$COMMAND_SAVE_DIR" ]
		then
			message_echo "ERROR: Directory [$COMMAND_SAVE_DIR] is not found." >&2
			exit 1
		fi
		;;
	load )
		misc_chk_privilege
		if database_maintain_chk_use
		then
			message_echo "ERROR: A temporary database exists." >&2
			message_echo "You must execute" >&2
			message_echo "        ${APPNAME} clean" >&2
			message_echo "before executing \"load\" command." >&2
			exit 1
		fi
		COMMAND_LOAD_FILE=$1
		shift || :
		_command_parse_args__chk_no_arg $#
		if [ -z "$COMMAND_LOAD_FILE" ]
		then
			message_echo "ERROR: No temporary database archive is specified." >&2
			exit 1
		fi
		if [ ! -f "$COMMAND_LOAD_FILE" ]
		then
			message_echo "ERROR: No such temporary database archive is found." >&2
			exit 1
		fi
		COMMAND_LOAD_FILE=`realpath "$COMMAND_LOAD_FILE"`
		;;
	glob | pkg )
		_command_parse_args__chk_glob_args $#
		;;
	options )
		_command_parse_args__chk_no_arg $#
		;;
	reconf | rmconf | forget | escape | restore | make )
		misc_chk_privilege
		temp_warn_obsolete_temp_db >&2
		_command_parse_args__chk_glob_args $#
		;;
	pkgsanity )
		misc_chk_privilege
		temp_warn_obsolete_temp_db >&2
		;;
	packupgrade )
		misc_chk_privilege
		COMMAND_OPERATION=$1
		shift || :
		case $COMMAND_OPERATION in
		create | clean )
			_command_parse_args__chk_no_arg $#
			;;
		crop )
			COMMAND_PACKUPGRADE_SAVEPATH=${1:-${PROGRAM}-upgrade.tar.gz}
			shift || :
			_command_parse_args__chk_no_arg $#
			[ ! -d "${COMMAND_PACKUPGRADE_SAVEPATH}" ] ||  COMMAND_PACKUPGRADE_SAVEPATH=${COMMAND_PACKUPGRADE_SAVEPATH}/${PROGRAM}-upgrade.tar.gz
			;;
		*)
			message_echo "ERROR: Invalid operation [$COMMAND_OPERATION]; it must be \"create\" \"crop\" or \"clean\"." >&2
			exit 1
			;;
		esac
		;;
	show )
		case $COMMAND_SHOW_SUBJECT in
		todo | done | redo | resolved | failure | deleted | restored | fossil | conflict | leaves | obsolete )
			database_query_chk_preparation_completion;;
		esac
		temp_warn_obsolete_temp_db >&2
		if ! expr "$1" : '@.*' > /dev/null
		then
			COMMAND_SHOW_SUBJECT=${1:-todo}
			shift || :
		else
			COMMAND_SHOW_SUBJECT=todo
		fi
		COMMAND_SHOW_OPTIONS=$1
		if expr "$COMMAND_SHOW_OPTIONS" : '@.*' > /dev/null
		then
			COMMAND_SHOW_DEPTAG=`expr "$COMMAND_SHOW_OPTIONS," : '@\(.*\)' | cut -d , -f 1` || :
			COMMAND_SHOW_LEVEL=`expr "$COMMAND_SHOW_OPTIONS," : '@\(.*\)' | cut -d , -f 2` || :
			case $COMMAND_SHOW_DEPTAG in
			all | run | build | '' )	;;
			*)
				message_echo "ERROR: Invalid show option [$COMMAND_SHOW_OPTIONS]." >&2
				exit 1
				;;
			esac
			case $COMMAND_SHOW_LEVEL in
			full | direct | '' )	;;
			*)
				message_echo "ERROR: Invalid show option [$COMMAND_SHOW_OPTIONS]." >&2
				exit 1
				;;
			esac
			shift || :
		fi
		case $COMMAND_SHOW_SUBJECT in
		todo | done | redo | resolved | inst_by_pkg | inst_built_default | inst_built_custom | failure | taboo | freeze | need | noneed | restored | deleted | fossil | conflict | moved )
			_command_parse_args__chk_no_arg $#
			;;
		leaves | obsolete )
			COMMAND_SHOW_KEYWORD=$1
			case $COMMAND_SHOW_KEYWORD in
			'' | selected | unselected )	;;
			*)
				message_echo "ERROR: Invalid show $COMMAND_SHOW_SUBJECT keyword [$COMMAND_SHOW_KEYWORD]." >&2
				exit 1
				;;
			esac
			shift || :
			_command_parse_args__chk_no_arg $#
			;;
		build_conflict_pkgs | inst_conflict_pkgs | initrequirements | requirements | initdependents | dependents | conflict_files | status | errormessage )
			_command_parse_args__chk_glob_args $#
			;;
		*)
			message_echo "ERROR: Invalid subject [$COMMAND_SHOW_SUBJECT]." >&2
			exit 1
			;;
		esac
		;;
	all | prepare )
		COMMAND_DO_MODE=$COMMAND_MODE
		COMMAND_MODE=do
		misc_chk_privilege
		temp_warn_obsolete_temp_db >&2
		_command_parse_args__chk_no_arg $#
		;;
	redo | do )
		COMMAND_DO_MODE=${1:-all}
		shift || :
		case $COMMAND_DO_MODE in
		all | prepare );;
		*)
			message_echo "ERROR: Invalid operation mode [$COMMAND_DO_MODE]." >&2
			exit 1
			;;
		esac
		misc_chk_privilege
		temp_warn_obsolete_temp_db >&2
		_command_parse_args__chk_no_arg $#
		if [ "$COMMAND_DO_MODE" = prepare ]
		then
			COMMAND_RESTART=prepare
		else
			COMMAND_RESTART=
		fi
		;;
	*)
		message_echo "ERROR: Invalid command [$COMMAND_MODE]." >&2
		exit 1
		;;
	esac
	COMMAND_SHIFT=$(($num_args_init - $#))
}

# ============= Execute command operations which do not need package tools =============
command_all_exec_without_pkgtools ()
{
	case $COMMAND_MODE in
	clean | load )
		command_exec_without_pkgtools__notify_reset_options
		case $COMMAND_MODE in
		clean )
			message_echo "Starting to clean up the temporary database..."
			database_maintain_clean_all
			;;
		load )
			message_echo "Starting to load the temporary database from the archive..."
			database_maintain_load "$COMMAND_LOAD_FILE"
			;;
		esac
		[ $opt_no_opening_message = yes ] || message_echo "Done"
		exit
		;;
	esac
}

# ============= Execute command operations which are irrespective of option settings =============
command_all_exec_irrespective_of_saved_options ()
{
	local arcfile origins_match pkgs_match
	case $COMMAND_MODE in
	ok )
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		command_flexconf_update_ok "$COMMAND_OPERATION" "$@"
		exit
		;;
	taboo | freeze | need | noneed )
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		mkdir -p "${DBDIR}/journal"
		echo "$@" >> ${DBDIR}/journal/$COMMAND_MODE
		case $COMMAND_MODE in
		taboo )
			command_flexconf_update_taboo "$COMMAND_OPERATION" "$@"
			;;
		freeze )
			command_flexconf_update_freeze "$COMMAND_OPERATION" "$@"
			;;
		need )
			program_deregister_stage_complete DETERMINE_SPECIFIED_TARGETS
			program_deregister_stage_complete ALL_COMPLETE
			command_flexconf_update_need "$COMMAND_OPERATION" "$@"
			;;
		noneed )
			program_deregister_stage_complete INSPECT_PRIMARY_LEAF_PORTS
			program_deregister_stage_complete ALL_COMPLETE
			command_flexconf_update_noneed "$COMMAND_OPERATION" "$@"
			;;
		esac
		exit
		;;
	reselect )
		case $COMMAND_OPERATION in
		leaves )
			if ! deinstall_select_leaf_ports_to_delete force
			then
				case $? in
				2 )
					message_echo "INFO: No leaf port is found."
					;;
				3 )
					message_echo "INFO: Leaf ports are undefined because requirements of some ports are not fully inspected."
					;;
				esac
			else
				program_deregister_stage_complete COLLECT_LEAF_PORTS_TO_DELETE
				program_deregister_stage_complete ALL_COMPLETE
			fi
			;;
		obsolete )
			if ! deinstall_select_obsolete_ports_to_delete force
			then
				case $? in
				2 )
					message_echo "INFO: No obsolete package is found."
					;;
				esac
			else
				program_deregister_stage_complete COLLECT_OBSOLETE_PORTS_TO_DELETE
				program_deregister_stage_complete ALL_COMPLETE
			fi
			;;
		esac
		exit
		;;
	save )
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		arcfile=`realpath "$COMMAND_SAVE_DIR"`/${APPNAME}_`date +%Y%m%d_%H%M%S`.tar.gz
		message_echo "Starting to save the temporary database as [$arcfile]..."
		database_maintain_save "$arcfile"
		[ $opt_no_opening_message = yes ] || message_echo "Done"
		exit
		;;
	glob )
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		origins_match=`pkgsys_eval_ports_glob "$@"`
		if  [ -n "$origins_match" ]
		then
			message_echo "Evaluated flavored port origins are as follows:"
			echo "$origins_match"
		else
			message_echo "No matching flavored port origin is found."
		fi
		exit
		;;
	pkg )
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Evaluated installed package are as follows:"
		pkgs_match=`pkgsys_eval_ports_glob "$@" | while read origin
		do
			pkgsys_get_installed_pkg_from_origin "$origin"
		done`
		if  [ -n "$pkgs_match" ]
		then
			message_echo "Evaluated installed package are as follows:"
			echo "$pkgs_match"
		else
			message_echo "ERROR: No matching package is installed." >&2
			temp_terminate_process () { :; }
			exit 1
		fi
		exit
		;;
	reconf | rmconf | forget | escape | restore | pkgsanity | packupgrade )
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		case $COMMAND_MODE in
		reconf | rmconf )
			command_pkgs_port_option_conf "$@"
			;;
		forget )
			command_forget "$@"
			;;
		escape )
			command_pkgs_escape "$@"
			;;
		restore )
			command_pkgs_restore "$@"
			;;
		pkgsanity )
			command_pkgs_pkgsanity "$@"
			;;
		packupgrade )
			command_pkgs_packupgrade
			;;
		esac
		[ $opt_no_opening_message = yes ] || message_echo "Done"
		exit
		;;
	redo )
		command_do_redo__command_all_exec_irrespective_of_saved_options
		;;
	esac
}

# ============= Execute command operations which should be done without upgrade of tools =============
command_all_exec_before_tools_upgrade ()
{
	local glob origins errno
	case $COMMAND_MODE in
	reset )
		message_echo "Starting to reset the temporary database by preserving the initial snapshot of installed packages..."
		if [ ! -d "${DBDIR}" ]
		then
			[ $opt_no_opening_message = yes ] || message_echo "INFO: No temporary database is built yet."
		else
			database_maintain_reset "$COMMAND_OPERATION"
			[ $opt_no_opening_message = yes ] || message_echo "INFO: Option settings and taboo/freeze/need/noneed lists are preserved."
		fi
		[ $opt_no_opening_message = yes ] || message_echo "Done"
		exit
		;;
	options )
		options_show_all
		exit
		;;
	show )
		command_show "$@"
		exit
		;;
	make )
		command_pkgs_make_ports "$@"
		exit
		;;
	esac
}

# ============= Execute command operations which must be done before the database construction =============
command_all_exec_command_specific_preconfiguration ()
{
	case $COMMAND_MODE in
	do )
		command_do_do__command_all_exec_command_specific_preconfiguration
		;;
	redo )
		command_do_redo__command_all_exec_command_specific_preconfiguration
		;;
	esac
}

# ============= Execute command operations of database construction =============
command_all_exec_build_database ()
{
	case $COMMAND_MODE in
	do | redo )
		command_do_prepare
		;;
	esac
}

# ============= Execute command operations which must be done before the main process =============
command_all_exec_before_main ()
{
	case $COMMAND_MODE in
	do | redo )
		case $COMMAND_DO_MODE in
		prepare )
			command_do_end_at_prepare_complete
			exit
			;;
		esac
		;;
	esac
}

# ============= Execute command operations as the main process =============
command_all_exec_main ()
{
	case $COMMAND_MODE in
	do | redo )
		command_do_main
		command_do_ending_process
		;;
	esac
}
