#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# Overlay onto lib/libcommand_do.sh for portsreinstall-upgrade
# - "do" command operation -
# Copyright (C) 2018 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Set up the manifest =============
command_do_setup_manifest ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Setting up the list of ports to reinstall"
		pkg_get_pkgs_timestamps > ${DBDIR}/initial_pkgs_snapshot.csv
		message_echo
	}
	program_exec_and_record_completion SETUP_MANIFEST
}

# ============= Check of package paths =============
command_do_chk_pkg_paths ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='SETUP_MANIFEST'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Checking the paths of package files"
		while read -r cmd target argument additional
		do
			case $cmd in
				add | addtool )
					if ! pkgpath=`pkgsys_pkgname_to_pkgarc "$opt_packages_dir" "$target"`
					then
						message_echo "ERROR: Package for [$target] is not found. Aborting." >&2
						exit 1
					fi
					additional=$pkgpath
					;;
				delete | delete_pattern )
					argument=NA
					;;
				*)
					message_echo "ERROR: Invalid command [$cmd] [$target] [$argument]" >&2
					exit 1
			esac
			printf '%s\t%s\t%s\t%s\n' "$cmd" "$target" "$argument" "$additional"
		done < ${ETCDIR}/manifest.lst > ${DBDIR}/stage.loop_list/manifest_pkgpath
		cp /dev/null "${DBDIR}/stage.loop_list/failed_add"
		message_echo
	}
	program_exec_and_record_completion CHK_PKGS
}

# ============= (Re)installation of new and reinstalled ports =============
command_do_reinst_ports ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='CHK_PKGS'
	_program_exec_restartable_loop_operation__routine ()
	{
		local cmd target argument additional target_regexp timestamp_init timestamp_fin pkgtype_init pkgtype_fin datetime_init datetime_fin origin target_pkg pkg pkg_pattern pkgpath item conflict pkgs_pkgtools
		cmd=`echo "$1" | cut -f 1`
		target=`echo "$1" | cut -f 2`
		argument=`echo "$1" | cut -f 3`
		additional=`echo "$1" | cut -f 4`
		target_pkg=`pkgsys_get_installed_pkg_from_origin "$target"`
		if [ -n "$target_pkg" ]
		then
			target_regexp=`str_escape_regexp "$target_pkg"`
			timestamp_init=`grep -E "^${target_regexp}[[:space:]]" "${DBDIR}/initial_pkgs_snapshot.csv" | cut -f 2`
			timestamp_fin=`grep -E "^${target_regexp}[[:space:]]" "${ETCDIR}/final_pkgs_snapshot.csv" | cut -f 2`
			if [ -n "$timestamp_init" -a -n "$timestamp_fin" ]
			then
				pkgtype_init=`expr "$timestamp_init" : '\([^:]*\):' || :`
				pkgtype_fin=`expr "$timestamp_fin" : '\([^:]*\):' || :`
				if [ -n "$pkgtype_init" -a -n "$pkgtype_fin" -a "x$pkgtype_init" = "x$pkgtype_fin" ]
				then
					datetime_init=`expr "$timestamp_init" : '[^:]*:\([0-9]*\)' || :`
					datetime_fin=`expr "$timestamp_fin" : '[^:]*:\([0-9]*\)' || :`
					[ -n "$datetime_init" -a -n "$datetime_fin" -a "$datetime_init" -ge "$datetime_fin" ] && return
				fi
			fi
		fi
		case $cmd in
			delete )
				origin=$target
				if pkgsys_exists_from_orig "$origin"
				then
					target_pkg=`pkgsys_get_installed_pkg_from_origin "$origin"`
					if [ $opt_dry_run = yes ]
					then
						message_echo "Dry run: Delete $target_pkg"
					else
						message_echo "Deleting $target_pkg..."
						pkg_delete_f "$target_pkg" || \
							{ message_echo "(Ignored)" >&2; }
					fi
				fi
				;;
			delete_pattern )
				pkg_pattern=$target
				pkgsys_get_installed_pkg_from_glob "$pkg_pattern" | while read pkgname
				do
					if [ $opt_dry_run = yes ]
					then
						message_echo "Dry run: Delete $pkgname"
					else
						message_echo "Deleting $pkgname..."
						pkg_delete_f "$pkgname" || \
							{ message_echo "(Ignored)" >&2; }
					fi
				done
				;;
			add )
				pkg=$target
				pkgpath=$additional
				if ! pkg_info_e "$pkg"
				then
					if [ $opt_dry_run = yes ]
					then
						message_echo "Dry run: Add $pkgpath"
					else
						if ! pkg_add_fF "$pkgpath"
						then
							message_echo "(Continue anyway)" >&2
							item=`printf '%s\t%s' "$pkg" "$pkgpath"`
							fileedit_add_a_line_if_new "$item" "${DBDIR}/stage.loop_list/failed_add"
						fi
					fi
				fi
				;;
			addtool )
				pkgpath=$additional
				origin=`expr "$argument" : '\([^|]*\)|'`
				conflict=`expr "$argument" : '[^|]*|\(.*\)' | tr '|' '\n'`
				pkgs_pkgtools=`{
					echo "$conflict" | while read -r pkg_pattern
					do
						[ -n "$pkg_pattern" ] || continue
						pkgsys_get_installed_pkg_from_glob "$pkg_pattern" || :
					done
					pkgsys_get_installed_pkg_from_origin "$origin"
				} | sort -u`
				if [ $opt_dry_run = yes ]
				then
					echo $pkgs_pkgtools | while read pkgname
					do
						message_echo "Dry run: Delete $pkgname (package management tool)"
					done
					message_echo "Dry run: Add $pkgpath (package management tool)"
				else
					message_echo "Upgrading $pkgs_pkgtools..."
					pkg_delete_f $pkgs_pkgtools || \
						{ message_echo "(Ignored)" >&2; }
					pkg_add_tools "$pkgpath" || pkg_rescue_tools
				fi
				;;
		esac
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Smart update"
		pkg_rescue_tools
		program_exec_restartable_loop_operation manifest_pkgpath
		cp /dev/null "${DBDIR}/stage.loop_list/failed_add_again"
		message_echo
	}
	program_exec_and_record_completion SMART_UPDATE
}

# ============= Deinstallation of remaining old packages =============
# This may be needed if clean command was executed in the builder chroot environment.
command_do_deinst_remining_olg_pkgs ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='SMART_UPDATE'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Deinstalling remaining old packages if any..."
		pkg_get_pkgs_timestamps | \
			grep -vE "`pkgsys_pkgtools_pkgs_filter_regexp`"> ${TMPDIR}/current_pkgs_snapshot.csv
		grep -Fxv -f "${ETCDIR}/final_pkgs_snapshot.csv" "${TMPDIR}/current_pkgs_snapshot.csv" | \
			cut -f 1 | while read pkg_old
		do
			pkg_delete_f "$pkg_old"
		done
		message_echo
	}
	program_exec_and_record_completion DEINST_REMAINING_OLD_PKGS
}

# ============= Retrial of failed adding packages =============
command_do_retrial ()
{
	local PROGRAM_DEPENDS
	while [ `wc -l < ${DBDIR}/stage.loop_list/failed_add` -gt 0 ]
	do
		rm -f  "${DBDIR}/RETRIAL_ADDING:restart"
		PROGRAM_DEPENDS='DEINST_REMAINING_OLD_PKGS'
		_program_exec_restartable_loop_operation__routine ()
		{
			local target pkgpath item
			target=`echo "$1" | cut -f 1`
			pkgpath=`echo "$1" | cut -f 2`
			if ! pkg_info_e "$target"
			then
				if ! pkg_add_fF "$pkgpath"
				then
						message_echo "(Continue anyway)" >&2
						item=`printf '%s\t%s' "$target" "$pkgpath"`
						fileedit_add_a_line_if_new "$item" "${DBDIR}/stage.loop_list/failed_add_again"
				fi
			fi
		}
		_program_exec_and_record_completion__operation ()
		{
			message_section_title "Retrial of failed installation"
			program_exec_restartable_loop_operation failed_add
			touch "${DBDIR}/stage.loop_list/failed_add_again"
			if [ `wc -l < ${DBDIR}/stage.loop_list/failed_add_again` -eq 0 ] || diff -q "${DBDIR}/stage.loop_list/failed_add_again" "${DBDIR}/stage.loop_list/failed_add" > /dev/null 2>/dev/null
			then
				cp /dev/null "${DBDIR}/stage.loop_list/failed_add"
			else
				touch "${DBDIR}/RETRIAL_ADDING:restart"
				mv "${DBDIR}/stage.loop_list/failed_add_again" "${DBDIR}/stage.loop_list/failed_add"
				touch "${DBDIR}/stage.loop_list/failed_add_again"
			fi
			message_echo
		}
		program_exec_and_record_completion RETRIAL_ADDING
		[ -e "${DBDIR}/RETRIAL_ADDING:restart" ] && program_reset_loop_for_stage RETRIAL_ADDING
	done
}

# ============= Main operation of do =============
command_do_main ()
{
	# Set up the manifest
	command_do_setup_manifest

	# Check of package paths
	command_do_chk_pkg_paths

	# (Re)installation of new and reinstalled ports
	command_do_reinst_ports

	# Retrial of failed adding packages
	[ $opt_dry_run = yes ] || command_do_retrial
}

# ============= Notice of failure =============
command_do_failure_notice ()
{
	touch "${DBDIR}/stage.loop_list/failed_add_again"
	if [ `wc -l < ${DBDIR}/stage.loop_list/failed_add_again` -gt 0 ]
	then
		message_echo "ERROR: The following packages failed to (re)install." >&2
		${DBDIR}/stage.loop_list/failed_add_again | cut -f 1 | message_cat >&2
		exit 1
	fi
}

# ============= Ending process =============
command_do_ending_process ()
{
	command_do_failure_notice
	temp_terminate_process () { :; }
	if [ $opt_no_opening_message = yes ]
	then
		message_echo "Done as ${APPNAME}"
		return
	fi
	message_section_title "COMPLETELY DONE"
	message_echo "- E N D -"
}
