#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# Overlay onto lib/libcommand_do.sh for portsreinstall-chroot
# - "do" command operation -
# Copyright (C) 2018-2022 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Check the completion of build at the chroot environment =============
command_do_chk_build_completion ()
{
	[ -e "$opt_basedir/builder/${DBDIR_PROG}"/stage.complete/ALL_COMPLETE ]
}

# ============= Check whether the termination of portsreinstall at the chroot environment was a unionfs error =============
command_do_chk_unionfs_error_at_chroot ()
{
	[ -e "$opt_basedir/builder/${DBDIR_PROG}"/execflag/unionfs_error ]
}

# ============= Get the recent chroot exit status of the builder chroot environment =============
command_do_get_recent_chroot_exit_status ()
{
	cat "${DBDIR}/recent_exit_status"
}

# ============= Enter the shell of the guest =============
command_do_chroot_enter_shell ()
{
	local shellcmd exitstatus
	message_cat <<eof
=========================================================
You entered the builder chroot environment.
Complete the update of packages by portsreinstall(8),
Then exit to the target environment.
=========================================================

eof
	shellcmd=`id -P | sed -E 's|.*:([^:]*)$|\1|'`
	case `basename "$shellcmd"` in
	sh | ksh | zsh | '' )
		shellcmd="env PS1='\\\$@[builder]\h\\$ ' ${shellcmd:-/bin/sh}"
		;;
	bash )
		shellcmd="env PS1='\u@[builder]\h\\$ ' $shellcmd"
		;;
	esac
	chroot "$opt_basedir"/builder "$shellcmd"
	exitstatus=$?
	echo "$exitstatus" > ${DBDIR}/recent_exit_status
	return $exitstatus
}

# ============= Execute in the chroot environment automatically repeating by fixing the file systems =============
_command_do_auto_repeat_by_fixing_fs ()
{
	until "$@" || ! command_do_chk_unionfs_error_at_chroot
	do
		message_echo "INFO: Retrying by re-mounting the file systems for the builder chroot environment." >&2
		fs_unmount
		fs_mount
	done
}

# ============= Continue reinstallation in the guest =============
command_do_chroot_enter_fullcourse ()
{
	local exitstatus opt_quiet
	message_section_title "Full course main: Reinstallation process"
	[ $opt_batch_mode = yes ] && opt_quiet='-a '
	chroot "$opt_basedir"/builder /usr/bin/env ${PROGRAM} -S $opt_quiet
	exitstatus=$?
	echo "$exitstatus" > ${DBDIR}/recent_exit_status
	return $exitstatus
}

# ============= Enter the guest and execute portsreinstall(8) automatically =============
command_do_chroot_auto ()
{
	local exitstatus opt_quiet
	[ $opt_batch_mode = yes ] && opt_quiet='-a '
	message_cat <<eof
=========================================================
Automatic execution inside the builder chroot environment:
portsreinstall -S $@
=========================================================

eof
	chroot "$opt_basedir"/builder /usr/bin/env ${PROGRAM} -S $opt_quiet "$@"
	exitstatus=$?
	echo "$exitstatus" > ${DBDIR}/recent_exit_status
	return $exitstatus
}

# ============= Enter the guest =============
command_do_chroot_enter ()
{
	if [ $COMMAND_MODE = auto ]
	then
		[ $opt_fullcourse = yes ] && message_section_title "Full course main"
		_command_do_auto_repeat_by_fixing_fs command_do_chroot_auto "$@"
	elif [ $opt_fullcourse = no ]
	then
		if ! command_do_chroot_enter_shell
		then
			message_echo "INFO: The last exit status in the builder chroot environment was non-zero." >&2
			command_do_chk_unionfs_error_at_chroot && \
				message_echo "INFO: Executing ${APPNAME} again may be sufficient for the troubleshooting." >&2
		fi
	else
		_command_do_auto_repeat_by_fixing_fs command_do_chroot_enter_fullcourse
	fi
	return `command_do_get_recent_chroot_exit_status`
}

# ============= Update of the files affecting package installation of the host environment  =============
command_do_update_host_files ()
{
	local dirpath filepath
	[ "x$opt_share_port_pkgs_dirs" = xyes ] && return
	message_section_title "Update the files affecting package installation of the host environment"
	message_echo "INFO: targets are packages, distfiles, port options and ports/packages management tools configurations"
	for dirpath in "${DISTDIR}" "${PACKAGES}" "${PORT_DBDIR}"
	do
		(
			mkdir -p "$dirpath"
			cd $dirpath && find . -type f
		) | while read file
		do
			[ -e "$opt_basedir/builder$dirpath/$file" ] || rm "$dirpath/$file"
		done
		(
			cd "$opt_basedir"/builder$dirpath && find . -type f
		) | while read file
		do
			[ ! -e "$dirpath/$file" -o "$opt_basedir/builder$dirpath/$file" -nt "$dirpath/$file" ] || continue
			nodedir=`dirname "$dirpath/$file"`
			mkdir -p "$nodedir"
			cp -p "$opt_basedir/builder$dirpath/$file" "$dirpath/$file"
		done
		find -d $dirpath -mindepth 1 -type d -empty -delete
	done
	for filepath in "${LOCALBASE}"/etc/portsreinstall.conf "${LOCALBASE}"/etc/pkgtools.conf "${LOCALBASE}"/etc/pkg.conf /etc/make.conf "${DBDIR}"
	do
		if [ -e "$opt_basedir"/builder$filepath ]
		then
			[ ! -e $filepath -o "$opt_basedir"/builder$filepath -nt $filepath ] && cp -p "$opt_basedir"/builder$filepath $filepath
		else
			rm -f $filepath
		fi
	done
	message_echo
}

# ============= Ending process =============
command_do_ending_process ()
{
	temp_terminate_process () { :; }
	if [ $opt_no_opening_message = yes ]
	then
		message_echo "Done as ${APPNAME}"
		return
	fi
	message_section_title "COMPLETELY DONE"
	message_echo "- E N D -"
}

# ============= Initial clean up =============
command_do_chroot_cleanup  ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=
	_program_exec_and_record_completion__operation ()
	{
		fs_destroy
		message_echo
	}
	program_exec_and_record_completion CLEANUP_GUEST
}

# ============= Common process after exiting from the chroot environment =============
command_do_chroot_postprocess ()
{
		if ! command_do_chk_build_completion
		then
			message_echo "INFO: Exited from the builder chroot environment without completing the build." >&2
			exit 130
		fi
		message_echo "INFO: The package build completed."
		message_echo "Is it OK to apply the upgrade to the host environment? ([y]/n)"
		if ! message_query_yn_default_yes
		then
			message_echo "INFO: Terminated because the continuation is stopped by the user."
			exit 130
		fi
}

# ============= Starter process in the chroot environment: Update the ports tree =============
command_do_starter_portsnap ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='CLEANUP_GUEST'
	_program_exec_and_record_completion__operation ()
	{
		local mode stdout
		[ $opt_fullcourse = no ] && return
		message_section_title "Full course starter: Update the ports tree"
		pkgsys_update_portstree
		message_echo
	}
	program_exec_and_record_completion STARTER_GUEST_PORTS_TREE
}

# ============= Starter process of the full course: Update the package repository =============
command_do_starter_pkg ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='STARTER_GUEST_PORTS_TREE'
	_program_exec_and_record_completion__operation ()
	{
		local opts
		[ $opt_fullcourse = no ] && return
		message_section_title "Full course starter: Update the package repository"
		pkg_update_pkgrepository
		message_echo
	}
	program_exec_and_record_completion STARTER_GUEST_PKG_REPO
}

# ============= Starter process of the full course: Clean the temporary database =============
command_do_starter_clean ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='STARTER_GUEST_PKG_REPO'
	_program_exec_and_record_completion__operation ()
	{
		local opts
		[ $opt_fullcourse = no ] && return
		message_section_title "Full course starter: Clean the temporary database"
		if [ $opt_batch_mode = no ]
		then
			opts=
		else
			opts='-q'
		fi
		chroot "$opt_basedir"/builder /usr/bin/env ${PROGRAM} $opts clean force
		message_echo
	}
	program_exec_and_record_completion STARTER_GUEST_CLEAN
}

# ============= Main process in the chroot environment: Initiate the reinstallation process =============
command_do_main_init_resinst ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='STARTER_GUEST_CLEAN'
	_program_exec_and_record_completion__operation ()
	{
		local opts opt_smart
		[ $opt_fullcourse = no ] && return
		message_section_title "Full course main: Initiate the reinstallation process"
		opts=
		[ $opt_batch_mode = yes ] && opts='-Ya'
		opt_smart=
		[ $opt_upgraded_system = no ]  && opt_smart='-q'
		chroot "$opt_basedir"/builder /usr/bin/env ${PROGRAM} -SJCGjx $opts $opt_smart
		message_echo
	}
	program_exec_and_record_completion STARTER_GUEST_INIT_REINST
}

# ============= Enter the chroot environment =============
command_do_chroot_enter_if_incomplete ()
{
	local PROGRAM_DEPENDS
	command_do_chk_build_completion || program_deregister_stage_complete UPGRADE_GUEST
	PROGRAM_DEPENDS='STARTER_GUEST_INIT_REINST'
	_program_exec_and_record_completion__operation ()
	{
		local exitstatus
		command_do_chroot_enter "$@"
		exitstatus=`command_do_get_recent_chroot_exit_status`
		[ "$exitstatus" -gt 0 ] && exit $exitstatus
		command_do_chroot_postprocess
		message_echo
	}
	program_exec_and_record_completion UPGRADE_GUEST "$@"
}

# ============= Pack (re)installed packages at the chroot environment =============
command_do_chroot_pack_update ()
{
	local PROGRAM_DEPENDS
	if [ -e "$opt_basedir"/store/complete ]
	then
		if ! command_do_chk_build_completion
		then
			program_deregister_stage_complete PACK_PKGS
			rm -rf "$opt_basedir"/store/complete
		fi
	else
		program_deregister_stage_complete PACK_PKGS
	fi
	PROGRAM_DEPENDS='UPGRADE_GUEST'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Pack the upgrades"
		[ $opt_batch_mode = yes ] && opt_quiet='-a '
		chroot "$opt_basedir"/builder /bin/csh -c "${PROGRAM} -S $opt_quiet packupgrade create && ${PROGRAM} -S $opt_quiet packupgrade crop /.${PROGRAM}"
	}
	program_exec_and_record_completion PACK_PKGS
}

# ============= Update of the files affecting package installation of the host environment  =============
command_do_chroot_update_host_files ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='PACK_PKGS'
	_program_exec_and_record_completion__operation ()
	{
		command_do_update_host_files
	}
	program_exec_and_record_completion UPDATE_HOST
}

# ============= Update of packages at the host environment =============
command_do_chroot_update_host_pkgs ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='UPDATE_HOST'
	_program_exec_and_record_completion__operation ()
	{
		local storedir
		message_section_title "Extract the upgrade archive"
		storedir=$opt_basedir/store
		rm -rf "$storedir/work"
		mkdir "$storedir/work"
		tar xzf "$storedir/portsreinstall-upgrade.tar.gz" -C "$storedir/work"
		message_echo
	}
	program_exec_and_record_completion EXTRACT_SCRIPT_ARCHIVE
	PROGRAM_DEPENDS='EXTRACT_SCRIPT_ARCHIVE'
	_program_exec_and_record_completion__operation ()
	{
		local opt_quiet
		message_section_title "Update the packages at the host environment"
		opt_quiet=
		[ $opt_batch_mode = yes ] && opt_quiet='-a '
		$opt_basedir/store/work/portsreinstall-upgrade -S clean
		$opt_basedir/store/work/portsreinstall-upgrade -S $opt_quiet -P "${PACKAGES}/${PKGREPOSITORYSUBDIR}"
	}
	program_exec_and_record_completion UPGRADE_HOST
}

# ============= Main pre-operation of do/auto =============
command_do_pre ()
{
	# Build and mount of the chroot environment
	fs_build_chroot
	fs_mount
}

# ============= Main post-operation of do/auto =============
command_do_post ()
{
	# Pack (re)installed packages at the guest
	command_do_chroot_pack_update
	
	# Update of the files affecting package installation of the host environment 
	command_do_chroot_update_host_files
	
	# Update of packages at the host environment
	command_do_chroot_update_host_pkgs
}
