#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Commands of ports/packages operations -
# Copyright (C) 2018-2022 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Operation of reconf/rmconf command =============
command_pkgs_port_option_conf ()
{
	local make_target origin
	case $COMMAND_MODE in
	reconf )
		message_echo "Reconfigure the specified port options."
		make_target=config
		;;
	rmconf )
		message_echo "The specified port options are reset to the default."
		make_target=rmconfig
		;;
	esac
	message_echo "Affected parts of the temporary database are reset automatically."
	for origin in `pkgsys_eval_ports_glob "$@"`
	do
		message_echo "Re-configure $origin:"
		database_build_make "$origin" $make_target
		database_build_patch_reconf "$origin"
	done
	program_deregister_stage_complete PREPARE_FOR_INSPECT_ALL_DEPENDENCIES
	program_deregister_stage_complete ALL_COMPLETE
}

# ============= Operation of escape command =============
command_pkgs_escape ()
{
	local origin pkg backup_pkg
	message_echo "Backing up and deleting the following packages for a temporary escape:"
	message_echo
	for origin in `pkgsys_eval_ports_glob "$@"`
	do
		pkgsys_register_evaluated_globs add "${DBDIR}/taboo.list" "$origin"
		message_echo "  Registered $origin as taboo."
		pkg=`pkgsys_get_installed_pkg_from_origin "$origin"`
		[ -n "$pkg" ] || continue
		if backup_pkg=`pkgsys_get_backup_pkg "$origin"`
		then
			message_echo "INFO: A backup package for $pkg ($origin) already exists as $backup_pkg."
		elif backup_pkg=`pkgsys_create_backup_pkg "$pkg" "${DBDIR}/backup_packages"`
		then
			message_echo "  Backed up $pkg ($origin) into $backup_pkg"
		else
			message_echo "ERROR: Failed to back up $pkg ($origin)." >&2
			message_echo >&2
			continue
		fi
		pkg_delete_f "$pkg" || \
		{
			message_echo "ERROR: Failed to deinstall $pkg ($origin)." >&2
			message_echo >&2
		}
		message_echo "  Deinstalled $pkg ($origin)."
		message_echo
	done
	fileedit_combine_lists "${DBDIR}/conf/TABOO:PORTS.parsed" "${DBDIR}/taboo.list" > ${DBDIR}/taboo.all.list
}

# ============= Operation of restore command =============
command_pkgs_restore ()
{
	local tmp_done_orig origin pkg exists_old_origins origin_orig pkg_orig origin_replace pkg_replace backup_pkg
	message_echo "Restoring the following temporary escaped packages:"
	message_echo
	tmp_done_orig=${TMPDIR}/command_pkgs_restore::restore::done_orig
	cp /dev/null "$tmp_done_orig"
	for origin in `pkgsys_eval_ports_glob "$@"`
	do
		pkgsys_register_evaluated_globs remove "${DBDIR}/taboo.list" "$origin"
		message_echo "  Deregistered $origin from taboo."
		env LANG=C grep -Fx -q "$origin" "$tmp_done_orig" 2> /dev/null || :
		if pkgsys_exists_from_orig "$origin"
		then
			pkg=`pkgsys_get_installed_pkg_from_origin "$origin"`
			message_echo "WARNING: $pkg ($origin) is already installed." >&2
			message_echo >&2
			continue
		fi
		exists_old_origins=no
		for origin_orig in `database_query_initial_orgins "$origin"`
		do
			if [ "x$origin_orig" = "x$origin" ] && pkgsys_exists_from_orig "$origin_orig"
			then
				pkg_orig=`pkgsys_get_installed_pkg_from_origin "$origin_orig"`
				message_echo "WARNING: An original version of $origin ($pkg_orig, $origin_orig) is already installed." >&2
				message_echo >&2
				exists_old_origins=yes
			fi
		done
		[ $exists_old_origins = yes ] && continue
		origin_replace=`echo "$origin" \
			| sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern"`
		if [ "x$origin_replace" != "x$origin" ]
		then
			if pkgsys_exists_from_orig "$origin_replace"
			then
				pkg_replace=`pkgsys_get_installed_pkg_from_origin "$origin_replace"`
				message_echo "WARNING: A replacement of $origin ($pkg_replace, $origin_replace) is already installed." >&2
				message_echo >&2
				continue
			fi
			if backup_pkg=`pkgsys_get_backup_pkg "$origin_replace" 2> /dev/null`
			then
				message_echo "INFO: $origin is replaced with $origin_replace ($pkg_replace)."
				echo "$origin_replace" >> $tmp_done_orig
				origin=$origin_replace
			fi
		else
			backup_pkg=
		fi
		if [ -z "$backup_pkg" ] && ! backup_pkg=`pkgsys_get_backup_pkg "$origin" 2> /dev/null`
		then
			message_echo "ERROR: Backup for $origin is not found." >&2
			message_echo >&2
			continue
		fi
		pkg=`pkgsys_pkgarc_to_pkgname "$backup_pkg"`
		if reinstall_chk_forbidden_conflicts "$pkg"
		then
			message_echo "WARNING: $pkg ($origin) is skipped because it conflicts with installed packages." >&2
			message_echo >&2
			continue
		fi
		if ! pkg_add_fF "$backup_pkg"
		then
			message_echo "ERROR: Failed to restore $pkg ($origin)." >&2
			message_echo >&2
		fi
		message_echo "  Restored $pkg ($origin)."
		message_echo
	done
	fileedit_combine_lists "${DBDIR}/conf/TABOO:PORTS.parsed" "${DBDIR}/taboo.list" > ${DBDIR}/taboo.all.list
}

# ============= Operation of pkgsanity command =============
command_pkgs_pkgsanity ()
{
	local tmp_list nlines iline pkg origin port_path is_reinstall_encouraged
	tmp_list=${TMPDIR}/command_pkgs_pkgsanity:list
	if [ $# -eq 0 ]
	then
		message_echo "Sanity check of the installed files for each package:"
		message_echo
		pkg_info_Ea > $tmp_list.pkgs
		pkg_info_all_flavored_origins > $tmp_list.orgs
	else
		message_echo "Examining the installed files for each specified package:"
		message_echo
		pkgsys_eval_ports_glob "$@" > $tmp_list.orgs
		while read origin
		do
			pkgsys_get_installed_pkg_from_origin "$origin"
		done < $tmp_list.orgs | env LANG=C grep -v '^[[:space:]]*$' > $tmp_list.pkgs || {
			message_echo "WARNING: No such globs match any installed package." >&2
			temp_terminate_process () { :; }
			exit 1
		}
	fi
	message_echo "<< Phase 1: Check and fix duplicated packages registrations for the same port origin >>"
	message_echo
	while read origin
	do
		[ `pkgsys_get_installed_pkg_from_origin "$origin" | wc -l` -gt 1 ] || continue
		pkgsys_get_installed_pkg_from_origin "$origin" | while read pkg
		do
			echo "$((0+`pkg_check_sanity \"$pkg\" | wc -l`))" "$pkg"
		done | sort -g > $tmp_list.pkgs_for_an_org
		pkg_valid=`head -n 1 "$tmp_list.pkgs_for_an_org" | cut -d ' ' -f 2`
		message_echo "Port [$origin] has multiply registered packages."
		message_echo "The valid one will be [$pkg_valid]."
		message_echo "Invalid one(s) will be as follows and to be deleted:"
		sed 1d "$tmp_list.pkgs_for_an_org" | cut -d ' ' -f 2 > $tmp_list.pkgs_for_an_org_msg
		message_cat "$tmp_list.pkgs_for_an_org_msg"
		errout=/dev/stderr
		[ $opt_batch_mode = yes ] && errout=/dev/null
		if backup_pkg=`pkgsys_create_backup_pkg "$pkg_valid" "${DBDIR}/backup_packages"`
		message_echo
		then
			pkg_delete_f `cat "$tmp_list.pkgs_for_an_org"` 2> $errout || {
				message_echo "WARNING: Deletion of the broken packages may not be fully successful, but continuing anyway." >&2
				message_echo >&2
			}
			pkg_add_fF "$backup_pkg" || {
				message_echo "WARNING: Reinstallation of the most valid package failed, but continuing anyway." >&2
				message_echo >&2
			}
		else
			message_echo "WARNING: Backup of the most valid package failed, but continuing anyway." >&2
			pkg_delete_f `cat "$tmp_list.pkgs_for_an_org"` 2> $errout || {
				message_echo "WARNING: Deletion of the broken packages may not be fully successful, but continuing anyway." >&2
				message_echo >&2
			}
		fi
	done < $tmp_list.orgs
	
	message_echo "<< Phase 2: Check and mark broken packages for reinstallation >>"
	message_echo
	nlines=`wc -l < $tmp_list.pkgs`
	iline=1
	while [ $iline -le $nlines ]
	do
		pkg=`sed -n ${iline}p "$tmp_list.pkgs"`
		iline=$((${iline}+1))
		for origin in `pkg_info_flavored_origins "$pkg"`
		do
			[ -n "$origin" ] || continue
			env LANG=C grep -q -Fx "$origin" "${DBDIR}/damaged_package" 2>/dev/null && continue
			pkgsys_sanitychk_pkgcontents "$pkg" is_reinstall_encouraged && continue
			port_path=`pkgsys_get_portpath_from_origin "$origin"`
			if [ ! -d "$port_path" ]
			then
				message_echo "WARNING: $pkg ($origin) is obsolete." >&2
				message_echo >&2
				continue
			fi
			if [ $is_reinstall_encouraged = no ]
			then
				if [ $opt_batch_mode = no ]
				then
					message_echo "Do you want to reinstall it? (y/[n])"
					message_query_yn_default_no || continue
				fi
			else
				if [ $opt_batch_mode = no ]
				then
					message_echo "Do you want to reinstall it? ([y]/n)"
					message_query_yn_default_yes || continue
				fi
				database_record_reconf_recover_sanity "$origin"
			fi
		done
	done
}

# ============= Operation of packupgrade command =============
command_pkgs_packupgrade ()
{
	case ${COMMAND_OPERATION} in
	clean )
		command_pkgs_packupgrade_clean
		;;
	create )
		command_pkgs_packupgrade_create
		;;
	crop )
		command_pkgs_packupgrade_crop
		;;
	esac
}

# ============= Operation of packupgrade clean command =============
command_pkgs_packupgrade_clean ()
{
	program_deregister_stage_complete COMMAND_PACKUPGRADE_PREPARE
	rm -rf "${DBDIR}/command_packupgrade"
}

# ============= Stage of preparation in operation of packupgrade create command =============
command_pkgs_packupgrade_create__prepare ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=
	_program_exec_and_record_completion__operation ()
	{
		local dstdir
		message_section_title "Preparation"
		mkdir -p "${PACKAGES}/${PKGREPOSITORYSUBDIR}"
		cp "${DBDIR}/reinst_order.list" "${DBDIR}/stage.loop_list/command_packupgrade_pack"
		cp "${DBDIR}/stage.loop_list/ports_to_delete" "${DBDIR}/stage.loop_list/command_packupgrade_delete"
		dstdir=${DBDIR}/command_packupgrade
		rm -rf "$dstdir" "$dstdir.tar.gz"
		mkdir -p "$dstdir"
		install -d "$dstdir/etc"
		touch "$dstdir/etc/created_packages.lst"
		pkg_get_pkgs_timestamps > $dstdir/etc/final_pkgs_snapshot.csv
		message_echo
	}
	program_exec_and_record_completion COMMAND_PACKUPGRADE_PREPARE
}

# ============= Stage of creating the manifest of deletion in operation of packupgrade create command =============
command_pkgs_packupgrade_create__manifest_delete ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='COMMAND_PACKUPGRADE_PREPARE'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin
		origin=$1
		printf '%s\t%s\t%s\n' delete "$origin" NA >> ${DBDIR}/command_packupgrade/etc/manifest.lst.delete
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Create manifest of deletion"
		program_exec_restartable_loop_operation command_packupgrade_delete
		message_echo
	}
	program_exec_and_record_completion COMMAND_PACKUPGRADE_MANIFEST_DELETE
}

# ============= Stage of creating the manifest of reinstallation in operation of packupgrade create command =============
command_pkgs_packupgrade_create__manifest_reinst ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=COMMAND_PACKUPGRADE_PREPARE
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin dstdir tmp_manifest dbdir_new pkgname pkgarc timestamp_inst timestamp_arc origin_old origin_equiv
		origin=$1
		dstdir=${DBDIR}/command_packupgrade
		tmp_manifest=${TMPDIR}/packupgrade::manifest
		rm -rf "$tmp_manifest"
		dbdir_new=${DBDIR}/requires/$origin
		pkgname=`pkgsys_get_installed_pkg_from_origin "$origin"`
		[ -n "$pkgname" ] || return 0
		if ! env LANG=C grep -Fxq "$pkgname" "$dstdir/etc/created_packages.lst" 2> /dev/null
		then
			if pkg_info_e "$pkgname"
			then
				if pkgarc=`pkgsys_pkgname_to_pkgarc "${PACKAGESDIR}" "$pkgname"`
				then
					timestamp_inst=`pkg_get_pkg_timestamp "$pkgname" 2> /dev/null` || :
					timestamp_arc=`ls -lD %s "$pkgarc" 2> /dev/null | sed -E 's/[[:space:]]+/ /g' | cut -w -f 6 | env LANG=C grep -v '^$'` || :
				else
					timestamp_inst=1
					timestamp_arc=0
				fi
				if [ -z "$timestamp_inst" -o -z "$timestamp_arc" -o "$timestamp_arc" -lt "$timestamp_inst" ]
				then
					if ! ( cd ${PACKAGESDIR}; pkg_create_b "$pkgname" | message_cat )
					then
						message_echo "ERROR: Failed to create package [$pkgname for $origin]." >&2
						exit 1
					fi
				fi
			elif env LANG=C grep -Fxq "$origin" ${DBDIR}/success.run.full.list
			then
				origin_equiv=`database_query_get_equivalent_orgin "$origin"`
				if [ -z "$origin_equiv" ] || env LANG=C grep -Fxq "$origin_equiv" ${DBDIR}/success.run.full.list
				then
					message_echo "ERROR: Necessary package is missing [$origin]." >&2
					exit 1
				fi
			fi
			fileedit_add_a_line_if_new "$pkgname" "$dstdir/etc/created_packages.lst"
		fi
		if pkgsys_is_necessary_pkgtool "$origin"
		then
			pkg_patterns=`pkgsys_get_conflicting_pkgs_patterns install "$origin" | tr '\n' '|'`
			printf '%s\t%s\t%s\n' addtool "$pkgname" "$origin|$pkg_patterns" >> $tmp_manifest
		else
			if [ -e "${DBDIR}/moved_from/$origin/initial_orig" ]
			then
				while read origin_old
				do
					printf '%s\t%s\t%s\n' delete "$origin_old" NA
				done < ${DBDIR}/moved_from/$origin/initial_orig >> $tmp_manifest
			fi
			printf '%s\t%s\t%s\n' delete "$origin" NA >> $tmp_manifest
			pkgsys_get_conflicting_pkgs_patterns install "$origin" | while read pkg_pattern
			do
				printf '%s\t%s\t%s\n' delete_pattern "$pkg_pattern" NA >> $tmp_manifest
			done
			printf '%s\t%s\t%s\n' add "$pkgname" NA >> $tmp_manifest
		fi
		cat "$tmp_manifest" >> $dstdir/etc/manifest.lst.update
	}
	_program_exec_and_record_completion__operation ()
	{
		local PACKAGESDIR
		message_section_title "Creating manifest of (re)installation and package archives..."
		PACKAGESDIR=${PACKAGES}/${PKGREPOSITORYSUBDIR}
		mkdir -p "${PACKAGESDIR}"
		PACKAGESDIR=`realpath "${PACKAGESDIR}"`
		program_exec_restartable_loop_operation command_packupgrade_pack
		message_echo
	}
	program_exec_and_record_completion COMMAND_PACKUPGRADE_MANIFEST_REINST
}

# ============= Stage of packing in operation of packupgrade create command =============
command_pkgs_packupgrade_create__pack ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='COMMAND_PACKUPGRADE_MANIFEST_DELETE'
	_program_exec_and_record_completion__operation ()
	{
		local dstdir
		message_section_title "Packing for the upgrade at the target systems"
		dstdir=${DBDIR}/command_packupgrade
		cat "$dstdir/etc/manifest.lst.delete" "$dstdir/etc/manifest.lst.update" > $dstdir/etc/manifest.lst
		install -m 444 "${DBDIR}"/WITH_PKGNG "${DBDIR}"/conf/complete_setup.sh "${DBDIR}"/conf/setenv.sh "$dstdir/etc"
		install "${SHAREDIR}/bin/${APPNAME}-upgrade" "$dstdir"
		install -d "$dstdir/lib/upgrade" "$dstdir/man/man8"
		( cd "${LIBDIR}" && find . upgrade -depth 1 -type f ) | while read filepath
		do
			install -m 444 "${LIBDIR}/$filepath" "$dstdir/lib/$filepath"
		done
		install -m 444 "${MYPREFIX}/man/man8/${PROGRAM}-upgrade.8.gz" "$dstdir/man/man8"
		tar czf "$dstdir.tar.gz" -C "$dstdir" .
		touch "$dstdir/complete"
	}
	program_exec_and_record_completion COMMAND_PACKUPGRADE_PACK
}

# ============= Operation of packupgrade create command =============
command_pkgs_packupgrade_create ()
{
	if ! program_chk_stage_complete ALL_COMPLETE
	then
		message_echo "ERROR: The reinstallation of ports must be completed before using this command." >&2
		exit 1
	fi
	# Preparation
	command_pkgs_packupgrade_create__prepare
	# Create manifest of deletion
	command_pkgs_packupgrade_create__manifest_delete
	# Create manifest of (re)installation and package archives
	command_pkgs_packupgrade_create__manifest_reinst
	# Packing for the upgrade at the target systems
	command_pkgs_packupgrade_create__pack
}

# ============= Operation of packupgrade crop command =============
command_pkgs_packupgrade_crop ()
{
	local dstdir
	if ! program_chk_stage_complete ALL_COMPLETE
	then
		message_echo "ERROR: The reinstallation of ports must be completed before using this command." >&2
		exit 1
	fi
	dstdir=${DBDIR}/command_packupgrade
	if [ ! -e "$dstdir/complete" ]
	then
		message_echo "ERROR: The execution of \"packupgrade create\" command must be completed before using this command." >&2
		exit 1
	fi
	cp "$dstdir.tar.gz" "${COMMAND_PACKUPGRADE_SAVEPATH}"
	message_echo "INFO: The cropped archive is saved as [${COMMAND_PACKUPGRADE_SAVEPATH}]."
	message_echo
}

# ============= Operation of make command =============
command_pkgs_make_ports ()
{
	glob=$1
	shift || :
	origins=`pkgsys_eval_ports_glob "$glob"`
	[ -n "$origins" ] || message_echo "WARNING: No matching port for the glob pattern $glob" >&2
	echo "$origins" | while read origin
	do
		message_echo "========== [$origin] =========="
		reinstall_make_individual "$origin" "$@"
		errno=$?
		message_echo
		[ $errno -eq 0 ] || exit $errno
	done
}
