#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Operations for queries to the temporary database -
# Copyright (C) 2013-2022 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Show a list of failed ports with their reasons =============
database_query_show_list_failure ()
{
	local tmp_valid
	tmp_valid=${TMPDIR}/database_query_show_list_failure:valid
	if [ `cat "${DBDIR}/failed.list" 2> /dev/null | wc -l` -eq 0 ]
	then
		message_echo "INFO: No item is registered in this list."
		return 1
	fi
	env LANG=C grep -v -E -f "${DBDIR}/ports_to_delete.grep_pattern" "${DBDIR}/failed.list" 2> /dev/null \
		| while read origin
	do
		env LANG=C grep -q -Fx "$origin" "${DBDIR}/stage.loop_list/ports_to_delete" 2> /dev/null \
			|| echo "$origin"
	done > $tmp_valid
	if [ `cat "$tmp_valid" 2> /dev/null | wc -l` -eq 0 ]
	then
		message_echo "INFO: No valid item is registered in this list."
		return 1
	fi
	while read origin
	do
		note=`cat "${DBDIR}/notes/$origin/note_failtre" 2> /dev/null || :`
		resolved=no
		env LANG=C grep -q -Fx "$origin" "${DBDIR}/manually_done.list" 2> /dev/null && resolved=yes
		pkgtag=`cat "${DBDIR}/moved_from/$origin/pkgtag" 2> /dev/null || :`
		if [ $opt_batch_mode = no ]
		then
			case $resolved in
			no )	resolved=;;
			yes )	resolved=', resolved';;
			esac
			if [ -n "$note" ]
			then
				detail="while [$note]"
			else
				detail="by unrecorded reasons"
			fi
			if [ -n "$pkgtag" ]
			then
				echo "$origin ($pkgtag) (error $detail$resolved)"
			else
				echo "$origin (error $detail$resolved)"
			fi
		else
			printf "%s\t%s\t%s\t%s\n" "$origin" "$pkgtag" "$note" "$resolved"
		fi
	done < $tmp_valid
	:
}

# ============= Show a list of failed restoration of conflict =============
database_query_show_list_failed_conflicts_restoration ()
{
	if [ `cat "${DBDIR}/deleted_conflicts" 2> /dev/null | wc -l` -eq 0 ]
	then
		message_echo "INFO: No item is registered in this list."
		return 1
	fi
	env LANG=C grep -v -E -f "${DBDIR}/ports_to_delete.grep_pattern_col1" "${DBDIR}/deleted_conflicts" \
		| while read origin pkg
		do
			pkg_regexp=`str_escape_regexp "$pkg"`
			against=`env LANG=C grep -E "^$pkg_regexp:" "${DBDIR}/forbidden_conflicts" 2> /dev/null | cut -d : -f 2,3 | sort -u`
			if [ $opt_batch_mode = no ]
			then
				if [ -n "$pkg" ]
				then
					echo -n "$origin ($pkg)"
				else
					echo -n "$origin"
				fi
				if [ -n "$against" ]
				then
					echo -n " against "
					against=`echo "$against" | sed 's/:/(/;s/$/)/' | tr '\n' ' '`
					str_linearize_list_and "$against"
				else
					echo
				fi
			else
				against=`echo "$against" | tr '\n' '|' | sed 's/,$//'`
				printf '%s\t%s\t%s\n' "$origin" "$pkg" "$against"
			fi
		done
	:
}

# ============= Show installed ports which have been neither upgraded or reinstalled from the initial state =============
database_query_show_list_fossil ()
{
	local srclist origin pkg
	srclist=`ls "${DBDIR}"/fossil_pkgs/fossil_since_* | head -n 1`
	[ -e "$srclist" ] || return 0
	while read origin
	do
		pkg=`pkgsys_get_init_pkg_from_orig "$origin"`
		pkg_info_e "$pkg" || continue
		printf '%s\t%s\n' "$origin" "$pkg"
	done < $srclist
}

# ============= Insert initial flavored origins to a list of flavored origins =============
database_query_add_initial_origins ()
{
	local origin
	while read origin
	do
		echo "$origin"
		[ -e "${DBDIR}/moved_from/$origin/initial_orig" ] || continue
		cat "${DBDIR}/moved_from/$origin/initial_orig"
	done
	:
}

# ============= Get target attributes =============
database_query_get_target_attributes ()
{
	local prefix origin _is_all _is_target _is_requires_requirements _is_initial_requirements _is_requires_dependents _is_initial_dependents _is_requires_requirements_complement _is_relevant infofile tag level
	prefix=$1
	origin=$2
	_is_all=y
	_is_target=
	_is_requires_requirements=
	_is_initial_requirements=
	_is_requires_dependents=
	_is_initial_dependents=
	_is_requires_requirements_complement=
	_is_relevant=y
	if [ ! -e "${DBDIR}/target_all" -a -n "$opt_target_itself$opt_target_dependents$opt_target_requirements" ]
	then
		_is_all=
		tag=`options_get_dependency_type`
		level=full
		[ $opt_only_target_scope = yes ] && level=direct
		infofile=${DBDIR}/targets/$origin/attrs.${tag}.${level}
		if [ -e "$infofile" ]
		then
			. "$infofile"
		else
			_is_relevant=
		fi
	fi
	eval ${prefix}_is_all=\$\{_is_all\}
	eval ${prefix}_is_target=\$\{_is_target\}
	eval ${prefix}_is_requires_requirements=\$\{_is_requires_requirements\}
	eval ${prefix}_is_initial_requirements=\$\{_is_initial_requirements\}
	eval ${prefix}_is_requires_dependents=\$\{_is_requires_dependents\}
	eval ${prefix}_is_initial_dependents=\$\{_is_initial_dependents\}
	eval ${prefix}_is_requires_requirements_complement=\$\{_is_requires_requirements_complement\}
	eval ${prefix}_is_relevant=\$\{_is_relevant\}
}

# ============= Check whether (re/de)installation of a port is suppressed =============
database_query_is_a_port_suppressed ()
{
	local origin flags flag
	origin=$1
	flags=
	[ $opt_suppress_self_upadte = yes ] && flags=SUPPRESSED_SELF
	[ $opt_suppress_pkgtools_upadte = yes ] && flags="$flags SUPPRESSED_PKGNG"
	for flag in $flags
	do
		for db in initial moved_from
		do
			[ -e "${DBDIR}/$db/$origin/$flag" ] && return
		done
	done
	return 1
}

# ============= Check whether a port needs to be updated or upgraded =============
database_query_does_a_port_need_update ()
{
	local origin dbpath frompath new_version current_version
	origin=$1
	dbpath=${DBDIR}/requires/$origin
	frompath=${DBDIR}/moved_from/$origin
	[ -e "$dbpath/conf_updated" ] && return
	[ -e "$dbpath/new_version" ] || return
	new_version=`cat "$dbpath/new_version"`
	current_version=`cat "$frompath/current_version"`
	[ "x$current_version" != "x$new_version" ]
}

# ============= Check before operations of a command which need the temporary database completely prepared =============
database_query_chk_preparation_completion ()
{
	program_chk_stage_complete PREPARATION && return
	message_echo "ERROR: Database has to be built completely before executing this command." >&2
	exit 1
}

# ============= Get a make variable value of a port =============
database_query_get_makevar_val ()
{
	local origin variable dbdir cache value
	origin=$1
	variable=$2
	dbdir=${DBDIR}/requires/$origin
	cache=$dbdir/makevar/$variable
	if [ -e "$cache" ]
	then
		cat "$cache"
	else
		value=`database_build_make "$origin" -V "$variable"`
		if [ -d "$dbdir" ] && misc_is_superuser_privilege
		then
			[ -d "$dbdir/makevar" ] || mkdir "$dbdir/makevar"
			echo "$value" > $cache.tmp
			mv "$cache.tmp" "$cache"
		fi
		echo "$value"
	fi
}

# ============= Get a configured value of a port =============
database_query_get_config_val ()
{
	local origin variable dbdir cache value
	origin=$1
	variable=$2
	dbfile=${DBDIR}/conf/each_port/$origin/$variable.conf
	cat "$dbfile" 2> /dev/null || :
}

# ============= Check whether configurations for a port is default =============
database_query_is_default_conf ()
{
	local origin mode dbpath tmp_msg is_customized is_requiremnt_replaced files tmp_old tmp_new origin_requirement tmp_msg_customized
	origin=$1
	mode=$2
	dbpath=${DBDIR}/requires/$origin
	if [ ! -e "$dbpath/is_customized" ]
	then
		tmp_msg=${TMPDIR}/database_query_is_default_conf:msg
		cp /dev/null "$tmp_msg"
		is_customized=no
		if [ `ls "${DBDIR}/conf/each_port/$origin" 2> /dev/null | wc -l` -gt 0 ]
		then
			files=`ls "${DBDIR}/conf/each_port/$origin" | sed -E 's/^([^.]+).*/\1/' | str_concat_items_for_sentence`
			echo "Knobs and miscellaneous customization by $files," >> $tmp_msg
			is_customized=yes
		fi
		if ! diff -q "$dbpath/ports_options.default" "$dbpath/ports_options.current" > /dev/null 2>&1
		then
			echo "Non-default port options," >> $tmp_msg
			is_customized=yes
		fi
		if env LANG=C grep -q -Fx "$origin" "${DBDIR}/conf/NOPKG:PORTS.parsed" 2> /dev/null
		then
			echo "Explicit specification as non-default in ${APPNAME}.conf," >> $tmp_msg
			is_customized=yes
		fi
		tmp_old=${TMPDIR}/database_query_is_default_conf:old
		tmp_new=${TMPDIR}/database_query_is_default_conf:new
		is_requiremnt_replaced=no
		if fileedit_manipulate_old_new_lines "$dbpath/requirements.all.direct.orig" "$dbpath/requirements.all.direct" \
			"$tmp_old" "$tmp_new"
		then
			echo "Replacement in requirements:" >> $tmp_msg
			echo "-------- FROM --------" >> $tmp_msg
			sed 's/^/ /' "$tmp_old" >> $tmp_msg
			echo "--------  TO  --------" >> $tmp_msg
			sed 's/^/ /' "$tmp_new" >> $tmp_msg
			echo "----------------------" >> $tmp_msg
			is_customized=yes
		fi
		for origin_requirement in `cat "$dbpath/requirements.all.direct" 2> /dev/null`
		do
			database_query_is_default_conf "$origin_requirement" quiet && continue
			echo "Non-default requirement $origin_requirement," >> $tmp_msg
			is_customized=yes
		done
		{ [ $is_customized = yes ] && cat "$tmp_msg"; } > $dbpath/is_customized.tmp
		mv "$dbpath/is_customized.tmp" "$dbpath/is_customized"
	fi
	[ `wc -c < $dbpath/is_customized` -eq 0 ] && return
	if [ "x$mode" != xquiet ]
	then
		tmp_msg_customized=${TMPDIR}/database_query_is_default_conf:msg_customized
		message_echo "INFO: This port is configured to be non-default because of"
		sed 's/^/         /' "$dbpath/is_customized" > $tmp_msg_customized
		message_cat "$tmp_msg_customized"
		message_echo "      so the prebuilt package is not used."
	fi
	return 1
}

# ============= Output of lists in which each matching port is registered =============
database_query_show_list_inclusion_of_matching_port ()
{
	local grandtitle lists pkgnamedb deptag level isfirst origin_target pkg_target table_target
	grandtitle=$1
	lists=$2
	pkgnamedb=$3
	deptag=$4
	level=$5
	shift 5
	message_echo "[$grandtitle]"
	message_dependency_scope "$deptag" "$level"
	message_echo
	isfirst=y
	for origin_target in `pkgsys_eval_ports_glob "$@"`
	do
		pkg_target=
		for table_target in $pkgnamedb
		do
			pkg_target=`cat "${DBDIR}/$table_target/$origin_target/pkgtag" 2> /dev/null` || :
			[ -n "$pkg_target" ] && break
		done
		[ -n "$pkg_target" ] || continue
		isfirst=n
		match=
		for subject in `echo "$lists" | tr \| ' '`
		do
			database_query_exists_in_list "$origin_target" "$subject" "$deptag" "$level" \
				&& match="$match $subject"
		done
		if [ $opt_batch_mode = no ]
		then
			echo -n "$origin_target ($pkg_target): "
			echo "$match" | sed 's/^ *//;s/ /, /g'
		else
			printf '%s\t%s\t' "$origin_target" "$pkg_target"
			echo "$match" | sed 's/^ *//;s/ /,/g'
		fi
	done
	if [ "$isfirst" = y ]
	then
		message_echo "ERROR: No inspected port matches the glob(s)." >&2
		exit 1
	fi
	:
}

# ============= Output of "show" command for port lists =============
database_query_show_port_lists ()
{
	local grandtitle title list listdb pkgnamedb deptag level isfirst origin_target pkg_target table_target list_target
	grandtitle=$1
	title=$2
	list=$3
	listdb=$4
	pkgnamedb=$5
	deptag=$6
	level=$7
	shift 7
	message_echo "[$grandtitle]"
	message_dependency_scope "$deptag" "$level"
	message_echo
	isfirst=y
	for origin_target in `pkgsys_eval_ports_glob "$@"`
	do
		pkg_target=
		for table_target in $pkgnamedb
		do
			pkg_target=`cat "${DBDIR}/$table_target/$origin_target/pkgtag" 2> /dev/null` || :
			[ -n "$pkg_target" ] && break
		done
		[ -n "$pkg_target" ] || continue
		[ "$isfirst" = y ] || message_echo
		isfirst=n
		[ $opt_batch_mode = no ] && printf "$title\n" "$origin_target ($pkg_target)"
		list_target=
		for table_target in $listdb
		do
			list_target=${DBDIR}/$table_target/$origin_target/$list
			[ -e "$list_target" ] && break
		done
		[ -e "$list_target" ] || continue
		if [ $opt_batch_mode = no ]
		then
			while read origin
			do
				pkg=
				for table in $pkgnamedb
				do
					pkg=`cat "${DBDIR}/$table/$origin/pkgtag" 2> /dev/null` || :
					[ -n "$pkg" ] && break
				done
				[ -n "$pkg" ] || continue
				echo "$origin ($pkg)"
			done < $list_target
		else
			while read origin
			do
				pkg=
				for table in $pkgnamedb
				do
					pkg=`cat "${DBDIR}/$table/$origin/pkgtag" 2> /dev/null` || :
					[ -n "$pkg" ] && break
				done
				[ -n "$pkg" ] || continue
				printf '%s\t%s\t%s\t%s\n' "$origin_target" "$pkg_target" "$origin" "$pkg"
			done < $list_target
		fi
	done
	if [ "$isfirst" = y ]
	then
		message_echo "ERROR: No inspected port matches the glob(s)." >&2
		exit 1
	fi
	:
}

# ============= Output of "show" command for log files =============
database_query_show_log ()
{
	local grandtitle title list listdb pkgnamedb isfirst origin_target pkg_target table_target list_target
	grandtitle=$1
	title=$2
	list=$3
	listdb=$4
	pkgnamedb=$5
	shift 5
	message_echo "[$grandtitle]"
	message_echo
	isfirst=y
	for origin_target in `pkgsys_eval_ports_glob "$@"`
	do
		pkg_target=
		for table_target in $pkgnamedb
		do
			pkg_target=`cat "${DBDIR}/$table_target/$origin_target/pkgtag" 2> /dev/null` || :
			[ -n "$pkg_target" ] && break
		done
		[ -n "$pkg_target" ] || continue
		[ "$isfirst" = y ] || message_echo
		isfirst=n
		[ $opt_batch_mode = no ] && printf "$title\n" "$origin_target ($pkg_target)"
		list_target=
		for table_target in $listdb
		do
			list_target=${DBDIR}/$table_target/$origin_target/$list
			[ -e "$list_target" ] && break
		done
		[ -e "$list_target" ] || continue
		cat  < $list_target
		echo
	done
	if [ "$isfirst" = y ]
	then
		message_echo "ERROR: No inspected port matches the glob(s)." >&2
		exit 1
	fi
	:
}

# ============= Output of "show" command for two column lists =============
database_query_show_two_column_lists ()
{
	local grandtitle title list listdb pkgnamedb isfirst origin_target pkg_target table_target list_target
	grandtitle=$1
	title=$2
	list=$3
	listdb=$4
	pkgnamedb=$5
	shift 5
	message_echo "[$grandtitle]"
	message_echo
	isfirst=y
	for origin_target in `pkgsys_eval_ports_glob "$@"`
	do
		pkg_target=
		for table_target in $pkgnamedb
		do
			pkg_target=`cat "${DBDIR}/$table_target/$origin_target/pkgtag" 2> /dev/null` || :
			[ -n "$pkg_target" ] && break
		done
		[ -n "$pkg_target" ] || continue
		[ "$isfirst" = y ] || message_echo
		isfirst=n
		[ $opt_batch_mode = no ] && printf "$title\n" "$origin_target ($pkg_target)"
		list_target=
		for table_target in $listdb
		do
			list_target=${DBDIR}/$table_target/$origin_target/$list
			[ -e "$list_target" ] && break
		done
		[ -e "$list_target" ] || continue
		if [ $opt_batch_mode = no ]
		then
			sed 's/[[:space:]]/: /' < $list_target
		else
			cat  < $list_target
		fi
	done
	if [ "$isfirst" = y ]
	then
		message_echo "ERROR: No inspected port matches the glob(s)." >&2
		exit 1
	fi
	:
}

# ============= Output of "show" command for a single list =============
database_query_show_single_list ()
{
	local list pkgnamedb flag_filter_skip_unchanged flag_filter_only_target flag_negative_listdb tmpflag_exists put_blankline origin matches_negative matches_positive flag pkg table
	local currentorigin_is_alll currentorigin_is_target currentorigin_is_requires_requirements
	local currentorigin_is_initial_requirements currentorigin_is_requires_dependents
	local currentorigin_is_initial_dependents currentorigin_is_requires_requirements_complement currentorigin_is_relevant
	list=$1
	pkgnamedb=$2
	flag_filter_skip_unchanged=$3
	flag_filter_only_target=$4
	flag_negative_listdb=$5
	tmpflag_exists=${TMPDIR}/database_query_show_single_list::exists_item
	if [ `cat "${DBDIR}/$list" 2> /dev/null | wc -l` -eq 0 ]
	then
		message_echo "INFO: No item is registered in this list."
		return 1
	fi
	if ! program_chk_stage_complete PREPARATION
	then
		message_echo "WARNING: The temporary database is incomplete. The raw list is shown." >&2
		cat "${DBDIR}/$list"
		return
	fi
	rm -f "$tmpflag_exists"
	put_blankline=
	if [ -n "$flag_filter_only_target" \
		-a -n "$opt_target_itself$opt_target_dependents$opt_target_requirements" ]
	then
		message_echo "WARNING: Ports outside of the target scope are excluded." >&2
		put_blankline=y
	fi
	if [ -n "$flag_filter_skip_unchanged" -a $opt_skip_unchanged = yes ]
	then
		message_echo "WARNING: Ports which have been the newest with their all requirements from the first are excluded." >&2
		put_blankline=y
	fi
	[ -n "$put_blankline" ] && message_echo
	while read origin
	do
		matches_negative=no
		for negative_listdb in $flag_negative_listdb
		do
			if env LANG=C grep -qFx "$origin" "${DBDIR}/$negative_listdb" 2> /dev/null
			then
				matches_negative=yes
				break
			fi
		done
		[ $matches_negative = yes ] && continue
		if [ -n "$flag_filter_skip_unchanged" -a $opt_skip_unchanged = yes ]
		then
			matches_positive=no
			for flag in $flag_filter_skip_unchanged
			do
				if [ -e "${DBDIR}/requires/$origin/$flag" ]
				then
					matches_positive=yes
					break
				fi
			done
			[ $matches_positive = no ] && continue
		fi
		if [ -n "$flag_filter_only_target" ]
		then
			database_query_get_target_attributes currentorigin "$origin"
			[ -n "${currentorigin_is_relevant}" ] || continue
			database_query_is_a_port_suppressed "$origin" && continue
		fi
		pkg=
		for table in $pkgnamedb
		do
			pkg=`cat "${DBDIR}/$table/$origin/pkgtag" 2> /dev/null` || :
			[ -n "$pkg" ] && break
		done
		[ -n "$pkg" ] || continue
		touch "$tmpflag_exists"
		if [ $opt_batch_mode = no ]
		then
			echo "$origin ($pkg)"
		else
			printf '%s\t%s\n' "$origin" "$pkg"
		fi
	done < ${DBDIR}/$list
	[ -e "$tmpflag_exists" ] && return
	message_echo "INFO: No item is registered in this list."
	return 1
}

# ============= Check whether the upgrade is necessary for a port =============
database_query_is_necessary_upgrade ()
{
	local origin nodedir dbsuffix tmpfile_new tmpfile_old tmpfile_diff
	origin=$1
	nodedir=${DBDIR}/requires/$origin
	[ -e "$nodedir/installed_by_pkg" ] && return 1
	dbsuffix=`options_get_dependency_type`.`options_get_dependency_level`
	tmpfile_new=${TMPDIR}/database_query_is_necessary_upgrade:failed_requirements.new
	tmpfile_old=${TMPDIR}/database_query_is_necessary_upgrade:failed_requirements.old
	tmpfile_diff=${TMPDIR}/database_query_is_necessary_upgrade:failed_requirements.diff
	[ -e "$nodedir/failed_requirements.${dbsuffix}.previous" ] || return
	sort -u "$nodedir/failed_requirements.${dbsuffix}" > $tmpfile_new 2> /dev/null || :
	sort -u "$nodedir/failed_requirements.${dbsuffix}.previous" > $tmpfile_old 2> /dev/null || :
	fileedit_exists_old_lines "$tmpfile_old" "$tmpfile_new"
}

# ============= Show moved or replaced ports to alternatives =============
database_query_show_list_moved ()
{
	cat "${DBDIR}/moved_ports" 2> /dev/null | while read origin
	do
		lastorigin=$origin
		while [ -n "$lastorigin" -a -e "${DBDIR}/replace/$lastorigin/origin" ]
		do
			lastorigin=`cat "${DBDIR}/replace/$lastorigin/origin"`
		done
		[ -n "$lastorigin" ] || continue
		pkg=
		for table in $pkgnamedb
		do
			pkg=`cat "${DBDIR}/$table/$origin/pkgtag" 2> /dev/null` || :
			[ -n "$pkg" ] && break
		done
		[ -n "$pkg" ] || pkg='not installed'
		lastpkg=
		for table in moved_from obsolete initial
		do
			lastpkg=`cat "${DBDIR}/$table/$lastorigin/pkgtag" 2> /dev/null` || :
			[ -n "$lastpkg" ] && break
		done
		[ -n "$lastpkg" ] || lastpkg='not installed'
		if [ $opt_batch_mode = no ]
		then
			echo "$origin ($pkg) => $lastorigin ($lastpkg)"
		else
			printf '%s\t%s\t%s\t%s\n' "$origin" "$pkg" "$lastorigin" "$lastpkg"
		fi
	done
}

# ============= Actual operations of "show" command for a single list =============
database_query_show_single_list_exec ()
{
	local subject deptag level dbsuffix flag_filter_skip_unchanged flag_filter_only_target flag_negative_listdb pkgnamedb
	subject=$1
	deptag=$2
	level=$3
	dbsuffix=$deptag.$level
	flag_filter_skip_unchanged=
	flag_filter_only_target=
	flag_negative_listdb=
	pkgnamedb='moved_from obsolete initial'
	case $subject in
	todo )
		message_echo "The following ports remain in the (re)installation queue for the current do/redo process:"
		message_echo "It is noted that ports to be skipped can be included here."
		message_dependency_scope "$deptag" "$level"
		message_echo
		list=stage.loop_list/reinst_todo.remain
		[ ${DBDIR}/reinst_order.list -nt ${DBDIR}/$list ] && list=reinst_order.list
		flag_filter_skip_unchanged=necessary_upgrade.$dbsuffix
		flag_filter_only_target=y
		;;
	done )
		message_echo "The following ports have been successfully (re)installed or newly installed:"
		message_dependency_scope "$deptag" "$level"
		message_echo
		list=success.$dbsuffix.list
		flag_filter_skip_unchanged=necessary_upgrade_completed.$dbsuffix
		flag_filter_only_target=y
		;;
	redo )
		message_echo "The following ports need (re)installation after success of the all requirements:"
		message_dependency_scope "$deptag" "$level"
		message_echo
		list=todo_after_requirements_succeed.$dbsuffix.list
		flag_filter_skip_unchanged="necessary_upgrade.$dbsuffix necessary_upgrade_completed.$dbsuffix"
		flag_filter_only_target=y
		flag_negative_listdb=leaf_ports_to_delete.selected
		;;
	resolved )
		message_echo "The following ports had problems which have been manually resolved:"
		message_echo
		list=manually_done.list
		;;
	inst_by_pkg )
		message_echo "The following ports are configured default and installed by prebuilt packages"
		message_echo
		list=installation_complete_by_pkg.list
		;;
	inst_built_default )
		message_echo "The following ports are configured default and installed by ports"
		message_echo
		list=inst_by_port_with_default_conf.list
		;;
	inst_built_custom )
		message_echo "The following ports are configured non-default and installed by ports"
		message_echo
		list=inst_by_port_with_custom_conf.list
		;;
	failure )
		message_echo "The following ports experienced failures and kept to be old or uninstalled:"
		message_echo
		database_query_show_list_failure
		return
		;;
	fossil )
		message_echo "The following ports are neither upgraded or reinstalled from the initial state:"
		message_echo
		database_query_show_list_fossil
		return
		;;
	conflict )
		message_echo "The following ports are temporarily deleted due to conflict:"
		message_echo
		database_query_show_list_failed_conflicts_restoration
		return
		;;
	moved )
		message_echo "The following ports are moved/replaced to new alternatives:"
		message_echo
		database_query_show_list_moved
		return
		;;
	taboo )
		message_echo "The following ports are registered as taboo:"
		message_echo
		list=taboo.all.list
		;;
	freeze )
		message_echo "The following ports are registered to freeze:"
		message_echo
		list=freeze.all.list
		;;
	need )
		message_echo "The following ports are registered as necessary:"
		message_echo
		list=need.list
		;;
	noneed )
		message_echo "The following ports are registered as unnecessary:"
		message_echo
		list=noneed.list
		;;
	restored )
		message_echo "The following leaf, obsolete or unneeded ports had been once deleted but are to be or have been restored:"
		message_echo
		list=stage.loop_list/ports_to_restore
		pkgnamedb='moved_from obsolete initial'
		flag_filter_only_target=y
		;;
	deleted )
		message_echo "The following leaf, obsolete or unneeded ports are to be or have been deleted:"
		message_echo
		list=stage.loop_list/ports_to_delete
		pkgnamedb='moved_from obsolete initial'
		flag_filter_only_target=y
		;;
	leaves )
		if [ -z "$deptag" ]
		then
			message_echo "The following ports are all detected leaf ports:"
			list=leaf_ports
		else
			message_echo "The following ports are all $deptag leaf ports:"
			list=leaf_ports_to_delete.$deptag
		fi
		message_echo
		pkgnamedb='moved_from obsolete initial'
		flag_filter_only_target=y
		;;
	obsolete )
		if [ -z "$deptag" ]
		then
			message_echo "The following ports are all detected obsolete ports:"
			list=obsolete_ports.can_be_deleted
		else
			message_echo "The following ports are all $deptag obsolete ports:"
			list=obsolete_ports_to_delete.$deptag
		fi
		message_echo
		pkgnamedb='moved_from obsolete initial'
		flag_filter_only_target=y
		;;
	esac
	database_query_show_single_list "$list" "$pkgnamedb" \
		"$flag_filter_skip_unchanged" "$flag_filter_only_target" "$flag_negative_listdb"
}

# ============= Check whether a port is registered in a list =============
database_query_exists_in_list ()
{
	local origin subject deptag level tmp_list dbsuffix origin_esc
	origin=$1
	subject=$2
	deptag=$3
	level=$4
	tmp_list=${TMPDIR}/database_query_exists_in_list:list
	dbsuffix=$deptag.$level
	origin_esc=`str_escape_regexp "$origin"`
	case $subject in
	todo )
		list=stage.loop_list/reinst_todo.remain
		[ ${DBDIR}/reinst_order.list -nt ${DBDIR}/$list ] && list=reinst_order.list
		;;
	done )
		list=success.$dbsuffix.list
		;;
	redo )
		list=todo_after_requirements_succeed.$dbsuffix.list
		;;
	resolved )
		list=manually_done.list
		;;
	inst_by_pkg )
		list=installation_complete_by_pkg.list
		;;
	inst_built_default )
		list=inst_by_port_with_default_conf.list
		;;
	inst_built_custom )
		list=inst_by_port_with_custom_conf.list
		;;
	failure )
		list=failed.list
		;;
	fossil )
		list=`ls "${DBDIR}"/fossil_pkgs/fossil_since_* | head -n 1`
		;;
	conflict )
		env LANG=C grep -v -E -f "${DBDIR}/ports_to_delete.grep_pattern_col1" "${DBDIR}/deleted_conflicts" 2> /dev/null \
			| env LANG=C grep -q -E "^${origin_esc}[[:space:]]"
		return
		;;
	taboo )
		list=taboo.all.list
		;;
	freeze )
		list=freeze.all.list
		;;
	need )
		list=need.list
		;;
	noneed )
		list=noneed.list
		;;
	restored )
		list=stage.loop_list/ports_to_restore
		;;
	deleted )
		list=stage.loop_list/ports_to_delete
		;;
	esac
	env LANG=C grep -q -Fx "$origin" "${DBDIR}/$list" 2> /dev/null
}

# ============= Check whether the requirements of installed packages match the port configuration =============
database_query_dependency_matching ()
{
	local origin pkg tmp_inst tmp_db
	origin=$1
	[ -d "${DBDIR}/requires/$origin" ] || return
	tmp_inst=${TMPDIR}/database_query_dependency_matching.installed
	tmp_db=${TMPDIR}/database_query_dependency_matching.configured
	pkg=`database_build_get_new_pkgname "$origin"`
	[ -n "$pkg" ] || return
	pkg_info_e "$pkg" || return
	pkg_info_qr "$pkg" | env LANG=C grep -v '^[[:space:]]*$' | sort -u > $tmp_inst
	database_build_get_full_run_requirement_pkgs "$origin" > $tmp_db
	diff "$tmp_inst" "$tmp_db" > /dev/null 2>/dev/null
}

# ============= Check whether any of the requirements are locked because being missing ports to freeze =============
database_query_are_requirements_not_locked ()
{
	local origin
	origin=$1
	cat "${DBDIR}/requires/$origin/requirements.all.direct" 2> /dev/null | while read origin_requirement
	do
		pkg_requirement=`pkgsys_get_installed_pkg_from_origin "$origin_requirement"` || :
		if env LANG=C grep -q -Fx "$origin_requirement" "${DBDIR}/freeze.all.list" 2> /dev/null
		then
			[ -n "$pkg_requirement" ] || return
		elif [ -z "$pkg_requirement" ]
		then
			database_query_are_requirements_not_locked "$origin_requirement" || return
		fi
	done
	:
}

# ============= Check whether any of progress is made in the current run =============
database_query_is_any_progress ()
{
	[ `cat "${DBDIR}/new_success_in_current_run" 2> /dev/null | wc -l` -gt 0 ]
}

# ============= Get the all initial origins, separated by line feed =============
database_query_initial_orgins ()
{
	local origin origin_init
	origin=$1
	if [ -e "${DBDIR}/moved_from/$origin/old_origs" ]
	then
		for origin_init in `cat "${DBDIR}/moved_from/$origin/old_origs"`
		do
			[ -e "${DBDIR}/initial/$origin_init/installed_version" ] && echo "$origin_init"
		done
	else
		[ -e "${DBDIR}/initial/$origin/installed_version" ] && echo "$origin"
	fi
	:
}

# ============= Get an unflavored or flavored equivalence of an origin with the same package name, which is deemed the same version =============
# If the input is unflavored and a flavored equivalence is inspected as a required port, the flavored origin is returned.
# If the input is flavored and an unflavored equivalence is inspected as a required port, the unflavored origin is returned.
# Otherwise, a blank value is returned.
database_query_get_equivalent_orgin ()
{
	local origin origin_esc pkg_origin pkg_origin_esc flavor origin_unflavored_esc origin_search
	origin=$1
	origin_esc=`str_escape_regexp "$origin"`
	pkg_origin=`env LANG=C grep -E "[[:space:]]$origin_esc$" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" | cut -f 2 || :`
	[ -n "$pkg_origin" ] || return 0
	pkg_origin_esc=`str_escape_regexp "$pkg_origin"`
	flavor=`pkgsys_get_flavor_from_origin "$origin"`
	origin_search=$origin_unflavored_esc$
	[ -n "$flavor" ] || origin_search=$origin_esc@
	env LANG=C grep -E "^$pkg_origin_esc[[:space:]]$origin_search" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" | cut -f 2 | head -n 1 || :
}
