#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - File system operations -
# Copyright (C) 2018-2022 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Get the global path of a possibly not yet created file/directory =============
fs_global_path ()
{
	local $path_src
	path_src=$1
	if [ -e "$path_src" ]
	then
		realpath "$path_src"
	else
		expr "$path_src" : '^/' > /dev/null || echo -n `realpath .`
		echo "$path_src"
	fi
}

# ============= Inspect the privilege of the current environment on file system operation =============
fs_inspect_fs_privilege ()
{
	local tgv mp mp_regexp mount_privilege basedir
	[ -d "${TMPDIR}"/fs_privilege ] && return
	tgv=${TMPDIR}/fs_privilege/test_tg
	mkdir -p "$tgv" "${TMPDIR}"/fs_privilege/test_mp
	mp=`realpath "${TMPDIR}"/fs_privilege/test_mp`
	mp_regexp=`str_escape_regexp "$mp"`
	echo yes > ${TMPDIR}/fs_privilege/fs_privilege
	cat > ${TMPDIR}/fs_privilege/fslist << eof
devfs	devfs
fdescfs	null
procfs	proc
tmpfs	tmpfs
nullfs	"$tgv"
unionfs	"$tgv"
eof
	while read fs tg
	do
		if mount -t $fs "$tgv" "$mp" 2> /dev/null && umount "$mp"
		then
			echo yes
		else
			echo no
			echo no > ${TMPDIR}/fs_privilege/fs_privilege
		fi > ${TMPDIR}/fs_privilege/fs_privilege:$fs 2> /dev/null
		umount -f "$mp" 2> /dev/null || :
		[ -e "${TMPDIR}"/fs_privilege/basedir ] && continue
		mount -t $fs "$tgv" "$mp" 2> /dev/null && \
			df "$mp" > ${TMPDIR}/fs_privilege/df:$fs && \
			umount "$mp"
		real_mp=`sed 1d "${TMPDIR}"/fs_privilege/df:$fs | tail -n 1 | \
			sed -E 's/^.*[[:space:]][0-9]+%[[:space:]]+//'`
		echo "$real_mp" | sed -E "s/$mp_regexp$//" > ${TMPDIR}/fs_privilege/basedir
	done < ${TMPDIR}/fs_privilege/fslist
	mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege`
	if [ "x$mount_privilege" = xyes ]
	then
		mount -t nullfs /bin "$mp" 2> /dev/null
		if [ `ls "$mp" 2> /dev/null | wc -l` -gt 0 ]
		then
			echo yes
		else
			echo no
		fi > ${TMPDIR}/fs_privilege/nullfs_target_recognition
		umount -f "$mp" 2> /dev/null || :
		nullfs_target_recognition=`cat "${TMPDIR}"/fs_privilege/nullfs_target_recognition`
		if [ "x$nullfs_target_recognition" = xyes ]
		then
			message_echo "INFO: The current environment has the full required privilege of mounting/unmounting file systems."
		else
			message_echo "INFO: The current environment formally has the full required privilege of mounting/unmounting file systems but the recognition of nullfs/unionfs targets is incorrect."
		fi
	else
		echo no > ${TMPDIR}/fs_privilege/nullfs_target_recognition
		message_echo "INFO: The current environment does not have the privilege of mounting/unmounting for the following file system(s)."
		while read fs tg
		do
			 mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege:$fs`
			 [ "x$mount_privilege" = xyes ] || echo '  '$fs
		done < ${TMPDIR}/fs_privilege/fslist | message_cat
	fi
	basedir=`cat "${TMPDIR}"/fs_privilege/basedir 2> /dev/null || :`
	if [ -n "$basedir" ]
	then
		message_echo "INFO: The current environment will be a chroot/jail guest whose base directory is \"$basedir\"."
	fi
	if [ "x$opt_invalidate_mount_privilege" = xyes ]
	then
		message_echo "INFO: The privilege of mounting/unmounting in this environment is forcibly invalidated."
	fi
}

# ============= Check whether mounting file systems are available at the current environment =============
fs_chk_mount_privilege ()
{
	local mount_privilege
	fs_inspect_fs_privilege
	[ "x$opt_invalidate_mount_privilege" = xno ] || return
	mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege`
	nullfs_target_recognition=`cat "${TMPDIR}"/fs_privilege/nullfs_target_recognition`
	[ "x$nullfs_target_recognition" = xyes -a "x$mount_privilege" = xyes ]
}

# ============= Check whether mounting file systems are available at the current environment =============
fs_chk_unmount_privilege ()
{
	local mount_privilege
	fs_inspect_fs_privilege
	[ "x$opt_invalidate_mount_privilege" = xno ] || return
	mount_privilege=`cat "${TMPDIR}"/fs_privilege/fs_privilege`
	[ "x$mount_privilege" = xyes ]
}

# ============= Get the base directory the current environment (applicable in case of a chroot guest) =============
fs_get_system_basedir ()
{
	fs_inspect_fs_privilege >&2
	cat "${TMPDIR}"/fs_privilege/basedir 2> /dev/null || :
}

# ============= Get the regular expression pattern of the actual mount point =============
fs_get_actual_mount_point_pattern ()
{
	local mountpoint basedir mountpoint_real
	mountpoint=$1
	basedir=`fs_get_system_basedir`
	if [ -e  "$mountpoint" ]
	then
		mountpoint_real=`realpath "$mountpoint"`
	else
		curdir=$mountpoint
		curnode=$mountpoint
		relative=
		while { curbase=`basename "$curdir"`; curdir=`dirname "$curdir"`; [ -n "$curdir" ]; }
		do
			relative=$curbase/$relative
			if [ -e  "$curdir" ]
			then
				mountpoint_real=`realpath "$curdir"`/$relative
				break
			fi
		done
	fi
	mountpoint_real_full=`str_regularize_df_path "$basedir/$mountpoint_real"`
	str_escape_regexp "$mountpoint_real_full"
}

# ============= Get mount info at the descendant directory levels required for builder chroot environment =============
fs_get_descendant_mount_info ()
{
	local mountpoint mountpoint_real_regexp basedir basedir_ptn zpools_ptn ptns
	mountpoint=$1
	mountpoint_real_regexp=`fs_get_actual_mount_point_pattern "$mountpoint"` || return
	basedir=`fs_get_system_basedir`
	basedir_ptn=`str_escape_regexp "$basedir"`
	zpools_ptn=`zpool list -H -o name | str_escape_regexp_filter | tr \\n '|' | sed 's/|$//'`
	ptns='/|<above>:'
	[ -z "$zpools_ptn" ] || ptns="$ptns|$zpools_ptn"
	df | sed 1d | env LANG=C grep -E "%[[:space:]]+$mountpoint_real_regexp\/" | while read fs data
	do
		echo "$fs" | env LANG=C grep -Eq "^$ptns" && fs=normal
		mp_abs=`echo "$data" | sed -E  's|.*%[[:space:]]+(/.+)$|\1|'`
		mp=`echo "$mp_abs" | sed -E "s|^$basedir_ptn||"`
		relative=`echo "$mp_abs" | sed -E "s|^$mountpoint_real_regexp||"`
		printf '%s\t%s\t%s\n' "$fs" "$mp" "$relative"
	done
}

# ============= Repair the unionfs image if applicable and hidden =============
# Use the side effect of find(1) as a medicine for disappearance of lower layer images of unionfs.
# This treatment, however, can lose its effect when the symptom gets serious.
# When the effect is lost, "umount -A" after "shutdown now" may be the only practical solution.
fs_fix_unionfs_image_if_hidden ()
{
	local needlepath needlepath_cur needlepath_next needlepath_lowest_exist
	needlepath=$1
	needlepath_cur=$needlepath
	needlepath_lowest_exist=
	while :
	do
		[ -z "$needlepath_lowest_exist" -a -e "$needlepath_cur" ] && needlepath_lowest_exist=$needlepath_cur
		find -dx "$needlepath_cur" -maxdepth 0 > /dev/null 2>&1 || :
		needlepath_next=`dirname "$needlepath_cur"`
		[ "x$needlepath_cur" = "x$needlepath_next" ] && break
		needlepath_cur=$needlepath_next
	done
	[ -e "$needlepath" ] && return
	if [ -n "$needlepath_lowest_exist" ] && df "$needlepath_lowest_exist" | sed 1d | env LANG=C grep -q '^<above>:'
	then
		message_echo "ERROR: Failed to recover a lost mandatory file, probably due to the bug of unionfs: $needlepath" >&2
		mkdir -p "${DBDIR}/execflag"
		touch "${DBDIR}/execflag/unionfs_error"
	else
		message_echo "ERROR: Failed to recover a lost mandatory file due to unknown reasons: $needlepath" >&2
	fi
	exit 1
}

# ============= Check whether a directory is mounted properly =============
fs_chk_mounted ()
{
	local type target mountpoint target_ptn mountpoint_real_regexp tmpsrc
	type=$1
	target=`str_regularize_df_path "$2"`
	mountpoint=`str_regularize_df_path "$3"`
	target_ptn=`str_escape_regexp "$target"`
	mountpoint_real_regexp=`str_escape_regexp "$mountpoint"`
	tmpsrc=`df | sed 1d | env LANG=C grep -E "%[[:space:]]+$mountpoint_real_regexp$"`
	case $type in
		nullfs )
			echo "$tmpsrc" | env LANG=C grep -qE "^${target_ptn}[[:space:]]" && return
			;;
		unionfs )
			echo "$tmpsrc" | env LANG=C grep -qE "^<above>:${target_ptn}[[:space:]]" && return
			;;
		devfs | fdescfs | procfs | linprocfs | tmpfs )
			echo "$tmpsrc" | env LANG=C grep -q "^$type" && return
			;;
		*)
			message_echo "ERROR: Unsupported file system [$type]" >&2
			exit 1
			;;
	esac
	return 1
}

