#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Common functions of main programs -
# Copyright (C) 2018-2022 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Define the software version =============
main_set_version ()
{
	MYVERSION=4.1.1
	COMPATIBLE_VERSIONS='^(4\.[1]\.[0-9])$'
	# Template for development versions
# 	MYVERSION=4.1.0+toward_4.1.1_20220418123142
# 	COMPATIBLE_VERSIONS='^(4\.[0-1]\.[0-9]]|4\.[0-1]\.[0]+(|\+toward_4\.[0-1]\.[0-9]+_[0-9]+))$'
}

# ============= Parse options, arguments and control parameters =============
# All arguments/options of the main program must be passed.
main_parse_options_arguments ()
{
	# ============= Save arguments for upgraded restart =============
	options_dump_args "$@" > ${TMPDIR}/restart_command.sh

	# ============= Option check =============
	options_set_default

	options_getopts "$@" || :
	if [ $OPTIONS_ERRNO -eq 2 ]
	then
		message_echo "INTERNAL ERROR: In parsing options" >&2
		exit 1
	fi
	shift "${OPTIONS_SHIFT}"
	
	options_regularize
	
	# ============= Argument check for no-command options =============
	if [ $opt_help_mode -ne 0 -o $opt_show_version = yes ]
	then
		if [ $# -gt 0 ]
		then
			message_echo "SYNTAX ERROR: No command is allowed for showing Help or Version" >&2
			OPTIONS_ERRNO=1
		fi
	fi
	
	# ============= Output usage if the case of a help mode or option/argument errors =============
	if [ $OPTIONS_ERRNO -ne 0 ]
	then
		exit $OPTIONS_ERRNO
	elif [ $opt_help_mode -eq 1 ]
	then
		usage_short
		exit
	elif [ $opt_help_mode -eq 2 ]
	then
		usage_long | less -r
		exit
	fi
	
	# ============= Output version number =============
	if [ $opt_show_version = yes ]
	then
		message_version
		exit
	fi

	# ============= Set up variables for environment of ports and packages =============
	conf_setup_ports_envs
	conf_setup_packages_envs

	# ============= Execute command operations before getting the temporary database ready =============
	command_all_exec_before_db_creation "$@"

	# ============= Creation of temporary database directory =============
	database_maintain_create

	# ============= Argument check for conventional runs =============
	command_all_parse_args "$@"
}

# ============= Define the common termination messages =============
main_define_common_termination_messages ()
{
	temp_terminate_process_common ()
	{
		local errno msg_where
		errno=${1:-0}
		[ $opt_batch_mode = yes -o $errno -eq 0 ] && return
		msg_where=`temp_get_msg_current_stage`
		[ -n "$msg_where" ] && msg_where=" during $msg_where"
		message_echo
		if [ $errno -eq 130 ]
		then
			message_echo "INFO: Terminated at `message_timestamp`$msg_where."
			message_echo
			[ $opt_no_opening_message = yes ] && return
			message_echo " You can restart this process from the terminated point by"
		else
			message_echo "INFO: Aborted at `message_timestamp`$msg_where."
			message_echo
			[ $opt_no_opening_message = yes ] && return
			message_echo " You may restart this process from the aborted point by"
		fi
		message_echo "executing without options or arguments as:"
		if [ -n "$COMMAND_RESTART" ]
		then
			message_echo "  ${APPNAME} $COMMAND_RESTART"
		else
			message_echo "  ${APPNAME}"
		fi
	}
}

# ============= Set termination messages for special commands =============
main_set_termination_messages_special ()
{
}

# ============= Option settings =============
main_option_settings ()
{
	local optcomb_err
	# Load, renew and save option values
	optcomb_err=0
	if [ \( "x$opt_reload_conf" = xyes -o "x$opt_reset_targets" = xyes \) -a "x$COMMAND_MODE" != xredo ]
	then
		message_echo "ERROR: Options -L and -N are available only in the initial run of redo command." >&2
		message_echo >&2
		optcomb_err=1
	fi
	if [ "x$opt_batch_ports_only" = xyes -a "x$opt_interactive_ports_only" = xyes ]
	then
		message_echo "ERROR: Options -A and -I conflict with each other." >&2
		message_echo >&2
		optcomb_err=1
	fi
	if [ -e "${DBDIR}/saved_options.sh" ]
	then
		options_chk_invalid_optvals_renewal non_renewable || optcomb_err=$?
		{
			options_renewed_optvals M renewable_anytime || optcomb_err=$?
			options_renewed_optvals N renewable_in_redo_on_target || optcomb_err=$?
			options_renewed_optvals L renewable_in_redo_on_conf || optcomb_err=$?
		} > ${TMPDIR}/renewed_optvals.sh
		[ $optcomb_err -eq 0 ] || exit $optcomb_err
		. "${DBDIR}/saved_options.sh"
		. "${TMPDIR}/renewed_optvals.sh"
	fi
	misc_is_superuser_privilege && misc_get_all_vardefs | options_filter saved > ${DBDIR}/saved_options.sh
	:
}

# ============= Save the previous configuration if exists =============
main_save_prev_conf ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=''
	_program_exec_and_record_completion__operation ()
	{
		rm -rf "${DBDIR}/conf.prev"
		[ -d "${DBDIR}/conf" ] && \
			cp -Rp "${DBDIR}/conf" "${DBDIR}/conf.prev"
		:
	}
	program_exec_and_record_completion SAVE_PREV_CONF
}

# ============= Load the saved configuration =============
main_load_conf ()
{
	. "${DBDIR}/conf/setenv.sh"
}

# ============= Get complete configuration variable definitions by importing pkgtools.conf(5) if available =============
main_get_complete_conf ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='SAVE_PREV_CONF'
	_program_exec_and_record_completion__operation ()
	{
		local need_msg
		need_msg=no
		rm -rf "${DBDIR}/conf"
		mkdir -p "${DBDIR}/conf"
		[ "x`options_get_effective_opt_load_pkgtoolsconf 2> /dev/null`" != xno -a $opt_batch_mode = no ] \
			&& need_msg=yes
		[ $need_msg = yes ] && \
			message_section_title "Parsing pkgtools.conf (by using installed portupgrade)"
		conf_get_complete_var_defs > ${DBDIR}/conf/complete_setup.sh
		[ $need_msg = yes ] &&  { message_echo "===> ok"; message_echo; }
		:
	}
	program_exec_and_record_completion GET_COMPLETE_CONF_VAR_DEF
}

# ============= Parse the configuration =============
main_parse_conf ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='GET_COMPLETE_CONF_VAR_DEF'
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Parsing the configuration"
		conf_manipulate_available_var_defs
		. "${DBDIR}/conf/manipulated_defs.sh"
		# ALT_MOVED_*
		conf_build_effective_MOVED
		# Environmental variables
		conf_setup_effective_env > ${DBDIR}/conf/setenv.sh
		. "${DBDIR}/conf/setenv.sh"
		# HOLD_*
		conf_parse_vars_for_each_port_glob HOLD
		# TABOO_*
		conf_parse_vars_for_each_port_glob TABOO
		fileedit_combine_lists "${DBDIR}/conf/TABOO:PORTS.parsed" "${DBDIR}/taboo.list" \
			> ${DBDIR}/taboo.all.list
		# FREEZE_*
		conf_parse_vars_for_each_port_glob FREEZE
		fileedit_combine_lists "${DBDIR}/conf/FREEZE:PORTS.parsed" "${DBDIR}/freeze.list" \
			> ${DBDIR}/freeze.all.list
		# REBUILD_*
		conf_parse_vars_for_each_port_glob NOPKG
		# REPLACE_*
		conf_build_replacement_patterns_from_REPLACE
		# CONFLICT_*
		conf_parse_vars_for_each_port_glob_with_bound_val CONFLICT TARGET DEF
		# BUILDCONFLICT_*
		conf_parse_vars_for_each_port_glob_with_bound_val BUILDCONFLICT TARGET DEF
		# INSTCONFLICT_*
		conf_parse_vars_for_each_port_glob_with_bound_val INSTCONFLICT TARGET DEF
		# MARG_*
		conf_parse_vars_for_each_port_glob_with_bound_val MARG TARGET DEF
		# MENV_*
		conf_parse_vars_for_each_port_glob_with_bound_val MENV TARGET DEF
		# BEFOREBUILD_*
		conf_parse_vars_for_each_port_glob_with_bound_val BEFOREBUILD TARGET COMMAND
		# BEFOREDEINSTALL_*
		conf_parse_vars_for_each_port_glob_with_bound_val BEFOREDEINSTALL TARGET COMMAND
		# AFTERINSTALL_*
		conf_parse_vars_for_each_port_glob_with_bound_val AFTERINSTALL TARGET COMMAND
		message_echo
	}
	program_exec_and_record_completion PARSE_CONF
}

# ============= Set up parameters based on options, arguments, environment =============
main_setup_parameters ()
{
	# ============= Termination messages during construction of the temporary database =============
	main_define_common_termination_messages
	temp_reset_termination_messages_common
	main_set_termination_messages_special
	
	# ============= Opening title =============
	
	message_credit
	command_all_chk_need_opening_notice && message_opening_notice
	message_echo
	
	# ============= Execute command operations which do not need package tools =============
	
	command_all_exec_without_pkgtools "$@"
	misc_is_superuser_privilege && database_maintain_mark_use
	# ============= Definition of environment dependent functions =============
	
	pkgsys_def_pkgtools
	
	# ============= Option settings =============
	
	# Execute command operations which are not affected by saved option settings
	command_all_exec_irrespective_of_saved_options "$@"
	
	# Load, renew and save option values
	main_option_settings
	
	# Show option values
	message_show_option_settings
	
	# Execute command operations which should be carried out just after completing the option settings
	command_all_exec_just_after_option_settings "$@"

	# ============= Command-specific pre-configuration =============
	
	# Execute command operations which must be done before the database construction
	command_all_exec_command_specific_preconfiguration "$@"
	
	# ============= Configurations =============
	
	# Save the previous configuration if exists
	main_save_prev_conf
	
	# Get complete configuration variable definitions by importing pkgtools.conf(5) if available
	main_get_complete_conf
	
	# Parse the configuration
	main_parse_conf
	
	# Load the saved configuration
	main_load_conf
	if [ $opt_just_save_options = yes ]
	then
		message_echo "(Save-options-only mode)"
		exit
	fi
}

# ============= Operation without packages management tools =============
main_operation_without_pkg_management_tools ()
{
	# Execute command operations which should be done without upgrade of tools
	command_all_exec_before_tools_upgrade "$@"

	# Check whether the temporary database is newer than the ports tree and refresh if so
	database_maintain_refresh_if_obsolete
}

# ============= Collect all installed packages =============
main_collect_all_installed_packages ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=''
	_program_exec_and_record_completion__operation ()
	{
		local fossil_path
		message_section_title "Collecting all installed packages"
		mkdir -p "${DBDIR}/fossil_pkgs"
		fossil_path=${DBDIR}/fossil_pkgs/fossil_since_`pkgsys_get_timestamp_portstree`
		if [ ! -e "$fossil_path" ]
		then
			pkg_info_all_flavored_origins > $fossil_path.tmp
			mv "$fossil_path.tmp" "$fossil_path"
		fi
		if [ ! -e "${DBDIR}/installed_ports" ]
		then
			cp "$fossil_path" "${DBDIR}/installed_ports.tmp"
			mv "${DBDIR}/installed_ports.tmp" "${DBDIR}/installed_ports"
		fi
		pkgsys_gen_init_pkg_origin_table
		message_echo
	}
	program_exec_and_record_completion COLLECT_ALL_INSTALLED_PACKAGES
}

# ============= Determine all target ports of tools which have to be up-to-date =============
main_determine_all_target_ports_of_tools ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS=
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Determining all target ports of tools which have to be up-to-date"
		{
			[ "$PKGSYS_USE_PKGNG" = yes ] && pkgsys_portsmgmt_pkg
			pkgsys_is_dialog4ports_used && pkgsys_portsmgmt_dialog4ports
			[ -n "$MYPORTORIGIN" ] && echo "$MYPORTORIGIN"
		} 2> /dev/null > ${DBDIR}/stage.loop_list/tools_to_inspect
		cp "${DBDIR}/stage.loop_list/tools_to_inspect" "${DBDIR}/stage.loop_list/tools_to_inspect_initial"
		message_echo
	}
	program_exec_and_record_completion DETERMINE_ALL_TARGET_PORTS_OF_TOOLS
}

# ============= Preliminary inspection of initially installed tools which have to be up-to-date =============
# (No need depend on PARSE_CONF because INSPECT_ALL_DEPENDENCIES will take the task.)
main_preliminary_inspection_of_initial_tools ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='DETERMINE_ALL_TARGET_PORTS_OF_TOOLS'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin
		origin=$1
		database_build_setup_initial_node "$origin"
	}
	_program_exec_and_record_completion__operation ()
	{
		local DEPTH_INDEX
		message_section_title "Preliminary inspection of initially installed tools which have to be up-to-date"
		program_exec_restartable_loop_operation tools_to_inspect_initial
		database_build_post_inspect_initial_dependencies
		message_echo
	}
	program_exec_and_record_completion PRELIMINARY_INSPECTION_OF_INITIAL_TOOLS
}

# ============= Preliminary inspection of tools which have to be up-to-date =============
# (No need depend on PARSE_CONF because INSPECT_ALL_DEPENDENCIES will take the task.)
main_preliminary_inspection_of_tools ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='PRELIMINARY_INSPECTION_OF_INITIAL_TOOLS'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin
		origin=$1
		database_build_inspect_dependencies "$origin"
	}
	_program_exec_and_record_completion__operation ()
	{
		local DEPTH_INDEX
		message_section_title "Preliminary inspection of tools which have to be up-to-date"
		cp /dev/null "${DBDIR}/done_required_ports_to_inspect"
		DEPTH_INDEX='--'
		program_exec_restartable_loop_operation tools_to_inspect
		database_build_post_inspect_dependencies
		message_echo
	}
	program_exec_and_record_completion PRELIMINARY_INSPECTION_OF_TOOLS
}

# ============= Upgrade of pkg(8) if new =============
# (No need depend on PARSE_CONF because REINSTALLATION will take the task.)
main_upgrade_pkg8_if_new ()
{
	local PROGRAM_DEPENDS
	if [ \( "$COMMAND_MODE" = do -o "$COMMAND_MODE" = redo \) \
		-a $opt_dry_run = no -a $opt_suppress_pkgtools_upadte = no \
		-a "$PKGSYS_USE_PKGNG" = yes ]
	then
		PROGRAM_DEPENDS='PRELIMINARY_INSPECTION_OF_TOOLS'
		_program_exec_and_record_completion__operation ()
		{
			local _MSG_CURRENT_STAGE_general origin
			_MSG_CURRENT_STAGE_general="pkg(8) upgrade"
			origin=`pkgsys_portsmgmt_pkg`
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			message_section_title "Upgrade of $origin if new"
			touch "${DBDIR}/target_all"
			cp "${DBDIR}/requires/$origin/requirements.all.direct.src" "${DBDIR}/requires/$origin/requirements.all.full"
			reinstall_exec "$origin"
			reinstall_restore_conflicts
			rm -f "${DBDIR}/target_all"
			temp_set_msg_current_stage
			message_echo
		}
		program_exec_and_record_completion UPGRADE_PKGNG
	fi
}

# ============= Upgrade of dialog4ports(1) if new =============
# (No need depend on PARSE_CONF because REINSTALLATION will take the task.)
main_upgrade_dialog4ports1_if_new ()
{
	local PROGRAM_DEPENDS
	if [ \( "$COMMAND_MODE" = do -o "$COMMAND_MODE" = redo \) \
		-a $opt_dry_run = no -a $opt_suppress_pkgtools_upadte = no ] \
		&& pkgsys_is_dialog4ports_used
	then
		PROGRAM_DEPENDS='PRELIMINARY_INSPECTION_OF_TOOLS'
		_program_exec_and_record_completion__operation ()
		{
			local _MSG_CURRENT_STAGE_general origin
			_MSG_CURRENT_STAGE_general="dialog4ports(1) upgrade"
			origin=`pkgsys_portsmgmt_dialog4ports`
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			message_section_title "Upgrade of $origin if new"
			touch "${DBDIR}/target_all"
			cp "${DBDIR}/requires/$origin/requirements.all.direct.src" "${DBDIR}/requires/$origin/requirements.all.full"
			reinstall_exec "$origin"
			reinstall_restore_conflicts
			rm -f "${DBDIR}/target_all"
			temp_set_msg_current_stage
			message_echo
		}
		program_exec_and_record_completion UPGRADE_DIALOG4PORTS
	fi
}

# ============= Upgrade of this utility if new =============
# (No need depend on PARSE_CONF because REINSTALLATION will take the task.)
main_self_upgrade ()
{
	local PROGRAM_DEPENDS
	if [ \( "$COMMAND_MODE" = do -o "$COMMAND_MODE" = redo \) \
		-a $opt_dry_run = no -a $opt_suppress_self_upadte = no \
		-a -n "$MYPORTORIGIN" ]
	then
		PROGRAM_DEPENDS='PRELIMINARY_INSPECTION_OF_TOOLS'
		_program_exec_and_record_completion__operation ()
		{
			local _MSG_CURRENT_STAGE_general
			_MSG_CURRENT_STAGE_general="pkgng upgrade"
			temp_set_msg_current_stage "${_MSG_CURRENT_STAGE_general}"
			message_section_title "Upgrade of this utility if new"
			touch "${DBDIR}/target_all"
			cp "${DBDIR}/requires/$MYPORTORIGIN/requirements.all.direct.src" "${DBDIR}/requires/$MYPORTORIGIN/requirements.all.full"
			reinstall_exec "$MYPORTORIGIN"
			reinstall_restore_conflicts
			rm -f "${DBDIR}/target_all"
			temp_set_msg_current_stage
			message_echo
		}
		program_exec_and_record_completion UPGRADE_SELF
	fi
	if [ "x`${APPNAME} -aV 2> /dev/null`" != "x$MYVERSION" ]
	then
		message_echo "INFO: ${APPNAME} is upgraded and the temporary database needs refresh."
		database_maintain_clean_for_self_upgrade || :
		message_echo "INFO: Restarting with the new version."
		message_echo
		temp_trap_for_invoking_new_version
		. "${TMPDIR}"/restart_command.sh
		exit
	fi
}

# ============= Overlay onto the temporary database by reflecting changes in configuration =============
main_reflect_conf_changes ()
{
	local PROGRAM_DEPENDS
	PROGRAM_DEPENDS='PARSE_CONF'
	_program_exec_and_record_completion__operation ()
	{
		local tmpfile_diff tmpfile_old tmpfile_new key
		[ -d "${DBDIR}/conf.prev" ] || return 0
		message_section_title "Overlay onto the temporary database by reflecting changes in configuration"
		tmpfile_old=${TMPDIR}/PATCH_TO_TMPDB_REFLECT_CONF_CHANGES::old
		tmpfile_new=${TMPDIR}/PATCH_TO_TMPDB_REFLECT_CONF_CHANGES::new
		tmpfile_updated_ports=${TMPDIR}/PATCH_TO_TMPDB_REFLECT_CONF_CHANGES::updated_ports
		if fileedit_manipulate_old_new_lines \
			"${DBDIR}/conf.prev/setenv.sh" "${DBDIR}/conf/setenv.sh" "$tmpfile_old" "$tmpfile_new"
		then
			if env LANG=C grep -q -e ^LOCALBASE= -e ^LINUXBASE= -e ^PORTSDIR= "$tmpfile_old" "$tmpfile_new"
			then
				message_echo "ERROR: Migration of the temporary database is unavailable because LOCALBASE, LINUXBASE or PORTSDIR was changed." >&2
				message_echo "        ${APPNAME} clean" >&2
				message_echo "must be executed in advance." >&2
				exit 1
			fi
		fi
		cut -s -d '|' -f 1,2 "${DBDIR}/conf.prev/MOVED_ALT.parsed" | tr '|' '\t' > ${TMPDIR}/MOVED_ALT.old
		cut -s -d '|' -f 1,2 "${DBDIR}/conf/MOVED_ALT.parsed" | tr '|' '\t' > ${TMPDIR}/MOVED_ALT.new
		if fileedit_manipulate_old_new_lines \
			"${TMPDIR}/MOVED_ALT.old" "${TMPDIR}/MOVED_ALT.new" "$tmpfile_old" "$tmpfile_new"
		then
			cat "$tmpfile_old" "$tmpfile_new" | while read from to
			do
				echo "$from"
				[ -n "$to" ] && echo "$to"
			done
		fi > $tmpfile_updated_ports
		sort -u "${DBDIR}/conf/NOPKG:PORTS.parsed" 2> /dev/null > ${TMPDIR}/NOPKG:PORTS.parsed.new || :
		sort -u "${DBDIR}/conf.prev/NOPKG:PORTS.parsed" 2> /dev/null > ${TMPDIR}/NOPKG:PORTS.parsed.old || :
		diff "${TMPDIR}/NOPKG:PORTS.parsed.old" "${TMPDIR}/NOPKG:PORTS.parsed.new" | sed -n 's/^[<>] //p' >> $tmpfile_updated_ports
		if fileedit_manipulate_old_new_lines \
			"${DBDIR}/conf.prev/REPLACE.csv" "${DBDIR}/conf/REPLACE.csv" "$tmpfile_old" "$tmpfile_new"
		then
			cat "$tmpfile_old" "$tmpfile_new" | while read from to
			do
				echo "$from"
				[ -n "$to" ] && echo "$to"
			done
		fi >> $tmpfile_updated_ports
		[ `wc -l < $tmpfile_updated_ports` -gt 0 ] && rm -f "${DBDIR}/REPLACE.complete_sed_pattern"
		[ -d "${DBDIR}/conf/each_port" ] && find "${DBDIR}/conf/each_port" -depth 2 \
			| while read dbpath
		do
			origin=`str_dirpath_to_origin "$dbpath"`
			dbpath_prev=${DBDIR}/conf.prev/each_port/$origin
			diff -r "$dbpath_prev" "$dbpath" > /dev/null 2>&1 && continue
			echo "$origin"
		done >> $tmpfile_updated_ports
		[ -d "${DBDIR}/conf.prev/each_port" ] && find "${DBDIR}/conf.prev/each_port" -depth 2 \
			| while read dbpath_prev
		do
			origin=`str_dirpath_to_origin "$dbpath_prev"`
			dbpath=${DBDIR}/conf/each_port/$origin
			[ -d "$dbpath" ] && continue
			echo "$origin"
		done >> $tmpfile_updated_ports
		if [ `wc -l < $tmpfile_updated_ports` -gt 0 ]
		then
			sort -u "$tmpfile_updated_ports" | while read origin
			do
				message_echo "Reset for $origin"
				database_build_patch_reconf "$origin"
			done
			program_deregister_stage_complete PREPARE_FOR_INSPECT_ALL_DEPENDENCIES
			program_deregister_stage_complete ALL_COMPLETE
		fi
		message_echo
	}
	program_exec_and_record_completion PATCH_TO_TMPDB_REFLECT_CONF_CHANGES
}
