#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Wrappers for hiding version differences in the Ports/Packages system -
# Copyright (C) 2013-2022 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
PKGSYS_USE_PKGNG=yes	# no: legacy pkg_* tools, yes: the new generation package (pkgng)
PKGSYS_CMD_PKG_DELETE='pkg delete'	# Corresponding command for pkg_delete
PKGSYS_AVR_REFETCH_TIMES_PER_SITE=1	# Average number (integer) of retrials for retrieving package or distfiles per mirror site
PKGSYS_AVR_REFETCH_TIMES_FOR_CHKSUMERR=2	#  Number (integer) of retrials for check sum error in retrieving a package

# ============= Get the time stamp of the ports tree =============
pkgsys_get_timestamp_portstree ()
{
	stat -t %s -f %m "${PORTS_INDEX_DB}" 2> /dev/null || :
}

# ============= Update the ports tree =============
pkgsys_update_portstree ()
{
		if [ -e "$PORTSNAP_WORKDIR/INDEX" ]
		then
			mode=update
		else
			mode=extract
		fi
		if [ $opt_batch_mode = no ]
		then
			stdout=/dev/stdout
		else
			stdout=/dev/null
		fi
		portsnap fetch $mode > $stdout
}

# ============= Check implementation of the ports tree =============
pkgsys_chk_ports_tree_implementation ()
{
	fs_fix_unionfs_image_if_hidden "${PORTSDIR}/Mk/bsd.port.mk" || :
	fs_fix_unionfs_image_if_hidden "${PORTSDIR}/Makefile" || :
	if [ ! -d "${PORTSDIR}" ]
	then
		message_echo "ERROR: Ports directory ${PORTSDIR} is not found." >&2
		exit 1
	fi
	if [ ! -e "${PORTSDIR}/Makefile" -o ! -e "${PORTSDIR}/Mk/bsd.port.mk" ]
	then
		message_echo "ERROR: Ports tree ${PORTSDIR} is missing, broken or incompatible." >&2
		exit 1
	fi
}

# ============= System defined value for the ports/packages =============
pkgsys_sysvar ()
{
	local var tmp_work
	var=$1
	tmp_work=${TMPDIR}/pkgsys_sysvar:work
	rm -rf "$tmp_work"
	mkdir "$tmp_work"
	fs_fix_unionfs_image_if_hidden "${PORTSDIR}/Mk/bsd.port.mk"
	make -C "$tmp_work" -f "${PORTSDIR}/Mk/bsd.port.mk" -V "$var" 2> /dev/null
}

# ============= Get the file name of package check sum file =============
pkgsys_pkgchksum_file ()
{
	echo CHECKSUM.MD5
}

# ============= Get the origin of the currently used pkg(8) =============
# NOTE: Assumed to be unflavored.
pkgsys_portsmgmt_pkg ()
{
	local origin_unflavored
	if [ ! -e "${DBDIR}/PKG_ORIGIN" ]
	then
		origin_unflavored=`pkgsys_sysvar PKG_ORIGIN` || :
		[ -n "$origin_unflavored" -a -d "${PORTSDIR}/$origin_unflavored" ] || origin_unflavored=ports-mgmt/pkg
		echo "$origin_unflavored" > ${DBDIR}/PKG_ORIGIN
	fi
	cat "${DBDIR}/PKG_ORIGIN"
}

# ============= Get the origin of the currently used dialog4ports(1) =============
# NOTE: Assumed to be unflavored.
pkgsys_portsmgmt_dialog4ports ()
{
	local origin_unflavored
	if [ ! -e "${DBDIR}/DIALOGPORT" ]
	then
		origin_unflavored=`pkgsys_sysvar DIALOGPORT` || :
		[ -n "$origin_unflavored" -a -d "${PORTSDIR}/$origin_unflavored" ] || origin_unflavored=ports-mgmt/dialog4ports
		echo "$origin_unflavored" > ${DBDIR}/DIALOGPORT
	fi
	cat "${DBDIR}/DIALOGPORT"
}

# ============= Check whether a port is indispensable for the standard function of the ports/packages system =============
pkgsys_is_pkgtool ()
{
	local origin origin_unflavored
	origin=$1
	origin_unflavored=`echo "$origin" | sed 's/@.*$//'`
	case $origin_unflavored in
	ports-mgmt/pkg | ports-mgmt/pkg-devel | ports-mgmt/dialog4ports | ports-mgmt/dialog4ports-static )
		;;
	*)	return 1
		;;
	esac
}

# ============= Check whether a port is indispensable for package operations =============
pkgsys_is_necessary_pkgtool ()
{
	local origin origin_unflavored
	origin=$1
	origin_unflavored=`echo "$origin" | sed 's/@.*$//'`
	[ x"$WITH_PKGNG" = x'yes' -a \( x"$origin_unflavored" = x'ports-mgmt/pkg' -o x"$origin_unflavored" = x'ports-mgmt/pkg-devel' \) ]
}

# ============= Get the extended regular expression pattern of ports for pkg(8) =============
pkgsys_pkgtools_ports_filter_regexp ()
{
	echo '^ports-mgmt/(pkg|pkg-devel)(|@.*)$'
}

# ============= Get the extended regular expression pattern of package names for pkg(8) =============
pkgsys_pkgtools_pkgs_filter_regexp ()
{
	echo '^(pkg|pkg-devel)-[0-9]\.'
}

# ============= Check whether the dialog for selecting port options is dialog4ports =============
pkgsys_is_dialog4ports_used ()
{
	[ -n "`pkgsys_sysvar DIALOG4PORTS`" ]
}

# ============= Get the number of mirror sites for legacy packages =============
pkgsys_num_mirrorsites ()
{
	local siteroots
	siteroots=$1
	echo "$siteroots" | tr '|' '\n' | wc -l
}

# ============= Get a URL one of mirror sites =============
pkgsys_url_from_mirrors ()
{
	local siteroots subdir nsites id_site site platform version
	siteroots=$1
	subdir=$2
	nsites=`pkgsys_num_mirrorsites "$siteroots"`
	id_site=`(set +e; random -e $nsites; echo $?)`
	site=`echo "$siteroots" | tr '|' '\n' | sed -n $((${id_site}+1))p`
	platform=`uname -p`
	version=`uname -r | cut -d - -f 1,2 | tr [:upper:] [:lower:]`
	echo -n "$site"
	printf "$subdir\n" "$platform" "$version"
}

# ============= Fetch a file from one of mirror sites =============
pkgsys_fetch_from_mirrors ()
{
	local siteroots subdir filename dst tmp_work fetch itrial origdir url
	siteroots=$1
	subdir=$2
	filename=$3
	dst=$4
	tmp_work=${TMPDIR}/pkgsys_fetch_from_mirrors:work
	rm -rf "$tmp_work"
	mkdir "$tmp_work"
	fetch=`pkgsys_sysvar FETCH_CMD`
	itrial=$((`pkgsys_num_mirrorsites "$siteroots"`*$PKGSYS_AVR_REFETCH_TIMES_PER_SITE))
	origdir=`pwd`
	cd "$tmp_work"
	while [ $itrial -ge 1 ]
	do
		url=`pkgsys_url_from_mirrors "$siteroots" "$subdir"`$filename
		message_echo "INFO: Fetching from $url:"
		$fetch "$url"&& break
		rm -f "$filename"
		itrial=$(($itrial-1))
	done
	cd "$origdir"
	[ -e "$tmp_work/$filename" ] || return
	mv "$tmp_work/$filename" "$dst"
}

# ============= Get the package check sums file ready =============
pkgsys_ready_checksum_file ()
{
	local chksumfile
	tmp_savedpath=${TMPDIR}/pkgsys_ready_checksum_file:savedpath
	rm -f "$tmp_savedpath"
	chksumfile=`pkgsys_pkgchksum_file`
	if [ ! -e "${DBDIR}/checksum/$chksumfile" ]
	then
		[ -d "${DBDIR}/checksum" ] || mkdir "${DBDIR}/checksum"
		if ! pkgsys_fetch_from_mirrors "${PACKAGECHECKSUMROOTS}" "${PACKAGECHECKSUMDIR}" \
			"$chksumfile" "${DBDIR}/checksum"
		then
			message_echo "WARNING: No check sum file is available." >&2
			return 1
		fi
	fi
	echo "${DBDIR}/checksum/$chksumfile" > $tmp_savedpath
}

# ============= Get the location of a check sums file fetched by pkgsys_ready_checksum_file =============
pkgsys_ready_checksum_file__fetched_file ()
{
	cat "${TMPDIR}/pkgsys_ready_checksum_file:savedpath"
}

# ============= Fetch a legacy package from one of remote sites =============
pkgsys_fetch_legacy_remote ()
{
	local pkg tmp_work tmp_pkgpath pkg_regexp checksumpath validMD5 fetchedMD5 needs_fetch itrial
	pkg=$1
	tmp_work=${TMPDIR}/pkgsys_fetch_legacy_remote:work
	tmp_pkgpath=${TMPDIR}/pkgsys_fetch_legacy_remote:pkgpath
	rm -rf "$tmp_work"
	mkdir "$tmp_work"
	pkg_regexp=`str_escape_regexp "$pkg"`
	pkgsys_ready_checksum_file || return
	checksumpath=`pkgsys_ready_checksum_file__fetched_file`
	validMD5=`env LANG=C grep -m 1 -E -e "^MD5[[:space:]]*\($pkg_regexp\.tbz\)[[:space:]]*=" "$checksumpath" | sed -E "s/^[^=]*=[[:space:]]*(.*)/\1/"`
	if [ -z "$validMD5" ]
	then
		message_echo "WARNING: No check sum for $pkg.tbz." >&2
		return 1
	fi
	needs_fetch=yes
	mkdir -p "${PKGREPOSITORY}"
	if [ -e "${PKGREPOSITORY}/$pkg.tbz" ]
	then
		if [ -e "${PKGREPOSITORY}/$pkg.md5=$validMD5.tbz" ]
		then
			fetchedMD5=`md5 "${PKGREPOSITORY}/$pkg.md5=$validMD5.tbz" | sed -E "s/^[^=]*=[[:space:]]*(.*)/\1/"`
			[ "x$fetchedMD5" = "x$validMD5" ] || rm "${PKGREPOSITORY}/$pkg.md5=$fetchedMD5.tbz"
		fi
		if [ -e "${PKGREPOSITORY}/$pkg.md5=$validMD5.tbz" ]
		then
			ln -f "${PKGREPOSITORY}/$pkg.md5=$fetchedMD5.tbz" "${PKGREPOSITORY}/$pkg.tbz"
		else
			fetchedMD5=`md5 "${PKGREPOSITORY}/$pkg.tbz" | sed -E "s/^[^=]*=[[:space:]]*(.*)/\1/"`
			if [ "x$fetchedMD5" = "x$validMD5" ]
			then
				needs_fetch=no
			else
				mv "${PKGREPOSITORY}/$pkg.tbz" "${PKGREPOSITORY}/$pkg.md5=$fetchedMD5.tbz"
			fi
		fi
	fi
	if [ $needs_fetch = yes ]
	then
		itrial=$PKGSYS_AVR_REFETCH_TIMES_FOR_CHKSUMERR
		while [ $itrial -ge 1 ]
		do
			if pkgsys_fetch_from_mirrors "${PACKAGEROOTS}" "${PACKAGEDIR}" \
				"$pkg.tbz" "$tmp_work"
			then
				fetchedMD5=`md5 "$tmp_work/$pkg.tbz" | sed -E "s/^[^=]*=[[:space:]]*(.*)/\1/"`
				[ "x$fetchedMD5" = "x$validMD5" ] && break
				message_echo "WARNING: Check sum mismatches for $pkg.tbz." >&2
			fi
			itrial=$(($itrial-1))
		done
		[ $itrial -ge 1 ] || return
		mv "$tmp_work/$pkg.tbz" "${PKGREPOSITORY}"
	fi
	echo "${PKGREPOSITORY}/$pkg.tbz" > $tmp_pkgpath
}

# ============= Get the location of a package fetched by pkgsys_fetch_legacy_remote =============
pkgsys_fetch_legacy_remote__fetched_pkg ()
{
	cat "${TMPDIR}/pkgsys_fetch_legacy_remote:pkgpath"
}

# ============= Check whether the dependency of a legacy package is the latest =============
pkgsys_is_dependency_of_a_legacypkg_latest ()
{
	local pkgarc tmp_extract tmp_contents tmp_origin tmp_pkg pkg nlines iline origin_req pkg_req pkg_new
	pkgarc=$1
	tmp_extract=${TMPDIR}/pkgng:pkgsys_is_dependency_of_a_legacypkg_latest:extract
	tmp_contents=${TMPDIR}/pkgng:pkgsys_is_dependency_of_a_legacypkg_latest:contents
	tmp_origin=${TMPDIR}/pkgng:pkgsys_is_dependency_of_a_legacypkg_latest:origin
	tmp_pkg=${TMPDIR}/pkgng:pkgsys_is_dependency_of_a_legacypkg_latest:pkg
	pkg=`pkgsys_pkgarc_to_pkgname "$pkgarc"`
	[ -e "$pkgarc" ] || return
	rm -rf "$tmp_extract"
	mkdir "$tmp_extract"
	tar xf "$pkgarc" -C "$tmp_extract" +CONTENTS
	env LANG=C grep -e '^@pkgdep[[:space:]]' -e '^@comment[[:space:]]*DEPORIGIN:' "$tmp_extract/+CONTENTS" \
		| sed 's/^@pkgdep[[:space:]]*//;s/^@comment[[:space:]]*DEPORIGIN://' > $tmp_contents
	nlines=`wc -l < $tmp_contents`
	iline=1
	while [ $iline -le $nlines ]
	do
		origin_req=`sed -n ${iline}p "$tmp_contents"`
		pkg_req=`sed -n $(($iline+1))p "$tmp_contents"`
		iline=$(($iline+2))
		pkg_new=`cat "${DBDIR}/requires/$origin_req/new_version" 2> /dev/null` || :
		if [ -z "$pkg_new" -o "$pkg_new" != "$pkg_req" ]
		then
			message_echo "WARNING: Requirements of remote package $pkg are not the latest." >&2
			return 1
		fi
	done
	:
}

# ============= Check whether legacy package tools are available =============
pkgsys_is_legacy_tool_available ()
{
	which -s pkg_info
}

# ============= Load portsnap.conf(5) for important directories  =============
pkgsys_load_portsnap_conf ()
{
	if which -s portsnap
	then
		PORTSNAP_WORKDIR=`sed -n '/^[[:space:]]*WORKDIR[[:space:]]*=[[:space:]]*\([^[:space:]#]*\)/s//\1/p' /etc/portsnap.conf || :`
		PORTSNAP_WORKDIR=${PORTSNAP_WORKDIR:-/var/db/portsnap}
		PORTSNAP_PORTSDIR=`sed -n '/^[[:space:]]*PORTSDIR[[:space:]]*=[[:space:]]*\([^[:space:]#]*\)/s//\1/p' /etc/portsnap.conf || :`
		PORTSNAP_PORTSDIR=${PORTSNAP_PORTSDIR:-/usr/ports}
	else
		PORTSNAP_WORKDIR=
		PORTSNAP_PORTSDIR=
	fi
}

# ============= Define wrapper commands for hiding the differences between pkg_* tools and pkgng =============
pkgsys_def_pkgtools ()
{
	if [ "${DBDIR}/WITH_PKGNG" -nt /etc/make.conf -o \( -e "${DBDIR}/WITH_PKGNG" -a ! -e /etc/make.conf \) ]
	then
		PKGSYS_USE_PKGNG=`cat "${DBDIR}/WITH_PKGNG"`
	else
		PKGSYS_USE_PKGNG=`pkgsys_sysvar WITH_PKG | tr '[:upper:]' '[:lower:]'`
		[ -n "$PKGSYS_USE_PKGNG" ] || PKGSYS_USE_PKGNG=`pkgsys_sysvar WITH_PKGNG | tr '[:upper:]' '[:lower:]'`
		if [ -d "${DBDIR}" ] && misc_is_superuser_privilege
		then
			echo "$PKGSYS_USE_PKGNG" > ${DBDIR}/WITH_PKGNG.tmp
			mv "${DBDIR}/WITH_PKGNG.tmp" "${DBDIR}/WITH_PKGNG"
		fi
	fi
	if [ "x$PKGSYS_USE_PKGNG" = xyes ]
	then
		export WITH_PKG=yes
		export WITH_PKGNG=yes
		PKGSYS_CMD_PKG_DELETE='pkg delete'
		pkg_is_tool_ready ()
		{
			TMPDIR=/dev/null ASSUME_ALWAYS_YES=yes  PACKAGESITE=file:///nonexistent \
				pkg info -x 'pkg(-devel)?$' >/dev/null 2>&1
		}
		pkg_is_tool_available ()
		{
			local dev_out dev_err
			if [ -x /usr/sbin/pkg ]
			then
				dev_out=/dev/stdout
				dev_err=/dev/stderr
				if [ $opt_batch_mode = yes ]
				then
					dev_out=/dev/null
					dev_err=/dev/null
				fi
				pkg_is_tool_ready && return
				env ASSUME_ALWAYS_YES=yes pkg bootstrap -f > $dev_out 2> $dev_err
				pkg_is_tool_ready
			else
				which -s pkg
			fi
		}
		pkg_info_Ea ()
		{
			pkg info -qa 2> /dev/null
		}
# 		pkg_info_qoa ()
# 		{
# 			pkg info -qoa 2> /dev/null
# 		}
# 		pkg_info_qox ()
# 		{
# 			pkg info -qox "$@" 2> /dev/null
# 		}
# 		pkg_info_qoX ()
# 		{
# 			pkg info -qox "$@" 2> /dev/null
# 		}
		pkg_info_qO ()
		{
			pkg info -qO "$@" 2> /dev/null
		}
		pkg_info_qo ()
		{
			pkg info -qo "$@" 2> /dev/null
		}
		pkg_info_qr ()
		{
			pkg info -qd "$@" 2> /dev/null
		}
		pkg_info_e ()
		{
			pkg info -qe "$@" 2> /dev/null
		}
		pkg_info_eO ()
		{
			pkg info -qeO "$@" 2> /dev/null
		}
		pkg_info_Eg ()
		{
			pkg info -Eg "$@" 2> /dev/null
		}
		pkg_info_qR ()
		{
			pkg info -qr "$@" 2> /dev/null
		}
		pkg_info_Ex ()
		{
			pkg info -Ex "$@" 2> /dev/null
		}
		pkg_info_qL ()
		{
			pkg info -ql "$@" 2> /dev/null
		}
		pkg_info_qs ()
		{
			# Return the total storage space occupied by the installed files in bytes
			pkg info -qs "$@" 2> /dev/null | sed 's/KiB$/*1024/;s/MiB$/*1024^2/;s/GiB$/*1024^3/;s/B$//' | tr '\n' + | sed 's/+$//' | bc -l
		}
		pkg_info_flavors ()
		# For an unflavored pattern, all matching flavors are returned.
		{
			local glob_unflavored
			glob_unflavored=$1
			pkg query -g '%At\t%Av' "$glob_unflavored" 2> /dev/null | env LANG=C grep -E '^flavor[[:space:]]' | cut -f 2
		}
		pkg_info_flavored_origins ()
		# For an unflavored pattern, all matching flavors are returned.
		{
			local glob_unflavored origin_unflavored
			glob_unflavored=$1
			origin_unflavored=`pkg_info_qo "$glob_unflavored" 2> /dev/null || :`
			flavors=`pkg_info_flavors "$glob_unflavored" 2> /dev/null || :`
			if [ -z "$flavors" ]
			then
				echo "$origin_unflavored"
			else
				for flavor in $flavors
				do
					echo "$origin_unflavored@$flavor"
				done
			fi
		}
		pkg_info_all_flavored_origins ()
		{
			local tmp_flavored tmp_flavored_ptn
			tmp_flavored_ptn=${TMPDIR}/pkg_info_all_flavored_origins:flavored_ptn
			pkg query '%o\t%At\t%Av' 2> /dev/null | env LANG=C grep -E '^[^[:space:]]+[[:space:]]flavor[[:space:]]' | cut -f 1,3 | tr '\t' @ | sed 's/@$//'
			pkg query '%n\t%At' 2> /dev/null | env LANG=C grep -E '^[^[:space:]]+[[:space:]]flavor$' | cut -f 1 > $tmp_flavored_ptn
			pkg query '%n\t%o' 2> /dev/null | env LANG=C grep -vFx -f "$tmp_flavored_ptn" | cut -f 2
		}
		pkg_check_sanity ()
		{
			local pkg tmp_stdout tmp_stderr
			pkg=$1
			tmp_stdout=${TMPDIR}/pkgng:pkg_check_sanity:stdout
			tmp_stderr=${TMPDIR}/pkgng:pkg_check_sanity:stderr
			pkg check -s "$pkg" > $tmp_stdout 2> $tmp_stderr || :
			{
				env LANG=C grep '^[^:]*: checksum mismatch for ' "$tmp_stderr" | sed 's/^[^:]*: checksum mismatch for //' || :
				env LANG=C grep '^[^:]*: missing file ' "$tmp_stderr" | sed 's/^[^:]*: missing file //' || :
				if env LANG=C grep -q '^pkg: .*: No such file or directory$' "$tmp_stderr"	# For the old specification of pkg(8)
				then
					pkg info -ql "$pkg" 2> /dev/null | while read filepath
					do
						[ -e "$filepath" ] || echo "$filepath"
					done
				fi
			} | sort -u || :
		}
		pkg_which ()
		{
			local filepath
			filepath=$1
			pkg which -q "$filepath" 2> /dev/null
		}
		pkg_info_gen_pkg_origin_table ()
		{
			local pkgname origin
			#	pkg query -g '%n-%v\t%o' \* 2> /dev/null > ${DBDIR}/installed_ports:pkg_vs_origin.tbl
			for pkgname in `pkg info -qa 2> /dev/null`
			do
				for origin in `pkg_info_flavored_origins "$pkgname"`
				do
					printf '%s\t%s\n' "$pkgname" "$origin"
				done
			done
		}
		pkg_create_b ()
		{
			pkg create "$@"
		}
		pkg_delete_f ()
		{
			local opt_del opt_quit dev_out dev_err
			opt_del=
			[ $opt_no_exec_inst_script = yes ] && opt_del='-D'
			opt_quit=
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				opt_quit='-q'
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			pkg delete -fqy $opt_del "$@" > $dev_out 2> $dev_err || :
			pkg_is_tool_ready || return 0	# If pkg(8) is deinstalled successfully
			pkg info -e $opt_quit "$@" > /dev/null 2>&1 || return 0
			pkg delete -fy $opt_del "$@" > $dev_out 2> $dev_err	# Countermeasure for a bug found for pkg-1.3.4 (at least not until 1.2.7_4)
		}
		pkg_add_tools ()
		{
			local pkgarc tmp_extract prefix prefix_parent pkg opt_quit dev_out dev_err
			pkgarc=$1
			tmp_extract=${TMPDIR}/pkgng:pkg_add_tools:extract
			opt_quit=
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				opt_quit='-q'
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			rm -rf "$tmp_extract"
			mkdir "$tmp_extract"
			tar xf "$pkgarc" -C "$tmp_extract" > $dev_out 2> $dev_err
			pkg=`pkgsys_pkgarc_to_pkgname "$pkgarc"`
			if env ASSUME_ALWAYS_YES=YES $tmp_extract/usr/local/sbin/pkg-static add $opt_quit "$pkgarc" > $dev_out 2> $dev_err
			then
				message_echo "INFO: $pkg is successfully registered."
			else
				message_echo "WARNING: Failed to register $pkg, but the process is continued." >&2
				return 1
			fi
		}
		pkg_add_f_common ()
		{
			local option pkgarc pkg pkg_tool pkg_gen opt_add opt_quit dev_out dev_err
			option=$1
			shift || :
			pkg_tool=
			pkg_gen=
			for pkgarc in "$@"
			do
				pkg=`basename "$pkgarc"`
				if expr "$pkg" : '^pkg-[0-9][0-9]*\..*' > /dev/null
				then
					pkg_tool=$pkgarc
				else
					pkg_gen="$pkg_gen $pkgarc"
				fi
			done
			[ -n "$pkg_tool" ] && pkg_add_tools "$pkg_tool"
			if [ -n "$pkg_gen" ]
			then
				rm -rf "${TMPDIR}/pkg_add_f"
				mkdir -p "${TMPDIR}/pkg_add_f"
				opt_add=
				[ $opt_no_exec_inst_script = yes ] && opt_add='-I'
				opt_quit=
				dev_out=/dev/stdout
				dev_err=/dev/stderr
				if [ $opt_batch_mode = yes ]
				then
					opt_quit='-q'
					dev_out=/dev/null
					dev_err=/dev/null
				fi
				( cd "${TMPDIR}/pkg_add_f" && ln -s $pkg_gen && env ASSUME_ALWAYS_YES=YES pkg add -M $option $opt_quit $opt_add * > $dev_out 2> $dev_err )
			fi
		}
		pkg_add_f ()
		{
			pkg_add_f_common '' "$@"
		}
		pkg_add_fF ()
		{
			pkg_add_f_common '-f' "$@"
		}
		pkg_inst_verify_pkg ()
		{
			local pkg pkgarc
			pkg=$1
			pkgarc=`pkgsys_pkgname_to_pkgarc "${PKGNG_PKG_CACHEDIR}" "$pkg"` || return
			tar tf "$pkgarc" > /dev/null 2>&1
		}
		pkg_inst_remote_fetch ()
		{
			local pkg opt_quit
			pkg=$1
			opt_quit=
			[ $opt_batch_mode = yes ] && opt_quit='-q'
			pkg fetch -yU $opt_quit "$pkg"
		}
		pkg_inst_remote_verify_fetch ()
		{
			local pkg
			pkg=$1
			pkg_inst_verify_pkg "$pkg" && return
			pkg_inst_remote_fetch "$pkg" && pkg_inst_verify_pkg "$pkg"
		}
		pkg_inst_remote ()
		{
			local pkg pkgarc opt_add opt_quit dev_out dev_err
			pkg=$1
			pkg_inst_remote_verify_fetch "$pkg" || return
			pkgarc=`pkgsys_pkgname_to_pkgarc "${PKGNG_PKG_CACHEDIR}" "$pkg"` || return
			rm -rf "${TMPDIR}/pkg_inst_remote"
			mkdir -p "${TMPDIR}/pkg_inst_remote"
			opt_add=
			[ $opt_no_exec_inst_script = yes ] && opt_add='-I'
			opt_quit=
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				opt_quit='-q'
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			( cd "${TMPDIR}/pkg_inst_remote" && ln -s "$pkgarc" && env ASSUME_ALWAYS_YES=YES pkg add -fM $opt_quit $opt_add * > $dev_out 2> $dev_err )
		}
		pkg_inst_wild_verify_pkg ()
		{
			local pkg pkgarc
			pkg=$1
			pkgarc=`pkgsys_pkgname_to_pkgarc "${PKGNG_PKG_CACHEDIR}" "$pkg"` || return
			pkgsys_is_dependency_of_a_legacypkg_latest "$pkgarc"
		}
		pkg_inst_remote_wild_fetch ()
		{
			local pkg
			pkg=$1
			if pkg_is_tool_available
			then
				pkg_inst_remote "$pkg" && return
			fi
			pkgsys_is_legacy_tool_available || return
			message_echo "INFO: Trying to use a legacy package and convert it to pkgng."
			pkgsys_fetch_legacy_remote "$pkg"
		}
		pkg_inst_remote_wild_verify_fetch ()
		{
			local pkg
			pkg=$1
			pkg_inst_wild_verify_pkg "$pkg" && return
			pkg_inst_remote_wild_fetch "$pkg" && pkg_inst_wild_verify_pkg "$pkg"
		}
		pkg_inst_remote_wild ()
		{
			local pkg pkgarc opt_add dev_out dev_err
			pkg=$1
			if pkg_inst_remote_wild_verify_fetch "$pkg"
			then
				pkgarc=`pkgsys_fetch_legacy_remote__fetched_pkg`
				rm -rf "${TMPDIR}/pkg_inst_remote_wild"
				mkdir -p "${TMPDIR}/pkg_inst_remote_wild"
				opt_add=
				[ $opt_no_exec_inst_script = yes ] && opt_add='-I'
				dev_out=/dev/stdout
				dev_err=/dev/stderr
				if [ $opt_batch_mode = yes ]
				then
					dev_out=/dev/null
					dev_err=/dev/null
				fi
				( cd "${TMPDIR}/pkg_inst_remote_wild" && ln -s "$pkgarc" && pkg_add -ifF $opt_add * > $dev_out 2> $dev_err) || return
				message_echo "INFO: Trying to convert the installed legacy package to pkgng."
				pkg2ng > $dev_out 2> $dev_err || :
				message_echo "INFO: Checking whether the conversion is successful."
				pkg info -qe "$pkg" 2> /dev/null
			fi
		}
		pkg_get_remote_repository_version ()
		{
			local origin origin_unflavored pkg
			origin=$1
			origin_unflavored=`expr "$origin" : '\([^@]*\)'` || return
			[ -n "$origin_unflavored" ] || return
			pkg fetch -qyU "$origin_unflavored" || return
			pkg=`pkg rquery -U %n-%v "$origin_unflavored"` || return
			echo "$pkg"
		}
		pkg_get_pkgs_timestamps ()
		{
			pkg query '%n-%v\tng-epoch:%t' "$@" 2> /dev/null
		}
		pkg_get_pkg_timestamp ()
		{
			local pkg
			pkg=$1
			pkg query '%t' "$pkg" 2> /dev/null
		}
		pkg_loadconf ()
		{
			local pkg_conf
			# Deafult configuration for pkg(1)
			PKGNG_PACKAGESITE='http://pkg.freebsd.org/${ABI}/latest'
			PKGNG_SRV_MIRRORS=YES
			PKGNG_PKG_DBDIR=/var/db/pkg
			PKGNG_PKG_CACHEDIR=/var/cache/pkg
			PKGNG_PORTSDIR=/usr/ports
			PKGNG_PUBKEY=/etc/ssl/pkg.conf
			PKGNG_HANDLE_RC_SCRIPTS=NO
			PKGNG_PKG_MULTIREPOS=NO
			PKGNG_ASSUME_ALWAYS_YES=NO
			PKGNG_SYSLOG=YES
			PKGNG_SHLIBS=NO
			PKGNG_AUTODEPS=NO
			PKGNG_PORTAUDIT_SITE='http=//portaudit.FreeBSD.org/auditfile.tbz'
			# Load configuration for pkg(1)
			pkg_conf=`pkg query %Fp pkg 2> /dev/null | env LANG=C grep '/etc/pkg\.conf\.sample$' | sed 's/\.sample$//'` || :
			[ -n "$pkg_conf" ] || pkg_conf=${MYPREFIX}/etc/pkg.conf
			env LANG=C grep -v -e '^[[:space:]]*#' -e '^[[:space:]]*$' "$pkg_conf" 2> /dev/null \
				| env LANG=C grep -e '^[[:space:]]*[A-Z0-9_]*[[:space:]]*:[[:space:]]*.*' \
				| while read srcline
			do
				var=`expr "$srcline" : '^[[:space:]]*\([A-Z0-9_]*\)[[:space:]]*:.*'` || :
				val=`expr "$srcline" : '^[[:space:]]*[A-Z0-9_]*[[:space:]]*:[[:space:]]*\(.*\)'` || :
				eval PKGNG_$var=\$val
				misc_get_all_vardefs | env LANG=C grep "^PKGNG_$var="
			done > ${TMPDIR}/pkgsys_def_pkgtools:pkg.conf.sh
			. "${TMPDIR}/pkgsys_def_pkgtools:pkg.conf.sh"
		}
		pkg_rescue_tools ()
		{
			local origin_portsmgmt packagepath checksumpath pkgname is_successful dev_out dev_err
			origin_portsmgmt=`pkgsys_portsmgmt_pkg`
			pkg_is_tool_ready && return
			packagepath=`pkgsys_get_backup_pkg "$origin_portsmgmt"` && \
				pkg_add_tools "$packagepath" && return
			pkg_is_tool_available && return
			message_echo "WARNING: WITH_PKG or WITH_PKGNG is set, but pkgng is still missing. It is installed now." >&2
			pkgsys_ready_checksum_file || return
			message_echo "INFO: Installing pkgng by legacy package tool."
			checksumpath=`pkgsys_ready_checksum_file__fetched_file`
			pkgname=`sed 's/^MD5[[:space:]]*(//;s/\.tbz)[[:space:]]*=[^=]*$//' "$checksumpath" \
				| env LANG=C grep -m 1 '^pkg-[0-9]'` || :
			[ -n "$pkgname" ] && pkg_inst_remote_wild "$pkgname" && return
			message_echo "INFO: Failed by package, so installing pkgng by port."
			env LANG=C grep -Ev '^[[:space:]]*WITH_PKG(|NG)=' /etc/make.conf > ${TMPDIR}/make.conf 2> /dev/null || :
			echo WITHOUT_PKG=yes >> ${TMPDIR}/make.conf
			echo WITHOUT_PKGNG=yes >> ${TMPDIR}/make.conf
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			( set -e
				unset WITH_PKG
				unset WITH_PKGNG
				unset WITHOUT_PKG
				unset WITHOUT_PKGNG
				message_echo "INFO: Attempting deinstallation of pkg(8) to make sure."
				portsmgmt_port_path=`pkgsys_get_portpath_from_origin "$origin_portsmgmt"`
				fs_fix_unionfs_image_if_hidden "${TMPDIR}/make.conf"
				env __MAKE_CONF="${TMPDIR}/make.conf" make -C "$portsmgmt_port_path" deinstall > $dev_out 2> $dev_err || :
				message_echo "INFO: Attempting (re)installation by pkg(8)."
				env __MAKE_CONF="${TMPDIR}/make.conf" make -C "$portsmgmt_port_path" reinstall clean > $dev_out 2> $dev_err
			) && {
				pkg2ng > $dev_out 2> $dev_err || :
				pkg_is_tool_available
			}
		}
		pkg_update_pkgrepository ()
		{
			local opts
			pkg_rescue_tools || return 0
			if [ $opt_batch_mode = no ]
			then
				opts=
			else
				opts='-q'
			fi
			pkg update $opts
		}
		if ! pkg_rescue_tools
		then
			message_echo "WARNING: Pkgng is still missing, but continuing for the time being." >&2
		fi
		pkg_loadconf
	elif ! pkgsys_is_legacy_tool_available
	then
		message_echo "ERROR: Pkgng is disabled although the legacy packages tools are unavailable. Resolve the problem manually." >&2
		exit 1
	else
		unset WITH_PKG
		unset WITH_PKGNG
		PKGSYS_USE_PKGNG=no
		PKGSYS_CMD_PKG_DELETE='pkg_delete'
		pkg_is_tool_ready ()
		{
			pkgsys_is_legacy_tool_available
		}
		pkg_is_tool_available ()
		{
			pkgsys_is_legacy_tool_available
		}
		pkg_info_Ea ()
		{
			pkg_info -Ea 2> /dev/null
		}
# 		pkg_info_qoa ()
# 		{
# 			pkg_info -qoa 2> /dev/null
# 		}
# 		pkg_info_qox ()
# 		{
# 			pkg_info -qox "$@" 2> /dev/null
# 		}
# 		pkg_info_qoX ()
# 		{
# 			pkg_info -qoX "$@" 2> /dev/null
# 		}
		pkg_info_qO ()
		{
			pkg_info -qO "$@" 2> /dev/null
		}
		pkg_info_qo ()
		{
			pkg_info -qo "$@" 2> /dev/null
		}
		pkg_info_qr ()
		{
			pkg_info -qr "$@" 2> /dev/null | sed -n 's/^@pkgdep[[:space:]]*//p'
		}
		pkg_info_e ()
		{
			pkg_info -qe "$@" 2> /dev/null
		}
		pkg_info_eO ()
		{
			[ `pkg_info -qO "$@" 2> /dev/null | wc -l` -gt 0 ]
		}
		pkg_info_Eg ()
		{
			pkg_info -E "$@" 2> /dev/null
		}
		pkg_info_qR ()
		{
			pkg_info -qR "$@" 2> /dev/null | env LANG=C grep -v '^$'
		}
		pkg_info_Ex ()
		{
			pkg_info -Ex "$@" 2> /dev/null
		}
		pkg_info_qL ()
		{
			pkg_info -qL "$@" 2> /dev/null
		}
		pkg_info_qs ()
		{
			# Return the total storage space occupied by the installed files in bytes
			pkg_info -qs "$@" 2> /dev/null | sed 's/[^0-9]*/*1024/' | tr '\n' + | sed 's/+$//' | bc -l
		}
		pkg_info_flavors ()
		{
			pkg_info -qe "$@" 2> /dev/null && echo
			:
		}
		pkg_info_flavored_origins ()
		{
			local glob_unflavored tmp_stdout
			glob_unflavored=$1
			pkg_info_qo "$glob_unflavored" 2> /dev/null
		}
		pkg_info_all_flavored_origins ()
		{
			pkg_info_qoa 2> /dev/null
		}
		pkg_check_sanity ()
		{
			local pkg tmp_stdout tmp_stderr
			pkg=$1
			tmp_stdout=${TMPDIR}/pkgng:pkg_check_sanity:stdout
			tmp_stderr=${TMPDIR}/pkgng:pkg_check_sanity:stderr
			pkg_info -qg "$pkg" > $tmp_stdout 2> $tmp_stderr || :
			sed -n 's/ fails the original MD5 checksum$//p' "$tmp_stdout" 2> /dev/null || :
			sed -nE "s/^pkg_info: (.*) doesn't exist$/\1/p" "$tmp_stderr" 2> /dev/null || :
		}
		pkg_which ()
		{
			local filepath
			filepath=$1
			pkg_info -qW "$filepath" 2> /dev/null
		}
		pkg_info_gen_pkg_origin_table ()
		{
			pkg_info -aE 2> /dev/null | while read pkg
			do
				origin=`pkg_info -qo "$pkg" 2> /dev/null`
				printf '%s\t%s\n' "$pkg" "$origin"
			done
		}
		pkg_create_b ()
		{
			local dev_out dev_err
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			pkg_create -b "$@" > $dev_out 2> $dev_err
		}
		pkg_delete_f ()
		{
			local opt_del dev_out dev_err
			opt_del=
			[ $opt_no_exec_inst_script = yes ] && opt_del='-D'
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			pkg_delete -f $opt_del "$@" > $dev_out 2> $dev_err
		}
		pkg_add_f ()
		{
			local opt_add dev_out dev_err
			rm -rf "${TMPDIR}/pkg_add_f"
			mkdir -p "${TMPDIR}/pkg_add_f"
			ln -s "$@" "${TMPDIR}/pkg_add_f"
			opt_add=
			[ $opt_no_exec_inst_script = yes ] && opt_add='-I'
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			( cd "${TMPDIR}/pkg_add_f" && pkg_add -if $opt_add * > $dev_out 2> $dev_err )
		}
		pkg_add_fF ()
		{
			local opt_add dev_out dev_err
			rm -rf "${TMPDIR}/pkg_add_f"
			mkdir -p "${TMPDIR}/pkg_add_f"
			ln -s "$@" "${TMPDIR}/pkg_add_f"
			opt_add=
			[ $opt_no_exec_inst_script = yes ] && opt_add='-I'
			dev_out=/dev/stdout
			dev_err=/dev/stderr
			if [ $opt_batch_mode = yes ]
			then
				dev_out=/dev/null
				dev_err=/dev/null
			fi
			( cd "${TMPDIR}/pkg_add_f" && pkg_add -ifF $opt_add * > $dev_out 2> $dev_err )
		}
		pkg_inst_verify_pkg ()
		{
			local pkg pkgarc
			pkg=$1
			pkgarc=`pkgsys_fetch_legacy_remote__fetched_pkg` || return
			pkgsys_is_dependency_of_a_legacypkg_latest "$pkgarc"
		}
		pkg_inst_remote_fetch ()
		{
			local pkg
			pkg=$1
			pkgsys_fetch_legacy_remote "$pkg"
		}
		pkg_inst_remote_verify_fetch ()
		{
			local pkg
			pkg=$1
			pkg_inst_verify_pkg "$pkg" && return
			pkg_inst_remote_fetch "$pkg" && pkg_inst_verify_pkg "$pkg"
		}
		pkg_inst_remote ()
		{
			local pkg pkgarc opt_add dev_out dev_err
			pkg=$1
			if pkg_inst_remote_verify_fetch "$pkg"
			then
				pkgarc=`pkgsys_fetch_legacy_remote__fetched_pkg`
				rm -rf "${TMPDIR}/pkg_inst_remote"
				mkdir -p "${TMPDIR}/pkg_inst_remote"
				ln -s "$@" "${TMPDIR}/pkg_add_f"
				opt_add=
				[ $opt_no_exec_inst_script = yes ] && opt_add='-I'
				dev_out=/dev/stdout
				dev_err=/dev/stderr
				if [ $opt_batch_mode = yes ]
				then
					dev_out=/dev/null
					dev_err=/dev/null
				fi
				( cd "${TMPDIR}/pkg_inst_remote" && ls -s "$pkgarc" && pkg_add -ifF $opt_add * > $dev_out 2> $dev_err )
			fi
		}
		pkg_inst_wild_verify_pkg ()
		{
			pkg_inst_verify_pkg "$@"
		}
		pkg_inst_remote_wild_fetch ()
		{
			pkg_inst_remote_fetch "$@"
		}
		pkg_inst_remote_wild_verify_fetch ()
		{
			pkg_inst_remote_verify_fetch "$@"
		}
		pkg_inst_remote_wild ()
		{
			pkg_inst_remote "$@"
		}
		pkg_get_remote_repository_version ()
		{
			local origin checksumpath version_regexp pkgbase_regexp pkg
			origin=$1
			pkgsys_ready_checksum_file > /dev/null || return
			checksumpath=`pkgsys_ready_checksum_file__fetched_file`
			version_regexp=`database_build_make "$origin" -V PORTVERSION | str_escape_regexp_filter`
			pkgbase_regexp=`database_build_get_new_pkgname "$origin" | sed -E "s/$version_regexp$//" | str_escape_regexp_filter`
			pkg=`sed -nE "s/^MD5[[:space:]]*\(($pkgbase_regexp.*)\.tbz\)[[:space:]]*=.*/\1/p" "$checksumpath" | head -n 1`
			echo "$pkg"
		}
		pkg_get_pkgs_timestamps ()
		{
			stat -t %s -f $'legacy:%m\t%N' ${PKG_DBDIR} 2> /dev/null || :
		}
		pkg_get_pkg_timestamp ()
		{
			local pkg portdb
			pkg=$1
			portdb=`echo "$pkg" | sed 's/-[0-9].*//'`
			stat -t %s -f %m "${PKG_DBDIR}/$portdb" 2> /dev/null || :
		}
		pkg_loadconf () { :; }
		pkg_rescue_tools () { :; }
		pkg_update_pkgrepository () { :; }
	fi
}

# ============= Get the unflavored part of a flavored origin =============
pkgsys_get_unflavored_origin ()
{
	local origin origin_unflavored
	origin=$1
	origin_unflavored=`echo "$origin" | sed 's/@.*$//'`
	echo "$origin_unflavored"
}

# ============= Get the port path from a flavored origin =============
pkgsys_get_portpath_from_origin ()
{
	local origin origin_unflavored
	origin=$1
	origin_unflavored=`pkgsys_get_unflavored_origin "$origin"`
	echo "${PORTSDIR}/$origin_unflavored"
}

# ============= Get the flavor from a flavored origin =============
pkgsys_get_flavor_from_origin ()
{
	local origin flavor
	origin=$1
	flavor=`echo "$origin" | sed -E 's/^[^@]*@?//'`
	echo "$flavor"
}

# ============= Compose a flavored origin name =============
pkgsys_compose_flavored_origin ()
{
	local origin flavor
	origin=$1
	flavor=$2
	origin_unflavored=`pkgsys_get_unflavored_origin "$origin"`
	[ -z "$flavor" ] || flavor=@$flavor
	echo "$origin_unflavored$flavor"
}

# ============= Check existence of a port for a flavored origin =============
pkgsys_exists_port ()
{
	local origin port_path
	origin=$1
	port_path=`pkgsys_get_portpath_from_origin "$origin"`
	[ -d "$port_path" ]
}

# ============= Get the installed package name from a flavored origin =============
pkgsys_get_installed_pkg_from_origin ()
{
	local origin origin_unflavored flavor_origin flavor_pkg
	origin=$1
	origin_unflavored=`pkgsys_get_unflavored_origin "$origin"`
	flavor_origin=`pkgsys_get_flavor_from_origin "$origin"`
	if [ -n "$flavor_origin" ]
	then
		for pkgname in `pkg_info_qO "$origin_unflavored" 2> /dev/null`
		do
			for flavor_pkg in `pkg_info_flavors "$pkgname"`
			do
				if [ "x$flavor_origin" = "x$flavor_pkg" ]
				then
					echo "$pkgname"
					return
				fi
			done
		done
	else
		pkg_info_qO "$origin_unflavored" 2> /dev/null
	fi
	:
}

# ============= Get the installed package name from glob patterns =============
pkgsys_get_installed_pkg_from_glob ()
{
	local glob regexp
	for glob in "$@"
	do
		[ -z "$glob" ] && continue
		if regexp=`expr "$glob" : ':\(.*\)'`
		then
			pkg_info_Ex "$regexp"
		else
			pkg_info_Eg "$glob"
		fi
	done | sort -u
}

# ============= Check existence of an installed package for a flavored origin =============
pkgsys_exists_from_orig ()
{
	local origin origin_unflavored flavor_origin flavor_pkg
	origin=$1
	origin_unflavored=`pkgsys_get_unflavored_origin "$origin"`
	pkg_info_eO "$origin_unflavored" 2> /dev/null || return
	flavor_origin=`pkgsys_get_flavor_from_origin "$origin"`
	for flavor_pkg in `pkg_info_flavors "$origin_unflavored"`
	do
		[ "x$flavor_origin" = "x$flavor_pkg" ] && return
	done
	false
}

# ============= Generate the package names vs origins table at the initial state =============
pkgsys_gen_init_pkg_origin_table ()
{
	[ -e "${DBDIR}/installed_ports:pkg_vs_origin.tbl" ] && return
	pkg_info_gen_pkg_origin_table > ${DBDIR}/installed_ports:pkg_vs_origin.tbl.tmp || :
	mv "${DBDIR}/installed_ports:pkg_vs_origin.tbl.tmp" "${DBDIR}/installed_ports:pkg_vs_origin.tbl"
}

# ============= Check existence of initially or currently installed package for a flavored origin =============
pkgsys_exists_or_existed_from_orig ()
{
	local origin
	origin=$1
	cut -f 2 "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null \
		| env LANG=C grep -q -Fx "$origin" || pkgsys_exists_from_orig "$origin"
}

# ============= Get the name of an initially installed package for a flavored origin =============
pkgsys_get_init_pkg_from_orig ()
{
	local origin tmppkg origin_regexp npkgs origin_unflavored flavor_origin flavor_pkg
	origin=$1
	tmppkg=${TMPDIR}/pkgsys_get_init_pkg_from_orig::pkg
	origin_regexp=`str_escape_regexp "$origin"`
	sed -n -E "/[[:space:]]$origin_regexp$/p" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null \
		| cut -f 1 > $tmppkg || :
	npkgs=`wc -l < $tmppkg`
	if [ $npkgs -gt 0 ]
	then
		cat "$tmppkg"
	else
		origin_unflavored=`pkgsys_get_unflavored_origin "$origin"`
		flavor_origin=`pkgsys_get_flavor_from_origin "$origin"`
		for pkgname in `pkg_info_qO "$origin_unflavored" 2> /dev/null`
		do
			for flavor_pkg in `pkg_info_flavors "$pkgname"`
			do
				if [ "x$flavor_origin" = "x$flavor_pkg" ]
				then
					echo "$pkgname"
					return
				fi
			done
		done
	fi
	:
}

# ============= Get the package name of this utility =============
pkgsys_get_my_current_pkg ()
{
	pkg_info_Ex "${PROGRAM}-[0-9].*"
}

# ============= Get the flavored origin of this utility =============
pkgsys_get_my_origin ()
{
	local pkgname
	for pkgname in `pkgsys_get_my_current_pkg`
	do
		pkg_info_flavored_origins "$pkgname"
	done
}

# ============= Get the flavored origin of an initially installed package by ambiguous matching =============
pkgsys_init_pkg_orig_by_ambiguous_matching ()
{
	local pkg origin tmporigin ambsuffix len_pkg pkg_regexp norigins
	pkg=$1
	tmporigin=${TMPDIR}/pkgsys_init_pkg_orig_by_ambiguous_matching::origin
	pkg_info_flavored_origins "$pkg" > $tmporigin
	[ `wc -l < $tmporigin` -gt 0 ] && { cat "$tmporigin"; return; }
	ambsuffix=
	norigins=0
	len_pkg=`echo -n "$pkg" | wc -c`
	if [ $len_pkg -gt 0 ]
	then
		while :
		do
			pkg_regexp=`str_escape_regexp "$pkg"`$ambsuffix
			env LANG=C grep -E "^${pkg_regexp}[[:space:]]" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null \
				| cut -f 2 > $tmporigin
			norigins=`wc -l < $tmporigin`
			[ $norigins -gt 0 ] && break
			ambsuffix='[a-zA-Z0-9.,_+-]*'
			len_pkg=$(($len_pkg-1))
			[ $len_pkg -gt 0 ] || break
			pkg=`echo -n "$pkg" | head -c $len_pkg`
		done
	fi
	[ $norigins -eq 1 ] || return
	cat "$tmporigin"
}

# ============= A part of message indicating tools for showing concerned issues in UPDATING =============
pkgsys_show_pkg_updating_commands ()
{
	if [ "x$PKGSYS_USE_PKGNG" = xyes ]
	then
		if which -s pkg_updating
		then
			echo 'pkg-updating(8) or pkg_updating(1)'
		else
			echo 'pkg-updating(8)'
		fi
	elif which -s pkg_updating
	then
		echo 'pkg_updating(1)'
	fi
}

# ============= Evaluation of flavored ports globs =============
pkgsys_eval_ports_glob ()
{
	local pkglist unflavored_origlist tmp_flavors
	pkglist=${DBDIR}/pkgsys_eval_ports_glob:pkg.lst
	unflavored_origlist=${DBDIR}/pkgsys_eval_ports_glob:origin_unflavored.lst
	tmp_flavors=${TMPDIR}/pkgsys_eval_ports_glob:flavors
	# Test the access privilege as the superuser or not
	if [ ! -r "$pkglist" ]
	then
		if touch "$pkglist" 2>/dev/null
		then
			rm "$pkglist"
		else
			pkglist=${TMPDIR}/pkgsys_eval_ports_glob:pkg.lst
		fi
	fi
	if [ ! -r "$unflavored_origlist" ]
	then
		if touch "$unflavored_origlist" 2>/dev/null
		then
			rm "$unflavored_origlist"
		else
			unflavored_origlist=${TMPDIR}/pkgsys_eval_ports_glob:origin.lst
		fi
	fi
	# Preparation of customized databases
	[ -f "$pkglist" ] \
		|| cut -d \| -f 1 "${PORTS_INDEX_DB}" > $pkglist
	[ -f "$unflavored_origlist" ] \
		|| cut -d \| -f 2 "${PORTS_INDEX_DB}" \
		| sed -E "s/^`str_escape_regexp "${PORTSDIR}"`\///" > $unflavored_origlist
	# Evaluation
	while [ $# -gt 0 ]
	do
		glob=$1
		shift
		expr "x$glob" : '^x-' > /dev/null 2>&1 && continue
		glob_regexp=`str_convert_portsglob_to_regexp_pattern "$glob"`
		if expr "$glob" : '.*/' > /dev/null 2>&1
		then
			if expr "$glob" : '.*@' > /dev/null 2>&1
			then
				glob_regexp_unflavored=`expr "$glob_regexp" : '\([^@]*\)' || :`$
				glob_regexp_flavor=^`expr "$glob_regexp" : '[^@]*@\([^@]*\)' || :`
					
				env LANG=C grep -E "$glob_regexp_unflavored" "$unflavored_origlist" 2>&1 | while read origin_unflavored
				do
					fs_fix_unionfs_image_if_hidden "${PORTSDIR}/$origin_unflavored"
					make -C "${PORTSDIR}/$origin_unflavored" -V FLAVORS 2> /dev/null | \
						tr '[:space:]' '\n' | env LANG=C grep -v '^$' | env LANG=C grep -E "$glob_regexp_flavor" | sed -E "s|^|$origin_unflavored@|"
				done
				{
					pkg_info_all_flavored_origins
					cut -f 2 "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null
				} | env LANG=C grep -E "$glob_regexp" 2>&1 || :
			else
				env LANG=C grep -E "$glob_regexp" "$unflavored_origlist" 2>&1 | while read origin_unflavored
				do
					origin_unflavored_rpl=`str_escape_replaceval "$origin_unflavored"`
					fs_fix_unionfs_image_if_hidden "${PORTSDIR}/$origin_unflavored"
					make -C "${PORTSDIR}/$origin_unflavored" -V FLAVORS 2> /dev/null | \
						tr '[:space:]' '\n' | env LANG=C grep -v '^$' > $tmp_flavors || :
					if [ `wc -l < $tmp_flavors` -gt 0 ]
					then
						sed -E "s|^|$origin_unflavored_rpl@|" "$tmp_flavors"
					else
						echo "$origin_unflavored"
					fi
				done
			fi
			glob_regexp_allflavors=`echo "$glob_regexp" | sed 's/$$/(|@.*)$/'`
			{
				pkg_info_all_flavored_origins
				cut -f 2 "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null
			} | env LANG=C grep -E "$glob_regexp_allflavors" 2>&1 || :
		else
			if expr "$glob" : '[a-z][a-zA-Z0-9_.+-]*[a-zA-Z0-9_.+]$' > /dev/null 2>&1 && \
				[ `expr "$glob" : '.*-[0-9]' 2>&1` -eq 0 ]
			then
				glob_regexp2=`expr "$glob_regexp" : '\(.*\)\$$' 2>&1 || :`'-[0-9]'
			else
				glob_regexp2=$glob_regexp
			fi
			env LANG=C grep -n -E "$glob_regexp2" "$pkglist" 2>&1 | cut -d : -f 1 \
				| while read index
			do
				sed -n ${index}p "$unflavored_origlist"
			done | while read origin_unflavored
			do
				origin_unflavored_rpl=`str_escape_replaceval "$origin_unflavored"`
				fs_fix_unionfs_image_if_hidden "${PORTSDIR}/$origin_unflavored"
				make -C "${PORTSDIR}/$origin_unflavored" -V FLAVORS 2> /dev/null | \
					tr '[:space:]' '\n' | env LANG=C grep -v '^$' > $tmp_flavors || :
				[ `wc -l < $tmp_flavors` -gt 0 ] || echo > $tmp_flavors
				sed -E "s/^/$origin_unflavored_rpl /" "$tmp_flavors"
			done | while read origin_unflavored flavor
			do
				if [ -n "$flavor" ]
				then
					if make -C "${PORTSDIR}/$origin_unflavored" package-name FLAVOR=$flavor 2> /dev/null | \
					env LANG=C grep -qE "$glob_regexp2"
					then
						echo "$origin_unflavored@$flavor"
					fi
				else
					if make -C "${PORTSDIR}/$origin_unflavored" package-name 2> /dev/null | \
					env LANG=C grep -qE "$glob_regexp2"
					then
						echo "$origin_unflavored"
					fi
				fi
			done || :
			glob_regexp2=`echo "$glob_regexp" | sed -E 's/\$*$//' 2>&1 || :`'[[:space:]]'
			env LANG=C grep -E "$glob_regexp2" "${DBDIR}/installed_ports:pkg_vs_origin.tbl" 2> /dev/null \
				| cut -f 2 || :
			for pkgname in `pkg_info_Ex "$glob_regexp"`
			do
				pkg_info_flavored_origins "$pkgname"
			done
		fi
	done | sort -u
}

# ============= Create a back-up package archive =============
pkgsys_create_backup_pkg ()
{
	local pkgname dstdir origin backup_pkg_old pkgname_ptn backup_pkg dbpath pkgpath
	pkgname=$1
	dstdir=$2
	rm -rf "${TMPDIR}"/package.tmp
	mkdir "${TMPDIR}"/package.tmp
	for origin in `pkg_info_flavored_origins "$pkgname"`
	do
		if backup_pkg_old=`pkgsys_get_backup_pkg "$origin"` && \
			[ ! -e "${DBDIR}/requires/$origin/installed_timestamp" -o \
				"$backup_pkg_old" -nt "${DBDIR}/requires/$origin/installed_timestamp" ]
		then
			echo "$backup_pkg_old"
			return
		fi
		if ( cd "${TMPDIR}"/package.tmp && pkg_create_b "$pkgname" > /dev/null )
		then
			pkgname_ptn=`str_escape_regexp "$pkgname"`
			backup_pkg=`ls "${TMPDIR}"/package.tmp | \
				env LANG=C grep -m 1 -E "^${pkgname_ptn}\.(txz|tbz|tgz|tar|pkg)$"` || :
		fi
		if [ -z "$backup_pkg" ]
		then
			message_echo "WARNING: Failed to create backup package for $pkgname." >&2
			return 1
		fi
		dbpath=${DBDIR}/backup/$origin
		mkdir -p "$dbpath"
		pkg_info_qL "$pkgname" > $dbpath/previously_installed_files
		mkdir -p "$dstdir"
		mv "${TMPDIR}/package.tmp/$backup_pkg" "$dstdir"
		pkgpath=$dstdir/$backup_pkg
		cat "${DBDIR}/backup_pkgarcs.lst" 2> /dev/null | \
			while read origin_cur pkgpath_cur
			do
				[ "$pkgpath_cur" = "$pkgpath" ] && continue
				if [ "$origin_cur" = "$origin" ]
				then
					rm -f "$pkgpath_cur"
				else
					printf '%s\t%s\n' "$origin_cur" "$pkgpath_cur"
				fi
			done > ${DBDIR}/backup_pkgarcs.lst.tmp
		printf '%s\t%s\n' "$origin" "$pkgpath" >> ${DBDIR}/backup_pkgarcs.lst.tmp
		mv "${DBDIR}/backup_pkgarcs.lst.tmp" "${DBDIR}/backup_pkgarcs.lst"
		echo "$pkgpath"
	done
}

# ============= Delete a back-up package archive for a flavored port origin =============
pkgsys_delete_backup_pkg ()
{
	local origin origin_regexp
	origin=$1
	origin_regexp=`str_escape_regexp "$origin"`
	env LANG=C grep -E "^${origin_regexp}[[:space:]]" "${DBDIR}/backup_pkgarcs.lst" 2> /dev/null \
		| cut -f 2 | while read pkgpath_cur
		do
			rm -f "$pkgpath_cur"
		done
	env LANG=C grep -v -E "^${origin_regexp}[[:space:]]" "${DBDIR}/backup_pkgarcs.lst" \
		2> /dev/null > ${DBDIR}/backup_pkgarcs.lst.tmp || :
	mv "${DBDIR}/backup_pkgarcs.lst.tmp" "${DBDIR}/backup_pkgarcs.lst"
}

# ============= Get an existing package archive path for a flavored port origin =============
pkgsys_get_backup_pkg ()
{
	local origin tmpnewest origin_regexp
	origin=$1
	tmpnewest=${TMPDIR}/pkgsys_get_backup_pkg::newest
	origin_regexp=`str_escape_regexp "$origin"`
	rm -f "$tmpnewest"
	env LANG=C grep -E "^${origin_regexp}[[:space:]]" "${DBDIR}/backup_pkgarcs.lst" 2> /dev/null \
		| cut -f 2 | while read pkgpath
	do
		pkgpath_newest=`cat "$tmpnewest" 2> /dev/null` || :
		[ -e "$pkgpath" ] || continue
		[ -z "$pkgpath_newest" -o "$pkgpath" -nt "$pkgpath_newest" ] || continue
		echo "$pkgpath" > $tmpnewest
	done
	cat "$tmpnewest" 2> /dev/null
}

# ============= Get a file list to be restored by the current backup package for a flavored port origin =============
pkgsys_get_restored_files_by_backup_pkg ()
{
	local origin
	origin=$1
	cat "${DBDIR}/backup/$origin/previously_installed_files" 2> /dev/null || :
}

# ============= Check whether any file match restored files by the current backup package for a flavored port origin =============
pkgsys_chk_match_to_restored_files_by_backup_pkg ()
{
	local origin filelist dbfile
	origin=$1
	filelist=$2
	dbfile=${DBDIR}/backup/$origin/previously_installed_files
	env LANG=C grep -qFx -f "$filelist" "$dbfile" 2> /dev/null
}

# ============= Get a package name from a package archive file name =============
pkgsys_pkgarc_to_pkgname ()
{
	local pkgfile_path
	pkgfile_path=$1
	basename "$pkgfile_path" | sed -E 's/\.(txz|tbz|tgz|tar|pkg)$//'
}

# ============= Get the file name of an existing package archive for a package name =============
pkgsys_pkgname_to_pkgarc ()
{
	local pkgdir pkgname pkgname_ptn pkgnode
	pkgdir=$1
	pkgname=$2
	[ -n "$pkgname" ] || return 1
	[ -d "$pkgdir" ] || return 1
	if [ "x$PKGSYS_USE_PKGNG" = xyes ]
	then
		pkgname_ptn=`str_escape_regexp "$pkgname"`
		pkgnode=`ls "$pkgdir" 2> /dev/null | env LANG=C grep -m 1 -E "^${pkgname_ptn}\.(txz|tbz|tgz|tar|pkg)$"` || :
	elif [ -e "$pkgdir/$pkgname.tbz" ]
	then
		pkgnode=$pkgname.tbz
	fi
	[ -n "$pkgnode" ] || return 1
	echo "$pkgdir/$pkgnode"
}

# ============= Get flavored port origins matching a glob pattern even if nonexistent =============
pkgsys_eval_ports_glob_even_if_nonexistent ()
{
	local glob_pattern
	glob_pattern=$1
	{
		pkgsys_eval_ports_glob "$glob_pattern" 2> /dev/null || :
		echo "$glob_pattern" | env LANG=C grep -E '^[a-z]+/[a-zA-Z0-9_.+-]+(|@[a-zA-Z0-9_.+-]+)$' || :
	} | env LANG=C grep -v -e '^$' | sort -u
}

# ============= Evaluate glob patterns and add/remove non-existing/existing ones of them to/from a file =============
pkgsys_register_evaluated_globs ()
{
	local mode listpath dirpath tmp_evaluated
	mode=$1
	listpath=$2
	shift 2
	dirpath=`dirname "$listpath"`
	tmp_evaluated=${TMPDIR}/pkgsys_register_evaluated_globs:pkgsys_eval_ports_glob
	echo "$@" | sed -E 's/[ :]+/\
/g' | env LANG=C grep -v '^$' | sort -u | while read -r glob
	do
		pkgsys_eval_ports_glob "$glob" > $tmp_evaluated
		[ `wc -l < $tmp_evaluated` -ge 1 ] || \
		{
			message_echo "WARNING: No matching ports/package glob [$glob]." >&2
			continue
		}
		cat "$tmp_evaluated"
	done | while read origin
	do
		mkdir -p "$dirpath"
		case $mode in
		remove )	fileedit_rm_a_line "$origin" "$listpath"
			;;
		add )	fileedit_add_a_line_if_new "$origin" "$listpath"
			;;
		esac
	done
}

# ============= Evaluate glob patterns for installed packages =============
pkgsys_eval_installed_pkgs_globs ()
{
	local tmp_evaluated
	tmp_evaluated=${TMPDIR}/pkgsys_eval_installed_pkgs_globs:origins
	rm -f "$tmp_evaluated"
	pkgsys_register_evaluated_globs add "$tmp_evaluated" "$@"
	[ -e "$tmp_evaluated" ] || return 0
	while read origin
	do
		pkgsys_exists_or_existed_from_orig "$origin" || echo "$origin"
	done < $tmp_evaluated
}

# ============= Get glob patterns of conflicting packages of a port =============
pkgsys_get_conflicting_pkgs_patterns ()
{
	local mode origin conflicts conflicts_makevar conflicts_config
	mode=$1
	origin=$2
	conflicts=`database_query_get_makevar_val "$origin" CONFLICTS`
	case $mode in
	build )
		conflicts_makevar=`database_query_get_makevar_val "$origin" CONFLICTS_BUILD`
		conflicts_config=`database_query_get_config_val "$origin" BUILDCONFLICT`
		;;
	install )
		conflicts_makevar=`database_query_get_makevar_val "$origin" CONFLICTS_INSTALL`
		conflicts_config=`database_query_get_config_val "$origin" INSTCONFLICT`
		;;
	*)
		conflicts_makevar=
		conflicts_config=
		;;
	esac
	echo "$conflicts $conflicts_makevar $conflicts_config" | tr ' ' '\n' | env LANG=C grep -v '^$' | sort -u
}	

# ============= Get conflicting installed packages of a port =============
pkgsys_get_conflicting_installed_pkgs ()
{
	local mode origin tmp_conflicts
	mode=$1
	origin=$2
	tmp_conflicts=${TMPDIR}/pkgsys_get_conflicting_installed_pkgs::conflicts
	pkgsys_get_conflicting_pkgs_patterns "$mode" "$origin" | while read pkg_pattern
	do
		pkgsys_get_installed_pkg_from_glob "$pkg_pattern" || :
	done > $tmp_conflicts
	cat "$tmp_conflicts"
	[ `wc -l < $tmp_conflicts` -gt 0 ]
}	

# ============= Check whether a package conflicts with a port =============
pkgsys_chk_conflict_by_a_pkg ()
{
	local mode origin pkg tmp_conflicts_ptn
	mode=$1
	origin=$2
	pkg=$3
	tmp_conflicts_ptn=${TMPDIR}/pkgsys_chk_conflict_by_a_pkg::conflicts_ptn
	pkgsys_get_conflicting_pkgs_patterns "$mode" "$origin" \
		| str_convert_glob_to_regexp_pattern > $tmp_conflicts_ptn
	echo "$pkg" | env LANG=C grep -q -E -f "$tmp_conflicts_ptn"
}	

# ============= Check whether installed files are lost or broken for a package =============
pkgsys_sanitychk_pkgcontents ()
{
	local pkg var_is_reinstall_encouraged _is_reinstall_encouraged tmp_sanity nlines iline src filename icol filename_esc pkg_owner origin
	pkg=$1
	var_is_reinstall_encouraged=$2
	tmp_sanity=${TMPDIR}/pkgsys_sanitychk_pkgcontents:sanity
	pkg_check_sanity "$pkg" > $tmp_sanity || :
	eval "$var_is_reinstall_encouraged=no"
	[ `wc -l < $tmp_sanity` -eq 0 ] && return
	nlines=`wc -l < $tmp_sanity`
	iline=1
	_is_reinstall_encouraged=no
	while [ $iline -le $nlines ]
	do
		filename=`sed -n ${iline}p "$tmp_sanity"`
		iline=$(($iline+1))
		if [ ! -e "$filename" ]
		then
			_is_reinstall_encouraged=yes
			break
		fi
		if expr "$filename" : '.*/include/.*' > /dev/null
		then
			_is_reinstall_encouraged=yes
			break
		fi
		filename_esc=`str_escape_regexp "$filename"`
		if file "$filename" | sed -E "s/^$filename_esc:[[:space:]]//" | env LANG=C grep -q '^ELF '
		then
			_is_reinstall_encouraged=yes
			break
		fi
		pkg_owner=`pkg_which "$filename"`
		if [ "$pkg" != "$pkg_owner" ]
		then
			_is_reinstall_encouraged=yes
			break
		fi
	done
	eval "$var_is_reinstall_encouraged=\$_is_reinstall_encouraged"
	for origin in `pkg_info_flavored_origins "$pkg"`
	do
		if [ $opt_batch_mode = no ]
		then
			message_echo "[$pkg ($origin)]"
			sed 's/^/  /' "$tmp_sanity"
			message_echo
		else
			pkg_replace=`str_escape_replaceval "$pkg"`
			origin_replace=`str_escape_replaceval "$origin"`
			sed "s/^/$pkg_replace\\\\$origin_replace\\\\$_is_reinstall_encouraged\\\\/" "$tmp_sanity" | tr '\\' '\t'
		fi
	done
	return 1
}

# ============= Check whether the port options database is once saved =============
pkgsys_exists_saved_port_options_timestamps ()
{
	[ -d "${DBDIR}/ls_dbdir" ]
}

# ============= Get the current all timestamp information of port options =============
pkgsys_get_current_port_options_timestamp ()
{
	local portdb_needle_regexp
	portdb_needle_regexp=`str_escape_regexp "$1"`
	{
		ls -lD %Y%m%d:%H%M%S "${PORT_DBDIR}" | if [ -n "$portdb_needle_regexp" ]
		then
			env LANG=C grep -E "[[:space:]]$portdb_needle_regexp$" || :
		else
			cat
		fi
	} 2> /dev/null | cut -w -f 6,7 | while read datetime portdb
	do
		[ -d "${PORT_DBDIR}/$portdb" ] || continue
		if [ -e "${PORT_DBDIR}/$portdb/options" ]
		then
			datetime=`ls -lD %Y%m%d:%H%M%S "${PORT_DBDIR}/$portdb/options" 2> /dev/null | cut -w -f 6`
		fi
		printf '%s\t%s\n' "$portdb" "$datetime"
	done
}

# ============= Get the saved all timestamp information of port options =============
pkgsys_get_saved_port_options_timestamps_all ()
{
	mkdir -p "${DBDIR}/ls_dbdir"
	cat "${DBDIR}/ls_dbdir/"*.log 2> /dev/null || :
}

# ============= Convert a list of port origins to port options timestamp log names =============
pkgsys_conv_portorigin_to_port_option_timestamp_logname ()
{
	sed 's|/|_|'
}

# ============= Get the file name of the port options database of a port =============
pkgsys_get_port_option_database ()
{
	local origin
	origin=$1
	if pkgsys_is_dialog4ports_used
	then
		echo "$origin" | sed 's|/|_|;s/@.*//'
	else
		database_build_make "$origin" -V UNIQUENAME
	fi
}

# ============= Save the timestamp information of port options of a port =============
pkgsys_save_port_option_timestamp ()
{
	local origin portoptlog portoptdb
	origin=$1
	portoptlog=`echo "$origin" | pkgsys_conv_portorigin_to_port_option_timestamp_logname`
	portoptdb=`pkgsys_get_port_option_database "$origin"`
	mkdir -p "${DBDIR}/ls_dbdir"
	pkgsys_get_current_port_options_timestamp "$portoptdb" > ${DBDIR}/ls_dbdir/$portoptlog.log 2> /dev/null || :
}

# ============= Get changed port options from the saved point =============
pkgsys_get_changed_port_options ()
{
	local saved_log current_log tmp_log
	saved_log=$1
	current_log=$2
	tmp_log=${TMPDIR}/pkgsys_get_changed_port_options.log
	{
		env LANG=C grep -vxF -f "$current_log" "$saved_log" || :
		env LANG=C grep -vxF -f "$saved_log" "$current_log" || :
	} | cut -w -f 1 | env LANG=C grep -v '^$' | sort -u > $tmp_log
	if pkgsys_is_dialog4ports_used
	then
		env LANG=C grep '_' "$tmp_log" || :
	else
		cat "$tmp_log"
	fi
}

# ============= Convert a list of port options database names to port globs =============
pkgsys_conv_portoptiondbs_to_globs ()
{
	if pkgsys_is_dialog4ports_used
	then
		sed 's|_|/|'
	else
		cat
	fi
}

# ============= Register nonexistent port options databases =============
pkgsys_register_list_nonexistent_portopriondb ()
{
	local dbpath
	dbpath=${DBDIR}/ls_dbdir/%NONEXISTENT%.log
	mkdir -p "${DBDIR}/ls_dbdir"
	cat > $dbpath.new
	touch  "$dbpath"
	sort -u "$dbpath.new" "$dbpath" > $dbpath.tmp
	mv "$dbpath.tmp" "$dbpath"
}

# ============= Order a list of installed packages by storage space occupied by the installed files =============
pkgsys_order_pkgs_by_size ()
{
	while read pkg
	do
		size=`pkg_info_qs "$pkg" || :`
		[ -n "$size" ] && echo "$size"$'\t'"$pkg"
	done | sort -g | cut -f 2
}
