#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Reinstallation processes -
# Copyright (C) 2013-2022 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
# The following variables are available only within reinstall_exec and functions invoked by it.
REINSTALL_PKGTAG=	# Tag indicating version changes by reinstallation of the current port
REINSTALL_ORIGPKGTAG=	# Tag indicating the flavored port origin and version changes by reinstallation of the current port
REINSTALL_CURRENTPKG=	# Currently installed package name of the current port
REINSTALL_ORIGIN=	# Flavored port origin of the current port
REINSTALL_DBNODE_DIR=	# Path of the "requires" section of the temporary database for the current port
REINSTALL_IS_CURRENTPKG_LATEST=	# Currently installed package name of the current port
REINSTALL_DBSUFFIX=	# Database suffix for the evaluation method of dependencies

# ============= Skip reinstallation by showing messages =============
reinstall_skip ()
{
	local message
	message=$1
	message_echo "($message)"
	message_target_relations "$REINSTALL_ORIGIN"
}

# ============= Check whether a package is forbidden due to conflict (only by referring to the record) =============
reinstall_quick_chk_forbidden_conflicts ()
{
	local pkg pkg_regexp_esc
	pkg=$1
	pkg_regexp_esc=`str_escape_regexp "$pkg"`
	env LANG=C grep -qE "^$pkg_regexp_esc:" "${DBDIR}/forbidden_conflicts" 2> /dev/null
}

# ============= Check whether a package is forbidden due to conflict (referring to both the record and the actual installation) =============
reinstall_chk_forbidden_conflicts ()
{
	local pkg tmp_forbidden pkg_regexp_esc
	pkg=$1
	tmp_forbidden=${TMPDIR}/reinstall_chk_forbidden_conflicts:forbidden
	pkg_regexp_esc=`str_escape_regexp "$pkg"`
	env LANG=C grep -E "^$pkg_regexp_esc:" "${DBDIR}/forbidden_conflicts" > $tmp_forbidden 2> /dev/null || return
	pkg_info_e `cut -d : -f 3 "$tmp_forbidden"`
}

# ============= Restore a package if it is temporarily deinstalled =============
reinstall_restore_if_temporarily_deinstalled ()
{
	local tmp_delete
	[ $opt_fetch_only = no -a $opt_dry_run = no ] || return 0
	[ -n "$REINSTALL_CURRENTPKG" ] && return
	tmp_list=${TMPDIR}/reinstall_restore_if_temporarily_deinstalled
	rm -rf "$tmp_list".*
	[ -e "$REINSTALL_FROMNODE_DIR/backedup_pkgfile" ] || return 0
	while read backup_pkg
	do
		[ -z "$backup_pkg" -o ! -e "$backup_pkg" ] && continue
		echo "$backup_pkg" >> $tmp_list.backedup
		pkg=`pkgsys_pkgarc_to_pkgname "$backup_pkg"`
		if reinstall_chk_forbidden_conflicts "$pkg"
		then
			echo "$backup_pkg" >> $tmp_list.avoid
		else
			echo "$backup_pkg" >> $tmp_list.restore
		fi
	done < $REINSTALL_FROMNODE_DIR/backedup_pkgfile
	[ -e "$tmp_list.backedup" ] || return 0
	message_echo "INFO: This port is temporarily deinstalled, so restoration will be attempted."
	if [ -e "$tmp_list.avoid" ]
	then
		message_echo "INFO: The following backup(s) are avoided because of conflict with installed packages."
		message_cat < $tmp_list.avoid
	fi
	if [ -e "$tmp_list.restore" ]
	then
		message_echo "INFO: The following backup(s) will be restored."
		message_cat < $tmp_list.restore
		while read backup_pkg
		do
			pkg_add_fF "$backup_pkg" || echo "$backup_pkg" >> $tmp_list.failed
		done < $tmp_list.restore
		if [ -e "$tmp_list.failed" ]
		then
			message_echo "WARNING: Failed to restore the following backup, but continuing anyway." >&2
			message_cat < $tmp_list.failed
		fi
	fi
	:
}

# ============= Skip reinstallation by showing messages if a flavored origin is in a list =============
reinstall_skip_if_in_a_list ()
{
	local message list mode
	message=$1
	list=$2
	mode=$3
	env LANG=C grep -q -Fx "$REINSTALL_ORIGIN" "${DBDIR}/$list" 2> /dev/null || return
	REINSTALL_ORIGIN_equiv=`database_query_get_equivalent_orgin "$REINSTALL_ORIGIN"`
	if [ -n "$REINSTALL_ORIGIN_equiv" ]
	then
		env LANG=C grep -q -Fx "$REINSTALL_ORIGIN_equiv" "${DBDIR}/$list" 2> /dev/null || return
	fi
	[ "x$mode" = xrestore ] && reinstall_restore_if_temporarily_deinstalled
	reinstall_skip "$message" || :
}

# ============= Get the make arguments =============
reinstall_setup_make_args ()
{
	local mode
	mode=$1
	{
		for key in LOCALBASE LINUXBASE PORT_DBDIR PORTSDIR DISTDIR PACKAGES PKGREPOSITORY
		do
			eval echo $key=\$$key
		done
		[ $opt_avoid_vulner = yes ] || echo 'DISABLE_VULNERABILITIES=yes'
		case $mode in
		anymode )
			;;
		'' | distinct )
			[ $opt_batch_ports_only = yes ] && echo 'BATCH=yes'
			[ $opt_interactive_ports_only = yes ] && echo 'INTERACTIVE=yes'
			;;
		esac
		if [ $opt_apply_default_config = yes ]
		then
			pkgsys_is_dialog4ports_used && echo 'NO_DIALOG=yes'
		fi
		dbdir=$REINSTALL_DBNODE_DIR
		[ -d "$dbdir" ] || dbdir=${DBDIR}/conf/each_port/$REINSTALL_ORIGIN
		cat "$dbdir/MARG.conf" 2> /dev/null || :
		flavor=`pkgsys_get_flavor_from_origin "$REINSTALL_ORIGIN"`
		[ -z "$flavor" ] || echo "FLAVOR=$flavor"
	} | tr '\n' ' '
}

# ============= Get the make environment variables =============
reinstall_setup_make_envs ()
{
	database_build_setup_make_envs "$REINSTALL_ORIGIN"
}

# ============= Common comand to execute make command =============
_reinstall_make_common ()
{
	local mode port_path MAKE_ARGS MAKE_ENVS
	mode=$1
	shift
	MAKE_ARGS=`reinstall_setup_make_args $mode`
	MAKE_ENVS=`reinstall_setup_make_envs`
	port_path=`pkgsys_get_portpath_from_origin "$REINSTALL_ORIGIN"`
	fs_fix_unionfs_image_if_hidden "$port_path"
	env $MAKE_ENVS make -C "$port_path" "$@" $MAKE_ARGS
}

# ============= Execute make command without restricting for BATCH or INTERACTIVE ports =============
reinstall_make_anymode ()
{
	_reinstall_make_common anymode "$@"
}

# ============= Execute make command =============
reinstall_make ()
{
	_reinstall_make_common '' "$@"
}

# ============= Error process during reinstallation =============
reinstall_errproc ()
{
	local position msg port_path
	position=$1
	msg=$2
	database_build_update_pkgname "$REINSTALL_ORIGIN" > /dev/null
	message_echo "ERROR: In $position." >&2
	message_echo "The port/package in concern is $REINSTALL_ORIGPKGTAG." >&2
	[ -n "$msg" ] && message_echo "($msg)" >&2
	message_echo >&2
	port_path=`pkgsys_get_portpath_from_origin "$REINSTALL_ORIGIN"`
	{ fs_fix_unionfs_image_if_hidden "$port_path" \
		&& pkgsys_chk_ports_tree_implementation; } || exit 1
	mkdir -p "${DBDIR}/notes/$REINSTALL_ORIGIN"
	echo "$position" > ${DBDIR}/notes/$REINSTALL_ORIGIN/note_failtre
	database_record_failure "$REINSTALL_ORIGIN" noclean
	message_report_failure_for_a_port "$REINSTALL_ORIGIN"
}

# ============= Check the latest stage for a port =============
reinstall_chk_stage ()
{
	local stagetag
	stagetag=$1
	[ -e "${DBDIR}/status.ports/$REINSTALL_ORIGIN/$stagetag" ]
}

# ============= Check completion of a stage for a port =============
reinstall_chk_stage_complete ()
{
	local stagetag
	stagetag=$1
	[ -e "${DBDIR}/status.ports/$REINSTALL_ORIGIN/complete/$stagetag" ]
}

# ============= Register the latest stage for a port =============
reinstall_register_stage ()
{
	local stagetag parentdir
	stagetag=$1
	parentdir=${DBDIR}/status.ports/$REINSTALL_ORIGIN
	mkdir -p "$parentdir"
	touch "$parentdir/$stagetag"
}

# ============= Register completion of a stage for a port =============
reinstall_register_stage_complete ()
{
	local stagetag parentdir
	stagetag=$1
	parentdir=${DBDIR}/status.ports/$REINSTALL_ORIGIN/complete
	mkdir -p "$parentdir"
	touch "$parentdir/$stagetag"
}

# ============= Deregister the latest stage for a port =============
reinstall_deregister_stage ()
{
	local stagetag
	stagetag=$1
	rm -f "${DBDIR}/status.ports/$REINSTALL_ORIGIN/$stagetag"
}

# ============= Deregister completion of a stage for a port =============
reinstall_deregister_stage_complete ()
{
	local stagetag
	stagetag=$1
	rm -f "${DBDIR}/status.ports/$REINSTALL_ORIGIN/complete/$stagetag"
}

# ============= Back up and delete conflict =============
# Package names of conflict are given via stdin. 
reinstall_backup_and_delete_conflicts ()
{
	local pkg origin origin_equiv origin_regexp_esc origin_equiv_regexp_esc backup_pkgdir backup_pkg
	message_echo "INFO: Deinstalling conflicting packages for $REINSTALL_ORIGPKGTAG."
	while read pkg
	do
		for origin in `pkg_info_flavored_origins "$pkg"`
		do
			origin_equiv=`database_query_get_equivalent_orgin "$origin"`
			message_echo "INFO: Backing up and deleting a conflict, $origin ($pkg)."
			origin_regexp_esc=`str_escape_regexp "$origin"`
			origin_equiv_regexp_esc=`str_escape_regexp "$origin_equiv"`
			if [ -d "${DBDIR}/requires/$origin" ]
			then
				backup_pkgdir=${DBDIR}/backup_packages
			else
				backup_pkgdir=${PKGREPOSITORY}
			fi
			mkdir -p "$backup_pkgdir"
			if backup_pkg=`pkgsys_get_backup_pkg "$origin"` || backup_pkg=`pkgsys_get_backup_pkg "$origin_equiv"`
			then
				message_echo "INFO: backup package already exists as $backup_pkg"
			elif ! backup_pkg=`pkgsys_create_backup_pkg "$pkg" "$backup_pkgdir"`
			then
				message_echo "WARNING: Failed to create the backup package, the conflict is kept installed." >&2
				continue
			fi
			env LANG=C grep -v -E "^${origin_regexp_esc}[[:space:]]" "${DBDIR}/deleted_conflicts" \
				> ${DBDIR}/deleted_conflicts.tmp 2> /dev/null || :
			if [ -n "$origin_equiv_regexp_esc" ]
			then
				env LANG=C grep -v -E "^${origin_equiv_regexp_esc}[[:space:]]" "${DBDIR}/deleted_conflicts.tmp" \
					> ${DBDIR}/deleted_conflicts.tmp2 2> /dev/null || :
				mv "${DBDIR}/deleted_conflicts.tmp2" "${DBDIR}/deleted_conflicts.tmp"
			fi
			printf '%s\t%s\n' "$origin" "$pkg" >> ${DBDIR}/deleted_conflicts.tmp
			[ -n "$origin_equiv" ] && printf '%s\t%s\n' "$origin_equiv" "$pkg" >> ${DBDIR}/deleted_conflicts.tmp
			mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
			pkg_delete_f "$pkg" || \
			{
				message_echo "WARNING: Failed to deinstall $pkg by $PKGSYS_CMD_PKG_DELETE." >&2
			}
		done
	done
	cat "${DBDIR}/deleted_conflicts" 2> /dev/null | sort -u > ${DBDIR}/deleted_conflicts.tmp
	mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
}

# ============= Back up and delete remaining actual install conflict (for installation by package) =============
reinstall_backup_and_delete_remaining_install_conflicts__by_pkg ()
{
	local pkg tmp_conflicts
	pkg=$1
	tmp_conflicts=${TMPDIR}/reinstall_backup_and_delete_remaining_install_conflicts__by_pkg
	message_echo "(Checking installation conflict...)"
	pkgsys_get_conflicting_installed_pkgs install "$REINSTALL_ORIGIN" > $tmp_conflicts.pkgs || :
	while read pkg_conflict
	do
		fileedit_add_a_line_if_new "$pkg_conflict:$REINSTALL_ORIGIN:$pkg" \
			"${DBDIR}/forbidden_conflicts"
	done < $tmp_conflicts.pkgs
	reinstall_backup_and_delete_conflicts < $tmp_conflicts.pkgs
}

# ============= Back up and delete remaining actual install conflict (for installation by port) =============
reinstall_backup_and_delete_remaining_install_conflicts ()
{
	local stagedir tmp_conflicts db_conflict REINSTALL_ORIGIN_equiv
	tmp_conflicts=${TMPDIR}/reinstall_backup_and_delete_remaining_install_conflicts
	db_conflict=$REINSTALL_DBNODE_DIR/possible_additional_conflict.csv
	message_echo "(Checking installation conflict...)"
	rm -rf "$db_conflict.tmp" "$tmp_conflicts".*
	stagedir=`database_query_get_makevar_val "$REINSTALL_ORIGIN" STAGEDIR`
	pkgsys_get_conflicting_installed_pkgs install "$REINSTALL_ORIGIN" > $tmp_conflicts.pkgs || :
	REINSTALL_ORIGIN_equiv=`database_query_get_equivalent_orgin "$REINSTALL_ORIGIN"`
	( set -e
		cd "$stagedir"
		find . -not -type d
	) | sed 's|^\.||' | while read filepath
	do
		[ ! -e "$filepath" ] && continue
		pkg=`pkg_which "$filepath" || :`
		[ -z "$pkg" ] && continue
		env LANG=C grep -qFx "$pkg" "$tmp_conflicts.pkgs" && continue
		for origin in `pkg_info_flavored_origins "$pkg"`
		do
			[ x"$origin" = x"$REINSTALL_ORIGIN" ] && continue
			[ x"$origin" = x"$REINSTALL_ORIGIN_equiv" ] && continue
			origin_equiv=`database_query_get_equivalent_orgin "$origin"`
			[ x"$origin_equiv" = x"$REINSTALL_ORIGIN" ] && continue
			printf '%s\t%s\n' "$pkg" "$filepath" >> $db_conflict.tmp
		done
	done
	if [ -e "$db_conflict.tmp" ]
	then
		while read pkg filepath
		do
			message_echo "INFO: Possible additional conflict with $pkg: $filepath"
		done < $db_conflict.tmp
		message_echo "INFO: The possible additional conflict packages will be escaped just in case."
		mv "$db_conflict.tmp" "$db_conflict"
		cut -f 1 "$db_conflict" >> $tmp_conflicts.pkgs
	else
		rm -f "$db_conflict.tmp" "$db_conflict"
	fi
	while read pkg_conflict
	do
		fileedit_add_a_line_if_new "$pkg_conflict:$REINSTALL_ORIGIN:$REINSTALL_NEWPKGNAME" \
			"${DBDIR}/forbidden_conflicts"
	done < $tmp_conflicts.pkgs
	reinstall_backup_and_delete_conflicts < $tmp_conflicts.pkgs
}

# ============= Remove needless possible additional conflict =============
# Use after installation of the target port and before restoration of its possible additional conflict.
reinstall_remove_needless_possible_conflict ()
{
	local db_conflict db_forbidden suffix tmp_db
	db_conflict=$REINSTALL_DBNODE_DIR/possible_additional_conflict.csv
	db_forbidden=${DBDIR}/forbidden_conflicts
	tmp_db=${TMPDIR}/reinstall_remove_needless_possible_conflict:db
	[ -e "$db_conflict" ] || return 0
	if [ ! -e "$db_forbidden" ]
	then
		rm -fr "$db_conflict"
		return
	fi
	rm -rf "$db_conflict".*
	touch $db_conflict.tmp
	while read pkg filepath
	do
		if which -s "$filepath"
		then
			echo printf '%s\t%s\n' "$pkg" "$filepath" >> $db_conflict.tmp
			message_echo "WARNING: Unregistered additional conflict with $pkg: $filepath" >&2
		else
			message_echo "INFO: The above notice of the possible additional conflict was needless: $pkg: $filepath"
		fi
	done < $db_conflict
	cut -f 1 "$db_conflict" | sort -u > $tmp_db.conflict_pkg.old
	cut -f 1 "$db_conflict.tmp" | sort -u > $tmp_db.conflict_pkg.new
	suffix=`echo ":$REINSTALL_ORIGIN:$REINSTALL_NEWPKGNAME" | str_escape_replaceval_filter`
	env LANG=C grep -vFx -f "$tmp_db.conflict_pkg.new" "$tmp_db.conflict_pkg.old" | \
		sed -E "s/$/$suffix/" > $tmp_db.conflict_pkg.needless.filter
	env LANG=C grep -vFx -f "$tmp_db.conflict_pkg.needless.filter" "$db_forbidden" > $db_forbidden.tmp || :
	mv "$db_forbidden.tmp" "$db_forbidden"
	mv "$db_conflict.tmp" "$db_conflict"
	[ `wc -l < $db_conflict` -gt 0 ] || rm "$db_conflict"
}

# ============= Restoration of backed up conflict =============
reinstall_restore_conflicts ()
{
	local origin_current tmpsrc origin_current_equiv
	origin_current=$1
	[ $opt_fetch_only = no -a $opt_dry_run = no ] || return 0
	[ -e "${DBDIR}/deleted_conflicts" ] || return 0
	tmpsrc=${TMPDIR}/reinstall_restore_conflicts::deleted_conflicts
	cp "${DBDIR}/deleted_conflicts" "$tmpsrc"
	origin_current_equiv=`database_query_get_equivalent_orgin "$origin_current"`
	while read origin pkg
	do
		pkg_regexp_esc=`str_escape_regexp "$pkg"`
		origins_init=`database_query_initial_orgins "$origin" | env LANG=C grep -vFx "$origin" || :`
		origin_replace=`echo "$origin" \
			| sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern"`
		[ "x$origin_replace" = "x$origin" ] && origin_replace=
		is_to_dereg_from_list=no
		origin_equiv=`database_query_get_equivalent_orgin "$origin"`
		if [ -n "$origin" -a \( "x$origin" = "x$origin_current" -o "x$origin" = "x$origin_current_equiv" \) ] || \
			[ -n "$origin_equiv" -a "x$origin_equiv" = "x$origin_current" ] || \
			pkg_info_e "$pkg" || pkgsys_exists_from_orig "$origin"
		then
			is_to_dereg_from_list=yes
		else
			for origin_orig in $origins_init $origin_replace
			do
				if [ -n "$origin_orig" ] && pkgsys_exists_from_orig "$origin_orig"
				then
					is_to_dereg_from_list=yes
					break
				fi
			done
		fi
		if pkgsys_chk_conflict_by_a_pkg install "$REINSTALL_ORIGIN" "$pkg"
		then
			fileedit_add_a_line_if_new "$pkg:$REINSTALL_ORIGIN:$REINSTALL_NEWPKGNAME" "${DBDIR}/forbidden_conflicts"
			is_skipped=yes
		elif reinstall_chk_forbidden_conflicts "$pkg"
		then
			is_skipped=yes
		else
			is_skipped=no
		fi
		if [ $is_skipped = yes ]
		then
			message_echo "INFO: Restoration of a conflict, $origin ($pkg), is avoided because it conflicts with installed packages."
		fi
		if [ $is_to_dereg_from_list = yes ]
		then
			pkg_current=
			origin_current=
			for origin_trial in $origin $origin_equiv $origins_init $origin_replace
			do
				pkg_trial=`pkgsys_get_installed_pkg_from_origin "$origin_trial" || :`
				[ -z "$pkg_trial" ] && continue
				pkg_current="$pkg_current, $pkg_trial"
				origin_current="$origin_current, $origin_trial"
			done
			pkg_current=`expr "$pkg_current" : ', \(.*\)'` || pkg_current=none
			origin_current=`expr "$origin_current" : ', \(.*\)'` || origin_current=none
			pkgname_msg=$pkg
			origin_msg=$origin
			[ "x$pkg_current" = "x$pkg" ] || pkgname_msg="$pkg => $pkg_current"
			[ "x$origin_current" = "x$origin" -o "x$origin_current" = "x$origin_equiv" ] || origin_msg="$origin => $origin_current"
			if [ $is_skipped = yes ]
			then
				message_echo "WARNING: Conflicting package is installed: $origin_msg ($pkgname_msg)"
			else
				message_echo "INFO: $origin_msg ($pkgname_msg) is already restored."
				env LANG=C grep -v -E "[[:space:]]$pkg_regexp_esc$" "${DBDIR}/deleted_conflicts" \
					> ${DBDIR}/deleted_conflicts.tmp || :
				mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
			fi
			continue
		fi
		[ $is_skipped = yes ] && continue
		if env LANG=C grep -q -Fx -e "$origin" -e "$origin_equiv" -e "$origin_replace" "${DBDIR}/taboo.all.list" 2> /dev/null
		then
			message_echo "INFO: Restoration of a conflict, $origin ($pkg), is avoided because it is taboo."
			continue
		fi
		message_echo "INFO: Restoring a former conflict, $origin ($pkg)."
		if ! backup_pkg=`pkgsys_get_backup_pkg "$origin"` || ! backup_pkg=`pkgsys_get_backup_pkg "$origin_equiv"`
		then
			message_echo "WARNING: No backup exists, gave up." >&2
			continue
		fi
		if pkg_add_fF "$backup_pkg"
		then
			env LANG=C grep -v -E "[[:space:]]$pkg_regexp_esc$" "${DBDIR}/deleted_conflicts" \
				> ${DBDIR}/deleted_conflicts.tmp || :
			mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
		else
			message_echo "WARNING: Failed to restore. Note that your system may experience troubles by this error." >&2
		fi
	done < $tmpsrc
}

# ============= Check whether the all non-looped requirements are installed =============
reinstall_are_requirements_ready ()
{
	[ -e "$REINSTALL_DBNODE_DIR/requirements.all.direct" ] || return 0
	while read origin
	do
		if ! pkgsys_exists_from_orig "$origin"
		then
			origin_equiv=`database_query_get_equivalent_orgin "$origin"`
			if [ -n "$origin_equiv" ] && ! pkgsys_exists_from_orig "$origin_equiv"
			then
				return 1
			fi
		fi
	done < $REINSTALL_DBNODE_DIR/requirements.all.direct
}

# ============= Get the all non-looped requirements ready for a port by restarting them if deinstalled =============
reinstall_chk_and_restore_requirements ()
{
	local tmp_restore tmp_isfailed
	tmp_restore=${TMPDIR}/reinstall_setup_requirements:restore
	tmp_isfailed=${TMPDIR}/reinstall_setup_requirements:isfailed
	cp /dev/null "$tmp_restore"
	rm -f "$tmp_isfailed"
	cat "$REINSTALL_DBNODE_DIR/requirements.all.direct" 2> /dev/null \
		| while read origin
	do
		pkgsys_exists_from_orig "$origin" && continue
		origin_equiv=`database_query_get_equivalent_orgin "$origin"`
		pkgsys_exists_from_orig "$origin_equiv" && continue
		if env LANG=C grep -q -Fx "$origin" "${DBDIR}/taboo.all.list" 2> /dev/null \
			|| { [ -n "$origin_equiv" ] &&  env LANG=C grep -q -Fx "$origin_equiv" "${DBDIR}/taboo.all.list" 2> /dev/null; }
		then
			message_echo "INFO: Restoration of a requirement [$origin] is avoided because it is set taboo."
			touch "$tmp_isfailed"
			break
		fi
		if [ -n "$origin_equiv" ]
		then
			origins_init=`{ database_query_initial_orgins "$origin"; database_query_initial_orgins "$origin_equiv"; } | env LANG=C grep -vFx -e "$origin" -e "$origin_equiv" || :`
		else
			origins_init=`{ database_query_initial_orgins "$origin"; } | env LANG=C grep -vFx "$origin" || :`
		fi
		origin_replace=`echo "$origin" \
			| sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern"`
		origin_equiv_replace=`echo "$origin_equiv" \
			| sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern"`
		[ "x$origin_replace" = "x$origin" ] && origin_replace=
		[ "x$origin_equiv_replace" = "x$origin_equiv" ] && origin_equiv_replace=
		[ -n "$origin_replace" ] || origin_replace=$origin_equiv_replace
		is_installed=no
		for origin_orig in $origins_init $origin_replace $origin_equiv_replace
		do
			[ "x$origin_orig" = "x$origin" ] && continue
			if pkgsys_exists_from_orig "$origin_orig"
			then
				is_installed=yes
				break
			fi
			if env LANG=C grep -q -Fx "$origin_orig" "${DBDIR}/taboo.all.list" 2> /dev/null
			then
				message_echo "INFO: Restoration of a requirement [$origin_orig] is avoided because it is set taboo."
				touch "$tmp_isfailed"
				is_installed=taboo
				break
			fi
		done
		[ $is_installed = yes ] && continue
		[ $is_installed = taboo ] && break
		for origin_orig in $origin $origin_equiv $origins_init $origin_replace $origin_equiv_replace
		do
			pkgarc=`pkgsys_get_backup_pkg "$origin_orig"` && break
		done
		if [ -z "$pkgarc" ]
		then
			if env LANG=C grep -q -Fx "$origin" "${DBDIR}/failed.list" 2> /dev/null || \
				{ [ -n "$origin_equiv" ] && env LANG=C grep -q -Fx "$origin_equiv" "${DBDIR}/failed.list" 2> /dev/null; }
			then
				touch "$tmp_isfailed"
				break
			fi
			continue
		fi
		printf '%s\t%s\n' "$origin_orig" "$pkgarc" >> $tmp_restore
	done
	[ -e "$tmp_isfailed" ] && return 1
	while read origin pkgarc
	do
		pkg=`pkgsys_pkgarc_to_pkgname "$pkgarc"`
		pkg_regexp_esc=`str_escape_regexp "$pkg"`
		if reinstall_chk_forbidden_conflicts "$pkg"
		then
			message_echo "INFO: Restoration of a requirement [$origin ($pkg)] is avoided because it conflicts with installed packages."
			continue
		fi
		message_echo "INFO: Restoring a backed-up requirement [$origin ($pkg)]."
		if pkg_add_fF "$pkgarc"
		then
			env LANG=C grep -v -E "[[:space:]]$pkg_regexp_esc$" "${DBDIR}/deleted_conflicts" \
				> ${DBDIR}/deleted_conflicts.tmp 2> /dev/null || :
			mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
		else
			message_echo "WARNING: Failed to restore by the backed-up package." >&2
		fi
		
	done < $tmp_restore
	:
}

# ============= Back-up of the currently installed package =============
reinstall_pkg_backup ()
{
	local backup_pkg pkg
	reinstall_chk_stage_complete PKG_BACKUP && return
	pkg=`echo "$REINSTALL_CURRENTPKG" | tr ' ' '\n' | env LANG=C grep -v '^$' | tail -n 1`
	if [ -n "$pkg" ]
	then
		message_echo "-- (Creating temporary backup package for $REINSTALL_ORIGPKGTAG)"
		if backup_pkg=`pkgsys_create_backup_pkg "$pkg" "${DBDIR}/backup_packages"`
		then
			fileedit_add_a_line_if_new "$pkg" "$REINSTALL_FROMNODE_DIR/backedup_version"
			fileedit_add_a_line_if_new "$backup_pkg" "$REINSTALL_FROMNODE_DIR/backedup_pkgfile"
		else
			message_echo "WARNING: Failed to create the backup package, but ignored by hoping success." >&2
			return 1
		fi
	fi
	reinstall_register_stage_complete PKG_BACKUP
}

# ============= Deinstallation of the currently installed package =============
# For a flavored package, its unflavored one is also deinstalled if existing.
# For an unflavored package, its all flavored ones are also deinstalled if existing.
reinstall_deinstall ()
{
	local tmp_installedpkg REINSTALL_ORIGIN_equiv installed_pkgs dev_out dev_err flavor
	dev_out=/dev/stdout
	dev_err=/dev/stderr
	if [ $opt_batch_mode = yes ]
	then
		dev_out=/dev/null
		dev_err=/dev/null
	fi
	tmp_installedpkg=${TMPDIR}/reinstall_deinstall:installedpkg
	pkgsys_get_installed_pkg_from_origin "$REINSTALL_ORIGIN" > $tmp_installedpkg
	[ `wc -l < $tmp_installedpkg` -gt 0 ] || return 0
	installed_pkgs=`tr '\n' ' ' < $tmp_installedpkg | sed 's/ *$//'`
	REINSTALL_ORIGIN_equiv=`database_query_get_equivalent_orgin "$REINSTALL_ORIGIN"`
	if pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN" || pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN_equiv"
	then
		message_echo "INFO: Deinstalling $installed_pkgs by $PKGSYS_CMD_PKG_DELETE."
		pkg_delete_f $installed_pkgs || \
		{
			message_echo "WARNING: Failed to deinstall." >&2
		}
	else
		flavor=`pkgsys_get_flavor_from_origin "$REINSTALL_ORIGIN"`
		if [ -n "$flavor" ]
		then
			pkgsys_get_unflavored_origin "$REINSTALL_ORIGIN" >> $tmp_installedpkg
		else
		fi
		while read pkg
		do
			message_echo "INFO: Deinstalling $pkg by $PKGSYS_CMD_PKG_DELETE." >&2
			pkg_delete_f "$pkg" || \
			{
				message_echo "WARNING: Failed to deinstall." >&2
			}
		done < $tmp_installedpkg
	fi
	message_echo "-- (Trying to deinstall by ports to make sure. This may cause negligible warnings.)"
	reinstall_make deinstall > $dev_out 2> $dev_err|| \
	{
		message_echo "WARNING: Failed to deinstall $REINSTALL_CURRENTPKG by make deinstall." >&2
	}
}

# ============= Deinstallation of installed packages for old ports of the current one =============
reinstall_deinstall_old_ports ()
{
	[ -e "$REINSTALL_FROMNODE_DIR/old_origs" ] || return 0
	while read origin_old
	do
		pkgsys_get_installed_pkg_from_origin "$origin_old"
		origin_old_equiv=`database_query_get_equivalent_orgin "$origin_old"`
		pkgsys_get_installed_pkg_from_origin "$origin_old_equiv"
	done < $REINSTALL_FROMNODE_DIR/old_origs | reinstall_backup_and_delete_conflicts
}

# ============= Recovery after failure of installation of the new package =============
reinstall_failed_install_recover ()
{
	local backedup_version backup_pkg dev_out dev_err REINSTALL_ORIGIN_equiv
	dev_out=/dev/stdout
	dev_err=/dev/stderr
	if [ $opt_batch_mode = yes ]
	then
		dev_out=/dev/null
		dev_err=/dev/null
	fi
	reinstall_chk_stage_complete FAILED_INSTALL.RECOVER && return
	message_echo "INFO: Trying to deinstall the failed/terminated installation (Ignore failures)."
	if [ -n "$REINSTALL_CURRENTPKG" ]
	then
		pkg_delete_f "$REINSTALL_CURRENTPKG" || :
	fi
	message_echo "INFO: Trying to deinstall by ports to make sure (This may cause negligible warnings)."
	reinstall_make deinstall > $dev_out 2> $dev_err|| :
	backedup_version=`cat "$REINSTALL_FROMNODE_DIR/backedup_version" 2> /dev/null || :`
	REINSTALL_ORIGIN_equiv=`database_query_get_equivalent_orgin "$REINSTALL_ORIGIN"`
	if env LANG=C grep -q -Fx "$REINSTALL_ORIGIN" "${DBDIR}/taboo.all.list" 2> /dev/null || \
		{ [ -n "$REINSTALL_ORIGIN_equiv" ]  && env LANG=C grep -q -Fx "$REINSTALL_ORIGIN_equiv" "${DBDIR}/taboo.all.list" 2> /dev/null; }
	then
		message_echo "INFO: Restoration of the backup of $backedup_version is avoided because it is taboo."
	elif [ -n "$backedup_version" ]
	then
		if reinstall_chk_forbidden_conflicts "$backedup_version"
		then
			message_echo "INFO: Restoration of the backup of $backedup_version, is avoided because it conflicts with installed packages."
		else
			message_echo "INFO: Restoring the backup of $backedup_version."
			if [ -e "$REINSTALL_FROMNODE_DIR/backedup_pkgfile" ]
			then
				while read backup_pkg
				do
					if [ ! -e "$backup_pkg" ]
					then
						message_echo "WARNING: The backup file $backup_pkg doesn't exist, gave up." >&2
					elif ! pkg_add_fF "$backup_pkg"
					then
						message_echo "WARNING: Failed to restore $backedup_version. Note that your system may experience troubles by this error." >&2
					fi
				done < $REINSTALL_FROMNODE_DIR/backedup_pkgfile
			else
				message_echo "WARNING: No backup was saved, gave up." >&2
			fi
		fi
	fi
	reinstall_register_stage_complete FAILED_INSTALL.RECOVER
}

# ============= Report an installation success to the all dependents =============
reinstall_tell_update_to_depandents ()
{
	local REINSTALL_ORIGIN_equiv tag level dbsuffix
	pkgsys_is_pkgtool "$REINSTALL_ORIGIN" && return
	REINSTALL_ORIGIN_equiv=`database_query_get_equivalent_orgin "$REINSTALL_ORIGIN"`
	pkgsys_is_pkgtool "$REINSTALL_ORIGIN_equiv" && return
	reinstall_chk_stage_complete TELL_UPDATE_TO_DEPANDENTSL && return
	for tag in all run build none
	do
		for level in full direct
		do
			dbsuffix=${tag}.${level}
			{
				cat "$REINSTALL_DBNODE_DIR/dependents.$dbsuffix" "$REINSTALL_DBNODE_DIR/dependents.$dbsuffix.orig"
				[ -e "$REINSTALL_DBNODE_DIR/succeeded_once" ] || cat "$REINSTALL_DBNODE_DIR/ignored_dependents.$tag" "$REINSTALL_DBNODE_DIR/ignored_dependents.$tag.orig"
			} 2> /dev/null \
				| sort -u \
				| while read origin_dependent
			do
				echo "$origin_dependent"
				origin_dependent_equiv=`database_query_get_equivalent_orgin "$origin_dependent"`
				[ -z "$origin_dependent_equiv" ] || echo "$origin_dependent_equiv"
			done | while read origin_dependent
			do
				[ -d "${DBDIR}/requires/$origin_dependent" ] || continue
				touch "${DBDIR}/requires/$origin_dependent/need_reinstall_due_to_upgraded_requirements.$dbsuffix"
				fileedit_rm_a_line "$origin_dependent" \
					"${DBDIR}/success.$dbsuffix.list"
				fileedit_add_a_line_if_new "$origin_dependent" \
					"${DBDIR}/todo_after_requirements_succeed.$dbsuffix.list"
			done
		done
	done
	reinstall_register_stage_complete TELL_UPDATE_TO_DEPANDENTS
}

# ============= Closing operations after a successful installation =============
reinstall_closing_operations_after_successful_install ()
{
	local tag level
	reinstall_chk_stage_complete CLOSING_OPERATIONS_AFTER_SUCCESSFUL_INSTALL && return
	database_build_update_pkgname "$REINSTALL_ORIGIN" > /dev/null
	database_build_update_pkgtag "$REINSTALL_ORIGIN"
	for tag in all run build none
	do
		for level in full direct
		do
			rm -f "$REINSTALL_DBNODE_DIR/need_reinstall_due_to_upgraded_requirements.${tag}.${level}"
			[ -e "$REINSTALL_DBNODE_DIR/failed_requirements.${tag}.${level}" ] || continue
			cp "$REINSTALL_DBNODE_DIR/failed_requirements.${tag}.${level}" \
				"$REINSTALL_DBNODE_DIR/failed_requirements.${tag}.${level}.previous"
		done
	done
	{
		echo "$REINSTALL_ORIGIN"
		database_query_initial_orgins "$REINSTALL_ORIGIN"
	} | sort -u | while read initial_orig
	do
		initial_orig_regexp=`str_escape_regexp "$initial_orig"`
		initial_orig_equiv=`database_query_get_equivalent_orgin "$initial_orig"`
		initial_orig_equiv_regexp=`str_escape_regexp "$initial_orig_equiv"`
		env LANG=C grep -E "^${initial_orig_regexp}[[:space:]]" "${DBDIR}/deleted_conflicts" 2> /dev/null \
			| cut -f 2 | while read initial_pkg
		do
			pkg_regexp=`str_escape_regexp "$initial_pkg"`
			env LANG=C grep -v -E "^${pkg_regexp}:" "${DBDIR}/forbidden_conflicts" \
				> ${DBDIR}/forbidden_conflicts.tmp 2> /dev/null || :
			mv "${DBDIR}/forbidden_conflicts.tmp" "${DBDIR}/forbidden_conflicts"
		done
		env LANG=C grep -v -E "^${initial_orig_regexp}[[:space:]]" "${DBDIR}/deleted_conflicts" \
			> ${DBDIR}/deleted_conflicts.tmp 2> /dev/null || :
		mv "${DBDIR}/deleted_conflicts.tmp" "${DBDIR}/deleted_conflicts"
		pkgsys_delete_backup_pkg "$initial_orig"
	done
	rm -f "$REINSTALL_FROMNODE_DIR/backedup_pkgfile"
	database_record_success "$REINSTALL_ORIGIN"
	message_echo "===>  Done successfully"
	reinstall_register_stage_complete CLOSING_OPERATIONS_AFTER_SUCCESSFUL_INSTALL
}

# ============= Fetch missing distfiles of a port without sanity check =============
reinstall_fetch_missing_distfiles ()
{
	local port_path tmp_fetch
	port_path=`pkgsys_get_portpath_from_origin "$REINSTALL_ORIGIN"` || return
	tmp_fetch=${TMPDIR}/reinstall_fetch_missing_distfiles:fetch.sh
	tmp_missing_fetch=${TMPDIR}/reinstall_fetch_missing_distfiles:missing_fetch.sh
	rm -rf "$tmp_fetch.pre"
	reinstall_make_anymode fetch-list 2> /dev/null > $tmp_fetch.src || return
	env LANG=C grep '^SHA256[[:space:]]' "$port_path/distinfo" | \
		sed -E 's/^SHA256[[:space:]]+\(([^)]*)\).*/\1/' | \
		while read relative_distfile_path
	do
		if ! { testhash=`( cd "${DISTDIR}" && sha256 "$relative_distfile_path" ) 2> /dev/null` && \
			env LANG=C grep -qxF "$testhash" "$port_path/distinfo"; }
		then
			relative_distfile_path_ptn=`str_escape_regexp "|| echo \"$relative_distfile_path\" not fetched; }"`
			if env LANG=C grep -E "$relative_distfile_path_ptn$" "$tmp_fetch.src"
			then
				subdir_distfile=`dirname "$relative_distfile_path"`
				[ "x$subdir_distfile" = x. ] || echo "mkdir -p \"${DISTDIR}/$subdir_distfile\"" >> $tmp_fetch.pre
			fi
		fi
	done | env LANG=C grep -v '^[[:space:]]*$' > $tmp_fetch.main || :
	[ -e "$tmp_fetch.pre" ] && sort -u "$tmp_fetch.pre"
	cat "$tmp_fetch.main"
}

# ============= Execute a command by recording the standard output and error output into a file in case of error =============
reinstall_execcmd_tee_errlog ()
{
	local func tmp_err
	func=$1
	tmp_err=${TMPDIR}/reinstall_execcmd_tee_errlog::error
	rm -f "$tmp_err"
	if [ $opt_batch_mode = yes ]
	then
		{ {
			$func
		} 2>&1 || touch "$tmp_err"; } > "$REINSTALL_DBNODE_DIR/error.log"
	else
		{ {
			$func
		} 2>&1 || touch "$tmp_err"; } | tee "$REINSTALL_DBNODE_DIR/error.log"
	fi
	[ -e "$tmp_err" ] && return 1
	rm -f "$REINSTALL_DBNODE_DIR/error.log"
	return
}

# ============= Execute a command to output to stdout by recording the error output into a file in case of error =============
reinstall_execcmd_getstdout_errlog ()
{
	local func
	func=$1
	if $func 2> $REINSTALL_DBNODE_DIR/error.log
	then
		rm -f "$REINSTALL_DBNODE_DIR/error.log"
		return
	fi
	cat "$REINSTALL_DBNODE_DIR/error.log" >&2
	return 1
}

# ============= Check whether any required package is missing or too old for build by ports =============
# Return status 0 for no problem about missing ports
reinstall_chk_missing_requirement ()
{
	local tmp_filter tmp_missing
	tmp_filter=${TMPDIR}/reinstall_chk_missing_requirement::requirements.all_but_test.unflavored
	tmp_missing=${TMPDIR}/reinstall_chk_missing_requirement::missing_requirements
	message_echo "Checking whether any required package is missing or too old..."
	sed 's/@.*//' "$REINSTALL_DBNODE_DIR/requirements.all.full" > $tmp_filter || :
	reinstall_make missing 2> /dev/null | env LANG=C grep -Fx -f "$tmp_filter" | while read missing_orig
	do
		missing_equiv=`database_query_get_equivalent_orgin "$missing_orig"`
		[ -n "$missing_equiv" ] && ! pkg_info_eO "$missing_equiv" && echo "$missing_orig"
	done > $tmp_missing
	if [ `wc -l < $tmp_missing` -gt 0 ]
	then
		message_echo "Found missing/too old requirements:"
		message_cat < $tmp_missing
		{
			echo "Found missing/too old requirements:"
			cat "$tmp_missing"
		} > "$REINSTALL_DBNODE_DIR/error.log"
		reinstall_restore_conflicts "$REINSTALL_ORIGIN"
		reinstall_restore_if_temporarily_deinstalled
		for tag in all run build none
		do
			for level in full direct
			do
				fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" \
					"${DBDIR}/todo_after_requirements_succeed.${tag}.${level}.list"
			done
		done
		fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/failed.list"
		reinstall_skip 'Skipped because of missing/too old packages'
		return 1
	fi
}

# ============= Check whether the current port is marked to skip =============
# Return status 1 for port to skip
reinstall_exec_chk_skip ()
{
	local tag level tmpdb
	local currentorigin_is_all currentorigin_is_target currentorigin_is_requires_requirements
	local currentorigin_is_initial_requirements currentorigin_is_requires_dependents
	local currentorigin_is_initial_dependents currentorigin_is_requires_requirements_complement
	local currentorigin_is_relevant
	tmpdb=${TMPDIR}/reinstall_exec_chk_skip
	database_query_get_target_attributes currentorigin "$REINSTALL_ORIGIN"
	if [ -z "${currentorigin_is_all}" -a -z "${currentorigin_is_relevant}" ]
	then
		reinstall_skip 'Skipped because being irrelevant'
		return 1
	fi
	reinstall_skip_if_in_a_list 'Skipped because being a leaf port' leaf_ports_to_delete.selected && return 1
	rm -f "$REINSTALL_DBNODE_DIR/this_is_skipped_build_requirement"
	if expr "$REINSTALL_CURRENTPKG" : '.* .*' > /dev/null
	then
		message_echo "WARNING: Multiple packages are registered for this port. Reinstallation is needed to fix it." >&2
	elif env LANG=C grep -q -Fx "$REINSTALL_ORIGIN" "${DBDIR}/damaged_package" 2> /dev/null
	then
		message_echo "WARNING: Installed files have invalid checksums for this port. Reinstallation is needed to fix it." >&2
	else
		if [ -e "${DBDIR}/target_all" ]
		then
			if [ $REINSTALL_IS_CURRENTPKG_LATEST = yes ]
			then
				reinstall_restore_if_temporarily_deinstalled
				reinstall_skip 'Skipped because being already latest'
				return 1
			fi
		elif [ ! -e "$REINSTALL_DBNODE_DIR/conf_updated" ]
		then
			if [ -e "$REINSTALL_DBNODE_DIR/installed_by_pkg" ] && database_query_is_default_conf "$REINSTALL_ORIGIN" quiet
			then
				reinstall_restore_if_temporarily_deinstalled
				reinstall_skip 'Skipped because already upgraded with a prebuilt package'
				return 1
			fi
			if [ -e "$REINSTALL_DBNODE_DIR/installed_by_freezing" ]
			then
				reinstall_restore_if_temporarily_deinstalled
				reinstall_skip 'Skipped because the reinstallation has been deemed already completed by freezing'
				return 1
			fi
			reinstall_skip_if_in_a_list 'Skipped because the reinstallation has been already completed' \
				"success.${REINSTALL_DBSUFFIX}.list" restore && return 1
			if [ $opt_skip_unchanged = yes ]
			then
				if [ ! -e "$REINSTALL_DBNODE_DIR/necessary_upgrade.${REINSTALL_DBSUFFIX}" ]
				then
					reinstall_restore_if_temporarily_deinstalled
					if [ -e "$REINSTALL_FROMNODE_DIR/installed_version" ]
					then
						reinstall_skip 'Skipped because being already latest as well as the all requirements'
					else
						touch "$REINSTALL_DBNODE_DIR/this_is_skipped_build_requirement"
						reinstall_skip 'Skipped because being an only-build-time dependency of already latest packages'
					fi
					return 1
				fi
			fi
			if [ \( $opt_skip_unchanged = no -a -e "$REINSTALL_DBNODE_DIR/succeeded_once" \) \
				-o \( $opt_skip_unchanged = yes -a $REINSTALL_IS_CURRENTPKG_LATEST = yes \) ]
			then
				if [ ! -e "$REINSTALL_DBNODE_DIR/need_reinstall_due_to_upgraded_requirements.${REINSTALL_DBSUFFIX}" ]
				then
					if ! database_query_is_necessary_upgrade "$REINSTALL_ORIGIN"
					then
						reinstall_restore_if_temporarily_deinstalled
						fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" \
							"${DBDIR}/todo_after_requirements_succeed.${REINSTALL_DBSUFFIX}.list"
						reinstall_skip 'Skipped because being already latest or failed as well as the all requirements'
						return 1
					fi
				fi
			fi
		fi
		reinstall_skip_if_in_a_list 'Marked to be manually-done' manually_done.list restore && return 1
	fi
	reinstall_skip_if_in_a_list 'Skipped because being a hold package' conf/HOLD:PORTS.parsed restore && return 1
	if database_query_is_a_port_suppressed "$REINSTALL_ORIGIN"
	then
		reinstall_restore_if_temporarily_deinstalled
		reinstall_skip 'Skipped because being suppressed'
		return 1
	fi
	reinstall_skip_if_in_a_list 'Ignored because being taboo' taboo.all.list && return 1
	if [ $opt_fetch_only = no ] && \
		! reinstall_are_requirements_ready && \
		! reinstall_chk_and_restore_requirements
	then
		reinstall_restore_if_temporarily_deinstalled
		for tag in all run build none
		do
			for level in full direct
			do
				fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" \
					"${DBDIR}/todo_after_requirements_succeed.${tag}.${level}.list"
			done
		done
		reinstall_skip 'Skipped because the requirements cannot be ready'
		return 1
	fi
}

# ============= Reinstallation of the current origin: Check sanity =============
# Return status 0 for no sanity problem
reinstall_exec_reinstall_check_sanity ()
{
	local check_sanity_msg func_pkg_inst_remote_verify_fetch func_pkg_inst_verify_pkg tmp_fetch_dir tmp_fetch_missing
	if ! reinstall_chk_stage_complete CHECK_SANITY
	then
		message_echo "Sanity Check..."
		_reinstall_exec__tmpcmd () { reinstall_make_anymode check-sanity; }
		if ! check_sanity_msg=`reinstall_execcmd_getstdout_errlog _reinstall_exec__tmpcmd`
		then
			echo "$check_sanity_msg"
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'check sanity'
			if [ $opt_fetch_only = yes ]
			then
				if [ $opt_inst_by_pkg_if_can = yes ]
				then
					if ! reinstall_chk_stage_complete FAILOVER_FETCH_PKG && \
						database_query_is_default_conf "$REINSTALL_ORIGIN"
					then
						message_echo "INFO: The configuration of this port is default, so use of a prebuilt package is attempted."
						func_pkg_inst_remote_verify_fetch=pkg_inst_remote_verify_fetch
						func_pkg_inst_verify_pkg=pkg_inst_verify_pkg
						if [ $opt_use_legacy_pkg_for_missing_pkgng = yes ]
						then
							func_pkg_inst_remote_verify_fetch=pkg_inst_remote_wild_verify_fetch
							func_pkg_inst_verify_pkg=pkg_inst_wild_verify_pkg
						fi
						if ! $func_pkg_inst_verify_pkg "$REINSTALL_NEWPKGNAME"
						then
							message_echo "Trying to fetch the package because of the fetch only mode..."
							if $func_pkg_inst_remote_verify_fetch "$REINSTALL_NEWPKGNAME"
							then
								reinstall_register_stage_complete FAILOVER_FETCH_PKG
							else
								message_echo "WARNING: Failed to fetch package for $REINSTALL_NEWPKGNAME"
							fi
						fi
					fi
				fi
				if ! reinstall_chk_stage_complete FAILOVER_FETCH
				then
					tmp_fetch_missing=${TMPDIR}/reinstall_exec:fetch.sh
					reinstall_fetch_missing_distfiles > $tmp_fetch_missing
					if [ `wc -l < $tmp_fetch_missing` -gt 0 ]
					then
						message_echo "Trying to fetch the distfiles because of the fetch only mode..."
						tmp_fetch_dir=${TMPDIR}/reinstall_exec:fetch_dir
						rm -rf "$tmp_fetch_dir"
						mkdir -p "$tmp_fetch_dir"
						if ( cd "$tmp_fetch_dir" && sh "$tmp_fetch_missing" )
						then
							reinstall_register_stage_complete FAILOVER_FETCH
						else
							message_echo "WARNING: Failed to fetch distfiles"
						fi
					fi
				fi
				message_fetch_only
			fi
			return 1
		fi
		message_echo
		reinstall_register_stage_complete CHECK_SANITY
	fi
}

# ============= Reinstallation of the current origin: Installation by package =============
# Return status 0 for success
reinstall_exec_reinstall_by_pkg ()
{
	local pkg func_pkg_inst_remote_verify_fetch func_pkg_inst_remote
	if [ $REINSTALL_IS_FROZEN = yes ] || [ $opt_inst_by_pkg_if_can = yes ] && database_query_is_default_conf "$REINSTALL_ORIGIN"
	then
		if [ $REINSTALL_IS_FROZEN = yes ]
		then
			pkg=`pkg_get_remote_repository_version "$REINSTALL_ORIGIN" 2> /dev/null || :`
			if [ -z "$pkg" ]
			then
				message_echo "WARNING: No package for $REINSTALL_ORIGIN is found in the repository." >&2
			elif [ "$pkg" != "$REINSTALL_NEWPKGNAME" ]
			then
				message_echo "WARNING: (For freezing) A different package version ($pkg) is found for $REINSTALL_ORIGIN in the repository." >&2
			fi
		else
			pkg=$REINSTALL_NEWPKGNAME
		fi
		message_echo "INFO: The configuration of this port is default, so use of a prebuilt package is attempted."
		if reinstall_chk_stage in_bypkg
		then
			message_echo "(Restarting the previously terminated (re)installation-by-package process...)"
		else
			reinstall_register_stage in_bypkg
		fi
		if [ -n "$pkg" ] && [ "$pkg" = "$REINSTALL_NEWPKGNAME" ] && ! reinstall_chk_stage_complete INSTALL_BY_PKG
		then
			if ! reinstall_chk_stage FAILED_INSTALL_BY_PKG
			then
				if [ $opt_fetch_only = no -a "x$PKGSYS_USE_PKGNG" = xyes ] \
					&& pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN"
				then
					fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_default_conf.list"
					fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_custom_conf.list"
					fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/installation_complete_by_pkg.list"
					if reinstall_deinstall && pkg_is_tool_available && pkg_info_e "$pkg"
					then
						reinstall_register_stage_complete INSTALL_BY_PKG
					else
						reinstall_register_stage FAILED_INSTALL_BY_PKG
					fi
				else
					func_pkg_inst_remote_verify_fetch=pkg_inst_remote_verify_fetch
					func_pkg_inst_remote=pkg_inst_remote
					if [ $opt_use_legacy_pkg_for_missing_pkgng = yes ]
					then
						func_pkg_inst_remote_verify_fetch=pkg_inst_remote_wild_verify_fetch
						func_pkg_inst_remote=pkg_inst_remote_wild
					fi
					if $func_pkg_inst_remote_verify_fetch "$pkg"
					then
						if [ $opt_fetch_only = yes ]
						then
							message_fetch_only
						else
							fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_default_conf.list"
							fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_custom_conf.list"
							fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/installation_complete_by_pkg.list"
							reinstall_backup_and_delete_remaining_install_conflicts__by_pkg "$pkg"
							reinstall_pkg_backup || :
							reinstall_register_stage in_add_pkg
							reinstall_deinstall_old_ports
							reinstall_deinstall
							if $func_pkg_inst_remote "$pkg"
							then
								if database_query_dependency_matching "$REINSTALL_ORIGIN"
								then
									reinstall_register_stage_complete INSTALL_BY_PKG
								else
									message_echo "INFO: The requirements of the package mismatch the configuration."
									message_echo "INFO: The installed package will be deleted and installation by port will be attempted instead."
									pkg_delete_f "$pkg" || :
									reinstall_register_stage FAILED_INSTALL_BY_PKG
								fi
							else
								reinstall_register_stage FAILED_INSTALL_BY_PKG
							fi
							if pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN"
							then
								pkg_is_tool_available || pkg_rescue_tools
							fi
						fi
					else
						reinstall_register_stage FAILED_INSTALL_BY_PKG
					fi
				fi
			fi
			if [ $opt_fetch_only = no ]
			then
				if reinstall_chk_stage FAILED_INSTALL_BY_PKG && \
					reinstall_chk_stage in_add_pkg
				then
					reinstall_failed_install_recover
				fi
			fi
		fi
		if [ $opt_fetch_only = no ] && reinstall_chk_stage_complete INSTALL_BY_PKG
		then
			touch "$REINSTALL_DBNODE_DIR/installed_timestamp"
			touch "$REINSTALL_DBNODE_DIR/installed_by_pkg"
			fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" "${DBDIR}/installation_complete_by_pkg.list"
			fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_custom_conf.list"
			fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_default_conf.list"
			reinstall_deregister_stage in_bypkg
			reinstall_tell_update_to_depandents
			reinstall_closing_operations_after_successful_install
			return
		fi
		reinstall_deregister_stage in_bypkg
		if [ $opt_fetch_only = yes ]
		then
			message_echo "INFO: Continue to fetch distfiles in case of installation by port."
		elif [ $REINSTALL_IS_FROZEN = yes ]
		then
			message_echo "WARNING: Correct (re)installation-by-package is unsuccessful, so retrying to freeze." >&2
		else
			message_echo "WARNING: (Re)installation-by-package is unsuccessful, so retrying by using port." >&2
		fi
	fi
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/installation_complete_by_pkg.list"
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_custom_conf.list"
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_default_conf.list"
	return 1
}

# ============= Reinstallation of the current origin: Freeze the port =============
# Return status 0 for no need of freezing
reinstall_exec_reinstall_freeze_if_necessary ()
{
	local pkg pkg_current
	message_echo "(Port to freeze)"
	_reinstall_exec__tmpcmd () { pkg_get_remote_repository_version "$REINSTALL_ORIGIN"; }
	if ! pkg=`reinstall_execcmd_getstdout_errlog _reinstall_exec__tmpcmd` || [ -z "$pkg" ]
	then
		reinstall_restore_conflicts "$REINSTALL_ORIGIN"
		reinstall_restore_if_temporarily_deinstalled
		reinstall_errproc 'freezing'
		reinstall_skip "ERROR: Failed to get the repository version."
		return 1
	fi
	pkg_current=`pkgsys_get_installed_pkg_from_origin "$REINSTALL_ORIGIN"`
	if [ "x$pkg" = "x$pkg_current" ]
	then
		message_echo "INFO: The latest repository version $pkg installed, deemed success."
		message_echo "WARNING: This action may cause problems due to the version/option mismatch." >&2
		reinstall_tell_update_to_depandents
		touch "$REINSTALL_DBNODE_DIR/installed_by_freezing"
		reinstall_closing_operations_after_successful_install
		return
	fi
	message_echo "INFO: The latest repository version $pkg will be installed."
	reinstall_pkg_backup || :
	reinstall_deinstall_old_ports
	reinstall_deinstall
	reinstall_backup_and_delete_remaining_install_conflicts__by_pkg "$pkg"
	_reinstall_exec__tmpcmd () { pkg_inst_remote "$pkg"; }
	if reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
	then
		message_echo "INFO: Deemed success."
		message_echo "WARNING: This action may cause problems due to the version/option mismatch." >&2
		reinstall_tell_update_to_depandents
		touch "$REINSTALL_DBNODE_DIR/installed_by_freezing"
		reinstall_closing_operations_after_successful_install
		return
	fi
	message_echo "ERROR: Failed install the version in the repository for a port to freeze. Dependents are locked." >&2
	reinstall_restore_conflicts "$REINSTALL_ORIGIN"
	reinstall_restore_if_temporarily_deinstalled
	reinstall_errproc 'freezing'
	reinstall_skip 'Skipped because being a port to freeze.'
	return 1
}

# ============= Reinstallation of the current origin: Avoid build if any requirement port to freeze is unfrozen =============
# Return status 0 for no concern about freezing requirements
reinstall_exec_reinstall_avoid_if_any_unfrozen_requirements_exists ()
{
	if env LANG=C grep -qFx "$REINSTALL_DBNODE_DIR/requirements.all.full" "${DBDIR}/freeze.all.list" 2> /dev/null
	then
		reinstall_restore_conflicts
		{
			pkgsys_get_conflicting_installed_pkgs build "$REINSTALL_ORIGIN" || :
			pkgsys_get_conflicting_installed_pkgs install "$REINSTALL_ORIGIN" || :
		} | reinstall_backup_and_delete_conflicts
		if ! database_query_are_requirements_not_locked "$REINSTALL_ORIGIN"
		then
			reinstall_skip 'Skipped because of missing requirements to freeze.'
			return 1
		fi
	fi
}

# ============= Reinstallation of the current origin: Prebuild process =============
# Return status 1 for error end
reinstall_exec_reinstall_prebuild ()
{
	if [ $opt_fetch_only = no ]
	then
		if reinstall_chk_stage in_prebuild
		then
			message_echo "(Restarting the previously terminated pre-build process...)"
		else
			reinstall_restore_conflicts
			reinstall_register_stage in_prebuild
		fi
	fi
	if [ $opt_fetch_only = no  -a -e "$REINSTALL_DBNODE_DIR/BEFOREBUILD.conf" ] && \
		! reinstall_chk_stage_complete BEFOREBUILD
	then
		message_echo "-- BEFOREBUILD operations (start)"
		_reinstall_exec__tmpcmd () { sh -e "$REINSTALL_DBNODE_DIR/BEFOREBUILD.conf"; }
		if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
		then
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'BEFOREBUILD operations'
			return 1
		fi
		message_echo "-- BEFOREBUILD operations (end)"
		reinstall_register_stage_complete BEFOREBUILD
	fi
	if [ $opt_fetch_only = no ] && ! reinstall_chk_stage_complete CLEAN_BEFORE_BUILD
	then
		_reinstall_exec__tmpcmd () { reinstall_make_anymode clean NOCLEANDEPENDS=yes; }
		if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
		then
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'clean before build'
			return 1
		fi
		message_echo
		reinstall_register_stage_complete CLEAN_BEFORE_BUILD
	fi
	reinstall_deregister_stage in_prebuild
}

# ============= Reinstallation of the current origin: Fetch process =============
# Return status 1 for error end
reinstall_exec_reinstall_fetch ()
{
	local dev_out dev_err
	dev_out=/dev/stdout
	dev_err=/dev/stderr
	if [ $opt_batch_mode = yes ]
	then
		dev_out=/dev/null
		dev_err=/dev/null
	fi
	if ! reinstall_chk_stage_complete FETCH
	then
		if ! reinstall_chk_stage FAILED_FETCH
		then
			reinstall_make_anymode checksum > $dev_out > $dev_err || \
				reinstall_register_stage FAILED_FETCH
		fi
		if reinstall_chk_stage FAILED_FETCH
		then
			if ! reinstall_chk_stage_complete FAILED_FETCH.RETRIAL_1
			then
				message_echo "INFO: Refetching distfiles for $REINSTALL_ORIGPKGTAG."
				{
					reinstall_make_anymode fetch FETCH_ARGS=-Ap > $dev_out > $dev_err &&
						reinstall_make_anymode checksum > $dev_out > $dev_err
				} || reinstall_register_stage FAILED_REFETCH_1
				reinstall_register_stage_complete FAILED_FETCH.RETRIAL_1
			fi
			if reinstall_chk_stage FAILED_REFETCH_1
			then
				if ! reinstall_chk_stage_complete FAILED_FETCH.RETRIAL_2.DISTCLEAN
				then
					message_echo "INFO: Cleaning distfiles for the second refetch for $REINSTALL_ORIGPKGTAG."
					reinstall_make_anymode distclean > $dev_out > $dev_err || :
					reinstall_register_stage_complete FAILED_FETCH.RETRIAL_2.DISTCLEAN
				fi
				if ! reinstall_chk_stage_complete FAILED_FETCH.RETRIAL_2
				then
					message_echo "INFO: Refetching distfiles as the second retrial for $REINSTALL_ORIGPKGTAG."
					_reinstall_exec__tmpcmd () { reinstall_make_anymode fetch FETCH_ARGS=-Ap; }
					if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
					then
						reinstall_restore_conflicts "$REINSTALL_ORIGIN"
						reinstall_restore_if_temporarily_deinstalled
						reinstall_errproc 'fetch'
						return
					fi
					_reinstall_exec__tmpcmd () { reinstall_make_anymode checksum; }
					if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
					then
						reinstall_restore_conflicts "$REINSTALL_ORIGIN"
						reinstall_restore_if_temporarily_deinstalled
						reinstall_errproc 'checksum'
						return
					fi
					reinstall_register_stage_complete FAILED_FETCH.RETRIAL_2
				fi
			fi
		fi
		reinstall_register_stage_complete FETCH
	fi
}

# ============= Reinstallation of the current origin: Check matching of interactive or non-interactive mode=============
# Return status 1 for error end
reinstall_exec_reinstall_chk_interactive_mode ()
{
	local to_skip msg_is_interactive
	if [ $opt_batch_ports_only = yes -o $opt_interactive_ports_only = yes ]
	then
		to_skip=no
		case `database_query_get_makevar_val "$REINSTALL_ORIGIN" IS_INTERACTIVE` in
			yes )	msg_is_interactive='interactive'
				[ $opt_batch_ports_only = yes ] && to_skip=yes
				;;
			'' )	msg_is_interactive='not interactive'
				[ $opt_interactive_ports_only = yes ] && to_skip=yes
				;;
		esac
		if [ $to_skip = yes ]
		then
			reinstall_restore_if_temporarily_deinstalled
			database_build_update_pkgname "$REINSTALL_ORIGIN" > /dev/null
			message_echo "INFO: Further processes for this port are skipped because it is $msg_is_interactive."
			return 1
		fi
	fi
}

# ============= Reinstallation of the current origin: Build process=============
# Return status 1 for error end
reinstall_exec_reinstall_build ()
{
	local flag_restarted_build build_args is_build_env_modified conflicts_install dev_out dev_err REINSTALL_ORIGIN_equiv
	dev_out=/dev/stdout
	dev_err=/dev/stderr
	REINSTALL_ORIGIN_equiv=`database_query_get_equivalent_orgin "$REINSTALL_ORIGIN"`
	if [ $opt_batch_mode = yes ]
	then
		dev_out=/dev/null
		dev_err=/dev/null
	fi
	if reinstall_chk_stage in_build
	then
		message_echo "(Restarting the previously terminated build process...)"
		flag_restarted_build=yes
	else
		reinstall_register_stage in_build
		flag_restarted_build=no
	fi
	if reinstall_chk_stage in_retrial_build
	then
		message_echo "(Restarting the previously terminated retrial build process...)"
		build_args='MAKE_JOBS_UNSAFE=yes'
	else
		build_args=
	fi
	if ! reinstall_chk_stage_complete BUILD
	then
		pkgsys_get_conflicting_installed_pkgs build "$REINSTALL_ORIGIN" \
			| reinstall_backup_and_delete_conflicts
		reinstall_chk_missing_requirement || return
		if ! reinstall_make build $build_args > $dev_out 2> $dev_err
		then
			reinstall_register_stage in_retrial_build
			[ $flag_restarted_build = yes ] && message_echo "INFO: This failure may be due to restarting from a terminated build."
			is_build_env_modified=no
			if conflicts_install=`pkgsys_get_conflicting_installed_pkgs install "$REINSTALL_ORIGIN"`
			then
				message_echo "INFO: Install-only conflicts are deinstalled for retrial because they may have effects on build."
				echo "$conflicts_install" | reinstall_backup_and_delete_conflicts
				is_build_env_modified=yes
			fi
			if pkgsys_exists_from_orig "$REINSTALL_ORIGIN" "$REINSTALL_ORIGIN_equiv"
			then
				message_echo "INFO: The currently installed package for this port is deinstalled for retrial because it may have effects on build."
				reinstall_pkg_backup || :
				reinstall_deinstall_old_ports
				reinstall_deinstall
				is_build_env_modified=yes
			fi
			reinstall_deregister_stage_complete CLEAN_BEFORE_BUILD
			message_echo "INFO: Retrying the build process after cleaning for $REINSTALL_ORIGPKGTAG."
			_reinstall_exec__tmpcmd () { reinstall_make clean; }
			if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
			then
				if pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN"
				then
					pkg_is_tool_available || pkg_rescue_tools
				fi
				reinstall_restore_conflicts "$REINSTALL_ORIGIN"
				reinstall_restore_if_temporarily_deinstalled
				reinstall_errproc 'clean after build failure'
				return 1
			fi
			reinstall_register_stage_complete CLEAN_BEFORE_BUILD
			_reinstall_exec__tmpcmd () { reinstall_make build MAKE_JOBS_UNSAFE=yes; }
			if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
			then
				if pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN"
				then
					pkg_is_tool_available || pkg_rescue_tools
				fi
				reinstall_restore_conflicts "$REINSTALL_ORIGIN"
				reinstall_restore_if_temporarily_deinstalled
				if [ `cat "$REINSTALL_DBNODE_DIR/failed_requirements.build.full" 2> /dev/null | wc -l` -eq 0 ]
				then
					reinstall_errproc 'build'
				else
					message_echo "INFO: The build failure might be due to the build failure(s) of the following requirements:"
					message_cat "$REINSTALL_DBNODE_DIR/failed_requirements.build.full"
					reinstall_errproc 'requirements build'
				fi
				return 1
			fi
			reinstall_deregister_stage in_retrial_build
		fi
		if pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN"
		then
			pkg_is_tool_available || pkg_rescue_tools
		fi
		reinstall_register_stage_complete BUILD
	fi
	reinstall_deregister_stage in_retrial_build
	reinstall_deregister_stage in_build
}

# ============= Reinstallation of the current origin: Stage process=============
# Return status 1 for error end
reinstall_exec_reinstall_stage ()
{
	if reinstall_chk_stage in_stage
	then
		message_echo "(Restarting the previously terminated staging process...)"
	else
		reinstall_register_stage in_stage
	fi
	if ! reinstall_chk_stage_complete STAGE
	then
		_reinstall_exec__tmpcmd () { reinstall_make stage; }
		if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
		then
			reinstall_restore_conflicts "$REINSTALL_ORIGIN"
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'STAGE operations'
			return 1
		fi
		reinstall_register_stage_complete STAGE
	fi
	reinstall_deregister_stage in_stage
}

# ============= Reinstallation of the current origin: Install process =============
# Return status 1 for error end
reinstall_exec_reinstall_install ()
{
	local insttarget dev_out dev_err REINSTALL_ORIGIN_equiv
	dev_out=/dev/stdout
	dev_err=/dev/stderr
	REINSTALL_ORIGIN_equiv=`database_query_get_equivalent_orgin "$REINSTALL_ORIGIN"`
	if [ $opt_batch_mode = yes ]
	then
		dev_out=/dev/null
		dev_err=/dev/null
	fi
	if [ -e "$REINSTALL_FROMNODE_DIR/installed_version" ]
	then
		insttarget=reinstall
	else
		insttarget=install
	fi
	if reinstall_chk_stage in_install
	then
		message_echo "(Restarting the previously terminated installation process...)"
	else
		reinstall_register_stage in_install
	fi
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_default_conf.list"
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_custom_conf.list"
	fileedit_rm_a_line "$REINSTALL_ORIGIN" "${DBDIR}/installation_complete_by_pkg.list"
	reinstall_pkg_backup || :
	if [ -e "$REINSTALL_DBNODE_DIR/BEFOREDEINSTALL.conf" ] && ! reinstall_chk_stage_complete BEFOREDEINSTALL
	then
		message_echo "-- BEFOREDEINSTALL operations (start)"
		_reinstall_exec__tmpcmd () { sh -e "$REINSTALL_DBNODE_DIR/BEFOREDEINSTALL.conf"; }
		if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
		then
			reinstall_restore_conflicts "$REINSTALL_ORIGIN"
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'BEFOREDEINSTALL operations'
			return 1
		fi
		message_echo "-- BEFOREDEINSTALL operations (end)"
		reinstall_register_stage_complete BEFOREDEINSTALL
	fi
	if ! reinstall_chk_stage_complete INSTALL
	then
		if pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN"
		then
			reinstall_backup_and_delete_remaining_install_conflicts
			reinstall_deinstall_old_ports
			reinstall_deinstall
		else
			reinstall_deinstall_old_ports
			reinstall_deinstall
			reinstall_backup_and_delete_remaining_install_conflicts
		fi
		if ! reinstall_chk_stage FAILED_INSTALL
		then
			if reinstall_make $insttarget > $dev_out 2> $dev_err || \
				{
					message_echo "INFO: Cleaning up for retrial."
					reinstall_make deinstall > $dev_out 2> $dev_err \
						|| message_echo "WARNING: Continuing by hoping a success." >&2
					message_echo "INFO: Retrying the installation."
					_reinstall_exec__tmpcmd () { reinstall_make $insttarget MAKE_JOBS_UNSAFE=yes; }
					reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
				}
			then
				touch "$REINSTALL_DBNODE_DIR/installed_timestamp"
				reinstall_register_stage_complete INSTALL
			else
				reinstall_register_stage FAILED_INSTALL
			fi
		fi
		if reinstall_chk_stage FAILED_INSTALL
		then
			if pkgsys_is_necessary_pkgtool "$REINSTALL_ORIGIN"
			then
				pkg_is_tool_available || pkg_rescue_tools
			else
				reinstall_failed_install_recover
			fi
			if [ -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf" ] && \
				! reinstall_chk_stage_complete FAILED_INSTALL.AFTERINSTALL
			then
				message_echo "-- AFTERINSTALL operations (start)"
				if ! sh -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf"
				then
					message_echo "WARNING: Failed in AFTERINSTALL operations." >&2
					message_echo "---- (The process is continued anyway)"
				fi
				message_echo "-- AFTERINSTALL operations (end)"
				reinstall_register_stage_complete FAILED_INSTALL.AFTERINSTALL
			fi
			reinstall_restore_conflicts "$REINSTALL_ORIGIN"
			reinstall_restore_if_temporarily_deinstalled
			reinstall_errproc 'install'
			return 1
		fi
	fi
	reinstall_remove_needless_possible_conflict
	reinstall_restore_if_temporarily_deinstalled
	if ! reinstall_chk_stage_complete UPDATE_CONFLICTS
	then
		pkg_info_qL "$REINSTALL_NEWPKGNAME" > ${TMPDIR}/reinstall_exec:currently_installed_files
		cat "${DBDIR}/backup_pkgarcs.lst" 2> /dev/null \
			| while read origin_bak pkgpath_bak
		do
			pkg_bak=`pkgsys_pkgarc_to_pkgname "$pkgpath_bak"`
			origin_bak_equiv=`database_query_get_equivalent_orgin "$origin_bak"`
			if [ "x$origin_bak" = "x$REINSTALL_ORIGIN" -o "x$origin_bak_equiv" = "x$REINSTALL_ORIGIN" -o "x$origin_bak" = "x$REINSTALL_ORIGIN_equiv" ]
			then
				[ "x$pkg_bak" = "x$REINSTALL_NEWPKGNAME" ] && continue
			elif reinstall_quick_chk_forbidden_conflicts "$pkg_bak"
			then
				continue
			elif ! pkgsys_chk_match_to_restored_files_by_backup_pkg \
				"$origin_bak" "${TMPDIR}/reinstall_exec:currently_installed_files"
			then
				continue
			fi
			fileedit_add_a_line_if_new "$pkg_bak:$REINSTALL_ORIGIN:$REINSTALL_NEWPKGNAME" \
				"${DBDIR}/forbidden_conflicts"
		done
		reinstall_register_stage_complete UPDATE_CONFLICTS
	fi
	if database_query_is_default_conf "$REINSTALL_ORIGIN" quiet
	then
		fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_default_conf.list"
	else
		fileedit_add_a_line_if_new "$REINSTALL_ORIGIN" "${DBDIR}/inst_by_port_with_custom_conf.list"
	fi
	if [ -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf" ] && ! reinstall_chk_stage_complete AFTERINSTALL
	then
		message_echo "-- AFTERINSTALL operations (start)"
		_reinstall_exec__tmpcmd () { sh -e "$REINSTALL_DBNODE_DIR/AFTERINSTALL.conf"; }
		if ! reinstall_execcmd_tee_errlog _reinstall_exec__tmpcmd
		then
			reinstall_restore_conflicts "$REINSTALL_ORIGIN"
			reinstall_errproc 'AFTERINSTALL operations'
			return 1
		fi
		message_echo "-- AFTERINSTALL operations (end)"
		reinstall_register_stage_complete AFTERINSTALL
	fi
	reinstall_deregister_stage in_install
}

# ============= Reinstallation of the current origin: Closing process after successful install =============
reinstall_exec_reinstall_close_successful_install ()
{
	local dev_out dev_err
	dev_out=/dev/stdout
	dev_err=/dev/stderr
	if [ $opt_batch_mode = yes ]
	then
		dev_out=/dev/null
		dev_err=/dev/null
	fi
	reinstall_restore_conflicts "$REINSTALL_ORIGIN"
	if ! reinstall_chk_stage_complete CLEAN_AFTER_INSTALL
	then
		if ! reinstall_make clean > $dev_out 2> $dev_err
		then
			message_echo "WARNING: Failed to clean $REINSTALL_ORIGPKGTAG." >&2
		fi
		reinstall_register_stage_complete CLEAN_AFTER_INSTALL
	fi
	reinstall_tell_update_to_depandents
	reinstall_closing_operations_after_successful_install
}

# ============= Reinstallation of the current origin =============
# Return status 1 for error or incomplete end
reinstall_exec_reinstall ()
{
	local insttarget instdesc
	if [ -e "$REINSTALL_FROMNODE_DIR/installed_version" ]
	then
		insttarget=reinstall
		instdesc='a reinstallation'
	else
		insttarget=install
		instdesc='an installation'
	fi
	temp_set_msg_current_stage "$instdesc process for $REINSTALL_ORIGPKGTAG $PROGRAM_STEP_COUNTER"
	message_target_relations "$REINSTALL_ORIGIN"
	message_echo "------- Starting $instdesc process --------"
	reinstall_exec_reinstall_check_sanity || return
	if [ $opt_dry_run = yes ]
	then
		message_dry_run
		return
	fi
	reinstall_exec_reinstall_by_pkg && return
	if [ $REINSTALL_IS_FROZEN = yes ]
	then
		reinstall_exec_reinstall_freeze_if_necessary || return
		return
	fi
	reinstall_exec_reinstall_avoid_if_any_unfrozen_requirements_exists || return
	reinstall_exec_reinstall_prebuild || return
	reinstall_exec_reinstall_fetch || return
	if [ $opt_fetch_only = yes ]
	then
		message_fetch_only
		return
	fi
	reinstall_exec_reinstall_chk_interactive_mode || return
	reinstall_exec_reinstall_build || return
	reinstall_exec_reinstall_stage || return
	reinstall_exec_reinstall_install || return
	reinstall_exec_reinstall_close_successful_install
}

# ============= Reinstallation for a flavored origin =============
reinstall_exec ()
{
	local REINSTALL_ORIGIN REINSTALL_DBNODE_DIR REINSTALL_FROMNODE_DIR
	local REINSTALL_CURRENTPKG REINSTALL_IS_CURRENTPKG_LATEST
	local REINSTALL_PKGTAG REINSTALL_ORIGPKGTAG REINSTALL_DBSUFFIX
	local REINSTALL_NEWPKGNAME REINSTALL_IS_FROZEN
	REINSTALL_ORIGIN=$1
	REINSTALL_DBNODE_DIR=${DBDIR}/requires/$REINSTALL_ORIGIN
	REINSTALL_FROMNODE_DIR=${DBDIR}/moved_from/$REINSTALL_ORIGIN
	REINSTALL_CURRENTPKG=`database_build_update_pkgname "$REINSTALL_ORIGIN" | tr '\n' ' ' | sed 's/ *$//'`
	REINSTALL_IS_CURRENTPKG_LATEST=no
	database_build_is_currentpkg_latest "$REINSTALL_ORIGIN" && REINSTALL_IS_CURRENTPKG_LATEST=yes
	database_build_update_pkgtag "$REINSTALL_ORIGIN"
	REINSTALL_PKGTAG=`cat "$REINSTALL_FROMNODE_DIR/pkgtag"`
	REINSTALL_ORIGPKGTAG="$REINSTALL_ORIGIN ($REINSTALL_PKGTAG)"
	REINSTALL_DBSUFFIX=`options_get_dependency_type`.`options_get_dependency_level`
	REINSTALL_NEWPKGNAME=`database_build_get_new_pkgname "$REINSTALL_ORIGIN"`
	REINSTALL_IS_FROZEN=no
	env LANG=C grep -qFx "$REINSTALL_ORIGIN" "${DBDIR}/freeze.all.list" 2> /dev/null && REINSTALL_IS_FROZEN=yes
	message_stage_title "$PROGRAM_STEP_COUNTER $REINSTALL_ORIGPKGTAG"
	reinstall_exec_chk_skip && reinstall_exec_reinstall || :
	message_echo
}

# ============= Indivisual make of a port for user command operations =============
reinstall_make_individual ()
{
	local REINSTALL_ORIGIN REINSTALL_DBNODE_DIR
	REINSTALL_ORIGIN=$1
	REINSTALL_DBNODE_DIR=${DBDIR}/requires/$REINSTALL_ORIGIN
	shift || :
	reinstall_make_anymode "$@"
}

