
--
-- Copyright (C) 2022  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--

with system;
with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;


with interfaces.c;
use type interfaces.c.unsigned_short;

with ada.numerics.generic_elementary_functions;
with ada.finalization;
with unchecked_deallocation;

with text_io; use text_io;



package body cyl2obj is -- intersection of two cylinders, for ZPMs
	package fmath is new
			ada.Numerics.generic_elementary_functions( float );
	use fmath;


------------ begin private procs --------------------------

function sqr( x:float ) return float is
begin
	return x*x;
end sqr;



procedure myassert( condition : boolean;  flag: integer:=0 ) is
begin
  if condition=false then
  		put("ASSERTION Failed!  ");
		if flag /= 0 then
			put_line( "cyl2obj @ " & integer'image(flag) );
		end if;
		new_line;
  		raise program_error;
  end if;
end myassert;



------------ end private procs --------------------------


procedure initialize( bar: in out ball ) is
begin
	bar.vert := new varray;
	bar.onncolr := new varray;
	bar.offcolr := new varray;
	bar.elem := new earray;
end initialize;

procedure vfree is new unchecked_deallocation(varray,vap);
procedure efree is new unchecked_deallocation(earray,eap);

procedure finalize( bar: in out ball ) is
begin
	vfree( bar.vert );
	vfree( bar.onncolr );
	vfree( bar.offcolr );
	efree( bar.elem );
	--text_io.put_line("cyl2 Free");
end finalize;




-- variation on c++ version:
-- use this when possible so we don't have to 
-- redefine colors, texture coords:
procedure recenter( bar: in out ball;  dxc, dyc, dzc : float ) is
	vmx: constant integer := nparts*96;
	k,j: integer;
begin

	-- body:
	for p in 0..nparts-1 loop
		k:=p*96;
		for h in 0..31 loop
			j:=h*3;
			bar.vert(j+1+k) := bar.vert(j+1+k)+dxc; --1st: 1
			bar.vert(j+2+k) := bar.vert(j+2+k)+dyc;
			bar.vert(j+3+k) := bar.vert(j+3+k)+dzc; --last: (np-1)*96+93+3 = np*96
		end loop; -- for h
	end loop; -- for p

	-- 2 endcaps:
	for k in 0..7 loop
		j:=k*3;
		bar.vert(vmx+j+1) := bar.vert(vmx+j+1)+dxc; --1st: np*96+1
		bar.vert(vmx+j+2) := bar.vert(vmx+j+2)+dyc;
		bar.vert(vmx+j+3) := bar.vert(vmx+j+3)+dzc; --last: np*96+21+3 = np*96+24 = nvert
	end loop; -- for k

end recenter;




procedure getcen( bar: ball; xxcc,yycc,zzcc: in out float ) is
begin
	xxcc:=bar.oxc;
	yycc:=bar.oyc;
	zzcc:=bar.ozc;
end;



-- here is the new preferred way to update ball pos;
-- it tracks (dx,dz) internally, then calls recenter.
-- It is much cheaper than calling setcyl2.
procedure updcyl2( bar: in out ball; xc,yc,zc : float ) is
	dx : constant float := xc - bar.oxc;
	dy : constant float := yc - bar.oyc;
	dz : constant float := zc - bar.ozc;
begin
	bar.oxc:=xc;
	bar.oyc:=yc;
	bar.ozc:=zc;
	recenter(bar, dx,dy,dz); --[ada version] vanishing ball
end updcyl2;



procedure setcyl2( bar: in out ball; xc,yc,zc, rr : float ) is

	vmx : constant integer := nparts*96;
	y0,y1,z0,z1,x0,x1 : float;
	k,m,j : integer := 0;
	jj : glushort;
	dy : constant float := rr/float(nparts+1);
	ytop : constant float := float(nparts)*dy; --26dec15
	xtop : constant float := sqrt(rr*rr - ytop*ytop);
	ztop : constant float := xtop;

--// end caps are:
--// top:  p1=(+xtop,ytop,+ztop), p2=(+xtop,ytop,-ztop),
--//       p3=(-xtop,ytop,-ztop), p4=(-xtop,ytop,+ztop)
--//
--// bot:  p1=(+xtop,-ytop,+ztop), p2(-xtop,-ytop,+ztop),
--//       p3=(-xtop,-ytop,-ztop), p4(+xtop,-ytop,-ztop);

begin


	for p in 0..nparts-1 loop

		y0:= float(p) * dy;
		y1:= y0 + dy;
		z0:= sqrt( rr*rr - y0*y0 );
		z1:= sqrt( rr*rr - y1*y1 );
		x0:= z0;
		x1:= z1;
		m := k;


		--top front ccw
		bar.vert(k+ 1):=-x0;  bar.vert(k+ 2):=+y0;  bar.vert(k+ 3):=+z0;
		bar.vert(k+ 4):=+x0;  bar.vert(k+ 5):=+y0;  bar.vert(k+ 6):=+z0;
		bar.vert(k+ 7):=+x1;  bar.vert(k+ 8):=+y1;  bar.vert(k+ 9):=+z1;
		bar.vert(k+10):=-x1;  bar.vert(k+11):=+y1;  bar.vert(k+12):=+z1;  
		k:=k+12;

		--bot front ccw
		bar.vert(k+ 1):=-x1;  bar.vert(k+ 2):=-y1;  bar.vert(k+ 3):=+z1;
		bar.vert(k+ 4):=+x1;  bar.vert(k+ 5):=-y1;  bar.vert(k+ 6):=+z1;
		bar.vert(k+ 7):=+x0;  bar.vert(k+ 8):=-y0;  bar.vert(k+ 9):=+z0;
		bar.vert(k+10):=-x0;  bar.vert(k+11):=-y0;  bar.vert(k+12):=+z0;  
		k:=k+12;

		--top rear ccw
		bar.vert(k+ 1):=+x0;  bar.vert(k+ 2):=+y0;  bar.vert(k+ 3):=-z0;
		bar.vert(k+ 4):=-x0;  bar.vert(k+ 5):=+y0;  bar.vert(k+ 6):=-z0;
		bar.vert(k+ 7):=-x1;  bar.vert(k+ 8):=+y1;  bar.vert(k+ 9):=-z1;
		bar.vert(k+10):=+x1;  bar.vert(k+11):=+y1;  bar.vert(k+12):=-z1;  
		k:=k+12;

		--bot rear ccw
		bar.vert(k+ 1):=+x1;  bar.vert(k+ 2):=-y1;  bar.vert(k+ 3):=-z1;
		bar.vert(k+ 4):=-x1;  bar.vert(k+ 5):=-y1;  bar.vert(k+ 6):=-z1;
		bar.vert(k+ 7):=-x0;  bar.vert(k+ 8):=-y0;  bar.vert(k+ 9):=-z0;
		bar.vert(k+10):=+x0;  bar.vert(k+11):=-y0;  bar.vert(k+12):=-z0;  
		k:=k+12;

---------------------------- 4 rectangles of 8 -------------------------

		--top left ccw
		bar.vert(k+ 1):=-x0;  bar.vert(k+ 2):=+y0;  bar.vert(k+ 3):=-z0;
		bar.vert(k+ 4):=-x0;  bar.vert(k+ 5):=+y0;  bar.vert(k+ 6):=+z0;
		bar.vert(k+ 7):=-x1;  bar.vert(k+ 8):=+y1;  bar.vert(k+ 9):=+z1;
		bar.vert(k+10):=-x1;  bar.vert(k+11):=+y1;  bar.vert(k+12):=-z1;  
		k:=k+12;

		--bot left ccw
		bar.vert(k+ 1):=-x1;  bar.vert(k+ 2):=-y1;  bar.vert(k+ 3):=-z1;
		bar.vert(k+ 4):=-x1;  bar.vert(k+ 5):=-y1;  bar.vert(k+ 6):=+z1;
		bar.vert(k+ 7):=-x0;  bar.vert(k+ 8):=-y0;  bar.vert(k+ 9):=+z0;
		bar.vert(k+10):=-x0;  bar.vert(k+11):=-y0;  bar.vert(k+12):=-z0;  
		k:=k+12;

		--top right ccw
		bar.vert(k+ 1):=+x0;  bar.vert(k+ 2):=+y0;  bar.vert(k+ 3):=+z0;
		bar.vert(k+ 4):=+x0;  bar.vert(k+ 5):=+y0;  bar.vert(k+ 6):=-z0;
		bar.vert(k+ 7):=+x1;  bar.vert(k+ 8):=+y1;  bar.vert(k+ 9):=-z1;
		bar.vert(k+10):=+x1;  bar.vert(k+11):=+y1;  bar.vert(k+12):=+z1;  
		k:=k+12;

		--bot right ccw
		bar.vert(k+ 1):=+x1;  bar.vert(k+ 2):=-y1;  bar.vert(k+ 3):=+z1;
		bar.vert(k+ 4):=+x1;  bar.vert(k+ 5):=-y1;  bar.vert(k+ 6):=-z1;
		bar.vert(k+ 7):=+x0;  bar.vert(k+ 8):=-y0;  bar.vert(k+ 9):=-z0;
		bar.vert(k+10):=+x0;  bar.vert(k+11):=-y0;  bar.vert(k+12):=+z0;  
		k:=k+12;


		-- begin set colors:
		for j in 0..15 loop
			bar.offcolr(3*j+m+1):=0.7;  
				bar.offcolr(3*j+m+2):=0.0;  
					bar.offcolr(3*j+m+3):=0.0; --drk.red front/back

			bar.onncolr(3*j+m+1):=1.0;  
				bar.onncolr(3*j+m+2):=0.3;  
					bar.onncolr(3*j+m+3):=0.3; --brt.red front/back
		end loop;
		for j in 16..31 loop
			bar.offcolr(3*j+m+1):=0.0;  
				bar.offcolr(3*j+m+2):=0.0;  
					bar.offcolr(3*j+m+3):=0.7; --drk.blue sides
			bar.onncolr(3*j+m+1):=0.3;  
				bar.onncolr(3*j+m+2):=0.3;  
					bar.onncolr(3*j+m+3):=1.0; --brt.blue sides
		end loop;
		-- end set colors:

	end loop; --for p


myassert(k=nvert-24, 1001);
myassert(k=vmx, 1002);



	-- begin handle 2 green endcaps:

	bar.vert(vmx+ 1):=+xtop;  bar.vert(vmx+ 2):=+ytop;  bar.vert(vmx+ 3):=+ztop;
	bar.vert(vmx+ 4):=+xtop;  bar.vert(vmx+ 5):=+ytop;  bar.vert(vmx+ 6):=-ztop;
	bar.vert(vmx+ 7):=-xtop;  bar.vert(vmx+ 8):=+ytop;  bar.vert(vmx+ 9):=-ztop;
	bar.vert(vmx+10):=-xtop;  bar.vert(vmx+11):=+ytop;  bar.vert(vmx+12):=+ztop;

	bar.vert(vmx+13):=+xtop;  bar.vert(vmx+14):=-ytop;  bar.vert(vmx+15):=+ztop;
	bar.vert(vmx+16):=-xtop;  bar.vert(vmx+17):=-ytop;  bar.vert(vmx+18):=+ztop;
	bar.vert(vmx+19):=-xtop;  bar.vert(vmx+20):=-ytop;  bar.vert(vmx+21):=-ztop;
	bar.vert(vmx+22):=+xtop;  bar.vert(vmx+23):=-ytop;  bar.vert(vmx+24):=-ztop;

	--last:  np*96+24 = nvert

	-- set colr green
	for i in 0..7 loop
		bar.offcolr(vmx+3*i+1):=0.0;  
			bar.offcolr(vmx+3*i+2):=0.7;  
				bar.offcolr(vmx+3*i+3):=0.0; --drk.grn
		bar.onncolr(vmx+3*i+1):=0.3;  
			bar.onncolr(vmx+3*i+2):=1.0;  
				bar.onncolr(vmx+3*i+3):=0.3; --brt.grn
	end loop;

	-- end handle 2 green endcaps:



	-- now offset object vertices from (0,0,0) to new center @ (xc,yc,zc)
	bar.oxc:=xc;
	bar.oyc:=yc;
	bar.ozc:=zc;
	bar.orr:=rr;
	recenter(bar, xc,yc,zc);



	-- finally set all of the element indices:
	for i in 0..nparts*8+1 loop -- 2 endcaps + np*8 rectangles
		jj := glushort(i*4);
		j := i*6;
		bar.elem(j+1):=jj+0;  bar.elem(j+2):=jj+1;  bar.elem(j+3):=jj+2;
		bar.elem(j+4):=jj+2;  bar.elem(j+5):=jj+3;  bar.elem(j+6):=jj+0;
	end loop;
	-- 1st: 1
	-- last: (np*8+1)*6+6 = 48*np+12 = nelm

end setcyl2;




-- note:  the shaders for these objects must have two 
-- input "layouts", as well as whatever uniforms are needed:
--
-- layout(location=0) in vec3 vertPosName
-- layout(location=1) in vec3 vertRgbName
--
-- ...where their actual names can be whatever is convenient
--
use gl;
use glext;
use glext.binding;
use gl.binding;
procedure draw( bar: ball; 
	vertbuff, rgbbuff, elembuff : gluint; on:boolean:=false ) is
begin

	-- 0th attribute:  vertices
	glBindBuffer(gl_array_buffer, vertbuff);
	glBufferData(gl_array_buffer, glsizeiptr(4*nvert), 
		bar.vert(1)'address, gl_static_draw);
	glEnableVertexAttribArray(0);
	glVertexAttribPointer(0,3,gl_float,gl_false,0, system.null_address);

	-- 1st attribute:  color
	glBindBuffer(gl_array_buffer, rgbbuff);
	if on then
		glBufferData(gl_array_buffer, glsizeiptr(4*nvert), 
			bar.onncolr(1)'address, gl_static_draw);
	else
		glBufferData(gl_array_buffer, glsizeiptr(4*nvert), 
			bar.offcolr(1)'address, gl_static_draw);
	end if;
	glEnableVertexAttribArray(1);
	glVertexAttribPointer(1,3,gl_float,gl_true,0, system.null_address);

	-- element indices:
	glBindBuffer(gl_element_array_buffer, elembuff);
	glBufferData(gl_element_array_buffer, glsizeiptr(2*nelm), 
		bar.elem(1)'address, gl_static_draw);

	glDrawElements( gl_triangles, glint(nvert), 
		gl_unsigned_short, system.null_address );

	glDisableVertexAttribArray(0);
	glDisableVertexAttribArray(1);

end draw;

end cyl2obj;

