# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: Downloader.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Download pieces from remote peers.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type EXPIRE_TIME: C{int}
@var EXPIRE_TIME: number of seconds after which disconnected seeds are expired

"""

from DebTorrent.CurrentRateMeasure import Measure
from DebTorrent.bitfield import Bitfield
from random import shuffle
from DebTorrent.clock import clock
import logging

logger = logging.getLogger('DebTorrent.BT1.Downloader')

EXPIRE_TIME = 60 * 60

class PerIPStats:
    """Statistics relating to downloads from a single peer.
    
    @type numgood: C{int}
    @ivar numgood: the number of good pieces received
    @type bad: C{dictionary}
    @ivar bad: keys are piece numbers, values are the number of bad copies
        of the piece received from the peer
    @type numconnections: C{int}
    @ivar numconnections: the number of connections made to the peer
    @type lastdownload: L{SingleDownload}
    @ivar lastdownload: the most recent SingleDownload instance
    @type peerid: C{string}
    @ivar peerid: the peer's ID
    
    """
    
    def __init__(self, ip):
        """Initialize the statistics.
        
        @type ip: C{string}
        @param ip: the IP address of the peer (not used)
        
        """
        
        self.numgood = 0
        self.bad = {}
        self.numconnections = 0
        self.lastdownload = None
        self.peerid = None

class BadDataGuard:
    """Process good and bad received pieces from a single peer.
    
    @type download: L{SingleDownload}
    @ivar download: the SingleDownload instance
    @type ip: C{string}
    @ivar ip: IP address of the peer
    @type downloader: L{Downloader}
    @ivar downloader: the Downloader instance
    @type stats: L{PerIPStats}
    @ivar stats: the PerIPStats instance
    @type lastindex: C{int}
    @ivar lastindex: the last good piece that was received
    
    """
    
    def __init__(self, download):
        """Initialize the class.
        
        @type download: L{SingleDownload}
        @param download: the SingleDownload instance for the download
        
        """
        
        self.download = download
        self.ip = download.ip
        self.downloader = download.downloader
        self.stats = self.downloader.perip[self.ip]
        self.lastindex = None

    def failed(self, index, bump = False):
        """Process the failed piece.
        
        @type index: C{int}
        @param index: the piece that failed
        @type bump: C{boolean}
        @param bump: whether to decrease the interest level in the 
            L{PiecePicker.PiecePicker} (optional, defaults to False)
        
        """
        
        self.stats.bad.setdefault(index, 0)
        self.downloader.gotbaddata[self.ip] = 1
        self.stats.bad[index] += 1
        if len(self.stats.bad) > 1:
            if self.download is not None:
                self.downloader.try_kick(self.download)
            elif self.stats.numconnections == 1 and self.stats.lastdownload is not None:
                self.downloader.try_kick(self.stats.lastdownload)
        if len(self.stats.bad) >= 3 and len(self.stats.bad) > int(self.stats.numgood/30):
            self.downloader.try_ban(self.ip)
        elif bump:
            self.downloader.picker.bump(index)

    def good(self, index):
        """Process the successful piece.
        
        @type index: C{int}
        @param index: the piece that succeeded
        
        """
        
        # lastindex is a hack to only increase numgood by one for each good
        # piece, however many chunks come from the connection(s) from this IP
        if index != self.lastindex:
            self.stats.numgood += 1
            self.lastindex = index

class SingleDownload:
    """Manage downloads from a single peer.
    
    @type downloader: L{Downloader}
    @ivar downloader: the Downloader instance
    @type connection: L{Connecter.Connection}
    @ivar connection: the connection to the peer
    @type choked: C{boolean}
    @ivar choked: whether the peer is choking the download
    @type interested: C{boolean}
    @ivar interested: whether the peer is interesting
    @type active_requests: C{list} of (C{int}, C{int}, C{int})
    @ivar active_requests: the requests that have been sent, the piece, offset
        within the piece, and the length of the request
    @type measure: L{DebTorrent.CurrentRateMeasure.Measure}
    @ivar measure: for measuring the download rate from the peer
    @type peermeasure: L{DebTorrent.CurrentRateMeasure.Measure}
    @ivar peermeasure: for measuring the download rate of the peer
    @type have: L{DebTorrent.bitfield.Bitfield}
    @ivar have: the bitfield the peer has
    @type last: C{float}
    @ivar last: the last time a chunk was received from the peer
    @type last2: C{float}
    @ivar last2: the last time a chunk or an unchoke was received
    @type example_interest: C{int}
    @ivar example_interest: an example piece to request
    @type backlog: C{int}
    @ivar backlog: the current backlog of chunk requests
    @type ip: C{string}
    @ivar ip: the IP address of the peer
    @type guard: L{BadDataGuard}
    @ivar guard: the guard to use to process pieces
    
    """
    
    def __init__(self, downloader, connection):
        """Initialize the instance.
        
        @type downloader: L{Downloader}
        @param downloader: the parent Downloader instance
        @type connection: L{Connecter.Connection}
        @param connection: the connection to the peer
        
        """
        
        self.downloader = downloader
        self.connection = connection
        self.choked = True
        self.interested = False
        self.active_requests = []
        self.measure = Measure(downloader.max_rate_period)
        self.peermeasure = Measure(downloader.max_rate_period)
        self.have = Bitfield(downloader.numpieces)
        self.last = -1000
        self.last2 = -1000
        self.example_interest = None
        self.backlog = 2
        self.ip = connection.get_ip()
        self.guard = BadDataGuard(self)

    def _backlog(self, just_unchoked):
        """Calculate the backlog of chunk requests to the peer.
        
        @type just_unchoked: C{boolean}
        @param just_unchoked: whether the connection was just unchoked
        @rtype: C{int}
        @return: the new backlog
        
        """
        
        self.backlog = min(
            2+int(4*self.measure.get_rate()/self.downloader.chunksize),
            (2*just_unchoked)+self.downloader.queue_limit() )
        if self.backlog > 50:
            self.backlog = max(50, self.backlog * 0.075)
        return self.backlog
    
    def disconnected(self):
        """Remove the newly disconnected peer."""
        self.downloader.lost_peer(self)
        if self.have.complete():
            self.downloader.picker.lost_seed()
        else:
            for i in xrange(len(self.have)):
                if self.have[i]:
                    self.downloader.picker.lost_have(i)
        if self.have.complete() and self.downloader.storage.is_endgame():
            self.downloader.add_disconnected_seed(self.connection.get_readable_id())
        self._letgo()
        self.guard.download = None

    def _letgo(self):
        """Remove the oustanding requests to the peer.
        
        For each active request that was unfulfilled by the peer, inform the 
        Storage that the request was lost, and send interested messages to any 
        remaining peers that have the piece.
        
        """
        
        if self.downloader.queued_out.has_key(self):
            del self.downloader.queued_out[self]
        if not self.active_requests:
            return
        if self.downloader.endgamemode:
            self.active_requests = []
            return
        lost = {}
        for index, begin, length in self.active_requests:
            self.downloader.storage.request_lost(index, begin, length)
            lost[index] = 1
        lost = lost.keys()
        self.active_requests = []
        if self.downloader.paused:
            return
        ds = [d for d in self.downloader.downloads if not d.choked]
        shuffle(ds)
        for d in ds:
            d._request_more()
        for d in self.downloader.downloads:
            if d.choked and not d.interested:
                for l in lost:
                    if d.have[l] and self.downloader.storage.do_I_have_requests(l):
                        d.send_interested()
                        break

    def got_choke(self):
        """Update the choked status and remove any active requests."""
        if not self.choked:
            self.choked = True
            self._letgo()

    def got_unchoke(self):
        """Update the choked status and request any needed pieces."""
        if self.choked:
            self.choked = False
            if self.interested:
                self._request_more(new_unchoke = True)
            self.last2 = clock()

    def is_choked(self):
        """Get the choked status of the connection.
        
        @rtype: C{boolean}
        @return: whether the peer is choking the connection
        
        """
        
        return self.choked

    def is_interested(self):
        """Get the interest in the peer.
        
        @rtype: C{boolean}
        @return: whether the peer is interesting
        
        """
        
        return self.interested

    def send_interested(self):
        """Send the interested message to the peer."""
        if not self.interested:
            self.interested = True
            self.connection.send_interested()
            if not self.choked:
                self.last2 = clock()

    def send_not_interested(self):
        """Send the not interested message to the peer."""
        if self.interested:
            self.interested = False
            self.connection.send_not_interested()

    def got_piece(self, index, begin, piece):
        """Process a received chunk.
        
        Add the newly received chunk to the Storage, remove any oustanding
        requests for it, and request more chunks from the peer.
        
        @type index: C{int}
        @param index: the piece index
        @type begin: C{int}
        @param begin: the offset within the piece
        @type piece: C{string}
        @param piece: the chunk
        @rtype: C{boolean}
        @return: whether the piece was accepted by the Storage (valid)
        
        """
        
        length = len(piece)
        try:
            self.active_requests.remove((index, begin, length))
        except ValueError:
            logger.warning('received an unrequested piece from '+self.ip+': '+
                           str(index)+', '+str(begin)+'-'+str(begin+len(piece)))
            self.downloader.discarded += length
            return False
        if self.downloader.endgamemode:
            self.downloader.all_requests.remove((index, begin, length))
        self.last = clock()
        self.last2 = clock()
        self.measure.update_rate(length)
        self.downloader.measurefunc(length)
        if not self.downloader.storage.piece_came_in(index, begin, piece, self.guard):
            self.downloader.piece_flunked(index)
            return False
        if self.downloader.storage.do_I_have(index):
            self.downloader.picker.complete(index)
        if self.downloader.endgamemode:
            for d in self.downloader.downloads:
                if d is not self:
                  if d.interested:
                    if d.choked:
                        assert not d.active_requests
                        d.fix_download_endgame()
                    else:
                        try:
                            d.active_requests.remove((index, begin, length))
                        except ValueError:
                            continue
                        d.connection.send_cancel(index, begin, length)
                        d.fix_download_endgame()
                  else:
                      assert not d.active_requests
        self._request_more()
        self.downloader.check_complete(index)
        return self.downloader.storage.do_I_have(index)

    def _request_more(self, new_unchoke = False):
        """Request more chunks from the peer.
        
        @type new_unchoke: C{boolean}
        @param new_unchoke: whether this request was the result of a recent
            unchoke (optional, defaults to False)
        
        """
        
        assert not self.choked
        if self.downloader.endgamemode:
            self.fix_download_endgame(new_unchoke)
            return
        if self.downloader.paused:
            return
        if len(self.active_requests) >= self._backlog(new_unchoke):
            if not (self.active_requests or self.backlog):
                self.downloader.queued_out[self] = 1
            return
        lost_interests = []
        while len(self.active_requests) < self.backlog:
            interest = self.downloader.picker.next(self.have,
                               self.downloader.storage.do_I_have_requests,
                               self.downloader.too_many_partials())
            if interest is None:
                break
            self.example_interest = interest
            self.send_interested()
            loop = True
            while len(self.active_requests) < self.backlog and loop:
                begin, length = self.downloader.storage.new_request(interest)
                self.downloader.picker.requested(interest)
                self.active_requests.append((interest, begin, length))
                self.connection.send_request(interest, begin, length)
                self.downloader.chunk_requested(length)
                if not self.downloader.storage.do_I_have_requests(interest):
                    loop = False
                    lost_interests.append(interest)
        if not self.active_requests:
            self.send_not_interested()
        if lost_interests:
            for d in self.downloader.downloads:
                if d.active_requests or not d.interested:
                    continue
                if d.example_interest is not None and self.downloader.storage.do_I_have_requests(d.example_interest):
                    continue
                for lost in lost_interests:
                    if d.have[lost]:
                        break
                else:
                    continue
                interest = self.downloader.picker.next(d.have,
                                   self.downloader.storage.do_I_have_requests,
                                   self.downloader.too_many_partials())
                if interest is None:
                    d.send_not_interested()
                else:
                    d.example_interest = interest
        if self.downloader.storage.is_endgame():
            self.downloader.start_endgame()


    def fix_download_endgame(self, new_unchoke = False):
        """Request more chunks from the peer in endgame mode.
        
        @type new_unchoke: C{boolean}
        @param new_unchoke: whether this request was the result of a recent
            unchoke (optional, defaults to False)
        
        """
        
        if self.downloader.paused:
            return
        if len(self.active_requests) >= self._backlog(new_unchoke):
            if not (self.active_requests or self.backlog) and not self.choked:
                self.downloader.queued_out[self] = 1
            return
        want = [a for a in self.downloader.all_requests if self.have[a[0]] and a not in self.active_requests]
        if not (self.active_requests or want):
            self.send_not_interested()
            return
        if want:
            self.send_interested()
        if self.choked:
            return
        shuffle(want)
        del want[self.backlog - len(self.active_requests):]
        self.active_requests.extend(want)
        for piece, begin, length in want:
            self.connection.send_request(piece, begin, length)
            self.downloader.chunk_requested(length)

    def got_have(self, index):
        """Receive a Have message from the peer.
        
        @type index: C{int}
        @param index: the piece the peer now has
        @rtype: C{boolean}
        @return: whether the peer is now a seed
        
        """
        
        self.downloader.totalmeasure.update_rate(self.downloader.storage.piece_lengths[index])
        self.peermeasure.update_rate(self.downloader.storage.piece_lengths[index])
        if not self.have[index]:
            self.have[index] = True
            self.downloader.picker.got_have(index)
            if self.have.complete():
                self.downloader.picker.became_seed()
                if not self.downloader.storage.get_amount_left():
                    self.downloader.add_disconnected_seed(self.connection.get_readable_id())
                    self.connection.close()
            elif self.downloader.endgamemode:
                self.fix_download_endgame()
            elif ( not self.downloader.paused
                   and not self.downloader.picker.is_blocked(index)
                   and self.downloader.storage.do_I_have_requests(index) ):
                if not self.choked:
                    self._request_more()
                else:
                    self.send_interested()
        return self.have.complete()

    def _check_interests(self):
        """Check if the peer is now interesting."""
        if self.interested or self.downloader.paused:
            return
        for i in xrange(len(self.have)):
            if ( self.have[i] and not self.downloader.picker.is_blocked(i)
                 and ( self.downloader.endgamemode
                       or self.downloader.storage.do_I_have_requests(i) ) ):
                self.send_interested()
                return

    def got_have_bitfield(self, have):
        """Receive a Bitfield message from the peer.
        
        @type have: L{DebTorrent.bitfield.Bitfield}
        @param have: the bitfield received from the peer
        @rtype: C{boolean}
        @return: whether the peer is a seed
        
        """
        
        if not self.downloader.storage.get_amount_left() and have.complete():
            if self.downloader.super_seeding:
                self.connection.send_bitfield(have.tostring()) # be nice, show you're a seed too
            self.connection.close()
            self.downloader.add_disconnected_seed(self.connection.get_readable_id())
            return False
        self.have = have
        if have.complete():
            self.downloader.picker.got_seed()
        else:
            for i in xrange(len(have)):
                if have[i]:
                    self.downloader.picker.got_have(i)
        if self.downloader.endgamemode and not self.downloader.paused:
            for piece, begin, length in self.downloader.all_requests:
                if self.have[piece]:
                    self.send_interested()
                    break
        else:
            self._check_interests()
        return have.complete()

    def get_rate(self):
        """Get the current download rate from the peer.
        
        @rtype: C{float}
        @return: the peer's download rate
        
        """
        
        return self.measure.get_rate()

    def is_snubbed(self):
        """Check if the peer is snubbing the download.
        
        @rtype: C{boolean}
        @return: whether the peer is snubbing the connection
        
        """
        
        if ( self.interested and not self.choked
             and clock() - self.last2 > self.downloader.snub_time ):
            for index, begin, length in self.active_requests:
                self.connection.send_cancel(index, begin, length)
            self.got_choke()    # treat it just like a choke
        return clock() - self.last > self.downloader.snub_time


class Downloader:
    """A collection of all single downloads.
    
    @type storage: L{StorageWrapper.StorageWrapper}
    @ivar storage: the StorageWrapper instance
    @type picker: L{PiecePicker.PiecePicker}
    @ivar picker: the PiecePicker instance
    @type max_rate_period: C{float}
    @ivar max_rate_period: maximum amount of time to guess the current 
            rate estimate represents
    @type measurefunc: C{method}
    @ivar measurefunc: the method to call to add downloaded data to the
            measurement of the download rate
    @type totalmeasure: L{DebTorrent.CurrentRateMeasure.Measure}
    @ivar totalmeasure: for measuring the total download rate from all peers
    @type numpieces: C{int}
    @ivar numpieces: total number of pieces in the download
    @type chunksize: C{int}
    @ivar chunksize: the number of bytes to query for per request
    @type snub_time: C{float}
    @ivar snub_time: seconds to wait for data to come in over a connection 
            before assuming it's semi-permanently choked
    @type kickfunc: C{method}
    @ivar kickfunc: method to call to kick a peer
    @type banfunc: C{method}
    @ivar banfunc: method to call to ban a peer
    @type disconnectedseeds: C{dictionary}
    @ivar disconnectedseeds: seeds that have recently been seen, keys are the
        seed's peer ID, values are the last time the seed was seen
    @type downloads: C{list} of C{SingleDownload}
    @ivar downloads: the active downloads from peers
    @type perip: C{dictionary}
    @ivar perip: keys are the IP addresses of peers, values are their L{PerIPStats}
    @type gotbaddata: C{dictionary}
    @ivar gotbaddata: keys are the IP addresses that have sent bad data
    @type kicked: C{dictionary}
    @ivar kicked: keys are the IP address, values are the peer ID for peers
        that have been kicked
    @type banned: C{dictionary}
    @ivar banned: keys are the IP addresses of peers that have been banned,
        values are their peer IDs
    @type kickbans_ok: C{boolean}
    @ivar kickbans_ok: whether to automatically kick/ban peers that send 
            bad data
    @type kickbans_halted: C{boolean}
    @ivar kickbans_halted: not used
    @type super_seeding: C{boolean}
    @ivar super_seeding: whether we are in super-seed mode
    @type endgamemode: C{boolean}
    @ivar endgamemode: whether the download is in end-game mode
    @type endgame_queued_pieces: C{list} of C{int}
    @ivar endgame_queued_pieces: the list of pieces that are queued in end-game mode
    @type all_requests: C{list} of (C{int}, C{int}, C{int})
    @ivar all_requests: all outstanding requests to all peers
    @type discarded: C{long}
    @ivar discarded: the amount of downloaded data that has been discarded
    @type download_rate: C{float}
    @ivar download_rate: the maximum rate to download at
    @type bytes_requested: C{int}
    @ivar bytes_requested: the number of bytes in oustanding requests
    @type last_time: C{float}
    @ivar last_time: the last time the queue limit was calculated
    @type queued_out: C{dictionary}
    @ivar queued_out: keys are L{SingleDownload} that are queued waiting for download
    @type requeueing: C{boolean}
    @ivar requeueing: whether requeueing is currently underway
    @type paused: C{boolean}
    @ivar paused: whether the download is paused
    
    """
    
    def __init__(self, storage, picker, max_rate_period,
                 numpieces, chunksize, measurefunc, snub_time,
                 kickbans_ok, kickfunc, banfunc):
        """Initialize the instance.
        
        @type storage: L{StorageWrapper.StorageWrapper}
        @param storage: the StorageWrapper instance
        @type picker: L{PiecePicker.PiecePicker}
        @param picker: the PiecePicker instance
        @type max_rate_period: C{float}
        @param max_rate_period: maximum amount of time to guess the current 
            rate estimate represents
        @type numpieces: C{int}
        @param numpieces: total number of pieces in the download
        @type chunksize: C{int}
        @param chunksize: the number of bytes to query for per request
        @type measurefunc: C{method}
        @param measurefunc: the method to call to add downloaded data to the
            measurement of the download rate
        @type snub_time: C{float}
        @param snub_time: seconds to wait for data to come in over a connection 
            before assuming it's semi-permanently choked
        @type kickbans_ok: C{boolean}
        @param kickbans_ok: whether to automatically kick/ban peers that send 
            bad data
        @type kickfunc: C{method}
        @param kickfunc: method to call to kick a peer
        @type banfunc: C{method}
        @param banfunc: method to call to ban a peer
        
        """
        
        self.storage = storage
        self.picker = picker
        self.max_rate_period = max_rate_period
        self.measurefunc = measurefunc
        self.totalmeasure = Measure(max_rate_period*(float(storage.datalength)/float(len(storage.piece_lengths)))/storage.request_size)
        self.numpieces = numpieces
        self.chunksize = chunksize
        self.snub_time = snub_time
        self.kickfunc = kickfunc
        self.banfunc = banfunc
        self.disconnectedseeds = {}
        self.downloads = []
        self.perip = {}
        self.gotbaddata = {}
        self.kicked = {}
        self.banned = {}
        self.kickbans_ok = kickbans_ok
        self.kickbans_halted = False
        self.super_seeding = False
        self.endgamemode = False
        self.endgame_queued_pieces = []
        self.all_requests = []
        self.discarded = 0L
#        self.download_rate = 25000  # 25K/s test rate
        self.download_rate = 0
        self.bytes_requested = 0
        self.last_time = clock()
        self.queued_out = {}
        self.requeueing = False
        self.paused = False

    def set_download_rate(self, rate):
        """Set the maximum download rate for all downloads.
        
        @type rate: C{float}
        @param rate: maximum kB/s to download at (0 = no limit)
        
        """
        
        self.download_rate = rate * 1000
        self.bytes_requested = 0

    def queue_limit(self):
        """Get the maximum number of bytes to request.
        
        @rtype: C{int}
        @return: the limit on the number of bytes to request
        
        """
        
        if not self.download_rate:
            return 10e10    # that's a big queue!
        t = clock()
        self.bytes_requested -= (t - self.last_time) * self.download_rate
        self.last_time = t
        if not self.requeueing and self.queued_out and self.bytes_requested < 0:
            self.requeueing = True
            q = self.queued_out.keys()
            shuffle(q)
            self.queued_out = {}
            for d in q:
                d._request_more()
            self.requeueing = False
        if -self.bytes_requested > 5*self.download_rate:
            self.bytes_requested = -5*self.download_rate
        return max(int(-self.bytes_requested/self.chunksize),0)

    def chunk_requested(self, size):
        """Add the new request size to the tally.
        
        @type size: C{int}
        @param size: the number of bytes that were requested
        
        """
        
        self.bytes_requested += size

    external_data_received = chunk_requested

    def make_download(self, connection):
        """Create a new L{SingleDownload} instance for a new connection.
        
        @type connection: L{Connecter.Connection}
        @param connection: the connection that was received
        @rtype: L{SingleDownload}
        @return: the newly created SingleDownload instance
        
        """
        
        ip = connection.get_ip()
        if self.perip.has_key(ip):
            perip = self.perip[ip]
        else:
            perip = self.perip.setdefault(ip, PerIPStats(ip))
        perip.peerid = connection.get_readable_id()
        perip.numconnections += 1
        d = SingleDownload(self, connection)
        perip.lastdownload = d
        self.downloads.append(d)
        return d

    def piece_flunked(self, index):
        """Request a failed piece from other peers.
        
        @type index: C{int}
        @param index: the piece index that failed
        
        """
        
        if self.paused:
            return
        if self.endgamemode:
            if self.downloads:
                while self.storage.do_I_have_requests(index):
                    nb, nl = self.storage.new_request(index)
                    self.all_requests.append((index, nb, nl))
                for d in self.downloads:
                    d.fix_download_endgame()
                return
            self._reset_endgame()
            return
        ds = [d for d in self.downloads if not d.choked]
        shuffle(ds)
        for d in ds:
            d._request_more()
        ds = [d for d in self.downloads if not d.interested and d.have[index]]
        for d in ds:
            d.example_interest = index
            d.send_interested()

    def has_downloaders(self):
        """Get the number of active downloads.
        
        @rtype: C{int}
        @return: the number of active download connections
        
        """
        
        return len(self.downloads)

    def lost_peer(self, download):
        """Remove a lost peer from the collection of downloads.
        
        @type download: L{SingleDownload}
        @param download: the download peer that was lost
        
        """
        
        ip = download.ip
        self.perip[ip].numconnections -= 1
        if self.perip[ip].lastdownload == download:
            self.perip[ip].lastdownload = None
        self.downloads.remove(download)
        if self.endgamemode and not self.downloads: # all peers gone
            self._reset_endgame()

    def _reset_endgame(self):
        """Stop the endgame mode."""
        self.storage.reset_endgame(self.all_requests)
        self.endgamemode = False
        self.all_requests = []
        self.endgame_queued_pieces = []


    def add_disconnected_seed(self, id):
        """Save the time of a disconnected seed.
        
        @type id: C{string}
        @param id: the peer ID of the disconnected seed
        
        """
        
#        if not self.disconnectedseeds.has_key(id):
#            self.picker.seed_seen_recently()
        self.disconnectedseeds[id]=clock()
        self.expire_disconnected_seeds()

    def expire_disconnected_seeds(self):
        """Expire old disconnected seeds."""
        expired = []
        for id,t in self.disconnectedseeds.items():
            if clock() - t > EXPIRE_TIME:     #Expire old seeds after so long
                expired.append(id)
        for id in expired:
#            self.picker.seed_disappeared()
            del self.disconnectedseeds[id]

    def num_disconnected_seeds(self):
        """Calculate the number or recently seen seeds.
        
        @rtype: C{int}
        @return: the number of recently seen disconnected seeds
        
        """
        
        # first expire old ones
        self.expire_disconnected_seeds()
        return len(self.disconnectedseeds)

    def _check_kicks_ok(self):
        """Check whether peers can be kicked for bad data.
        
        @rtype: C{boolean}
        @return: whether it is OK to kick peers
        
        """
        
        if len(self.gotbaddata) > 10:
            self.kickbans_ok = False
            self.kickbans_halted = True
        return self.kickbans_ok and len(self.downloads) > 2

    def try_kick(self, download):
        """If allowed, kick a peer.
        
        @type download: L{SingleDownload}
        @param download: the peer's download connection
        
        """
        
        if self._check_kicks_ok():
            download.guard.download = None
            ip = download.ip
            id = download.connection.get_readable_id()
            self.kicked[ip] = id
            self.perip[ip].peerid = id
            logger.warning('kicking peer '+ip+' ('+id+')')
            self.kickfunc(download.connection)
        
    def try_ban(self, ip):
        """If allowed, ban a peer.
        
        @type ip: C{string}
        @param ip: the IP address of the peer
        
        """
        
        if self._check_kicks_ok():
            logger.warning('banning peer IP '+ip)
            self.banfunc(ip)
            self.banned[ip] = self.perip[ip].peerid
            if self.kicked.has_key(ip):
                del self.kicked[ip]

    def set_super_seed(self):
        """Enable super-seed mode."""
        self.super_seeding = True

    def check_complete(self, index):
        """Check whether the download is complete.
        
        If it is complete, send the piece to the connected seeds and then
        disconnect them.
        
        @type index: C{int}
        @param index: the last received piece
        @rtype: C{boolean}
        @return: whether the download is complete
        
        """
        
        if self.endgamemode and not self.all_requests:
            self.endgamemode = False
        if self.endgame_queued_pieces and not self.endgamemode:
            self.requeue_piece_download()
        if not self.storage.get_amount_left():
            assert not self.all_requests
            assert not self.endgamemode
            for d in [i for i in self.downloads if i.have.complete()]:
                d.connection.send_have(index)   # be nice, tell the other seed you completed
                self.add_disconnected_seed(d.connection.get_readable_id())
                d.connection.close()
            return True
        return False

    def too_many_partials(self):
        """Check whether there are too many outstanding incomplete pieces.
        
        @rtype: C{boolean}
        @return: if the number of incomplete pieces is greater than half the
            number of connected downloads
        
        """
        
        return len(self.storage.dirty) > (len(self.downloads)/2)


    def cancel_piece_download(self, pieces):
        """Cancel any outstanding requests for the pieces.
        
        @type pieces: C{list} of C{int}
        @param pieces: the list of pieces to cancel
        
        """
        
        if self.endgamemode:
            if self.endgame_queued_pieces:
                for piece in pieces:
                    try:
                        self.endgame_queued_pieces.remove(piece)
                    except:
                        pass
            new_all_requests = []
            for index, nb, nl in self.all_requests:
                if index in pieces:
                    self.storage.request_lost(index, nb, nl)
                else:
                    new_all_requests.append((index, nb, nl))
            self.all_requests = new_all_requests

        for d in self.downloads:
            hit = False
            for index, nb, nl in d.active_requests:
                if index in pieces:
                    hit = True
                    d.connection.send_cancel(index, nb, nl)
                    if not self.endgamemode:
                        self.storage.request_lost(index, nb, nl)
            if hit:
                d.active_requests = [ r for r in d.active_requests
                                      if r[0] not in pieces ]
                d._request_more()
            if not self.endgamemode and d.choked:
                d._check_interests()

    def requeue_piece_download(self, pieces = []):
        """Request more pieces.
        
        @type pieces: C{list} of C{int}
        @param pieces: the list of pieces to requeue

        """
        
        if self.endgame_queued_pieces:
            for piece in pieces:
                if not piece in self.endgame_queued_pieces:
                    self.endgame_queued_pieces.append(piece)
            pieces = self.endgame_queued_pieces
        if self.endgamemode:
            if self.all_requests:
                self.endgame_queued_pieces = pieces
                return
            self.endgamemode = False
            self.endgame_queued_pieces = None
           
        ds = [d for d in self.downloads]
        shuffle(ds)
        for d in ds:
            if d.choked:
                d._check_interests()
            else:
                d._request_more()

    def start_endgame(self):
        """Switch to endgame mode."""
        assert not self.endgamemode
        logger.info('Switching to endgame mode')
        self.endgamemode = True
        assert not self.all_requests
        for d in self.downloads:
            if d.active_requests:
                assert d.interested and not d.choked
            for request in d.active_requests:
                assert not request in self.all_requests
                self.all_requests.append(request)
        for d in self.downloads:
            d.fix_download_endgame()

    def pause(self, flag):
        """Pause or unpause the download.
        
        @type flag: C{boolean}
        @param flag: whether to pause of unpause.
        
        """
        
        self.paused = flag
        if flag:
            for d in self.downloads:
                for index, begin, length in d.active_requests:
                    d.connection.send_cancel(index, begin, length)
                d._letgo()
                d.send_not_interested()
            if self.endgamemode:
                self._reset_endgame()
        else:
            shuffle(self.downloads)
            for d in self.downloads:
                d._check_interests()
                if d.interested and not d.choked:
                    d._request_more()
