/* LineSpectralFrequencies.cpp
 *
 * Copyright (C) 2016 David Weenink
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 djmw 20160421  Initial version
*/

#include "LineSpectralFrequencies.h"
#include "NUM2.h"

#include "oo_DESTROY.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_COPY.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_EQUAL.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_CAN_WRITE_AS_ENCODING.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_WRITE_TEXT.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_WRITE_BINARY.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_READ_TEXT.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_READ_BINARY.h"
#include "LineSpectralFrequencies_def.h"
#include "oo_DESCRIPTION.h"
#include "LineSpectralFrequencies_def.h"

Thing_implement (LineSpectralFrequencies, Sampled, 1);

void structLineSpectralFrequencies :: v_info () {
	structDaata :: v_info ();
	MelderInfo_writeLine (U"Time domain: ", xmin, U" to ", xmax, U" (s).");
	MelderInfo_writeLine (U"Number of frequencies: ", maximumNumberOfFrequencies);
	MelderInfo_writeLine (U"Number of frames: ", nx);
	MelderInfo_writeLine (U"Time step: ", dx, U" (s).");
	MelderInfo_writeLine (U"First frame at: ", x1, U" (s).");
}

void LineSpectralFrequencies_Frame_init (LineSpectralFrequencies_Frame me, int numberOfFrequencies) {
	my frequencies = NUMvector<double> (1, numberOfFrequencies);
	my numberOfFrequencies = numberOfFrequencies;
}

static void LineSpectralFrequencies_init (LineSpectralFrequencies me, double tmin, double tmax, long nt, double dt, double t1, int numberOfFrequencies, double maximumFrequency) {
	my maximumFrequency = maximumFrequency;
	my maximumNumberOfFrequencies = numberOfFrequencies;
	Sampled_init (me, tmin, tmax, nt, dt, t1);
	my d_frames = NUMvector<structLineSpectralFrequencies_Frame> (1, nt);
}

autoLineSpectralFrequencies LineSpectralFrequencies_create (double tmin, double tmax, long nt, double dt, double t1, int numberOfFrequencies, double maximumFrequency) {
	try {
		autoLineSpectralFrequencies me = Thing_new (LineSpectralFrequencies);
		LineSpectralFrequencies_init (me.get(), tmin, tmax, nt, dt, t1, numberOfFrequencies, maximumFrequency);
		return me;
	} catch (MelderError) {
		Melder_throw (U"LineSpectralFrequencies not created.");
	}
}

void LineSpectralFrequencies_drawFrequencies (LineSpectralFrequencies me, Graphics g, double tmin, double tmax, double fmin, double fmax, bool garnish) {
	if (tmax <= tmin) {
		tmin = my xmin;
		tmax = my xmax;
	}
	integer itmin, itmax;
	if (! Sampled_getWindowSamples (me, tmin, tmax, & itmin, & itmax)) {
		return;
	}
	if (fmax <= fmin) {
		double f1max, f2min; 
		autoNUMvector<double> f1 (itmin, itmax), f2 (itmin, itmax);
		for (integer iframe = itmin; iframe <= itmax; iframe ++) {
			f1 [iframe] = my d_frames [iframe]. frequencies [1];
			f2 [iframe] = my d_frames [iframe]. frequencies [my d_frames[iframe]. numberOfFrequencies];
		}
		NUMvector_extrema (f1.peek(), itmin, itmax, & fmin, & f1max);
		NUMvector_extrema (f2.peek(), itmin, itmax, & f2min, & fmax);
	}
	if (fmax == fmin) {
		fmin = 0;
		fmax += 0.5;
	}

	Graphics_setInner (g);
	Graphics_setWindow (g, tmin, tmax, fmin, fmax);
	for (integer iframe = itmin; iframe <= itmax; iframe ++) {
		LineSpectralFrequencies_Frame lsf = & my d_frames[iframe];
		double x = Sampled_indexToX (me, iframe);
		for (long ifreq = 1; ifreq <= lsf -> numberOfFrequencies; ifreq++) {
			double y = lsf -> frequencies [ifreq];
			if (y >= fmin && y <= fmax) { 
				Graphics_speckle (g, x, y);
			}
		}
	}
	Graphics_unsetInner (g);
	if (garnish) {
		Graphics_drawInnerBox (g);
		Graphics_textBottom (g, true, U"Time (seconds)");
		Graphics_textLeft (g, true, U"Frequency (Hz)");
		Graphics_marksBottom (g, 2, true, true, false);
		Graphics_marksLeft (g, 2, true, true, false);
	}
}

autoMatrix LineSpectralFrequencies_downto_Matrix (LineSpectralFrequencies me) {
	try {
		autoMatrix thee = Matrix_create (my xmin, my xmax, my nx, my dx, my x1, 0.5, 0.5 + my maximumNumberOfFrequencies, my maximumNumberOfFrequencies, 1.0, 1.0);
		for (long j = 1; j <= my nx; j++) {
			LineSpectralFrequencies_Frame lsf = & my d_frames[j];
			for (long i = 1; i <= lsf -> numberOfFrequencies; i++) {
				thy z [i][j] = lsf -> frequencies [i];
			}
		}
		return thee;
	} catch (MelderError) {
		Melder_throw (me, U": no Matrix with linear prediction coefficients created.");
	}
}

/* End of file LineSpectralFrequencies.cpp */
