var count = (function toFixed(BigNumber) {
    var start = +new Date(),
        log,
        error,
        u,
        passed = 0,
        total = 0;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function assertException(func, message) {
        var actual;
        total++;
        try {
            func();
        } catch (e) {
            actual = e;
        }
        if (actual && actual.name == 'BigNumber Error') {
            passed++;
            //log('\n Expected and actual: ' + actual);
        } else {
            error('\n Test number: ' + total + ' failed');
            error('\n Expected: ' + message + ' to raise a BigNumber Error.');
            error(' Actual:   ' + (actual || 'no exception'));
            //process.exit();
        }
    }

    function T(expected, value, decimalPlaces){
        assert(String(expected), new BigNumber(value).toFixed(decimalPlaces));
    }

    log('\n Testing toFixed...');

    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : 1E9
    });

    // https://developer.mozilla.org/en/JavaScript/Reference/Global_Objects/Number/toFixed
    // Javascript:            2.26.toFixed(5)  returns  2.26000
    // Javascript:           -2.26.toFixed(5)  returns -2.26
    // BigNumber:  new BigNumber(-2.26).toFixed(5) returns -2.26000

    T('100.0', 99.9512986, 1);
    T('10.0',  9.95036, 1);
    T('1.0', 0.99, 1);
    T('0.10', 0.09906, 2);
    T('0.010', 0.0098034, 3);

    T('NaN', NaN, 2);
    T('Infinity', 1/0, 2);
    T('-Infinity', -1/0, 2);

    T('1111111111111111111111.00000000', '1111111111111111111111', 8);
    T('0.1', 0.1, 1);
    T('0.10', 0.1, 2);
    T('0.100', 0.1, 3);
    T('0.01', 0.01, 2);
    T('0.010', 0.01, 3);
    T('0.0100', 0.01, 4);
    T('0.00', 0.001, 2);
    T('0.001', 0.001, 3);
    T('0.0010', 0.001, 4);
    T('1.0000', 1, 4);
    T('1.0', 1, 1);
    T('1', 1, 0);
    T('12', 12, 0);
    T('1', 1.1, 0);
    T('12', 12.1, 0);
    T('1', 1.12, 0);
    T('12', 12.12, 0);
    T('0.0000006', 0.0000006, 7);
    T('0.00000006', 0.00000006, 8);
    T('0.000000060', 0.00000006, 9);
    T('0.0000000600', 0.00000006, 10);
    T('0', 0, 0);
    T('0.0', 0, 1);
    T('0.00', 0, 2);

    T('-1111111111111111111111.00000000', '-1111111111111111111111', 8);
    T('-0.1', -0.1, 1);
    T('-0.10', -0.1, 2);
    T('-0.100', -0.1, 3);
    T('-0.01', -0.01, 2);
    T('-0.010', -0.01, 3);
    T('-0.0100', -0.01, 4);
    T('-0.00', -0.001, 2);
    T('-0.001', -0.001, 3);
    T('-0.0010', -0.001, 4);
    T('-1.0000', -1, 4);
    T('-1.0', -1, 1);
    T('-1', -1, 0);
    T('-1', -1.1, 0);
    T('-12', -12.1, 0);
    T('-1', -1.12, 0);
    T('-12', -12.12, 0);
    T('-0.00000', -0.0000006, 5);
    T('-0.0000006', -0.0000006, 7);
    T('-0.00000006', -0.00000006, 8);
    T('-0.000000060', -0.00000006, 9);
    T('-0.0000000600', -0.00000006, 10);
    T('0', 0, 0);
    T('0', -0, 0);
    T('0.0', -0, 1);
    T('0.00', -0, 2);
    T('0.00', '-0.0', 2);
    T('0.00', '-0.0000', 2);
    T('0.0000', -0, 4);

    T('1000', 1000, u);
    T('0.00001', 0.00001, u);
    T('0.00001', 0.00001, 5);
    T('0.00000000000000000010', '0.0000000000000000001', 20);
    T('0.00001000000000000', 0.00001, 17);
    T('1.00000000000000000', 1, 17);
    T('1000000000000000128', '1000000000000000128', u);
    T('100000000000000128.0', '100000000000000128', 1);
    T('10000000000000128.00', '10000000000000128', 2);
    T('10000000000000128.00000000000000000000', '10000000000000128', 20);
    T('0', 0, u);
    T('-42.000', -42, 3);
    T('-1000000000000000128', '-1000000000000000128', u);
    T('-0.00000000000000000010', '-0.0000000000000000001', 20);
    T('0.12312312312312300000', '0.123123123123123', 20);

    T('1', 0.5, 0);
    T('-1', -0.5, 0);
    T('1.3', 1.25, 1);
    T('234.2041', 234.20405, 4);
    T('234.2041', '234.204050000000000000000000000000006', 4);

    BigNumber.config({ROUNDING_MODE : 0});

    T('6552606716539719300271040797809220.3', '6552606716539719300271040797809220.237838405', 1);
    T('25605410.260045950231371', '25605410.260045950231370974066', 15);
    T('-65593283.0000', '-65593283', 4);
    T('-2238743875407788208860272230040067273281097722822075126.88000000000', '-2238743875407788208860272230040067273281097722822075126.88', 11);
    T('-1714042659419404211.0000000', '-1714042659419404211', 7);
    T('580861301.000', '580861301', 3);
    T('-495746734.000', '-495746734', 3);
    T('-909962202.00000', '-909962202', 5);
    T('-6588.00', '-6588', 2);
    T('-89235424125324.000000', '-89235424125324', 6);
    T('-0.0001', '-0.0000000000177', 4);
    T('-0.1', '-0.000000000000007', 1);
    T('7086047552021418140460170703933026989776558656.864197000000000', '7086047552021418140460170703933026989776558656.864197', 15);
    T('-0.0000000001', '-0.0000000000000000007707', 10);
    T('6613833500.133407255728', '6613833500.133407255727325566781771', 12);
    T('-0.60', '-0.6', 2);
    T('-5800577761301861465745732694236894834.347199790459214393940', '-5800577761301861465745732694236894834.34719979045921439394', 21);
    T('65495133621931312001260.90', '65495133621931312001260.89870016055223846451749943336585697', 2);
    T('-0.0000001', '-0.0000000000021727264987520611266069544793244453321336', 7);
    T('5138338134667513058267845351654.0073', '5138338134667513058267845351654.007282', 4);
    T('-419591612781413402.1606094651083282', '-419591612781413402.16060946510832813452', 16);
    T('-842984708.00000', '-842984708', 5);
    T('6094679992788973717749530934167375519948195193196.00000000000', '6094679992788973717749530934167375519948195193196', 11);
    T('3070926.00', '3070926', 2);
    T('-49738484.22445560175', '-49738484.2244556017464380224443566512165424404', 11);
    T('0.01', '0.007252154601', 2);
    T('0.01', '0.0000000008450464516429601', 2);
    T('64191341.1515', '64191341.151484', 4);
    T('-0.0000000000000004274', '-0.00000000000000042733957907641417564294', 19);
    T('-307434.0', '-307434', 1);
    T('-7155930111672774753753.84501482871023', '-7155930111672774753753.845014828710226', 14);
    T('-204638.0', '-204638', 1);
    T('67354.81', '67354.801797', 2);
    T('0.000042911675', '0.000042911674899533228107357062', 12);
    T('-57865060735863.5903347110', '-57865060735863.590334711', 10);
    T('-0.00889', '-0.0088857', 5);
    T('655593535442553089245040020344044.000000000000', '655593535442553089245040020344044', 12);
    T('-2077237944012133216588518213.9402', '-2077237944012133216588518213.9401561759783653376', 4);
    T('3354.50', '3354.49314543', 2);
    T('-0.00001', '-0.000000000000091204118667391473401958802734362777499449233', 5);
    T('-2817626898661648124242602088971648489124280903.00000000000000000000000', '-2817626898661648124242602088971648489124280903', 23);
    T('-0.1138', '-0.1137406345875745478881089475981775971743', 4);
    T('372625.84', '372625.836933301', 2);
    T('-0.0001', '-0.0000000000030295', 4);
    T('-0.20133282610692', '-0.2013328261069183505168093992697', 14);
    T('-674022381434.84086', '-674022381434.840858317', 5);
    T('527.32194354633', '527.32194354632981665427056720389485101', 11);
    T('0.0000000195361247565', '0.0000000195361247564008390550978147981600411', 19);
    T('-2163002984808367617.0000', '-2163002984808367617', 4);
    T('-4169664275471960031191605200147601022249457082609196211576.20000000000000000000000', '-4169664275471960031191605200147601022249457082609196211576.2', 23);
    T('-0.00000000000555130', '-0.0000000000055512969161980904744035', 17);
    T('119869956144.00', '119869956144', 2);
    T('1.039', '1.038251882646323867920972358898559', 3);
    T('-2.0', '-2', 1);
    T('7410.197760931', '7410.197760930626987673', 9);
    T('6638110167526.00', '6638110167526', 2);
    T('30974929342171050427055113.00', '30974929342171050427055113', 2);
    T('43480882950025886012415275271059692966.000', '43480882950025886012415275271059692966', 3);
    T('-0.1', '-0.0000004587260575000190182731', 1);
    T('-71828326035.100387343565950831', '-71828326035.100387343565950830073177108125', 18);
    T('-552954262470932495380914766735683420381508430032672.0000000000000', '-552954262470932495380914766735683420381508430032672', 13);
    T('-0.001', '-0.000000006438270583885530971500187308778545384', 3);
    T('-1241063214283490729682249.3542720483313661438', '-1241063214283490729682249.35427204833136614379131332498', 19);
    T('50729670708122152413903981650812767483.000000000000000000', '50729670708122152413903981650812767483', 18);
    T('-0.8', '-0.73086389033', 1);
    T('0.000000001', '0.0000000000035654453258182342271756', 9);
    T('-37426128484776251838279429.00000', '-37426128484776251838279429', 5);
    T('-8858815238086078991845344.00', '-8858815238086078991845344', 2);
    T('715713000263082248.000', '715713000263082248', 3);
    T('-1290.35664066483299', '-1290.35664066483298067128266141875', 14);
    T('-0.00000008453', '-0.0000000845275307904197645672104', 11);
    T('-435325479515.000000', '-435325479515', 6);
    T('-55.026', '-55.0259854614935863191307', 3);
    T('-111827547395883838719196940005789974871114521059.00000000000000000000000', '-111827547395883838719196940005789974871114521059', 23);
    T('-0.80', '-0.8', 2);
    T('-8.17139161520009', '-8.1713916152000874707093159619488493836845705', 14);
    T('769000136593201.985181', '769000136593201.98518095', 6);
    T('31159491831470011.07740834271415', '31159491831470011.0774083427141446999', 14);
    T('89059.299592037965784', '89059.2995920379657839664222340962987295864685207', 15);
    T('0.1', '0.06', 1);
    T('-4570519.0000', '-4570519', 4);
    T('-0.0000001518', '-0.000000151731483260580950465152613377577649917', 10);
    T('-5555052692.071431452', '-5555052692.07143145120571654571684469634938224716', 9);
    T('0.01', '0.0000000735211529638724323811564431527925439', 2);
    T('-230061001007826663915617395042288.847216199000', '-230061001007826663915617395042288.847216199', 12);
    T('-6453992990066244249806.375375876294395', '-6453992990066244249806.3753758762943945461637495930201559', 15);
    T('-0.01', '-0.0000026377450289082142641925', 2);
    T('-363618346.0', '-363618346', 1);
    T('0.01', '0.00000000009', 2);
    T('65432262394897701468848832.7', '65432262394897701468848832.607978068443545803141', 1);
    T('567040263395365029.000', '567040263395365029', 3);
    T('7431958669891786786543853147278.580', '7431958669891786786543853147278.58', 3);
    T('0.0580', '0.0579400993437', 4);
    T('-5364310483511374687804726517179019251624044430754.23807000000000', '-5364310483511374687804726517179019251624044430754.23807', 14);
    T('81096.00', '81096', 2);
    T('-0.000000766905', '-0.00000076690484881136751175461310751844214625312811858607148', 12);
    T('837.0', '837', 1);
    T('-0.000001', '-0.0000000003241605066758965441271096586120022721670744987061', 6);
    T('18848356500282.000000', '18848356500282', 6);
    T('-6865183293234767102.91096748000', '-6865183293234767102.91096748', 11);

    BigNumber.config({ROUNDING_MODE : 1});

    T('0.3', '0.3', 1);
    T('-200258348374.3', '-200258348374.3595802551014614089785610548492885372322083789', 1);
    T('-8996550690041157178188143247448286757711.5580857413', '-8996550690041157178188143247448286757711.55808574133329', 10);
    T('-3172413669280032477.00', '-3172413669280032477', 2);
    T('6547946357820.750067107731812021675', '6547946357820.750067107731812021675351468709784004', 21);
    T('24188393190716631282750407915133817062572333100239.0000000000000000000', '24188393190716631282750407915133817062572333100239', 19);
    T('539982361589798324286450574560330534901309503.82000000', '539982361589798324286450574560330534901309503.82', 8);
    T('829898800701640360552652.0000', '829898800701640360552652', 4);
    T('-1585782773394.4', '-1585782773394.44501382110847', 1);
    T('-7604844176594943774211951887242195107.399', '-7604844176594943774211951887242195107.399576743428669618164', 3);
    T('-0.00', '-0.0000000000000005153003506839', 2);
    T('0.000', '0.00003', 3);
    T('-5454249481540317712.560293859013731302', '-5454249481540317712.56029385901373130215526009738012974642338', 18);
    T('15352607654972198961429295651185206861818141054385638.00000000000000000', '15352607654972198961429295651185206861818141054385638', 17);
    T('91494535411039025233040.292', '91494535411039025233040.29224903220309368507011', 3);
    T('2043369316.0', '2043369316', 1);
    T('-0.0000000782350618457', '-0.0000000782350618457630647420312027682238301541350414', 19);
    T('2122652.0', '2122652', 1);
    T('-0.00841365', '-0.00841365099301981489219310202029642', 8);
    T('0.0', '0.0007035', 1);
    T('-0.00000000000000374916', '-0.0000000000000037491685778894015479084539735777088', 20);
    T('534883638.00', '534883638', 2);
    T('-2.0', '-2', 1);
    T('-5634442247266825358399629799939027370665.0000000000000000000', '-5634442247266825358399629799939027370665', 19);
    T('3331187169219186569122.000000000', '3331187169219186569122', 9);
    T('0.0000000000', '0.0000000000006604395609805032330367635527', 10);
    T('-0.0000000000000576901', '-0.00000000000005769013292086168690493327', 19);
    T('-661.8', '-661.828596629053201916486', 1);
    T('-6073555395665254434249128854999349235744174928042756.1153000000000', '-6073555395665254434249128854999349235744174928042756.1153', 13);
    T('-5013086846966.000', '-5013086846966', 3);
    T('0.004015', '0.00401523226833', 6);
    T('3140295374107374579919222510.1462722', '3140295374107374579919222510.1462722819395532', 7);
    T('-0.000000', '-0.00000000000000799876460379334679831886', 6);
    T('-0.00', '-0.0003', 2);
    T('-0.026760766726884267', '-0.026760766726884267750393307117624838556001925491', 18);
    T('-20821740502968847857923433558797.0899', '-20821740502968847857923433558797.08997487843745', 4);
    T('0.00000000002', '0.000000000020346524414696573703092255317751132892', 11);
    T('45492073832204366341299301624.000000000', '45492073832204366341299301624', 9);
    T('-22166905155222048275755520049666636.000000000000000', '-22166905155222048275755520049666636', 15);
    T('-0.0047749609085292677147', '-0.00477496090852926771478756451899887586311084', 22);
    T('0.0000000000066505751320542', '0.000000000006650575132054209449586010993453988006221893652', 25);
    T('-5987959.1630276586738', '-5987959.163027658673856586', 13);
    T('-1666.00', '-1666', 2);
    T('634686619827216246836355610829.00000', '634686619827216246836355610829', 5);
    T('-4775591515709.469953191321103401', '-4775591515709.4699531913211034017874029213583522324151968262', 18);
    T('-0.00209248872211', '-0.002092488722118037239933953031151990638700454763', 14);
    T('4592542353054190400811174084.8269203906', '4592542353054190400811174084.826920390637', 10);
    T('-2719622410429690231797721788738949434271171.000000000', '-2719622410429690231797721788738949434271171', 9);
    T('7481081126833353571249877375770446139269602514514.000000000000000', '7481081126833353571249877375770446139269602514514', 15);
    T('-717308374940.1455264110', '-717308374940.145526411005821148802846366045624291352081913', 10);
    T('-0.000000007', '-0.000000007385554720977', 9);
    T('-0.000861389378754506340', '-0.000861389378754506340296565696012302905659179798', 21);
    T('-5053251493102402525630380749.678', '-5053251493102402525630380749.6785380807585', 3);
    T('-4.0', '-4', 1);
    T('-91608.490784509096643367058', '-91608.49078450909664336705884736640894006', 21);
    T('0.000', '0.000000008', 3);
    T('-60759551763816025544062865222.0000000000', '-60759551763816025544062865222', 10);
    T('1052.9354577213309', '1052.93545772133093448862729794627180236152', 13);
    T('45267509474.0', '45267509474', 1);
    T('3157315152.00', '3157315152', 2);
    T('-0.5', '-0.5', 1);
    T('0.0057546', '0.005754639024965241832260499185230186', 7);
    T('-0.000', '-0.0006', 3);
    T('3230791870040253164174259086.74466486260', '3230791870040253164174259086.7446648626', 11);
    T('-4344423254203589122222715762047.000000', '-4344423254203589122222715762047', 6);
    T('0.000000', '0.00000000000000000005', 6);
    T('-0.00000000', '-0.0000000000000032995964759', 8);
    T('29342667392627.13800502090', '29342667392627.1380050209', 11);
    T('3553322646.0000', '3553322646', 4);
    T('0.0000000', '0.000000000000711595888783922', 7);
    T('128714169681975260747586179339689165715468750318817.0', '128714169681975260747586179339689165715468750318817', 1);
    T('-203846045708188631.7', '-203846045708188631.714385911896349', 1);
    T('6732937084326281547153681265380857817.680', '6732937084326281547153681265380857817.68', 3);
    T('-80645.000', '-80645', 3);
    T('0.0091340467346905226', '0.0091340467346905226577831006050168789952519932147436292', 19);
    T('765350910755641789513513452874635492095713.000000000000000000', '765350910755641789513513452874635492095713', 18);
    T('7619917853284.00', '7619917853284', 2);
    T('-7584198211496430223.4315410358218089229', '-7584198211496430223.43154103582180892294235232', 19);
    T('-2946097370887492968389.2786430820000', '-2946097370887492968389.278643082', 13);
    T('-400129748.000', '-400129748', 3);
    T('-76362970460470638842230785495147305876.114384970200', '-76362970460470638842230785495147305876.1143849702', 12);
    T('334029398796150851211670406157956664329.0000000000000', '334029398796150851211670406157956664329', 13);
    T('-0.000000000000', '-0.000000000000000000881892851839338487244179354', 12);
    T('-70475264003.0000', '-70475264003', 4);
    T('0.080', '0.0809', 3);
    T('16917787280140934145228.000', '16917787280140934145228', 3);
    T('25909519297999779346.000000000', '25909519297999779346', 9);
    T('-355224.19720319810644', '-355224.19720319810644496969482', 14);
    T('8482635122899359930368306655493255835883236244.0000000000000', '8482635122899359930368306655493255835883236244', 13);
    T('41268046802033799763539654473098094614587.00000000000000000', '41268046802033799763539654473098094614587', 17);
    T('8564065673925609331469179144.0', '8564065673925609331469179144', 1);
    T('-231732424.30838', '-231732424.3083887160672296965273691583237367', 5);
    T('-597740668669015342275289261436747905744110289638.43026849', '-597740668669015342275289261436747905744110289638.430268497449', 8);
    T('8297509140414599.000000', '8297509140414599', 6);
    T('-17320207086212822899.00', '-17320207086212822899', 2);
    T('0.00', '0.000000000002', 2);
    T('4972.517305', '4972.51730546661166855901714', 6);
    T('0.000000', '0.0000000162819653537555368724511902456803362906', 6);
    T('29485932208290816133111088923502731.000000', '29485932208290816133111088923502731', 6);
    T('0.0000', '0.000000000006', 4);

    BigNumber.config({ROUNDING_MODE : 2});

    T('7810971819938620325901899057970512790433.0000000000000000', '7810971819938620325901899057970512790433', 16);
    T('-623395.0', '-623395', 1);
    T('81334094079413997693749807.000000000000', '81334094079413997693749807', 12);
    T('790143966.5', '790143966.406169480041929356421', 1);
    T('-0.000485', '-0.0004851099615478', 6);
    T('843581583868277359263.9721265', '843581583868277359263.97212648230399373761717', 7);
    T('-921038771017147427324121032.000000', '-921038771017147427324121032', 6);
    T('-823.0', '-823', 1);
    T('505.0', '505', 1);
    T('-866703339332502.8636000', '-866703339332502.8636', 7);
    T('46.0', '46', 1);
    T('60042202798790129685.0000000', '60042202798790129685', 7);
    T('-1847392.0000', '-1847392', 4);
    T('-4580251343840510165820631684073999841789879.0000000000000000000000', '-4580251343840510165820631684073999841789879', 22);
    T('3501378778000.539063748827265447', '3501378778000.5390637488272654464266281832626835121535730779', 18);
    T('0.000000004686', '0.000000004685387489140232', 12);
    T('-0.0', '-0.0006', 1);
    T('-445677508.00', '-445677508', 2);
    T('0.0002', '0.00015099822561012723', 4);
    T('-2992505698074499692367794831646886.0000000000000000', '-2992505698074499692367794831646886', 16);
    T('5297873.59', '5297873.5869265978056497873793730268', 2);
    T('72978439324039191443182.00000000000', '72978439324039191443182', 11);
    T('0.66028387', '0.660283869505570207917431271006277981914716620876', 8);
    T('-232008389442485.0', '-232008389442485', 1);
    T('-2.371568', '-2.37156855939356279513952542004820909', 6);
    T('-0.36285', '-0.3628526972092455731139', 5);
    T('-8119.0', '-8119', 1);
    T('171846434549810597775.8595264010', '171846434549810597775.859526401', 10);
    T('-765674124885597.6363120', '-765674124885597.636312053095690367', 7);
    T('2682390283.000', '2682390283', 3);
    T('7799670515191143398434802.0000000000', '7799670515191143398434802', 10);
    T('731916154123696.1935', '731916154123696.193488323779184681349292529227446', 4);
    T('-793941153937799250384382615553258074.00000', '-793941153937799250384382615553258074', 5);
    T('651002.00', '651002', 2);
    T('-61317874871.00', '-61317874871', 2);
    T('8673.00', '8673', 2);
    T('-14490519736189597435728386282.030655', '-14490519736189597435728386282.0306558561264348956674', 6);
    T('0.001', '0.00000000000002', 3);
    T('0.000009', '0.000008920984342', 6);
    T('0.0000920957086395973679873', '0.000092095708639597367987279557138375172816422475', 25);
    T('28125849913667924088615.25407023317734', '28125849913667924088615.2540702331773399078402875044891', 14);
    T('87.55538597', '87.55538596435839691343147', 8);
    T('939705902083425775202905619379.0000000', '939705902083425775202905619379', 7);
    T('7303936681469130614.8345000', '7303936681469130614.8345', 7);
    T('-5831852886782.68010000', '-5831852886782.6801', 8);
    T('-0.0000000', '-0.0000000213769530812', 7);
    T('3939846383149740457.10095420654', '3939846383149740457.1009542065397674933246710194598175764', 11);
    T('590849835.0000', '590849835', 4);
    T('-8275957863.1479060805971940', '-8275957863.14790608059719409463511261267671122', 16);
    T('88601607840.97', '88601607840.97', 2);
    T('-87.0', '-87', 1);
    T('-0.00000000000059590334996', '-0.0000000000005959033499645216206226546715679135941845273', 23);
    T('-20156162002512717731841.00', '-20156162002512717731841', 2);
    T('-7651429939400798.00000000', '-7651429939400798', 8);
    T('4054038.8271920833198564466870907131', '4054038.8271920833198564466870907130849589640579687753824767', 28);
    T('0.000001', '0.0000000003', 6);
    T('275300507469257287911954639302505046260690.99000000', '275300507469257287911954639302505046260690.99', 8);
    T('642144861480907968070.771743458405005477', '642144861480907968070.77174345840500547656346', 18);
    T('-889957451436631506823037014.12137026759445866404763', '-889957451436631506823037014.1213702675944586640476325', 23);
    T('-0.06219606', '-0.062196067085235955835989272145756234722461994836174091995', 8);
    T('-74213.0', '-74213', 1);
    T('-1874944878399124299917948535549036295737418419.000', '-1874944878399124299917948535549036295737418419', 3);
    T('83048495.1430455759922', '83048495.1430455759921014663361993386532235', 13);
    T('-755506912.624491023612801925849', '-755506912.6244910236128019258495078668411577879440686', 21);
    T('5321125710271045378833902102391466551386841083.00000000000', '5321125710271045378833902102391466551386841083', 11);
    T('0.000004284427957821426529823151', '0.0000042844279578214265298231502546967595510818572110299927', 30);
    T('472565886204.00000', '472565886204', 5);
    T('31931010708026796882.0000000', '31931010708026796882', 7);
    T('-4082183405971.00', '-4082183405971', 2);
    T('43182719822263089032.0', '43182719822263089032', 1);
    T('-74632497435577935327787910318876432.000000', '-74632497435577935327787910318876432', 6);
    T('24236715602376164671.99119879565302925270845390802', '24236715602376164671.99119879565302925270845390801276995046', 29);
    T('0.0002148', '0.00021473151048663639093342995904027', 7);
    T('11552045377675783444804198150417717022744929.278', '11552045377675783444804198150417717022744929.2775103900527', 3);
    T('0.0000007575', '0.000000757403681490224', 10);
    T('-855.71', '-855.7117714955299728683318516625249', 2);
    T('40834479174168915057442292427179095154620364684.0000000000000000000', '40834479174168915057442292427179095154620364684', 19);
    T('270321444.000', '270321444', 3);
    T('-30995028620287.03', '-30995028620287.032123746540311643282', 2);
    T('-0.0', '-0.00009', 1);
    T('-602729508268591597438320913.60658950848712687', '-602729508268591597438320913.60658950848712687918', 17);
    T('-548567216043758894618461964890.694772532380790460115', '-548567216043758894618461964890.6947725323807904601157', 21);
    T('82.0', '82', 1);
    T('-55514405531090.06405191', '-55514405531090.06405191938240530830698', 8);
    T('748298620904632711442812.4683', '748298620904632711442812.46822', 4);
    T('682294636690140.7964265', '682294636690140.7964264385841075', 7);
    T('-0.000000000', '-0.00000000000000000291', 9);
    T('-6848806396346459932451282602470846060855488728.540000000000', '-6848806396346459932451282602470846060855488728.54', 12);
    T('5276393.209', '5276393.2086599096026388085281', 3);
    T('0.4', '0.4', 1);
    T('8375483913668.000', '8375483913668', 3);
    T('34817563688185836561554596.15000', '34817563688185836561554596.15', 5);
    T('4367672.009', '4367672.00813881', 3);
    T('-5013698209882323019634267579122589538355461264705883707.000000000000000000000000', '-5013698209882323019634267579122589538355461264705883707', 24);
    T('0.001', '0.000000000000000745707393395393623687701759', 3);
    T('76676945635878017064402467.00000', '76676945635878017064402467', 5);
    T('-382435678755006.0000000', '-382435678755006', 7);
    T('0.5', '0.5', 1);
    T('-3063.0', '-3063', 1);
    T('-3.0', '-3', 1);

    BigNumber.config({ROUNDING_MODE : 3});

    T('0.0000', '0.00000000000584', 4);
    T('-79055907138737329232.165677999092552368', '-79055907138737329232.1656779990925523674', 18);
    T('47206303530045536472421888342339.00000000', '47206303530045536472421888342339', 8);
    T('3974.0', '3974', 1);
    T('1296297430771.39', '1296297430771.391670729445980444292', 2);
    T('0.000016', '0.00001686928031', 6);
    T('-0.00001', '-0.0000000000008', 5);
    T('4883938724383.000', '4883938724383', 3);
    T('-5.4115622307', '-5.411562230628539329345282639155988044655000352687981880842', 10);
    T('-0.01', '-0.0000000000155280115429218804426906540524', 2);
    T('-2730930776035874709906.00000', '-2730930776035874709906', 5);
    T('8419336352549424092369623.00000', '8419336352549424092369623', 5);
    T('52897.4', '52897.4', 1);
    T('-783171007682720320053252521365465473997085.0', '-783171007682720320053252521365465473997085', 1);
    T('0.8', '0.8', 1);
    T('-0.00000000085280', '-0.000000000852796916216780032747621063045628213284', 14);
    T('809486781065644418.9', '809486781065644418.939', 1);
    T('-9.0', '-9', 1);
    T('2399395.00', '2399395', 2);
    T('-243707862514768528758611224374.0000000000000', '-243707862514768528758611224374', 13);
    T('-87906.00', '-87906', 2);
    T('-36206954271055061276640398631031143034683624940748.9970000000000000000000', '-36206954271055061276640398631031143034683624940748.997', 22);
    T('0.00', '0.0000000000565885579252765288683939564608114096713', 2);
    T('-8391656174313988740342135972218668561645437.0', '-8391656174313988740342135972218668561645437', 1);
    T('14571897947566372593360811.00000000000', '14571897947566372593360811', 11);
    T('-473156035623.70', '-473156035623.690373046925674588826', 2);
    T('2798706581453003.177239111', '2798706581453003.17723911167031', 9);
    T('607781292230803273523112111.091726721140', '607781292230803273523112111.09172672114', 12);
    T('-0.3', '-0.3', 1);
    T('3008522535782692059736852844809784198247000184.0', '3008522535782692059736852844809784198247000184', 1);
    T('14227.00', '14227', 2);
    T('7841760307718014116576857993957.0000', '7841760307718014116576857993957', 4);
    T('-0.00001', '-0.00000000000002596', 5);
    T('5247199732982111073.000000000', '5247199732982111073', 9);
    T('-12.0', '-12', 1);
    T('0.000', '0.00005', 3);
    T('-1383703062.0000', '-1383703062', 4);
    T('0.0051019245305169', '0.00510192453051698060366202631871161193', 16);
    T('-45565092550117861497905653558246322559799.0', '-45565092550117861497905653558246322559799', 1);
    T('-0.284', '-0.284', 3);
    T('-8962.00', '-8962', 2);
    T('49335989571.2173023', '49335989571.217302304037', 7);
    T('-9.0', '-9', 1);
    T('-26112007394118580909080488694986628.00000000', '-26112007394118580909080488694986628', 8);
    T('-0.0000001', '-0.0000000000004', 7);
    T('0.0000002341', '0.0000002341509570562590239', 10);
    T('14533914.08264220677', '14533914.0826422067727128413377656933218789408', 11);
    T('42170538403476604541324854403569069340562310909332579546.37600000000000000', '42170538403476604541324854403569069340562310909332579546.376', 17);
    T('-4.0', '-4', 1);
    T('-67678179481337765.66210000', '-67678179481337765.6621', 8);
    T('257419830944838.0000000', '257419830944838', 7);
    T('-324773537769898472911.277781339985241561477923', '-324773537769898472911.2777813399852415614779220499583265', 24);
    T('-528485032702878171887338835619232399832.00000000000', '-528485032702878171887338835619232399832', 11);
    T('-710189041499049633659311234.000', '-710189041499049633659311234', 3);
    T('643084101102033.063753808679', '643084101102033.0637538086790219378107', 12);
    T('825884431.000', '825884431', 3);
    T('-543871854882673630364514781674939771106.38551140781759', '-543871854882673630364514781674939771106.385511407817581261', 14);
    T('-798526.00', '-798526', 2);
    T('297795786834144418744.0000', '297795786834144418744', 4);
    T('0.00000', '0.0000000000016163586667107', 5);
    T('0.0000000000', '0.00000000000007848361792826553641', 10);
    T('-5.007308350876984680', '-5.007308350876984679530067590680338870257350613322626', 18);
    T('-427.00', '-427', 2);
    T('-515119.313431282058', '-515119.31343128205773478896638', 12);
    T('-8.49', '-8.4804', 2);
    T('2477.751471782877', '2477.751471782877726695914405313733773564569949039123496962', 12);
    T('0.00000000', '0.000000000000000065892', 8);
    T('7138.0', '7138.013', 1);
    T('0.0000000000', '0.0000000000000004333337165355354317801382736832134', 10);
    T('8658428891542726719576602828199407563445702034.00000000', '8658428891542726719576602828199407563445702034', 8);
    T('0.3', '0.3', 1);
    T('-56313246520419173728621429021902681217573545.50000000000000', '-56313246520419173728621429021902681217573545.5', 14);
    T('-23.039752413764', '-23.03975241376398747770561587976', 12);
    T('-19445726051618345902355920469.00000', '-19445726051618345902355920469', 5);
    T('-0.1', '-0.040277481632393097542018589044285854989493888417936', 1);
    T('-0.000000000001', '-0.000000000000000000288835285344547651291408208502608124', 12);
    T('79030772762.92000', '79030772762.92', 5);
    T('5700622112017639331911483394352164374087.00000000000000000', '5700622112017639331911483394352164374087', 17);
    T('-90770899291595539313072309812380381240470505.000000000000', '-90770899291595539313072309812380381240470505', 12);
    T('-41647595.5561', '-41647595.55603', 4);
    T('222575273379018396575101654735581517852852286433973.4686000000000000', '222575273379018396575101654735581517852852286433973.4686', 16);
    T('56275673822913529965328835.15006015303', '56275673822913529965328835.150060153034', 11);
    T('-88903.69011778', '-88903.690117778621375708068659403797174645788', 8);
    T('-115251.00', '-115251', 2);
    T('-0.00001', '-0.0000079321934570619592061', 5);
    T('9222185695996633788650562039735238902841130971959266.000000', '9222185695996633788650562039735238902841130971959266', 6);
    T('-327381709334.18527398150845', '-327381709334.185273981508440153589241628279291', 14);
    T('1.2070939', '1.207093941595', 7);
    T('5631244336598787152545539244.614', '5631244336598787152545539244.614953046957418773850871532', 3);
    T('7.585318', '7.58531893327', 6);
    T('509177.0', '509177', 1);
    T('-5922122769788.000000', '-5922122769788', 6);
    T('204517097255774161136650103483266240865357614.000000000000', '204517097255774161136650103483266240865357614', 12);
    T('-0.008817568208764', '-0.00881756820876361559226942398227', 15);
    T('80111694.00', '80111694', 2);
    T('-453349.000', '-453349', 3);
    T('85122981744559018.00000000', '85122981744559018', 8);
    T('0.027007', '0.027007420598954545854969297272268125483242465539', 6);
    T('-2699105308.4', '-2699105308.3051738674733271523825325743', 1);
    T('-0.000000000001', '-0.00000000000000883409579454508262166084385', 12);

    BigNumber.config({ROUNDING_MODE : 4, EXPONENTIAL_AT : 0});

    T('733744593401073823825766410831877679446.0000000000000000000', '733744593401073823825766410831877679446', 19);
    T('-64.6849459', '-64.6849458687691227978', 7);
    T('-0.000000', '-0.00000000009', 6);
    T('-62537287527837589298857228059657673223234916.95923265430000000', '-62537287527837589298857228059657673223234916.9592326543', 17);
    T('3393668096256773847245721315080265089731.000000', '3393668096256773847245721315080265089731', 6);
    T('0.0', '0.0000000000000056674956638008432348702401392', 1);
    T('72516372734.6', '72516372734.6447', 1);
    T('-418.28', '-418.2800731793741351', 2);
    T('0.00', '0.001', 2);
    T('8366217346845756726.00000000', '8366217346845756726', 8);
    T('-0.000000', '-0.0000000000000092034548636370987112234384736726', 6);
    T('0.35', '0.35474830751442135112334772517193392', 2);
    T('64703289793894.5830', '64703289793894.58296866', 4);
    T('-0.000000000000000036', '-0.000000000000000036461242408590182363418943891', 18);
    T('5494508405056449117588.631948', '5494508405056449117588.631948458033233759999', 6);
    T('-0.0', '-0.00393971618499838726739122333520030506235698', 1);
    T('375581290738585826632.00000000', '375581290738585826632', 8);
    T('254.96635275802300887', '254.96635275802300886544776010389418575738792480979736', 17);
    T('21492347.69467571391498624445', '21492347.6946757139149862444482880595559468', 20);
    T('313576441324233.0000000', '313576441324233', 7);
    T('-8617.57089319166', '-8617.57089319166205141433276411273552139017', 11);
    T('0.00000000014928376395', '0.00000000014928376394768755724252424022348932365406839448223', 20);
    T('2.497624215622999568569', '2.49762421562299956856937732557160934323114203547832675', 21);
    T('-288056106511636272271570523193926801058294502.000', '-288056106511636272271570523193926801058294502', 3);
    T('-4798171567575228724.140', '-4798171567575228724.1400144613946894', 3);
    T('8.49008', '8.49007655210654', 5);
    T('-56633605.000', '-56633605', 3);
    T('-4147381274685866.81753', '-4147381274685866.8175313', 5);
    T('-534460490015293367127173277346694900936058.0000', '-534460490015293367127173277346694900936058', 4);
    T('182707431911537249021116759327712693311898345504618668.43327000000000000000000', '182707431911537249021116759327712693311898345504618668.43327', 23);
    T('210005324615278.4586839690045963321032', '210005324615278.458683969004596332103244549279', 22);
    T('779837001772884165637922377221951347134695.644834', '779837001772884165637922377221951347134695.6448338', 6);
    T('-0.000001', '-0.00000064188301390033596845335767993853284632527964514979079', 6);
    T('13.0', '13', 1);
    T('0.0000001269', '0.0000001269060795648365813491128357427111184222', 10);
    T('18446632248354.00', '18446632248354', 2);
    T('-1229249.79', '-1229249.7897249259', 2);
    T('49082.0', '49082', 1);
    T('-61.0', '-61', 1);
    T('-893.0', '-893', 1);
    T('5002282278.56974877690066484', '5002282278.569748776900664839184116538222902', 17);
    T('41372.00', '41372', 2);
    T('-4732022445962399687294885123498809.7625585825095', '-4732022445962399687294885123498809.7625585825095', 13);
    T('-55484242.036895641', '-55484242.036895641', 9);
    T('-41427133134.52583323427907663268339', '-41427133134.525833234279076632683393992706825', 23);
    T('0.0', '0.00004300614085218825243480119971669264977421', 1);
    T('-472025754597316278339412186866.7010659789', '-472025754597316278339412186866.701065978877597089729906019843', 10);
    T('2.50869', '2.50869082352024406', 5);
    T('-87154.000', '-87154', 3);
    T('-0.0000000000071859183950976', '-0.0000000000071859183950975905578927950225183102336392', 25);
    T('-10827138340266.44', '-10827138340266.44057558913', 2);
    T('0.0000702', '0.0000701871613973530533582225769977', 7);
    T('-4878237667448133503521416059497.376626000000', '-4878237667448133503521416059497.376626', 12);
    T('8984687671347686046766702.01181862480', '8984687671347686046766702.0118186248', 11);
    T('0.00000', '0.00000000076447711430809658905538', 5);
    T('-184825.87743766370213607', '-184825.87743766370213606656669525883145102', 17);
    T('-147007021210433212254058959161555.747000000000', '-147007021210433212254058959161555.747', 12);
    T('641778680063060681850989295.4', '641778680063060681850989295.44394602812788754296411563', 1);
    T('0.00', '0.000000000000000261099857', 2);
    T('-69168585539.981086198312', '-69168585539.9810861983124008597329', 12);
    T('-2441638.37765', '-2441638.37765405543', 5);
    T('0.00', '0.00000005', 2);
    T('-17865480328519059654833853633.668588983', '-17865480328519059654833853633.6685889825124607147114', 9);
    T('5552449880075592383242782.42', '5552449880075592383242782.42273775283', 2);
    T('-3.47445833771303714', '-3.4744583377130371421642122331960174443', 17);
    T('-5543608156416096833838191852961329666675.00000000000000', '-5543608156416096833838191852961329666675', 14);
    T('-6836042.000', '-6836042', 3);
    T('0.00000912', '0.0000091177854552839774258855562071866', 8);
    T('55986.00', '55986', 2);
    T('0.0', '0.02', 1);
    T('-0.0000', '-0.00000001', 4);
    T('889659799897707059043156.0', '889659799897707059043156', 1);
    T('0.000000', '0.00000000000000003', 6);
    T('0.00000000000008179581450', '0.00000000000008179581449515246320423213581212698189913041', 23);
    T('-981.4210', '-981.421', 4);
    T('-3455913777070673553360330519337.00000000000000', '-3455913777070673553360330519337', 14);
    T('0.00000000081700', '0.00000000081700228716634187537384600725', 14);
    T('4822037.000', '4822037', 3);
    T('-0.0000000064360437778', '-0.0000000064360437777672775206464043731', 19);
    T('5290480.66496927901282867', '5290480.66496927901282867027658062129', 17);
    T('-408612952552043095.000', '-408612952552043095', 3);
    T('-513.0', '-513', 1);
    T('-14964.000', '-14964', 3);
    T('-19903753117563849440806175537514112075543387952.00000', '-19903753117563849440806175537514112075543387952', 5);
    T('249443.0175143', '249443.0175142727704683419749', 7);
    T('-74343756463586673907431837633.24', '-74343756463586673907431837633.23687651', 2);
    T('-27.679', '-27.6785', 3);
    T('-64259172643579684888737979.618833554815821332098', '-64259172643579684888737979.618833554815821332097716331', 21);
    T('17230955766539391.00000000', '17230955766539391', 8);
    T('582751886981240.177', '582751886981240.177383845390864465', 3);
    T('191002031658424.51', '191002031658424.51247456500221220552332700327795408', 2);
    T('-479015175846069070145.000000000', '-479015175846069070145', 9);
    T('135507.00', '135507', 2);
    T('-1819380757840.7931545894175', '-1819380757840.79315458941752033110677996441946221', 13);
    T('56729395152942371.34113220', '56729395152942371.3411322', 8);
    T('0.00000', '0.000000009', 5);
    T('231631.31091406798', '231631.310914067979514934639396351953448', 11);
    T('0.00', '0.00044659', 2);
    T('-637697600918508531049573689760309.5119477000000000', '-637697600918508531049573689760309.5119477', 16);
    T('-6.19247283', '-6.19247282501186063445037314', 8);

    BigNumber.config({ROUNDING_MODE : 5});

    T('0.00', '0.00000000000000070362', 2);
    T('682377946933004839.0000000', '682377946933004839', 7);
    T('0.000000000', '0.00000000000000301722508588270616971784651380892069087869575', 9);
    T('-356330174906.7', '-356330174906.737986270704', 1);
    T('771875969530483104163034745359742192923504.000000000000000', '771875969530483104163034745359742192923504', 15);
    T('-85.57612133364', '-85.576121333642541652128540737082', 11);
    T('-796870619982006783.00000000', '-796870619982006783', 8);
    T('985819.0', '985819', 1);
    T('0.000000', '0.00000000000000000007093034243958', 6);
    T('-0.0000', '-0.0000000006400178083107075310177579449', 4);
    T('-105570971372984.0', '-105570971372984', 1);
    T('4387745564661852845308.94229815083918', '4387745564661852845308.9422981508391799473309882', 14);
    T('-3978043797116373.2', '-3978043797116373.159631', 1);
    T('5.8961281424853232622595', '5.8961281424853232622594846548715356650909645537495738043051', 22);
    T('-0.0000081539970551874384495', '-0.000008153997055187438449482684680130340977472796176926', 25);
    T('-65.0', '-65', 1);
    T('-24012940205869.000', '-24012940205869', 3);
    T('0.000000', '0.000000000000000066985234999507396', 6);
    T('2935194283452.35738951157832206907124', '2935194283452.357389511578322069071242729133894', 23);
    T('542060.7771', '542060.77713479586821060591807145320317', 4);
    T('890815036528241639534803534813.592588608837168425865025', '890815036528241639534803534813.59258860883716842586502472472', 24);
    T('685877749980391051.17', '685877749980391051.1715', 2);
    T('47820177163147405231185068730213.894293', '47820177163147405231185068730213.89429254869826585947429821', 6);
    T('1536364305.00', '1536364305', 2);
    T('-85825125.5810188', '-85825125.5810188298', 7);
    T('-4983288214158634188.4572989', '-4983288214158634188.4572988600705359104', 7);
    T('0.0000000000', '0.0000000000000000017929073', 10);
    T('-1684136337600384671.00000000', '-1684136337600384671', 8);
    T('567733137127609543.000000', '567733137127609543', 6);
    T('607675679770387.000', '607675679770387', 3);
    T('43727409067319.0', '43727409067319', 1);
    T('-77274559773372606.00', '-77274559773372606', 2);
    T('-28855643486070486857836.00000', '-28855643486070486857836', 5);
    T('-915356570870.00', '-915356570870.00041832252262829437239', 2);
    T('0.0', '0.00000000000004843552301428650828', 1);
    T('-33295218517967296146307127.00000', '-33295218517967296146307127', 5);
    T('279738383433666359352.0000000', '279738383433666359352', 7);
    T('-30674028853766084158066396602.97', '-30674028853766084158066396602.97484778446', 2);
    T('-0.323', '-0.32275988418995752362737918723', 3);
    T('6129049976563004.000', '6129049976563004', 3);
    T('-6.0', '-6', 1);
    T('-66243945975700325606152500782471023774538548318.0000000000000000000000', '-66243945975700325606152500782471023774538548318', 22);
    T('-138286369547644243875.0000000', '-138286369547644243875', 7);
    T('5099535.00', '5099535', 2);
    T('675.0', '675', 1);
    T('0.0002717', '0.0002716602757', 7);
    T('4649780225385110311483722.0000', '4649780225385110311483722', 4);
    T('-22143045418555913412107884765016867198665971.10000000000000000000000', '-22143045418555913412107884765016867198665971.1', 23);
    T('-71055639496050521326328010675.00000000', '-71055639496050521326328010675', 8);
    T('0.00000000', '0.000000000077562570903960521', 8);
    T('-0.000000021017500', '-0.000000021017500243085615932617', 15);
    T('0.000000000000000024697', '0.00000000000000002469732084636233381651368431', 21);
    T('760509941556107507425043304787283113918044183872.000000000000000000000000', '760509941556107507425043304787283113918044183872', 24);
    T('-553489807182304143570302092614431846999972192129146.0608020000000000', '-553489807182304143570302092614431846999972192129146.060802', 16);
    T('4616442457861540834390983504493684542810.99300000000', '4616442457861540834390983504493684542810.993', 11);
    T('5836715007847225.0000', '5836715007847225', 4);
    T('1056180.997', '1056180.99676966515555844225990248546', 3);

    BigNumber.config({ROUNDING_MODE : 6});

    T('-6614662975368684488885953285955838893900074215956.00', '-6614662975368684488885953285955838893900074215956', 2);
    T('8642393776898.000', '8642393776898', 3);
    T('3612207878528.1106344698085142865', '3612207878528.1106344698085142865438924419', 19);
    T('977910386.93', '977910386.9343537', 2);
    T('-8937937492688941768401276851642629965923372.0', '-8937937492688941768401276851642629965923372', 1);
    T('-8327876772303865517454752161.4376112752921904774', '-8327876772303865517454752161.43761127529219047742910886', 19);
    T('-27707.00', '-27707', 2);
    T('-7287595765894242586375.92700435484', '-7287595765894242586375.927004354837723619581861634319', 11);
    T('-0.00000', '-0.000000000000023670858459165486137218740358207859', 5);
    T('574676836802209077.64156', '574676836802209077.641563875647945277493356873895', 5);
    T('-3041038455237222898218053653661.87952947815', '-3041038455237222898218053653661.87952947815', 11);
    T('-628244132307.000', '-628244132307', 3);
    T('316566935167341670725238.000', '316566935167341670725238', 3);
    T('-77953301569468294124105528.0', '-77953301569468294124105528', 1);
    T('0.0046', '0.00460227833968584', 4);
    T('4323265763616518980379929221104.0', '4323265763616518980379929221104', 1);
    T('1674500565014237781637959673471730398.1120033', '1674500565014237781637959673471730398.1120032995511774', 7);
    T('872559441430499650732600166.00000', '872559441430499650732600166', 5);
    T('-87858304.0000', '-87858304', 4);
    T('-4158788.000', '-4158788', 3);
    T('3983930437416823631395.9093', '3983930437416823631395.90934402583657', 4);
    T('-14.5531937', '-14.5531936852106573016020290135814233645752955297443336', 7);
    T('0.00000000002201', '0.00000000002200503795474854372849141363413146996972', 14);
    T('0.0000000000000', '0.00000000000000000894955653982033503846831364474746320232', 13);
    T('61564779.00', '61564779', 2);
    T('-7.0', '-7', 1);
    T('-0.000000000000028455976228', '-0.00000000000002845597622815028653703372125435938812845106', 24);
    T('728982423193598397582409707715766595902844.0000000', '728982423193598397582409707715766595902844', 7);
    T('14538075860529.2697937480', '14538075860529.269793748', 10);
    T('0.000', '0.00000000000000000007021566096', 3);
    T('-5136066535080.86830591678842264063462546263', '-5136066535080.86830591678842264063462546262988980471309228558', 29);
    T('-3026751112367460839746524832112404665.000000000000000', '-3026751112367460839746524832112404665', 15);
    T('-6694275398250090562.00', '-6694275398250090562', 2);
    T('-50643997750937725516578696636829589244661366402743.000000000', '-50643997750937725516578696636829589244661366402743', 9);
    T('2197374413.79714', '2197374413.7971355344377249354470626290268959873', 5);
    T('0.0', '0.005020192658713355', 1);
    T('-254956356194995318790657817617390203938.180000000', '-254956356194995318790657817617390203938.18', 9);
    T('0.000', '0.000000000090726', 3);
    T('-644164371869124953225087.14694842524000', '-644164371869124953225087.14694842524', 14);
    T('-0.00000157175189018', '-0.0000015717518901763532026126676445445', 17);
    T('883545726717772254369.08868386', '883545726717772254369.0886838593581297250598', 8);
    T('-8890332541208932844712909996569216422867971798.000000000000000', '-8890332541208932844712909996569216422867971798', 15);
    T('-0.000000000000', '-0.000000000000000019481454049470983547912235', 12);
    T('740944636504591431790473918513.0000', '740944636504591431790473918513', 4);
    T('-5679793084762496855854965276883350654245022549131067599715.800000', '-5679793084762496855854965276883350654245022549131067599715.8', 6);
    T('51991398836962962151.4032644162159', '51991398836962962151.4032644162159099221', 13);
    T('66958838053011544557752193.00000', '66958838053011544557752193', 5);
    T('-670579843.2118208913941229137016', '-670579843.21182089139412291370155773837719883548843299002', 22);
    T('-153613650899356538114572207.94248678450152160836', '-153613650899356538114572207.9424867845015216083607156497', 20);
    T('-737846251706379871052.2861168', '-737846251706379871052.2861168267591416289887782899835', 7);
    T('-65908055.0', '-65908055', 1);
    T('3237039120306392313.80906374769', '3237039120306392313.80906374769434099092272015749692609', 11);
    T('83640693821.00', '83640693821', 2);
    T('-921776606869800052510163986912.6037663480501', '-921776606869800052510163986912.603766348050146856757561176111', 13);
    T('6303883932563373.0', '6303883932563373', 1);
    T('6373283811.5454531478622699', '6373283811.5454531478622698611321575271', 16);
    T('6097017834070937658.000000000', '6097017834070937658', 9);
    T('24524.17471335606758880', '24524.1747133560675887965670636565353165', 17);
    T('82671976754257522256837287487178420454352.00000', '82671976754257522256837287487178420454352', 5);
    T('-4134121508878287.32516638661', '-4134121508878287.32516638661094488', 11);
    T('706494960751225536339363333.135485602348', '706494960751225536339363333.13548560234793', 12);
    T('0.0000000', '0.00000000000000000005260494554647051', 7);
    T('-2172823046.605754378126140867773679', '-2172823046.60575437812614086777367899344184998336', 24);
    T('7182216670065915047.3', '7182216670065915047.265', 1);
    T('-0.0001', '-0.00011347856', 4);
    T('-1620.4750509689199484099', '-1620.475050968919948409915128915793493745056891952647', 19);
    T('-2107001274569567393.00000000', '-2107001274569567393', 8);
    T('0.00', '0.0000000004', 2);
    T('-0.000', '-0.00000000000000000009622600511499459674774308', 3);
    T('-1749656594942211427.14080828808148814976439530', '-1749656594942211427.1408082880814881497643953004161', 26);
    T('-2483383057229762.00000000', '-2483383057229762', 8);
    T('0.000000', '0.0000005', 6);
    T('0.5', '0.5', u);

    T('-0', '-0.5', 0);

    T('10.5', '10.5', u);
    T('0.05', '0.05', u);
    T('0.4', '0.4', u);
    T('0.6', '0.6', u);
    T('1.5', '1.5', u);
    T('-1.5', '-1.5', u);

    BigNumber.config({ROUNDING_MODE : 4});

    T('-535564000.00', '-535564000', 2);
    T('-80000000.000', '-80000000', 3);
    T('-240.0', '-240', 1);
    T('-7520000000.0', '-7520000000', 1);
    T('306550.000', '306550', 3);
    T('800000000.0', '800000000', 1);
    T('-454000000.00000', '-454000000', 5);
    T('60.0', '60', 1);
    T('-4700000.000', '-4700000', 3);
    T('25000.00', '25000', 2);
    T('6847.0', '6847', 1);
    T('780.0', '780', 1);
    T('-50.0', '-50', 1);
    T('700.0', '700', 1);
    T('412598000.0000', '412598000', 4);
    T('-84581600.0', '-84581600', 1);
    T('5590000.00', '5590000', 2);
    T('-66000.0', '-66000', 1);
    T('600.0', '600', 1);
    T('-513335000.000', '-513335000', 3);
    T('-40.0', '-40', 1);
    T('3401.00', '3401', 2);
    T('8600.00', '8600', 2);
    T('-5410000.0', '-5410000', 1);
    T('5527170.0', '5527170', 1);
    T('-3710700.0', '-3710700', 1);
    T('-13230000.0', '-13230000', 1);

    T('-52536202527.13891931933498708496518913773517016037224', '-5.253620252713891931933498708496518913773517016037224E10', u);
    T('0.0000000000000008761383898703999300297605784533730922', '8.761383898703999300297605784533730922E-16', u);
    T('-0.0000000000000000000000000000000000004', '-4E-37', u);
    T('2832044194940.47654015529986948208', '2.83204419494047654015529986948208E12', u);
    T('0.00000000000000000000000064668155275864837', '6.4668155275864837E-25', u);
    T('4.34128088694', '4.34128088694E0', u);
    T('-851957.6572615611436542', '-8.519576572615611436542E5', u);
    T('-89097468286.2561077879598798580253771544265682053', '-8.90974682862561077879598798580253771544265682053E10', u);
    T('-0.000000000000000000000000000000000000000000000030409767', '-3.0409767E-47', u);
    T('0.000000000000000000004295112484112357722747956581254', '4.295112484112357722747956581254E-21', u);
    T('-839203207475651.6542808578', '-8.392032074756516542808578E14', u);
    T('-0.00000000000000000078216', '-7.8216E-19', u);
    T('-1782290274935701247734.21262413032385', '-1.78229027493570124773421262413032385E21', u);
    T('31669451104144801337076433457.18511076', '3.166945110414480133707643345718511076E28', u);
    T('0.000000000000000000000000000000000000472351852761', '4.72351852761E-37', u);
    T('-50155220217523568901083623.78842580174913602672593731', '-5.015522021752356890108362378842580174913602672593731E25', u);
    T('-291.37', '-2.9137E2', u);
    T('-52474611936456205886027195352961212383.2279441143', '-5.24746119364562058860271953529612123832279441143E37', u);
    T('-326.9376463292543', '-3.269376463292543E2', u);
    T('0.00000000000044', '4.4E-13', u);
    T('0.000000087792449', '8.7792449E-8', u);
    T('497835793870987132484033.938845920610565887398960253648', '4.97835793870987132484033938845920610565887398960253648E23', u);
    T('-33960439442302770058.74863962', '-3.396043944230277005874863962E19', u);
    T('0.00000000000000000000000000000000000000020694696587293782', '2.0694696587293782E-40', u);
    T('-0.8904319539', '-8.904319539E-1', u);
    T('-86429552798.4', '-8.64295527984E10', u);
    T('0.000000000000000000000000000003', '3E-30', u);
    T('108595082003321932.8178949736608320767', '1.085950820033219328178949736608320767E17', u);
    T('611126.73674534242367484008848341500334', '6.1112673674534242367484008848341500334E5', u);
    T('4311.0597590893026685282662406924', '4.3110597590893026685282662406924E3', u);
    T('0.000000000000000000000000000000000000000073309221', '7.3309221E-41', u);
    T('-6864844854510.144926572184', '-6.864844854510144926572184E12', u);
    T('-0.038', '-3.8E-2', u);
    T('0.00000000000000794946369399591', '7.94946369399591E-15', u);
    T('56020.6861005', '5.60206861005E4', u);
    T('-0.00000000000896738912765063552834206018847', '-8.96738912765063552834206018847E-12', u);
    T('0.00000000006755925523685348357189087468', '6.755925523685348357189087468E-11', u);
    T('0.000000000000000000000000000000000072', '7.2E-35', u);
    T('0.00000000000000000000000000000000000000000000000000006128', '6.128E-53', u);
    T('-27010668344373144545282.061712584328482438690936897', '-2.7010668344373144545282061712584328482438690936897E22', u);
    T('-0.000066720042169954586184178645', '-6.6720042169954586184178645E-5', u);
    T('-0.00000000000000000000000000000000000003516', '-3.516E-38', u);
    T('0.007458', '7.458E-3', u);
    T('0.0670339647456587432138292225', '6.70339647456587432138292225E-2', u);
    T('-49.59', '-4.959E1', u);
    T('5.94', '5.94E0', u);
    T('9185110977234279483303912931774514.67', '9.18511097723427948330391293177451467E33', u);
    T('-127.11408592808048991219551891022999276273412536879573107525', '-1.2711408592808048991219551891022999276273412536879573107525E2', u);
    T('0.00000013', '1.3E-7', u);
    T('-0.00000000000000000000000000000000156', '-1.56E-33', u);
    T('22', '2.2E1', u);
    T('6644378197598065.943168375926', '6.644378197598065943168375926E15', u);
    T('-0.00000000000000465403429687260725429254606083299909', '-4.65403429687260725429254606083299909E-15', u);
    T('0.00000653586429286761580785105', '6.53586429286761580785105E-6', u);
    T('-779132060.497697052423647343175672388834', '-7.79132060497697052423647343175672388834E8', u);
    T('0.0000000000000000085996828590423169591835759758087548466494', '8.5996828590423169591835759758087548466494E-18', u);
    T('-0.000000000000256285784', '-2.56285784E-13', u);
    T('0.0000000000000017947', '1.7947E-15', u);
    T('0.000000000060671249811126224597672813512109', '6.0671249811126224597672813512109E-11', u);
    T('-800627106331596202.032250866963247864855868', '-8.00627106331596202032250866963247864855868E17', u);
    T('-98402249.9358755191632299360339880127190049662097654206', '-9.84022499358755191632299360339880127190049662097654206E7', u);
    T('-0.000000000000000000000014149882794818263963', '-1.4149882794818263963E-23', u);
    T('-7060324311175919534626797409919718746390.10555', '-7.06032431117591953462679740991971874639010555E39', u);
    T('821393911651350506858922608971571483.39201', '8.2139391165135050685892260897157148339201E35', u);
    T('-721517626817.43734235970422357278172', '-7.2151762681743734235970422357278172E11', u);
    T('-5124.504476910146380807047771', '-5.124504476910146380807047771E3', u);
    T('0.0000000005697544867401308959870722352682904', '5.697544867401308959870722352682904E-10', u);
    T('-0.00012871208613777', '-1.2871208613777E-4', u);
    T('8567233928771894245.8242651905951092028701228506224699', '8.5672339287718942458242651905951092028701228506224699E18', u);
    T('-0.0000000094881461545668728', '-9.4881461545668728E-9', u);
    T('6045245.020177069527457656633132622', '6.045245020177069527457656633132622E6', u);
    T('-7210271571379007035869444.1417241308887367042323559687840525', '-7.2102715713790070358694441417241308887367042323559687840525E24', u);
    T('-2620220239077.66', '-2.62022023907766E12', u);
    T('-0.001916564704980402478', '-1.916564704980402478E-3', u);
    T('5371.947750065160966666612070818997', '5.371947750065160966666612070818997E3', u);
    T('0.00000077036439397954895237124076', '7.7036439397954895237124076E-7', u);
    T('-0.000000000002280137669447351084609656999367909206747800792', '-2.280137669447351084609656999367909206747800792E-12', u);
    T('635945056777163867933996244.44612913038', '6.3594505677716386793399624444612913038E26', u);
    T('5', '5E0', u);
    T('48309.56', '4.830956E4', u);
    T('2138.45865823930129961216391905305999251296833471869532168', '2.13845865823930129961216391905305999251296833471869532168E3', u);
    T('3706073870.96925990345144', '3.70607387096925990345144E9', u);
    T('-798.62', '-7.9862E2', u);
    T('-521968106907526.65373750732297', '-5.2196810690752665373750732297E14', u);
    T('0.7224996555', '7.224996555E-1', u);
    T('-0.00000000000059452644215310796452483703709', '-5.9452644215310796452483703709E-13', u);
    T('350818063006798017016671125.0434068891113', '3.508180630067980170166711250434068891113E26', u);
    T('7968708424180299716816047189442601.3735762244', '7.9687084241802997168160471894426013735762244E33', u);
    T('-0.5355520816318664361284739093852209293628562271329701', '-5.355520816318664361284739093852209293628562271329701E-1', u);
    T('-906.9275233025623299867798', '-9.069275233025623299867798E2', u);
    T('0.00045587', '4.5587E-4', u);
    T('0.000004', '4E-6', u);
    T('5857276704537340524.1431417907437', '5.8572767045373405241431417907437E18', u);
    T('-3843554476637626791.87474208180361', '-3.84355447663762679187474208180361E18', u);
    T('-60535943484683045495384222.313768035329647860053456457477', '-6.0535943484683045495384222313768035329647860053456457477E25', u);
    T('0.0000000000000004444869762788836674404079577885262473', '4.444869762788836674404079577885262473E-16', u);
    T('-0.005874931444844844350595746989373586841269114316', '-5.874931444844844350595746989373586841269114316E-3', u);
    T('0.09', '9E-2', u);
    T('86280010336.8017418950328178', '8.62800103368017418950328178E10', u);
    T('227901.6305936588696902705189877626178015851261755', '2.279016305936588696902705189877626178015851261755E5', u);
    T('-0.00002923', '-2.923E-5', u);
    T('451.1251729802129507', '4.511251729802129507E2', u);
    T('-2586077212890917651232310958.88201646914374247083', '-2.58607721289091765123231095888201646914374247083E27', u);
    T('-0.08', '-8E-2', u);
    T('7.996020827302694833122463304879649129610781362', '7.996020827302694833122463304879649129610781362E0', u);
    T('-0.00000000000000000000000007933204829049175909186498799838', '-7.933204829049175909186498799838E-26', u);
    T('-0.0000000000001882498592930501880542338117957263473489255', '-1.882498592930501880542338117957263473489255E-13', u);
    T('-1172052648945437104144.7753679481957698417677', '-1.1720526489454371041447753679481957698417677E21', u);
    T('287937985683292603.441526238031', '2.87937985683292603441526238031E17', u);
    T('-0.0000000000134355091714731837192225455005', '-1.34355091714731837192225455005E-11', u);
    T('0.000000000000000000000000000026170541344981279884209612', '2.6170541344981279884209612E-29', u);
    T('0.0041336403', '4.1336403E-3', u);
    T('8955011.51436146124079765380599870608206410057', '8.95501151436146124079765380599870608206410057E6', u);
    T('76343535604093', '7.6343535604093E13', u);
    T('-579109804573982925.9', '-5.791098045739829259E17', u);
    T('-16248730541202.41645330499728414237866592099124270393892244', '-1.624873054120241645330499728414237866592099124270393892244E13', u);
    T('299078130777696473774921407.8921386301', '2.990781307776964737749214078921386301E26', u);
    T('0.00000000000005008659860139108', '5.008659860139108E-14', u);
    T('-277409247924.47', '-2.7740924792447E11', u);
    T('-0.516', '-5.16E-1', u);
    T('-0.0000000000000000000000056338952424532568447556', '-5.6338952424532568447556E-24', u);
    T('-0.000000217955', '-2.17955E-7', u);
    T('8905335075021645253458124.63415988932826552', '8.90533507502164525345812463415988932826552E24', u);
    T('-0.0044', '-4.4E-3', u);
    T('-0.2373963', '-2.373963E-1', u);
    T('-0.79088027341157244765444311372339', '-7.9088027341157244765444311372339E-1', u);
    T('0.0026', '2.6E-3', u);
    T('-835918193339364260673226.934', '-8.35918193339364260673226934E23', u);
    T('-0.0009', '-9E-4', u);
    T('6', '6E0', u);
    T('-314214420212243283843403549.4470864901529365489404218274', '-3.142144202122432838434035494470864901529365489404218274E26', u);
    T('0.00000000791545594251', '7.91545594251E-9', u);
    T('191942230233.90984962570015373025283255557', '1.9194223023390984962570015373025283255557E11', u);
    T('-0.0000000000078881545375574067765343323', '-7.8881545375574067765343323E-12', u);
    T('-16802261058.36705', '-1.680226105836705E10', u);
    T('0.0000000000000056017241520290338574647512149076190238', '5.6017241520290338574647512149076190238E-15', u);
    T('-0.00000000003763962', '-3.763962E-11', u);
    T('-0.000000000511737717217408462541465239', '-5.11737717217408462541465239E-10', u);
    T('-0.000000000000000000000000000000000000000050368224614', '-5.0368224614E-41', u);
    T('-0.018634708832478724474964945976701', '-1.8634708832478724474964945976701E-2', u);
    T('6', '6E0', u);
    T('-0.0000000000000000000745072058021090744018544591692', '-7.45072058021090744018544591692E-20', u);
    T('724257535284104810664.4073', '7.242575352841048106644073E20', u);
    T('12769.45510988046008277708155820993631711046', '1.276945510988046008277708155820993631711046E4', u);
    T('631804730657678655927.137998704033924856965', '6.31804730657678655927137998704033924856965E20', u);
    T('0.00000000000000000000000000000000798840870767288241343529097', '7.98840870767288241343529097E-33', u);
    T('1696465.807207', '1.696465807207E6', u);
    T('0.00000000000000000000000000000079804991500258221476735155', '7.9804991500258221476735155E-31', u);
    T('-0.000000000000000000000598137708', '-5.98137708E-22', u);
    T('0.000000000000000000000000065555369684822', '6.5555369684822E-26', u);
    T('-35599704691194312186568857.06582774697883475886191', '-3.559970469119431218656885706582774697883475886191E25', u);
    T('-0.0000000000000794908227013', '-7.94908227013E-14', u);
    T('-0.0000000499730429264884151845723427055364', '-4.99730429264884151845723427055364E-8', u);
    T('0.00000000000000064', '6.4E-16', u);
    T('-572705', '-5.72705E5', u);
    T('-0.000000000559274332', '-5.59274332E-10', u);
    T('2924074735520802294144158.124247615661785268262895466645', '2.924074735520802294144158124247615661785268262895466645E24', u);
    T('0.00000000000000000000001792', '1.792E-23', u);
    T('-839757046612045970742723218746.08104510687271206453', '-8.3975704661204597074272321874608104510687271206453E29', u);
    T('607849.09099201775466906352465117043', '6.0784909099201775466906352465117043E5', u);
    T('-4654202085161.7295919153', '-4.6542020851617295919153E12', u);
    T('350765025122274578259836720029858792277.93', '3.5076502512227457825983672002985879227793E38', u);
    T('-0.000000000000000000000000000000000000528363561265895', '-5.28363561265895E-37', u);
    T('-7', '-7E0', u);
    T('4.57', '4.57E0', u);
    T('-1214468.83483364', '-1.21446883483364E6', u);
    T('-1.75890665039235813638669857754779113', '-1.75890665039235813638669857754779113E0', u);
    T('-0.0001520617551991821956064', '-1.520617551991821956064E-4', u);
    T('0.000000000000000000000000000000000000000000316862530006', '3.16862530006E-43', u);
    T('3', '3E0', u);
    T('82', '8.2E1', u);
    T('-0.0000000000000000000000000000000773334716331177010647839', '-7.73334716331177010647839E-32', u);
    T('708885924.332539635179', '7.08885924332539635179E8', u);
    T('-64497329.71357360701588', '-6.449732971357360701588E7', u);
    T('-0.7187', '-7.187E-1', u);
    T('0.00000000000000000000000000000000000038135349688', '3.8135349688E-37', u);
    T('0.00000000000000000000000000000000182717532099214054', '1.82717532099214054E-33', u);
    T('19869092956779031884883390565185379047759848968501117153.4', '1.98690929567790318848833905651853790477598489685011171534E55', u);
    T('620913861673.81857852920469094171232255417901159', '6.2091386167381857852920469094171232255417901159E11', u);
    T('235690494.2', '2.356904942E8', u);
    T('1952.848789', '1.952848789E3', u);
    T('0.0000000000000000000000593857592909', '5.93857592909E-23', u);
    T('-0.00000000000000688558900049482997723', '-6.88558900049482997723E-15', u);
    T('-0.00000000000000000000000000007338347173289341804737', '-7.338347173289341804737E-29', u);
    T('0.000000000000000202545915248826284620971210675', '2.02545915248826284620971210675E-16', u);
    T('-724.280380099716494328413118123', '-7.24280380099716494328413118123E2', u);
    T('-0.000000000000018969640124102058547183921', '-1.8969640124102058547183921E-14', u);
    T('429623.23189016762963579', '4.2962323189016762963579E5', u);
    T('-4440068176059478189.38961985114362', '-4.44006817605947818938961985114362E18', u);
    T('388081523018426906543780329974051543542101.8221256689223182', '3.880815230184269065437803299740515435421018221256689223182E41', u);
    T('454795752022.4749683607481963197156049975854', '4.547957520224749683607481963197156049975854E11', u);
    T('-5.97', '-5.97E0', u);
    T('-0.0000000651', '-6.51E-8', u);
    T('0.000307481649349842471', '3.07481649349842471E-4', u);
    T('-0.06005081686761154653757227654183441415861126', '-6.005081686761154653757227654183441415861126E-2', u);
    T('0.00000000000000000000063579065', '6.3579065E-22', u);
    T('900814801036731395557.6', '9.008148010367313955576E20', u);
    T('4999', '4.999E3', u);
    T('-7.3827865388155464662280547066599644534268122420403', '-7.3827865388155464662280547066599644534268122420403E0', u);
    T('-17278399836268786075488221.51', '-1.727839983626878607548822151E25', u);
    T('-0.0660146', '-6.60146E-2', u);
    T('-781591906217028941735734533.343797893312168609257042055753', '-7.81591906217028941735734533343797893312168609257042055753E26', u);
    T('-6278408649', '-6.278408649E9', u);
    T('-0.000000000001', '-1E-12', u);
    T('0.0018', '1.8E-3', u);
    T('-437548970.255395047611134138987663791895761194803418316513197', '-4.37548970255395047611134138987663791895761194803418316513197E8', u);
    T('-147885337323804329368530390734663845475933.77830435', '-1.4788533732380432936853039073466384547593377830435E41', u);
    T('-753868987492754718849345711047082845957.640888486201188367', '-7.53868987492754718849345711047082845957640888486201188367E38', u);
    T('-340168909098957.1241', '-3.401689090989571241E14', u);
    T('-0.000000000000000000000000000000000000000585436735853', '-5.85436735853E-40', u);
    T('0.0000000000000000756425339599187041535729604907751951301172', '7.56425339599187041535729604907751951301172E-17', u);
    T('4632870.933', '4.632870933E6', u);
    T('-4563559260.01', '-4.56355926001E9', u);
    T('-0.00000000000000000000000294407952905593673865', '-2.94407952905593673865E-24', u);
    T('0.00000000039555331064814', '3.9555331064814E-10', u);
    T('-0.0000000000000000000000000000000066', '-6.6E-33', u);
    T('0.00000000000000000000000002796', '2.796E-26', u);
    T('0.006420974922080335', '6.420974922080335E-3', u);
    T('62.09', '6.209E1', u);
    T('0.000000000000000000000000000000000000000092019', '9.2019E-41', u);
    T('31705406794.02881', '3.170540679402881E10', u);
    T('85.877', '8.5877E1', u);
    T('6183.522312', '6.183522312E3', u);
    T('4299.15237475410462987616810012961', '4.29915237475410462987616810012961E3', u);
    T('21900171506665946787994.9913441726189153092', '2.19001715066659467879949913441726189153092E22', u);
    T('64432017665108304653.37748', '6.443201766510830465337748E19', u);
    T('-859958206403439.8555459676', '-8.599582064034398555459676E14', u);
    T('332.8856507997622171514594697481203953250774320862536', '3.328856507997622171514594697481203953250774320862536E2', u);
    T('-0.0000000000000000000000000000007425', '-7.425E-31', u);
    T('8.72361', '8.72361E0', u);
    T('-0.0000000000000000000000000000000000000000000000025453', '-2.5453E-48', u);
    T('129360586484988816847066041509256285.9834653851528560934', '1.293605864849888168470660415092562859834653851528560934E35', u);
    T('-0.000000000000007222', '-7.222E-15', u);
    T('-8098841403716381679540887.942710395311867043476849928102', '-8.098841403716381679540887942710395311867043476849928102E24', u);
    T('-596359917165924096.74516895354130304259526279190192', '-5.9635991716592409674516895354130304259526279190192E17', u);
    T('383728239984415462011.156111377322400391221501061013146359', '3.83728239984415462011156111377322400391221501061013146359E20', u);
    T('-0.0000089', '-8.9E-6', u);
    T('-0.000000000000000000000000008687857', '-8.687857E-27', u);
    T('0.0000361135882323181', '3.61135882323181E-5', u);
    T('-790530028343151500857311598079293.168169782009497538', '-7.90530028343151500857311598079293168169782009497538E32', u);
    T('120280775320592531520673579931715496.413898', '1.20280775320592531520673579931715496413898E35', u);
    T('-0.00000000000000330496', '-3.30496E-15', u);
    T('0.00000000000000000000000000000000000000000009', '9E-44', u);
    T('0.493800687846', '4.93800687846E-1', u);
    T('0.00000000000000000000000000000000000000000000428917279063', '4.28917279063E-45', u);
    T('2439161986343305.3977219293086130659', '2.4391619863433053977219293086130659E15', u);
    T('0.00000000000000000000000000000001236416', '1.236416E-32', u);
    T('-0.00000074534400936482826343789037997822705', '-7.4534400936482826343789037997822705E-7', u);
    T('-21.1', '-2.11E1', u);
    T('2572114652026268493653871881146558385361920.28081168806628264', '2.57211465202626849365387188114655838536192028081168806628264E42', u);
    T('17124033483109159133818176268510442295426753249741457.875034', '1.7124033483109159133818176268510442295426753249741457875034E52', u);
    T('-0.5129122224574351008', '-5.129122224574351008E-1', u);
    T('-0.007927', '-7.927E-3', u);
    T('-0.0000920899747', '-9.20899747E-5', u);
    T('-0.004894338278755', '-4.894338278755E-3', u);
    T('0.0000000000000000000061703', '6.1703E-21', u);
    T('2326089407921647116369956641187720.969', '2.326089407921647116369956641187720969E33', u);
    T('1166197836238509', '1.166197836238509E15', u);
    T('3671781131154060908657.603137323342926541681254', '3.671781131154060908657603137323342926541681254E21', u);
    T('0.0000000000000000000000000000000000000000000002260279', '2.260279E-46', u);
    T('-0.00000000000000000000000000000000000000000000000000404', '-4.04E-51', u);
    T('-652408413688213856070194.6216304473505375385271', '-6.524084136882138560701946216304473505375385271E23', u);
    T('-6437.3786', '-6.4373786E3', u);
    T('-0.00024', '-2.4E-4', u);
    T('-0.000000000000000000000000000004', '-4E-30', u);
    T('0.0000000000000009079496181315516965', '9.079496181315516965E-16', u);
    T('-33759.06239', '-3.375906239E4', u);
    T('-8249214.8143', '-8.2492148143E6', u);
    T('76023.094878', '7.6023094878E4', u);
    T('4750348149604126293082788358593492149985525371.3055161011554', '4.7503481496041262930827883585934921499855253713055161011554E45', u);
    T('-347056242813575345437098678.19382', '-3.4705624281357534543709867819382E26', u);
    T('-0.0044334994986839', '-4.4334994986839E-3', u);
    T('-77579240.8814471085', '-7.75792408814471085E7', u);
    T('0.00000000000005880190315', '5.880190315E-14', u);
    T('5512594339298216692.2504515931248375537394173161332', '5.5125943392982166922504515931248375537394173161332E18', u);
    T('897599613823052.78098021722556', '8.9759961382305278098021722556E14', u);
    T('491849.74046716999671925', '4.9184974046716999671925E5', u);
    T('0.00000000000000000000000000000026', '2.6E-31', u);
    T('83831959917154625137362002.826519817779', '8.3831959917154625137362002826519817779E25', u);
    T('0.0000000004824', '4.824E-10', u);
    T('-0.0000000000000000000539954390007731983666483', '-5.39954390007731983666483E-20', u);
    T('-5', '-5E0', u);
    T('-0.0000000000000000000000000000000052105973', '-5.2105973E-33', u);
    T('-51180818493453796222392326680632879', '-5.1180818493453796222392326680632879E34', u);
    T('-0.0000000000000000000000000000000000240888314887894116', '-2.40888314887894116E-35', u);
    T('-3.9884198', '-3.9884198E0', u);
    T('-0.00292509420999319058434270507', '-2.92509420999319058434270507E-3', u);
    T('-0.0471410502674988132847954859528693281', '-4.71410502674988132847954859528693281E-2', u);
    T('-0.000383921003997213038329810979734179482203337059136937948', '-3.83921003997213038329810979734179482203337059136937948E-4', u);
    T('868847355738186906225977.83933455357403132497074451153826473', '8.6884735573818690622597783933455357403132497074451153826473E23', u);
    T('-74428485419.93786255558241', '-7.442848541993786255558241E10', u);
    T('9.08353695983999662727523423580372295', '9.08353695983999662727523423580372295E0', u);
    T('0.000000000000000000000000000000000000174340386756706026938', '1.74340386756706026938E-37', u);
    T('0.00000000000000000000000003488223852390463813101', '3.488223852390463813101E-26', u);
    T('6235893648937978707207460050453092026130252190198479.3', '6.2358936489379787072074600504530920261302521901984793E51', u);
    T('288685778491241.9787664674', '2.886857784912419787664674E14', u);
    T('0.000132721936833718', '1.32721936833718E-4', u);
    T('-878.0561026970809281319399445768', '-8.780561026970809281319399445768E2', u);
    T('0.00000000000000000000000000000164205860568335794182', '1.64205860568335794182E-30', u);
    T('198.2447277', '1.982447277E2', u);
    T('594893202379402597086528939812.686961108921', '5.94893202379402597086528939812686961108921E29', u);
    T('815799379961431517563630941318305822656819.1819089536', '8.157993799614315175636309413183058226568191819089536E41', u);
    T('7990092124827731099787724282235162054016305443.9448', '7.9900921248277310997877242822351620540163054439448E45', u);
    T('663687681604087645522022865972.07146532571', '6.6368768160408764552202286597207146532571E29', u);
    T('318669729285258750300170721357707298590338.45839229964442', '3.1866972928525875030017072135770729859033845839229964442E41', u);
    T('-555699450886771049.424695515896915466268294356460403', '-5.55699450886771049424695515896915466268294356460403E17', u);
    T('-0.00000000000000000000000000000000000000000001095071857', '-1.095071857E-44', u);
    T('-9150620505556.07974725572726628538', '-9.15062050555607974725572726628538E12', u);
    T('-0.00000000003194584', '-3.194584E-11', u);
    T('-1913473.7214461866163373699149123', '-1.9134737214461866163373699149123E6', u);
    T('226008983260172522.282738382841159863', '2.26008983260172522282738382841159863E17', u);
    T('62674367141762718476.245255', '6.2674367141762718476245255E19', u);
    T('-2', '-2E0', u);
    T('-0.00000009178', '-9.178E-8', u);
    T('0.00000000000000000629836181986423672328404795028', '6.29836181986423672328404795028E-18', u);
    T('93586.02926647672', '9.358602926647672E4', u);
    T('-0.0000000000000000000000005', '-5E-25', u);
    T('-9207421318.2448691844836734269513121808514798', '-9.2074213182448691844836734269513121808514798E9', u);
    T('-0.00000007307221392104978', '-7.307221392104978E-8', u);
    T('-0.0000000000000066886943766783493967753683408289', '-6.6886943766783493967753683408289E-15', u);
    T('-543304673129193.8', '-5.433046731291938E14', u);
    T('-0.000000000000000136', '-1.36E-16', u);
    T('-5548881134560.03611729206353518', '-5.54888113456003611729206353518E12', u);
    T('12277666390.0592100569639588009142346505770647', '1.22776663900592100569639588009142346505770647E10', u);
    T('-49886360742145249688090891445025578471363.463145336476', '-4.9886360742145249688090891445025578471363463145336476E40', u);
    T('289734846955085106069130291953704075.5686911', '2.897348469550851060691302919537040755686911E35', u);
    T('0.00000000046461', '4.6461E-10', u);
    T('-62109960217702169495599.7856590374740584451730995867043611', '-6.21099602177021694955997856590374740584451730995867043611E22', u);
    T('-0.00543899591266188574678', '-5.43899591266188574678E-3', u);
    T('-0.000000000000655098180667', '-6.55098180667E-13', u);
    T('16003803057504466354568876.721', '1.6003803057504466354568876721E25', u);
    T('4.54663', '4.54663E0', u);
    T('-0.0000090363058536', '-9.0363058536E-6', u);
    T('0.00000000000000000000000000000000000000000000683169237734869', '6.83169237734869E-45', u);
    T('-0.000007999285941677', '-7.999285941677E-6', u);
    T('-0.0013289268682646', '-1.3289268682646E-3', u);
    T('-676279445952.4758664434048894660778454544', '-6.762794459524758664434048894660778454544E11', u);
    T('-0.00000000000009075526387', '-9.075526387E-14', u);
    T('0.00000086974105048792231226652741684821263913885781939473', '8.6974105048792231226652741684821263913885781939473E-7', u);
    T('0.000000066874978182439602812', '6.6874978182439602812E-8', u);
    T('69460738505039771437626.5674', '6.94607385050397714376265674E22', u);
    T('-496842225856320580039833.32412974875059', '-4.9684222585632058003983332412974875059E23', u);
    T('-0.00000000089027363409225605342063967', '-8.9027363409225605342063967E-10', u);
    T('0.000000000000000000000000000000000000000000013177025997876', '1.3177025997876E-44', u);
    T('0.00000000000000000000000000000000000000000000000000767', '7.67E-51', u);
    T('0.00000000000000000000051646585515', '5.1646585515E-22', u);
    T('-77.0523658189', '-7.70523658189E1', u);
    T('-2381011509343581230.1741806527782135341132', '-2.3810115093435812301741806527782135341132E18', u);
    T('15355.8', '1.53558E4', u);
    T('-6.99530180722252243035032591', '-6.99530180722252243035032591E0', u);
    T('0.00000000000000000000000000000000000174189621024', '1.74189621024E-36', u);
    T('-0.30429214832427054994311306579079362506404', '-3.0429214832427054994311306579079362506404E-1', u);
    T('0.0000053681637', '5.3681637E-6', u);
    T('-0.0000000000000000000580794214565094', '-5.80794214565094E-20', u);
    T('0.00000000000000000000000000000000000000718', '7.18E-39', u);
    T('0.00308106507', '3.08106507E-3', u);
    T('-778826601182.03', '-7.7882660118203E11', u);
    T('0.07', '7E-2', u);
    T('-18429554314600.2502715009477102928634264509718651532399233353', '-1.84295543146002502715009477102928634264509718651532399233353E13', u);
    T('1', '1E0', u);
    T('-0.007671287604999', '-7.671287604999E-3', u);
    T('7887909.816542736', '7.887909816542736E6', u);
    T('749', '7.49E2', u);
    T('0.00000000000000000000000000000000000000000000000000000000701', '7.01E-57', u);
    T('19.83', '1.983E1', u);
    T('6.09402754', '6.09402754E0', u);
    T('424650217237.5591', '4.246502172375591E11', u);
    T('-0.000049564535268301', '-4.9564535268301E-5', u);
    T('-7149629423.847728553', '-7.149629423847728553E9', u);
    T('886867.3497161434434805387149126261591812', '8.868673497161434434805387149126261591812E5', u);
    T('5.26322950282017332236528', '5.26322950282017332236528E0', u);
    T('-814.59758571746371817362831870047066741472348685917', '-8.1459758571746371817362831870047066741472348685917E2', u);
    T('-0.3417309109605504', '-3.417309109605504E-1', u);
    T('0.000000057039362368628403676633', '5.7039362368628403676633E-8', u);
    T('-71880.393174366874491172325', '-7.1880393174366874491172325E4', u);
    T('0.000000000000000000000000000000022058', '2.2058E-32', u);
    T('-2.4', '-2.4E0', u);
    T('228988936752996967.25464233211', '2.2898893675299696725464233211E17', u);
    T('-5080707117552421.40044791548207099220514344', '-5.08070711755242140044791548207099220514344E15', u);
    T('-0.00000000000000000000000000000000016', '-1.6E-34', u);
    T('-0.00003625156420604915204366585038046', '-3.625156420604915204366585038046E-5', u);
    T('464637966596604669241542203619416.3216624756173', '4.646379665966046692415422036194163216624756173E32', u);
    T('0.9087655476734088654458558078829027886267006357', '9.087655476734088654458558078829027886267006357E-1', u);
    T('38664124727384955142.767044860367795251877128009517', '3.8664124727384955142767044860367795251877128009517E19', u);
    T('-0.04762788797', '-4.762788797E-2', u);
    T('-0.00000005574134895606662617', '-5.574134895606662617E-8', u);
    T('-0.0003', '-3E-4', u);
    T('4722723.88281334997860228789', '4.72272388281334997860228789E6', u);
    T('-23995836727515866107660.4330429236850044610474', '-2.39958367275158661076604330429236850044610474E22', u);
    T('0.0000045734704681293061832577', '4.5734704681293061832577E-6', u);
    T('-0.0000000000000000000000000000000182740798445987382441', '-1.82740798445987382441E-32', u);
    T('255100617558.86883972286820342761', '2.5510061755886883972286820342761E11', u);
    T('0.064171494091240642357663125', '6.4171494091240642357663125E-2', u);
    T('0.000000000000000000000000000000000000000000000294', '2.94E-46', u);
    T('-281546266519431813455047.7', '-2.815462665194318134550477E23', u);
    T('-570257381822964838010877265', '-5.70257381822964838010877265E26', u);
    T('0.00000505461928', '5.05461928E-6', u);
    T('-0.0000000000000079383014842488', '-7.9383014842488E-15', u);
    T('4902.6138179376447898508', '4.9026138179376447898508E3', u);
    T('-0.0000715984', '-7.15984E-5', u);
    T('0.000035824586044205294062412584', '3.5824586044205294062412584E-5', u);
    T('0.000000000000000000000125960459847724978955518957', '1.25960459847724978955518957E-22', u);
    T('0.028825661929052', '2.8825661929052E-2', u);
    T('1', '1E0', u);
    T('5515129992862248795518.6683048631996974', '5.5151299928622487955186683048631996974E21', u);
    T('-0.00000000000000000000000000000000000000000080094098284978', '-8.0094098284978E-43', u);
    T('0.0087634344566767751845291518372154140005282350718533575', '8.7634344566767751845291518372154140005282350718533575E-3', u);
    T('-0.00000000000000000000000000087483068736', '-8.7483068736E-28', u);
    T('-0.00000000000000000000003486', '-3.486E-23', u);
    T('0.00000000000000000000000000000070504040638', '7.0504040638E-31', u);
    T('8303509586343.2443141446152924467090885760476794115914091746', '8.3035095863432443141446152924467090885760476794115914091746E12', u);
    T('-64990.51997', '-6.499051997E4', u);
    T('-655394931216227242937434026113830.5297263576723976532', '-6.553949312162272429374340261138305297263576723976532E32', u);
    T('-30636947149027568.3784157180980445146', '-3.06369471490275683784157180980445146E16', u);
    T('-0.00000000000000000000000000000000000001208233572003', '-1.208233572003E-38', u);
    T('-0.0000000000000011794152801815', '-1.1794152801815E-15', u);
    T('1071378.3293', '1.0713783293E6', u);
    T('-0.000000000003361089193578981719907614', '-3.361089193578981719907614E-12', u);
    T('0.000000000000000007517993065604', '7.517993065604E-18', u);
    T('-0.00000000000000000000000000523129187652012620622569673798', '-5.23129187652012620622569673798E-27', u);
    T('0.0005386441729450879234', '5.386441729450879234E-4', u);
    T('1956653.41224492311730384723219211', '1.95665341224492311730384723219211E6', u);
    T('6896', '6.896E3', u);
    T('0.0000000000744534597556481260346395203468455303201235567973', '7.44534597556481260346395203468455303201235567973E-11', u);
    T('-292388805074045762115572488335407720919180203157229382665.559', '-2.92388805074045762115572488335407720919180203157229382665559E56', u);
    T('0.0000000000161486329717318200324782054218311', '1.61486329717318200324782054218311E-11', u);
    T('-0.000000007268', '-7.268E-9', u);
    T('-301600646696.465', '-3.01600646696465E11', u);
    T('433163.236977123475732966', '4.33163236977123475732966E5', u);
    T('0.00000000000002', '2E-14', u);
    T('77436648026076439631578359.44359576567', '7.743664802607643963157835944359576567E25', u);
    T('548548914365689512587508708513155531344527.51142574657215155', '5.4854891436568951258750870851315553134452751142574657215155E41', u);
    T('-34.831073457011317265120446795022234120450280951265737281038', '-3.4831073457011317265120446795022234120450280951265737281038E1', u);
    T('-845.7703', '-8.457703E2', u);
    T('-4855853589907390754493', '-4.855853589907390754493E21', u);
    T('-0.000000000000000000000000000000000000000000000036', '-3.6E-47', u);
    T('-0.00000000000000000000000000000000000000000000000000000065', '-6.5E-55', u);
    T('-45187081404585.42115773362507', '-4.518708140458542115773362507E13', u);
    T('-0.000323660879015412727290006070402289', '-3.23660879015412727290006070402289E-4', u);
    T('0.0000000026140558644933293588', '2.6140558644933293588E-9', u);
    T('845511912406451608.452066253429799', '8.45511912406451608452066253429799E17', u);
    T('0.0000000004438916316433824501475', '4.438916316433824501475E-10', u);
    T('-1.475814', '-1.475814E0', u);
    T('-0.5938633', '-5.938633E-1', u);
    T('-0.0000000000000000000000000000001', '-1E-31', u);
    T('-5261012744021996107539939108.077860134', '-5.261012744021996107539939108077860134E27', u);
    T('-0.0000000000000000000000000000000000000000000048147761739', '-4.8147761739E-45', u);
    T('-768545505862649955130.58486163645042434421335', '-7.6854550586264995513058486163645042434421335E20', u);
    T('-8704255.29406077171056688276987554', '-8.70425529406077171056688276987554E6', u);
    T('-0.00000000049', '-4.9E-10', u);
    T('-0.0000274005502702', '-2.74005502702E-5', u);
    T('0.0000000000000000000000000000899782060188', '8.99782060188E-29', u);
    T('2665032804405760472644.6495163233309696', '2.6650328044057604726446495163233309696E21', u);
    T('-0.000000000000000030326850268940214303', '-3.0326850268940214303E-17', u);
    T('0.0000612367883', '6.12367883E-5', u);
    T('303609712270628568267916524914027172014687335.2872885723795', '3.036097122706285682679165249140271720146873352872885723795E44', u);
    T('0.000475307072678608341', '4.75307072678608341E-4', u);
    T('-0.889983831753818831790592735', '-8.89983831753818831790592735E-1', u);
    T('-0.00066889767092099512366082', '-6.6889767092099512366082E-4', u);
    T('-82', '-8.2E1', u);
    T('0.000000000000000021', '2.1E-17', u);
    T('-0.000000000000000000000000000000000000000000000179184', '-1.79184E-46', u);
    T('-0.00000153', '-1.53E-6', u);
    T('1515.50800161303015840906362948489', '1.51550800161303015840906362948489E3', u);
    T('-0.000000065937444987966499626906986814867840872', '-6.5937444987966499626906986814867840872E-8', u);
    T('-8.9396126344740419192589432399447838129620466', '-8.9396126344740419192589432399447838129620466E0', u);
    T('-14954.93625698463935831538650997', '-1.495493625698463935831538650997E4', u);
    T('9174945', '9.174945E6', u);
    T('0.000000000000000000000000000000002802944759681178397', '2.802944759681178397E-33', u);
    T('-43661507562386856914454118482535735284321568.8260068', '-4.36615075623868569144541184825357352843215688260068E43', u);
    T('-0.0000000000000053', '-5.3E-15', u);
    T('0.000000000000000088261', '8.8261E-17', u);
    T('-3.5', '-3.5E0', u);
    T('-0.0000000892690903256739084318630208528', '-8.92690903256739084318630208528E-8', u);
    T('-6407107615634731793662054.643592714114255633352022546773', '-6.407107615634731793662054643592714114255633352022546773E24', u);
    T('0.000000000000008', '8E-15', u);
    T('-4113623798185597530479042260306405.169879539509', '-4.113623798185597530479042260306405169879539509E33', u);
    T('-0.0000000000000000000000005949543693626028147554', '-5.949543693626028147554E-25', u);
    T('-16486202099937.552352885944', '-1.6486202099937552352885944E13', u);
    T('6904428088990.55342459', '6.90442808899055342459E12', u);
    T('-82273669444.3011132490462', '-8.22736694443011132490462E10', u);
    T('-0.00000000000416644372132', '-4.16644372132E-12', u);
    T('-0.051168126904004539033', '-5.1168126904004539033E-2', u);
    T('-0.000034398785', '-3.4398785E-5', u);
    T('-714608413034.632588711', '-7.14608413034632588711E11', u);
    T('31.7594171820640021654374655039', '3.17594171820640021654374655039E1', u);
    T('87.4', '8.74E1', u);
    T('0.00000000000000000000000483', '4.83E-24', u);
    T('24.61775061167239603598621023987484627713284203', '2.461775061167239603598621023987484627713284203E1', u);
    T('-0.0000000000000000000000000000005620531', '-5.620531E-31', u);
    T('36514.26972606881494', '3.651426972606881494E4', u);
    T('-0.00000000000000000000000000000000000000804143880989122008869', '-8.04143880989122008869E-39', u);
    T('-0.0000000000000000000000000000000000000736020246', '-7.36020246E-38', u);
    T('-0.00000000000510047724869054', '-5.10047724869054E-12', u);
    T('905030924720385935277.8553538426257506345310051398387757', '9.050309247203859352778553538426257506345310051398387757E20', u);
    T('-9.18', '-9.18E0', u);
    T('32679523370662392206339543790659744869129539414', '3.2679523370662392206339543790659744869129539414E46', u);
    T('90.2504892', '9.02504892E1', u);
    T('-0.0000000000000000000000050446054506203873', '-5.0446054506203873E-24', u);
    T('136427761390644472964118311128712026.442', '1.36427761390644472964118311128712026442E35', u);

    assert('-1478635511205065720411441088234634087783353090531769.175603436034231639743395165930222246484607747' +
    '0460440794925193040344223656739741115687100160257607638', new BigNumber('-1478635511205065720411441088234634087783' +
    '353090531769.175603436034231639743395165930222246484607747046044079492519304034422365673974111568710016025760763' +
    '83664408162624867351605015229777701990865326902932136430587015275752666401315258528531504569482239').toFixed(100));

    BigNumber.config({ERRORS : true});

    T('123.450', '12.345e1', new BigNumber('3'));
    T('123.45', '12.345e1', null);
    T('123.45', '12.345e1', u);
    T('123', '12.345e1', 0);
    T('123', '12.345e1', -0);
    T('123', '12.345e1', '-0');
    T('123', '12.345e1', '-0.0000000');

    assertException(function () {new BigNumber(1.23).toFixed(NaN)}, "(1.23).toFixed(NaN)");
    assertException(function () {new BigNumber(1.23).toFixed('NaN')}, "(1.23).toFixed('NaN')");
    assertException(function () {new BigNumber(1.23).toFixed([])}, "(1.23).toFixed([])");
    assertException(function () {new BigNumber(1.23).toFixed({})}, "(1.23).toFixed({})");
    assertException(function () {new BigNumber(1.23).toFixed('')}, "(1.23).toFixed('')");
    assertException(function () {new BigNumber(1.23).toFixed(' ')}, "(1.23).toFixed(' ')");
    assertException(function () {new BigNumber(1.23).toFixed('hello')}, "(1.23).toFixed('hello')");
    assertException(function () {new BigNumber(1.23).toFixed('\t')}, "(1.23).toFixed('\t')");
    assertException(function () {new BigNumber(1.23).toFixed(new Date)}, "(1.23).toFixed(new Date)");
    assertException(function () {new BigNumber(1.23).toFixed(new RegExp)}, "(1.23).toFixed(new RegExp)");
    assertException(function () {new BigNumber(1.23).toFixed(2.01)}, "(1.23).toFixed(2.01)");
    assertException(function () {new BigNumber(1.23).toFixed(10.5)}, "(1.23).toFixed(10.5)");
    assertException(function () {new BigNumber(1.23).toFixed('1.1e1')}, "(1.23).toFixed('1.1e1')");
    assertException(function () {new BigNumber(1.23).toFixed(true)}, "(1.23).toFixed(true)");
    assertException(function () {new BigNumber(1.23).toFixed(false)}, "(1.23).toFixed(false)");
    assertException(function () {new BigNumber(1.23).toFixed(function (){})}, "(1.23).toFixed(function (){})");

    assertException(function () {new BigNumber('12.345e1').toFixed('-1')}, ".toFixed('-1')");
    assertException(function () {new BigNumber('12.345e1').toFixed(-23)}, ".toFixed(-23)");
    assertException(function () {new BigNumber('12.345e1').toFixed(1e9 + 1)}, ".toFixed(1e9 + 1)");
    assertException(function () {new BigNumber('12.345e1').toFixed(1e9 + 0.1)}, ".toFixed(1e9 + 1)");
    assertException(function () {new BigNumber('12.345e1').toFixed('-0.01')}, ".toFixed('-0.01')");
    assertException(function () {new BigNumber('12.345e1').toFixed('-1e-1')}, ".toFixed('-1e-1')");
    assertException(function () {new BigNumber('12.345e1').toFixed(Infinity)}, ".toFixed(Infinity)");
    assertException(function () {new BigNumber('12.345e1').toFixed('-Infinity')}, ".toFixed('-Infinity')");

    BigNumber.config({ERRORS : false});

    T('Infinity', Infinity, 0);
    T('Infinity', Infinity, NaN);
    T('Infinity', Infinity, null);
    T('Infinity', Infinity, Infinity);
    T('NaN', NaN, -Infinity);

    T('123.450', '12.345e1', new BigNumber(3));
    T('123.45', '12.345e1', null);
    T('123.45', '12.345e1', u);
    T('123.45', '12.345e1', NaN);
    T('123.45', '12.345e1', 'NaN');
    T('123.45', '12.345e1', []);
    T('123.45', '12.345e1', {});
    T('123.45', '12.345e1', '');
    T('123.45', '12.345e1', ' ');
    T('123.45', '12.345e1', 'hello');
    T('123.45', '12.345e1', '\t');
    T('123.45', '12.345e1', new Date);
    T('123.45', '12.345e1', new RegExp);
    
    T('123', '12.345e1', -0);
    
    T('123.4500000', '12.345e1', 7.5);
    T('123.45000000000', '12.345e1', '1.1e1');

    T('123.5', '12.345e1', 1);
    T('123.45', '12.345e1', '-1');
    T('123.45', '12.345e1', -23);
    T('123.45', '12.345e1', 1e9 + 1);
    T('123.45', '12.345e1', 1e9 + 0.1);
    T('123.45', '12.345e1', '-0.01');
    T('123.45', '12.345e1', '-1e-1');
    T('123.45', '12.345e1', Infinity);
    T('123.45', '12.345e1', '-Infinity');

    // ROUND_HALF_CEIL
    // Rounds towards nearest neighbour. If equidistant, rounds towards Infinity
    BigNumber.config({ERRORS : true, ROUNDING_MODE : 7});

    T('0.1', '0.05', 1);
    T('1', 0.5, 0);
    T('1', 0.54, 0);
    T('1', 0.55, 0);
    T('1', 0.56, 0);
    T('-1', -0.54, 0);
    T('-0', -0.5, 0);                  // test no. 1307
    T('-1', -0.56, 0);
    T('-0.5', -0.5, 1);
    T('1.3', 1.25, 1);
    T('-1.2', -1.25, 1);
    T('234.2041', 234.20405, 4);
    T('-234.2040', -234.20405, 4);
    T('234.2041', '234.204050000000000000000000000000006', 4);
    T('-234.2045', '-234.20449', 4);
    T('-234.2041', '-234.204050000000000000000000000000006', 4);
    T('999.0', 999, 1);
    T('1000', 999.5, 0);
    T('-999', -999.5, 0);
    T('-999.5', -999.5, 1);
    T('1.00000000000000000', '1.000000000000000000005', 17)
    T('1.00000000000000000001', '1.000000000000000000005', 20)
    T('-1.00000000000000000', '-1.000000000000000000005', 17)
    T('-1.00000000000000000000', '-1.000000000000000000005', 20)

    // ROUND_HALF_FLOOR
    // Rounds towards nearest neighbour. If equidistant, rounds towards -Infinity
    BigNumber.config({ROUNDING_MODE : 8});

    T('0.0', '0.05', '1');
    T('0', 0.5, '0');
    T('1', 0.54, '0');
    T('1', 0.55, '0');
    T('1', 0.56, '0');
    T('-1', -0.54, '0');
    T('-1', -0.5, '0');
    T('-1', -0.56, '0');
    T('-0.5', -0.5, '1');
    T('1.2', 1.25, '1');
    T('-1.3', -1.25, '1');
    T('234.2040', 234.20405, '4');
    T('-234.2041', -234.20405, '4');
    T('234.2040', '234.20404999999999999999999999999999', '4');
    T('234.2041', '234.204050000000000000000000000000006', '4');
    T('-234.2045', '-234.20449', '4');
    T('-234.2041', '-234.204050000000000000000000000000006', '4');
    T('999.0', 999, '1');
    T('999', 999.5, '0');
    T('-1000', -999.5, '0');
    T('-999.5', -999.5, '1');
    T('1.00000000000000000', '1.000000000000000000005', '17')
    T('1.00000000000000000000', '1.000000000000000000005', '20')
    T('-1.00000000000000000', '-1.000000000000000000005', '17')
    T('-1.00000000000000000001', '-1.000000000000000000005', '20')
    
    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : 1E9
    });

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];;
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;