//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/LatticeTypeSelectionForm.cpp
//! @brief     Implements class LatticeTypeSelectionForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleDesigner/LatticeTypeSelectionForm.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Sample/InterferenceItems.h"
#include "GUI/Model/Sample/Lattice2DItems.h"
#include <QCheckBox>
#include <QLabel>

LatticeTypeSelectionForm::LatticeTypeSelectionForm(
    QWidget* parent, Interference2DAbstractLatticeItem* interferenceItem,
    SampleEditorController* ec)
    : ISelectionContainerForm(parent, ec)
    , m_interferenceItem(interferenceItem)
{
    initUI(interferenceItem->latticeTypeSelection());
}

void LatticeTypeSelectionForm::createContent()
{
    auto* currentLatticeType = m_interferenceItem->latticeTypeSelection().currentItem();
    const auto valueProperties = currentLatticeType->geometryValues(false);
    const bool vertical = valueProperties.size() > 2;

    const auto onValueChange = [this](double newValue, DoubleProperty& d) {
        m_ec->setDensityRelatedValue(m_interferenceItem, newValue, d);
    };
    LayerEditorUtil::addMultiPropertyToGrid(m_gridLayout, 1, valueProperties, onValueChange,
                                            vertical, false);

    m_integrateOverXiCheckBox = new QCheckBox("Integrate over Xi", this);
    m_integrateOverXiCheckBox->setChecked(m_interferenceItem->xiIntegration());
    m_gridLayout->addWidget(m_integrateOverXiCheckBox, 1, m_gridLayout->columnCount());
    connect(m_integrateOverXiCheckBox, &QCheckBox::stateChanged,
            [this] { m_ec->setIntegrateOverXi(this, m_integrateOverXiCheckBox->isChecked()); });

    const int colOfXiLabel = m_gridLayout->columnCount();
    LayerEditorUtil::addMultiPropertyToGrid(m_gridLayout, m_gridLayout->columnCount(),
                                            {&currentLatticeType->latticeRotationAngle()},
                                            onValueChange, vertical, true);
    m_xiLabel = qobject_cast<QLabel*>(
        m_gridLayout->itemAtPosition(vertical ? 0 : 1, colOfXiLabel)->widget());
    ASSERT(m_xiLabel);

    updateXiVisibility();
}

void LatticeTypeSelectionForm::onIntegrateOverXiChanged()
{
    QSignalBlocker b(m_integrateOverXiCheckBox);
    m_integrateOverXiCheckBox->setChecked(m_interferenceItem->xiIntegration());
    updateXiVisibility();
}

void LatticeTypeSelectionForm::updateXiVisibility()
{
    m_xiLabel->setVisible(!m_interferenceItem->xiIntegration());
    m_xiLabel->buddy()->setVisible(!m_interferenceItem->xiIntegration());
}

Interference2DAbstractLatticeItem* LatticeTypeSelectionForm::interferenceItem() const
{
    return m_interferenceItem;
}
