/**
 * Go SDK for OpenFGA
 *
 * API version: 1.x
 * Website: https://openfga.dev
 * Documentation: https://openfga.dev/docs
 * Support: https://openfga.dev/community
 * License: [Apache-2.0](https://github.com/openfga/go-sdk/blob/main/LICENSE)
 *
 * NOTE: This file was auto generated by OpenAPI Generator (https://openapi-generator.tech). DO NOT EDIT.
 */

package openfga

import (
	"bytes"
	"context"
	"errors"
	"io"
	"log"
	"net/http"
	"net/url"
	"strings"
	"time"

	"github.com/openfga/go-sdk/internal/utils/retryutils"
	"github.com/openfga/go-sdk/telemetry"
)

// Linger please
var (
	_ context.Context
)

type RequestOptions struct {
	Headers map[string]string `json:"headers,omitempty"`
}

type OpenFgaApi interface {

	/*
		 * BatchCheck Send a list of `check` operations in a single request
		 * The `BatchCheck` API functions nearly identically to `Check`, but instead of checking a single user-object relationship BatchCheck accepts a list of relationships to check and returns a map containing `BatchCheckItem` response for each check it received.

	An associated `correlation_id` is required for each check in the batch. This ID is used to correlate a check to the appropriate response. It is a string consisting of only alphanumeric characters or hyphens with a maximum length of 36 characters. This `correlation_id` is used to map the result of each check to the item which was checked, so it must be unique for each item in the batch. We recommend using a UUID or ULID as the `correlation_id`, but you can use whatever unique identifier you need as long  as it matches this regex pattern: `^[\w\d-]{1,36}$`

	NOTE: The maximum number of checks that can be passed in the `BatchCheck` API is configurable via the [OPENFGA_MAX_CHECKS_PER_BATCH_CHECK](https://openfga.dev/docs/getting-started/setup-openfga/configuration#OPENFGA_MAX_CHECKS_PER_BATCH_CHECK) environment variable. If `BatchCheck` is called using the SDK, the SDK can split the batch check requests for you.

	For more details on how `Check` functions, see the docs for `/check`.

	### Examples
	#### A BatchCheckRequest
	```json
	{
	  "checks": [
	     {
	       "tuple_key": {
	         "object": "document:2021-budget"
	         "relation": "reader",
	         "user": "user:anne",
	       },
	       "contextual_tuples": {...}
	       "context": {}
	       "correlation_id": "01JA8PM3QM7VBPGB8KMPK8SBD5"
	     },
	     {
	       "tuple_key": {
	         "object": "document:2021-budget"
	         "relation": "reader",
	         "user": "user:bob",
	       },
	       "contextual_tuples": {...}
	       "context": {}
	       "correlation_id": "01JA8PMM6A90NV5ET0F28CYSZQ"
	     }
	   ]
	}
	```

	Below is a possible response to the above request. Note that the result map's keys are the `correlation_id` values from the checked items in the request:
	```json
	{
	   "result": {
	     "01JA8PMM6A90NV5ET0F28CYSZQ": {
	       "allowed": false,
	       "error": {"message": ""}
	    },
	     "01JA8PM3QM7VBPGB8KMPK8SBD5": {
	       "allowed": true,
	       "error": {"message": ""}
	    }
	}
	```

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiBatchCheckRequest
	*/
	BatchCheck(ctx context.Context, storeId string) ApiBatchCheckRequest

	/*
	 * BatchCheckExecute executes the request
	 * @return BatchCheckResponse
	 */
	BatchCheckExecute(r ApiBatchCheckRequest) (BatchCheckResponse, *http.Response, error)

	/*
		 * Check Check whether a user is authorized to access an object
		 * The Check API returns whether a given user has a relationship with a given object in a given store.
	The `user` field of the request can be a specific target, such as `user:anne`, or a userset (set of users) such as `group:marketing#member` or a type-bound public access `user:*`.
	To arrive at a result, the API uses: an authorization model, explicit tuples written through the Write API, contextual tuples present in the request, and implicit tuples that exist by virtue of applying set theory (such as `document:2021-budget#viewer@document:2021-budget#viewer`; the set of users who are viewers of `document:2021-budget` are the set of users who are the viewers of `document:2021-budget`).
	A `contextual_tuples` object may also be included in the body of the request. This object contains one field `tuple_keys`, which is an array of tuple keys. Each of these tuples may have an associated `condition`.
	You may also provide an `authorization_model_id` in the body. This will be used to assert that the input `tuple_key` is valid for the model specified. If not specified, the assertion will be made against the latest authorization model ID. It is strongly recommended to specify authorization model id for better performance.
	You may also provide a `context` object that will be used to evaluate the conditioned tuples in the system. It is strongly recommended to provide a value for all the input parameters of all the conditions, to ensure that all tuples be evaluated correctly.
	By default, the Check API caches results for a short time to optimize performance. You may specify a value of `HIGHER_CONSISTENCY` for the optional `consistency` parameter in the body to inform the server that higher conisistency is preferred at the expense of increased latency. Consideration should be given to the increased latency if requesting higher consistency.
	The response will return whether the relationship exists in the field `allowed`.

	Some exceptions apply, but in general, if a Check API responds with `{allowed: true}`, then you can expect the equivalent ListObjects query to return the object, and viceversa.
	For example, if `Check(user:anne, reader, document:2021-budget)` responds with `{allowed: true}`, then `ListObjects(user:anne, reader, document)` may include `document:2021-budget` in the response.
	## Examples
	### Querying with contextual tuples
	In order to check if user `user:anne` of type `user` has a `reader` relationship with object `document:2021-budget` given the following contextual tuple
	```json
	{
	  "user": "user:anne",
	  "relation": "member",
	  "object": "time_slot:office_hours"
	}
	```
	the Check API can be used with the following request body:
	```json
	{
	  "tuple_key": {
	    "user": "user:anne",
	    "relation": "reader",
	    "object": "document:2021-budget"
	  },
	  "contextual_tuples": {
	    "tuple_keys": [
	      {
	        "user": "user:anne",
	        "relation": "member",
	        "object": "time_slot:office_hours"
	      }
	    ]
	  },
	  "authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"
	}
	```
	### Querying usersets
	Some Checks will always return `true`, even without any tuples. For example, for the following authorization model
	```python
	model
	  schema 1.1
	type user
	type document
	  relations
	    define reader: [user]
	```
	the following query
	```json
	{
	  "tuple_key": {
	     "user": "document:2021-budget#reader",
	     "relation": "reader",
	     "object": "document:2021-budget"
	  }
	}
	```
	will always return `{ "allowed": true }`. This is because usersets are self-defining: the userset `document:2021-budget#reader` will always have the `reader` relation with `document:2021-budget`.
	### Querying usersets with difference in the model
	A Check for a userset can yield results that must be treated carefully if the model involves difference. For example, for the following authorization model
	```python
	model
	  schema 1.1
	type user
	type group
	  relations
	    define member: [user]
	type document
	  relations
	    define blocked: [user]
	    define reader: [group#member] but not blocked
	```
	the following query
	```json
	{
	  "tuple_key": {
	     "user": "group:finance#member",
	     "relation": "reader",
	     "object": "document:2021-budget"
	  },
	  "contextual_tuples": {
	    "tuple_keys": [
	      {
	        "user": "user:anne",
	        "relation": "member",
	        "object": "group:finance"
	      },
	      {
	        "user": "group:finance#member",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      {
	        "user": "user:anne",
	        "relation": "blocked",
	        "object": "document:2021-budget"
	      }
	    ]
	  },
	}
	```
	will return `{ "allowed": true }`, even though a specific user of the userset `group:finance#member` does not have the `reader` relationship with the given object.
	### Requesting higher consistency
	By default, the Check API caches results for a short time to optimize performance. You may request higher consistency to inform the server that higher consistency should be preferred at the expense of increased latency. Care should be taken when requesting higher consistency due to the increased latency.
	```json
	{
	  "tuple_key": {
	     "user": "group:finance#member",
	     "relation": "reader",
	     "object": "document:2021-budget"
	  },
	  "consistency": "HIGHER_CONSISTENCY"
	}
	```

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiCheckRequest
	*/
	Check(ctx context.Context, storeId string) ApiCheckRequest

	/*
	 * CheckExecute executes the request
	 * @return CheckResponse
	 */
	CheckExecute(r ApiCheckRequest) (CheckResponse, *http.Response, error)

	/*
	 * CreateStore Create a store
	 * Create a unique OpenFGA store which will be used to store authorization models and relationship tuples.
	 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
	 * @return ApiCreateStoreRequest
	 */
	CreateStore(ctx context.Context) ApiCreateStoreRequest

	/*
	 * CreateStoreExecute executes the request
	 * @return CreateStoreResponse
	 */
	CreateStoreExecute(r ApiCreateStoreRequest) (CreateStoreResponse, *http.Response, error)

	/*
	 * DeleteStore Delete a store
	 * Delete an OpenFGA store. This does not delete the data associated with the store, like tuples or authorization models.
	 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
	 * @param storeId
	 * @return ApiDeleteStoreRequest
	 */
	DeleteStore(ctx context.Context, storeId string) ApiDeleteStoreRequest

	/*
	 * DeleteStoreExecute executes the request
	 */
	DeleteStoreExecute(r ApiDeleteStoreRequest) (*http.Response, error)

	/*
		 * Expand Expand all relationships in userset tree format, and following userset rewrite rules.  Useful to reason about and debug a certain relationship
		 * The Expand API will return all users and usersets that have certain relationship with an object in a certain store.
	This is different from the `/stores/{store_id}/read` API in that both users and computed usersets are returned.
	Body parameters `tuple_key.object` and `tuple_key.relation` are all required.
	A `contextual_tuples` object may also be included in the body of the request. This object contains one field `tuple_keys`, which is an array of tuple keys. Each of these tuples may have an associated `condition`.
	The response will return a tree whose leaves are the specific users and usersets. Union, intersection and difference operator are located in the intermediate nodes.

	## Example
	To expand all users that have the `reader` relationship with object `document:2021-budget`, use the Expand API with the following request body
	```json
	{
	  "tuple_key": {
	    "object": "document:2021-budget",
	    "relation": "reader"
	  },
	  "authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"
	}
	```
	OpenFGA's response will be a userset tree of the users and usersets that have read access to the document.
	```json
	{
	  "tree":{
	    "root":{
	      "type":"document:2021-budget#reader",
	      "union":{
	        "nodes":[
	          {
	            "type":"document:2021-budget#reader",
	            "leaf":{
	              "users":{
	                "users":[
	                  "user:bob"
	                ]
	              }
	            }
	          },
	          {
	            "type":"document:2021-budget#reader",
	            "leaf":{
	              "computed":{
	                "userset":"document:2021-budget#writer"
	              }
	            }
	          }
	        ]
	      }
	    }
	  }
	}
	```
	The caller can then call expand API for the `writer` relationship for the `document:2021-budget`.
	### Expand Request with Contextual Tuples

	Given the model
	```python
	model
	    schema 1.1

	type user

	type folder
	    relations
	        define owner: [user]

	type document
	    relations
	        define parent: [folder]
	        define viewer: [user] or writer
	        define writer: [user] or owner from parent
	```
	and the initial tuples
	```json
	[{
	    "user": "user:bob",
	    "relation": "owner",
	    "object": "folder:1"
	}]
	```

	To expand all `writers` of `document:1` when `document:1` is put in `folder:1`, the first call could be

	```json
	{
	  "tuple_key": {
	    "object": "document:1",
	    "relation": "writer"
	  },
	  "contextual_tuples": {
	    "tuple_keys": [
	      {
	        "user": "folder:1",
	        "relation": "parent",
	        "object": "document:1"
	      }
	    ]
	  }
	}
	```
	this returns:
	```json
	{
	  "tree": {
	    "root": {
	      "name": "document:1#writer",
	      "union": {
	        "nodes": [
	          {
	            "name": "document:1#writer",
	            "leaf": {
	              "users": {
	                "users": []
	              }
	            }
	          },
	          {
	            "name": "document:1#writer",
	            "leaf": {
	              "tupleToUserset": {
	                "tupleset": "document:1#parent",
	                "computed": [
	                  {
	                    "userset": "folder:1#owner"
	                  }
	                ]
	              }
	            }
	          }
	        ]
	      }
	    }
	  }
	}
	```
	This tells us that the `owner` of `folder:1` may also be a writer. So our next call could be to find the `owners` of `folder:1`
	```json
	{
	  "tuple_key": {
	    "object": "folder:1",
	    "relation": "owner"
	  }
	}
	```
	which gives
	```json
	{
	  "tree": {
	    "root": {
	      "name": "folder:1#owner",
	      "leaf": {
	        "users": {
	          "users": [
	            "user:bob"
	          ]
	        }
	      }
	    }
	  }
	}
	```

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiExpandRequest
	*/
	Expand(ctx context.Context, storeId string) ApiExpandRequest

	/*
	 * ExpandExecute executes the request
	 * @return ExpandResponse
	 */
	ExpandExecute(r ApiExpandRequest) (ExpandResponse, *http.Response, error)

	/*
	 * GetStore Get a store
	 * Returns an OpenFGA store by its identifier
	 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
	 * @param storeId
	 * @return ApiGetStoreRequest
	 */
	GetStore(ctx context.Context, storeId string) ApiGetStoreRequest

	/*
	 * GetStoreExecute executes the request
	 * @return GetStoreResponse
	 */
	GetStoreExecute(r ApiGetStoreRequest) (GetStoreResponse, *http.Response, error)

	/*
		 * ListObjects List all objects of the given type that the user has a relation with
		 * The ListObjects API returns a list of all the objects of the given type that the user has a relation with.
	 To arrive at a result, the API uses: an authorization model, explicit tuples written through the Write API, contextual tuples present in the request, and implicit tuples that exist by virtue of applying set theory (such as `document:2021-budget#viewer@document:2021-budget#viewer`; the set of users who are viewers of `document:2021-budget` are the set of users who are the viewers of `document:2021-budget`).
	An `authorization_model_id` may be specified in the body. If it is not specified, the latest authorization model ID will be used. It is strongly recommended to specify authorization model id for better performance.
	You may also specify `contextual_tuples` that will be treated as regular tuples. Each of these tuples may have an associated `condition`.
	You may also provide a `context` object that will be used to evaluate the conditioned tuples in the system. It is strongly recommended to provide a value for all the input parameters of all the conditions, to ensure that all tuples be evaluated correctly.
	By default, the Check API caches results for a short time to optimize performance. You may specify a value of `HIGHER_CONSISTENCY` for the optional `consistency` parameter in the body to inform the server that higher conisistency is preferred at the expense of increased latency. Consideration should be given to the increased latency if requesting higher consistency.
	The response will contain the related objects in an array in the "objects" field of the response and they will be strings in the object format `<type>:<id>` (e.g. "document:roadmap").
	The number of objects in the response array will be limited by the execution timeout specified in the flag OPENFGA_LIST_OBJECTS_DEADLINE and by the upper bound specified in the flag OPENFGA_LIST_OBJECTS_MAX_RESULTS, whichever is hit first.
	The objects given will not be sorted, and therefore two identical calls can give a given different set of objects.
		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiListObjectsRequest
	*/
	ListObjects(ctx context.Context, storeId string) ApiListObjectsRequest

	/*
	 * ListObjectsExecute executes the request
	 * @return ListObjectsResponse
	 */
	ListObjectsExecute(r ApiListObjectsRequest) (ListObjectsResponse, *http.Response, error)

	/*
		 * ListStores List all stores
		 * Returns a paginated list of OpenFGA stores and a continuation token to get additional stores.
	The continuation token will be empty if there are no more stores.

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @return ApiListStoresRequest
	*/
	ListStores(ctx context.Context) ApiListStoresRequest

	/*
	 * ListStoresExecute executes the request
	 * @return ListStoresResponse
	 */
	ListStoresExecute(r ApiListStoresRequest) (ListStoresResponse, *http.Response, error)

	/*
		 * ListUsers List the users matching the provided filter who have a certain relation to a particular type.
		 * The ListUsers API returns a list of all the users of a specific type that have a relation to a given object.
	 To arrive at a result, the API uses: an authorization model, explicit tuples written through the Write API, contextual tuples present in the request, and implicit tuples that exist by virtue of applying set theory (such as `document:2021-budget#viewer@document:2021-budget#viewer`; the set of users who are viewers of `document:2021-budget` are the set of users who are the viewers of `document:2021-budget`).
	An `authorization_model_id` may be specified in the body. If it is not specified, the latest authorization model ID will be used. It is strongly recommended to specify authorization model id for better performance.
	You may also specify `contextual_tuples` that will be treated as regular tuples. Each of these tuples may have an associated `condition`.
	You may also provide a `context` object that will be used to evaluate the conditioned tuples in the system. It is strongly recommended to provide a value for all the input parameters of all the conditions, to ensure that all tuples be evaluated correctly.
	The response will contain the related users in an array in the "users" field of the response. These results may include specific objects, usersets
	or type-bound public access. Each of these types of results is encoded in its own type and not represented as a string.In cases where a type-bound public access result is returned (e.g. `user:*`), it cannot be inferred that all subjects
	of that type have a relation to the object; it is possible that negations exist and checks should still be queried
	on individual subjects to ensure access to that document.The number of users in the response array will be limited by the execution timeout specified in the flag OPENFGA_LIST_USERS_DEADLINE and by the upper bound specified in the flag OPENFGA_LIST_USERS_MAX_RESULTS, whichever is hit first.
	The returned users will not be sorted, and therefore two identical calls may yield different sets of users.
		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiListUsersRequest
	*/
	ListUsers(ctx context.Context, storeId string) ApiListUsersRequest

	/*
	 * ListUsersExecute executes the request
	 * @return ListUsersResponse
	 */
	ListUsersExecute(r ApiListUsersRequest) (ListUsersResponse, *http.Response, error)

	/*
		 * Read Get tuples from the store that matches a query, without following userset rewrite rules
		 * The Read API will return the tuples for a certain store that match a query filter specified in the body of the request.
	The API doesn't guarantee order by any field.
	It is different from the `/stores/{store_id}/expand` API in that it only returns relationship tuples that are stored in the system and satisfy the query.
	In the body:
	1. `tuple_key` is optional. If not specified, it will return all tuples in the store.
	2. `tuple_key.object` is mandatory if `tuple_key` is specified. It can be a full object (e.g., `type:object_id`) or type only (e.g., `type:`).
	3. `tuple_key.user` is mandatory if tuple_key is specified in the case the `tuple_key.object` is a type only. If tuple_key.user is specified, it needs to be a full object (e.g., `type:user_id`).
	## Examples
	### Query for all objects in a type definition
	To query for all objects that `user:bob` has `reader` relationship in the `document` type definition, call read API with body of
	```json
	{
	 "tuple_key": {
	     "user": "user:bob",
	     "relation": "reader",
	     "object": "document:"
	  }
	}
	```
	The API will return tuples and a continuation token, something like
	```json
	{
	  "tuples": [
	    {
	      "key": {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-06T15:32:11.128Z"
	    }
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}
	```
	This means that `user:bob` has a `reader` relationship with 1 document `document:2021-budget`. Note that this API, unlike the List Objects API, does not evaluate the tuples in the store.
	The continuation token will be empty if there are no more tuples to query.
	### Query for all stored relationship tuples that have a particular relation and object
	To query for all users that have `reader` relationship with `document:2021-budget`, call read API with body of
	```json
	{
	  "tuple_key": {
	     "object": "document:2021-budget",
	     "relation": "reader"
	   }
	}
	```
	The API will return something like
	```json
	{
	  "tuples": [
	    {
	      "key": {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-06T15:32:11.128Z"
	    }
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}
	```
	This means that `document:2021-budget` has 1 `reader` (`user:bob`).  Note that, even if the model said that all `writers` are also `readers`, the API will not return writers such as `user:anne` because it only returns tuples and does not evaluate them.
	### Query for all users with all relationships for a particular document
	To query for all users that have any relationship with `document:2021-budget`, call read API with body of
	```json
	{
	  "tuple_key": {
	      "object": "document:2021-budget"
	   }
	}
	```
	The API will return something like
	```json
	{
	  "tuples": [
	    {
	      "key": {
	        "user": "user:anne",
	        "relation": "writer",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-05T13:42:12.356Z"
	    },
	    {
	      "key": {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-06T15:32:11.128Z"
	    }
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}
	```
	This means that `document:2021-budget` has 1 `reader` (`user:bob`) and 1 `writer` (`user:anne`).

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiReadRequest
	*/
	Read(ctx context.Context, storeId string) ApiReadRequest

	/*
	 * ReadExecute executes the request
	 * @return ReadResponse
	 */
	ReadExecute(r ApiReadRequest) (ReadResponse, *http.Response, error)

	/*
	 * ReadAssertions Read assertions for an authorization model ID
	 * The ReadAssertions API will return, for a given authorization model id, all the assertions stored for it.
	 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
	 * @param storeId
	 * @param authorizationModelId
	 * @return ApiReadAssertionsRequest
	 */
	ReadAssertions(ctx context.Context, storeId string, authorizationModelId string) ApiReadAssertionsRequest

	/*
	 * ReadAssertionsExecute executes the request
	 * @return ReadAssertionsResponse
	 */
	ReadAssertionsExecute(r ApiReadAssertionsRequest) (ReadAssertionsResponse, *http.Response, error)

	/*
		 * ReadAuthorizationModel Return a particular version of an authorization model
		 * The ReadAuthorizationModel API returns an authorization model by its identifier.
	The response will return the authorization model for the particular version.

	## Example
	To retrieve the authorization model with ID `01G5JAVJ41T49E9TT3SKVS7X1J` for the store, call the GET authorization-models by ID API with `01G5JAVJ41T49E9TT3SKVS7X1J` as the `id` path parameter.  The API will return:
	```json
	{
	  "authorization_model":{
	    "id":"01G5JAVJ41T49E9TT3SKVS7X1J",
	    "type_definitions":[
	      {
	        "type":"user"
	      },
	      {
	        "type":"document",
	        "relations":{
	          "reader":{
	            "union":{
	              "child":[
	                {
	                  "this":{}
	                },
	                {
	                  "computedUserset":{
	                    "object":"",
	                    "relation":"writer"
	                  }
	                }
	              ]
	            }
	          },
	          "writer":{
	            "this":{}
	          }
	        }
	      }
	    ]
	  }
	}
	```
	In the above example, there are 2 types (`user` and `document`). The `document` type has 2 relations (`writer` and `reader`).
		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @param id
		 * @return ApiReadAuthorizationModelRequest
	*/
	ReadAuthorizationModel(ctx context.Context, storeId string, id string) ApiReadAuthorizationModelRequest

	/*
	 * ReadAuthorizationModelExecute executes the request
	 * @return ReadAuthorizationModelResponse
	 */
	ReadAuthorizationModelExecute(r ApiReadAuthorizationModelRequest) (ReadAuthorizationModelResponse, *http.Response, error)

	/*
		 * ReadAuthorizationModels Return all the authorization models for a particular store
		 * The ReadAuthorizationModels API will return all the authorization models for a certain store.
	OpenFGA's response will contain an array of all authorization models, sorted in descending order of creation.

	## Example
	Assume that a store's authorization model has been configured twice. To get all the authorization models that have been created in this store, call GET authorization-models. The API will return a response that looks like:
	```json
	{
	  "authorization_models": [
	    {
	      "id": "01G50QVV17PECNVAHX1GG4Y5NC",
	      "type_definitions": [...]
	    },
	    {
	      "id": "01G4ZW8F4A07AKQ8RHSVG9RW04",
	      "type_definitions": [...]
	    },
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}
	```
	If there are no more authorization models available, the `continuation_token` field will be empty
	```json
	{
	  "authorization_models": [
	    {
	      "id": "01G50QVV17PECNVAHX1GG4Y5NC",
	      "type_definitions": [...]
	    },
	    {
	      "id": "01G4ZW8F4A07AKQ8RHSVG9RW04",
	      "type_definitions": [...]
	    },
	  ],
	  "continuation_token": ""
	}
	```

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiReadAuthorizationModelsRequest
	*/
	ReadAuthorizationModels(ctx context.Context, storeId string) ApiReadAuthorizationModelsRequest

	/*
	 * ReadAuthorizationModelsExecute executes the request
	 * @return ReadAuthorizationModelsResponse
	 */
	ReadAuthorizationModelsExecute(r ApiReadAuthorizationModelsRequest) (ReadAuthorizationModelsResponse, *http.Response, error)

	/*
		 * ReadChanges Return a list of all the tuple changes
		 * The ReadChanges API will return a paginated list of tuple changes (additions and deletions) that occurred in a given store, sorted by ascending time. The response will include a continuation token that is used to get the next set of changes. If there are no changes after the provided continuation token, the same token will be returned in order for it to be used when new changes are recorded. If the store never had any tuples added or removed, this token will be empty.
	You can use the `type` parameter to only get the list of tuple changes that affect objects of that type.
	When reading a write tuple change, if it was conditioned, the condition will be returned.
	When reading a delete tuple change, the condition will NOT be returned regardless of whether it was originally conditioned or not.

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiReadChangesRequest
	*/
	ReadChanges(ctx context.Context, storeId string) ApiReadChangesRequest

	/*
	 * ReadChangesExecute executes the request
	 * @return ReadChangesResponse
	 */
	ReadChangesExecute(r ApiReadChangesRequest) (ReadChangesResponse, *http.Response, error)

	/*
		 * Write Add or delete tuples from the store
		 * The Write API will transactionally update the tuples for a certain store. Tuples and type definitions allow OpenFGA to determine whether a relationship exists between an object and an user.
	In the body, `writes` adds new tuples and `deletes` removes existing tuples. When deleting a tuple, any `condition` specified with it is ignored.
	The API is not idempotent by default: if, later on, you try to add the same tuple key (even if the `condition` is different), or if you try to delete a non-existing tuple, it will throw an error.
	To allow writes when an identical tuple already exists in the database, set `"on_duplicate": "ignore"` on the `writes` object.
	To allow deletes when a tuple was already removed from the database, set `"on_missing": "ignore"` on the `deletes` object.
	If a Write request contains both idempotent (ignore) and non-idempotent (error) operations, the most restrictive action (error) will take precedence. If a condition fails for a sub-request with an error flag, the entire transaction will be rolled back. This gives developers explicit control over the atomicity of the requests.
	The API will not allow you to write tuples such as `document:2021-budget#viewer@document:2021-budget#viewer`, because they are implicit.
	An `authorization_model_id` may be specified in the body. If it is, it will be used to assert that each written tuple (not deleted) is valid for the model specified. If it is not specified, the latest authorization model ID will be used.
	## Example
	### Adding relationships
	To add `user:anne` as a `writer` for `document:2021-budget`, call write API with the following
	```json
	{
	  "writes": {
	    "tuple_keys": [
	      {
	        "user": "user:anne",
	        "relation": "writer",
	        "object": "document:2021-budget"
	      }
	    ],
	    "on_duplicate": "ignore"
	  },
	  "authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"
	}
	```
	### Removing relationships
	To remove `user:bob` as a `reader` for `document:2021-budget`, call write API with the following
	```json
	{
	  "deletes": {
	    "tuple_keys": [
	      {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      }
	    ],
	    "on_missing": "ignore"
	  }
	}
	```

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiWriteRequest
	*/
	Write(ctx context.Context, storeId string) ApiWriteRequest

	/*
	 * WriteExecute executes the request
	 * @return map[string]interface{}
	 */
	WriteExecute(r ApiWriteRequest) (map[string]interface{}, *http.Response, error)

	/*
	 * WriteAssertions Upsert assertions for an authorization model ID
	 * The WriteAssertions API will upsert new assertions for an authorization model id, or overwrite the existing ones. An assertion is an object that contains a tuple key, the expectation of whether a call to the Check API of that tuple key will return true or false, and optionally a list of contextual tuples.
	 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
	 * @param storeId
	 * @param authorizationModelId
	 * @return ApiWriteAssertionsRequest
	 */
	WriteAssertions(ctx context.Context, storeId string, authorizationModelId string) ApiWriteAssertionsRequest

	/*
	 * WriteAssertionsExecute executes the request
	 */
	WriteAssertionsExecute(r ApiWriteAssertionsRequest) (*http.Response, error)

	/*
		 * WriteAuthorizationModel Create a new authorization model
		 * The WriteAuthorizationModel API will add a new authorization model to a store.
	Each item in the `type_definitions` array is a type definition as specified in the field `type_definition`.
	The response will return the authorization model's ID in the `id` field.

	## Example
	To add an authorization model with `user` and `document` type definitions, call POST authorization-models API with the body:
	```json
	{
	  "type_definitions":[
	    {
	      "type":"user"
	    },
	    {
	      "type":"document",
	      "relations":{
	        "reader":{
	          "union":{
	            "child":[
	              {
	                "this":{}
	              },
	              {
	                "computedUserset":{
	                  "object":"",
	                  "relation":"writer"
	                }
	              }
	            ]
	          }
	        },
	        "writer":{
	          "this":{}
	        }
	      }
	    }
	  ]
	}
	```
	OpenFGA's response will include the version id for this authorization model, which will look like
	```
	{"authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"}
	```

		 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
		 * @param storeId
		 * @return ApiWriteAuthorizationModelRequest
	*/
	WriteAuthorizationModel(ctx context.Context, storeId string) ApiWriteAuthorizationModelRequest

	/*
	 * WriteAuthorizationModelExecute executes the request
	 * @return WriteAuthorizationModelResponse
	 */
	WriteAuthorizationModelExecute(r ApiWriteAuthorizationModelRequest) (WriteAuthorizationModelResponse, *http.Response, error)
}

// OpenFgaApiService OpenFgaApi service
type OpenFgaApiService service

type ApiBatchCheckRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *BatchCheckRequest
	options    RequestOptions
}

func (r ApiBatchCheckRequest) Body(body BatchCheckRequest) ApiBatchCheckRequest {
	r.body = &body
	return r
}

func (r ApiBatchCheckRequest) Options(options RequestOptions) ApiBatchCheckRequest {
	r.options = options
	return r
}

func (r ApiBatchCheckRequest) Execute() (BatchCheckResponse, *http.Response, error) {
	return r.ApiService.BatchCheckExecute(r)
}

/*
  - BatchCheck Send a list of `check` operations in a single request
  - The `BatchCheck` API functions nearly identically to `Check`, but instead of checking a single user-object relationship BatchCheck accepts a list of relationships to check and returns a map containing `BatchCheckItem` response for each check it received.

An associated `correlation_id` is required for each check in the batch. This ID is used to correlate a check to the appropriate response. It is a string consisting of only alphanumeric characters or hyphens with a maximum length of 36 characters. This `correlation_id` is used to map the result of each check to the item which was checked, so it must be unique for each item in the batch. We recommend using a UUID or ULID as the `correlation_id`, but you can use whatever unique identifier you need as long  as it matches this regex pattern: `^[\w\d-]{1,36}$`

NOTE: The maximum number of checks that can be passed in the `BatchCheck` API is configurable via the [OPENFGA_MAX_CHECKS_PER_BATCH_CHECK](https://openfga.dev/docs/getting-started/setup-openfga/configuration#OPENFGA_MAX_CHECKS_PER_BATCH_CHECK) environment variable. If `BatchCheck` is called using the SDK, the SDK can split the batch check requests for you.

For more details on how `Check` functions, see the docs for `/check`.

### Examples
#### A BatchCheckRequest
```json

	{
	  "checks": [
	     {
	       "tuple_key": {
	         "object": "document:2021-budget"
	         "relation": "reader",
	         "user": "user:anne",
	       },
	       "contextual_tuples": {...}
	       "context": {}
	       "correlation_id": "01JA8PM3QM7VBPGB8KMPK8SBD5"
	     },
	     {
	       "tuple_key": {
	         "object": "document:2021-budget"
	         "relation": "reader",
	         "user": "user:bob",
	       },
	       "contextual_tuples": {...}
	       "context": {}
	       "correlation_id": "01JA8PMM6A90NV5ET0F28CYSZQ"
	     }
	   ]
	}

```

Below is a possible response to the above request. Note that the result map's keys are the `correlation_id` values from the checked items in the request:
```json

	{
	   "result": {
	     "01JA8PMM6A90NV5ET0F28CYSZQ": {
	       "allowed": false,
	       "error": {"message": ""}
	    },
	     "01JA8PM3QM7VBPGB8KMPK8SBD5": {
	       "allowed": true,
	       "error": {"message": ""}
	    }
	}

```

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiBatchCheckRequest
*/
func (a *OpenFgaApiService) BatchCheck(ctx context.Context, storeId string) ApiBatchCheckRequest {
	return ApiBatchCheckRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return BatchCheckResponse
 */
func (a *OpenFgaApiService) BatchCheckExecute(r ApiBatchCheckRequest) (BatchCheckResponse, *http.Response, error) {
	const (
		operationName = "BatchCheck"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    BatchCheckResponse
	)

	path := "/stores/{store_id}/batch-check"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiCheckRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *CheckRequest
	options    RequestOptions
}

func (r ApiCheckRequest) Body(body CheckRequest) ApiCheckRequest {
	r.body = &body
	return r
}

func (r ApiCheckRequest) Options(options RequestOptions) ApiCheckRequest {
	r.options = options
	return r
}

func (r ApiCheckRequest) Execute() (CheckResponse, *http.Response, error) {
	return r.ApiService.CheckExecute(r)
}

/*
  - Check Check whether a user is authorized to access an object
  - The Check API returns whether a given user has a relationship with a given object in a given store.

The `user` field of the request can be a specific target, such as `user:anne`, or a userset (set of users) such as `group:marketing#member` or a type-bound public access `user:*`.
To arrive at a result, the API uses: an authorization model, explicit tuples written through the Write API, contextual tuples present in the request, and implicit tuples that exist by virtue of applying set theory (such as `document:2021-budget#viewer@document:2021-budget#viewer`; the set of users who are viewers of `document:2021-budget` are the set of users who are the viewers of `document:2021-budget`).
A `contextual_tuples` object may also be included in the body of the request. This object contains one field `tuple_keys`, which is an array of tuple keys. Each of these tuples may have an associated `condition`.
You may also provide an `authorization_model_id` in the body. This will be used to assert that the input `tuple_key` is valid for the model specified. If not specified, the assertion will be made against the latest authorization model ID. It is strongly recommended to specify authorization model id for better performance.
You may also provide a `context` object that will be used to evaluate the conditioned tuples in the system. It is strongly recommended to provide a value for all the input parameters of all the conditions, to ensure that all tuples be evaluated correctly.
By default, the Check API caches results for a short time to optimize performance. You may specify a value of `HIGHER_CONSISTENCY` for the optional `consistency` parameter in the body to inform the server that higher conisistency is preferred at the expense of increased latency. Consideration should be given to the increased latency if requesting higher consistency.
The response will return whether the relationship exists in the field `allowed`.

Some exceptions apply, but in general, if a Check API responds with `{allowed: true}`, then you can expect the equivalent ListObjects query to return the object, and viceversa.
For example, if `Check(user:anne, reader, document:2021-budget)` responds with `{allowed: true}`, then `ListObjects(user:anne, reader, document)` may include `document:2021-budget` in the response.
## Examples
### Querying with contextual tuples
In order to check if user `user:anne` of type `user` has a `reader` relationship with object `document:2021-budget` given the following contextual tuple
```json

	{
	  "user": "user:anne",
	  "relation": "member",
	  "object": "time_slot:office_hours"
	}

```
the Check API can be used with the following request body:
```json

	{
	  "tuple_key": {
	    "user": "user:anne",
	    "relation": "reader",
	    "object": "document:2021-budget"
	  },
	  "contextual_tuples": {
	    "tuple_keys": [
	      {
	        "user": "user:anne",
	        "relation": "member",
	        "object": "time_slot:office_hours"
	      }
	    ]
	  },
	  "authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"
	}

```
### Querying usersets
Some Checks will always return `true`, even without any tuples. For example, for the following authorization model
```python
model

	schema 1.1

type user
type document

	relations
	  define reader: [user]

```
the following query
```json

	{
	  "tuple_key": {
	     "user": "document:2021-budget#reader",
	     "relation": "reader",
	     "object": "document:2021-budget"
	  }
	}

```
will always return `{ "allowed": true }`. This is because usersets are self-defining: the userset `document:2021-budget#reader` will always have the `reader` relation with `document:2021-budget`.
### Querying usersets with difference in the model
A Check for a userset can yield results that must be treated carefully if the model involves difference. For example, for the following authorization model
```python
model

	schema 1.1

type user
type group

	relations
	  define member: [user]

type document

	relations
	  define blocked: [user]
	  define reader: [group#member] but not blocked

```
the following query
```json

	{
	  "tuple_key": {
	     "user": "group:finance#member",
	     "relation": "reader",
	     "object": "document:2021-budget"
	  },
	  "contextual_tuples": {
	    "tuple_keys": [
	      {
	        "user": "user:anne",
	        "relation": "member",
	        "object": "group:finance"
	      },
	      {
	        "user": "group:finance#member",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      {
	        "user": "user:anne",
	        "relation": "blocked",
	        "object": "document:2021-budget"
	      }
	    ]
	  },
	}

```
will return `{ "allowed": true }`, even though a specific user of the userset `group:finance#member` does not have the `reader` relationship with the given object.
### Requesting higher consistency
By default, the Check API caches results for a short time to optimize performance. You may request higher consistency to inform the server that higher consistency should be preferred at the expense of increased latency. Care should be taken when requesting higher consistency due to the increased latency.
```json

	{
	  "tuple_key": {
	     "user": "group:finance#member",
	     "relation": "reader",
	     "object": "document:2021-budget"
	  },
	  "consistency": "HIGHER_CONSISTENCY"
	}

```

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiCheckRequest
*/
func (a *OpenFgaApiService) Check(ctx context.Context, storeId string) ApiCheckRequest {
	return ApiCheckRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return CheckResponse
 */
func (a *OpenFgaApiService) CheckExecute(r ApiCheckRequest) (CheckResponse, *http.Response, error) {
	const (
		operationName = "Check"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    CheckResponse
	)

	path := "/stores/{store_id}/check"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiCreateStoreRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	body       *CreateStoreRequest
	options    RequestOptions
}

func (r ApiCreateStoreRequest) Body(body CreateStoreRequest) ApiCreateStoreRequest {
	r.body = &body
	return r
}

func (r ApiCreateStoreRequest) Options(options RequestOptions) ApiCreateStoreRequest {
	r.options = options
	return r
}

func (r ApiCreateStoreRequest) Execute() (CreateStoreResponse, *http.Response, error) {
	return r.ApiService.CreateStoreExecute(r)
}

/*
 * CreateStore Create a store
 * Create a unique OpenFGA store which will be used to store authorization models and relationship tuples.
 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
 * @return ApiCreateStoreRequest
 */
func (a *OpenFgaApiService) CreateStore(ctx context.Context) ApiCreateStoreRequest {
	return ApiCreateStoreRequest{
		ApiService: a,
		ctx:        ctx,
	}
}

/*
 * Execute executes the request
 * @return CreateStoreResponse
 */
func (a *OpenFgaApiService) CreateStoreExecute(r ApiCreateStoreRequest) (CreateStoreResponse, *http.Response, error) {
	const (
		operationName = "CreateStore"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    CreateStoreResponse
	)

	path := "/stores"

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, "")
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"body": requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiDeleteStoreRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	options    RequestOptions
}

func (r ApiDeleteStoreRequest) Options(options RequestOptions) ApiDeleteStoreRequest {
	r.options = options
	return r
}

func (r ApiDeleteStoreRequest) Execute() (*http.Response, error) {
	return r.ApiService.DeleteStoreExecute(r)
}

/*
 * DeleteStore Delete a store
 * Delete an OpenFGA store. This does not delete the data associated with the store, like tuples or authorization models.
 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
 * @param storeId
 * @return ApiDeleteStoreRequest
 */
func (a *OpenFgaApiService) DeleteStore(ctx context.Context, storeId string) ApiDeleteStoreRequest {
	return ApiDeleteStoreRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 */
func (a *OpenFgaApiService) DeleteStoreExecute(r ApiDeleteStoreRequest) (*http.Response, error) {
	const (
		operationName = "DeleteStore"
		httpMethod    = http.MethodDelete
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
	)

	path := "/stores/{store_id}"
	if r.storeId == "" {
		return nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return httpResponse, err
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return httpResponse, nil
	}

	// should never have reached this
	return nil, reportError("Error not handled properly")
}

type ApiExpandRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *ExpandRequest
	options    RequestOptions
}

func (r ApiExpandRequest) Body(body ExpandRequest) ApiExpandRequest {
	r.body = &body
	return r
}

func (r ApiExpandRequest) Options(options RequestOptions) ApiExpandRequest {
	r.options = options
	return r
}

func (r ApiExpandRequest) Execute() (ExpandResponse, *http.Response, error) {
	return r.ApiService.ExpandExecute(r)
}

/*
  - Expand Expand all relationships in userset tree format, and following userset rewrite rules.  Useful to reason about and debug a certain relationship
  - The Expand API will return all users and usersets that have certain relationship with an object in a certain store.

This is different from the `/stores/{store_id}/read` API in that both users and computed usersets are returned.
Body parameters `tuple_key.object` and `tuple_key.relation` are all required.
A `contextual_tuples` object may also be included in the body of the request. This object contains one field `tuple_keys`, which is an array of tuple keys. Each of these tuples may have an associated `condition`.
The response will return a tree whose leaves are the specific users and usersets. Union, intersection and difference operator are located in the intermediate nodes.

## Example
To expand all users that have the `reader` relationship with object `document:2021-budget`, use the Expand API with the following request body
```json

	{
	  "tuple_key": {
	    "object": "document:2021-budget",
	    "relation": "reader"
	  },
	  "authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"
	}

```
OpenFGA's response will be a userset tree of the users and usersets that have read access to the document.
```json

	{
	  "tree":{
	    "root":{
	      "type":"document:2021-budget#reader",
	      "union":{
	        "nodes":[
	          {
	            "type":"document:2021-budget#reader",
	            "leaf":{
	              "users":{
	                "users":[
	                  "user:bob"
	                ]
	              }
	            }
	          },
	          {
	            "type":"document:2021-budget#reader",
	            "leaf":{
	              "computed":{
	                "userset":"document:2021-budget#writer"
	              }
	            }
	          }
	        ]
	      }
	    }
	  }
	}

```
The caller can then call expand API for the `writer` relationship for the `document:2021-budget`.
### Expand Request with Contextual Tuples

Given the model
```python
model

	schema 1.1

type user

type folder

	relations
	    define owner: [user]

type document

	relations
	    define parent: [folder]
	    define viewer: [user] or writer
	    define writer: [user] or owner from parent

```
and the initial tuples
```json

	[{
	    "user": "user:bob",
	    "relation": "owner",
	    "object": "folder:1"
	}]

```

To expand all `writers` of `document:1` when `document:1` is put in `folder:1`, the first call could be

```json

	{
	  "tuple_key": {
	    "object": "document:1",
	    "relation": "writer"
	  },
	  "contextual_tuples": {
	    "tuple_keys": [
	      {
	        "user": "folder:1",
	        "relation": "parent",
	        "object": "document:1"
	      }
	    ]
	  }
	}

```
this returns:
```json

	{
	  "tree": {
	    "root": {
	      "name": "document:1#writer",
	      "union": {
	        "nodes": [
	          {
	            "name": "document:1#writer",
	            "leaf": {
	              "users": {
	                "users": []
	              }
	            }
	          },
	          {
	            "name": "document:1#writer",
	            "leaf": {
	              "tupleToUserset": {
	                "tupleset": "document:1#parent",
	                "computed": [
	                  {
	                    "userset": "folder:1#owner"
	                  }
	                ]
	              }
	            }
	          }
	        ]
	      }
	    }
	  }
	}

```
This tells us that the `owner` of `folder:1` may also be a writer. So our next call could be to find the `owners` of `folder:1`
```json

	{
	  "tuple_key": {
	    "object": "folder:1",
	    "relation": "owner"
	  }
	}

```
which gives
```json

	{
	  "tree": {
	    "root": {
	      "name": "folder:1#owner",
	      "leaf": {
	        "users": {
	          "users": [
	            "user:bob"
	          ]
	        }
	      }
	    }
	  }
	}

```

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiExpandRequest
*/
func (a *OpenFgaApiService) Expand(ctx context.Context, storeId string) ApiExpandRequest {
	return ApiExpandRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return ExpandResponse
 */
func (a *OpenFgaApiService) ExpandExecute(r ApiExpandRequest) (ExpandResponse, *http.Response, error) {
	const (
		operationName = "Expand"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ExpandResponse
	)

	path := "/stores/{store_id}/expand"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiGetStoreRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	options    RequestOptions
}

func (r ApiGetStoreRequest) Options(options RequestOptions) ApiGetStoreRequest {
	r.options = options
	return r
}

func (r ApiGetStoreRequest) Execute() (GetStoreResponse, *http.Response, error) {
	return r.ApiService.GetStoreExecute(r)
}

/*
 * GetStore Get a store
 * Returns an OpenFGA store by its identifier
 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
 * @param storeId
 * @return ApiGetStoreRequest
 */
func (a *OpenFgaApiService) GetStore(ctx context.Context, storeId string) ApiGetStoreRequest {
	return ApiGetStoreRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return GetStoreResponse
 */
func (a *OpenFgaApiService) GetStoreExecute(r ApiGetStoreRequest) (GetStoreResponse, *http.Response, error) {
	const (
		operationName = "GetStore"
		httpMethod    = http.MethodGet
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    GetStoreResponse
	)

	path := "/stores/{store_id}"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiListObjectsRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *ListObjectsRequest
	options    RequestOptions
}

func (r ApiListObjectsRequest) Body(body ListObjectsRequest) ApiListObjectsRequest {
	r.body = &body
	return r
}

func (r ApiListObjectsRequest) Options(options RequestOptions) ApiListObjectsRequest {
	r.options = options
	return r
}

func (r ApiListObjectsRequest) Execute() (ListObjectsResponse, *http.Response, error) {
	return r.ApiService.ListObjectsExecute(r)
}

/*
  - ListObjects List all objects of the given type that the user has a relation with
  - The ListObjects API returns a list of all the objects of the given type that the user has a relation with.
    To arrive at a result, the API uses: an authorization model, explicit tuples written through the Write API, contextual tuples present in the request, and implicit tuples that exist by virtue of applying set theory (such as `document:2021-budget#viewer@document:2021-budget#viewer`; the set of users who are viewers of `document:2021-budget` are the set of users who are the viewers of `document:2021-budget`).

An `authorization_model_id` may be specified in the body. If it is not specified, the latest authorization model ID will be used. It is strongly recommended to specify authorization model id for better performance.
You may also specify `contextual_tuples` that will be treated as regular tuples. Each of these tuples may have an associated `condition`.
You may also provide a `context` object that will be used to evaluate the conditioned tuples in the system. It is strongly recommended to provide a value for all the input parameters of all the conditions, to ensure that all tuples be evaluated correctly.
By default, the Check API caches results for a short time to optimize performance. You may specify a value of `HIGHER_CONSISTENCY` for the optional `consistency` parameter in the body to inform the server that higher conisistency is preferred at the expense of increased latency. Consideration should be given to the increased latency if requesting higher consistency.
The response will contain the related objects in an array in the "objects" field of the response and they will be strings in the object format `<type>:<id>` (e.g. "document:roadmap").
The number of objects in the response array will be limited by the execution timeout specified in the flag OPENFGA_LIST_OBJECTS_DEADLINE and by the upper bound specified in the flag OPENFGA_LIST_OBJECTS_MAX_RESULTS, whichever is hit first.
The objects given will not be sorted, and therefore two identical calls can give a given different set of objects.
  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiListObjectsRequest
*/
func (a *OpenFgaApiService) ListObjects(ctx context.Context, storeId string) ApiListObjectsRequest {
	return ApiListObjectsRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return ListObjectsResponse
 */
func (a *OpenFgaApiService) ListObjectsExecute(r ApiListObjectsRequest) (ListObjectsResponse, *http.Response, error) {
	const (
		operationName = "ListObjects"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ListObjectsResponse
	)

	path := "/stores/{store_id}/list-objects"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiListStoresRequest struct {
	ctx               context.Context
	ApiService        OpenFgaApi
	pageSize          *int32
	continuationToken *string
	name              *string
	options           RequestOptions
}

func (r ApiListStoresRequest) PageSize(pageSize int32) ApiListStoresRequest {
	r.pageSize = &pageSize
	return r
}
func (r ApiListStoresRequest) ContinuationToken(continuationToken string) ApiListStoresRequest {
	r.continuationToken = &continuationToken
	return r
}
func (r ApiListStoresRequest) Name(name string) ApiListStoresRequest {
	r.name = &name
	return r
}

func (r ApiListStoresRequest) Options(options RequestOptions) ApiListStoresRequest {
	r.options = options
	return r
}

func (r ApiListStoresRequest) Execute() (ListStoresResponse, *http.Response, error) {
	return r.ApiService.ListStoresExecute(r)
}

/*
  - ListStores List all stores
  - Returns a paginated list of OpenFGA stores and a continuation token to get additional stores.

The continuation token will be empty if there are no more stores.

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @return ApiListStoresRequest
*/
func (a *OpenFgaApiService) ListStores(ctx context.Context) ApiListStoresRequest {
	return ApiListStoresRequest{
		ApiService: a,
		ctx:        ctx,
	}
}

/*
 * Execute executes the request
 * @return ListStoresResponse
 */
func (a *OpenFgaApiService) ListStoresExecute(r ApiListStoresRequest) (ListStoresResponse, *http.Response, error) {
	const (
		operationName = "ListStores"
		httpMethod    = http.MethodGet
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ListStoresResponse
	)

	path := "/stores"

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}

	if r.pageSize != nil {
		localVarQueryParams.Add("page_size", parameterToString(*r.pageSize, ""))
	}
	if r.continuationToken != nil {
		localVarQueryParams.Add("continuation_token", parameterToString(*r.continuationToken, ""))
	}
	if r.name != nil {
		localVarQueryParams.Add("name", parameterToString(*r.name, ""))
	}
	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, "")
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"body": requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiListUsersRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *ListUsersRequest
	options    RequestOptions
}

func (r ApiListUsersRequest) Body(body ListUsersRequest) ApiListUsersRequest {
	r.body = &body
	return r
}

func (r ApiListUsersRequest) Options(options RequestOptions) ApiListUsersRequest {
	r.options = options
	return r
}

func (r ApiListUsersRequest) Execute() (ListUsersResponse, *http.Response, error) {
	return r.ApiService.ListUsersExecute(r)
}

/*
  - ListUsers List the users matching the provided filter who have a certain relation to a particular type.
  - The ListUsers API returns a list of all the users of a specific type that have a relation to a given object.
    To arrive at a result, the API uses: an authorization model, explicit tuples written through the Write API, contextual tuples present in the request, and implicit tuples that exist by virtue of applying set theory (such as `document:2021-budget#viewer@document:2021-budget#viewer`; the set of users who are viewers of `document:2021-budget` are the set of users who are the viewers of `document:2021-budget`).

An `authorization_model_id` may be specified in the body. If it is not specified, the latest authorization model ID will be used. It is strongly recommended to specify authorization model id for better performance.
You may also specify `contextual_tuples` that will be treated as regular tuples. Each of these tuples may have an associated `condition`.
You may also provide a `context` object that will be used to evaluate the conditioned tuples in the system. It is strongly recommended to provide a value for all the input parameters of all the conditions, to ensure that all tuples be evaluated correctly.
The response will contain the related users in an array in the "users" field of the response. These results may include specific objects, usersets
or type-bound public access. Each of these types of results is encoded in its own type and not represented as a string.In cases where a type-bound public access result is returned (e.g. `user:*`), it cannot be inferred that all subjects
of that type have a relation to the object; it is possible that negations exist and checks should still be queried
on individual subjects to ensure access to that document.The number of users in the response array will be limited by the execution timeout specified in the flag OPENFGA_LIST_USERS_DEADLINE and by the upper bound specified in the flag OPENFGA_LIST_USERS_MAX_RESULTS, whichever is hit first.
The returned users will not be sorted, and therefore two identical calls may yield different sets of users.
  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiListUsersRequest
*/
func (a *OpenFgaApiService) ListUsers(ctx context.Context, storeId string) ApiListUsersRequest {
	return ApiListUsersRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return ListUsersResponse
 */
func (a *OpenFgaApiService) ListUsersExecute(r ApiListUsersRequest) (ListUsersResponse, *http.Response, error) {
	const (
		operationName = "ListUsers"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ListUsersResponse
	)

	path := "/stores/{store_id}/list-users"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiReadRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *ReadRequest
	options    RequestOptions
}

func (r ApiReadRequest) Body(body ReadRequest) ApiReadRequest {
	r.body = &body
	return r
}

func (r ApiReadRequest) Options(options RequestOptions) ApiReadRequest {
	r.options = options
	return r
}

func (r ApiReadRequest) Execute() (ReadResponse, *http.Response, error) {
	return r.ApiService.ReadExecute(r)
}

/*
  - Read Get tuples from the store that matches a query, without following userset rewrite rules
  - The Read API will return the tuples for a certain store that match a query filter specified in the body of the request.

The API doesn't guarantee order by any field.
It is different from the `/stores/{store_id}/expand` API in that it only returns relationship tuples that are stored in the system and satisfy the query.
In the body:
1. `tuple_key` is optional. If not specified, it will return all tuples in the store.
2. `tuple_key.object` is mandatory if `tuple_key` is specified. It can be a full object (e.g., `type:object_id`) or type only (e.g., `type:`).
3. `tuple_key.user` is mandatory if tuple_key is specified in the case the `tuple_key.object` is a type only. If tuple_key.user is specified, it needs to be a full object (e.g., `type:user_id`).
## Examples
### Query for all objects in a type definition
To query for all objects that `user:bob` has `reader` relationship in the `document` type definition, call read API with body of
```json

	{
	 "tuple_key": {
	     "user": "user:bob",
	     "relation": "reader",
	     "object": "document:"
	  }
	}

```
The API will return tuples and a continuation token, something like
```json

	{
	  "tuples": [
	    {
	      "key": {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-06T15:32:11.128Z"
	    }
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}

```
This means that `user:bob` has a `reader` relationship with 1 document `document:2021-budget`. Note that this API, unlike the List Objects API, does not evaluate the tuples in the store.
The continuation token will be empty if there are no more tuples to query.
### Query for all stored relationship tuples that have a particular relation and object
To query for all users that have `reader` relationship with `document:2021-budget`, call read API with body of
```json

	{
	  "tuple_key": {
	     "object": "document:2021-budget",
	     "relation": "reader"
	   }
	}

```
The API will return something like
```json

	{
	  "tuples": [
	    {
	      "key": {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-06T15:32:11.128Z"
	    }
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}

```
This means that `document:2021-budget` has 1 `reader` (`user:bob`).  Note that, even if the model said that all `writers` are also `readers`, the API will not return writers such as `user:anne` because it only returns tuples and does not evaluate them.
### Query for all users with all relationships for a particular document
To query for all users that have any relationship with `document:2021-budget`, call read API with body of
```json

	{
	  "tuple_key": {
	      "object": "document:2021-budget"
	   }
	}

```
The API will return something like
```json

	{
	  "tuples": [
	    {
	      "key": {
	        "user": "user:anne",
	        "relation": "writer",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-05T13:42:12.356Z"
	    },
	    {
	      "key": {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      },
	      "timestamp": "2021-10-06T15:32:11.128Z"
	    }
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}

```
This means that `document:2021-budget` has 1 `reader` (`user:bob`) and 1 `writer` (`user:anne`).

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiReadRequest
*/
func (a *OpenFgaApiService) Read(ctx context.Context, storeId string) ApiReadRequest {
	return ApiReadRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return ReadResponse
 */
func (a *OpenFgaApiService) ReadExecute(r ApiReadRequest) (ReadResponse, *http.Response, error) {
	const (
		operationName = "Read"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ReadResponse
	)

	path := "/stores/{store_id}/read"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiReadAssertionsRequest struct {
	ctx                  context.Context
	ApiService           OpenFgaApi
	storeId              string
	authorizationModelId string
	options              RequestOptions
}

func (r ApiReadAssertionsRequest) Options(options RequestOptions) ApiReadAssertionsRequest {
	r.options = options
	return r
}

func (r ApiReadAssertionsRequest) Execute() (ReadAssertionsResponse, *http.Response, error) {
	return r.ApiService.ReadAssertionsExecute(r)
}

/*
 * ReadAssertions Read assertions for an authorization model ID
 * The ReadAssertions API will return, for a given authorization model id, all the assertions stored for it.
 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
 * @param storeId
 * @param authorizationModelId
 * @return ApiReadAssertionsRequest
 */
func (a *OpenFgaApiService) ReadAssertions(ctx context.Context, storeId string, authorizationModelId string) ApiReadAssertionsRequest {
	return ApiReadAssertionsRequest{
		ApiService:           a,
		ctx:                  ctx,
		storeId:              storeId,
		authorizationModelId: authorizationModelId,
	}
}

/*
 * Execute executes the request
 * @return ReadAssertionsResponse
 */
func (a *OpenFgaApiService) ReadAssertionsExecute(r ApiReadAssertionsRequest) (ReadAssertionsResponse, *http.Response, error) {
	const (
		operationName = "ReadAssertions"
		httpMethod    = http.MethodGet
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ReadAssertionsResponse
	)

	path := "/stores/{store_id}/assertions/{authorization_model_id}"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))
	if r.authorizationModelId == "" {
		return returnValue, nil, reportError("authorizationModelId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"authorization_model_id"+"}", url.PathEscape(parameterToString(r.authorizationModelId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiReadAuthorizationModelRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	id         string
	options    RequestOptions
}

func (r ApiReadAuthorizationModelRequest) Options(options RequestOptions) ApiReadAuthorizationModelRequest {
	r.options = options
	return r
}

func (r ApiReadAuthorizationModelRequest) Execute() (ReadAuthorizationModelResponse, *http.Response, error) {
	return r.ApiService.ReadAuthorizationModelExecute(r)
}

/*
  - ReadAuthorizationModel Return a particular version of an authorization model
  - The ReadAuthorizationModel API returns an authorization model by its identifier.

The response will return the authorization model for the particular version.

## Example
To retrieve the authorization model with ID `01G5JAVJ41T49E9TT3SKVS7X1J` for the store, call the GET authorization-models by ID API with `01G5JAVJ41T49E9TT3SKVS7X1J` as the `id` path parameter.  The API will return:
```json

	{
	  "authorization_model":{
	    "id":"01G5JAVJ41T49E9TT3SKVS7X1J",
	    "type_definitions":[
	      {
	        "type":"user"
	      },
	      {
	        "type":"document",
	        "relations":{
	          "reader":{
	            "union":{
	              "child":[
	                {
	                  "this":{}
	                },
	                {
	                  "computedUserset":{
	                    "object":"",
	                    "relation":"writer"
	                  }
	                }
	              ]
	            }
	          },
	          "writer":{
	            "this":{}
	          }
	        }
	      }
	    ]
	  }
	}

```
In the above example, there are 2 types (`user` and `document`). The `document` type has 2 relations (`writer` and `reader`).
  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @param id
  - @return ApiReadAuthorizationModelRequest
*/
func (a *OpenFgaApiService) ReadAuthorizationModel(ctx context.Context, storeId string, id string) ApiReadAuthorizationModelRequest {
	return ApiReadAuthorizationModelRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
		id:         id,
	}
}

/*
 * Execute executes the request
 * @return ReadAuthorizationModelResponse
 */
func (a *OpenFgaApiService) ReadAuthorizationModelExecute(r ApiReadAuthorizationModelRequest) (ReadAuthorizationModelResponse, *http.Response, error) {
	const (
		operationName = "ReadAuthorizationModel"
		httpMethod    = http.MethodGet
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ReadAuthorizationModelResponse
	)

	path := "/stores/{store_id}/authorization-models/{id}"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))
	if r.id == "" {
		return returnValue, nil, reportError("id is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"id"+"}", url.PathEscape(parameterToString(r.id, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiReadAuthorizationModelsRequest struct {
	ctx               context.Context
	ApiService        OpenFgaApi
	storeId           string
	pageSize          *int32
	continuationToken *string
	options           RequestOptions
}

func (r ApiReadAuthorizationModelsRequest) PageSize(pageSize int32) ApiReadAuthorizationModelsRequest {
	r.pageSize = &pageSize
	return r
}
func (r ApiReadAuthorizationModelsRequest) ContinuationToken(continuationToken string) ApiReadAuthorizationModelsRequest {
	r.continuationToken = &continuationToken
	return r
}

func (r ApiReadAuthorizationModelsRequest) Options(options RequestOptions) ApiReadAuthorizationModelsRequest {
	r.options = options
	return r
}

func (r ApiReadAuthorizationModelsRequest) Execute() (ReadAuthorizationModelsResponse, *http.Response, error) {
	return r.ApiService.ReadAuthorizationModelsExecute(r)
}

/*
  - ReadAuthorizationModels Return all the authorization models for a particular store
  - The ReadAuthorizationModels API will return all the authorization models for a certain store.

OpenFGA's response will contain an array of all authorization models, sorted in descending order of creation.

## Example
Assume that a store's authorization model has been configured twice. To get all the authorization models that have been created in this store, call GET authorization-models. The API will return a response that looks like:
```json

	{
	  "authorization_models": [
	    {
	      "id": "01G50QVV17PECNVAHX1GG4Y5NC",
	      "type_definitions": [...]
	    },
	    {
	      "id": "01G4ZW8F4A07AKQ8RHSVG9RW04",
	      "type_definitions": [...]
	    },
	  ],
	  "continuation_token": "eyJwayI6IkxBVEVTVF9OU0NPTkZJR19hdXRoMHN0b3JlIiwic2siOiIxem1qbXF3MWZLZExTcUoyN01MdTdqTjh0cWgifQ=="
	}

```
If there are no more authorization models available, the `continuation_token` field will be empty
```json

	{
	  "authorization_models": [
	    {
	      "id": "01G50QVV17PECNVAHX1GG4Y5NC",
	      "type_definitions": [...]
	    },
	    {
	      "id": "01G4ZW8F4A07AKQ8RHSVG9RW04",
	      "type_definitions": [...]
	    },
	  ],
	  "continuation_token": ""
	}

```

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiReadAuthorizationModelsRequest
*/
func (a *OpenFgaApiService) ReadAuthorizationModels(ctx context.Context, storeId string) ApiReadAuthorizationModelsRequest {
	return ApiReadAuthorizationModelsRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return ReadAuthorizationModelsResponse
 */
func (a *OpenFgaApiService) ReadAuthorizationModelsExecute(r ApiReadAuthorizationModelsRequest) (ReadAuthorizationModelsResponse, *http.Response, error) {
	const (
		operationName = "ReadAuthorizationModels"
		httpMethod    = http.MethodGet
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ReadAuthorizationModelsResponse
	)

	path := "/stores/{store_id}/authorization-models"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}

	if r.pageSize != nil {
		localVarQueryParams.Add("page_size", parameterToString(*r.pageSize, ""))
	}
	if r.continuationToken != nil {
		localVarQueryParams.Add("continuation_token", parameterToString(*r.continuationToken, ""))
	}
	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiReadChangesRequest struct {
	ctx               context.Context
	ApiService        OpenFgaApi
	storeId           string
	type_             *string
	pageSize          *int32
	continuationToken *string
	startTime         *time.Time
	options           RequestOptions
}

func (r ApiReadChangesRequest) Type_(type_ string) ApiReadChangesRequest {
	r.type_ = &type_
	return r
}
func (r ApiReadChangesRequest) PageSize(pageSize int32) ApiReadChangesRequest {
	r.pageSize = &pageSize
	return r
}
func (r ApiReadChangesRequest) ContinuationToken(continuationToken string) ApiReadChangesRequest {
	r.continuationToken = &continuationToken
	return r
}
func (r ApiReadChangesRequest) StartTime(startTime time.Time) ApiReadChangesRequest {
	r.startTime = &startTime
	return r
}

func (r ApiReadChangesRequest) Options(options RequestOptions) ApiReadChangesRequest {
	r.options = options
	return r
}

func (r ApiReadChangesRequest) Execute() (ReadChangesResponse, *http.Response, error) {
	return r.ApiService.ReadChangesExecute(r)
}

/*
  - ReadChanges Return a list of all the tuple changes
  - The ReadChanges API will return a paginated list of tuple changes (additions and deletions) that occurred in a given store, sorted by ascending time. The response will include a continuation token that is used to get the next set of changes. If there are no changes after the provided continuation token, the same token will be returned in order for it to be used when new changes are recorded. If the store never had any tuples added or removed, this token will be empty.

You can use the `type` parameter to only get the list of tuple changes that affect objects of that type.
When reading a write tuple change, if it was conditioned, the condition will be returned.
When reading a delete tuple change, the condition will NOT be returned regardless of whether it was originally conditioned or not.

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiReadChangesRequest
*/
func (a *OpenFgaApiService) ReadChanges(ctx context.Context, storeId string) ApiReadChangesRequest {
	return ApiReadChangesRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return ReadChangesResponse
 */
func (a *OpenFgaApiService) ReadChangesExecute(r ApiReadChangesRequest) (ReadChangesResponse, *http.Response, error) {
	const (
		operationName = "ReadChanges"
		httpMethod    = http.MethodGet
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    ReadChangesResponse
	)

	path := "/stores/{store_id}/changes"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}

	if r.type_ != nil {
		localVarQueryParams.Add("type", parameterToString(*r.type_, ""))
	}
	if r.pageSize != nil {
		localVarQueryParams.Add("page_size", parameterToString(*r.pageSize, ""))
	}
	if r.continuationToken != nil {
		localVarQueryParams.Add("continuation_token", parameterToString(*r.continuationToken, ""))
	}
	if r.startTime != nil {
		localVarQueryParams.Add("start_time", parameterToString(*r.startTime, ""))
	}
	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiWriteRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *WriteRequest
	options    RequestOptions
}

func (r ApiWriteRequest) Body(body WriteRequest) ApiWriteRequest {
	r.body = &body
	return r
}

func (r ApiWriteRequest) Options(options RequestOptions) ApiWriteRequest {
	r.options = options
	return r
}

func (r ApiWriteRequest) Execute() (map[string]interface{}, *http.Response, error) {
	return r.ApiService.WriteExecute(r)
}

/*
  - Write Add or delete tuples from the store
  - The Write API will transactionally update the tuples for a certain store. Tuples and type definitions allow OpenFGA to determine whether a relationship exists between an object and an user.

In the body, `writes` adds new tuples and `deletes` removes existing tuples. When deleting a tuple, any `condition` specified with it is ignored.
The API is not idempotent by default: if, later on, you try to add the same tuple key (even if the `condition` is different), or if you try to delete a non-existing tuple, it will throw an error.
To allow writes when an identical tuple already exists in the database, set `"on_duplicate": "ignore"` on the `writes` object.
To allow deletes when a tuple was already removed from the database, set `"on_missing": "ignore"` on the `deletes` object.
If a Write request contains both idempotent (ignore) and non-idempotent (error) operations, the most restrictive action (error) will take precedence. If a condition fails for a sub-request with an error flag, the entire transaction will be rolled back. This gives developers explicit control over the atomicity of the requests.
The API will not allow you to write tuples such as `document:2021-budget#viewer@document:2021-budget#viewer`, because they are implicit.
An `authorization_model_id` may be specified in the body. If it is, it will be used to assert that each written tuple (not deleted) is valid for the model specified. If it is not specified, the latest authorization model ID will be used.
## Example
### Adding relationships
To add `user:anne` as a `writer` for `document:2021-budget`, call write API with the following
```json

	{
	  "writes": {
	    "tuple_keys": [
	      {
	        "user": "user:anne",
	        "relation": "writer",
	        "object": "document:2021-budget"
	      }
	    ],
	    "on_duplicate": "ignore"
	  },
	  "authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"
	}

```
### Removing relationships
To remove `user:bob` as a `reader` for `document:2021-budget`, call write API with the following
```json

	{
	  "deletes": {
	    "tuple_keys": [
	      {
	        "user": "user:bob",
	        "relation": "reader",
	        "object": "document:2021-budget"
	      }
	    ],
	    "on_missing": "ignore"
	  }
	}

```

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiWriteRequest
*/
func (a *OpenFgaApiService) Write(ctx context.Context, storeId string) ApiWriteRequest {
	return ApiWriteRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return map[string]interface{}
 */
func (a *OpenFgaApiService) WriteExecute(r ApiWriteRequest) (map[string]interface{}, *http.Response, error) {
	const (
		operationName = "Write"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    map[string]interface{}
	)

	path := "/stores/{store_id}/write"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}

type ApiWriteAssertionsRequest struct {
	ctx                  context.Context
	ApiService           OpenFgaApi
	storeId              string
	authorizationModelId string
	body                 *WriteAssertionsRequest
	options              RequestOptions
}

func (r ApiWriteAssertionsRequest) Body(body WriteAssertionsRequest) ApiWriteAssertionsRequest {
	r.body = &body
	return r
}

func (r ApiWriteAssertionsRequest) Options(options RequestOptions) ApiWriteAssertionsRequest {
	r.options = options
	return r
}

func (r ApiWriteAssertionsRequest) Execute() (*http.Response, error) {
	return r.ApiService.WriteAssertionsExecute(r)
}

/*
 * WriteAssertions Upsert assertions for an authorization model ID
 * The WriteAssertions API will upsert new assertions for an authorization model id, or overwrite the existing ones. An assertion is an object that contains a tuple key, the expectation of whether a call to the Check API of that tuple key will return true or false, and optionally a list of contextual tuples.
 * @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
 * @param storeId
 * @param authorizationModelId
 * @return ApiWriteAssertionsRequest
 */
func (a *OpenFgaApiService) WriteAssertions(ctx context.Context, storeId string, authorizationModelId string) ApiWriteAssertionsRequest {
	return ApiWriteAssertionsRequest{
		ApiService:           a,
		ctx:                  ctx,
		storeId:              storeId,
		authorizationModelId: authorizationModelId,
	}
}

/*
 * Execute executes the request
 */
func (a *OpenFgaApiService) WriteAssertionsExecute(r ApiWriteAssertionsRequest) (*http.Response, error) {
	const (
		operationName = "WriteAssertions"
		httpMethod    = http.MethodPut
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
	)

	path := "/stores/{store_id}/assertions/{authorization_model_id}"
	if r.storeId == "" {
		return nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))
	if r.authorizationModelId == "" {
		return nil, reportError("authorizationModelId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"authorization_model_id"+"}", url.PathEscape(parameterToString(r.authorizationModelId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return httpResponse, err
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return httpResponse, nil
	}

	// should never have reached this
	return nil, reportError("Error not handled properly")
}

type ApiWriteAuthorizationModelRequest struct {
	ctx        context.Context
	ApiService OpenFgaApi
	storeId    string
	body       *WriteAuthorizationModelRequest
	options    RequestOptions
}

func (r ApiWriteAuthorizationModelRequest) Body(body WriteAuthorizationModelRequest) ApiWriteAuthorizationModelRequest {
	r.body = &body
	return r
}

func (r ApiWriteAuthorizationModelRequest) Options(options RequestOptions) ApiWriteAuthorizationModelRequest {
	r.options = options
	return r
}

func (r ApiWriteAuthorizationModelRequest) Execute() (WriteAuthorizationModelResponse, *http.Response, error) {
	return r.ApiService.WriteAuthorizationModelExecute(r)
}

/*
  - WriteAuthorizationModel Create a new authorization model
  - The WriteAuthorizationModel API will add a new authorization model to a store.

Each item in the `type_definitions` array is a type definition as specified in the field `type_definition`.
The response will return the authorization model's ID in the `id` field.

## Example
To add an authorization model with `user` and `document` type definitions, call POST authorization-models API with the body:
```json

	{
	  "type_definitions":[
	    {
	      "type":"user"
	    },
	    {
	      "type":"document",
	      "relations":{
	        "reader":{
	          "union":{
	            "child":[
	              {
	                "this":{}
	              },
	              {
	                "computedUserset":{
	                  "object":"",
	                  "relation":"writer"
	                }
	              }
	            ]
	          }
	        },
	        "writer":{
	          "this":{}
	        }
	      }
	    }
	  ]
	}

```
OpenFGA's response will include the version id for this authorization model, which will look like
```
{"authorization_model_id": "01G50QVV17PECNVAHX1GG4Y5NC"}
```

  - @param ctx context.Context - for authentication, logging, cancellation, deadlines, tracing, etc. Passed from http.Request or context.Background().
  - @param storeId
  - @return ApiWriteAuthorizationModelRequest
*/
func (a *OpenFgaApiService) WriteAuthorizationModel(ctx context.Context, storeId string) ApiWriteAuthorizationModelRequest {
	return ApiWriteAuthorizationModelRequest{
		ApiService: a,
		ctx:        ctx,
		storeId:    storeId,
	}
}

/*
 * Execute executes the request
 * @return WriteAuthorizationModelResponse
 */
func (a *OpenFgaApiService) WriteAuthorizationModelExecute(r ApiWriteAuthorizationModelRequest) (WriteAuthorizationModelResponse, *http.Response, error) {
	const (
		operationName = "WriteAuthorizationModel"
		httpMethod    = http.MethodPost
	)
	var (
		requestStarted = time.Now()
		requestBody    interface{}
		returnValue    WriteAuthorizationModelResponse
	)

	path := "/stores/{store_id}/authorization-models"
	if r.storeId == "" {
		return returnValue, nil, reportError("storeId is required and must be specified")
	}

	path = strings.ReplaceAll(path, "{"+"store_id"+"}", url.PathEscape(parameterToString(r.storeId, "")))

	localVarHeaderParams := make(map[string]string)
	localVarQueryParams := url.Values{}
	if r.body == nil {
		return returnValue, nil, reportError("body is required and must be specified")
	}

	// to determine the Content-Type header
	localVarHTTPContentTypes := []string{"application/json"}

	// set Content-Type header
	localVarHTTPContentType := selectHeaderContentType(localVarHTTPContentTypes)
	if localVarHTTPContentType != "" {
		localVarHeaderParams["Content-Type"] = localVarHTTPContentType
	}

	// to determine the Accept header
	localVarHTTPHeaderAccepts := []string{"application/json"}

	// set Accept header
	localVarHTTPHeaderAccept := selectHeaderAccept(localVarHTTPHeaderAccepts)
	if localVarHTTPHeaderAccept != "" {
		localVarHeaderParams["Accept"] = localVarHTTPHeaderAccept
	}
	// body params
	requestBody = r.body

	// if any override headers were in the options, set them now
	for header, val := range r.options.Headers {
		localVarHeaderParams[header] = val
	}

	retryParams := a.client.cfg.RetryParams
	for i := 0; i < retryParams.MaxRetry+1; i++ {
		req, err := a.client.prepareRequest(r.ctx, path, httpMethod, requestBody, localVarHeaderParams, localVarQueryParams)
		if err != nil {
			return returnValue, nil, err
		}

		httpResponse, err := a.client.callAPI(req)
		if err != nil || httpResponse == nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, http.Header{}, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to network error (error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		responseBody, err := io.ReadAll(httpResponse.Body)
		_ = httpResponse.Body.Close()
		httpResponse.Body = io.NopCloser(bytes.NewBuffer(responseBody))
		if err != nil {
			if i < retryParams.MaxRetry {
				timeToWait := retryutils.GetTimeToWait(i, retryParams.MaxRetry, retryParams.MinWaitInMs, httpResponse.Header, operationName)
				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to error parsing response body (err=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}
			return returnValue, httpResponse, err
		}

		if httpResponse.StatusCode >= http.StatusMultipleChoices {
			err := a.client.handleAPIError(httpResponse, responseBody, requestBody, operationName, r.storeId)
			if err != nil && i < retryParams.MaxRetry {
				timeToWait := time.Duration(0)
				var fgaApiRateLimitExceededError FgaApiRateLimitExceededError
				var fgaApiInternalError FgaApiInternalError
				switch {
				case errors.As(err, &fgaApiRateLimitExceededError):
					timeToWait = err.(FgaApiRateLimitExceededError).GetTimeToWait(i, *retryParams)
				case errors.As(err, &fgaApiInternalError):
					timeToWait = err.(FgaApiInternalError).GetTimeToWait(i, *retryParams)
				}

				if timeToWait > 0 {
					if a.client.cfg.Debug {
						log.Printf("\nWaiting %v to retry %v (%v %v) due to api retryable error (status code %v, error=%v) on attempt %v. Request body: %v\n", timeToWait, operationName, req.Method, req.URL, httpResponse.StatusCode, err, i, requestBody)
					}
					time.Sleep(timeToWait)
					continue
				}
			}

			return returnValue, httpResponse, err
		}

		err = a.client.decode(&returnValue, responseBody, httpResponse.Header.Get("Content-Type"))
		if err != nil {
			newErr := GenericOpenAPIError{
				body:  responseBody,
				error: err.Error(),
			}
			return returnValue, httpResponse, newErr
		}

		metrics := telemetry.GetMetrics(telemetry.TelemetryFactoryParameters{Configuration: a.client.cfg.Telemetry})

		var attrs, queryDuration, requestDuration, _ = metrics.BuildTelemetryAttributes(
			operationName,
			map[string]interface{}{
				"storeId": r.storeId,
				"body":    requestBody,
			},
			req,
			httpResponse,
			requestStarted,
			i,
		)

		if requestDuration > 0 {
			_, _ = metrics.RequestDuration(requestDuration, attrs)
		}

		if queryDuration > 0 {
			_, _ = metrics.QueryDuration(queryDuration, attrs)
		}

		return returnValue, httpResponse, nil
	}

	// should never have reached this
	return returnValue, nil, reportError("Error not handled properly")
}
