'''
Defines the VmsObjectLibrary class which is used to obtain a list of modules
and the external symbols they each contain in an OpenVMS object library.
'''

import string
from VmsSymbol import VmsSymbol

class VmsObjectModule:

    '''Helper class maintains a list of function and data declarations that are
    stored in a module (translation unit) that is part of a library.'''

    def __init__(self, moduleData):
	self.__name = moduleData[0]
	self.__symbols = {}
	for line in moduleData[1:]:
	    symbols = string.split(string.strip(line))
	    for symbol in symbols:
		self.__symbols[symbol] = VmsSymbol(symbol)

    def name(self):
	return self.__name

    def analyze(self, analysis):
	for name in self.__symbols.keys():
	    if analysis.symbols.has_key(name):
		self.__symbols[name]=analysis.symbols[name]

    def demangle(self, mangledName, demangledName):
	self.__symbols[mangledName].demangle(demangledName)

    def lookup(self, name):
	return self.__symbols[name]

    def symbols(self):

	names=self.__symbols.keys()
	names.sort()
	return map(self.lookup, names)

    def isFunction(self, name):
	return self.__symbols[name].isFunction()

    def isData(self, name):
	return self.__symbols[name].isData()

    def isUndefined(self, name):
	return self.__symbols[name].isUndefined()

    def isAmbiguous(self, name):
	return self.__symbols[name].isAmbiguous()

    def functions(self):
	names=filter(self.isFunction, self.__symbols.keys())
	names.sort()
	return map(self.lookup, names)

    def data(self):
	names=filter(self.isData, self.__symbols.keys())
	names.sort()
	return map(self.lookup, names)

    def undefined(self):
	names=filter(self.isUndefined, self.__symbols.keys())
	names.sort()
	return map(self.lookup, names)

    def ambiguous(self):
	names=filter(self.isAmbiguous, self.__symbols.keys())
	names.sort()
	return map(self.lookup, names)

    def display(self, details=0):

	print
	print 'Module %s' % self.name()
	print

	functions=self.functions()
	if functions:
	    print 'Functions:'
	    for symbol in functions:
		symbol.display(details)
	else:
	    print 'no functions'

	data=self.data()
	if data:
	    print 'Data:'
	    for symbol in data:
		symbol.display(details)
	else:
	    print 'no data'

	undefined=self.undefined()
	if undefined:
	    print 'Undefined symbols:'
	    for symbol in undefined:
		symbol.display(details)
	else:
	    print 'no undefined symbols'

	ambiguous=self.ambiguous()
	if ambiguous:
	    print 'Ambiguous symbols:'
	    for symbol in ambiguous:
		symbol.display(details)
	else:
	    print 'no ambiguous symbols'

def isvowel(c):
    return c=='A' or c=='a' or\
	   c=='E' or c=='e' or\
	   c=='I' or c=='i' or\
	   c=='O' or c=='o' or\
	   c=='U' or c=='u'

def _shorten(name):

    # E.g., __INIT_TCPSOCKETMTFACTORY13A??? becomes
    #           1         2         3
    # 0123456789012345678901234567890
    # CXX$INTTCPSCKTMTFCTRY13A13BH0RC
    #                         0123456
    #			      <-hash>

    shortName='CXX$'
    l=len(name)
    wasVowel=0
    i=0
    for c in name:
	if isvowel(c) and not wasVowel:
	    wasVowel=1
	    # XXX I honestly don't know what it does if more than two vowels in
	    # a row occur.
	elif c != '_':
	    shortName=shortName + c
	    wasVowel=0
	if len(shortName)==31-7:
	    break
    return shortName

class VmsObjectLibrary:
    '''OpenVMS Object Library class maintains a list of VmsObjectModules'''

    def __init__(self, libraryFileName, repository, major, minor, micro):

	self.libraryFileName = libraryFileName
	import re
	exp=re.compile('[:\[\]]+')
	nameext=exp.split(self.libraryFileName)[-1]
	# this is what os.path.basename(self.libraryFileName) should do on VMS
	# but doesn't.
	self.simpleName = string.split(nameext,'.')[0]
	self.repository = repository
	self.major = major
	self.minor = minor
	self.micro = micro
	self.gsmajor = major
	self.gsminor = minor * 1000 + micro
	self.parseLibraryListing()
	self.analyze()
	self.demangle()

    def parseLibraryListing(self):

	import os

	os.system(
	    'library/list=%s/names %s'
	    % (self.libraryFileName, self.libraryFileName)
	)
	listingFileName = self.libraryFileName + '.lis'
	listing = open(listingFileName)
	contents = listing.read()
	moduleContents = string.split(contents, 'Module ')
	self.objectModules = []
	for moduleData in moduleContents[1:]:
	    lines = string.split(moduleData, '\n')
	    self.objectModules.append(VmsObjectModule(lines))
	del listing
	os.remove(listingFileName)

    def demangle(self):

	import os
	symbols=[]
	for module in self.objectModules:
	    for symbol in module.symbols():
		symbols.append((module.name(), symbol.name()))

	demanglerCommandName=self.simpleName + '.demangle'
	demanglerCommandFile=open(demanglerCommandName,'w')
	demanglerCommandFile.write(
	    '$ cxxdemangler /repository=%s\n' % self.repository
	)
	for symbol in symbols:
	    demanglerCommandFile.write(symbol[1] + '\n')
	demanglerCommandFile.close()

	p=os.popen('@ %s' % demanglerCommandName)
	lines=p.readlines()
	if len(lines) < len(symbols):
	    raise Exception(
		'''Truncated demangler output.'''
	    )
	elif len(lines) > len(symbols):
	    raise Exception(
		'''Demangler output contains extra garbage.'''
	    )

	module=0

	for symbol in range(len(symbols)):
	    moduleName=symbols[symbol][0]
	    mangledName=symbols[symbol][1]
	    demangledName=lines[symbol][:-1]   # remove \n

	    # advance the module index to the next one.
	    while module < len(self.objectModules) and\
		self.objectModules[module].name() != moduleName:
		module = module + 1
	    if module == len(self.objectModules):
		raise Exception(
		    '''Programming error: module %s not found in list while
		    demangling names.'''
		    % moduleName
		)
	    self.objectModules[module].demangle(mangledName, demangledName)
	os.remove(demanglerCommandName)

    def analyze(self):
	import os
	from ObjectFileAnalysis import ObjectFileAnalysis
	for module in self.objectModules:

	    # supposed to be able to to analyze/object... lib/include=module
	    # but that doesn't seem to work on VAX!
	    os.system(\
		'LIBRARY/EXTRACT=%s/OUTPUT=%s.OBJ %s' %\
		(module.name(), module.name(), self.libraryFileName)
	    )

	    analysis=ObjectFileAnalysis(module.name())
	    analysis.analyze()
	    module.analyze(analysis)
	    os.remove('''%s.OBJ''' % module.name())
	    
    def display(self, detailed=0):
	for module in self.objectModules:
	    module.display(detailed)

    def vaxOptionsFile(self):
	import re
	self.optionsFileName = '''%s_vax.opt''' % self.simpleName
	f=open(self.optionsFileName,'w')
	f.write('''! %(optionsFileName)s - shareable image options file for 
!\t\t\t  %(libraryFileName)s object library.

GSMATCH=LEQUAL,%(gsmajor)d,%(gsminor)d

CLUSTER=%(simpleName)s_CLUSTER
COLLECT=%(simpleName)s_CLUSTER,%(simpleName)s_FUNCTIONS

'''	% self.__dict__)

	for module in self.objectModules:

	    f.write('\n! Module %s\n\n' % module.name())

	    # Note that this is not quite as elaborate as below.  In
	    # particulare, these regular expressions do not require that the
	    # string match all the way to the end.

	    ptrToInitPattern1 = '''^void\s+\(\*const\s+_p__init_%s''' \
		% module.name()
	    ptrToInitPattern2 = '''^_P_INIT_%s''' % module.name()
	    ptrToInitPattern3 = '''^_PINIT_%s''' % module.name()
	    ptrToInitPattern4 = '^' + _shorten('_PINIT_' + module.name())
	    ptrToInitRoutine = '(%s|%s|%s|%s)' % (
		ptrToInitPattern1, ptrToInitPattern2, ptrToInitPattern3,
		ptrToInitPattern4
	    )
	    ptrToInitExp = re.compile(ptrToInitRoutine)

	    for symbol in module.data():
		demangled=symbol.demangledName()
		if not ptrToInitExp.search(demangled):
		    mangled=symbol.mangledName()
		    padding=32-len(mangled)
		    line = 'UNIVERSAL = %s%*s ! %s' % (
			mangled, padding, '', demangled
		    )

		    # The linker is hateful, so we must wrap lines:

		    while line:
			next=line[:80]+'\n'
			f.write(next)
			line=line[80:]
			if line:
			    line = '%44s ! %s' % ('',line)

    def vaxXferVector(self):
	import re
	self.marName='''%s_vax.mar''' % self.simpleName
	f=open(self.marName, 'w')
	f.write(
'''; %(marName)s\t- Shareable image transfer vector table for
;\t\t\t  %(libraryFileName)s object library.

\t.TITLE\t%(simpleName)s_VEC %(simpleName)s shareable image
\t.IDENT\t"%(major)d.%(minor)d.%(micro)d"
\t.DSABL\tGBL

\t.PSECT\t%(simpleName)s_FUNCTIONS PIC, USR, CON, REL, LCL, SHR, -
\t\tEXE, RD, NOWRT, QUAD

'''	% self.__dict__)

	for module in self.objectModules:

	    # in case it's demangled:
	    moduleInitPattern1 =\
'''^void\s+__[Ii][Nn][Ii][Tt]_%s[0-9A-Fa-f]+\s*\(\s*void\s*\)\s$'''\
		% module.name()

	    # currently, the compiler doesn't mangle the init routine:
	    moduleInitPattern2 =\
		'''^__[Ii][Nn][Ii][Tt]_%s[0-9A-Fa-f]+$'''\
		% module.name()

	    # long names will be shortened to match the following (Note that
	    # this is tricky because the name contains a hexidecimal has number
	    # before having another hash number appended to the end.  The latter
	    # can contain letters that aren't hexidecimal digits.)
	    moduleInitPattern3\
		= '^' + _shorten('__INIT_%s' % module.name()) + '[0-9a-zA-Z]+$'

	    # match one of the above:
	    moduleInitRoutine = '(%s|%s|%s)' % (
		moduleInitPattern1, moduleInitPattern2, moduleInitPattern3
	    )
	    moduleInitExp = re.compile(moduleInitRoutine)
	    f.write('\n\t; Module %s\n\n' % module.name())
	    for symbol in module.functions():
		if not moduleInitExp.search(symbol.demangledName()):
		    f.write(
			'\t_CALL\t%s\t; %s\n'
			% (symbol.mangledName(), symbol.demangledName())
		    )
	f.write('''	.END''')
