/*
 * DISTRHO Plugin Framework (DPF)
 * Copyright (C) 2012-2021 Filipe Coelho <falktx@falktx.com>
 *
 * Permission to use, copy, modify, and/or distribute this software for any purpose with
 * or without fee is hereby granted, provided that the above copyright notice and this
 * permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD
 * TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "TopLevelWidgetPrivateData.hpp"
#include "WidgetPrivateData.hpp"
#include "WindowPrivateData.hpp"
#include "pugl.hpp"

START_NAMESPACE_DGL

// -----------------------------------------------------------------------

TopLevelWidget::PrivateData::PrivateData(TopLevelWidget* const s, Window& w)
    : self(s),
      selfw(s),
      window(w)
{
    window.pData->topLevelWidgets.push_back(self);
}

TopLevelWidget::PrivateData::~PrivateData()
{
    window.pData->topLevelWidgets.remove(self);
}

bool TopLevelWidget::PrivateData::keyboardEvent(const KeyboardEvent& ev)
{
    // ignore event if we are not visible
    if (! selfw->pData->visible)
        return false;

    // give top-level widget chance to catch this event first
    if (self->onKeyboard(ev))
        return true;

    // propagate event to all subwidgets recursively
    return selfw->pData->giveKeyboardEventForSubWidgets(ev);
}

bool TopLevelWidget::PrivateData::specialEvent(const SpecialEvent& ev)
{
    // ignore event if we are not visible
    if (! selfw->pData->visible)
        return false;

    // give top-level widget chance to catch this event first
    if (self->onSpecial(ev))
        return true;

    // propagate event to all subwidgets recursively
    return selfw->pData->giveSpecialEventForSubWidgets(ev);
}

bool TopLevelWidget::PrivateData::characterInputEvent(const CharacterInputEvent& ev)
{
    // ignore event if we are not visible
    if (! selfw->pData->visible)
        return false;

    // give top-level widget chance to catch this event first
    if (self->onCharacterInput(ev))
        return true;

    // propagate event to all subwidgets recursively
    return selfw->pData->giveCharacterInputEventForSubWidgets(ev);
}

bool TopLevelWidget::PrivateData::mouseEvent(const MouseEvent& ev)
{
    // ignore event if we are not visible
    if (! selfw->pData->visible)
        return false;

    MouseEvent rev = ev;

    if (window.pData->autoScaling)
    {
        const double autoScaleFactor = window.pData->autoScaleFactor;

        rev.pos.setX(ev.pos.getX() / autoScaleFactor);
        rev.pos.setY(ev.pos.getY() / autoScaleFactor);
        rev.absolutePos.setX(ev.absolutePos.getX() / autoScaleFactor);
        rev.absolutePos.setY(ev.absolutePos.getY() / autoScaleFactor);
    }

    // give top-level widget chance to catch this event first
    if (self->onMouse(ev))
        return true;

    // propagate event to all subwidgets recursively
    return selfw->pData->giveMouseEventForSubWidgets(rev);
}

bool TopLevelWidget::PrivateData::motionEvent(const MotionEvent& ev)
{
    // ignore event if we are not visible
    if (! selfw->pData->visible)
        return false;

    MotionEvent rev = ev;

    if (window.pData->autoScaling)
    {
        const double autoScaleFactor = window.pData->autoScaleFactor;

        rev.pos.setX(ev.pos.getX() / autoScaleFactor);
        rev.pos.setY(ev.pos.getY() / autoScaleFactor);
        rev.absolutePos.setX(ev.absolutePos.getX() / autoScaleFactor);
        rev.absolutePos.setY(ev.absolutePos.getY() / autoScaleFactor);
    }

    // give top-level widget chance to catch this event first
    if (self->onMotion(ev))
        return true;

    // propagate event to all subwidgets recursively
    return selfw->pData->giveMotionEventForSubWidgets(rev);
}

bool TopLevelWidget::PrivateData::scrollEvent(const ScrollEvent& ev)
{
    // ignore event if we are not visible
    if (! selfw->pData->visible)
        return false;

    ScrollEvent rev = ev;

    if (window.pData->autoScaling)
    {
        const double autoScaleFactor = window.pData->autoScaleFactor;

        rev.pos.setX(ev.pos.getX() / autoScaleFactor);
        rev.pos.setY(ev.pos.getY() / autoScaleFactor);
        rev.absolutePos.setX(ev.absolutePos.getX() / autoScaleFactor);
        rev.absolutePos.setY(ev.absolutePos.getY() / autoScaleFactor);
        rev.delta.setX(ev.delta.getX() / autoScaleFactor);
        rev.delta.setY(ev.delta.getY() / autoScaleFactor);
    }

    // give top-level widget chance to catch this event first
    if (self->onScroll(ev))
        return true;

    // propagate event to all subwidgets recursively
    return selfw->pData->giveScrollEventForSubWidgets(rev);
}

void TopLevelWidget::PrivateData::fallbackOnResize()
{
    puglFallbackOnResize(window.pData->view);
}

// -----------------------------------------------------------------------

END_NAMESPACE_DGL
