"""
Data sources:
- Forenames: https://forebears.io/kenya/forenames
- Surnames: https://forebears.io/kenya/surnames

Last updated: 2023 (based on latest available data from Forebears.io)

Note: Name frequencies are based on statistical incidence in the Kenyan population
and are represented as decimal fractions (e.g., 0.023 = 2.3% of population).
"""

from collections import OrderedDict

from .. import Provider as PersonProvider


class Provider(PersonProvider):
    """
    Kenyan English person name provider.

    This class provides methods to generate authentic Kenyan names with
    appropriate frequency distributions based on statistical data.

    Name formats follow common Kenyan patterns with optional prefixes.
    """

    # Name formats for Kenyan English

    formats_female = (
        "{{first_name_female}} {{last_name}}",
        "{{first_name_female}} {{last_name}}",
        "{{first_name_female}} {{last_name}}",
        "{{first_name_female}} {{last_name}}",
        "{{first_name_female}} {{last_name}}",
        "{{prefix_female}} {{first_name_female}} {{last_name}}",
        "{{prefix_female}} {{first_name_female}} {{last_name}}",
    )

    formats_male = (
        "{{first_name_male}} {{last_name}}",
        "{{first_name_male}} {{last_name}}",
        "{{first_name_male}} {{last_name}}",
        "{{first_name_male}} {{last_name}}",
        "{{first_name_male}} {{last_name}}",
        "{{prefix_male}} {{first_name_male}} {{last_name}}",
        "{{prefix_male}} {{first_name_male}} {{last_name}}",
    )

    formats = formats_female + formats_male

    # Male first names with frequency weights
    # Source: https://forebears.io/kenya/forenames
    # Data represents the statistical frequency of each name in the Kenyan population

    first_names_male = OrderedDict(
        [
            ("John", 0.01059880),
            ("Joseph", 0.00985244),
            ("Peter", 0.00899436),
            ("Brian", 0.00842100),
            ("David", 0.00769793),
            ("James", 0.00712314),
            ("Samuel", 0.00588357),
            ("Daniel", 0.00573521),
            ("Stephen", 0.00571003),
            ("Kelvin", 0.00529927),
            ("Ibrahim", 0.00501938),
            ("Mohamed", 0.00482652),
            ("Dennis", 0.00481827),
            ("Emmanuel", 0.00458416),
            ("Paul", 0.00454081),
            ("Victor", 0.00440481),
            ("Moses", 0.00421542),
            ("Kevin", 0.00416383),
            ("Wanjiru", 0.00400511),
            ("Michael", 0.00394054),
            ("Francis", 0.00389962),
            ("Charles", 0.00358813),
            ("Vincent", 0.00353974),
            ("Alex", 0.00338782),
            ("Simon", 0.00335786),
            ("George", 0.00329155),
            ("Otieno", 0.00325414),
            ("Patrick", 0.00325365),
            ("Evans", 0.00298701),
            ("Collins", 0.00297805),
            ("Kennedy", 0.00282166),
            ("Mwangi", 0.00277793),
            ("Isaac", 0.00270628),
            ("Edwin", 0.00266795),
            ("Ali", 0.00261930),
            ("Erick", 0.00260290),
            ("Felix", 0.00256436),
            ("Joshua", 0.00250288),
            ("Abdi", 0.00248951),
            ("Martin", 0.00246938),
            ("Robert", 0.00239626),
            ("Amos", 0.00237748),
            ("Denis", 0.00231707),
            ("Hassan", 0.00229591),
            ("Julius", 0.00225978),
            ("Benson", 0.00224179),
            ("Benard", 0.00223845),
            ("Fredrick", 0.00216248),
            ("Nicholas", 0.00214224),
            ("Jackson", 0.00212671),
            ("Samwel", 0.00212501),
            ("Ian", 0.00208823),
            ("Eric", 0.00207789),
            ("Odhiambo", 0.00204522),
            ("Maina", 0.00203718),
            ("Richard", 0.00201608),
            ("Ochieng", 0.00199652),
            ("Antony", 0.00198471),
            ("Juma", 0.00196672),
            ("William", 0.00191878),
            ("Philip", 0.00176519),
            ("Samson", 0.00172114),
            ("Shadrack", 0.00164016),
            ("Ahmed", 0.00162822),
            ("Joel", 0.00161042),
            ("Wilson", 0.00158865),
            ("Thomas", 0.00158734),
            ("Hussein", 0.00156704),
            ("Omondi", 0.00154408),
            ("Kamau", 0.00151823),
            ("Onyango", 0.00149865),
            ("Geoffrey", 0.00149709),
            ("Bonface", 0.00148111),
            ("Hillary", 0.00146602),
            ("Jacob", 0.00146280),
            ("Andrew", 0.00145712),
            ("Gideon", 0.00144090),
            ("Benjamin", 0.00142924),
            ("Mark", 0.00142384),
            ("Timothy", 0.00142268),
            ("Adan", 0.00141512),
            ("Sammy", 0.00139469),
            ("Dominic", 0.00138627),
            ("Wafula", 0.00137849),
            ("Salim", 0.00133919),
            ("Titus", 0.00130714),
            ("Henry", 0.00129952),
            ("Elijah", 0.00128514),
            ("Anthony", 0.00127553),
            ("Edward", 0.00125285),
            ("Josphat", 0.00122370),
            ("Allan", 0.00121884),
            ("Wambui", 0.00121763),
            ("Stanley", 0.00119567),
            ("Gilbert", 0.00119446),
            ("Njoroge", 0.00119150),
            ("Kyalo", 0.00119038),
            ("Kariuki", 0.00118630),
            ("Ismail", 0.00118334),
            ("Christopher", 0.00118094),
            ("Wanjiku", 0.00117384),
            ("Omar", 0.00115533),
            ("Njeri", 0.00115047),
            ("Mutua", 0.00114218),
            ("Nelson", 0.00113134),
            ("Ouma", 0.00112873),
            ("Alfred", 0.00112674),
            ("Clinton", 0.00112542),
            ("Enock", 0.00112536),
            ("Kenneth", 0.00111929),
            ("Elvis", 0.00110586),
            ("Duncan", 0.00106500),
            ("Suleiman", 0.00106247),
            ("Rashid", 0.00106120),
            ("Dickson", 0.00104068),
            ("Valentine", 0.00102958),
            ("Lawrence", 0.00101049),
            ("Wanjala", 0.00100587),
            ("Muthoni", 0.00100218),
            ("Solomon", 0.00099151),
            ("Meshack", 0.00098847),
            ("Caleb", 0.00098829),
            ("Wambua", 0.00098767),
            ("Simiyu", 0.00095730),
            ("Abraham", 0.00095521),
            ("Jeremiah", 0.00095396),
            ("Boniface", 0.00095117),
            ("Hamisi", 0.00093238),
            ("Kibet", 0.00092122),
            ("Gabriel", 0.00091470),
            ("Kimani", 0.00090269),
            ("Raphael", 0.00089405),
            ("Justus", 0.00087231),
            ("Mutuku", 0.00086017),
            ("Reuben", 0.00085883),
            ("Derrick", 0.00085669),
            ("Ezekiel", 0.00084862),
            ("Njuguna", 0.00082147),
            ("Ronald", 0.00082048),
            ("Wekesa", 0.00081538),
            ("Bernard", 0.00080333),
            ("Lewis", 0.00079906),
            ("Leonard", 0.00079217),
            ("Silas", 0.00078660),
            ("Festus", 0.00077876),
            ("Jonathan", 0.00077431),
            ("Godfrey", 0.00076202),
            ("Geofrey", 0.00076047),
            ("Said", 0.00075641),
            ("Musyoka", 0.00075213),
            ("Douglas", 0.00074267),
            ("Dancan", 0.00073235),
            ("Eliud", 0.00071725),
            ("Kioko", 0.00071181),
            ("Pius", 0.00070365),
            ("Morris", 0.00069747),
            ("Elias", 0.00069663),
            ("Macharia", 0.00069376),
            ("Musa", 0.00069313),
            ("Mathew", 0.00069223),
            ("Barasa", 0.00067566),
            ("Fred", 0.00067359),
            ("Frankline", 0.00066942),
            ("Wairimu", 0.00065575),
            ("Isaiah", 0.00065154),
            ("Harrison", 0.00064623),
            ("Mutinda", 0.00063934),
            ("Newton", 0.00063865),
            ("Osman", 0.00063364),
            ("Wycliffe", 0.00063187),
            ("Karanja", 0.00062945),
            ("Mohammed", 0.00062718),
            ("Yusuf", 0.00062010),
            ("Nyambura", 0.00061332),
            ("Owino", 0.00060952),
            ("Wangui", 0.00060902),
            ("Issa", 0.00060773),
            ("Albert", 0.00060345),
            ("Okoth", 0.00059945),
            ("Jared", 0.00059885),
            ("Job", 0.00059604),
            ("Rodgers", 0.00059473),
            ("Yussuf", 0.00059157),
            ("Wangari", 0.00058641),
            ("Derick", 0.00058155),
            ("Johnson", 0.00057715),
            ("Oduor", 0.00057365),
            ("Oscar", 0.00056972),
            ("Nickson", 0.00056160),
            ("Wilfred", 0.00056119),
            ("Kiprotich", 0.00055858),
            ("Njoki", 0.00054847),
            ("Wanyonyi", 0.00054601),
            ("Nyongesa", 0.00054303),
            ("Kinyua", 0.00054281),
            ("Cyrus", 0.00053942),
            ("Abubakar", 0.00053288),
            ("Aron", 0.00052774),
            ("Kipkoech", 0.00052562),
            ("Melvin", 0.00051888),
            ("Gitonga", 0.00051713),
            ("Walter", 0.00051668),
            ("Kiptoo", 0.00051320),
            ("Mbugua", 0.00050781),
            ("Mohamud", 0.00050666),
            ("Tonny", 0.00050644),
            ("Waweru", 0.00050502),
            ("Onesmus", 0.00050109),
            ("Zakaria", 0.00050079),
            ("Wesley", 0.00050003),
            ("Ramadhan", 0.00049921),
            ("Haron", 0.00049813),
            ("Chege", 0.00049621),
            ("Daud", 0.00049439),
            ("Ngugi", 0.00048938),
            ("Tom", 0.00048619),
            ("Mwendwa", 0.00048476),
            ("Mike", 0.00048418),
            ("Cosmas", 0.00048390),
            ("Alvin", 0.00048196),
            ("Steven", 0.00048105),
            ("Kiprono", 0.00048079),
            ("Njeru", 0.00047934),
            ("Ezra", 0.00047608),
            ("Muriithi", 0.00046695),
            ("Abel", 0.00046645),
            ("Davis", 0.00046572),
            ("Mwende", 0.00046507),
            ("Maurice", 0.00046475),
            ("Mburu", 0.00045626),
            ("Gerald", 0.00045609),
            ("Kiprop", 0.00045293),
            ("Ken", 0.00045244),
            ("Musyoki", 0.00044883),
            ("Irungu", 0.00044799),
            ("Noah", 0.00044369),
            ("Maxwell", 0.00043967),
            ("Mwaura", 0.00043963),
            ("Cheruiyot", 0.00043840),
            ("Austine", 0.00043663),
            ("Nyaga", 0.00043514),
            ("Abdihakim", 0.00043361),
            ("Elisha", 0.00043067),
            ("Kimutai", 0.00043004),
            ("Gitau", 0.00042991),
            ("Kiplagat", 0.00042989),
            ("Laban", 0.00042715),
            ("Kipkorir", 0.00042514),
            ("Josephat", 0.00042402),
            ("Muli", 0.00042235),
            ("Yahya", 0.00042127),
            ("Noor", 0.00041747),
            ("Justine", 0.00041551),
            ("Oliver", 0.00041471),
            ("Steve", 0.00041337),
            ("Issack", 0.00041080),
            ("Mungai", 0.00040933),
            ("Ndungu", 0.00040644),
            ("Nathan", 0.00040577),
            ("Harun", 0.00040473),
            ("Augustine", 0.00040277),
            ("Mutisya", 0.00040253),
            ("Alexander", 0.00040218),
            ("Eugene", 0.00039897),
            ("Wachira", 0.00039365),
            ("Ayub", 0.00039305),
            ("Tony", 0.00039229),
            ("Cornelius", 0.00039162),
            ("Boaz", 0.00039111),
            ("Mutunga", 0.00038646),
            ("Aden", 0.00038610),
            ("Abdalla", 0.00038601),
            ("Bakari", 0.00038396),
            ("Mwaniki", 0.00038154),
            ("Dan", 0.00037849),
            ("Chris", 0.00037800),
            ("Japheth", 0.00037577),
            ("Philemon", 0.00037411),
            ("Ng'ang'a", 0.00037022),
            ("Benedict", 0.00035532),
            ("Maxwel", 0.00035370),
            ("Eugine", 0.00035232),
            ("Raymond", 0.00035161),
            ("Lucas", 0.00035107),
            ("Njenga", 0.00034831),
            ("Japhet", 0.00034751),
            ("Simion", 0.00034718),
            ("Ernest", 0.00034530),
            ("Ndung'u", 0.00034526),
            ("Munene", 0.00034142),
            ("Murithi", 0.00034008),
            ("Baraka", 0.00033803),
            ("Humphrey", 0.00033703),
            ("Mwanzia", 0.00033548),
            ("Hosea", 0.00033381),
            ("Nicholus", 0.00033364),
            ("Calvin", 0.00033189),
            ("Munyao", 0.00033075),
            ("Kipruto", 0.00032887),
            ("Linus", 0.00032744),
            ("Reagan", 0.00032719),
            ("Danson", 0.00032582),
            ("Arnold", 0.00032405),
            ("Robinson", 0.00032250),
            ("Evance", 0.00032030),
            ("Ismael", 0.00031814),
            ("Sylvester", 0.00031535),
            ("Charo", 0.00031345),
            ("Musembi", 0.00031321),
            ("Muriuki", 0.00031034),
            ("Muema", 0.00030902),
            ("Idris", 0.00030838),
            ("Willy", 0.00030732),
            ("Riziki", 0.00030645),
            ("Kinyanjui", 0.00030615),
            ("Mugo", 0.00030561),
            ("Teddy", 0.00030300),
            ("Clement", 0.00030028),
            ("Kilonzo", 0.00029812),
            ("Peterson", 0.00029639),
            ("Mwikali", 0.00029557),
            ("Kiplangat", 0.00029510),
            ("Kipngetich", 0.00029456),
            ("Mumo", 0.00029425),
            ("Kipchumba", 0.00029367),
            ("Rotich", 0.00029123),
            ("Feisal", 0.00029078),
            ("Kuria", 0.00029009),
            ("Wamalwa", 0.00028987),
            ("Ambrose", 0.00028985),
            ("Anderson", 0.00028838),
            ("Alphonce", 0.00028810),
            ("Jeff", 0.00028663),
            ("Mulwa", 0.00028585),
            ("Kipkemoi", 0.00028549),
            ("Billy", 0.00028460),
            ("Shem", 0.00028203),
            ("Muchiri", 0.00028151),
            ("Adam", 0.00028052),
            ("Makau", 0.00028043),
            ("Jack", 0.00028026),
            ("Mueni", 0.00027877),
            ("Kimathi", 0.00027857),
            ("Frank", 0.00027810),
            ("Koech", 0.00027769),
            ("Fidelis", 0.00027754),
            ("Jamal", 0.00027683),
            ("Wainaina", 0.00027518),
            ("Kipchirchir", 0.00027510),
            ("Saidi", 0.00027501),
            ("Makokha", 0.00027497),
            ("Roy", 0.00027432),
            ("Okumu", 0.00027303),
            ("Hezron", 0.00027184),
            ("Noel", 0.00027184),
            ("Kassim", 0.00027084),
            ("Micah", 0.00027069),
            ("Tobias", 0.00026938),
            ("Ndegwa", 0.00026858),
            ("Kenedy", 0.00026760),
            ("Hesbon", 0.00026616),
            ("Muiruri", 0.00026599),
            ("Amani", 0.00026482),
            ("Lameck", 0.00026370),
            ("Njagi", 0.00026279),
            ("Kemboi", 0.00026253),
            ("Katana", 0.00026205),
            ("Elly", 0.00026110),
            ("Erastus", 0.00026080),
            ("Nicodemus", 0.00025959),
            ("Mwangangi", 0.00025886),
            ("Mutiso", 0.00025696),
            ("Swaleh", 0.00025650),
            ("Johnstone", 0.00025642),
            ("Mugambi", 0.00025527),
            ("Morgan", 0.00025493),
            ("Nganga", 0.00025486),
            ("Hamza", 0.00025352),
            ("Omari", 0.00025242),
            ("Levis", 0.00025149),
            ("Godwin", 0.00025147),
            ("Momanyi", 0.00025137),
            ("Shaban", 0.00025042),
            ("Korir", 0.00024998),
            ("Daudi", 0.00024646),
            ("Jimmy", 0.00024638),
            ("Wanja", 0.00024614),
            ("Jonah", 0.00024232),
            ("Oluoch", 0.00024132),
            ("Karimi", 0.00023996),
            ("Abdul", 0.00023949),
            ("Ndirangu", 0.00023923),
            ("Odongo", 0.00023713),
            ("Wanyama", 0.00023595),
            ("Calvince", 0.00023418),
            ("Weldon", 0.00023336),
            ("Opiyo", 0.00023126),
            ("Wanjohi", 0.00023076),
            ("Sifuna", 0.00023012),
            ("Nuru", 0.00022996),
            ("Gedion", 0.00022789),
            ("Mustafa", 0.00022727),
            ("Wycliff", 0.00022692),
            ("Rama", 0.00022649),
            ("Terry", 0.00022649),
            ("Keneth", 0.00022629),
            ("Langat", 0.00022584),
            ("Githinji", 0.00022541),
            ("Keith", 0.00022420),
            ("Dismas", 0.00022411),
            ("Yunis", 0.00022195),
            ("Zacharia", 0.00022131),
            ("Josiah", 0.00022092),
            ("Gregory", 0.00022023),
            ("Muthui", 0.00021861),
            ("Lucky", 0.00021679),
            ("Joash", 0.00021582),
            ("Isaack", 0.00021547),
            ("Tyson", 0.00021487),
            ("Mbithi", 0.00021478),
            ("Jesse", 0.00021368),
            ("Luka", 0.00021360),
            ("Kipkirui", 0.00021297),
            ("Pascal", 0.00021243),
            ("Gibson", 0.00021232),
            ("Nduku", 0.00021180),
            ("Kiplimo", 0.00021141),
            ("Joram", 0.00021139),
            ("Njogu", 0.00021137),
            ("Nyakundi", 0.00021131),
            ("Njiru", 0.00021042),
            ("Emanuel", 0.00020973),
            ("Zakayo", 0.00020869),
            ("Okello", 0.00020811),
            ("Ryan", 0.00020755),
            ("Maingi", 0.00020474),
            ("Owuor", 0.00020394),
            ("Murimi", 0.00020308),
            ("Isack", 0.00020284),
            ("Zablon", 0.00020204),
            ("Musau", 0.00020191),
            ("Kelly", 0.00020178),
            ("Abdallah", 0.00020139),
            ("Rajab", 0.00020012),
            ("Sila", 0.00020012),
            ("Aggrey", 0.00019980),
            ("Mumbi", 0.00019887),
            ("Franklin", 0.00019822),
            ("Phelix", 0.00019721),
            ("Muturi", 0.00019634),
            ("Kipkurui", 0.00019582),
            ("Elphas", 0.00019459),
            ("Mwiti", 0.00019397),
            ("Collince", 0.00019377),
            ("Robin", 0.00019351),
            ("Munyoki", 0.00019122),
            ("Ephantus", 0.00019075),
            ("Muia", 0.00019049),
            ("Silvester", 0.00018969),
            ("Gordon", 0.00018934),
            ("Innocent", 0.00018725),
            ("Kirui", 0.00018721),
            ("Amin", 0.00018684),
            ("Donald", 0.00018611),
            ("Justin", 0.00018595),
            ("Kinuthia", 0.00018585),
            ("Kipsang", 0.00018565),
            ("Kiarie", 0.00018559),
            ("Milton", 0.00018505),
            ("Obed", 0.00018420),
            ("Owen", 0.00018408),
            ("Bramwel", 0.00018395),
            ("Zachary", 0.00018390),
            ("Nduta", 0.00018371),
            ("Wambugu", 0.00018338),
            ("Fidel", 0.00018334),
            ("Aftin", 0.00018332),
            ("Seth", 0.00018321),
            ("Carlos", 0.00018276),
            ("Mukhwana", 0.00018274),
            ("Ndunge", 0.00018192),
            ("Cleophas", 0.00018189),
            ("Kimanzi", 0.00018099),
            ("Mutie", 0.00017900),
            ("Javan", 0.00017878),
            ("Luke", 0.00017766),
            ("Ondieki", 0.00017742),
            ("Micheal", 0.00017708),
            ("Elkana", 0.00017704),
            ("Bahati", 0.00017686),
            ("Nehemiah", 0.00017557),
            ("Amon", 0.00017285),
            ("Kanini", 0.00017261),
            ("Makori", 0.00017231),
            ("Musee", 0.00017133),
            ("Aaron", 0.00017127),
            ("Kimeu", 0.00017062),
            ("Phillip", 0.00016950),
            ("Kipkosgei", 0.00016907),
            ("Ronny", 0.00016619),
            ("Nixon", 0.00016576),
            ("Karani", 0.00016501),
            ("Livingstone", 0.00016486),
            ("Wesonga", 0.00016440),
            ("Wangila", 0.00016220),
            ("Marvin", 0.00016205),
            ("Simeon", 0.00016205),
            ("Liban", 0.00016194),
            ("Dancun", 0.00016151),
            ("Phineas", 0.00016039),
            ("Kosgei", 0.00016032),
            ("Bryan", 0.00016002),
            ("Abednego", 0.00015795),
            ("Masika", 0.00015777),
            ("Nahashon", 0.00015712),
            ("Athuman", 0.00015615),
            ("Ngari", 0.00015576),
            ("Nzioka", 0.00015570),
            ("Stephene", 0.00015492),
            ("Masinde", 0.00015456),
            ("Athman", 0.00015443),
            ("Mathias", 0.00015278),
            ("Vitalis", 0.00015242),
            ("Bett", 0.00015235),
            ("Mboya", 0.00015205),
            ("Tsuma", 0.00015201),
            ("Kipkemboi", 0.00015160),
            ("Joe", 0.00015110),
            ("Mutai", 0.00015080),
            ("Mugendi", 0.00015067),
            ("Kipngeno", 0.00015058),
            ("Muthengi", 0.00015028),
            ("Jairus", 0.00015026),
            ("Nyamai", 0.00014987),
            ("Kipyegon", 0.00014967),
            ("Njue", 0.00014916),
            ("Gathoni", 0.00014909),
            ("Sam", 0.00014903),
            ("Muthoka", 0.00014881),
            ("Muthama", 0.00014719),
            ("Rogers", 0.00014710),
            ("Ngigi", 0.00014708),
            ("Dalmas", 0.00014700),
            ("Rono", 0.00014687),
            ("Nderitu", 0.00014644),
            ("Nzomo", 0.00014641),
            ("Duke", 0.00014564),
            ("Clifford", 0.00014520),
            ("Bravin", 0.00014473),
            ("Edgar", 0.00014430),
            ("Washington", 0.00014410),
            ("Nyawira", 0.00014380),
            ("Thuo", 0.00014302),
            ("Willis", 0.00014300),
            ("Oloo", 0.00014203),
        ]
    )

    # Female first names with frequency weights
    # Source: https://forebears.io/kenya/forenames
    # Data represents the statistical frequency of each name in the Kenyan population

    first_names_female = OrderedDict(
        [
            ("Mary", 0.00966109),
            ("Faith", 0.00956659),
            ("Mercy", 0.00946263),
            ("Sharon", 0.00644013),
            ("Esther", 0.00630065),
            ("Elizabeth", 0.00494676),
            ("Jane", 0.00459681),
            ("Grace", 0.00453507),
            ("Ruth", 0.00428457),
            ("Lucy", 0.00418074),
            ("Caroline", 0.00413472),
            ("Christine", 0.00362275),
            ("Nancy", 0.00353028),
            ("Margaret", 0.00350726),
            ("Ann", 0.00346193),
            ("Lilian", 0.00330208),
            ("Joyce", 0.00322583),
            ("Purity", 0.00321959),
            ("Eunice", 0.00318776),
            ("Susan", 0.00314055),
            ("Jackline", 0.00306724),
            ("Rose", 0.00304990),
            ("Beatrice", 0.00293480),
            ("Diana", 0.00288876),
            ("Cynthia", 0.00282212),
            ("Catherine", 0.00265037),
            ("Brenda", 0.00261368),
            ("Sarah", 0.00253089),
            ("Naomi", 0.00251838),
            ("Irene", 0.00251804),
            ("Alice", 0.00248275),
            ("Fatuma", 0.00239363),
            ("Gladys", 0.00230073),
            ("Agnes", 0.00227969),
            ("Vivian", 0.00227734),
            ("Lydia", 0.00222244),
            ("Dorcas", 0.00220204),
            ("Joan", 0.00218211),
            ("Hellen", 0.00212097),
            ("Janet", 0.00209925),
            ("Pauline", 0.00201971),
            ("Florence", 0.00197222),
            ("Sheila", 0.00188729),
            ("Maureen", 0.00187880),
            ("Winnie", 0.00175485),
            ("Joy", 0.00162316),
            ("Emily", 0.00157833),
            ("Miriam", 0.00155944),
            ("Abigael", 0.00146701),
            ("Halima", 0.00146699),
            ("Judith", 0.00146435),
            ("Damaris", 0.00145112),
            ("Tabitha", 0.00144449),
            ("Nafula", 0.00143924),
            ("Charity", 0.00141132),
            ("Linet", 0.00138780),
            ("Millicent", 0.00135714),
            ("Amina", 0.00134947),
            ("Leah", 0.00134895),
            ("Anne", 0.00132405),
            ("Martha", 0.00131524),
            ("Josephine", 0.00130645),
            ("Sylvia", 0.00128248),
            ("Kemunto", 0.00124281),
            ("Yvonne", 0.00118612),
            ("Daisy", 0.00116904),
            ("Nelly", 0.00116533),
            ("Betty", 0.00111372),
            ("Stella", 0.00111346),
            ("Doreen", 0.00110262),
            ("Salome", 0.00110186),
            ("Veronica", 0.00110162),
            ("Judy", 0.00109944),
            ("Fridah", 0.00109942),
            ("Teresia", 0.00106508),
            ("Monica", 0.00106001),
            ("Atieno", 0.00104766),
            ("Rebecca", 0.00104293),
            ("Caren", 0.00103809),
            ("Hannah", 0.00103526),
            ("Carolyne", 0.00103109),
            ("Moraa", 0.00101894),
            ("Everlyne", 0.00101354),
            ("Juliet", 0.00100607),
            ("Josphine", 0.00099965),
            ("Monicah", 0.00099330),
            ("Marion", 0.00097896),
            ("Kerubo", 0.00097817),
            ("Gloria", 0.00097603),
            ("Irine", 0.00096674),
            ("Winfred", 0.00096249),
            ("Akinyi", 0.00094826),
            ("Violet", 0.00093675),
            ("Rosemary", 0.00089973),
            ("Dorothy", 0.00088688),
            ("Jacinta", 0.00087945),
            ("Mariam", 0.00087818),
            ("Achieng", 0.00084935),
            ("Maurine", 0.00081659),
            ("Nekesa", 0.00080562),
            ("Linda", 0.00079942),
            ("Regina", 0.00079476),
            ("Rachael", 0.00079027),
            ("Cecilia", 0.00079005),
            ("Peris", 0.00077377),
            ("Asha", 0.00076463),
            ("Rael", 0.00074600),
            ("Emmaculate", 0.00072889),
            ("Patricia", 0.00071542),
            ("Teresa", 0.00070739),
            ("Kwamboka", 0.00070691),
            ("Veronicah", 0.00070292),
            ("Rehema", 0.00070283),
            ("Doris", 0.00069482),
            ("Pamela", 0.00069363),
            ("Mildred", 0.00069298),
            ("Celestine", 0.00068922),
            ("Edith", 0.00068629),
            ("Nanjala", 0.00067039),
            ("Lydiah", 0.00066545),
            ("Everline", 0.00066376),
            ("Victoria", 0.00065025),
            ("Zipporah", 0.00064461),
            ("Annah", 0.00063193),
            ("Khadija", 0.00062733),
            ("Beth", 0.00062206),
            ("Margret", 0.00061373),
            ("Risper", 0.00061075),
            ("Virginia", 0.00061019),
            ("Deborah", 0.00060548),
            ("Mercyline", 0.00060226),
            ("Sophia", 0.00058635),
            ("Ivy", 0.00058483),
            ("Peninah", 0.00057410),
            ("Beryl", 0.00057285),
            ("Anna", 0.00056693),
            ("Silvia", 0.00056443),
            ("Jelagat", 0.00056225),
            ("Aisha", 0.00056164),
            ("Winny", 0.00055173),
            ("Serah", 0.00054633),
            ("Maria", 0.00054009),
            ("Naomy", 0.00051167),
            ("Julia", 0.00051063),
            ("Anyango", 0.00050936),
            ("Angela", 0.00050932),
            ("Magdaline", 0.00050312),
            ("Sandra", 0.00050219),
            ("Mirriam", 0.00049988),
            ("Stacy", 0.00049394),
            ("Immaculate", 0.00049062),
            ("Hilda", 0.00048988),
            ("Mourine", 0.00048936),
            ("Belinda", 0.00048232),
            ("Loise", 0.00047895),
            ("Tracy", 0.00047796),
            ("Viola", 0.00047762),
            ("Habiba", 0.00047714),
            ("Milka", 0.00047692),
            ("Emma", 0.00047593),
            ("Neema", 0.00047576),
            ("Evaline", 0.00047340),
            ("Rahma", 0.00047274),
            ("Moureen", 0.00046863),
            ("Evalyne", 0.00045565),
            ("Rukia", 0.00045330),
            ("Roseline", 0.00045246),
            ("Nyaboke", 0.00045071),
            ("Loice", 0.00044276),
            ("Norah", 0.00043265),
            ("Rahab", 0.00042898),
            ("Naliaka", 0.00042641),
            ("Harriet", 0.00042287),
            ("Moreen", 0.00041473),
            ("Dorine", 0.00039955),
            ("Selina", 0.00039460),
            ("Getrude", 0.00039024),
            ("Akoth", 0.00038605),
            ("Karen", 0.00037854),
            ("Rachel", 0.00037603),
            ("Salma", 0.00036845),
            ("Stellah", 0.00036698),
            ("Priscilla", 0.00036625),
            ("Phoebe", 0.00036502),
            ("Joyline", 0.00036131),
            ("Jebet", 0.00035578),
            ("Rhoda", 0.00035573),
            ("Saumu", 0.00035532),
            ("Dorcus", 0.00035167),
            ("Chebet", 0.00035081),
            ("Racheal", 0.00035042),
            ("Teresiah", 0.00034947),
            ("Jeniffer", 0.00034938),
            ("Jecinta", 0.00033958),
            ("Juliana", 0.00033641),
            ("Patience", 0.00033621),
            ("Roselyne", 0.00033191),
            ("Felister", 0.00033161),
            ("Laura", 0.00033032),
            ("Penina", 0.00033010),
            ("Quinter", 0.00032479),
            ("Sheilah", 0.00032349),
            ("Rodah", 0.00032313),
            ("Sabina", 0.00031773),
            ("Valary", 0.00031747),
            ("Bosibori", 0.00031507),
            ("Prisca", 0.00031419),
            ("Clare", 0.00031237),
            ("Metrine", 0.00030870),
            ("Jepkoech", 0.00030812),
            ("Wendy", 0.00030591),
            ("Dinah", 0.00030470),
            ("Sally", 0.00030129),
            ("Edna", 0.00029879),
            ("Dianah", 0.00029395),
            ("Consolata", 0.00028935),
            ("Jerop", 0.00028644),
            ("Lucia", 0.00028635),
            ("Carol", 0.00028393),
            ("Sofia", 0.00028382),
            ("Lavender", 0.00028216),
            ("Fancy", 0.00027909),
            ("Jeruto", 0.00027812),
            ("Priscah", 0.00027806),
            ("Emilly", 0.00027365),
            ("Phylis", 0.00027333),
            ("Brendah", 0.00027309),
            ("Velma", 0.00027212),
            ("Angeline", 0.00026860),
            ("Jemimah", 0.00026726),
            ("Anita", 0.00026655),
            ("Carren", 0.00026253),
            ("Elosy", 0.00026238),
            ("Naom", 0.00026095),
            ("Lorna", 0.00025940),
            ("Hadija", 0.00025929),
            ("Emmah", 0.00025763),
            ("Prudence", 0.00025692),
            ("Jemutai", 0.00025640),
            ("Magdalene", 0.00025560),
            ("Jepkorir", 0.00025555),
            ("Annet", 0.00025454),
            ("Nelima", 0.00025357),
            ("Francisca", 0.00025350),
            ("Gesare", 0.00025221),
            ("Hawa", 0.00024785),
            ("Rita", 0.00024715),
            ("Linah", 0.00024465),
            ("Hildah", 0.00024361),
            ("Hafsa", 0.00024301),
            ("Jepchumba", 0.00024137),
            ("Jeptoo", 0.00024035),
            ("Jesca", 0.00023761),
            ("Jennifer", 0.00023750),
            ("Jerotich", 0.00023724),
            ("Jepchirchir", 0.00023262),
            ("Jerono", 0.00022932),
            ("Mwanamisi", 0.00022748),
            ("Eva", 0.00022729),
            ("Leila", 0.00022351),
            ("Annastacia", 0.00022200),
            ("Felistus", 0.00021999),
            ("Vallary", 0.00021824),
            ("Nyanchama", 0.00021701),
            ("Lenah", 0.00021595),
            ("Vicky", 0.00021375),
            ("Phanice", 0.00021321),
            ("Cherotich", 0.00020783),
            ("Fiona", 0.00020781),
            ("June", 0.00020660),
            ("Evelyne", 0.00020483),
            ("Jedidah", 0.00020483),
            ("Jael", 0.00020256),
            ("Maryann", 0.00020241),
            ("Emmy", 0.00020163),
            ("Zainab", 0.00019971),
            ("Milkah", 0.00019865),
            ("Frida", 0.00019587),
            ("Jepkemoi", 0.00019572),
            ("Sheilla", 0.00019554),
            ("Lorine", 0.00019487),
            ("Priscillah", 0.00019336),
            ("Sylivia", 0.00019235),
            ("Keziah", 0.00019230),
            ("Jepkosgei", 0.00019178),
            ("Anastacia", 0.00019071),
            ("Janeth", 0.00018909),
            ("Christabel", 0.00018893),
            ("Kemuma", 0.00018772),
            ("Mwanajuma", 0.00018740),
            ("Claire", 0.00018479),
            ("Clara", 0.00018367),
            ("Philis", 0.00018295),
            ("Lynn", 0.00018259),
            ("Lillian", 0.00018256),
            ("Morine", 0.00018224),
            ("Dolphine", 0.00018200),
            ("Melvine", 0.00018062),
            ("Mitchelle", 0.00018030),
            ("Makena", 0.00017902),
            ("Flora", 0.00017576),
            ("Bridget", 0.00017542),
            ("Sophy", 0.00017371),
            ("Debora", 0.00017155),
            ("Brigid", 0.00017062),
            ("Phyllis", 0.00016607),
            ("Edinah", 0.00016470),
            ("Chepkemoi", 0.00015926),
            ("Bridgit", 0.00015814),
            ("Centrine", 0.00015782),
            ("Maryanne", 0.00015732),
            ("Sheillah", 0.00015721),
            ("Jenipher", 0.00015710),
            ("Benta", 0.00015674),
            ("Lindah", 0.00015620),
            ("Farida", 0.00015598),
            ("Zainabu", 0.00015561),
            ("Maimuna", 0.00015559),
            ("Jamila", 0.00015557),
            ("Felista", 0.00015540),
            ("Natasha", 0.00015389),
            ("Chelangat", 0.00015330),
            ("Saida", 0.00015218),
            ("Chepkorir", 0.00015196),
            ("Bilha", 0.00015011),
            ("Chepkoech", 0.00014831),
            ("Rabecca", 0.00014739),
            ("Milcah", 0.00014726),
            ("Philomena", 0.00014451),
            ("Eddah", 0.00014441),
            ("Weddy", 0.00014441),
            ("Sara", 0.00014374),
            ("Laureen", 0.00014367),
            ("Ivone", 0.00014235),
        ]
    )

    # Combined first names dictionary
    first_names: OrderedDict[str, float] = OrderedDict()
    first_names.update(first_names_male)
    first_names.update(first_names_female)

    # Kenyan surnames with frequency weights
    # Source: https://forebears.io/kenya/surnames
    # Data represents the statistical frequency of each surname in the Kenyan population

    last_names = OrderedDict(
        [
            ("Otieno", 0.01185858),
            ("Mohamed", 0.01075451),
            ("Mwangi", 0.01038549),
            ("Odhiambo", 0.00760428),
            ("Maina", 0.00758120),
            ("Ochieng", 0.00728878),
            ("Ali", 0.00581335),
            ("Kamau", 0.00568845),
            ("Omondi", 0.00568520),
            ("Onyango", 0.00566218),
            ("Juma", 0.00520575),
            ("Hassan", 0.00509232),
            ("Abdi", 0.00496608),
            ("Wambui", 0.00483273),
            ("Atieno", 0.00483204),
            ("Wanjiku", 0.00470425),
            ("Njoroge", 0.00467513),
            ("Njeri", 0.00459377),
            ("Kariuki", 0.00437833),
            ("Akinyi", 0.00425928),
            ("Wafula", 0.00403169),
            ("Achieng", 0.00399618),
            ("Muthoni", 0.00390685),
            ("Ouma", 0.00389180),
            ("Ahmed", 0.00365510),
            ("Kimani", 0.00342333),
            ("Hussein", 0.00335421),
            ("Adan", 0.00324856),
            ("Mutua", 0.00323485),
            ("Abdullahi", 0.00318444),
            ("Adhiambo", 0.00315336),
            ("Cheruiyot", 0.00309847),
            ("Njuguna", 0.00304994),
            ("Kibet", 0.00302969),
            ("Simiyu", 0.00297783),
            ("Macharia", 0.00292088),
            ("Wanjala", 0.00277257),
            ("Barasa", 0.00270568),
            ("Wambua", 0.00266876),
            ("Wairimu", 0.00259689),
            ("Chebet", 0.00258405),
            ("Koech", 0.00257486),
            ("Ibrahim", 0.00252510),
            ("Omar", 0.00252211),
            ("Nyambura", 0.00250880),
            ("Rotich", 0.00246079),
            ("Karanja", 0.00245174),
            ("Anyango", 0.00242930),
            ("Wangui", 0.00233913),
            ("Wekesa", 0.00232939),
            ("Okoth", 0.00230821),
            ("Langat", 0.00228714),
            ("Chepkemoi", 0.00227987),
            ("Mutuku", 0.00226696),
            ("Owino", 0.00213890),
            ("Wangari", 0.00211406),
            ("Njoki", 0.00211256),
            ("Awuor", 0.00208825),
            ("Musyoka", 0.00205847),
            ("Nyongesa", 0.00205169),
            ("Cherotich", 0.00203770),
            ("Wanyonyi", 0.00202984),
            ("Chepkoech", 0.00201876),
            ("Korir", 0.00200572),
            ("Chelangat", 0.00199011),
            ("Kiptoo", 0.00198645),
            ("Mwendwa", 0.00194370),
            ("Mbugua", 0.00192400),
            ("Chege", 0.00190851),
            ("Waweru", 0.00190765),
            ("Kinyua", 0.00189531),
            ("Ngugi", 0.00188742),
            ("Kiprotich", 0.00186848),
            ("Kipkoech", 0.00184004),
            ("Oduor", 0.00180750),
            ("Mburu", 0.00177094),
            ("Akoth", 0.00176540),
            ("Kirui", 0.00176159),
            ("Mwende", 0.00175633),
            ("Kioko", 0.00175388),
            ("Gitonga", 0.00175360),
            ("Rono", 0.00171274),
            ("Auma", 0.00170708),
            ("Mwaura", 0.00167055),
            ("Muriithi", 0.00167049),
            ("Gitau", 0.00166999),
            ("Munene", 0.00166267),
            ("Wachira", 0.00162774),
            ("Irungu", 0.00161414),
            ("Njeru", 0.00159517),
            ("Kemboi", 0.00158103),
            ("Mutai", 0.00157707),
            ("Cherono", 0.00155533),
            ("Kimutai", 0.00154743),
            ("Ndungu", 0.00152040),
            ("Nyaga", 0.00151951),
            ("Mutinda", 0.00151614),
            ("Ndung'u", 0.00151566),
            ("Chepkorir", 0.00151527),
            ("Ng'ang'a", 0.00149524),
            ("Chepkirui", 0.00148268),
            ("Kiprono", 0.00147575),
            ("Mungai", 0.00144706),
            ("Bett", 0.00142190),
            ("Muriuki", 0.00141980),
            ("Mwenda", 0.00140230),
            ("Yussuf", 0.00139567),
            ("Njenga", 0.00138443),
            ("Said", 0.00137857),
            ("Osman", 0.00137846),
            ("Kiprop", 0.00136367),
            ("Mohamud", 0.00136183),
            ("Kiplagat", 0.00134968),
            ("Kipkorir", 0.00134530),
            ("Awino", 0.00132584),
            ("John", 0.00132309),
            ("Mwaniki", 0.00132278),
            ("Charo", 0.00130667),
            ("Murithi", 0.00130397),
            ("Mugo", 0.00125578),
            ("Kosgei", 0.00125500),
            ("Makokha", 0.00123937),
            ("Kyalo", 0.00123876),
            ("Karisa", 0.00123166),
            ("Ngetich", 0.00121722),
            ("Okumu", 0.00121014),
            ("Kinyanjui", 0.00119918),
            ("Chepngetich", 0.00118920),
            ("Peter", 0.00118850),
            ("Maalim", 0.00118792),
            ("Issack", 0.00118101),
            ("Muli", 0.00117761),
            ("Kazungu", 0.00116841),
            ("Katana", 0.00115890),
            ("Kimathi", 0.00114494),
            ("Kiplangat", 0.00113536),
            ("Wamalwa", 0.00112348),
            ("Momanyi", 0.00112228),
            ("Musyoki", 0.00111425),
            ("Kuria", 0.00110812),
            ("Mutisya", 0.00109803),
            ("Cheptoo", 0.00109567),
            ("Kipngetich", 0.00109418),
            ("Wainaina", 0.00108714),
            ("Chemutai", 0.00108099),
            ("Muchiri", 0.00107276),
            ("Nyakundi", 0.00106187),
            ("Wanyama", 0.00105098),
            ("Sheikh", 0.00104944),
            ("Mohammed", 0.00103814),
            ("Mugambi", 0.00103419),
            ("Kipchumba", 0.00102298),
            ("Nekesa", 0.00102248),
            ("Wanja", 0.00102021),
            ("Moraa", 0.00100522),
            ("Odongo", 0.00100327),
            ("Opiyo", 0.00099723),
            ("Noor", 0.00098785),
            ("Kahindi", 0.00098655),
            ("Muiruri", 0.00097603),
            ("Chepngeno", 0.00097361),
            ("Mutunga", 0.00096676),
            ("Mueni", 0.00096133),
            ("Njagi", 0.00095236),
            ("Ndegwa", 0.00094489),
            ("Karimi", 0.00094448),
            ("Nganga", 0.00094210),
            ("Nyabuto", 0.00093649),
            ("Wanjohi", 0.00093432),
            ("Kipruto", 0.00093103),
            ("David", 0.00092904),
            ("Aden", 0.00092874),
            ("Musa", 0.00092828),
            ("Chepkwony", 0.00092564),
            ("Sang", 0.00091808),
            ("Okello", 0.00091650),
            ("Oluoch", 0.00091096),
            ("Ndirangu", 0.00089738),
            ("Munyao", 0.00089649),
            ("Murimi", 0.00089649),
            ("Mumo", 0.00089301),
            ("Kipchirchir", 0.00088857),
            ("Mwikali", 0.00088389),
            ("Njogu", 0.00087718),
            ("Makena", 0.00087088),
            ("Owuor", 0.00086938),
            ("Ngeno", 0.00086624),
            ("Mukami", 0.00086581),
            ("Kerubo", 0.00086050),
            ("Githinji", 0.00085091),
            ("Joseph", 0.00084669),
            ("Kipkirui", 0.00084589),
            ("Makau", 0.00084089),
            ("Kendi", 0.00083649),
            ("Mwanzia", 0.00082822),
            ("Ondieki", 0.00081471),
            ("Mutheu", 0.00079864),
            ("Mulwa", 0.00079818),
            ("Muema", 0.00079775),
            ("Kilonzo", 0.00079355),
            ("Cherop", 0.00079030),
            ("Muturi", 0.00078030),
            ("Kiarie", 0.00077846),
            ("Musembi", 0.00077670),
            ("Waithira", 0.00077237),
            ("Mutiso", 0.00076769),
            ("Kipkemoi", 0.00076583),
            ("Karani", 0.00076141),
            ("Mwita", 0.00076001),
            ("Waithera", 0.00075825),
            ("Kirimi", 0.00075814),
            ("Gatwiri", 0.00075801),
            ("Wambugu", 0.00075749),
            ("Mumbi", 0.00075583),
            ("Ruto", 0.00075063),
            ("Makori", 0.00074760),
            ("Nanjala", 0.00073980),
            ("Maingi", 0.00073807),
            ("Chacha", 0.00073487),
            ("Mwangangi", 0.00073422),
            ("Muhumed", 0.00073257),
            ("Nduta", 0.00073090),
            ("Abdalla", 0.00072571),
            ("Njiru", 0.00071836),
            ("Wanjiru", 0.00071713),
            ("Tanui", 0.00070793),
            ("Nasimiyu", 0.00070630),
            ("Lagat", 0.00070600),
            ("Mwiti", 0.00070442),
            ("Kinuthia", 0.00070379),
            ("Kawira", 0.00070035),
            ("Ogutu", 0.00069920),
            ("Farah", 0.00069047),
            ("Rutto", 0.00068530),
            ("Guyo", 0.00068521),
            ("Oloo", 0.00068519),
            ("Musau", 0.00068508),
            ("Chirchir", 0.00068339),
            ("Yegon", 0.00068010),
            ("Nduku", 0.00067960),
            ("Salim", 0.00067584),
            ("Kwamboka", 0.00067449),
            ("Kinya", 0.00067309),
            ("Were", 0.00066293),
            ("Too", 0.00065353),
            ("Mutuma", 0.00065165),
            ("Odero", 0.00064543),
            ("James", 0.00064361),
            ("Abdirahman", 0.00064257),
            ("Bakari", 0.00063850),
            ("Mutwiri", 0.00063785),
            ("Gakii", 0.00063766),
            ("Omollo", 0.00063759),
            ("Mbithi", 0.00063348),
            ("Wesonga", 0.00062558),
            ("Mboya", 0.00061033),
            ("Kipsang", 0.00060966),
            ("Muia", 0.00059888),
            ("Masinde", 0.00059292),
            ("Gathoni", 0.00059175),
            ("Muthui", 0.00059145),
            ("Kiplimo", 0.00059097),
            ("Kipngeno", 0.00058985),
            ("Kipkemboi", 0.00058796),
            ("Paul", 0.00058790),
            ("Thuo", 0.00058437),
            ("Nkatha", 0.00057887),
            ("Wasike", 0.00057705),
            ("Ngigi", 0.00056945),
            ("Kipyegon", 0.00056823),
            ("Mokaya", 0.00056702),
            ("Mutembei", 0.00056587),
            ("Wandera", 0.00056529),
            ("Nderitu", 0.00056176),
            ("Mwai", 0.00055832),
            ("Nyawira", 0.00055503),
            ("Kimeu", 0.00055347),
            ("Jepchirchir", 0.00054991),
            ("Okeyo", 0.00054857),
            ("Naliaka", 0.00054820),
            ("Chepkurui", 0.00054634),
            ("Maiyo", 0.00054628),
            ("Jeptoo", 0.00054571),
            ("Samuel", 0.00054205),
            ("Bundi", 0.00053666),
            ("Ngari", 0.00053623),
            ("Mbogo", 0.00053374),
            ("Kipkosgei", 0.00053255),
            ("Kipkurui", 0.00053192),
            ("Chelimo", 0.00052947),
            ("Baya", 0.00052497),
            ("Ngala", 0.00052447),
            ("Waithaka", 0.00052040),
            ("Omari", 0.00052010),
            ("Aoko", 0.00051962),
            ("Kimanzi", 0.00051709),
            ("Ooko", 0.00051581),
            ("Njue", 0.00051410),
            ("Mutugi", 0.00051397),
            ("Ndunge", 0.00051286),
            ("Kenga", 0.00051046),
            ("Jepkoech", 0.00050942),
            ("Bosire", 0.00050671),
            ("Wangechi", 0.00050652),
            ("Ntinyari", 0.00050440),
            ("Kathambi", 0.00050355),
            ("Morara", 0.00050074),
            ("Nzioka", 0.00050013),
            ("Sifuna", 0.00049944),
            ("Kanini", 0.00049931),
            ("Mukhwana", 0.00049825),
            ("Njau", 0.00049407),
            ("Mwirigi", 0.00049151),
            ("Munga", 0.00049093),
            ("Tsuma", 0.00048902),
            ("Kirwa", 0.00048885),
            ("Kigen", 0.00048868),
            ("Mathenge", 0.00048606),
            ("Daniel", 0.00048118),
            ("Odiwuor", 0.00048090),
            ("Kassim", 0.00047889),
            ("Mutie", 0.00047839),
            ("Oketch", 0.00047711),
            ("Mose", 0.00047631),
            ("Kithinji", 0.00047282),
            ("Kathure", 0.00047172),
            ("Gichuki", 0.00047038),
            ("Oyugi", 0.00047010),
            ("Chepchirchir", 0.00046834),
            ("Tonui", 0.00046637),
            ("Mutethia", 0.00046594),
            ("Muthomi", 0.00046579),
            ("Jemutai", 0.00046206),
            ("Safari", 0.00046165),
            ("Cheruto", 0.00045951),
            ("Ekai", 0.00045946),
            ("Wangeci", 0.00045821),
            ("Biwott", 0.00045667),
            ("Tarus", 0.00045606),
            ("Koskei", 0.00045602),
            ("Keter", 0.00045357),
            ("Nyaboke", 0.00045340),
            ("Muthee", 0.00045173),
            ("Kamande", 0.00045106),
            ("Masika", 0.00044968),
            ("Khaemba", 0.00044859),
            ("Njambi", 0.00044825),
            ("Wangila", 0.00044760),
            ("Jerop", 0.00044693),
            ("Mogaka", 0.00044597),
            ("Issak", 0.00044493),
            ("Odera", 0.00044487),
            ("Kitsao", 0.00044411),
            ("Dahir", 0.00044366),
            ("Jebet", 0.00044296),
            ("Muthoka", 0.00044212),
            ("Haji", 0.00044097),
            ("Muigai", 0.00044084),
            ("Kanana", 0.00044075),
            ("Ojwang", 0.00044045),
            ("Waswa", 0.00043900),
            ("Nzomo", 0.00043692),
            ("Munyoki", 0.00043668),
            ("Osoro", 0.00043616),
            ("Kemei", 0.00043558),
            ("Kombo", 0.00043499),
            ("Hamisi", 0.00043402),
            ("Wanyoike", 0.00043374),
            ("Chesang", 0.00043296),
            ("Mugendi", 0.00043292),
            ("Owiti", 0.00043120),
            ("Jeruto", 0.00042858),
            ("Jepchumba", 0.00042646),
            ("Ronoh", 0.00042609),
            ("Yego", 0.00042581),
            ("Baraka", 0.00042319),
            ("Wahome", 0.00042315),
            ("Ireri", 0.00042109),
            ("Ogola", 0.00042075),
            ("Mbithe", 0.00041882),
            ("Aluoch", 0.00041860),
            ("Salat", 0.00041687),
            ("Nkirote", 0.00041349),
            ("Bii", 0.00041113),
            ("Nyamai", 0.00041068),
            ("Musyimi", 0.00041029),
            ("Ndanu", 0.00040879),
            ("Ismail", 0.00040756),
            ("Mbuthia", 0.00040596),
            ("Muendo", 0.00040479),
            ("Ngure", 0.00040271),
            ("Limo", 0.00040230),
            ("Ochola", 0.00040162),
            ("Moses", 0.00040143),
            ("Rop", 0.00040082),
            ("Nyokabi", 0.00040080),
            ("Komen", 0.00040011),
            ("Kihara", 0.00039872),
            ("Mmbone", 0.00039708),
            ("Mumbua", 0.00039673),
            ("Murugi", 0.00039649),
            ("Boru", 0.00039548),
            ("Ndinda", 0.00039483),
            ("Sikuku", 0.00039415),
            ("Muthama", 0.00039318),
            ("Kiragu", 0.00039296),
            ("Jepkosgei", 0.00039290),
            ("Mukiri", 0.00039255),
            ("Alio", 0.00039143),
            ("Wanjira", 0.00039104),
            ("Mokua", 0.00039093),
            ("Opondo", 0.00039000),
            ("Wario", 0.00038671),
            ("Muinde", 0.00038567),
            ("Suleiman", 0.00038398),
            ("Orina", 0.00038209),
            ("Khisa", 0.00038092),
            ("Ouko", 0.00037895),
            ("Jepkemboi", 0.00037742),
            ("Michael", 0.00037672),
            ("Murage", 0.00037508),
            ("Jerotich", 0.00037319),
            ("Kibe", 0.00037293),
            ("Gitari", 0.00037248),
            ("Chepchumba", 0.00037137),
            ("Ndolo", 0.00037044),
            ("Simon", 0.00036990),
            ("Mugure", 0.00036979),
            ("Abdille", 0.00036977),
            ("Abdallah", 0.00036704),
            ("Odoyo", 0.00036696),
            ("Wawira", 0.00036665),
            ("Kaingu", 0.00036455),
            ("Ombati", 0.00036336),
            ("Muhonja", 0.00036182),
            ("Muriungi", 0.00036174),
            ("Kimeli", 0.00036146),
            ("Stephen", 0.00036061),
            ("Nelima", 0.00035994),
            ("Musili", 0.00035890),
            ("Jepkorir", 0.00035730),
            ("Deng", 0.00035728),
            ("Toroitich", 0.00035652),
            ("Obonyo", 0.00035502),
            ("Ndiritu", 0.00035418),
            ("Omwenga", 0.00035412),
            ("Bashir", 0.00035383),
            ("Kalama", 0.00035349),
            ("Sigei", 0.00035225),
            ("Nabwire", 0.00035165),
            ("Kurgat", 0.00035128),
            ("Maritim", 0.00034989),
            ("Okinyi", 0.00034870),
            ("Wamaitha", 0.00034790),
            ("Emmanuel", 0.00034606),
            ("Rashid", 0.00034554),
            ("Mbuvi", 0.00034470),
            ("Thuku", 0.00034272),
            ("Sammy", 0.00034201),
            ("Bahati", 0.00034106),
            ("Obiero", 0.00034091),
            ("Ndege", 0.00033707),
            ("Koome", 0.00033673),
            ("Masai", 0.00033614),
            ("Gacheri", 0.00033612),
            ("Charles", 0.00033521),
            ("Nthenya", 0.00033510),
            ("Kinoti", 0.00033473),
            ("Muteti", 0.00033227),
            ("Shikuku", 0.00033060),
            ("Francis", 0.00033001),
            ("Mwanza", 0.00032878),
            ("Omolo", 0.00032614),
            ("Nzuki", 0.00032564),
            ("Mutegi", 0.00032508),
            ("Kiptanui", 0.00032350),
            ("Nzioki", 0.00032313),
            ("Ayuma", 0.00032211),
            ("Masha", 0.00032174),
            ("Chumba", 0.00032066),
            ("Jillo", 0.00032044),
            ("Nduati", 0.00031992),
            ("Kitheka", 0.00031968),
            ("Jerono", 0.00031958),
            ("Wayua", 0.00031890),
            ("Mureithi", 0.00031849),
            ("Ewoi", 0.00031631),
            ("Muthini", 0.00031505),
            ("Khalif", 0.00031488),
            ("Sila", 0.00031481),
            ("Mwihaki", 0.00031388),
            ("Maroa", 0.00031353),
            ("Ekiru", 0.00031323),
            ("Jelimo", 0.00031159),
            ("Ngina", 0.00030849),
            ("Ojiambo", 0.00030795),
            ("Chengo", 0.00030764),
            ("Kagendo", 0.00030730),
            ("Gichuhi", 0.00030704),
            ("Matheka", 0.00030680),
            ("Mulei", 0.00030654),
            ("Kangogo", 0.00030650),
            ("Brian", 0.00030615),
            ("George", 0.00030585),
            ("Mulongo", 0.00030524),
            ("Murigi", 0.00030327),
            ("Kiptum", 0.00030325),
            ("Kipyego", 0.00030314),
            ("Ndambuki", 0.00030273),
            ("Kandie", 0.00030236),
            ("Muasya", 0.00030219),
            ("Warui", 0.00030197),
            ("Kipkogei", 0.00030046),
            ("Chebii", 0.00029998),
            ("Kombe", 0.00029961),
            ("Jepleting", 0.00029924),
            ("Adow", 0.00029797),
            ("Ngatia", 0.00029777),
            ("Musungu", 0.00029725),
            ("Njihia", 0.00029716),
            ("Nyakio", 0.00029714),
            ("Murunga", 0.00029422),
            ("Amondi", 0.00029350),
            ("Wamuyu", 0.00029264),
            ("Abdikadir", 0.00029240),
            ("Muindi", 0.00029233),
            ("Mwalimu", 0.00029169),
            ("Chelagat", 0.00029119),
            ("Mutemi", 0.00029095),
            ("Thomas", 0.00029095),
            ("Ndwiga", 0.00028952),
            ("Serem", 0.00028926),
            ("Athman", 0.00028872),
            ("Yusuf", 0.00028865),
            ("Mong'are", 0.00028792),
            ("Kimanthi", 0.00028735),
            ("Nyang'au", 0.00028729),
            ("Adongo", 0.00028450),
            ("Mulinge", 0.00028369),
            ("Muya", 0.00028302),
            ("Chemtai", 0.00028272),
            ("Nyamweya", 0.00028136),
            ("Syombua", 0.00028034),
            ("Kalume", 0.00027991),
            ("Miriti", 0.00027943),
            ("Mahat", 0.00027934),
            ("Bwire", 0.00027923),
            ("Kamene", 0.00027644),
            ("Wanza", 0.00027419),
            ("Muchui", 0.00027402),
            ("Muoki", 0.00027341),
            ("Ndichu", 0.00027319),
            ("Yator", 0.00027274),
            ("Kimaiyo", 0.00027261),
            ("Ndunda", 0.00027233),
            ("Gikonyo", 0.00027181),
            ("Bosibori", 0.00027163),
            ("Mwania", 0.00027046),
            ("Muthuri", 0.00026984),
            ("Ogolla", 0.00026923),
            ("Thuranira", 0.00026895),
            ("Musimbi", 0.00026856),
            ("Wabwire", 0.00026689),
            ("Nyaguthii", 0.00026639),
            ("Munywoki", 0.00026568),
            ("Muhia", 0.00026566),
            ("Kadenge", 0.00026366),
            ("Thoya", 0.00026343),
            ("Keya", 0.00026317),
            ("Okech", 0.00026265),
            ("Kiio", 0.00026208),
            ("Mongare", 0.00026068),
            ("Mwongeli", 0.00025721),
            ("Ongeri", 0.00025715),
            ("Onsongo", 0.00025704),
            ("Kennedy", 0.00025643),
            ("Mbatha", 0.00025561),
            ("Joshua", 0.00025550),
            ("Nyangau", 0.00025392),
            ("Githaiga", 0.00025373),
            ("Maithya", 0.00025342),
            ("Edin", 0.00025175),
            ("Kanyi", 0.00025104),
            ("Kogo", 0.00025007),
            ("Muchoki", 0.00024942),
            ("Maloba", 0.00024859),
            ("Isaac", 0.00024682),
            ("Kimotho", 0.00024669),
            ("Ndiema", 0.00024638),
            ("Muuo", 0.00024612),
            ("Etyang", 0.00024576),
            ("Kipkemei", 0.00024487),
            ("Ngui", 0.00024483),
            ("Julius", 0.00024478),
            ("Njuki", 0.00024450),
            ("Kipleting", 0.00024428),
            ("Wabwile", 0.00024327),
            ("Jepkirui", 0.00024197),
            ("Sakwa", 0.00024186),
            ("Kananu", 0.00024052),
            ("Masila", 0.00024021),
            ("Marwa", 0.00023935),
            ("Mutindi", 0.00023865),
            ("Kundu", 0.00023844),
            ("Amina", 0.00023805),
            ("Kang'ethe", 0.00023785),
            ("Mwaka", 0.00023783),
            ("Onyancha", 0.00023751),
            ("Galgallo", 0.00023716),
            ("Kaburu", 0.00023621),
            ("Godana", 0.00023458),
            ("Kilonzi", 0.00023443),
            ("Bare", 0.00023441),
            ("Muchai", 0.00023369),
            ("Wasonga", 0.00023298),
            ("Naibei", 0.00023237),
            ("Ngumbao", 0.00023216),
            ("Kiboi", 0.00023168),
            ("Koros", 0.00023164),
            ("Rioba", 0.00023127),
            ("Philip", 0.00023034),
            ("Nyambane", 0.00023032),
            ("Muthiani", 0.00023027),
            ("Ngunjiri", 0.00022982),
            ("King'ori", 0.00022941),
            ("Ndiwa", 0.00022928),
            ("Wangare", 0.00022923),
            ("Esekon", 0.00022908),
            ("Kivuva", 0.00022893),
            ("Kobia", 0.00022891),
            ("Akumu", 0.00022887),
            ("Ramadhan", 0.00022776),
            ("Esinyen", 0.00022744),
            ("Apiyo", 0.00022716),
            ("Kasyoka", 0.00022679),
            ("Wako", 0.00022612),
            ("Situma", 0.00022560),
            ("Kadzo", 0.00022547),
            ("Mbula", 0.00022527),
            ("Gichuru", 0.00022516),
            ("Oyoo", 0.00022484),
            ("William", 0.00022464),
            ("Gitahi", 0.00022460),
            ("Nyamawi", 0.00022382),
            ("Kibor", 0.00022350),
            ("Nzau", 0.00022341),
            ("Wawire", 0.00022304),
            ("Ngei", 0.00022293),
            ("Cheboi", 0.00022181),
            ("Kagwiria", 0.00022170),
            ("Okelo", 0.00022163),
            ("Jimale", 0.00022096),
            ("Moseti", 0.00022046),
            ("Patrick", 0.00022029),
            ("Martin", 0.00021906),
            ("Kitur", 0.00021869),
            ("Ogeto", 0.00021839),
            ("Kiiru", 0.00021810),
            ("Ogega", 0.00021802),
            ("Khamis", 0.00021782),
            ("Mawira", 0.00021724),
            ("Jepngetich", 0.00021711),
            ("Nur", 0.00021663),
            ("Boke", 0.00021592),
            ("Maundu", 0.00021576),
            ("Tuwei", 0.00021576),
            ("Nyangweso", 0.00021557),
            ("Wakhungu", 0.00021442),
            ("Kungu", 0.00021431),
            ("Wanjau", 0.00021353),
            ("Ekal", 0.00021327),
            ("Lumumba", 0.00021306),
            ("Sitienei", 0.00021288),
            ("Muraya", 0.00021239),
            ("Kiilu", 0.00021159),
            ("Nyamu", 0.00021046),
            ("Orwa", 0.00021035),
            ("Waruguru", 0.00021029),
            ("Kimaru", 0.00021024),
            ("Fondo", 0.00021007),
            ("Muktar", 0.00020979),
            ("Kangethe", 0.00020877),
            ("Jepkogei", 0.00020825),
            ("Nasambu", 0.00020741),
            ("Ekeno", 0.00020667),
            ("Ngige", 0.00020661),
            ("Obuya", 0.00020600),
            ("Okemwa", 0.00020509),
            ("Robert", 0.00020453),
            ("Robi", 0.00020427),
            ("Matano", 0.00020368),
            ("Mwongela", 0.00020299),
            ("Mutia", 0.00020297),
            ("Chumo", 0.00020240),
            ("Karambu", 0.00020238),
            ("Munyi", 0.00020219),
            ("Sitati", 0.00020186),
            ("Samson", 0.00020175),
            ("Jacob", 0.00020126),
            ("Gacheru", 0.00020110),
            ("Cherutich", 0.00020015),
            ("Abubakar", 0.00019981),
            ("Chepkemboi", 0.00019959),
            ("Abuga", 0.00019957),
            ("Mangale", 0.00019933),
            ("Muthike", 0.00019896),
            ("Jackson", 0.00019890),
            ("Mahamud", 0.00019883),
            ("Mawia", 0.00019879),
            ("Kimtai", 0.00019866),
            ("Nyagaka", 0.00019825),
            ("Wambura", 0.00019822),
            ("Odira", 0.00019790),
            ("Khamisi", 0.00019773),
            ("Kurui", 0.00019758),
            ("Murangiri", 0.00019589),
            ("Kairu", 0.00019584),
            ("Omwoyo", 0.00019498),
            ("Mbae", 0.00019493),
            ("Lelei", 0.00019457),
            ("Issa", 0.00019420),
            ("Kung'u", 0.00019340),
            ("Samoei", 0.00019311),
            ("Obare", 0.00019298),
            ("Theuri", 0.00019251),
            ("Samwel", 0.00019227),
            ("Wanga", 0.00019214),
            ("Roba", 0.00019153),
            ("Soita", 0.00019060),
            ("Mbaabu", 0.00019023),
            ("Mwero", 0.00019017),
            ("Ogada", 0.00019006),
            ("Lewa", 0.00018991),
            ("Kingori", 0.00018930),
            ("Obara", 0.00018922),
            ("Mutanu", 0.00018917),
            ("Sawe", 0.00018917),
            ("Murungi", 0.00018909),
            ("Awinja", 0.00018881),
            ("Nthiga", 0.00018876),
            ("Siele", 0.00018865),
            ("Baraza", 0.00018852),
            ("Nzai", 0.00018820),
            ("Mogire", 0.00018807),
            ("Mwinzi", 0.00018686),
            ("Maluki", 0.00018673),
            ("Muthengi", 0.00018593),
            ("Bor", 0.00018577),
            ("Galgalo", 0.00018545),
            ("Mbiti", 0.00018543),
            ("Mauti", 0.00018536),
            ("Oginga", 0.00018534),
            ("Dida", 0.00018523),
            ("Nzilani", 0.00018495),
            ("Micheni", 0.00018478),
            ("Mitei", 0.00018460),
            ("Munyua", 0.00018456),
            ("Ogweno", 0.00018376),
            ("Terer", 0.00018346),
            ("Nafula", 0.00018311),
            ("Titus", 0.00018305),
            ("Ekitela", 0.00018292),
            ("Ambani", 0.00018211),
            ("Amoit", 0.00018168),
            ("Halake", 0.00018136),
            ("Naserian", 0.00018025),
            ("Cheyech", 0.00018008),
            ("Mumbe", 0.00017973),
            ("Gati", 0.00017945),
            ("Maweu", 0.00017884),
            ("Karuga", 0.00017878),
            ("Ndoro", 0.00017832),
            ("Ruwa", 0.00017832),
            ("Mwadime", 0.00017822),
            ("Duba", 0.00017796),
            ("Kageha", 0.00017765),
            ("Muga", 0.00017763),
            ("Krop", 0.00017744),
            ("Daud", 0.00017728),
            ("Nyanchama", 0.00017722),
            ("Muchira", 0.00017703),
            ("Akello", 0.00017672),
            ("Emuria", 0.00017629),
            ("Chesire", 0.00017612),
            ("Musee", 0.00017586),
            ("Njoka", 0.00017555),
            ("Ondiek", 0.00017540),
            ("Anyona", 0.00017536),
            ("Zawadi", 0.00017531),
            ("Huka", 0.00017516),
            ("Mahamed", 0.00017499),
            ("Yakub", 0.00017475),
            ("Kaberia", 0.00017432),
            ("Kisilu", 0.00017412),
            ("Thiong'o", 0.00017393),
            ("Busienei", 0.00017386),
            ("Wavinya", 0.00017373),
            ("Mwinyi", 0.00017354),
            ("Ombui", 0.00017265),
            ("Mukundi", 0.00017235),
            ("Hillow", 0.00017189),
            ("Keitany", 0.00017174),
            ("Chomba", 0.00017139),
            ("Bulle", 0.00017116),
            ("Mwema", 0.00017105),
            ("Karuri", 0.00017103),
            ("Mwathi", 0.00017053),
            ("Soi", 0.00017029),
            ("Evans", 0.00016901),
            ("Imali", 0.00016897),
            ("Migwi", 0.00016834),
            ("Ratemo", 0.00016817),
            ("Mutio", 0.00016780),
            ("Alex", 0.00016778),
            ("Misiko", 0.00016758),
            ("Oyaro", 0.00016704),
            ("Abdulla", 0.00016693),
            ("Benson", 0.00016689),
            ("Kithome", 0.00016689),
            ("Mwamburi", 0.00016678),
            ("Ngome", 0.00016672),
            ("Ayieko", 0.00016670),
            ("Okuku", 0.00016670),
            ("Richard", 0.00016646),
            ("Eyanae", 0.00016618),
            ("Kabiru", 0.00016570),
            ("Gichohi", 0.00016557),
            ("Nyabuti", 0.00016546),
            ("Abdow", 0.00016540),
            ("Joel", 0.00016540),
            ("Kitonga", 0.00016514),
            ("Nyale", 0.00016503),
            ("Towett", 0.00016499),
            ("Mati", 0.00016490),
            ("Ekidor", 0.00016479),
            ("Mutahi", 0.00016470),
            ("Mmboga", 0.00016442),
            ("Oduori", 0.00016384),
            ("Changawa", 0.00016379),
            ("Nyangaresi", 0.00016373),
            ("Maranga", 0.00016332),
            ("Kalekye", 0.00016308),
            ("Wanje", 0.00016297),
            ("Kiama", 0.00016288),
            ("Muimi", 0.00016278),
            ("Gicheru", 0.00016195),
            ("Salah", 0.00016193),
            ("Eregae", 0.00016172),
            ("Jarso", 0.00016163),
            ("Sande", 0.00016133),
            ("Kakai", 0.00016122),
            ("Swaleh", 0.00016000),
            ("Chenangat", 0.00015990),
            ("Ngao", 0.00015936),
            ("Nyambu", 0.00015920),
            ("Onditi", 0.00015873),
            ("Bonaya", 0.00015840),
            ("Nandwa", 0.00015834),
            ("Chebon", 0.00015797),
            ("Mwakio", 0.00015784),
            ("Mnangat", 0.00015596),
            ("Lemayian", 0.00015593),
            ("Erupe", 0.00015565),
            ("Rugut", 0.00015557),
            ("Mule", 0.00015552),
            ("Gona", 0.00015524),
            ("Maitha", 0.00015511),
            ("Gachoki", 0.00015500),
            ("Musila", 0.00015498),
            ("Odinga", 0.00015476),
            ("Dennis", 0.00015446),
            ("Gichana", 0.00015444),
            ("Chai", 0.00015385),
            ("Muhoro", 0.00015385),
            ("Jama", 0.00015364),
            ("Nakhumicha", 0.00015359),
            ("Jemeli", 0.00015355),
            ("Mundia", 0.00015347),
            ("Junior", 0.00015299),
            ("Wamboi", 0.00015292),
            ("Vaati", 0.00015275),
            ("Fundi", 0.00015249),
            ("Menza", 0.00015219),
            ("Muchemi", 0.00015193),
            ("Victor", 0.00015178),
            ("Kituku", 0.00015141),
            ("Kiogora", 0.00015110),
            ("Makungu", 0.00015063),
            ("Pkemoi", 0.00015035),
            ("Jumba", 0.00014987),
            ("Wechuli", 0.00014955),
            ("Moturi", 0.00014935),
            ("Kajuju", 0.00014918),
            ("Mayaka", 0.00014905),
            ("Yatich", 0.00014853),
            ("Mbaka", 0.00014851),
            ("Kibiwott", 0.00014831),
            ("Mucheru", 0.00014825),
            ("Gure", 0.00014822),
            ("Ngaruiya", 0.00014814),
            ("Okongo", 0.00014792),
            ("Munguti", 0.00014783),
            ("Ngotho", 0.00014783),
            ("Mamo", 0.00014779),
            ("Muthami", 0.00014747),
            ("Benard", 0.00014738),
            ("Ondari", 0.00014725),
            ("Otiende", 0.00014714),
            ("Mzungu", 0.00014710),
            ("Ombasa", 0.00014671),
            ("Kimosop", 0.00014645),
            ("Jelagat", 0.00014632),
            ("Muchangi", 0.00014632),
            ("Aloo", 0.00014612),
            ("Abdiaziz", 0.00014606),
            ("Andrew", 0.00014582),
            ("Gedi", 0.00014582),
            ("Matara", 0.00014580),
            ("Kelvin", 0.00014500),
            ("Mwendia", 0.00014500),
            ("Karwitha", 0.00014411),
            ("Apondi", 0.00014374),
            ("Barongo", 0.00014372),
            ("Chepkwemoi", 0.00014370),
            ("Mukoya", 0.00014366),
            ("Minayo", 0.00014346),
            ("Masese", 0.00014344),
            ("Ziro", 0.00014331),
            ("Kwemoi", 0.00014314),
            ("Wakio", 0.00014272),
            ("Munyiva", 0.00014266),
            ("Saidi", 0.00014262),
            ("Chepkosgei", 0.00014246),
            ("Yaa", 0.00014223),
            ("Achola", 0.00014218),
            ("Mambo", 0.00014210),
            ("Jelle", 0.00014158),
            ("Tirop", 0.00014156),
            ("Mbui", 0.00014147),
            ("Komora", 0.00014132),
            ("Abdifatah", 0.00014130),
            ("Mageto", 0.00014125),
            ("Kamathi", 0.00014123),
            ("Kanja", 0.00014123),
            ("Gathogo", 0.00014078),
            ("Mutune", 0.00014078),
            ("Wabomba", 0.00014054),
            ("Kiptui", 0.00014047),
            ("Gatimu", 0.00014026),
            ("Jomo", 0.00014026),
            ("Meli", 0.00014010),
            ("Dagane", 0.00014002),
            ("Mutungi", 0.00013971),
            ("Mumba", 0.00013945),
            ("Weru", 0.00013941),
            ("Kabui", 0.00013893),
            ("Kaimenyi", 0.00013893),
            ("Kimemia", 0.00013880),
            ("Musya", 0.00013868),
            ("Jepkemei", 0.00013835),
            ("Saina", 0.00013818),
            ("Kaloki", 0.00013816),
            ("Elijah", 0.00013794),
            ("Kibiwot", 0.00013792),
            ("Kiti", 0.00013768),
            ("Nzisa", 0.00013751),
            ("Kiema", 0.00013727),
            ("Mogeni", 0.00013718),
            ("Wanjugu", 0.00013712),
            ("Birgen", 0.00013703),
            ("Shariff", 0.00013699),
            ("Gachanja", 0.00013694),
            ("Osiemo", 0.00013681),
            ("Onyiego", 0.00013679),
            ("Iminza", 0.00013638),
            ("Muange", 0.00013618),
            ("Musumba", 0.00013616),
            ("Bore", 0.00013603),
            ("Saruni", 0.00013588),
            ("Mwololo", 0.00013586),
            ("Abdinoor", 0.00013571),
            ("Awadh", 0.00013521),
            ("Jefwa", 0.00013504),
            ("Lumbasi", 0.00013504),
            ("Wangai", 0.00013467),
            ("Nyoike", 0.00013463),
            ("Wanzala", 0.00013450),
            ("Saitoti", 0.00013441),
            ("Adam", 0.00013421),
            ("Mogere", 0.00013365),
            ("Benjamin", 0.00013363),
            ("Kaari", 0.00013330),
            ("Muraguri", 0.00013322),
            ("Sidi", 0.00013320),
            ("Nyawa", 0.00013291),
            ("Kevin", 0.00013266),
            ("Omweri", 0.00013253),
            ("Maghanga", 0.00013242),
            ("Nafuna", 0.00013242),
            ("Angwenyi", 0.00013235),
        ]
    )

    # Prefixes for Kenyan English
    # Includes common professional, religious and honorific titles used in Kenya
    prefixes_male = ("Mr", "Dr", "Eng", "Prof", "Hon", "Rev", "Pastor", "Fr", "Bishop")

    prefixes_female = ("Mrs", "Ms", "Dr", "Eng", "Prof", "Hon", "Rev", "Pastor", "Sis")
