{-# LANGUAGE OverloadedStrings #-}
module KAT_PubKey.PSS (pssTests) where

import Crypto.PubKey.RSA
import qualified Crypto.PubKey.RSA.PSS as PSS

import Imports

-- Module contains one vector generated by the implementation itself and other
-- vectors from <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>

data VectorPSS = VectorPSS { message :: ByteString
                           , salt :: ByteString
                           , signature :: ByteString
                           }

rsaKeyInt = PrivateKey
    { private_pub = PublicKey
        { public_n = 0xa2ba40ee07e3b2bd2f02ce227f36a195024486e49c19cb41bbbdfbba98b22b0e577c2eeaffa20d883a76e65e394c69d4b3c05a1e8fadda27edb2a42bc000fe888b9b32c22d15add0cd76b3e7936e19955b220dd17d4ea904b1ec102b2e4de7751222aa99151024c7cb41cc5ea21d00eeb41f7c800834d2c6e06bce3bce7ea9a5
        , public_e = 0x010001
        , public_size = 128
        }
    , private_d = 0x50e2c3e38d886110288dfc68a9533e7e12e27d2aa56d2cdb3fb6efa990bcff29e1d2987fb711962860e7391b1ce01ebadb9e812d2fbdfaf25df4ae26110a6d7a26f0b810f54875e17dd5c9fb6d641761245b81e79f8c88f0e55a6dcd5f133abd35f8f4ec80adf1bf86277a582894cb6ebcd2162f1c7534f1f4947b129151b71
    , private_p = 0xd17f655bf27c8b16d35462c905cc04a26f37e2a67fa9c0ce0dced472394a0df743fe7f929e378efdb368eddff453cf007af6d948e0ade757371f8a711e278f6b
    , private_q = 0xc6d92b6fee7414d1358ce1546fb62987530b90bd15e0f14963a5e2635adb69347ec0c01b2ab1763fd8ac1a592fb22757463a982425bb97a3a437c5bf86d03f2f
    , private_dP = 0x9d0dbf83e5ce9e4b1754dcd5cd05bcb7b55f1508330ea49f14d4e889550f8256cb5f806dff34b17ada44208853577d08e4262890acf752461cea05547601bc4f
    , private_dQ = 0x1291a524c6b7c059e90e46dc83b2171eb3fa98818fd179b6c8bf6cecaa476303abf283fe05769cfc495788fe5b1ddfde9e884a3cd5e936b7e955ebf97eb563b1
    , private_qinv = 0xa63f1da38b950c9ad1c67ce0d677ec2914cd7d40062df42a67eb198a176f9742aac7c5fea14f2297662b84812c4defc49a8025ab4382286be4c03788dd01d69f
    }

rsaKey1 = PrivateKey
    { private_pub = PublicKey
        { public_n = 0xa56e4a0e701017589a5187dc7ea841d156f2ec0e36ad52a44dfeb1e61f7ad991d8c51056ffedb162b4c0f283a12a88a394dff526ab7291cbb307ceabfce0b1dfd5cd9508096d5b2b8b6df5d671ef6377c0921cb23c270a70e2598e6ff89d19f105acc2d3f0cb35f29280e1386b6f64c4ef22e1e1f20d0ce8cffb2249bd9a2137
        , public_e = 0x010001
        , public_size = 128
        }
    , private_d = 0x33a5042a90b27d4f5451ca9bbbd0b44771a101af884340aef9885f2a4bbe92e894a724ac3c568c8f97853ad07c0266c8c6a3ca0929f1e8f11231884429fc4d9ae55fee896a10ce707c3ed7e734e44727a39574501a532683109c2abacaba283c31b4bd2f53c3ee37e352cee34f9e503bd80c0622ad79c6dcee883547c6a3b325
    , private_p = 0xe7e8942720a877517273a356053ea2a1bc0c94aa72d55c6e86296b2dfc967948c0a72cbccca7eacb35706e09a1df55a1535bd9b3cc34160b3b6dcd3eda8e6443
    , private_q = 0xb69dca1cf7d4d7ec81e75b90fcca874abcde123fd2700180aa90479b6e48de8d67ed24f9f19d85ba275874f542cd20dc723e6963364a1f9425452b269a6799fd
    , private_dP = 0x28fa13938655be1f8a159cbaca5a72ea190c30089e19cd274a556f36c4f6e19f554b34c077790427bbdd8dd3ede2448328f385d81b30e8e43b2fffa027861979
    , private_dQ = 0x1a8b38f398fa712049898d7fb79ee0a77668791299cdfa09efc0e507acb21ed74301ef5bfd48be455eaeb6e1678255827580a8e4e8e14151d1510a82a3f2e729
    , private_qinv = 0x27156aba4126d24a81f3a528cbfb27f56886f840a9f6e86e17a44b94fe9319584b8e22fdde1e5a2e3bd8aa5ba8d8584194eb2190acf832b847f13a3d24a79f4d
    }

vectorInt = VectorPSS
    { message = "\x85\x9e\xef\x2f\xd7\x8a\xca\x00\x30\x8b\xdc\x47\x11\x93\xbf\x55\xbf\x9d\x78\xdb\x8f\x8a\x67\x2b\x48\x46\x34\xf3\xc9\xc2\x6e\x64\x78\xae\x10\x26\x0f\xe0\xdd\x8c\x08\x2e\x53\xa5\x29\x3a\xf2\x17\x3c\xd5\x0c\x6d\x5d\x35\x4f\xeb\xf7\x8b\x26\x02\x1c\x25\xc0\x27\x12\xe7\x8c\xd4\x69\x4c\x9f\x46\x97\x77\xe4\x51\xe7\xf8\xe9\xe0\x4c\xd3\x73\x9c\x6b\xbf\xed\xae\x48\x7f\xb5\x56\x44\xe9\xca\x74\xff\x77\xa5\x3c\xb7\x29\x80\x2f\x6e\xd4\xa5\xff\xa8\xba\x15\x98\x90\xfc"
    , salt = "\xe3\xb5\xd5\xd0\x02\xc1\xbc\xe5\x0c\x2b\x65\xef\x88\xa1\x88\xd8\x3b\xce\x7e\x61"
    , signature = "\x8d\xaa\x62\x7d\x3d\xe7\x59\x5d\x63\x05\x6c\x7e\xc6\x59\xe5\x44\x06\xf1\x06\x10\x12\x8b\xaa\xe8\x21\xc8\xb2\xa0\xf3\x93\x6d\x54\xdc\x3b\xdc\xe4\x66\x89\xf6\xb7\x95\x1b\xb1\x8e\x84\x05\x42\x76\x97\x18\xd5\x71\x5d\x21\x0d\x85\xef\xbb\x59\x61\x92\x03\x2c\x42\xbe\x4c\x29\x97\x2c\x85\x62\x75\xeb\x6d\x5a\x45\xf0\x5f\x51\x87\x6f\xc6\x74\x3d\xed\xdd\x28\xca\xec\x9b\xb3\x0e\xa9\x9e\x02\xc3\x48\x82\x69\x60\x4f\xe4\x97\xf7\x4c\xcd\x7c\x7f\xca\x16\x71\x89\x71\x23\xcb\xd3\x0d\xef\x5d\x54\xa2\xb5\x53\x6a\xd9\x0a\x74\x7e"
    }

{-
# mHash    = Hash(M)
# salt     = random string of octets
# M'       = Padding || mHash || salt
# H        = Hash(M')
# DB       = Padding || salt 
# dbMask   = MGF(H, length(DB))
# maskedDB = DB xor dbMask (leftmost bit set to
#            zero)
# EM       = maskedDB || H || 0xbc

# mHash:
37 b6 6a e0 44 58 43 35 3d 47 ec b0 b4 fd 14 c1 
10 e6 2d 6a 

# salt:

# M':
00 00 00 00 00 00 00 00 37 b6 6a e0 44 58 43 35 
3d 47 ec b0 b4 fd 14 c1 10 e6 2d 6a e3 b5 d5 d0 
02 c1 bc e5 0c 2b 65 ef 88 a1 88 d8 3b ce 7e 61 

# H:
df 1a 89 6f 9d 8b c8 16 d9 7c d7 a2 c4 3b ad 54 
6f be 8c fe 

# DB:
00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 
00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 
00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 
00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 
00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 
00 00 00 00 00 00 01 e3 b5 d5 d0 02 c1 bc e5 0c 
2b 65 ef 88 a1 88 d8 3b ce 7e 61 

# dbMask:
66 e4 67 2e 83 6a d1 21 ba 24 4b ed 65 76 b8 67 
d9 a4 47 c2 8a 6e 66 a5 b8 7d ee 7f bc 7e 65 af 
50 57 f8 6f ae 89 84 d9 ba 7f 96 9a d6 fe 02 a4 
d7 5f 74 45 fe fd d8 5b 6d 3a 47 7c 28 d2 4b a1 
e3 75 6f 79 2d d1 dc e8 ca 94 44 0e cb 52 79 ec 
d3 18 3a 31 1f c8 97 39 a9 66 43 13 6e 8b 0f 46 
5e 87 a4 53 5c d4 c5 9b 10 02 8d 

# maskedDB:
66 e4 67 2e 83 6a d1 21 ba 24 4b ed 65 76 b8 67 
d9 a4 47 c2 8a 6e 66 a5 b8 7d ee 7f bc 7e 65 af 
50 57 f8 6f ae 89 84 d9 ba 7f 96 9a d6 fe 02 a4 
d7 5f 74 45 fe fd d8 5b 6d 3a 47 7c 28 d2 4b a1 
e3 75 6f 79 2d d1 dc e8 ca 94 44 0e cb 52 79 ec 
d3 18 3a 31 1f c8 96 da 1c b3 93 11 af 37 ea 4a 
75 e2 4b db fd 5c 1d a0 de 7c ec 

# Encoded message EM:
66 e4 67 2e 83 6a d1 21 ba 24 4b ed 65 76 b8 67 
d9 a4 47 c2 8a 6e 66 a5 b8 7d ee 7f bc 7e 65 af 
50 57 f8 6f ae 89 84 d9 ba 7f 96 9a d6 fe 02 a4 
d7 5f 74 45 fe fd d8 5b 6d 3a 47 7c 28 d2 4b a1 
e3 75 6f 79 2d d1 dc e8 ca 94 44 0e cb 52 79 ec 
d3 18 3a 31 1f c8 96 da 1c b3 93 11 af 37 ea 4a 
75 e2 4b db fd 5c 1d a0 de 7c ec df 1a 89 6f 9d 
8b c8 16 d9 7c d7 a2 c4 3b ad 54 6f be 8c fe bc 
-}

vectorsKey1 =
    [
    -- Example 1.1
      VectorPSS
        { message = "\xcd\xc8\x7d\xa2\x23\xd7\x86\xdf\x3b\x45\xe0\xbb\xbc\x72\x13\x26\xd1\xee\x2a\xf8\x06\xcc\x31\x54\x75\xcc\x6f\x0d\x9c\x66\xe1\xb6\x23\x71\xd4\x5c\xe2\x39\x2e\x1a\xc9\x28\x44\xc3\x10\x10\x2f\x15\x6a\x0d\x8d\x52\xc1\xf4\xc4\x0b\xa3\xaa\x65\x09\x57\x86\xcb\x76\x97\x57\xa6\x56\x3b\xa9\x58\xfe\xd0\xbc\xc9\x84\xe8\xb5\x17\xa3\xd5\xf5\x15\xb2\x3b\x8a\x41\xe7\x4a\xa8\x67\x69\x3f\x90\xdf\xb0\x61\xa6\xe8\x6d\xfa\xae\xe6\x44\x72\xc0\x0e\x5f\x20\x94\x57\x29\xcb\xeb\xe7\x7f\x06\xce\x78\xe0\x8f\x40\x98\xfb\xa4\x1f\x9d\x61\x93\xc0\x31\x7e\x8b\x60\xd4\xb6\x08\x4a\xcb\x42\xd2\x9e\x38\x08\xa3\xbc\x37\x2d\x85\xe3\x31\x17\x0f\xcb\xf7\xcc\x72\xd0\xb7\x1c\x29\x66\x48\xb3\xa4\xd1\x0f\x41\x62\x95\xd0\x80\x7a\xa6\x25\xca\xb2\x74\x4f\xd9\xea\x8f\xd2\x23\xc4\x25\x37\x02\x98\x28\xbd\x16\xbe\x02\x54\x6f\x13\x0f\xd2\xe3\x3b\x93\x6d\x26\x76\xe0\x8a\xed\x1b\x73\x31\x8b\x75\x0a\x01\x67\xd0"
        , salt = "\xde\xe9\x59\xc7\xe0\x64\x11\x36\x14\x20\xff\x80\x18\x5e\xd5\x7f\x3e\x67\x76\xaf"
        , signature = "\x90\x74\x30\x8f\xb5\x98\xe9\x70\x1b\x22\x94\x38\x8e\x52\xf9\x71\xfa\xac\x2b\x60\xa5\x14\x5a\xf1\x85\xdf\x52\x87\xb5\xed\x28\x87\xe5\x7c\xe7\xfd\x44\xdc\x86\x34\xe4\x07\xc8\xe0\xe4\x36\x0b\xc2\x26\xf3\xec\x22\x7f\x9d\x9e\x54\x63\x8e\x8d\x31\xf5\x05\x12\x15\xdf\x6e\xbb\x9c\x2f\x95\x79\xaa\x77\x59\x8a\x38\xf9\x14\xb5\xb9\xc1\xbd\x83\xc4\xe2\xf9\xf3\x82\xa0\xd0\xaa\x35\x42\xff\xee\x65\x98\x4a\x60\x1b\xc6\x9e\xb2\x8d\xeb\x27\xdc\xa1\x2c\x82\xc2\xd4\xc3\xf6\x6c\xd5\x00\xf1\xff\x2b\x99\x4d\x8a\x4e\x30\xcb\xb3\x3c"
        }
    -- Example 1.2
    , VectorPSS
        { message = "\x85\x13\x84\xcd\xfe\x81\x9c\x22\xed\x6c\x4c\xcb\x30\xda\xeb\x5c\xf0\x59\xbc\x8e\x11\x66\xb7\xe3\x53\x0c\x4c\x23\x3e\x2b\x5f\x8f\x71\xa1\xcc\xa5\x82\xd4\x3e\xcc\x72\xb1\xbc\xa1\x6d\xfc\x70\x13\x22\x6b\x9e"
        , salt = "\xef\x28\x69\xfa\x40\xc3\x46\xcb\x18\x3d\xab\x3d\x7b\xff\xc9\x8f\xd5\x6d\xf4\x2d"
        , signature = "\x3e\xf7\xf4\x6e\x83\x1b\xf9\x2b\x32\x27\x41\x42\xa5\x85\xff\xce\xfb\xdc\xa7\xb3\x2a\xe9\x0d\x10\xfb\x0f\x0c\x72\x99\x84\xf0\x4e\xf2\x9a\x9d\xf0\x78\x07\x75\xce\x43\x73\x9b\x97\x83\x83\x90\xdb\x0a\x55\x05\xe6\x3d\xe9\x27\x02\x8d\x9d\x29\xb2\x19\xca\x2c\x45\x17\x83\x25\x58\xa5\x5d\x69\x4a\x6d\x25\xb9\xda\xb6\x60\x03\xc4\xcc\xcd\x90\x78\x02\x19\x3b\xe5\x17\x0d\x26\x14\x7d\x37\xb9\x35\x90\x24\x1b\xe5\x1c\x25\x05\x5f\x47\xef\x62\x75\x2c\xfb\xe2\x14\x18\xfa\xfe\x98\xc2\x2c\x4d\x4d\x47\x72\x4f\xdb\x56\x69\xe8\x43"
        }
    -- Example 1.3
    , VectorPSS
        { message = "\xa4\xb1\x59\x94\x17\x61\xc4\x0c\x6a\x82\xf2\xb8\x0d\x1b\x94\xf5\xaa\x26\x54\xfd\x17\xe1\x2d\x58\x88\x64\x67\x9b\x54\xcd\x04\xef\x8b\xd0\x30\x12\xbe\x8d\xc3\x7f\x4b\x83\xaf\x79\x63\xfa\xff\x0d\xfa\x22\x54\x77\x43\x7c\x48\x01\x7f\xf2\xbe\x81\x91\xcf\x39\x55\xfc\x07\x35\x6e\xab\x3f\x32\x2f\x7f\x62\x0e\x21\xd2\x54\xe5\xdb\x43\x24\x27\x9f\xe0\x67\xe0\x91\x0e\x2e\x81\xca\x2c\xab\x31\xc7\x45\xe6\x7a\x54\x05\x8e\xb5\x0d\x99\x3c\xdb\x9e\xd0\xb4\xd0\x29\xc0\x6d\x21\xa9\x4c\xa6\x61\xc3\xce\x27\xfa\xe1\xd6\xcb\x20\xf4\x56\x4d\x66\xce\x47\x67\x58\x3d\x0e\x5f\x06\x02\x15\xb5\x90\x17\xbe\x85\xea\x84\x89\x39\x12\x7b\xd8\xc9\xc4\xd4\x7b\x51\x05\x6c\x03\x1c\xf3\x36\xf1\x7c\x99\x80\xf3\xb8\xf5\xb9\xb6\x87\x8e\x8b\x79\x7a\xa4\x3b\x88\x26\x84\x33\x3e\x17\x89\x3f\xe9\xca\xa6\xaa\x29\x9f\x7e\xd1\xa1\x8e\xe2\xc5\x48\x64\xb7\xb2\xb9\x9b\x72\x61\x8f\xb0\x25\x74\xd1\x39\xef\x50\xf0\x19\xc9\xee\xf4\x16\x97\x13\x38\xe7\xd4\x70"
        , salt = "\x71\x0b\x9c\x47\x47\xd8\x00\xd4\xde\x87\xf1\x2a\xfd\xce\x6d\xf1\x81\x07\xcc\x77"
        , signature = "\x66\x60\x26\xfb\xa7\x1b\xd3\xe7\xcf\x13\x15\x7c\xc2\xc5\x1a\x8e\x4a\xa6\x84\xaf\x97\x78\xf9\x18\x49\xf3\x43\x35\xd1\x41\xc0\x01\x54\xc4\x19\x76\x21\xf9\x62\x4a\x67\x5b\x5a\xbc\x22\xee\x7d\x5b\xaa\xff\xaa\xe1\xc9\xba\xca\x2c\xc3\x73\xb3\xf3\x3e\x78\xe6\x14\x3c\x39\x5a\x91\xaa\x7f\xac\xa6\x64\xeb\x73\x3a\xfd\x14\xd8\x82\x72\x59\xd9\x9a\x75\x50\xfa\xca\x50\x1e\xf2\xb0\x4e\x33\xc2\x3a\xa5\x1f\x4b\x9e\x82\x82\xef\xdb\x72\x8c\xc0\xab\x09\x40\x5a\x91\x60\x7c\x63\x69\x96\x1b\xc8\x27\x0d\x2d\x4f\x39\xfc\xe6\x12\xb1"
        }
    -- Example 1.4
    , VectorPSS
        { message = "\xbc\x65\x67\x47\xfa\x9e\xaf\xb3\xf0"
        , salt = "\x05\x6f\x00\x98\x5d\xe1\x4d\x8e\xf5\xce\xa9\xe8\x2f\x8c\x27\xbe\xf7\x20\x33\x5e"
        , signature = "\x46\x09\x79\x3b\x23\xe9\xd0\x93\x62\xdc\x21\xbb\x47\xda\x0b\x4f\x3a\x76\x22\x64\x9a\x47\xd4\x64\x01\x9b\x9a\xea\xfe\x53\x35\x9c\x17\x8c\x91\xcd\x58\xba\x6b\xcb\x78\xbe\x03\x46\xa7\xbc\x63\x7f\x4b\x87\x3d\x4b\xab\x38\xee\x66\x1f\x19\x96\x34\xc5\x47\xa1\xad\x84\x42\xe0\x3d\xa0\x15\xb1\x36\xe5\x43\xf7\xab\x07\xc0\xc1\x3e\x42\x25\xb8\xde\x8c\xce\x25\xd4\xf6\xeb\x84\x00\xf8\x1f\x7e\x18\x33\xb7\xee\x6e\x33\x4d\x37\x09\x64\xca\x79\xfd\xb8\x72\xb4\xd7\x52\x23\xb5\xee\xb0\x81\x01\x59\x1f\xb5\x32\xd1\x55\xa6\xde\x87"
        }
    -- Example 1.5
    , VectorPSS
        { message = "\xb4\x55\x81\x54\x7e\x54\x27\x77\x0c\x76\x8e\x8b\x82\xb7\x55\x64\xe0\xea\x4e\x9c\x32\x59\x4d\x6b\xff\x70\x65\x44\xde\x0a\x87\x76\xc7\xa8\x0b\x45\x76\x55\x0e\xee\x1b\x2a\xca\xbc\x7e\x8b\x7d\x3e\xf7\xbb\x5b\x03\xe4\x62\xc1\x10\x47\xea\xdd\x00\x62\x9a\xe5\x75\x48\x0a\xc1\x47\x0f\xe0\x46\xf1\x3a\x2b\xf5\xaf\x17\x92\x1d\xc4\xb0\xaa\x8b\x02\xbe\xe6\x33\x49\x11\x65\x1d\x7f\x85\x25\xd1\x0f\x32\xb5\x1d\x33\xbe\x52\x0d\x3d\xdf\x5a\x70\x99\x55\xa3\xdf\xe7\x82\x83\xb9\xe0\xab\x54\x04\x6d\x15\x0c\x17\x7f\x03\x7f\xdc\xcc\x5b\xe4\xea\x5f\x68\xb5\xe5\xa3\x8c\x9d\x7e\xdc\xcc\xc4\x97\x5f\x45\x5a\x69\x09\xb4"
        , salt = "\x80\xe7\x0f\xf8\x6a\x08\xde\x3e\xc6\x09\x72\xb3\x9b\x4f\xbf\xdc\xea\x67\xae\x8e"
        , signature = "\x1d\x2a\xad\x22\x1c\xa4\xd3\x1d\xdf\x13\x50\x92\x39\x01\x93\x98\xe3\xd1\x4b\x32\xdc\x34\xdc\x5a\xf4\xae\xae\xa3\xc0\x95\xaf\x73\x47\x9c\xf0\xa4\x5e\x56\x29\x63\x5a\x53\xa0\x18\x37\x76\x15\xb1\x6c\xb9\xb1\x3b\x3e\x09\xd6\x71\xeb\x71\xe3\x87\xb8\x54\x5c\x59\x60\xda\x5a\x64\x77\x6e\x76\x8e\x82\xb2\xc9\x35\x83\xbf\x10\x4c\x3f\xdb\x23\x51\x2b\x7b\x4e\x89\xf6\x33\xdd\x00\x63\xa5\x30\xdb\x45\x24\xb0\x1c\x3f\x38\x4c\x09\x31\x0e\x31\x5a\x79\xdc\xd3\xd6\x84\x02\x2a\x7f\x31\xc8\x65\xa6\x64\xe3\x16\x97\x8b\x75\x9f\xad"
        }
    -- Example 1.6
    , VectorPSS
        { message = "\x10\xaa\xe9\xa0\xab\x0b\x59\x5d\x08\x41\x20\x7b\x70\x0d\x48\xd7\x5f\xae\xdd\xe3\xb7\x75\xcd\x6b\x4c\xc8\x8a\xe0\x6e\x46\x94\xec\x74\xba\x18\xf8\x52\x0d\x4f\x5e\xa6\x9c\xbb\xe7\xcc\x2b\xeb\xa4\x3e\xfd\xc1\x02\x15\xac\x4e\xb3\x2d\xc3\x02\xa1\xf5\x3d\xc6\xc4\x35\x22\x67\xe7\x93\x6c\xfe\xbf\x7c\x8d\x67\x03\x57\x84\xa3\x90\x9f\xa8\x59\xc7\xb7\xb5\x9b\x8e\x39\xc5\xc2\x34\x9f\x18\x86\xb7\x05\xa3\x02\x67\xd4\x02\xf7\x48\x6a\xb4\xf5\x8c\xad\x5d\x69\xad\xb1\x7a\xb8\xcd\x0c\xe1\xca\xf5\x02\x5a\xf4\xae\x24\xb1\xfb\x87\x94\xc6\x07\x0c\xc0\x9a\x51\xe2\xf9\x91\x13\x11\xe3\x87\x7d\x00\x44\xc7\x1c\x57\xa9\x93\x39\x50\x08\x80\x6b\x72\x3a\xc3\x83\x73\xd3\x95\x48\x18\x18\x52\x8c\x1e\x70\x53\x73\x92\x82\x05\x35\x29\x51\x0e\x93\x5c\xd0\xfa\x77\xb8\xfa\x53\xcc\x2d\x47\x4b\xd4\xfb\x3c\xc5\xc6\x72\xd6\xff\xdc\x90\xa0\x0f\x98\x48\x71\x2c\x4b\xcf\xe4\x6c\x60\x57\x36\x59\xb1\x1e\x64\x57\xe8\x61\xf0\xf6\x04\xb6\x13\x8d\x14\x4f\x8c\xe4\xe2\xda\x73"
        , salt = "\xa8\xab\x69\xdd\x80\x1f\x00\x74\xc2\xa1\xfc\x60\x64\x98\x36\xc6\x16\xd9\x96\x81"
        , signature = "\x2a\x34\xf6\x12\x5e\x1f\x6b\x0b\xf9\x71\xe8\x4f\xbd\x41\xc6\x32\xbe\x8f\x2c\x2a\xce\x7d\xe8\xb6\x92\x6e\x31\xff\x93\xe9\xaf\x98\x7f\xbc\x06\xe5\x1e\x9b\xe1\x4f\x51\x98\xf9\x1f\x3f\x95\x3b\xd6\x7d\xa6\x0a\x9d\xf5\x97\x64\xc3\xdc\x0f\xe0\x8e\x1c\xbe\xf0\xb7\x5f\x86\x8d\x10\xad\x3f\xba\x74\x9f\xef\x59\xfb\x6d\xac\x46\xa0\xd6\xe5\x04\x36\x93\x31\x58\x6f\x58\xe4\x62\x8f\x39\xaa\x27\x89\x82\x54\x3b\xc0\xee\xb5\x37\xdc\x61\x95\x80\x19\xb3\x94\xfb\x27\x3f\x21\x58\x58\xa0\xa0\x1a\xc4\xd6\x50\xb9\x55\xc6\x7f\x4c\x58"
        }
    ]

-- ==================================
-- Example 2: A 1025-bit RSA Key Pair
-- ==================================

rsaKey2 = PrivateKey
    { private_pub = PublicKey
        { public_n = 0x01d40c1bcf97a68ae7cdbd8a7bf3e34fa19dcca4ef75a47454375f94514d88fed006fb829f8419ff87d6315da68a1ff3a0938e9abb3464011c303ad99199cf0c7c7a8b477dce829e8844f625b115e5e9c4a59cf8f8113b6834336a2fd2689b472cbb5e5cabe674350c59b6c17e176874fb42f8fc3d176a017edc61fd326c4b33c9
        , public_e = 0x010001
        , public_size = 129
        }
    , private_d = 0x027d147e4673057377fd1ea201565772176a7dc38358d376045685a2e787c23c15576bc16b9f444402d6bfc5d98a3e88ea13ef67c353eca0c0ddba9255bd7b8bb50a644afdfd1dd51695b252d22e7318d1b6687a1c10ff75545f3db0fe602d5f2b7f294e3601eab7b9d1cecd767f64692e3e536ca2846cb0c2dd486a39fa75b1
    , private_p = 0x016601e926a0f8c9e26ecab769ea65a5e7c52cc9e080ef519457c644da6891c5a104d3ea7955929a22e7c68a7af9fcad777c3ccc2b9e3d3650bce404399b7e59d1
    , private_q = 0x014eafa1d4d0184da7e31f877d1281ddda625664869e8379e67ad3b75eae74a580e9827abd6eb7a002cb5411f5266797768fb8e95ae40e3e8a01f35ff89e56c079
    , private_dP = 0xe247cce504939b8f0a36090de200938755e2444b29539a7da7a902f6056835c0db7b52559497cfe2c61a8086d0213c472c78851800b171f6401de2e9c2756f31
    , private_dQ = 0xb12fba757855e586e46f64c38a70c68b3f548d93d787b399999d4c8f0bbd2581c21e19ed0018a6d5d3df86424b3abcad40199d31495b61309f27c1bf55d487c1
    , private_qinv = 0x564b1e1fa003bda91e89090425aac05b91da9ee25061e7628d5f51304a84992fdc33762bd378a59f030a334d532bd0dae8f298ea9ed844636ad5fb8cbdc03cad
    }

vectorsKey2 =
    [
    -- Example 2.1
      VectorPSS
        { message = "\xda\xba\x03\x20\x66\x26\x3f\xae\xdb\x65\x98\x48\x11\x52\x78\xa5\x2c\x44\xfa\xa3\xa7\x6f\x37\x51\x5e\xd3\x36\x32\x10\x72\xc4\x0a\x9d\x9b\x53\xbc\x05\x01\x40\x78\xad\xf5\x20\x87\x51\x46\xaa\xe7\x0f\xf0\x60\x22\x6d\xcb\x7b\x1f\x1f\xc2\x7e\x93\x60"
        , salt = "\x57\xbf\x16\x0b\xcb\x02\xbb\x1d\xc7\x28\x0c\xf0\x45\x85\x30\xb7\xd2\x83\x2f\xf7"
        , signature = "\x01\x4c\x5b\xa5\x33\x83\x28\xcc\xc6\xe7\xa9\x0b\xf1\xc0\xab\x3f\xd6\x06\xff\x47\x96\xd3\xc1\x2e\x4b\x63\x9e\xd9\x13\x6a\x5f\xec\x6c\x16\xd8\x88\x4b\xdd\x99\xcf\xdc\x52\x14\x56\xb0\x74\x2b\x73\x68\x68\xcf\x90\xde\x09\x9a\xdb\x8d\x5f\xfd\x1d\xef\xf3\x9b\xa4\x00\x7a\xb7\x46\xce\xfd\xb2\x2d\x7d\xf0\xe2\x25\xf5\x46\x27\xdc\x65\x46\x61\x31\x72\x1b\x90\xaf\x44\x53\x63\xa8\x35\x8b\x9f\x60\x76\x42\xf7\x8f\xab\x0a\xb0\xf4\x3b\x71\x68\xd6\x4b\xae\x70\xd8\x82\x78\x48\xd8\xef\x1e\x42\x1c\x57\x54\xdd\xf4\x2c\x25\x89\xb5\xb3"
        }
    -- Example 2.2
    , VectorPSS
        { message = "\xe4\xf8\x60\x1a\x8a\x6d\xa1\xbe\x34\x44\x7c\x09\x59\xc0\x58\x57\x0c\x36\x68\xcf\xd5\x1d\xd5\xf9\xcc\xd6\xad\x44\x11\xfe\x82\x13\x48\x6d\x78\xa6\xc4\x9f\x93\xef\xc2\xca\x22\x88\xce\xbc\x2b\x9b\x60\xbd\x04\xb1\xe2\x20\xd8\x6e\x3d\x48\x48\xd7\x09\xd0\x32\xd1\xe8\xc6\xa0\x70\xc6\xaf\x9a\x49\x9f\xcf\x95\x35\x4b\x14\xba\x61\x27\xc7\x39\xde\x1b\xb0\xfd\x16\x43\x1e\x46\x93\x8a\xec\x0c\xf8\xad\x9e\xb7\x2e\x83\x2a\x70\x35\xde\x9b\x78\x07\xbd\xc0\xed\x8b\x68\xeb\x0f\x5a\xc2\x21\x6b\xe4\x0c\xe9\x20\xc0\xdb\x0e\xdd\xd3\x86\x0e\xd7\x88\xef\xac\xca\xca\x50\x2d\x8f\x2b\xd6\xd1\xa7\xc1\xf4\x1f\xf4\x6f\x16\x81\xc8\xf1\xf8\x18\xe9\xc4\xf6\xd9\x1a\x0c\x78\x03\xcc\xc6\x3d\x76\xa6\x54\x4d\x84\x3e\x08\x4e\x36\x3b\x8a\xcc\x55\xaa\x53\x17\x33\xed\xb5\xde\xe5\xb5\x19\x6e\x9f\x03\xe8\xb7\x31\xb3\x77\x64\x28\xd9\xe4\x57\xfe\x3f\xbc\xb3\xdb\x72\x74\x44\x2d\x78\x58\x90\xe9\xcb\x08\x54\xb6\x44\x4d\xac\xe7\x91\xd7\x27\x3d\xe1\x88\x97\x19\x33\x8a\x77\xfe"
        , salt = "\x7f\x6d\xd3\x59\xe6\x04\xe6\x08\x70\xe8\x98\xe4\x7b\x19\xbf\x2e\x5a\x7b\x2a\x90"
        , signature = "\x01\x09\x91\x65\x6c\xca\x18\x2b\x7f\x29\xd2\xdb\xc0\x07\xe7\xae\x0f\xec\x15\x8e\xb6\x75\x9c\xb9\xc4\x5c\x5f\xf8\x7c\x76\x35\xdd\x46\xd1\x50\x88\x2f\x4d\xe1\xe9\xae\x65\xe7\xf7\xd9\x01\x8f\x68\x36\x95\x4a\x47\xc0\xa8\x1a\x8a\x6b\x6f\x83\xf2\x94\x4d\x60\x81\xb1\xaa\x7c\x75\x9b\x25\x4b\x2c\x34\xb6\x91\xda\x67\xcc\x02\x26\xe2\x0b\x2f\x18\xb4\x22\x12\x76\x1d\xcd\x4b\x90\x8a\x62\xb3\x71\xb5\x91\x8c\x57\x42\xaf\x4b\x53\x7e\x29\x69\x17\x67\x4f\xb9\x14\x19\x47\x61\x62\x1c\xc1\x9a\x41\xf6\xfb\x95\x3f\xbc\xbb\x64\x9d\xea"
        }
    -- Example 2.3
    , VectorPSS
        { message = "\x52\xa1\xd9\x6c\x8a\xc3\x9e\x41\xe4\x55\x80\x98\x01\xb9\x27\xa5\xb4\x45\xc1\x0d\x90\x2a\x0d\xcd\x38\x50\xd2\x2a\x66\xd2\xbb\x07\x03\xe6\x7d\x58\x67\x11\x45\x95\xaa\xbf\x5a\x7a\xeb\x5a\x8f\x87\x03\x4b\xbb\x30\xe1\x3c\xfd\x48\x17\xa9\xbe\x76\x23\x00\x23\x60\x6d\x02\x86\xa3\xfa\xf8\xa4\xd2\x2b\x72\x8e\xc5\x18\x07\x9f\x9e\x64\x52\x6e\x3a\x0c\xc7\x94\x1a\xa3\x38\xc4\x37\x99\x7c\x68\x0c\xca\xc6\x7c\x66\xbf\xa1"
        , salt = "\xfc\xa8\x62\x06\x8b\xce\x22\x46\x72\x4b\x70\x8a\x05\x19\xda\x17\xe6\x48\x68\x8c"
        , signature = "\x00\x7f\x00\x30\x01\x8f\x53\xcd\xc7\x1f\x23\xd0\x36\x59\xfd\xe5\x4d\x42\x41\xf7\x58\xa7\x50\xb4\x2f\x18\x5f\x87\x57\x85\x20\xc3\x07\x42\xaf\xd8\x43\x59\xb6\xe6\xe8\xd3\xed\x95\x9d\xc6\xfe\x48\x6b\xed\xc8\xe2\xcf\x00\x1f\x63\xa7\xab\xe1\x62\x56\xa1\xb8\x4d\xf0\xd2\x49\xfc\x05\xd3\x19\x4c\xe5\xf0\x91\x27\x42\xdb\xbf\x80\xdd\x17\x4f\x6c\x51\xf6\xba\xd7\xf1\x6c\xf3\x36\x4e\xba\x09\x5a\x06\x26\x7d\xc3\x79\x38\x03\xac\x75\x26\xae\xbe\x0a\x47\x5d\x38\xb8\xc2\x24\x7a\xb5\x1c\x48\x98\xdf\x70\x47\xdc\x6a\xdf\x52\xc6\xc4"
        }
    -- Example 2.4
    , VectorPSS
        { message = "\xa7\x18\x2c\x83\xac\x18\xbe\x65\x70\xa1\x06\xaa\x9d\x5c\x4e\x3d\xbb\xd4\xaf\xae\xb0\xc6\x0c\x4a\x23\xe1\x96\x9d\x79\xff"
        , salt = "\x80\x70\xef\x2d\xe9\x45\xc0\x23\x87\x68\x4b\xa0\xd3\x30\x96\x73\x22\x35\xd4\x40"
        , signature = "\x00\x9c\xd2\xf4\xed\xbe\x23\xe1\x23\x46\xae\x8c\x76\xdd\x9a\xd3\x23\x0a\x62\x07\x61\x41\xf1\x6c\x15\x2b\xa1\x85\x13\xa4\x8e\xf6\xf0\x10\xe0\xe3\x7f\xd3\xdf\x10\xa1\xec\x62\x9a\x0c\xb5\xa3\xb5\xd2\x89\x30\x07\x29\x8c\x30\x93\x6a\x95\x90\x3b\x6b\xa8\x55\x55\xd9\xec\x36\x73\xa0\x61\x08\xfd\x62\xa2\xfd\xa5\x6d\x1c\xe2\xe8\x5c\x4d\xb6\xb2\x4a\x81\xca\x3b\x49\x6c\x36\xd4\xfd\x06\xeb\x7c\x91\x66\xd8\xe9\x48\x77\xc4\x2b\xea\x62\x2b\x3b\xfe\x92\x51\xfd\xc2\x1d\x8d\x53\x71\xba\xda\xd7\x8a\x48\x82\x14\x79\x63\x35\xb4\x0b"
        }
    -- Example 2.5
    , VectorPSS
        { message = "\x86\xa8\x3d\x4a\x72\xee\x93\x2a\x4f\x56\x30\xaf\x65\x79\xa3\x86\xb7\x8f\xe8\x89\x99\xe0\xab\xd2\xd4\x90\x34\xa4\xbf\xc8\x54\xdd\x94\xf1\x09\x4e\x2e\x8c\xd7\xa1\x79\xd1\x95\x88\xe4\xae\xfc\x1b\x1b\xd2\x5e\x95\xe3\xdd\x46\x1f"
        , salt = "\x17\x63\x9a\x4e\x88\xd7\x22\xc4\xfc\xa2\x4d\x07\x9a\x8b\x29\xc3\x24\x33\xb0\xc9"
        , signature = "\x00\xec\x43\x08\x24\x93\x1e\xbd\x3b\xaa\x43\x03\x4d\xae\x98\xba\x64\x6b\x8c\x36\x01\x3d\x16\x71\xc3\xcf\x1c\xf8\x26\x0c\x37\x4b\x19\xf8\xe1\xcc\x8d\x96\x50\x12\x40\x5e\x7e\x9b\xf7\x37\x86\x12\xdf\xcc\x85\xfc\xe1\x2c\xda\x11\xf9\x50\xbd\x0b\xa8\x87\x67\x40\x43\x6c\x1d\x25\x95\xa6\x4a\x1b\x32\xef\xcf\xb7\x4a\x21\xc8\x73\xb3\xcc\x33\xaa\xf4\xe3\xdc\x39\x53\xde\x67\xf0\x67\x4c\x04\x53\xb4\xfd\x9f\x60\x44\x06\xd4\x41\xb8\x16\x09\x8c\xb1\x06\xfe\x34\x72\xbc\x25\x1f\x81\x5f\x59\xdb\x2e\x43\x78\xa3\xad\xdc\x18\x1e\xcf"
        }
    -- Example 2.6
    , VectorPSS
        { message = "\x04\x9f\x91\x54\xd8\x71\xac\x4a\x7c\x7a\xb4\x53\x25\xba\x75\x45\xa1\xed\x08\xf7\x05\x25\xb2\x66\x7c\xf1"
        , salt = "\x37\x81\x0d\xef\x10\x55\xed\x92\x2b\x06\x3d\xf7\x98\xde\x5d\x0a\xab\xf8\x86\xee"
        , signature = "\x00\x47\x5b\x16\x48\xf8\x14\xa8\xdc\x0a\xbd\xc3\x7b\x55\x27\xf5\x43\xb6\x66\xbb\x6e\x39\xd3\x0e\x5b\x49\xd3\xb8\x76\xdc\xcc\x58\xea\xc1\x4e\x32\xa2\xd5\x5c\x26\x16\x01\x44\x56\xad\x2f\x24\x6f\xc8\xe3\xd5\x60\xda\x3d\xdf\x37\x9a\x1c\x0b\xd2\x00\xf1\x02\x21\xdf\x07\x8c\x21\x9a\x15\x1b\xc8\xd4\xec\x9d\x2f\xc2\x56\x44\x67\x81\x10\x14\xef\x15\xd8\xea\x01\xc2\xeb\xbf\xf8\xc2\xc8\xef\xab\x38\x09\x6e\x55\xfc\xbe\x32\x85\xc7\xaa\x55\x88\x51\x25\x4f\xaf\xfa\x92\xc1\xc7\x2b\x78\x75\x86\x63\xef\x45\x82\x84\x31\x39\xd7\xa6"
        }
    ]

-- ==================================
-- Example 3: A 1026-bit RSA Key Pair
-- ==================================

rsaKey3 = PrivateKey
    { private_pub = PublicKey
        { public_n = 0x02f246ef451ed3eebb9a310200cc25859c048e4be798302991112eb68ce6db674e280da21feded1ae74880ca522b18db249385012827c515f0e466a1ffa691d98170574e9d0eadb087586ca48933da3cc953d95bd0ed50de10ddcb6736107d6c831c7f663e833ca4c097e700ce0fb945f88fb85fe8e5a773172565b914a471a443
        , public_e = 0x010001
        , public_size = 129
        }
    , private_d = 0x651451733b56de5ac0a689a4aeb6e6894a69014e076c88dd7a667eab3232bbccd2fc44ba2fa9c31db46f21edd1fdb23c5c128a5da5bab91e7f952b67759c7cff705415ac9fa0907c7ca6178f668fb948d869da4cc3b7356f4008dfd5449d32ee02d9a477eb69fc29266e5d9070512375a50fbbcc27e238ad98425f6ebbf88991
    , private_p = 0x01bd36e18ece4b0fdb2e9c9d548bd1a7d6e2c21c6fdc35074a1d05b1c6c8b3d558ea2639c9a9a421680169317252558bd148ad215aac550e2dcf12a82d0ebfe853
    , private_q = 0x01b1b656ad86d8e19d5dc86292b3a192fdf6e0dd37877bad14822fa00190cab265f90d3f02057b6f54d6ecb14491e5adeacebc48bf0ebd2a2ad26d402e54f61651
    , private_dP = 0x1f2779fd2e3e5e6bae05539518fba0cd0ead1aa4513a7cba18f1cf10e3f68195693d278a0f0ee72f89f9bc760d80e2f9d0261d516501c6ae39f14a476ce2ccf5
    , private_dQ = 0x011a0d36794b04a854aab4b2462d439a5046c91d940b2bc6f75b62956fef35a2a6e63c5309817f307bbff9d59e7e331bd363f6d66849b18346adea169f0ae9aec1
    , private_qinv = 0x0b30f0ecf558752fb3a6ce4ba2b8c675f659eba6c376585a1b39712d038ae3d2b46fcb418ae15d0905da6440e1513a30b9b7d6668fbc5e88e5ab7a175e73ba35
    }

vectorsKey3 =
    [
    -- Example 3.1
      VectorPSS
        { message = "\x59\x4b\x37\x33\x3b\xbb\x2c\x84\x52\x4a\x87\xc1\xa0\x1f\x75\xfc\xec\x0e\x32\x56\xf1\x08\xe3\x8d\xca\x36\xd7\x0d\x00\x57"
        , salt = "\xf3\x1a\xd6\xc8\xcf\x89\xdf\x78\xed\x77\xfe\xac\xbc\xc2\xf8\xb0\xa8\xe4\xcf\xaa"
        , signature = "\x00\x88\xb1\x35\xfb\x17\x94\xb6\xb9\x6c\x4a\x3e\x67\x81\x97\xf8\xca\xc5\x2b\x64\xb2\xfe\x90\x7d\x6f\x27\xde\x76\x11\x24\x96\x4a\x99\xa0\x1a\x88\x27\x40\xec\xfa\xed\x6c\x01\xa4\x74\x64\xbb\x05\x18\x23\x13\xc0\x13\x38\xa8\xcd\x09\x72\x14\xcd\x68\xca\x10\x3b\xd5\x7d\x3b\xc9\xe8\x16\x21\x3e\x61\xd7\x84\xf1\x82\x46\x7a\xbf\x8a\x01\xcf\x25\x3e\x99\xa1\x56\xea\xa8\xe3\xe1\xf9\x0e\x3c\x6e\x4e\x3a\xa2\xd8\x3e\xd0\x34\x5b\x89\xfa\xfc\x9c\x26\x07\x7c\x14\xb6\xac\x51\x45\x4f\xa2\x6e\x44\x6e\x3a\x2f\x15\x3b\x2b\x16\x79\x7f"
        }
    -- Example 3.2
    , VectorPSS
        { message = "\x8b\x76\x95\x28\x88\x4a\x0d\x1f\xfd\x09\x0c\xf1\x02\x99\x3e\x79\x6d\xad\xcf\xbd\xdd\x38\xe4\x4f\xf6\x32\x4c\xa4\x51"
        , salt = "\xfc\xf9\xf0\xe1\xf1\x99\xa3\xd1\xd0\xda\x68\x1c\x5b\x86\x06\xfc\x64\x29\x39\xf7"
        , signature = "\x02\xa5\xf0\xa8\x58\xa0\x86\x4a\x4f\x65\x01\x7a\x7d\x69\x45\x4f\x3f\x97\x3a\x29\x99\x83\x9b\x7b\xbc\x48\xbf\x78\x64\x11\x69\x17\x95\x56\xf5\x95\xfa\x41\xf6\xff\x18\xe2\x86\xc2\x78\x30\x79\xbc\x09\x10\xee\x9c\xc3\x4f\x49\xba\x68\x11\x24\xf9\x23\xdf\xa8\x8f\x42\x61\x41\xa3\x68\xa5\xf5\xa9\x30\xc6\x28\xc2\xc3\xc2\x00\xe1\x8a\x76\x44\x72\x1a\x0c\xbe\xc6\xdd\x3f\x62\x79\xbd\xe3\xe8\xf2\xbe\x5e\x2d\x4e\xe5\x6f\x97\xe7\xce\xaf\x33\x05\x4b\xe7\x04\x2b\xd9\x1a\x63\xbb\x09\xf8\x97\xbd\x41\xe8\x11\x97\xde\xe9\x9b\x11\xaf"
        }
    -- Example 3.3
    , VectorPSS
        { message = "\x1a\xbd\xba\x48\x9c\x5a\xda\x2f\x99\x5e\xd1\x6f\x19\xd5\xa9\x4d\x9e\x6e\xc3\x4a\x8d\x84\xf8\x45\x57\xd2\x6e\x5e\xf9\xb0\x2b\x22\x88\x7e\x3f\x9a\x4b\x69\x0a\xd1\x14\x92\x09\xc2\x0c\x61\x43\x1f\x0c\x01\x7c\x36\xc2\x65\x7b\x35\xd7\xb0\x7d\x3f\x5a\xd8\x70\x85\x07\xa9\xc1\xb8\x31\xdf\x83\x5a\x56\xf8\x31\x07\x18\x14\xea\x5d\x3d\x8d\x8f\x6a\xde\x40\xcb\xa3\x8b\x42\xdb\x7a\x2d\x3d\x7a\x29\xc8\xf0\xa7\x9a\x78\x38\xcf\x58\xa9\x75\x7f\xa2\xfe\x4c\x40\xdf\x9b\xaa\x19\x3b\xfc\x6f\x92\xb1\x23\xad\x57\xb0\x7a\xce\x3e\x6a\xc0\x68\xc9\xf1\x06\xaf\xd9\xee\xb0\x3b\x4f\x37\xc2\x5d\xbf\xbc\xfb\x30\x71\xf6\xf9\x77\x17\x66\xd0\x72\xf3\xbb\x07\x0a\xf6\x60\x55\x32\x97\x3a\xe2\x50\x51"
        , salt = "\x98\x6e\x7c\x43\xdb\xb6\x71\xbd\x41\xb9\xa7\xf4\xb6\xaf\xc8\x0e\x80\x5f\x24\x23"
        , signature = "\x02\x44\xbc\xd1\xc8\xc1\x69\x55\x73\x6c\x80\x3b\xe4\x01\x27\x2e\x18\xcb\x99\x08\x11\xb1\x4f\x72\xdb\x96\x41\x24\xd5\xfa\x76\x06\x49\xcb\xb5\x7a\xfb\x87\x55\xdb\xb6\x2b\xf5\x1f\x46\x6c\xf2\x3a\x0a\x16\x07\x57\x6e\x98\x3d\x77\x8f\xce\xff\xa9\x2d\xf7\x54\x8a\xea\x8e\xa4\xec\xad\x2c\x29\xdd\x9f\x95\xbc\x07\xfe\x91\xec\xf8\xbe\xe2\x55\xbf\xe8\x76\x2f\xd7\x69\x0a\xa9\xbf\xa4\xfa\x08\x49\xef\x72\x8c\x2c\x42\xc4\x53\x23\x64\x52\x2d\xf2\xab\x7f\x9f\x8a\x03\xb6\x3f\x7a\x49\x91\x75\x82\x86\x68\xf5\xef\x5a\x29\xe3\x80\x2c"
        }
    -- Example 3.4
    , VectorPSS
        { message = "\x8f\xb4\x31\xf5\xee\x79\x2b\x6c\x2a\xc7\xdb\x53\xcc\x42\x86\x55\xae\xb3\x2d\x03\xf4\xe8\x89\xc5\xc2\x5d\xe6\x83\xc4\x61\xb5\x3a\xcf\x89\xf9\xf8\xd3\xaa\xbd\xf6\xb9\xf0\xc2\xa1\xde\x12\xe1\x5b\x49\xed\xb3\x91\x9a\x65\x2f\xe9\x49\x1c\x25\xa7\xfc\xe1\xf7\x22\xc2\x54\x36\x08\xb6\x9d\xc3\x75\xec"
        , salt = "\xf8\x31\x2d\x9c\x8e\xea\x13\xec\x0a\x4c\x7b\x98\x12\x0c\x87\x50\x90\x87\xc4\x78"
        , signature = "\x01\x96\xf1\x2a\x00\x5b\x98\x12\x9c\x8d\xf1\x3c\x4c\xb1\x6f\x8a\xa8\x87\xd3\xc4\x0d\x96\xdf\x3a\x88\xe7\x53\x2e\xf3\x9c\xd9\x92\xf2\x73\xab\xc3\x70\xbc\x1b\xe6\xf0\x97\xcf\xeb\xbf\x01\x18\xfd\x9e\xf4\xb9\x27\x15\x5f\x3d\xf2\x2b\x90\x4d\x90\x70\x2d\x1f\x7b\xa7\xa5\x2b\xed\x8b\x89\x42\xf4\x12\xcd\x7b\xd6\x76\xc9\xd1\x8e\x17\x03\x91\xdc\xd3\x45\xc0\x6a\x73\x09\x64\xb3\xf3\x0b\xcc\xe0\xbb\x20\xba\x10\x6f\x9a\xb0\xee\xb3\x9c\xf8\xa6\x60\x7f\x75\xc0\x34\x7f\x0a\xf7\x9f\x16\xaf\xa0\x81\xd2\xc9\x2d\x1e\xe6\xf8\x36\xb8"
        }
    -- Example 3.5
    , VectorPSS
        { message = "\xfe\xf4\x16\x1d\xfa\xaf\x9c\x52\x95\x05\x1d\xfc\x1f\xf3\x81\x0c\x8c\x9e\xc2\xe8\x66\xf7\x07\x54\x22\xc8\xec\x42\x16\xa9\xc4\xff\x49\x42\x7d\x48\x3c\xae\x10\xc8\x53\x4a\x41\xb2\xfd\x15\xfe\xe0\x69\x60\xec\x6f\xb3\xf7\xa7\xe9\x4a\x2f\x8a\x2e\x3e\x43\xdc\x4a\x40\x57\x6c\x30\x97\xac\x95\x3b\x1d\xe8\x6f\x0b\x4e\xd3\x6d\x64\x4f\x23\xae\x14\x42\x55\x29\x62\x24\x64\xca\x0c\xbf\x0b\x17\x41\x34\x72\x38\x15\x7f\xab\x59\xe4\xde\x55\x24\x09\x6d\x62\xba\xec\x63\xac\x64"
        , salt = "\x50\x32\x7e\xfe\xc6\x29\x2f\x98\x01\x9f\xc6\x7a\x2a\x66\x38\x56\x3e\x9b\x6e\x2d"
        , signature = "\x02\x1e\xca\x3a\xb4\x89\x22\x64\xec\x22\x41\x1a\x75\x2d\x92\x22\x10\x76\xd4\xe0\x1c\x0e\x6f\x0d\xde\x9a\xfd\x26\xba\x5a\xcf\x6d\x73\x9e\xf9\x87\x54\x5d\x16\x68\x3e\x56\x74\xc9\xe7\x0f\x1d\xe6\x49\xd7\xe6\x1d\x48\xd0\xca\xeb\x4f\xb4\xd8\xb2\x4f\xba\x84\xa6\xe3\x10\x8f\xee\x7d\x07\x05\x97\x32\x66\xac\x52\x4b\x4a\xd2\x80\xf7\xae\x17\xdc\x59\xd9\x6d\x33\x51\x58\x6b\x5a\x3b\xdb\x89\x5d\x1e\x1f\x78\x20\xac\x61\x35\xd8\x75\x34\x80\x99\x83\x82\xba\x32\xb7\x34\x95\x59\x60\x8c\x38\x74\x52\x90\xa8\x5e\xf4\xe9\xf9\xbd\x83"
        }
    -- Example 3.6
    , VectorPSS
        { message = "\xef\xd2\x37\xbb\x09\x8a\x44\x3a\xee\xb2\xbf\x6c\x3f\x8c\x81\xb8\xc0\x1b\x7f\xcb\x3f\xeb"
        , salt = "\xb0\xde\x3f\xc2\x5b\x65\xf5\xaf\x96\xb1\xd5\xcc\x3b\x27\xd0\xc6\x05\x30\x87\xb3"
        , signature = "\x01\x2f\xaf\xec\x86\x2f\x56\xe9\xe9\x2f\x60\xab\x0c\x77\x82\x4f\x42\x99\xa0\xca\x73\x4e\xd2\x6e\x06\x44\xd5\xd2\x22\xc7\xf0\xbd\xe0\x39\x64\xf8\xe7\x0a\x5c\xb6\x5e\xd4\x4e\x44\xd5\x6a\xe0\xed\xf1\xff\x86\xca\x03\x2c\xc5\xdd\x44\x04\xdb\xb7\x6a\xb8\x54\x58\x6c\x44\xee\xd8\x33\x6d\x08\xd4\x57\xce\x6c\x03\x69\x3b\x45\xc0\xf1\xef\xef\x93\x62\x4b\x95\xb8\xec\x16\x9c\x61\x6d\x20\xe5\x53\x8e\xbc\x0b\x67\x37\xa6\xf8\x2b\x4b\xc0\x57\x09\x24\xfc\x6b\x35\x75\x9a\x33\x48\x42\x62\x79\xf8\xb3\xd7\x74\x4e\x2d\x22\x24\x26\xce"
        }
    ]

-- ==================================
-- Example 8: A 1031-bit RSA Key Pair
-- ==================================

rsaKey8 = PrivateKey
    { private_pub = PublicKey
        { public_n = 0x495370a1fb18543c16d3631e3163255df62be6eee890d5f25509e4f778a8ea6fbbbcdf85dff64e0d972003ab3681fbba6dd41fd541829b2e582de9f2a4a4e0a2d0900bef4753db3cee0ee06c7dfae8b1d53b5953218f9cceea695b08668edeaadced9463b1d790d5ebf27e9115b46cad4d9a2b8efab0561b0810344739ada0733f
        , public_e = 0x010001
        , public_size = 129
        }
    , private_d = 0x6c66ffe98980c38fcdeab5159898836165f4b4b817c4f6a8d486ee4ea9130fe9b9092bd136d184f95f504a607eac565846d2fdd6597a8967c7396ef95a6eeebb4578a643966dca4d8ee3de842de63279c618159c1ab54a89437b6a6120e4930afb52a4ba6ced8a4947ac64b30a3497cbe701c2d6266d517219ad0ec6d347dbe9
    , private_p = 0x08dad7f11363faa623d5d6d5e8a319328d82190d7127d2846c439b0ab72619b0a43a95320e4ec34fc3a9cea876422305bd76c5ba7be9e2f410c8060645a1d29edb
    , private_q = 0x0847e732376fc7900f898ea82eb2b0fc418565fdae62f7d9ec4ce2217b97990dd272db157f99f63c0dcbb9fbacdbd4c4dadb6df67756358ca4174825b48f49706d
    , private_dP = 0x05c2a83c124b3621a2aa57ea2c3efe035eff4560f33ddebb7adab81fce69a0c8c2edc16520dda83d59a23be867963ac65f2cc710bbcfb96ee103deb771d105fd85
    , private_dQ = 0x04cae8aa0d9faa165c87b682ec140b8ed3b50b24594b7a3b2c220b3669bb819f984f55310a1ae7823651d4a02e99447972595139363434e5e30a7e7d241551e1b9
    , private_qinv = 0x07d3e47bf686600b11ac283ce88dbb3f6051e8efd04680e44c171ef531b80b2b7c39fc766320e2cf15d8d99820e96ff30dc69691839c4b40d7b06e45307dc91f3f
    }

vectorsKey8 =
    [
    -- Example 8.1
      VectorPSS
        { message = "\x81\x33\x2f\x4b\xe6\x29\x48\x41\x5e\xa1\xd8\x99\x79\x2e\xea\xcf\x6c\x6e\x1d\xb1\xda\x8b\xe1\x3b\x5c\xea\x41\xdb\x2f\xed\x46\x70\x92\xe1\xff\x39\x89\x14\xc7\x14\x25\x97\x75\xf5\x95\xf8\x54\x7f\x73\x56\x92\xa5\x75\xe6\x92\x3a\xf7\x8f\x22\xc6\x99\x7d\xdb\x90\xfb\x6f\x72\xd7\xbb\x0d\xd5\x74\x4a\x31\xde\xcd\x3d\xc3\x68\x58\x49\x83\x6e\xd3\x4a\xec\x59\x63\x04\xad\x11\x84\x3c\x4f\x88\x48\x9f\x20\x97\x35\xf5\xfb\x7f\xda\xf7\xce\xc8\xad\xdc\x58\x18\x16\x8f\x88\x0a\xcb\xf4\x90\xd5\x10\x05\xb7\xa8\xe8\x4e\x43\xe5\x42\x87\x97\x75\x71\xdd\x99\xee\xa4\xb1\x61\xeb\x2d\xf1\xf5\x10\x8f\x12\xa4\x14\x2a\x83\x32\x2e\xdb\x05\xa7\x54\x87\xa3\x43\x5c\x9a\x78\xce\x53\xed\x93\xbc\x55\x08\x57\xd7\xa9\xfb"
        , salt = "\x1d\x65\x49\x1d\x79\xc8\x64\xb3\x73\x00\x9b\xe6\xf6\xf2\x46\x7b\xac\x4c\x78\xfa"
        , signature = "\x02\x62\xac\x25\x4b\xfa\x77\xf3\xc1\xac\xa2\x2c\x51\x79\xf8\xf0\x40\x42\x2b\x3c\x5b\xaf\xd4\x0a\x8f\x21\xcf\x0f\xa5\xa6\x67\xcc\xd5\x99\x3d\x42\xdb\xaf\xb4\x09\xc5\x20\xe2\x5f\xce\x2b\x1e\xe1\xe7\x16\x57\x7f\x1e\xfa\x17\xf3\xda\x28\x05\x2f\x40\xf0\x41\x9b\x23\x10\x6d\x78\x45\xaa\xf0\x11\x25\xb6\x98\xe7\xa4\xdf\xe9\x2d\x39\x67\xbb\x00\xc4\xd0\xd3\x5b\xa3\x55\x2a\xb9\xa8\xb3\xee\xf0\x7c\x7f\xec\xdb\xc5\x42\x4a\xc4\xdb\x1e\x20\xcb\x37\xd0\xb2\x74\x47\x69\x94\x0e\xa9\x07\xe1\x7f\xbb\xca\x67\x3b\x20\x52\x23\x80\xc5"
        }
    -- Example 8.2
    , VectorPSS
        { message = "\xe2\xf9\x6e\xaf\x0e\x05\xe7\xba\x32\x6e\xcc\xa0\xba\x7f\xd2\xf7\xc0\x23\x56\xf3\xce\xde\x9d\x0f\xaa\xbf\x4f\xcc\x8e\x60\xa9\x73\xe5\x59\x5f\xd9\xea\x08"
        , salt = "\x43\x5c\x09\x8a\xa9\x90\x9e\xb2\x37\x7f\x12\x48\xb0\x91\xb6\x89\x87\xff\x18\x38"
        , signature = "\x27\x07\xb9\xad\x51\x15\xc5\x8c\x94\xe9\x32\xe8\xec\x0a\x28\x0f\x56\x33\x9e\x44\xa1\xb5\x8d\x4d\xdc\xff\x2f\x31\x2e\x5f\x34\xdc\xfe\x39\xe8\x9c\x6a\x94\xdc\xee\x86\xdb\xbd\xae\x5b\x79\xba\x4e\x08\x19\xa9\xe7\xbf\xd9\xd9\x82\xe7\xee\x6c\x86\xee\x68\x39\x6e\x8b\x3a\x14\xc9\xc8\xf3\x4b\x17\x8e\xb7\x41\xf9\xd3\xf1\x21\x10\x9b\xf5\xc8\x17\x2f\xad\xa2\xe7\x68\xf9\xea\x14\x33\x03\x2c\x00\x4a\x8a\xa0\x7e\xb9\x90\x00\x0a\x48\xdc\x94\xc8\xba\xc8\xaa\xbe\x2b\x09\xb1\xaa\x46\xc0\xa2\xaa\x0e\x12\xf6\x3f\xbb\xa7\x75\xba\x7e"
        }
    -- Example 8.3
    , VectorPSS
        { message = "\xe3\x5c\x6e\xd9\x8f\x64\xa6\xd5\xa6\x48\xfc\xab\x8a\xdb\x16\x33\x1d\xb3\x2e\x5d\x15\xc7\x4a\x40\xed\xf9\x4c\x3d\xc4\xa4\xde\x79\x2d\x19\x08\x89\xf2\x0f\x1e\x24\xed\x12\x05\x4a\x6b\x28\x79\x8f\xcb\x42\xd1\xc5\x48\x76\x9b\x73\x4c\x96\x37\x31\x42\x09\x2a\xed\x27\x76\x03\xf4\x73\x8d\xf4\xdc\x14\x46\x58\x6d\x0e\xc6\x4d\xa4\xfb\x60\x53\x6d\xb2\xae\x17\xfc\x7e\x3c\x04\xbb\xfb\xbb\xd9\x07\xbf\x11\x7c\x08\x63\x6f\xa1\x6f\x95\xf5\x1a\x62\x16\x93\x4d\x3e\x34\xf8\x50\x30\xf1\x7b\xbb\xc5\xba\x69\x14\x40\x58\xaf\xf0\x81\xe0\xb1\x9c\xf0\x3c\x17\x19\x5c\x5e\x88\x8b\xa5\x8f\x6f\xe0\xa0\x2e\x5c\x3b\xda\x97\x19\xa7"
        , salt = "\xc6\xeb\xbe\x76\xdf\x0c\x4a\xea\x32\xc4\x74\x17\x5b\x2f\x13\x68\x62\xd0\x45\x29"
        , signature = "\x2a\xd2\x05\x09\xd7\x8c\xf2\x6d\x1b\x6c\x40\x61\x46\x08\x6e\x4b\x0c\x91\xa9\x1c\x2b\xd1\x64\xc8\x7b\x96\x6b\x8f\xaa\x42\xaa\x0c\xa4\x46\x02\x23\x23\xba\x4b\x1a\x1b\x89\x70\x6d\x7f\x4c\x3b\xe5\x7d\x7b\x69\x70\x2d\x16\x8a\xb5\x95\x5e\xe2\x90\x35\x6b\x8c\x4a\x29\xed\x46\x7d\x54\x7e\xc2\x3c\xba\xdf\x28\x6c\xcb\x58\x63\xc6\x67\x9d\xa4\x67\xfc\x93\x24\xa1\x51\xc7\xec\x55\xaa\xc6\xdb\x40\x84\xf8\x27\x26\x82\x5c\xfe\x1a\xa4\x21\xbc\x64\x04\x9f\xb4\x2f\x23\x14\x8f\x9c\x25\xb2\xdc\x30\x04\x37\xc3\x8d\x42\x8a\xa7\x5f\x96"
        }
    -- Example 8.4
    , VectorPSS
        { message = "\xdb\xc5\xf7\x50\xa7\xa1\x4b\xe2\xb9\x3e\x83\x8d\x18\xd1\x4a\x86\x95\xe5\x2e\x8a\xdd\x9c\x0a\xc7\x33\xb8\xf5\x6d\x27\x47\xe5\x29\xa0\xcc\xa5\x32\xdd\x49\xb9\x02\xae\xfe\xd5\x14\x44\x7f\x9e\x81\xd1\x61\x95\xc2\x85\x38\x68\xcb\x9b\x30\xf7\xd0\xd4\x95\xc6\x9d\x01\xb5\xc5\xd5\x0b\x27\x04\x5d\xb3\x86\x6c\x23\x24\xa4\x4a\x11\x0b\x17\x17\x74\x6d\xe4\x57\xd1\xc8\xc4\x5c\x3c\xd2\xa9\x29\x70\xc3\xd5\x96\x32\x05\x5d\x4c\x98\xa4\x1d\x6e\x99\xe2\xa3\xdd\xd5\xf7\xf9\x97\x9a\xb3\xcd\x18\xf3\x75\x05\xd2\x51\x41\xde\x2a\x1b\xff\x17\xb3\xa7\xdc\xe9\x41\x9e\xcc\x38\x5c\xf1\x1d\x72\x84\x0f\x19\x95\x3f\xd0\x50\x92\x51\xf6\xca\xfd\xe2\x89\x3d\x0e\x75\xc7\x81\xba\x7a\x50\x12\xca\x40\x1a\x4f\xa9\x9e\x04\xb3\xc3\x24\x9f\x92\x6d\x5a\xfe\x82\xcc\x87\xda\xb2\x2c\x3c\x1b\x10\x5d\xe4\x8e\x34\xac\xe9\xc9\x12\x4e\x59\x59\x7a\xc7\xeb\xf8"
        , salt = "\x02\x1f\xdc\xc6\xeb\xb5\xe1\x9b\x1c\xb1\x6e\x9c\x67\xf2\x76\x81\x65\x7f\xe2\x0a"
        , signature = "\x1e\x24\xe6\xe5\x86\x28\xe5\x17\x50\x44\xa9\xeb\x6d\x83\x7d\x48\xaf\x12\x60\xb0\x52\x0e\x87\x32\x7d\xe7\x89\x7e\xe4\xd5\xb9\xf0\xdf\x0b\xe3\xe0\x9e\xd4\xde\xa8\xc1\x45\x4f\xf3\x42\x3b\xb0\x8e\x17\x93\x24\x5a\x9d\xf8\xbf\x6a\xb3\x96\x8c\x8e\xdd\xc3\xb5\x32\x85\x71\xc7\x7f\x09\x1c\xc5\x78\x57\x69\x12\xdf\xeb\xd1\x64\xb9\xde\x54\x54\xfe\x0b\xe1\xc1\xf6\x38\x5b\x32\x83\x60\xce\x67\xec\x7a\x05\xf6\xe3\x0e\xb4\x5c\x17\xc4\x8a\xc7\x00\x41\xd2\xca\xb6\x7f\x0a\x2a\xe7\xaa\xfd\xcc\x8d\x24\x5e\xa3\x44\x2a\x63\x00\xcc\xc7"
        }
    -- Example 8.5
    , VectorPSS
        { message = "\x04\xdc\x25\x1b\xe7\x2e\x88\xe5\x72\x34\x85\xb6\x38\x3a\x63\x7e\x2f\xef\xe0\x76\x60\xc5\x19\xa5\x60\xb8\xbc\x18\xbd\xed\xb8\x6e\xae\x23\x64\xea\x53\xba\x9d\xca\x6e\xb3\xd2\xe7\xd6\xb8\x06\xaf\x42\xb3\xe8\x7f\x29\x1b\x4a\x88\x81\xd5\xbf\x57\x2c\xc9\xa8\x5e\x19\xc8\x6a\xcb\x28\xf0\x98\xf9\xda\x03\x83\xc5\x66\xd3\xc0\xf5\x8c\xfd\x8f\x39\x5d\xcf\x60\x2e\x5c\xd4\x0e\x8c\x71\x83\xf7\x14\x99\x6e\x22\x97\xef"
        , salt = "\xc5\x58\xd7\x16\x7c\xbb\x45\x08\xad\xa0\x42\x97\x1e\x71\xb1\x37\x7e\xea\x42\x69"
        , signature = "\x33\x34\x1b\xa3\x57\x6a\x13\x0a\x50\xe2\xa5\xcf\x86\x79\x22\x43\x88\xd5\x69\x3f\x5a\xcc\xc2\x35\xac\x95\xad\xd6\x8e\x5e\xb1\xee\xc3\x16\x66\xd0\xca\x7a\x1c\xda\x6f\x70\xa1\xaa\x76\x2c\x05\x75\x2a\x51\x95\x0c\xdb\x8a\xf3\xc5\x37\x9f\x18\xcf\xe6\xb5\xbc\x55\xa4\x64\x82\x26\xa1\x5e\x91\x2e\xf1\x9a\xd7\x7a\xde\xea\x91\x1d\x67\xcf\xef\xd6\x9b\xa4\x3f\xa4\x11\x91\x35\xff\x64\x21\x17\xba\x98\x5a\x7e\x01\x00\x32\x5e\x95\x19\xf1\xca\x6a\x92\x16\xbd\xa0\x55\xb5\x78\x50\x15\x29\x11\x25\xe9\x0d\xcd\x07\xa2\xca\x96\x73\xee"
        }
    -- Example 8.6
    , VectorPSS
        { message = "\x0e\xa3\x7d\xf9\xa6\xfe\xa4\xa8\xb6\x10\x37\x3c\x24\xcf\x39\x0c\x20\xfa\x6e\x21\x35\xc4\x00\xc8\xa3\x4f\x5c\x18\x3a\x7e\x8e\xa4\xc9\xae\x09\x0e\xd3\x17\x59\xf4\x2d\xc7\x77\x19\xcc\xa4\x00\xec\xdc\xc5\x17\xac\xfc\x7a\xc6\x90\x26\x75\xb2\xef\x30\xc5\x09\x66\x5f\x33\x21\x48\x2f\xc6\x9a\x9f\xb5\x70\xd1\x5e\x01\xc8\x45\xd0\xd8\xe5\x0d\x2a\x24\xcb\xf1\xcf\x0e\x71\x49\x75\xa5\xdb\x7b\x18\xd9\xe9\xe9\xcb\x91\xb5\xcb\x16\x86\x90\x60\xed\x18\xb7\xb5\x62\x45\x50\x3f\x0c\xaf\x90\x35\x2b\x8d\xe8\x1c\xb5\xa1\xd9\xc6\x33\x60\x92\xf0\xcd"
        , salt = "\x76\xfd\x4e\x64\xfd\xc9\x8e\xb9\x27\xa0\x40\x3e\x35\xa0\x84\xe7\x6b\xa9\xf9\x2a"
        , signature = "\x1e\xd1\xd8\x48\xfb\x1e\xdb\x44\x12\x9b\xd9\xb3\x54\x79\x5a\xf9\x7a\x06\x9a\x7a\x00\xd0\x15\x10\x48\x59\x3e\x0c\x72\xc3\x51\x7f\xf9\xff\x2a\x41\xd0\xcb\x5a\x0a\xc8\x60\xd7\x36\xa1\x99\x70\x4f\x7c\xb6\xa5\x39\x86\xa8\x8b\xbd\x8a\xbc\xc0\x07\x6a\x2c\xe8\x47\x88\x00\x31\x52\x5d\x44\x9d\xa2\xac\x78\x35\x63\x74\xc5\x36\xe3\x43\xfa\xa7\xcb\xa4\x2a\x5a\xaa\x65\x06\x08\x77\x91\xc0\x6a\x8e\x98\x93\x35\xae\xd1\x9b\xfa\xb2\xd5\xe6\x7e\x27\xfb\x0c\x28\x75\xaf\x89\x6c\x21\xb6\xe8\xe7\x30\x9d\x04\xe4\xf6\x72\x7e\x69\x46\x3e"
        }
    ]

doSignTest key i vector = testCase (show i) (Right (signature vector) @=? actual)
    where actual = PSS.signWithSalt (salt vector) Nothing PSS.defaultPSSParamsSHA1 key (message vector)

doVerifyTest key i vector = testCase (show i) (True @=? actual)
    where actual = PSS.verify PSS.defaultPSSParamsSHA1 (private_pub key) (message vector) (signature vector)

pssTests = testGroup "RSA-PSS"
    [ testGroup "signature internal"
        [ doSignTest rsaKeyInt katZero vectorInt ]
    , testGroup "verify internal"
        [ doVerifyTest rsaKeyInt katZero vectorInt ]
    , testGroup "signature key 1024" $ zipWith (doSignTest rsaKey1) [katZero..] vectorsKey1
    , testGroup "verify key 1024" $ zipWith (doVerifyTest rsaKey1) [katZero..] vectorsKey1
    , testGroup "signature key 1025" $ zipWith (doSignTest rsaKey2) [katZero..] vectorsKey2
    , testGroup "verify key 1025" $ zipWith (doVerifyTest rsaKey2) [katZero..] vectorsKey2
    , testGroup "signature key 1026" $ zipWith (doSignTest rsaKey3) [katZero..] vectorsKey3
    , testGroup "verify key 1026" $ zipWith (doVerifyTest rsaKey3) [katZero..] vectorsKey3
    , testGroup "signature key 1031" $ zipWith (doSignTest rsaKey8) [katZero..] vectorsKey8
    , testGroup "verify key 1031" $ zipWith (doVerifyTest rsaKey8) [katZero..] vectorsKey8
    ]
