/*
 * Copyright 2010 Jörg Ehrichs <joerg.ehichs@gmx.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tabletdaemon.h"
#include "devicehandler.h"
#include "wacomadaptor.h"
#include "wacomdeviceadaptor.h"

// KDE includes
#include <KDE/KPluginFactory>
#include <KDE/KAboutData>
#include <KDE/KNotification>
#include <KDE/KLocale>
#include <KDE/KLocalizedString>
#include <KDE/KIconLoader>
#include <KDE/KSharedConfig>

//Solid includes
#include <Solid/Device>
#include <Solid/DeviceNotifier>
#include <Solid/DeviceInterface>
#include <Solid/GenericInterface>

using namespace Wacom;

K_PLUGIN_FACTORY(WacomTabletFactory, registerPlugin<TabletDaemon>();)
K_EXPORT_PLUGIN(WacomTabletFactory("wacomtabletdaemon"))

namespace Wacom
{
/**
  * Private class of the TabletDaemon for the d-pointer
  *
  */
class TabletDaemonPrivate
{
public:
    DeviceHandler       *deviceHandler;  /**< Pointer to the tablet device */
    KSharedConfig::Ptr  profilesConfig;  /**< Shared pointer to the configuration file that holds all tablet profiles */
    KComponentData      applicationData; /**< Basic application data */
    KIconLoader         *iconLoader;     /**< Simple loader for the notification icon */
    QString             udi;             /**< current connected tablet device udi */
    QString             curProfile;      /**< currently active profile */
    bool                initPhase;       /**< used to suppress the tablet add notification on kded loading. */
};
}

TabletDaemon::TabletDaemon(QObject *parent, const QVariantList &args)
        : KDEDModule(parent), d_ptr(new TabletDaemonPrivate)
{
    Q_UNUSED(args);
    Q_D(TabletDaemon);

    KGlobal::locale()->insertCatalog("wacomtablet");

    KAboutData about("wacomtablet", "wacomtablet", ki18n("Graphic Tablet Configuration daemon"), "0.1.2.1",
                     ki18n("A wacom tablet control daemon"),
                     KAboutData::License_GPL,
                     ki18n("(c) 2010 Jörg Ehrichs"),
                     KLocalizedString(),
                     "http://www.etricceline.de");

    about.addAuthor(ki18n("Jörg Ehrichs"), ki18n("Maintainer") , "joerg.ehrichs@gmx.de");

    d->applicationData = KComponentData(about);
    d->iconLoader = new KIconLoader(d->applicationData);
    d->profilesConfig = KSharedConfig::openConfig("tabletprofilesrc", KConfig::SimpleConfig);
    d->deviceHandler = new DeviceHandler();

    //DBus connection
    new WacomAdaptor(this);
    new WacomDeviceAdaptor(d->deviceHandler);
    QDBusConnection::sessionBus().registerObject("/Tablet", this);
    QDBusConnection::sessionBus().registerObject("/Device", d->deviceHandler);
    QDBusConnection::sessionBus().registerService("org.kde.Wacom");

    //initial search for devices with solid
    bool foundDevice = false;
    d->initPhase = true;

    foreach(const Solid::Device &device, Solid::Device::allDevices()) {
        deviceAdded(device.udi());
        if (d->deviceHandler->isDeviceAvailable()) {
            foundDevice = true;
            break;
        }
    }

    // if no device via solid could be found try again with normal search over all devices (serial and such)
    if (!foundDevice) {
        d->deviceHandler->reloadDeviceInformation();
    }

    // on deviceAdded, check to see if the device was added
    connect(Solid::DeviceNotifier::instance(), SIGNAL(deviceAdded(const QString&)), SLOT(deviceAdded(const QString &)));
    // likewise if removed
    connect(Solid::DeviceNotifier::instance(), SIGNAL(deviceRemoved(const QString&)), SLOT(deviceRemoved(const QString &)));

    d->initPhase = false;
}

TabletDaemon::~TabletDaemon()
{
    delete this->d_ptr->deviceHandler;
    delete this->d_ptr->iconLoader;
    delete this->d_ptr;
}

void TabletDaemon::deviceAdded(const QString & udi)
{
    Q_D(TabletDaemon);

    // if we already have a device ... skip this step
    if (d->deviceHandler->isDeviceAvailable()) {
        return;
    }

    Solid::Device dev(udi);
    //@TODO add tablet interface to solid
    const Solid::GenericInterface *unknown = dev.as<Solid::GenericInterface>();
    if( !unknown )
	return;
    QStringList category = unknown->property("input.x11_options.Type").toStringList();
    QString     driver   = unknown->property("input.x11_driver").toString();
    if (category.contains("pad") || category.contains("stylus") || category.contains("eraser") || category.contains("cursor") || driver.contains("wacom")) {

        // remove old info and try to detect the tablet
        d->deviceHandler->reloadDeviceInformation();

        // if we found something notify about it and set the default profile to it
        if (d->deviceHandler->isDeviceAvailable()) {
            // maybe not the best solution. but it does work to surpress the notification dialog on kde startup
            if (!d->initPhase) {
                KNotification *notification = new KNotification("tabletAdded");
                notification->setTitle(i18n("Tablet added"));
                notification->setText(i18n("New %1 tablet added", d->deviceHandler->deviceName()));
                notification->setPixmap(d->iconLoader->loadIcon("input-tablet", KIconLoader::Panel));
                notification->setComponentData(d->applicationData);
                notification->sendEvent();
            }

            d->udi = udi;

            emit tabletAdded();

            //get last used profilename
            KSharedConfigPtr config = KSharedConfig::openConfig("wacomtablet-kderc");
            KConfigGroup generalGroup( config, "General" );

            QString profileName = generalGroup.readEntry( "lastprofile", QString() );

            if(profileName.isEmpty()) {
                setProfile("default");
            } else {
                setProfile(profileName);
            }
        }
    }
}

void TabletDaemon::deviceRemoved(const QString & udi)
{
    Q_D(TabletDaemon);
    if (d->deviceHandler->isDeviceAvailable()) {
        if (d->udi == udi) {
            KNotification *notification = new KNotification("tabletRemoved");
            notification->setTitle(i18n("Tablet removed"));
            notification->setText(i18n("Tablet %1 removed", d->deviceHandler->deviceName()));
            notification->setComponentData(d->applicationData);
            notification->sendEvent();
            d->deviceHandler->reloadDeviceInformation();

            emit tabletRemoved();
        }
    }
}

bool TabletDaemon::tabletAvailable() const
{
    Q_D(const TabletDaemon);
    return d->deviceHandler->isDeviceAvailable();
}

void TabletDaemon::setProfile(const QString& profile)
{
    Q_D(TabletDaemon);
    d->profilesConfig->reparseConfiguration();
    KConfigGroup deviceGroup = KConfigGroup(d->profilesConfig, d->deviceHandler->deviceName());
    KConfigGroup profileGroup = KConfigGroup(&deviceGroup, profile);

    if (profileGroup.groupList().isEmpty()) {
        notifyError(i18n("Profile <b>%1</b> does not exist", profile));
    } else {
        d->curProfile = profile;
        d->deviceHandler->applyProfile(&profileGroup);

        emit profileChanged(profile);

        //write as last used profile into the config file
        KSharedConfigPtr config = KSharedConfig::openConfig("wacomtablet-kderc");
        KConfigGroup generalGroup( config, "General" );

        generalGroup.writeEntry( "lastprofile", profile );
    }
}

QString TabletDaemon::profile() const
{
    Q_D(const TabletDaemon);
    return d->curProfile;
}

void TabletDaemon::notifyError(const QString &message) const
{
    Q_D(const TabletDaemon);
    KNotification *notification = new KNotification("tabletError");
    notification->setTitle(i18n("Graphic Tablet error"));
    notification->setText(message);
    notification->setComponentData(d->applicationData);
    notification->sendEvent();
}
