/** BEGIN COPYRIGHT BLOCK
 * Copyright (C) 2001 Sun Microsystems, Inc.  Used by permission.
 * Copyright (C) 2005 Red Hat, Inc.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * END COPYRIGHT BLOCK **/
/* 
 * util.c:  Miscellaneous stuffs
 *            
 * All blame to Mike McCool
 */

#include "libadmin/libadmin.h"
#include "base/util.h"
#include "private/pprio.h"
#include "prprf.h"

#ifdef XP_UNIX
#include <dirent.h>
#include <sys/types.h>
#include <fcntl.h>
#else
#include <base/file.h>
#include <sys/stat.h>
#endif /* WIN32? */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>     /* isdigit */

#define NUM_ENTRIES 64

#ifdef MCC_PROXY
char *
XP_GetString()
{
        return "ZAP";
}
#endif

#ifdef XP_WIN32
char *GetQueryNT(void)
{
    char *qs = getenv("QUERY_STRING");
    if(qs && (*qs == '\0'))
        qs = NULL;
    return qs;
}
#endif  /* XP_WIN32 */

void escape_for_shell(char *cmd) {
    register int x,y,l;

    l=strlen(cmd);
    for(x=0;cmd[x];x++) {
        if(strchr(" &;`'\"|*!?~<>^()[]{}$\\\x0A",cmd[x])){
            for(y=l+1;y>x;y--)
                cmd[y] = cmd[y-1];
            l++; /* length has been increased */
            cmd[x] = '\\';
            x++; /* skip the character */
        }
    }
}

int _admin_dumbsort(const void *s1, const void *s2)
{
    return strcmp(*((char **)s1), *((char **)s2));
}

#ifdef XP_UNIX /* WIN32 change */
/* Lists all files in a directory. */
char **list_directory(char *path, int dashA)  
{
    char **ar;
    DIR *ds;
    struct dirent *d;
    int n, p;

    n = NUM_ENTRIES;
    p = 0;

    ar = (char **) MALLOC(n * sizeof(char *));

    if(!(ds = opendir(path))) {
        return NULL;
    }

    while( (d = readdir(ds)) ) {
        if ( ( d->d_name[0] != '.' ) ||
	     ( dashA && d->d_name[1] &&
	       ( d->d_name[1] != '.' || d->d_name[2] ) ) ) {
            if(p == (n-1)) {
                n += NUM_ENTRIES;
                ar = (char **) REALLOC(ar, n*sizeof(char *));
            }
            /* 2: Leave space to add a trailing slash later */
            ar[p] = (char *) MALLOC(strlen(d->d_name) + 2);
            strcpy(ar[p++], d->d_name);
        }
    }
    closedir(ds);

    qsort((void *)ar, p, sizeof(char *), _admin_dumbsort);
    ar[p] = NULL;

    return ar;
}

#else /* WIN32 change */
/* Lists all files in a directory. */
char **list_directory(char *path, int dashA)  
{
    char **ar;
    SYS_DIR ds;
    SYS_DIRENT *d;
    int n, p;

    n = NUM_ENTRIES;
    p = 0;

    ar = (char **) MALLOC(n * sizeof(char *));

    if(!(ds = dir_open(path))) {
        return NULL;
    }

    while( (d = dir_read(ds)) ) {
        if ( ( d->d_name[0] != '.' ) ||
	     ( dashA && d->d_name[1] &&
	       ( d->d_name[1] != '.' || d->d_name[2] ) ) ) {
            if(p == (n-1)) {
                n += NUM_ENTRIES;
                ar = (char **) REALLOC(ar, n*sizeof(char *));
            }
            /* 2: Leave space to add a trailing slash later */
            ar[p] = (char *) MALLOC(strlen(d->d_name) + 2);
            strcpy(ar[p++], d->d_name);
        }
    }
    dir_close(ds);

    qsort((void *)ar, p, sizeof(char *), _admin_dumbsort);
    ar[p] = NULL;

    return ar;
}
#endif /* WIN32 */

int file_exists(char *fn)
{
    struct stat finfo;

    if(!stat(fn, &finfo))
        return 1;
    else
        return 0;
}

int get_file_size(char *fn)
{
    struct stat finfo;
    int ans = -1;
 
    if(!stat(fn, &finfo))  {
        ans = finfo.st_size;
    }  else  {
        report_error(FILE_ERROR, fn, "Could not get size of file.");
    }
    return ans;
}

/* return: mtime(f1) < mtime(f2) ? */
int mtime_is_earlier(char *file1, char *file2)
{
    struct stat fi1, fi2;

    if(stat(file1, &fi1))  {
        return -1;
    }
    if(stat(file2, &fi2))  {
        return -1;
    }
    return( (fi1.st_mtime < fi2.st_mtime) ? 1 : 0);
}

time_t get_mtime(char *fn)
{
    struct stat fi;

    if(stat(fn, &fi))
        return 0;
    return fi.st_mtime;
}

int all_numbers(char *target) 
{
    register int x=0;
  
    while(target[x])
        if(!isdigit(target[x++]))
            return 0;
    return 1;
}


int all_numbers_float(char *target) 
{
    register int x;
    int seenpt;

    for(x = 0, seenpt = 0; target[x]; ++x) {
        if((target[x] == '.') && (!seenpt))
            seenpt = 1;
        else if((!isdigit(target[x])) && seenpt)
            return 0;
    }
    return 1;
}

/* Get the current HTTP server URL. */
char *get_serv_url(void)
{
    return(getenv("SERVER_URL"));
}

/* ------------------------------- run_cmd -------------------------------- */


/* Previously in install. This is also pretty UNIX-ish. */

/* Nirmal: Added  code for Win32 implementation of this function. */

#include <signal.h>
#ifdef XP_UNIX
#include <sys/wait.h>
#endif /* XP_UNIX */


int run_cmd(char *cmd, FILE *closeme, struct runcmd_s *rm)
{
#ifdef WIN32
    HANDLE hproc;
    PROCESS_INFORMATION child;
    STARTUPINFO  siStartInfo ;
#else
    struct stat fi;
    int exstat;
    char *errmsg, tfn[128];
    FILE *f;
    int fd;
    pid_t pid;
#endif


#ifdef WIN32
    /* Nirmal:
	    For now, i will just spawn
	    a child in WINNT to execute the command. Communication to
	    the parent is done through stdout pipe, that was setup by
	    the parent.

    */	    
    hproc = OpenProcess(STANDARD_RIGHTS_REQUIRED, FALSE, GetCurrentProcessId());
    if (hproc == NULL) {
	    fprintf(stdout, "Content-type: text/html\n\n");
	    fflush(stdout);
	    report_error(SYSTEM_ERROR, NULL, "Could not open handle to myself");
	    return -1;  // stmt. not reached.
    }

    ZeroMemory(&child, sizeof(PROCESS_INFORMATION));
    ZeroMemory(&siStartInfo, sizeof(STARTUPINFO));
    siStartInfo.cb = sizeof(STARTUPINFO);
    siStartInfo.lpReserved = siStartInfo.lpReserved2 = NULL;
    siStartInfo.cbReserved2 = 0;
    siStartInfo.lpDesktop = NULL;
    siStartInfo.dwFlags =  STARTF_USESHOWWINDOW;
//	Several fields arent used when dwFlags is not set.    
//    siStartInfo.hStdInput = hChildStdinRd;
//    siStartInfo.hStdOutput = siStartInfo.hStdError = hChildStdoutWr;
    siStartInfo.wShowWindow = SW_HIDE;
									
    if ( ! CreateProcess(
	    NULL,	// pointer to name of executable module
	    cmd,	// pointer to command line string
	    NULL,	// pointer to process security attribute
	    NULL,	// pointer to thread security attributes
	    TRUE,	// handle inheritance flag
	    0,		// creation flags
	    NULL,	// pointer to new environment block
	    NULL,	// pointer to current directory name
	    &siStartInfo,	// pointer to STARTUPINFO
	    &child	// pointer to PROCESS_INFORMATION
	   ))
    {
        rm->title = "CreateProcess failed";
        rm->msg = "run_cmd: Can't create new process. ";
        rm->arg = "";
        rm->sysmsg = 1;
	return -1;
    }
    else 
        return 0;
#else
    sprintf(cmd, "%s > /tmp/startmsg.%d 2>&1 < /dev/null", cmd, getpid()); /*  */
    /* FUCK UNIX SIGNALS. */
    signal(SIGCHLD, SIG_DFL);
    switch( (pid = fork()) ) {
      case 0:
        /* Hmm. Work around an apparent bug in stdio. */
        if(closeme)
            close(fileno(closeme));
        execl("/bin/sh", "/bin/sh", "-c", cmd, (char *)NULL);
        /* DOH! */
        sprintf(tfn, "/tmp/startmsg.%d", getpid());
        if(!(f = fopen(tfn, "w")))
            exit(1);
        fprintf(f, "Exec of %s failed. The error was %s.\n", cmd, 
                system_errmsg());
        fclose(f);
        exit(1);
      case -1:
        rm->title = "Fork failed";
        rm->msg = "Can't create new process. %s";
        rm->arg = "";
        rm->sysmsg = 1;
        return -1;
      default:
        sprintf(tfn, "/tmp/startmsg.%d", getpid());

        if(waitpid(pid, &exstat, 0) == -1) {
            rm->title = "Can't wait for child";
            rm->msg = "Can't wait for process. %s";
            rm->arg = "";
            rm->sysmsg = 1;
            return -1;
        }
        if(exstat) {
            if(!(fd = open(tfn, O_RDONLY))) {
                rm->title = "Can't open error file";
                rm->msg = "Can't find error file %s.";
                rm->arg = cmd;
                rm->sysmsg = 1;
                return -1;
            }
            fstat(fd, &fi);
            if((fi.st_size > 0) && (fi.st_size < 8192)) {
                errmsg = (char *) MALLOC(fi.st_size + 1);
                read(fd, errmsg, fi.st_size);
                errmsg[fi.st_size] = '\0';
                close(fd);
                unlink(tfn);
                rm->title = "Command execution failed";
                rm->msg = "The command did not execute. "
                          "Here is the output:<p>\n<pre>\n%s\n</pre>\n";
                rm->arg = errmsg;
                rm->sysmsg = 0;
                return -1;
            }
            else {
                close(fd);
                unlink(tfn);

                rm->title = "Command execution failed";
                rm->msg = "The command didn't execute, and it did not produce "
                          "any output. Run <code>%s</code> from the command "
                          "line and examine the output.\n";
                rm->arg = cmd;
                rm->sysmsg = 0;
                return -1;
            }
        }
        unlink(tfn);
        return 0;
    }
#endif  /* WIN32 */

}



/* This is basically copy_file from the install section, with the error 
 * reporting changed to match the admin stuff.  Since some stuff depends
 * on copy_file being the install version, I'll cheat and call this one
 * cp_file. */
#ifdef XP_UNIX
 
#define COPY_BUFFER_SIZE        4096

void cp_file(char *sfile, char *dfile, int mode)
{
    int sfd, dfd, len;
    struct stat fi;
 
    char copy_buffer[COPY_BUFFER_SIZE];
    unsigned long read_len;

/* Make sure we're in the right umask */
    umask(022);

    if( (sfd = open(sfile, O_RDONLY)) == -1)
        report_error(FILE_ERROR, sfile, "Can't open file for reading.");
 
    fstat(sfd, &fi);
    if(!(S_ISREG(fi.st_mode))) {
        close(sfd);
        return;
    }
    len = fi.st_size;
 
    if( (dfd = open(dfile, O_RDWR | O_CREAT | O_TRUNC, mode)) == -1)
        report_error(FILE_ERROR, dfile, "Can't write to file.");
 
    while(len) {
        read_len = len>COPY_BUFFER_SIZE?COPY_BUFFER_SIZE:len;
 
        if ( (read_len = read(sfd, copy_buffer, read_len)) == -1) {
            report_error(FILE_ERROR, sfile, "Error reading file for copy.");
        }
 
        if ( write(dfd, copy_buffer, read_len) != read_len) {
            report_error(FILE_ERROR, dfile, "Error writing file for copy.");
        }
 
        len -= read_len;
    }
    close(sfd);
    close(dfd);
}

#else /* XP_WIN32 */
void cp_file(char *sfile, char *dfile, int mode)
{
    HANDLE sfd, dfd, MapHandle;
    PCHAR fp;
    DWORD BytesWritten = 0;
    DWORD len;

    if( (sfd = CreateFile(sfile, GENERIC_READ,
    		FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
    		OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL))
    			== INVALID_HANDLE_VALUE) {
        report_error(FILE_ERROR, "Cannot open file for reading", sfile);
    }
    len = GetFileSize(sfd, NULL);
    if( (dfd = CreateFile(dfile, GENERIC_READ | GENERIC_WRITE,
    	FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,CREATE_ALWAYS,
    	FILE_ATTRIBUTE_NORMAL, NULL)) == INVALID_HANDLE_VALUE) {
        report_error(FILE_ERROR, "Cannot open destination file for writing", 
		dfile);
    }
	if (len == 0)
		return;
    if( (MapHandle = CreateFileMapping(sfd, NULL, PAGE_READONLY,
    		0, 0, NULL)) == NULL) {
        report_error(FILE_ERROR, "Cannot create file mapping", sfile);
    }
    if (!(fp = MapViewOfFile(MapHandle, FILE_MAP_READ, 0, 0, 0))) {
        report_error(FILE_ERROR, "Cannot map file %s", sfile);
    }
    while ( len) {
    	if(!WriteFile(dfd, fp, len, &BytesWritten, NULL)) {
            report_error(FILE_ERROR, "Cannot write new file", dfile);
	    }
    	len -= BytesWritten;
    	fp += BytesWritten;
    }

    CloseHandle(sfd);
    UnmapViewOfFile(fp);
    CloseHandle(MapHandle);
    FlushFileBuffers(dfd);
    CloseHandle(dfd);
}
#endif

int delete_file(char *path)
{
#ifdef XP_UNIX
    return unlink(path);
#else
    return !(DeleteFile(path));
#endif
}

void create_dir(char *dir, int mode)
{
    if ((dir == (char *) NULL) || (strlen(dir) == 0)) {
       report_error(FILE_ERROR, "No directory is specified",
                         "Could not create a necessary directory.");
    }

    if(!file_exists(dir)) {
#ifdef XP_UNIX
        if(mkdir(dir, mode) == -1)  {
#else  /* XP_WIN32 */
        if(!CreateDirectory(dir, NULL)) {
            if (GetLastError() != ERROR_ALREADY_EXISTS)
#endif /* XP_WIN32 */
            report_error(FILE_ERROR, dir,
                         "Could not create a necessary directory.");
        }
    }
}

#ifdef XP_UNIX
SYS_FILE lf;
#elif defined(XP_WIN32)
HANDLE lf;
#endif

char *get_flock_path(void)
{
    char *result="";
    char *port=getenv("SERVER_PORT");
#ifdef XP_UNIX
    result=(char *) MALLOC(strlen("/tmp/lock.%%s.")+strlen(port)+4);
    sprintf(result, "/tmp/lock.%%s.%s", port);
#endif
    return result;
}

/* Open a file with locking, close a file with unlocking. */
FILE *fopen_l(char *path, char *mode)
{
    FILE *f = fopen(path, mode);
    char *lockpath;
    char *sn="admin-serv";
    char *flp=FILE_LOCK_PATH;
 
    if(f == NULL) return NULL;
    lockpath=(char *) MALLOC(strlen(sn)+strlen(flp)+16);
    sprintf(lockpath, flp, sn);
#ifdef XP_UNIX
    if( (lf=system_fopenRW(lockpath)) == SYS_ERROR_FD)
        report_error(FILE_ERROR, lockpath, "Could not open file.");
    if(system_flock(lf)==IO_ERROR)
        report_error(FILE_ERROR, lockpath, "Could not lock file.");
#elif defined(XP_WIN32)
    /* Using mutexes because if the CGI program dies, the mutex will be
     * automatically released by the OS for another process to grab.  
     * Semaphores do not have this property; and if the CGI program crashes,
     * the admin server would be effectively crippled.
     */
	if ( (lf = CreateMutex(NULL, 0, lockpath)) == NULL) {
        report_error(FILE_ERROR, lockpath, "Could not create admin mutex.");
	} else {
        if ( WaitForSingleObject(lf, 60*1000) == WAIT_FAILED) {
            report_error(FILE_ERROR, lockpath, "Unable to obtain mutex after 60 seconds.");
        }
	}
#endif /* XP_UNIX */
    return f;
}

void fclose_l(FILE *f)
{
    fclose(f);
#ifdef XP_UNIX
    if(system_ulock(lf)==IO_ERROR)
        report_error(FILE_ERROR, NULL, "Could not unlock lock file.");
    system_fclose(lf);
#elif defined(XP_WIN32)
    if (lf) {
        ReleaseMutex(lf);
        CloseHandle(lf);
    }
#endif /* XP_UNIX */
}

/* Ripped off from the client.  (Sorry, Lou.) */
/* */
/* The magic set of 64 chars in the uuencoded data */
unsigned char uuset[] = {
'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T',
'U','V','W','X','Y','Z','a','b','c','d','e','f','g','h','i','j','k','l','m','n',
'o','p','q','r','s','t','u','v','w','x','y','z','0','1','2','3','4','5','6','7',
'8','9','+','/' };
 
int do_uuencode(unsigned char *src, unsigned char *dst, int srclen)
{
   int  i, r;
   unsigned char *p;
 
/* To uuencode, we snip 8 bits from 3 bytes and store them as
6 bits in 4 bytes.   6*4 == 8*3 (get it?) and 6 bits per byte
yields nice clean bytes
 
It goes like this:
        AAAAAAAA BBBBBBBB CCCCCCCC
turns into the standard set of uuencode ascii chars indexed by numbers:
        00AAAAAA 00AABBBB 00BBBBCC 00CCCCCC
 
Snip-n-shift, snip-n-shift, etc....
 
*/
 
   for (p=dst,i=0; i < srclen; i += 3) {
                /* Do 3 bytes of src */
                register char b0, b1, b2;
 
                b0 = src[0];
                if (i==srclen-1)
                        b1 = b2 = '\0';
                else if (i==srclen-2) {
                        b1 = src[1];
                        b2 = '\0';
                }
                else {
                        b1 = src[1];
                        b2 = src[2];
                }
 
                *p++ = uuset[b0>>2];
                *p++ = uuset[(((b0 & 0x03) << 4) | ((b1 & 0xf0) >> 4))];
                *p++ = uuset[(((b1 & 0x0f) << 2) | ((b2 & 0xc0) >> 6))];
                *p++ = uuset[b2 & 0x3f];
                src += 3;
   }
   *p = 0;      /* terminate the string */
   r = (unsigned char *)p - (unsigned char *)dst;/* remember how many we did */
 
   /* Always do 4-for-3, but if not round threesome, have to go
          clean up the last extra bytes */
 
   for( ; i != srclen; i--)
                *--p = '=';
 
   return r;
}
 
char *alert_word_wrap(char *str, int width, char *linefeed)
{
    char *ans = NULL;
    int counter=0;
    int lsc=0, lsa=0;
    register int strc=0, ansc=0;
    register int x=0;
 
    /* assume worst case */
    ans = (char *) MALLOC((strlen(str)*strlen(linefeed))+32);
 
    for(strc=0, ansc=0; str[strc]; /*none*/)  {
        if(str[strc]=='\n')  {
            counter=0;
            lsc=0, lsa=0;
            for(x=0; linefeed[x]; x++)  {
                ans[ansc++]=linefeed[x];
            }
            strc++;
        }  else if(str[strc]=='\r')  {
            strc++;
        }  else if(str[strc]=='\\')  {
            ans[ansc++]='\\';
            ans[ansc++]=strc++;
        }  else  {
            if(counter==width)  {
                if(lsc && lsa)  {
                    strc=lsc;
                    ansc=lsa;
 
                    counter=0;
                    lsc=0, lsa=0;
                    for(x=0; linefeed[x]; x++)  {
                        ans[ansc++]=linefeed[x];
                    }
                    strc++;
                }  else  {
                /* else, you're a loser, I'm breaking your big word anyway */
                    counter=0;
                    lsc=0, lsa=0;
                    for(x=0; linefeed[x]; x++)  {
                        ans[ansc++]=linefeed[x];
                    }
                    strc++;
                }
            }  else  {
                if(str[strc] == ' ')  {
                    lsc=strc;
                    lsa=ansc;
                }
                ans[ansc++]=str[strc++];
                counter++;
            }
        }
    }
    ans[ansc]='\0';
    return ans;
}

void remove_directory(char *path)
{
    struct stat finfo;
    char **dirlisting;
    register int x=0;
    int stat_good = 0;
    char *fullpath = NULL;

#ifdef XP_UNIX
    stat_good = (lstat(path, &finfo) == -1 ? 0 : 1);
#else /* WIN32 */
    stat_good = (stat(path, &finfo) == -1 ? 0 : 1);
#endif /* XP_UNIX */

    if(!stat_good) return;

    if(S_ISDIR(finfo.st_mode))  {
        dirlisting = list_directory(path,1);
        if(!dirlisting) return;

        for(x=0; dirlisting[x]; x++)  {
            fullpath = (char *) MALLOC(strlen(path) +
                                       strlen(dirlisting[x]) + 4);
            sprintf(fullpath, "%s%c%s", path, FILE_PATHSEP, dirlisting[x]);
#ifdef XP_UNIX
            stat_good = (lstat(fullpath, &finfo) == -1 ? 0 : 1);
#else /* WIN32 */
            stat_good = (stat(fullpath, &finfo) == -1 ? 0 : 1);
#endif /* XP_UNIX */
            if(!stat_good) continue;
            if(S_ISDIR(finfo.st_mode))  {
                remove_directory(fullpath);
            }  else  {
                fprintf(stdout, "<i>Removing file</i> "
                                "<code>%s</code><br>\n", fullpath);
                unlink(fullpath);
            }
            FREE(fullpath);
        }
        fprintf(stdout, "<i>Removing directory</i> "
                        "<code>%s</code><br>\n", path);
#ifdef XP_UNIX
        rmdir(path);
#else /* XP_WIN32 */
                RemoveDirectory(path);
#endif /* XP_WIN32 */
    }  else  {
        fprintf(stdout, "<i>Removing file</i> <code>%s</code><br>\n", path);
        unlink(path);
    }
    return;
}

/* do fgets with a filebuffer *, instead of a File *.  Can't use util_getline
   because it complains if the line is too long.
   It does throw away <CR>s, though.
 */
NSAPI_PUBLIC char *system_gets( char *line, int size, filebuffer *fb )
{
  int	c;
  int	i = 0;

  while ( --size ) {
    switch ( c = filebuf_getc( fb ) ) {
    case IO_EOF:
      line[i] = '\0';
      return i ? line : NULL;
    case LF:
      line[i] = c;
      line[i+1] = '\0';
      return line;	/* got a line, and it fit! */
    case IO_ERROR:
      return NULL;
    case CR:
      ++size;
      break;
    default:
      line[i++] = c;
      break;
    }
  }
  /* if we got here, then we overran the buffer size */
  line[i] = '\0';
  return line;
}

#ifndef WIN32

/* make a zero length file, no matter how long it was before */
NSAPI_PUBLIC int
system_zero( SYS_FILE f )
{
  ftruncate( PR_FileDesc2NativeHandle( f ), 0 );
  return 0;
}

#endif

/***********************************************************************
** FUNCTION:	cookieValue
** DESCRIPTION:
**   Get the current value of the cookie variable
** INPUTS:	var - the name of the cookie variable
**		val - if non-NULL, set the in-memory copy of the var
** OUTPUTS:	None
** RETURN:	NULL if the var doesn't exist, else the value
** SIDE EFFECTS:
**	Eats memory
** RESTRICTIONS:
**	Don't screw around with the returned string, if anything else wants
**	to use it.
** MEMORY:	This is a memory leak, so only use it in CGIs
** ALGORITHM:
**	If it's never been called, build a memory structure of the
**	cookie variables.
**	Look for the passed variable, and return its value, or NULL
***********************************************************************/

NSAPI_PUBLIC char *
cookieValue( char *var, char *val )
{
  static char	**vars = NULL;
  static char	**vals = NULL;
  static int	numVars = -1;
  int		i;

  if ( numVars == -1 ) {	/* first time, init the structure */
    char	*cookie = getenv( "HTTP_COOKIE" );

    if ( cookie && *cookie ) {
      int	len = strlen( cookie );
      int	foundVal = 0;

      cookie = STRDUP( cookie );
      numVars = 0;
      vars = (char **)MALLOC( sizeof( char * ) );
      vals = (char **)MALLOC( sizeof( char * ) );
      vars[0] = cookie;
      for ( i = 0 ; i < len ; ++i ) {
	if ( ( ! foundVal ) && ( cookie[i] == '=' ) ) {	
	  vals[numVars++] = cookie + i + 1;
	  cookie[i] = '\0';
	  foundVal = 1;
	} else if ( ( cookie[i] == ';' ) && ( cookie[i+1] == ' ' ) ) {
	  cookie[i] = '\0';
	  vals = (char **) REALLOC( vals,
				    sizeof( char * ) * ( numVars + 1 ) );
	  vars = (char **) REALLOC( vars,
				    sizeof( char * ) * ( numVars + 1 ) );
	  vars[numVars] = cookie + i + 2;
	  i += 2;
	  foundVal = 0;
	}
      }
    } else {	/* no cookie, no vars */
      numVars = 0;
    }
  }
  for ( i = 0 ; i < numVars ; ++i ) {
    if ( strcmp( vars[i], var ) == 0 ) {
      if ( val ) {
	vals[i] = STRDUP( val );
      } else {
	return vals[i];
      }
    }
  }
  return NULL;
}

/***********************************************************************
** FUNCTION:	jsEscape
** DESCRIPTION:
**   Escape the usual suspects, so the parser javascript parser won't eat them
** INPUTS:	src - the string
** OUTPUTS:	NONE
** RETURN:	A malloced string, containing the escaped src
** SIDE EFFECTS:
**	None, except for more memory being eaten
** RESTRICTIONS:
**	None
** MEMORY:	One Malloc, you should free this if you care
***********************************************************************/

NSAPI_PUBLIC char *
jsEscape( char *src )
{
  int	needsEscaping = 0;
  int	i;
  char	*dest;

  for ( i = 0 ; src[i] ; ++i ) {
    if ( src[i] == '\\' || src[i] == '\'' || src[i] == '"' ) {
      ++needsEscaping;
    }
  }
  dest = (char *)MALLOC( i + needsEscaping + 1 );
  for ( i = 0 ; *src ; ++src ) {
    if ( ( *src == '\\' ) || ( *src == '\'' ) || ( *src == '"' ) ) {
      dest[i++] = '\\';	/* escape */
    }
    dest[i++] = *src;
  }
  dest[i] = '\0';
  return dest;
}

/***********************************************************************
** FUNCTION:	jsPWDialogSrc
** DESCRIPTION:
**   Put the source to the passwordDialog JavaScript function out.
** INPUTS:	inScript - if true, don't put <SCRIPT> stuff out
**		otherJS  - if nonNULL, other javascript to execute
** OUTPUTS:	None
** RETURN:	None
** SIDE EFFECTS:
**	clogs up stdout
** RESTRICTIONS:
**	Don't use this outside of a CGI, or before the Content-type:
** MEMORY:	No memory change
** ALGORITHM:
**	@+@What's really going on?
***********************************************************************/

NSAPI_PUBLIC void
jsPWDialogSrc( int inScript, char *otherJS )
{
  static int	srcSpewed = 0;

  otherJS = otherJS ? otherJS : "";

  if ( ! inScript ) {
    fprintf( stdout, "<SCRIPT LANGUAGE=\""MOCHA_NAME"\">\n" );
  }
  if ( ! srcSpewed ) {
    srcSpewed = 1;
    fprintf( stdout, "function passwordDialog( prompt, element ) {\n"
	     "    var dlg = window.open( '', 'dialog', 'height=60,width=500' );\n"
	     "    dlg.document.write(\n"
	     "        '<form name=f1 onSubmit=\"opener.document.'\n"
	     "        + element + '.value = goo.value; window.close(); "
	     "%s; return false;\">',\n"
	     "        prompt, '<input type=password name=goo></form>' );\n"
	     "    dlg.document.f1.goo.focus();\n"
	     "    dlg.document.close();\n"
	     "}\n", otherJS );
  }
  if ( ! inScript ) {
    fprintf( stdout, "</SCRIPT>\n" );
  }
}

/* this macro was copied from libldap/tmplout.c */

#define HREF_CHAR_ACCEPTABLE( c )	(( c >= '-' && c <= '9' ) ||	\
					 ( c >= '@' && c <= 'Z' ) ||	\
					 ( c == '_' ) ||		\
					 ( c >= 'a' && c <= 'z' ))

/* this function is copied from libldap/tmplout.c:strcat_escaped */
NSAPI_PUBLIC void htmladmin_strcat_escaped( char *s1, char *s2 )
{
    unsigned char	*q;
    char		*p, *hexdig = "0123456789ABCDEF";

    p = s1 + strlen( s1 );
    for ( q = (unsigned char *)s2; *q != '\0'; ++q ) {
	if ( HREF_CHAR_ACCEPTABLE( *q )) {
	    *p++ = *q;
	} else {
	    *p++ = '%';
	    *p++ = hexdig[ *q >> 4 ];
	    *p++ = hexdig[ *q & 0x0F ];
	}
    }

    *p = '\0';
}


NSAPI_PUBLIC char *htmladmin_strdup_escaped( char *s )
{
    char	*p;

    p = (char *)malloc( 3 * strlen( s ) + 1 );
    *p = '\0';
    htmladmin_strcat_escaped( p, s );
    return( p );
}

/* returns true if the given path is a valid directory, false otherwise */
NSAPI_PUBLIC int
util_is_dir_ok(const char *path)
{
    PRFileInfo prinfo;
    int ret = 0;

    if (path && *path &&
	(PR_SUCCESS == PR_GetFileInfo(path, &prinfo)) &&
	prinfo.type == PR_FILE_DIRECTORY) {
	ret = 1;
    }

    return ret;
}

/* returns true if the given path is a valid file, false otherwise */
NSAPI_PUBLIC int
util_is_file_ok(const char *path)
{
    PRFileInfo prinfo;
    int ret = 0;

    if (path && *path &&
	(PR_SUCCESS == PR_GetFileInfo(path, &prinfo)) &&
	prinfo.type == PR_FILE_FILE) {
	ret = 1;
    }

    return ret;
}

/* returns true if the file was found somewhere, false otherwise */
NSAPI_PUBLIC int
util_find_file_in_paths(
    char *filebuf, /* this will be filled in with the full path/filename if found, '\0' otherwise */
    size_t bufsize, /* size of filebuf e.g. sizeof(filebuf) */
    const char *filename, /* the base filename to look for */
    const char *path, /* path given by caller */
    const char *arpath, /* path relative to ADMSERV_ROOT */
    const char *nrpath /* path relative to NETSITE_ROOT */
)
{
    int retval = 0;
    const char *admroot = getenv("ADMSERV_CONF_DIR");

    /* try given path */
    PR_snprintf(filebuf, bufsize, "%s/%s", path, filename);
    retval = util_is_file_ok(filebuf);
    if (!retval && admroot) { /* look in ADMSERV_ROOT */
	PR_snprintf(filebuf, bufsize, "%s%s%s/%s", admroot,
		    (arpath && *arpath) ? "/" : "",
		    (arpath && *arpath) ? arpath : "",
		    filename);
	retval = util_is_file_ok(filebuf);
    }

    if (!retval) {
	filebuf[0] = '\0';
    }

    return retval;
}

NSAPI_PUBLIC char*
util_get_conf_dir(void)
{
    const char *admroot = getenv("ADMSERV_CONF_DIR");
    static char confdir[PATH_MAX];

    if (!confdir[0]) {
        if (util_is_dir_ok(admroot)) {
            PR_snprintf(confdir, sizeof(confdir), admroot);
        } else if (util_is_dir_ok(CONFIGDIR)) {
            PR_snprintf(confdir, sizeof(confdir), CONFIGDIR);
        } else {
            return NULL;
        }
    }

    return confdir;
}

NSAPI_PUBLIC char*
util_get_security_dir(void)
{
    const char *admroot = getenv("ADMSERV_CONF_DIR");
    static char secdir[PATH_MAX];

    if (!secdir[0]) {
        if (util_is_dir_ok(admroot)) {
            PR_snprintf(secdir, sizeof(secdir), admroot);
        } else if (util_is_dir_ok(SECURITYDIR)) {
            PR_snprintf(secdir, sizeof(secdir), SECURITYDIR);
        } else {
            return util_get_conf_dir();
        }
    }

    return secdir;
}

NSAPI_PUBLIC char*
util_get_log_dir(void)
{
    const char *admroot = getenv("ADMSERV_LOG_DIR");
    static char logdir[PATH_MAX];

    if (!logdir[0]) {
        if (util_is_dir_ok(admroot)) {
            PR_snprintf(logdir, sizeof(logdir), admroot);
        } else if (util_is_dir_ok(LOGDIR)) {
            PR_snprintf(logdir, sizeof(logdir), LOGDIR);
        } else {
            return NULL;
        }
    }

    return logdir;
}

NSAPI_PUBLIC char*
util_get_pid_dir(void)
{
    const char *admroot = getenv("ADMSERV_PID_DIR");
    static char piddir[PATH_MAX];

    if (!piddir[0]) {
        if (util_is_dir_ok(admroot)) {
            PR_snprintf(piddir, sizeof(piddir), admroot);
        } else if (util_is_dir_ok(PIDDIR)) {
            PR_snprintf(piddir, sizeof(piddir), PIDDIR);
        } else {
            return NULL;
        }
    }

    return piddir;
}

/* old style html dir == progpath/../html */
#define HTML_DIR "../html/"

NSAPI_PUBLIC const char*
util_get_html_dir(void)
{
    const char *admroot = getenv("ADMSERV_ROOT");
    const char *nsroot = getenv("NETSITE_ROOT");
    static char htmldir[PATH_MAX];

    if (!htmldir[0]) {
        if (util_is_dir_ok(HTMLDIR)) {
            PR_snprintf(htmldir, sizeof(htmldir), HTMLDIR);
        } else if (util_is_dir_ok(admroot)) {
            PR_snprintf(htmldir, sizeof(htmldir), "%s/../../bin/admin/admin/html", admroot);
        } else if (util_is_dir_ok(nsroot)) {
            PR_snprintf(htmldir, sizeof(htmldir), "%s/bin/admin/admin/html", nsroot);
        } else {
            PR_snprintf(htmldir, sizeof(htmldir), HTML_DIR);
        }
    }

    return htmldir;
}

NSAPI_PUBLIC const char*
util_get_icon_dir(void)
{
    const char *admroot = getenv("ADMSERV_ROOT");
    const char *nsroot = getenv("NETSITE_ROOT");
    static char icondir[PATH_MAX];

    if (!icondir[0]) {
        if (util_is_dir_ok(ICONDIR)) {
            PR_snprintf(icondir, sizeof(icondir), ICONDIR);
        } else if (util_is_dir_ok(admroot)) {
            PR_snprintf(icondir, sizeof(icondir), "%s/../../bin/admin/admin/icons", admroot);
        } else if (util_is_dir_ok(nsroot)) {
            PR_snprintf(icondir, sizeof(icondir), "%s/bin/admin/admin/icons", nsroot);
        } else {
            return NULL;
        }
    }

    return icondir;
}

/* return true if all of the chars in s are valid chars for use in
   file and directory names, and false otherwise.  This means that
   the string must begin with a letter or number, and must contain
   letters, numbers, '.', '-' and '_'.
   The main purpose of this is to see if a malicious client is sending
   us bogus path names in an attempt to gain access or DoS.
*/
NSAPI_PUBLIC int
util_is_valid_path_string(const char *s)
{
    int ret = 0;
    if (s) {
	if (isalnum(*s)) {
	    ret = 1;
	    for(; ret && *s; ++s) {
		ret = isalnum(*s) || (*s == '-') || (*s == '_') || (*s == '.');
	    }
	}
    }
    return ret;
}

/* try various ways to determine if the given name is a valid
   file or directory - this value is passed in as a form
   parameter, and our motto is "Don't trust the user!"
   If the given filetype is directory, also check to see if the optional
   given filename (may be NULL) is in the given directory
*/
NSAPI_PUBLIC int
util_verify_file_or_dir(
    const char *name, /* name of file or directory to check */
    PRFileType filetype, /* type of name */
    const char *childname, /* optional child file/dir to check inside given parent name */
    size_t childlen, /* only compare first childlen chars of childname - use -1 for entire string */
    PRFileType childtype /* type of child */
)
{
    int ret = 0;
    PRFileInfo fileinfo;
    /* first, just a simple access check */
    PRStatus status = PR_GetFileInfo(name, &fileinfo);
    ret = ((status == PR_SUCCESS) && (fileinfo.type == filetype));
    if (ret) {
        /* checks out ok - let's split it into the base name and the parent dir,
           open the parent dir, and see if the base name exists in the parent dir
        */
        char *copy = PL_strdup(name);
        size_t len = strlen(copy);
        char *ptr = &copy[len-1];
        /* get the basename - a really bad name may look like
           /path/foo/// or even ///////////////// */
        for (; (ptr > copy) && (*ptr == '/'); --ptr) {
            /* do nothing */
        }
        if ((ptr == copy) && (*ptr == '/')) {
            /* bad - string consists of nothing but '/' */
            ptr = NULL;
            ret = 0;
        } else {
            PRDir *pdir;
            PRDirEntry *pent;

            ret = 0;
            if (*ptr == '/') {
                *ptr = 0; /* terminate the string at the first trailing '/' */
            }
            ptr = strrchr(copy, '/');
            if (!ptr) {
                ptr = copy;
                copy = PL_strdup(".");
            } else {
                *ptr = 0;
                ++ptr;
                ptr = PL_strdup(ptr);
            }
            /* copy now points at the parent, ptr at the child */
            if ((pdir = PR_OpenDir(copy))) {
                for(pent = PR_ReadDir(pdir, PR_SKIP_BOTH); pent && !ret;
                    pent = PR_ReadDir(pdir, PR_SKIP_BOTH)) {
                    ret = !strcmp(pent->name, ptr);
                }
                PR_CloseDir(pdir);
            }
            if (ret && childname && (filetype == PR_FILE_DIRECTORY)) {
                ret = 0;
                /* we've verified that name is a valid directory - see if
                   the given filename exists in that directory */
                if ((pdir = PR_OpenDir(name))) {
                    for(pent = PR_ReadDir(pdir, PR_SKIP_BOTH); pent && !ret;
                        pent = PR_ReadDir(pdir, PR_SKIP_BOTH)) {
                        if (childlen > 0) {
                            ret = !strncmp(pent->name, childname, childlen);
                        } else {
                            ret = !strcmp(pent->name, childname);
                        }
                    }
                    PR_CloseDir(pdir);
                    if (ret) {
                        /* child exists - check type */
                        char *fullname = PR_smprintf("%s%c%s", name, FILE_PATHSEP, childname);
                        status = PR_GetFileInfo(fullname, &fileinfo);
                        ret = ((status == PR_SUCCESS) && (fileinfo.type == childtype));
                        PR_smprintf_free(fullname);
                    }
                }
            }
        }
        PL_strfree(copy);
        PL_strfree(ptr);
    }
    return ret;
}

NSAPI_PUBLIC int
util_psetHasObjectClass(PsetHndl pset, const char *ocname)
{
    int rval = 0;
    char *val = NULL;
    ValueType nodeObjectClass = psetGetObjectClass(pset, "" /* use root node */, &rval);
    ValueType iter = nodeObjectClass;

    rval = 0;
    while ((!rval) && ((val = *iter++))) {
        rval = !PL_strcasecmp(ocname, val);
    }

    deleteValue(nodeObjectClass);
    return rval;
}

